/* Combustion Analysis Tool (CAT)
   www.catool.org
   
   Filename: utils.c

   Purpose:  Provide common programming interface for processing acquired engine data
  
   Author:   Ben Brown
   Version:  1.2
   Date:     19.10.2015

   Revision: GPL Release

   Copyright (C) Xarin Limited, 2000-2024

      This program is free software: you can redistribute it and/or modify
      it under the terms of version 2 of the GNU General Public License as
      published by the Free Software Foundation.

      This program is distributed in the hope that it will be useful,
      but WITHOUT ANY WARRANTY; without even the implied warranty of
      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
      GNU General Public License for more details.

      You should have received a copy of the GNU General Public License
      along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* For Microsoft Visual C++ */
#ifdef _MSC_VER
#include "stdafx.h"
#else
#ifndef strnlen_s
#define strnlen_s(str, strsz) strlen((str))
#endif
#endif

#include <ctype.h>
#include <stdlib.h>
#include <float.h>
#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include <time.h>

#if !defined(__cplusplus)
typedef int bool;
#define true 1
#define false 0
#endif

extern unsigned int debug_level;

#include "utils.h"

#ifdef __WXMSW__
#include "wx/wx.h"
#include "utilities.h"
#endif

long fsize(FILE* file_handle)
{
	long file_position = 0;
	long return_code;
	
	file_position = ftell(file_handle);
	
	if (file_position == -1)
	{
		return_code = -1;
	}
	else if (fseek(file_handle,0,SEEK_END) == 0)
	{
		return_code = ftell(file_handle);
	}
	else
	{
		return_code = -1;
	}
		
	if ((file_position >= 0) && (fseek(file_handle,file_position,SEEK_SET) != 0))
	{
		return_code = -1;
	}

	return(return_code);
}

size_t fwrite_short(const short data,FILE* file_handle)
{
	return(fwrite(&data,2,1,file_handle));
}

size_t fwrite_char(const char* data,const size_t length,FILE* file_handle)
{
	return(fwrite(data,length,1,file_handle));
}

size_t fwrite_double(const double data,FILE* file_handle)
{
	return(fwrite(&data,8,1,file_handle));
}

size_t fwrite_null(const unsigned int count, FILE* file_handle)
{
	size_t result = 0;
	unsigned int counter;
	char zero = 0x00;

	for (counter = 0; counter < count; counter++)
	{
		result += fwrite(&zero, 1, 1, file_handle);
	}
	return(result);
}

size_t fwrite_float(const float data,FILE* file_handle)
{
	return(fwrite(&data,4,1,file_handle));
}

size_t fwrite_int(const int data,FILE* file_handle)
{
	return(fwrite(&data,4,1,file_handle));
}

size_t fwrite_uint(const unsigned int data,FILE* file_handle)
{
	return(fwrite(&data,4,1,file_handle));
}

size_t fwrite_padded_char(const char* data,const size_t length,FILE* file_handle)
{
	size_t char_length = 0;
	size_t loop;
	size_t written = 0;

	while ((char_length < length) && (data[char_length] != 0x00))
	{
		char_length += 1;
	}

	written += fwrite(data,char_length,1,file_handle);

	for(loop=0;loop<(length-char_length);loop++)
	{
		written += fwrite("\0",1,1,file_handle);
	}

	return(written);
}

size_t fwrite_string(const char* string, FILE* file, const size_t maximum_count)
{
	size_t length;
	unsigned int zero = 0;
	size_t result;

	if ((file == NULL) || (string == NULL))
	{
		return(0);
	}

	length = strnlen_s(string,maximum_count);

	if (length > 0)
	{
		result = fwrite(&length, 1, 4, file);
		result += fwrite(string, 1, length, file);
	}
	else
	{
		result = fwrite(&zero, 1, 4, file);
	}

	return(result);
}

size_t fread_string(char* string, FILE* file, const size_t maximum_count)
{
	unsigned int length;
	size_t result;
	size_t position;
	char null;

	if ((file == NULL) || (string == NULL) || (maximum_count == 0))
	{
		return(0);
	}

	result = fread(&length, 4, 1, file);

	if (result == 1)
	{
		if (length >= maximum_count)
		{
			result = fread(string, maximum_count, 1, file);
			string[maximum_count-1] = 0x00;

			for (position = maximum_count; position < length; position++)
			{
				fread(&null, 1, 1, file);
			}
		}
		else if (length > 0)
		{
			result = fread(string, length, 1, file);
			string[length] = 0x00;
		}
		else
		{
			string[0] = 0x00;
		}
	}

	return(result);
}

void swap_endian_2(void* input)
{
	char temp;
	char* variable = (char*)input;

	temp = variable[0];

	variable[0] = variable[1];
	variable[1] = temp;
}

void swap_endian_4(void* input)
{
	char temp[2];
	char* variable = (char*)input;

	temp[0] = variable[0];
	temp[1] = variable[1];

	variable[0] = variable[3];
	variable[1] = variable[2];
	variable[2] = temp[1];
	variable[3] = temp[0];
}

void swap_endian_8(void* input)
{
	char temp[4];
	char* variable = (char*)input;

	temp[0] = variable[0];
	temp[1] = variable[1];
	temp[2] = variable[2];
	temp[3] = variable[3];

	variable[0] = variable[7];
	variable[1] = variable[6];
	variable[2] = variable[5];
	variable[3] = variable[4];
	variable[4] = temp[3];
	variable[5] = temp[2];
	variable[6] = temp[1];
	variable[7] = temp[0];
}

/* This makes the assumption that the AVL time format is always fixed as:
  
        DD-MM-YY HH:MM:SS
  
   Note that years upto 79 are assumed upto 2079, above that 1980 onwards */

time_t convert_time(char* string_time)
{
	struct tm time_tmp;
	int tmp;

	time_tmp.tm_year = 109;
	time_tmp.tm_mon = 0;
	time_tmp.tm_mday = 1;
	time_tmp.tm_hour = 0;
	time_tmp.tm_min = 0;
	time_tmp.tm_sec = 0;
	time_tmp.tm_wday = 0;
	time_tmp.tm_yday = 0;
	time_tmp.tm_isdst = 0;
	
	if (string_time != NULL)
	{
		if ((strnlen_s(string_time,10) <= 10) &&
			((sscanf(string_time, "%04d/%02d/%04d", &time_tmp.tm_year, &time_tmp.tm_mon, &time_tmp.tm_mday) == 3) ||
		     (sscanf(string_time, "%04d.%02d.%04d", &time_tmp.tm_year, &time_tmp.tm_mon, &time_tmp.tm_mday) == 3) ||
			 (sscanf(string_time, "%04d-%02d-%04d", &time_tmp.tm_year, &time_tmp.tm_mon, &time_tmp.tm_mday) == 3)))
		{
			if ((time_tmp.tm_mday > 31) && (time_tmp.tm_year <= 31))
			{
				tmp = time_tmp.tm_year;
				time_tmp.tm_year = time_tmp.tm_mday;
				time_tmp.tm_mday = tmp;
			}

			if (time_tmp.tm_year < 80)
			{
				time_tmp.tm_year += 100;
			}
			else if (time_tmp.tm_year > 1900)
			{
				time_tmp.tm_year -= 1900;
			}
			else
			{
				/* Do Nothing */
			}

			time_tmp.tm_mon -= 1;
		}
		else if ((sscanf(string_time, "%04d/%02d/%04d %2d:%02d:%02d", &time_tmp.tm_year, &time_tmp.tm_mon, &time_tmp.tm_mday, &time_tmp.tm_hour, &time_tmp.tm_min, &time_tmp.tm_sec) == 6) ||
			     (sscanf(string_time, "%04d/%02d/%04d %2d:%02d", &time_tmp.tm_year, &time_tmp.tm_mon, &time_tmp.tm_mday, &time_tmp.tm_hour, &time_tmp.tm_min) == 5) ||
		         (sscanf(string_time, "%04d-%02d-%04d %2d:%02d:%02d", &time_tmp.tm_year, &time_tmp.tm_mon, &time_tmp.tm_mday, &time_tmp.tm_hour, &time_tmp.tm_min, &time_tmp.tm_sec) == 6) ||
			     (sscanf(string_time, "%04d-%02d-%04d %2d:%02d", &time_tmp.tm_year, &time_tmp.tm_mon, &time_tmp.tm_mday, &time_tmp.tm_hour, &time_tmp.tm_min) == 5) ||
				 ((string_time[8] == 0x20) &&
				 (sscanf(string_time, "%04d%02d%02d %1d%02d%02d", &time_tmp.tm_year, &time_tmp.tm_mon, &time_tmp.tm_mday, &time_tmp.tm_hour, &time_tmp.tm_min, &time_tmp.tm_sec) == 6) &&
				 (time_tmp.tm_hour <= 9)))
		{
			if ((time_tmp.tm_mday > 31) && (time_tmp.tm_year <= 31))
			{
				tmp = time_tmp.tm_year;
				time_tmp.tm_year = time_tmp.tm_mday;
				time_tmp.tm_mday = tmp;
			}

			if (time_tmp.tm_year < 80)
			{
				time_tmp.tm_year += 100;
			}
			else if (time_tmp.tm_year > 1900)
			{
				time_tmp.tm_year -= 1900;
			}
			else
			{
				/* Do Nothing */
			}

			time_tmp.tm_mon -= 1;
		}
		else if (sscanf(string_time, "%04d%02d%02d%02d%02d%02d", &time_tmp.tm_year, &time_tmp.tm_mon, &time_tmp.tm_mday, &time_tmp.tm_hour, &time_tmp.tm_min, &time_tmp.tm_sec) == 6)
		{
			if ((time_tmp.tm_mday > 31) && (time_tmp.tm_year <= 31))
			{
				tmp = time_tmp.tm_year;
				time_tmp.tm_year = time_tmp.tm_mday;
				time_tmp.tm_mday = tmp;
			}

			if (time_tmp.tm_year > 1900)
			{
				time_tmp.tm_year -= 1900;
			}

			time_tmp.tm_mon -= 1;
		}
		else
		{
			logmessage(WARNING, "Unknown time format\n");
		}

		logmessage(DEBUG, "%s: %02d.%02d.%04d %02d:%02d:%02d\n", string_time, time_tmp.tm_mday, time_tmp.tm_mon + 1, time_tmp.tm_year + 1900, time_tmp.tm_hour, time_tmp.tm_min, time_tmp.tm_sec);
	}
	
	if ((time_tmp.tm_hour >= 0) && (time_tmp.tm_hour <= 23) &&
		(time_tmp.tm_min >= 0) && (time_tmp.tm_min <= 59) &&
		(time_tmp.tm_sec >= 0) && (time_tmp.tm_sec <= 59) &&
		(time_tmp.tm_mday >= 1) && (time_tmp.tm_mday <= 31) &&
		(time_tmp.tm_year >= 80) && (time_tmp.tm_year <= 200))
	{
		return(mktime(&time_tmp));
	}
	else
	{
		logmessage(WARNING, "Could not convert time stamp to valid time (%s)\n", string_time);

		return(time(NULL));
	}
}

void create_time(const time_t* input_time, char* output_time)
{
	struct tm* time_tmp;

	if (output_time != NULL)
	{
		time_tmp = localtime(input_time);

		if (time_tmp != NULL)
		{
			(void)snprintf(output_time, 15, "%04d%02d%02d%02d%02d%02d", time_tmp->tm_year + 1900, time_tmp->tm_mon + 1, time_tmp->tm_mday, time_tmp->tm_hour, time_tmp->tm_min, time_tmp->tm_sec);

			output_time[14] = 0x00;
		}
	}
}

size_t _strnlen_s(const char* string, size_t max)
{
	size_t count = 0;

	if ((string == NULL) || (max == 0) || (max > 65536))
	{
		return(0);
	}

	while (*string && max)
	{
		count++;
		max--;
		string++;
	}

	return(count);
}

int case_insensitive_compare(const char* input_string, const char* compare_string,const size_t max_length)
{
	char truncated_input_string[4096];
	char upper_case_compare_string[4096];

	size_t char_pos;
	size_t input_len;
	size_t compare_len;

	input_len = _strnlen_s(input_string, max_length);
	compare_len = _strnlen_s(compare_string, max_length);
	
	if (compare_len != input_len)
	{
		return 0;
	}
		
	if (compare_len >= sizeof(upper_case_compare_string))
	{
		compare_len = sizeof(upper_case_compare_string) - 1;
	}
		
	strncpy(truncated_input_string,input_string,compare_len);
	truncated_input_string[compare_len] = 0x00;
	
	strncpy(upper_case_compare_string,compare_string,compare_len);
	upper_case_compare_string[compare_len] = 0x00;
	
	for (char_pos=0;char_pos<compare_len;char_pos++)
	{
		truncated_input_string[char_pos] = toupper(truncated_input_string[char_pos]);
		upper_case_compare_string[char_pos] = toupper(upper_case_compare_string[char_pos]);
	}

	if (strncmp(truncated_input_string,upper_case_compare_string,compare_len) == 0)
	{
		return 1;
	}
	else
	{
		return 0;
	}
}

void set_bit(unsigned char* byte,const unsigned int bit)
{
#ifdef _SAFE_MEMORY
	if (byte != NULL)
#endif
	{
		*byte |= 1 << bit;
	}
}

void clear_bit(unsigned char* byte,const unsigned int bit)
{
#ifdef _SAFE_MEMORY
	if (byte != NULL)
#endif
	{
		*byte &= 0xff - (1 << bit);
	}
}

int read_bit(const unsigned char* byte,const unsigned int bit)
{
#ifdef _SAFE_MEMORY
	if (byte == NULL)
	{
		return(0);
	}
#endif
	return (int)((*byte >> bit) & 0x01);
}

void read_line(const char* in,char* out[],const unsigned int max_number_of_arguments,const unsigned int max_argument_size,unsigned int* arguments,const unsigned int separator, const char custom)
{
	unsigned int start_position;
	unsigned int char_pos = 0;
	unsigned int argument;
	unsigned int string_length;
	unsigned int character;
	unsigned int in_length = (unsigned int)strnlen_s(in,65536);
	
	for (argument=0;argument<max_number_of_arguments;argument++)
	{
		out[argument][0] = '\0';
	}
		
	argument = 0;

	while ((char_pos < in_length) && (argument < max_number_of_arguments))
	{
		while ((char_pos < in_length) && ((((separator & SEPARATOR_SPACE) > 0) && (in[char_pos] == 0x20)) || 
			                              (((separator & SEPARATOR_TAB) > 0) && (in[char_pos] == 0x09)) || 
			                              (((separator & SEPARATOR_COMMA) > 0) && (in[char_pos] == 0x2C)) ||
			                              (((separator & SEPARATOR_CUSTOM) > 0) && (in[char_pos] == custom))))
		{
			if (((separator & SEPARATOR_COMMA) > 0) && (in[char_pos] == 0x2C))
			{
				argument++;
			}
			char_pos++;
		}

		/* 0x3B = ; */

		if ((char_pos < in_length) && ((in[char_pos] == 0x3B) || (in[char_pos] == 0x0A) || (in[char_pos] == 0x0D) || (in[char_pos] == 0x00)))
		{
			char_pos = in_length;
		}
		
		if (char_pos < in_length)
		{						
			if (in[char_pos] == 0x22)			/* 0x22 = " */
			{
				start_position = char_pos + 1;
				char_pos++;

				while ((char_pos < in_length) && (in[char_pos] != 0x22) && (in[char_pos] != 0x0A) && (in[char_pos] != 0x0D) && (in[char_pos] != 0x00))
				{
					char_pos++;
				}

				string_length = char_pos - start_position;
				char_pos++;
			}
			else
			{
				start_position = char_pos;
				char_pos++;

				while ((char_pos < in_length) && (((separator & SEPARATOR_SPACE) == 0) || (in[char_pos] != 0x20)) && 
					                             (((separator & SEPARATOR_COMMA) == 0) || (in[char_pos] != 0x2C)) && 
					                             (((separator & SEPARATOR_TAB) == 0) || (in[char_pos] != 0x09)) && 
					                             (((separator & SEPARATOR_CUSTOM) == 0) || (in[char_pos] != custom)) && 
					                             (in[char_pos] != 0x0A) && (in[char_pos] != 0x0D) && (in[char_pos] != 0x00))
				{
					char_pos++;
				}

				string_length = char_pos - start_position;
			}
			
			if ((char_pos < in_length) && (in[char_pos] == 0x2C) && ((separator & SEPARATOR_COMMA) > 0))
			{
				char_pos++;
			}
			
			if (string_length > max_argument_size - 1)
			{
				string_length = max_argument_size - 1;
			}
			
			if (string_length > 0)
			{
				for (character=0;character<string_length;character++)
				{
					out[argument][character] = in[start_position+character];
				}
			}
					
			out[argument][string_length] = 0x00;
			
			argument++;
		}
	}

	*arguments = argument;
}

#define MAX_VALUE_LENGTH 25

bool determine_range(const char* input, const size_t length, unsigned int* values, unsigned int number_of_values)
{
	size_t index;
	size_t stripped_index = 0;
	size_t value_length;
	const char* comma_position = NULL;
	const char* dash_position = NULL;
	char* input_copy = NULL;
	char value[MAX_VALUE_LENGTH+1];
	unsigned int value_index;
	unsigned int value_index_2;
	unsigned int temp;
	unsigned int value_index_counter;

	if ((input == NULL) || (values == NULL) || (number_of_values == 0))
	{
		return(false);
	}

	input_copy = (char*)malloc((length+1)*sizeof(char));
	if (input_copy == NULL)
	{
		return(false);
	}

	/* Strip out anything that isn't 0123456789,- */

	stripped_index = 0;
	for (index = 0; index < length; index++)
	{
		if (((input[index] >= 0x30) && (input[index] <= 0x39)) || (input[index] == 0x2C) || (input[index] == 0x2D))
		{
			input_copy[stripped_index] = input[index];
			stripped_index++;
		}
	}

	if (stripped_index < length)
	{
		input_copy[stripped_index] = 0x00;
	}

	input_copy[length] = 0x00;

	for (value_index_counter = 0; value_index_counter < number_of_values; value_index_counter++)
	{
		values[value_index_counter] = 0;
	}

	index = 0;

	while (index < stripped_index)
	{
		comma_position = strchr(&input_copy[index], ',');

		if (comma_position == NULL)
		{
			value_length = strnlen_s(&input_copy[index],MAX_VALUE_LENGTH);
		}
		else
		{
			value_length = comma_position - &input_copy[index];
		}

		if (value_length > MAX_VALUE_LENGTH)
		{
			value_length = MAX_VALUE_LENGTH;
		}

		strncpy(value, &input_copy[index], value_length);
		value[value_length] = 0x00;

		dash_position = strchr(value, '-');

		if (dash_position != NULL)
		{
			if (sscanf(value, "%25u-%25u", &value_index, &value_index_2) == 2)
			{
				if (value_index_2 < value_index)
				{
					temp = value_index_2;
					value_index_2 = value_index;
					value_index = temp;
				}

				if (value_index > number_of_values)
				{
					value_index = number_of_values;
				}

				if (value_index_2 > number_of_values)
				{
					value_index_2 = number_of_values;
				}

				if ((value_index > 0) && (value_index_2 > 0))
				{
					for (value_index_counter = value_index; value_index_counter <= value_index_2; value_index_counter++)
					{
						values[value_index_counter - 1] = 1;
					}
				}
			}
			else if (sscanf(value, "-%25u", &value_index) == 1)
			{
				if (value_index > number_of_values)
				{
					value_index = number_of_values;
				}

				if (value_index > 0)
				{
					for (value_index_counter = 1; value_index_counter <= value_index; value_index_counter++)
					{
						values[value_index_counter - 1] = 1;
					}
				}
			}
			else
			{
				/* Do Nothing */
			}
		}
		else
		{
			if (sscanf(value, "%25u", &value_index) == 1)
			{
				if ((value_index > 0) && (value_index < number_of_values))
				{
					values[value_index-1] = 1;
				}
			}
		}

		index += comma_position - &input_copy[index] + 1;
	}

	free(input_copy);

	return(true);
}

#ifdef _WIN32
#include <windows.h>
#endif

void logmessage(const unsigned int level, const char* format, ... )
{
    char output[1000];
    
    va_list arguments;
    va_start(arguments,format);
    
	(void)vsnprintf(output, sizeof(output), format, arguments);
	output[999] = 0x00;
    
    if (debug_level >= level)
    {
#ifdef __WXMSW__
        wxString message;
		
		if (level == 1)	/* FATAL */
		{
			message.Printf(_T("%s"), output);
			wxQuietMessageBox(message, _T("FATAL ERROR from catoolRT"), wxICON_ERROR | wxOK);
			
			DebugBreak();

			exit(EXIT_FAILURE);
		}
		else if (level <= 2)	/* MSG_ERROR */
		{
			message.Printf(_T("%s"), output);
			wxQuietMessageBox(message, _T("ERROR from catoolRT"), wxICON_ERROR | wxOK);
		}
		else
		{
			wxLogMessage(_T("%s"), output);
		}
#else
        printf("%s",output);

		if (level == 1) /* FATAL */
		{
#ifdef _WIN32
			DebugBreak();
#endif
			exit(EXIT_FAILURE);
		}
#endif
    }
    
    va_end(arguments);
}
