/* Combustion Analysis Tool (CAT)
   www.catool.org
   
   Filename: sdf.c

   Purpose:  Provide common programming interface for processing acquired engine data
  
   Author:   Ben Brown
   Version:  1.2
   Date:     02.03.16

   Revision: GPL Release

   Copyright (C) Xarin Limited, 2000-2024

      This program is free software: you can redistribute it and/or modify
      it under the terms of version 2 of the GNU General Public License as
      published by the Free Software Foundation.

      This program is distributed in the hope that it will be useful,
      but WITHOUT ANY WARRANTY; without even the implied warranty of
      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
      GNU General Public License for more details.

      You should have received a copy of the GNU General Public License
      along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* For Microsoft Visual C++ */
#ifdef _MSC_VER
#include "stdafx.h"
#endif

#include <stdio.h>
#include <string.h>
#include <time.h>

#if !defined(__cplusplus)
typedef int bool;
#define true 1
#define false 0
#endif

extern unsigned int debug_level;

#include "cat.h"

size_t fwrite_swap_short(short data, FILE* file_handle)
{
	swap_endian_2(&data);

	return(fwrite(&data, 2, 1, file_handle));
}

size_t fwrite_swap_long(long data, FILE* file_handle)
{
	swap_endian_4(&data);

	return(fwrite(&data, 4, 1, file_handle));
}

size_t fwrite_swap_float(float data, FILE* file_handle)
{
	swap_endian_4(&data);

	return(fwrite(&data, 4, 1, file_handle));
}

size_t fwrite_swap_double(double data, FILE* file_handle)
{
	swap_endian_8(&data);

	return(fwrite(&data, 8, 1, file_handle));
}

size_t fwrite_single_char(char data, FILE* file_handle)
{
	return(fwrite(&data, 1, 1, file_handle));
}

void sdf_write_file_hdr(FileData* file, const short num_of_comments, const short comment_size, FILE* output)
{
	char applicVer[8];
	short num_of_DATA_HDR_record = 1;
	short num_of_VECTOR_record;
	short num_of_CHANNEL_record;
	short num_of_UNIQUE_record = 0;
	short num_of_SCAN_STRUCT_record = 0;
	short num_of_XDATA_record = 0;
	short num_of_SCAN_BIG_record = 0;
	short num_of_COMMENT_record = num_of_comments;
	long offset_of_DATA_HDR_record;
	long offset_of_VECTOR_record;
	long offset_of_CHANNEL_record;
	long offset_of_YDATA_record;
	long offset_of_COMMENT_record;
	short number_of_channels;
	short channel;
	struct tm* file_time = NULL;

	if (output == NULL)
	{
		return;
	}

	if (file == NULL)
	{
		return;
	}

	number_of_channels = 0;
	for (channel = 0; channel < (short)file->number_of_channels; channel++)
	{
		if ((file->channel_data[channel].abscissa.type == ABSCISSA_TIME) && (file->channel_data[channel].loaded == true))
		{
			number_of_channels++;

			Set_Raw_Channel_Min_Max(file, channel);
		}
	}

	num_of_VECTOR_record = number_of_channels;
	num_of_CHANNEL_record = number_of_channels;

	fwrite_single_char(0x42, output);
	fwrite_single_char(0x00, output);

	file_time = localtime(&file->creation_time);
	
	fwrite_swap_short(10, output);
	fwrite_swap_long(80, output);
	fwrite_swap_short(3, output);
	fwrite_swap_short(-99, output);
	fwrite_swap_short(1900 + file_time->tm_year, output);
	fwrite_swap_short((file_time->tm_mon+1)*100 + file_time->tm_mday, output);
	fwrite_swap_short(file_time->tm_hour*100 + file_time->tm_min, output);
	memset(applicVer, 0, 8);
	fwrite(applicVer,8,1,output);
	fwrite_swap_short(num_of_DATA_HDR_record, output);
	fwrite_swap_short(num_of_VECTOR_record, output);
	fwrite_swap_short(num_of_CHANNEL_record, output);
	fwrite_swap_short(num_of_UNIQUE_record, output);
	fwrite_swap_short(num_of_SCAN_STRUCT_record, output);
	fwrite_swap_short(num_of_XDATA_record, output);

	offset_of_DATA_HDR_record = 2 + 80 + 156;
	offset_of_VECTOR_record = offset_of_DATA_HDR_record + 148;
	offset_of_CHANNEL_record = offset_of_VECTOR_record + (18 * number_of_channels);
	offset_of_COMMENT_record = offset_of_CHANNEL_record + (212 * number_of_channels);
	offset_of_YDATA_record = offset_of_COMMENT_record + comment_size;

	fwrite_swap_long(offset_of_DATA_HDR_record, output);
	fwrite_swap_long(offset_of_VECTOR_record, output);
	fwrite_swap_long(offset_of_CHANNEL_record, output);

	fwrite_swap_long(-1, output);
	fwrite_swap_long(-1, output);
	fwrite_swap_long(-1, output);

	fwrite_swap_long(offset_of_YDATA_record, output);

	fwrite_swap_short(num_of_SCAN_BIG_record, output);
	fwrite_swap_short(num_of_COMMENT_record, output);
	fwrite_swap_long(-1, output);
	fwrite_swap_long(offset_of_COMMENT_record, output);		/* Spec says this is offset_of_next_SDF_FILE */

	fwrite_swap_long(-1, output);							/* But spec is short 4 bytes so guess this is it */
}

void sdf_write_meas_hdr(FILE* output)
{
	char measTitle[60];

	if (output == NULL)
	{
		return;
	}

	fwrite_swap_short(11, output);
	fwrite_swap_long(156, output);
	fwrite_swap_long(-1, output);
	fwrite_swap_float(0.0f, output);
	fwrite_swap_float(0.0f, output);
	fwrite_swap_long(0, output);
	fwrite_swap_short(0, output);
	fwrite_swap_short(0, output);
	fwrite_swap_short(0, output);
	fwrite_swap_short(0, output);
	fwrite_swap_long(0, output);
	fwrite_swap_float(0.0f, output);
	memset(measTitle, 0, 60);
	fwrite(measTitle,60,1, output);
	fwrite_swap_float(0.0f, output);

	fwrite_swap_double(0.0, output);
	fwrite_swap_double(0.0, output);
	fwrite_swap_double(0.0, output);
	fwrite_swap_short(-99, output);
	fwrite_swap_short(1, output);
	fwrite_swap_short(-99, output);
	fwrite_swap_double(0.0, output);
	fwrite_swap_long(0, output);
	fwrite_swap_long(0, output);
	fwrite_swap_double(0.0, output);
}

void sdf_write_unit(const char* label, FILE* output)
{
	if (output == NULL)
	{
		return;
	}

	fwrite(label, 10, 1, output);
	fwrite_swap_float(1.0f, output);
	fwrite_single_char(0, output);
	fwrite_single_char(0, output);
	fwrite_single_char(2, output);
	fwrite_single_char(0, output);
	fwrite_single_char(0, output);
	fwrite_single_char(0, output);
	fwrite_single_char(0, output);
	fwrite_single_char(0, output);
}

void sdf_write_data_hdr(const FileData* file, FILE* output)
{
	char dataTitle[16];
	long num_of_points;
	double abscissa_firstX = 0.0;
	double abscissa_deltaX;
	short total_rows = 0;
	short total_cols = 1;
	unsigned int resolution;
	unsigned int channel;

	if (file == NULL)
	{
		return;
	}

	if (output == NULL)
	{
		return;
	}

	num_of_points = 0;
	resolution = 0;

	for (channel = 0; channel < file->number_of_channels; channel++)
	{
		if ((file->channel_data[channel].abscissa.type == ABSCISSA_TIME) && (file->channel_data[channel].loaded == true))
		{
			if (file->channel_data[channel].abscissa.resolution[0] != resolution)
			{
				if (resolution == 0)
				{
					resolution = file->channel_data[channel].abscissa.resolution[0];

					logmessage(NOTICE, "%s: Set resolution to %f ms\n", file->channel_data[channel].name, (float)((double)resolution * file->time_resolution));
				}
				else
				{
					logmessage(MSG_ERROR, "%s: Resolution different to other channels (%u vs %u) - File will not be valid!\n", file->channel_data[channel].name, file->channel_data[channel].abscissa.resolution[0], resolution);
				}
			}

			if (file->channel_data[channel].samples_per_cycle != num_of_points)
			{
				if (num_of_points == 0)
				{
					num_of_points = file->channel_data[channel].samples_per_cycle;

					logmessage(NOTICE, "%s: Set number of data points to %d\n", file->channel_data[channel].name, num_of_points);
				}
				else
				{
					logmessage(MSG_ERROR, "%s: Number of data points different to other channels (%u vs %d) - File will not be valid!\n", file->channel_data[channel].name, file->channel_data[channel].samples_per_cycle, num_of_points);
				}
			}

			total_rows++;
		}
	}

	abscissa_deltaX = (double)resolution * file->time_resolution;		/* ms */

	fwrite_swap_short(12, output);
	fwrite_swap_long(148, output);
	fwrite_swap_long(-1, output);
	memset(dataTitle, 0, 16);
	strncpy(dataTitle, "", 16);
	fwrite(dataTitle, 16, 1, output);
	fwrite_swap_short(1, output);		/* Time domain */
	fwrite_swap_short(0, output);		/* Time */
	fwrite_swap_short((short)num_of_points, output);
	fwrite_swap_short((short)num_of_points - 1, output);
	fwrite_swap_float((float)abscissa_firstX, output);
	fwrite_swap_float((float)abscissa_deltaX, output);
	fwrite_swap_short(0, output);		/* linear */
	fwrite_swap_short(3, output);		/* float */
	fwrite_swap_short(0, output);
	fwrite_swap_short(1, output);		/* short */
	fwrite_swap_short(1, output);
	fwrite_swap_short(0, output);
	fwrite_swap_short(0, output);
	fwrite_swap_short(0, output);
	fwrite_swap_short(1, output);
	fwrite_swap_long(0, output);
	fwrite_swap_short(total_rows, output);
	fwrite_swap_short(total_cols, output);

	sdf_write_unit("s", output);

	fwrite_swap_short(0, output);		/*  use channel header record for y unit */

	sdf_write_unit("", output);

	fwrite_swap_double(abscissa_firstX, output);
	fwrite_swap_double(abscissa_deltaX, output);
	fwrite_swap_short(1, output);
	fwrite_swap_short(1, output);
	fwrite_swap_long(num_of_points, output);
	fwrite_swap_long(num_of_points - 1, output);
	fwrite_swap_short(1, output);
	fwrite_swap_short(0, output);
	fwrite_swap_short(0, output);
}

void sdf_write_vector_hdr(const unsigned int channel, FILE* output)
{
	if (output == NULL)
	{
		return;
	}

	fwrite_swap_short(13, output);
	fwrite_swap_long(18, output);
	fwrite_swap_long(-1, output);
	fwrite_swap_short(channel, output);
	fwrite_swap_short(-1, output);
	fwrite_swap_short(48, output);
	fwrite_swap_short(0, output);
}

void sdf_write_window(FILE* output)
{
	if (output == NULL)
	{
		return;
	}

	fwrite_swap_short(0, output);
	fwrite_swap_short(0, output);
	fwrite_swap_float(0.0f, output);
	fwrite_swap_float(0.0f, output);
	fwrite_swap_float(0.0f, output);
	fwrite_swap_float(1.0f, output);
	fwrite_swap_float(1.0f, output);
}

void sdf_write_channel_hdr(const ChannelData* channel_data, const short channel, FILE* output)
{
	char channelLabel[30];
	char moduleId[12];
	char serialNum[12];
	char intLabel[10];

	if (channel_data == NULL)
	{
		return;
	}

	double channelScale = (double)(channel_data->max - channel_data->min) / 65534.0;
	double channelOffset = (double)(channel_data->max + channel_data->min) / 2.0;

	if (output == NULL)
	{
		return;
	}

	memset(channelLabel, 0, 30);
	memset(moduleId, 0, 12);
	memset(serialNum, 0, 12);
	memset(intLabel, 0, 10);

	strncpy(channelLabel, channel_data->name, 30);
	channelLabel[29] = 0x00;
	strncpy(moduleId, "", 12);
	strncpy(serialNum, "", 12);
	strncpy(intLabel, "V", 10);

	fwrite_swap_short(14, output);
	fwrite_swap_long(212, output);
	fwrite_swap_long(-1, output);
	fwrite(channelLabel, 30, 1, output);
	fwrite(moduleId, 12, 1, output);
	fwrite(serialNum, 12, 1, output);

	sdf_write_window(output);

	fwrite_swap_short(0, output);
	fwrite_swap_float(0.0f, output);
	fwrite_swap_float(2.0f, output);
	fwrite_swap_short(3, output);
	fwrite_swap_short(1, output);
	fwrite_swap_short(1, output);
	fwrite_swap_short(0, output);
	fwrite(intLabel, 10, 1, output);

	sdf_write_unit(channel_data->units, output);

	fwrite_swap_float(1.0f, output);
	fwrite_swap_float(1E6f, output);

	fwrite_swap_short(0, output);
	fwrite_swap_short(0, output);
	fwrite_swap_short(0, output);
	fwrite_swap_double(channelScale, output);
	fwrite_swap_double(channelOffset, output);

	fwrite_swap_double(0.0, output);
	fwrite_swap_double(0.0, output);
	fwrite_swap_double(0.0, output);
	fwrite_swap_double(0.0, output);
	fwrite_swap_double(0.0, output);

	fwrite_swap_short(channel, output);
	fwrite_swap_short(0, output);
}

void sdf_write_ydata_hdr(const ChannelData* channel_data, FILE* output)
{
	long recordSize;
	unsigned int point;
	double double_value;
	short short_value;

	if (channel_data == NULL)
	{
		return;
	}

	if (output == NULL)
	{
		return;
	}

	double channelScale = (double)(channel_data->max - channel_data->min) / 65534.0;
	double channelOffset = (double)(channel_data->max + channel_data->min) / 2.0;

	recordSize = 2 + 4 + 2*channel_data->samples_per_cycle;

	fwrite_swap_short(17, output);
	fwrite_swap_long(recordSize, output);

	if (channel_data->data_d != NULL)
	{
		for (point = 0; point < channel_data->samples_per_cycle; point++)
		{
			double_value = (channel_data->data_d[point] - channelOffset) / channelScale;

			short_value = (short)double_value;

			fwrite_swap_short(short_value, output);
		}
	}
	else
	{
		for (point = 0; point < channel_data->samples_per_cycle; point++)
		{
			double_value = ((double)channel_data->data[point] - channelOffset) / channelScale;

			short_value = (short)double_value;

			fwrite_swap_short(short_value, output);
		}
	}

	logmessage(DEBUG, "%s: Wrote %u bytes\n", channel_data->name, channel_data->samples_per_cycle * 2);
}

short sdf_write_comment_hdr(const char* comment, const short scope_type, const short scope_info, FILE* output)
{
	short recordSize;
	short comment_bytes;

	if (comment == NULL)
	{
		return(-1);
	}

	comment_bytes = (short)strnlen_s(comment,SIZEOF_DESCRIPTION);

	recordSize = 22 + comment_bytes;

	if (output == NULL)
	{
		return(recordSize);
	}

	fwrite_swap_short(20, output);
	fwrite_swap_short(recordSize, output);

	fwrite_swap_long(-1, output);
	fwrite_swap_long(24, output);
	fwrite_swap_long(comment_bytes, output);
	fwrite_swap_short(0, output);
	fwrite_swap_short(scope_type, output);
	fwrite_swap_short(scope_info, output);

	fwrite(comment, comment_bytes, 1, output);

	return(recordSize);
}

#ifdef _CATOOL_UNICODE_
bool save_sdf_file(FileData* file, const wchar_t* filename)
#else
bool save_sdf_file(FileData* file, const char* filename)
#endif
{
	FILE* output = NULL;
	unsigned int sdf_channel;
	unsigned int channel;
	short comment_size;
	short num_of_comments;

	if (file == NULL)
	{
		return(false);
	}

	comment_size = sdf_write_comment_hdr(file->engine.name, 0, -1, NULL);
	comment_size += sdf_write_comment_hdr(file->comment, 1, 0, NULL);
	sdf_channel = 0;
	num_of_comments = 2;
	for (channel = 0; channel < file->number_of_channels; channel++)
	{
		if ((file->channel_data[channel].abscissa.type == ABSCISSA_TIME) && (file->channel_data[channel].loaded == true))
		{
			comment_size += sdf_write_comment_hdr(file->channel_data[channel].description, 3, sdf_channel, NULL);

			sdf_channel++;
			num_of_comments++;
		}
	}

	if (sdf_channel == 0)
	{
		return(false);
	}

#ifdef _CATOOL_UNICODE_
	output = _wfopen(filename, L"wb");
#else
	output = fopen(filename, "wb");
#endif

	if (output == NULL)
	{
		return(false);
	}

	sdf_write_file_hdr(file,num_of_comments,comment_size,output);

	sdf_write_meas_hdr(output);

	sdf_write_data_hdr(file, output);

	sdf_channel = 0;
	for (channel = 0; channel < file->number_of_channels; channel++)
	{
		if ((file->channel_data[channel].abscissa.type == ABSCISSA_TIME) && (file->channel_data[channel].loaded == true))
		{
			sdf_write_vector_hdr(sdf_channel, output);

			sdf_channel++;
		}
	}

	sdf_channel = 0;
	for (channel = 0; channel < file->number_of_channels; channel++)
	{
		if ((file->channel_data[channel].abscissa.type == ABSCISSA_TIME) && (file->channel_data[channel].loaded == true))
		{
			sdf_write_channel_hdr(&file->channel_data[channel], sdf_channel, output);

			sdf_channel++;
		}
	}

	sdf_write_comment_hdr(file->engine.name, 0, -1, output);
	sdf_write_comment_hdr(file->comment, 1, 0, output);

	sdf_channel = 0;
	for (channel = 0; channel < file->number_of_channels; channel++)
	{
		if ((file->channel_data[channel].abscissa.type == ABSCISSA_TIME) && (file->channel_data[channel].loaded == true))
		{
			sdf_write_comment_hdr(file->channel_data[channel].description, 3, sdf_channel, output);

			sdf_channel++;
		}
	}

	for (channel = 0; channel < file->number_of_channels; channel++)
	{
		if ((file->channel_data[channel].abscissa.type == ABSCISSA_TIME) && (file->channel_data[channel].loaded == true))
		{
			sdf_write_ydata_hdr(&file->channel_data[channel], output);
		}
	}

	fclose(output);

	return(true);
}
