/* Combustion Analysis Tool (CAT)
   www.catool.org

   Filename: plugin.h

   Purpose:  Provide common programming interface for processing acquired engine data

   Author:   Ben Brown
   Version:  1.2
   Date:     08.03.2017

   Revision: GPL Release

   Copyright (C) Xarin Limited, 2000-2024

      This program is free software: you can redistribute it and/or modify
      it under the terms of version 2 of the GNU General Public License as
      published by the Free Software Foundation.

      This program is distributed in the hope that it will be useful,
      but WITHOUT ANY WARRANTY; without even the implied warranty of
      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
      GNU General Public License for more details.

      You should have received a copy of the GNU General Public License
      along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef __PLUGIN_H__
#define __PLUGIN_H__

typedef struct PluginFunctions PluginFunctions;
typedef struct Plugin Plugin;

typedef bool(*InitialisePluginChannelsFunction)(FileData* file, AnalysisChannel* analysis, const unsigned int channel, const unsigned int analysis_type, const Geometry geometry, void** config);
typedef bool(*CompletePluginChannelsFunction)(FileData* file, AnalysisChannel* analysis, const unsigned int channel, const unsigned int analysis_type, const Geometry geometry, void** config);
typedef void(*ReturnPluginAnalysisFunction)(FileData* file, AnalysisChannel* analysis, const unsigned int cycle, const unsigned int channel, const unsigned int analysis_type, const Geometry geometry, void* config, const unsigned int stats_cycles, const unsigned int stats_start_cycle, const float engine_speed, const unsigned int number_of_cycles, const unsigned int number_of_buffer_cycles);
typedef bool(*SavePluginConfigFunction)(FILE* config_file, const unsigned int version);
typedef bool(*LoadPluginConfigFunction)(FILE* config_file, const unsigned int version, const unsigned int sizeof_customer_config);
typedef bool(*CalculateVolumeFunction)(const FileData* file, const unsigned int channel, float** volume);
typedef bool(*ValidateAnalysisRequestsFunction)(const FileData* file, AnalysisChannel* analysis);
typedef bool(*register_plugin_function)(const char* name, const char* filename, const char* copyright, const unsigned int features, AnalysisConf* analysis_configuration, const unsigned int number_of_channels, PluginFunctions functions, const size_t interface_key);

typedef bool(*RegisterAnalysisChannelsFunction)(register_plugin_function register_plugin, const char* filename, PluginFunctions functions, const size_t interface_key);
typedef void(*logmessageFunction)(const unsigned int level, const char* format, ...);

struct PluginFunctions {
	InitialisePluginChannelsFunction InitialisePluginChannels;
	CompletePluginChannelsFunction CompletePluginChannels;
	ReturnPluginAnalysisFunction ReturnPluginAnalysis;
	SavePluginConfigFunction SavePluginConfig;
	LoadPluginConfigFunction LoadPluginConfig;
	CalculateVolumeFunction CalculateVolume;
	ValidateAnalysisRequestsFunction ValidateAnalysisRequests;
	logmessageFunction logmessage;
};

struct Plugin {
	char name[SIZEOF_CHANNEL_NAME];
	char filename[256];
	char copyright[256];
	unsigned int features;
	unsigned int number_of_channels;

	PluginFunctions functions;

	Plugin* next;
};

void initialise_catool(void);
void uninitialise_catool(void);
bool register_plugin(const char* name,
	const char* filename,
	const char* copyright,
	const unsigned int features,
	AnalysisConf* analysis_configuration,
	const unsigned int number_of_channels,
	PluginFunctions functions,
	const size_t interface_key);
bool plugin_validate_analysis_requests(const FileData* file, AnalysisChannel* analysis);
bool plugin_calculate_volume(const FileData* file, const unsigned int channel, float** volume);
unsigned int get_number_of_plugins(void);
Plugin* get_plugins(void);
unsigned int get_number_of_plugin_channels(void);
void* get_analysis_function(const unsigned int analysis_type);
void* get_complete_function(const unsigned int analysis_type);
void* get_initialise_function(const unsigned int analysis_type);
const char* get_analysis_name(const unsigned int analysis_type);
const char* get_analysis_description(const unsigned int analysis_type);
const char* get_analysis_units(const unsigned int analysis_type);
unsigned int get_number_of_analysis_channels(void);
const char* get_analysis_unique_id(const unsigned int analysis_type);
int get_analysis_type_from_unique_id(const char* unique_id);
unsigned int get_analysis_output_abscissa(const unsigned int analysis_type);
unsigned int get_analysis_abscissa_units(const unsigned int analysis_type);
unsigned int get_analysis_channel_validity(const unsigned int analysis_type, const unsigned int channel_type);
unsigned int get_analysis_abscissa_type(const unsigned int analysis_type);
float get_analysis_min(const unsigned int analysis_type);
float get_analysis_max(const unsigned int analysis_type);
float get_analysis_error_min(const unsigned int analysis_type);
float get_analysis_error_max(const unsigned int analysis_type);
float get_analysis_error_stddev(const unsigned int analysis_type);
float get_analysis_warning_min(const unsigned int analysis_type);
float get_analysis_warning_max(const unsigned int analysis_type);
float get_analysis_warning_stddev(const unsigned int analysis_type);
unsigned long long int get_analysis_calcs(const unsigned int analysis_type);

#endif