/* Combustion Analysis Tool (CAT)
   www.catool.org

   Filename: plugin.c

   Purpose:  Provide common programming interface for processing acquired engine data

   Author:   Ben Brown
   Version:  1.2
   Date:     08.03.2017

   Revision: GPL Release

   Copyright (C) Xarin Limited, 2000-2024

      This program is free software: you can redistribute it and/or modify
      it under the terms of version 2 of the GNU General Public License as
      published by the Free Software Foundation.

      This program is distributed in the hope that it will be useful,
      but WITHOUT ANY WARRANTY; without even the implied warranty of
      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
      GNU General Public License for more details.

      You should have received a copy of the GNU General Public License
      along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <stdlib.h>
#include <string.h>

#include "cat.h"

extern unsigned int debug_level;

AnalysisChannelConfig analysis_config;
Plugin* plugins = NULL;

bool Copy_Analysis_Config(const AnalysisConf* src, AnalysisConf* dest)
{
	unsigned int channel;

	if ((src == NULL) || (dest == NULL))
	{
		return(false);
	}

	dest->number = src->number;
	dest->analysis_calcs = src->analysis_calcs;
	dest->abscissa_type = src->abscissa_type;
	dest->output_abscissa = src->output_abscissa;
	dest->abscissa_units = src->abscissa_units;

	strncpy(dest->name, src->name, SIZEOF_CHANNEL_NAME);
	strncpy(dest->units, src->units, SIZEOF_UNITS);
	strncpy(dest->description, src->description, SIZEOF_DESCRIPTION);

	dest->min = src->min;
	dest->max = src->max;

	for (channel = 0; channel < NUMBER_OF_CHANNEL_TYPES; channel++)
	{
		dest->channel_validity[channel] = src->channel_validity[channel];
	}

	dest->warning_min = src->warning_min;
	dest->warning_max = src->warning_max;
	dest->warning_stddev = src->warning_stddev;
	dest->error_min = src->error_min;
	dest->error_max = src->error_max;
	dest->error_stddev = src->error_stddev;

	dest->AnalysisFunction = src->AnalysisFunction;
	dest->InitialiseFunction = src->InitialiseFunction;
	dest->CompleteFunction = src->CompleteFunction;

	strncpy(dest->unique_id, src->unique_id, SIZEOF_CHANNEL_NAME);

	return(true);
}

void initialise_catool()
{
	unsigned int channel;
	Plugin* plugin = NULL;

	analysis_config.number = NUMBER_OF_ANALYSIS_CHANNELS;

	if (analysis_config.config != NULL)
	{
		free(analysis_config.config);
	}

	while (plugins != NULL)
	{
		plugin = plugins;
		plugins = plugins->next;

		free(plugin);
	}

	analysis_config.config = (AnalysisConf*)malloc(analysis_config.number * sizeof(AnalysisConf));

	if (analysis_config.config == NULL)
	{
		logmessage(FATAL, "Cannot allocate memory\n");
	}

	for (channel = 0; channel < analysis_config.number; channel++)
	{
		if (analysis_configuration[channel].number != channel)
		{
			logmessage(FATAL, "Analysis configuration mismatched (%d)\n", channel);
		}

		Copy_Analysis_Config(&analysis_configuration[channel], &analysis_config.config[channel]);
	}
}

void uninitialise_catool()
{
	Plugin* plugin = NULL;

	if (analysis_config.config != NULL)
	{
		free(analysis_config.config);
		analysis_config.config = NULL;
	}

	if (plugins != NULL)
	{
		while (plugins != NULL)
		{
			plugin = plugins;
			plugins = plugins->next;

			free(plugin);
		}
	}
}

unsigned int get_number_of_analysis_channels()
{
	if (analysis_config.config == NULL)
	{
		return(0);
	}

	return(analysis_config.number);
}

unsigned int get_number_of_plugins()
{
	Plugin* plugin = NULL;
	unsigned int number_of_plugins = 0;

	plugin = plugins;

	while (plugin != NULL)
	{
		number_of_plugins++;

		plugin = plugin->next;
	}

	return(number_of_plugins);
}

unsigned int get_number_of_plugin_channels()
{
	return(get_number_of_analysis_channels() - NUMBER_OF_ANALYSIS_CHANNELS);
}

bool plugin_calculate_volume(const FileData* file, const unsigned int channel, float** volume)
{
	Plugin* plugin = plugins;

	while (plugin != NULL)
	{
		if ((plugin->functions.CalculateVolume != NULL) && (plugin->functions.CalculateVolume(file, channel, volume) == true))
		{
			return(true);
		}
		
		plugin = plugin->next;
	}

	return(false);
}

bool plugin_validate_analysis_requests(const FileData* file, AnalysisChannel* analysis)
{
	Plugin* plugin = plugins;

	while (plugin != NULL)
	{
		if (plugin->functions.ValidateAnalysisRequests != NULL)
		{
			plugin->functions.ValidateAnalysisRequests(file, analysis);
		}

		plugin = plugin->next;
	}

	return(true);
}

bool register_plugin(const char* name,
	const char* filename,
	const char* copyright,
	const unsigned int features,
	AnalysisConf* analysis_configuration,
	const unsigned int number_of_channels,
	PluginFunctions functions,
	const size_t interface_key)
{
	unsigned int channel;
	AnalysisConf* new_config = NULL;
	AnalysisConf* old_config = NULL;
	Plugin* plugin = NULL;

	size_t calc_interface_key =
		   (0x0D77C4EC +
			sizeof(FileData) * 2 +
			sizeof(AnalysisConf) * 4 +
			sizeof(Geometry) * 8 +
			sizeof(PluginFunctions) * 16 +
			sizeof(register_plugin_function) * 32 +
			sizeof(AnalysisConfig) * 64);

	if (interface_key != calc_interface_key)
	{
		logmessage(WARNING, "Plugin incompatible with this version of catool\n");
		return(false);
	}

	if ((analysis_config.number > 0) && (analysis_config.config == NULL))
	{
		return(false);
	}

	if ((analysis_config.number == 0) && (analysis_config.config != NULL))
	{
		return(false);
	}

	new_config = (AnalysisConf*)malloc((analysis_config.number + number_of_channels) * sizeof(AnalysisConf));

	if (new_config == NULL)
	{
		logmessage(FATAL, "Cannot allocate memory\n");
	}

	for (channel = 0; channel < analysis_config.number; channel++)
	{
		if (analysis_config.config[channel].number != channel)
		{
			logmessage(FATAL, "Analysis configuration mismatched (%d)\n", channel);
		}

		Copy_Analysis_Config(&analysis_config.config[channel], &new_config[channel]);
	}

	for (channel = 0; channel < number_of_channels; channel++)
	{
		Copy_Analysis_Config(&analysis_configuration[channel], &new_config[analysis_config.number + channel]);
		new_config[analysis_config.number + channel].number = analysis_config.number + channel;
		analysis_configuration[channel].number = analysis_config.number + channel;
	}

	old_config = analysis_config.config;

	analysis_config.config = new_config;

	analysis_config.number += number_of_channels;

	free(old_config);

	if (plugins == NULL)
	{
		plugins = (Plugin*)malloc(sizeof(Plugin));

		if (plugins == NULL)
		{
			logmessage(FATAL, "Out of memory\n");
		}

		plugin = plugins;
	}
	else
	{
		plugin = plugins;

		while (plugin->next != NULL)
		{
			plugin = plugin->next;
		}

		plugin->next = (Plugin*)malloc(sizeof(Plugin));

		if (plugin->next == NULL)
		{
			logmessage(FATAL, "Out of memory\n");
		}

		plugin = plugin->next;
	}

	plugin->next = NULL;
	strncpy(plugin->filename, filename, 256);
	plugin->filename[255] = 0x00;
	strncpy(plugin->name, name, SIZEOF_CHANNEL_NAME);
	plugin->name[SIZEOF_CHANNEL_NAME-1] = 0x00;
	strncpy(plugin->copyright, copyright, SIZEOF_CHANNEL_NAME);
	plugin->copyright[SIZEOF_CHANNEL_NAME - 1] = 0x00;
	plugin->features = features;
	plugin->number_of_channels = number_of_channels;

	plugin->functions.InitialisePluginChannels = functions.InitialisePluginChannels;
	plugin->functions.CompletePluginChannels = functions.CompletePluginChannels;
	plugin->functions.ReturnPluginAnalysis = functions.ReturnPluginAnalysis;
	plugin->functions.SavePluginConfig = functions.SavePluginConfig;
	plugin->functions.LoadPluginConfig = functions.LoadPluginConfig;
	plugin->functions.CalculateVolume = functions.CalculateVolume;
	plugin->functions.ValidateAnalysisRequests = functions.ValidateAnalysisRequests;

	return(true);
}

Plugin* get_plugins()
{
	return(plugins);
}

void* get_analysis_function(const unsigned int analysis_type)
{
	if (analysis_config.config == NULL)
	{
		return(NULL);
	}

	if (analysis_type >= get_number_of_analysis_channels())
	{
		return(NULL);
	}

	return(analysis_config.config[analysis_type].AnalysisFunction);
}

void* get_initialise_function(const unsigned int analysis_type)
{
	if (analysis_config.config == NULL)
	{
		return(NULL);
	}

	if (analysis_type >= get_number_of_analysis_channels())
	{
		return(NULL);
	}

	return(analysis_config.config[analysis_type].InitialiseFunction);
}

void* get_complete_function(const unsigned int analysis_type)
{
	if (analysis_config.config == NULL)
	{
		return(NULL);
	}

	if (analysis_type >= get_number_of_analysis_channels())
	{
		return(NULL);
	}

	return(analysis_config.config[analysis_type].CompleteFunction);
}

const char* get_analysis_unique_id(const unsigned int analysis_type)
{
	if (analysis_config.config == NULL)
	{
		return(NULL);
	}

	if (analysis_type >= get_number_of_analysis_channels())
	{
		return(NULL);
	}

	return(analysis_config.config[analysis_type].unique_id);
}

int get_analysis_type_from_unique_id(const char* unique_id)
{
	unsigned int channel;
	int analysis_type = -1;

	if (analysis_config.config == NULL)
	{
		return(-1);
	}

	if (unique_id == NULL)
	{
		return(-1);
	}

	for (channel = 0; channel < analysis_config.number; channel++)
	{
		if (strncmp(analysis_config.config[channel].unique_id, unique_id, SIZEOF_CHANNEL_NAME) == 0)
		{
			analysis_type = analysis_config.config[channel].number;
		}
	}

	return(analysis_type);
}

const char* get_analysis_name(const unsigned int analysis_type)
{
	if (analysis_config.config == NULL)
	{
		return(NULL);
	}

	if (analysis_type >= get_number_of_analysis_channels())
	{
		return(NULL);
	}

	return(analysis_config.config[analysis_type].name);
}

const char* get_analysis_description(const unsigned int analysis_type)
{
	if (analysis_config.config == NULL)
	{
		return(NULL);
	}

	if (analysis_type >= get_number_of_analysis_channels())
	{
		return(NULL);
	}

	return(analysis_config.config[analysis_type].description);
}

const char* get_analysis_units(const unsigned int analysis_type)
{
	if (analysis_config.config == NULL)
	{
		return(NULL);
	}

	if (analysis_type >= get_number_of_analysis_channels())
	{
		return(NULL);
	}

	return(analysis_config.config[analysis_type].units);
}

unsigned int get_analysis_output_abscissa(const unsigned int analysis_type)
{
	if (analysis_config.config == NULL)
	{
		return(ABSCISSA_UNKNOWN);
	}

	if (analysis_type >= get_number_of_analysis_channels())
	{
		return(ABSCISSA_UNKNOWN);
	}

	return(analysis_config.config[analysis_type].output_abscissa);
}

unsigned int get_analysis_abscissa_units(const unsigned int analysis_type)
{
	if (analysis_config.config == NULL)
	{
		return(UNITS_NONE);
	}

	if (analysis_type >= get_number_of_analysis_channels())
	{
		return(UNITS_NONE);
	}

	return(analysis_config.config[analysis_type].abscissa_units);
}

unsigned int get_analysis_channel_validity(const unsigned int analysis_type, const unsigned int channel_type)
{
	if (analysis_config.config == NULL)
	{
		return(0);
	}

	if (analysis_type >= get_number_of_analysis_channels())
	{
		return(0);
	}

	if (channel_type >= NUMBER_OF_CHANNEL_TYPES)
	{
		return(0);
	}

	return(analysis_config.config[analysis_type].channel_validity[channel_type]);
}

unsigned int get_analysis_abscissa_type(const unsigned int analysis_type)
{
	if (analysis_config.config == NULL)
	{
		return(ABSCISSA_UNKNOWN);
	}

	if (analysis_type >= get_number_of_analysis_channels())
	{
		return(ABSCISSA_UNKNOWN);
	}

	return(analysis_config.config[analysis_type].abscissa_type);
}

float get_analysis_min(const unsigned int analysis_type)
{
	if (analysis_config.config == NULL)
	{
		return(-FLT_MAX);
	}

	if (analysis_type >= get_number_of_analysis_channels())
	{
		return(-FLT_MAX);
	}

	return(analysis_config.config[analysis_type].min);
}

float get_analysis_max(const unsigned int analysis_type)
{
	if (analysis_config.config == NULL)
	{
		return(FLT_MAX);
	}

	if (analysis_type >= get_number_of_analysis_channels())
	{
		return(FLT_MAX);
	}

	return(analysis_config.config[analysis_type].max);
}

float get_analysis_error_min(const unsigned int analysis_type)
{
	if (analysis_config.config == NULL)
	{
		return(-FLT_MAX);
	}

	if (analysis_type >= get_number_of_analysis_channels())
	{
		return(-FLT_MAX);
	}

	return(analysis_config.config[analysis_type].error_min);
}

float get_analysis_error_max(const unsigned int analysis_type)
{
	if (analysis_config.config == NULL)
	{
		return(FLT_MAX);
	}

	if (analysis_type >= get_number_of_analysis_channels())
	{
		return(FLT_MAX);
	}

	return(analysis_config.config[analysis_type].error_max);
}

float get_analysis_error_stddev(const unsigned int analysis_type)
{
	if (analysis_config.config == NULL)
	{
		return(FLT_MAX);
	}

	if (analysis_type >= get_number_of_analysis_channels())
	{
		return(FLT_MAX);
	}

	return(analysis_config.config[analysis_type].error_stddev);
}

float get_analysis_warning_min(const unsigned int analysis_type)
{
	if (analysis_config.config == NULL)
	{
		return(-FLT_MAX);
	}

	if (analysis_type >= get_number_of_analysis_channels())
	{
		return(-FLT_MAX);
	}

	return(analysis_config.config[analysis_type].warning_min);
}

float get_analysis_warning_max(const unsigned int analysis_type)
{
	if (analysis_config.config == NULL)
	{
		return(FLT_MAX);
	}

	if (analysis_type >= get_number_of_analysis_channels())
	{
		return(FLT_MAX);
	}

	return(analysis_config.config[analysis_type].warning_max);
}

float get_analysis_warning_stddev(const unsigned int analysis_type)
{
	if (analysis_config.config == NULL)
	{
		return(FLT_MAX);
	}

	if (analysis_type >= get_number_of_analysis_channels())
	{
		return(FLT_MAX);
	}

	return(analysis_config.config[analysis_type].warning_stddev);
}

unsigned long long int get_analysis_calcs(const unsigned int analysis_type)
{
	if (analysis_config.config == NULL)
	{
		return(ULLONG_MAX);
	}

	if (analysis_type >= get_number_of_analysis_channels())
	{
		return(ULLONG_MAX);
	}

	return(analysis_config.config[analysis_type].analysis_calcs);
}