/* catoolRT
   www.xarin.com
   
   Filename: maths.c

   Purpose:  Realtime combustion data acquisition and analysis
  
   Author:   Ben Brown
   Version:  11.2 beta
   Date:     16.03.17

   Copyright (C) Xarin Limited, 2000-2024
   All Rights Reserved.
*/

#include "cat.h"
#include <stdlib.h>
#include <string.h>

/* http://hbfs.wordpress.com/2010/03/09/radix-sort-on-floating-point-numbers/ */

void radix_sort_indices_split_array(const float* array, unsigned int* index, size_t array_size, size_t start, size_t count, const unsigned char sort_type)
{
	size_t zeros;
	unsigned int* warray = NULL;
	unsigned int* tmp = NULL;
	uint32_t radix;
	uint32_t* iarray = (uint32_t*)array;
	unsigned int swap;
	size_t count_0;
	size_t count_1;
	size_t j;
	size_t d;

	if ((array == NULL) || 
		(index == NULL) || 
		(count == 0))
	{
		return;
	}

	warray = (unsigned int*)malloc(count * sizeof(unsigned int));

	if (warray == NULL)
	{
		return;
	}

	size_t offset = array_size - count + start;

	for (j = 0; j < count; j++)
	{
		d = j + offset;

		if (d >= array_size)
		{
			d -= array_size;
		}

		index[j] = (unsigned int)d;
	}

	for (radix = 1; radix; radix <<= 1)
	{
		count_0 = 0;
		zeros = 0;

		for (j = 0; j < count; ++j)
		{
			zeros += !(iarray[index[j]] & radix);
		}

		count_1 = zeros;

		for (j = 0; j < count; ++j)
		{
			if (iarray[index[j]] & radix)
			{
				if (count_1 < count)
				{
					warray[count_1] = index[j];
					++count_1;
				}
			}
			else
			{
				if (count_0 < count)
				{
					warray[count_0] = index[j];
					++count_0;
				}
			}
		}

		tmp = warray;
		warray = index;
		index = tmp;
	}

	if (zeros < count)
	{
		if (zeros > 0)
		{
			memcpy(warray + (count - zeros), index, zeros * sizeof(unsigned int));

			for (d = 0, j = count - 1; j >= zeros; j--, d++)
			{
				warray[d] = index[j];
			}
		}

		memcpy(index, warray, count * sizeof(unsigned int));
	}

	free(warray);

	if (sort_type == SORT_AXIS_DECENDING)
	{
		for (d = 0, j = count - 1; d < (count / 2); d++, j--)
		{
			swap = index[d];
			index[d] = index[j];
			index[j] = swap;
		}
	}
}

void radix_sort_indices(const float* array, unsigned int* index, size_t count, const unsigned char sort_type)
{
	size_t zeros;
	unsigned int* warray = NULL;
	unsigned int* tmp = NULL;
	uint32_t radix;
	uint32_t* iarray = (uint32_t*)array;
	unsigned int swap;
	size_t count_0;
	size_t count_1;
	size_t j;
	size_t d;

	if ((array == NULL) || 
		(index == NULL) || 
		(count == 0))
	{
		return;
	}

	warray = (unsigned int*)malloc(count*sizeof(unsigned int));

	if (warray == NULL)
	{
		return;
	}

	for (j = 0; j < count; j++)
	{
		index[j] = (unsigned int)j;
	}

	for (radix = 1; radix; radix <<= 1)
	{
		count_0 = 0;
		zeros = 0;

		for (j = 0; j < count; ++j)
		{
			zeros += !(iarray[index[j]] & radix);
		}

		count_1 = zeros;

		for (j = 0; j < count; ++j)
		{
			if (iarray[index[j]] & radix)
			{
				if (count_1 < count)
				{
					warray[count_1] = index[j];
					++count_1;
				}
			}
			else
			{
				if (count_0 < count)
				{
					warray[count_0] = index[j];
					++count_0;
				}
			}
		}

		tmp = warray;
		warray = index;
		index = tmp;
	}

	if (zeros < count)
	{
		if (zeros > 0)
		{
			memcpy(&warray[count - zeros], index, zeros * sizeof(unsigned int));

			for (d = 0, j = count - 1; j >= zeros; j--, d++)
			{
				warray[d] = index[j];
			}
		}

		memcpy(index, warray, count * sizeof(unsigned int));
	}

	free(warray);

	if (sort_type == SORT_AXIS_DECENDING)
	{
		for (d = 0, j = count - 1; d < count / 2; d++, j--)
		{
			swap = index[d];
			index[d] = index[j];
			index[j] = swap;
		}
	}
}

void radix_sort(float* array, size_t count)
{
	size_t zeros = 0;
	float* warray;
	float* tmp;
	size_t j;
	size_t d;
	uint32_t radix;

	if ((array == NULL) || 
		(count == 0))
	{
		return;
	}

	warray = (float*)malloc(count*sizeof(float));

	for (radix = 1; radix; radix <<= 1)
	{
		uint32_t* iarray = (uint32_t*)array;

		size_t count0 = 0;
		size_t count1 = 0;

		zeros = 0;
		for (j = 0; j < count; ++j)
		{
			zeros += !(iarray[j] & radix);
		}

		count1 = zeros;

		for (j = 0; j < count; ++j)
		{
			if (iarray[j] & radix)
			{
				if (count1 < count)
				{
					warray[count1] = array[j];
					++count1;
				}
			}
			else
			{
				if (count0 < count)
				{
					warray[count0] = array[j];
					++count0;
				}
			}
		}

		tmp = warray;
		warray = array;
		array = tmp;
	}

	if (zeros < count)
	{
		if (zeros > 0)
		{
			memcpy(warray + (count - zeros), array, zeros * sizeof(float));

			for (d = 0, j = count - 1; j >= zeros; j--, d++)
			{
				warray[d] = array[j];
			}
		}

		memcpy(array, warray, count * sizeof(float));
	}

	free(warray);
}

float median_sort(float* data, const size_t count)
{
	float result;

	if ((data == NULL) || 
		(count == 0))
	{
		return(0.0f);
	}

	if (count == 1)
	{
		return(data[0]);
	}

	radix_sort(data, count);

	if ((count % 2) == 0)
	{
		result = (data[count / 2] + data[(count / 2) - 1]) / 2.0f;
	}
	else
	{
		result = data[count / 2];
	}

	return(result);
}

float lookup_2d(const float* x_axis, const float* y_axis, const unsigned int data_points, const float x)
{
	unsigned int column;

#ifdef _SAFE_MEMORY
	if ((x_axis == NULL) || 
		(y_axis == NULL) || 
		(data_points == 0))
	{
		return(0.0f);
	}
#endif

	if (x <= x_axis[0])
	{
		return(y_axis[0]);
	}
	else if (x >= x_axis[data_points - 1])
	{
		return(y_axis[data_points - 1]);
	}
	else
	{
		column = 1;
		while (x_axis[column] < x)
		{
			column++;
		}

		return(y_axis[column - 1] + (y_axis[column] - y_axis[column - 1]) * (x - x_axis[column - 1]) / (x_axis[column] - x_axis[column - 1]));
	}
}

float lookup_2dT(Table2D* table, const float x)
{
#ifdef _SAFE_MEMORY
	if (table == NULL)
	{
		return(0.0f);
	}
#endif

	return(lookup_2d(table->x, table->y, table->x_breakpoints, x));
}

float lookup_3dT(Table3D* table, const float x, const float y)
{
#ifdef _SAFE_MEMORY
	if (table == NULL)
	{
		return(0.0f);
	}
#endif

	return(lookup_3d(table->x, table->y, table->z, table->x_breakpoints, table->y_breakpoints, x, y));
}

#ifdef Z_AXIS
#error Z_AXIS macro is already defined
#else
#define Z_AXIS(ROW,COLUMN)   z_axis[((ROW)*x_data_points) + (COLUMN)]
#endif

float lookup_3d(const float* x_axis, const float* y_axis, const float* z_axis, const unsigned int x_data_points, const unsigned int y_data_points, const float x, const float y)
{
	unsigned int row;
	unsigned int column;
	float column_1_value;
	float column_2_value;
	float row_f;

#ifdef _SAFE_MEMORY
	if ((x_axis == NULL) || 
		(y_axis == NULL) || 
		(z_axis == NULL) || 
		(x_data_points == 0) || 
		(y_data_points == 0))
	{
		return(0.0f);
	}
#endif

	if (x <= x_axis[0])
	{
		column = 0;
	}
	else if (x >= x_axis[x_data_points - 1])
	{
		column = x_data_points - 1;
	}
	else
	{
		column = 1;
		while (x_axis[column] < x)
		{
			column++;
		}
	}

	if (y <= y_axis[0])
	{
		row = 0;
	}
	else if (y >= y_axis[y_data_points - 1])
	{
		row = y_data_points - 1;
	}
	else
	{
		row = 1;
		while (y_axis[row] < x)
		{
			row++;
		}
	}

	row_f = (y - y_axis[row - 1]) / (y_axis[row] - y_axis[row - 1]);

	column_1_value = Z_AXIS(row - 1,column - 1) + ((Z_AXIS(row,column - 1) - Z_AXIS(row - 1,column - 1)) * row_f);

	column_2_value = Z_AXIS(row - 1,column) + ((Z_AXIS(row,column) - Z_AXIS(row - 1,column)) * row_f);

	return(column_1_value + (column_2_value - column_1_value) * (x - x_axis[column - 1]) / (x_axis[column] - x_axis[column - 1]));
}

#ifdef TABLE_Z_AXIS
#error TABLE_Z_AXIS macro is already defined
#else
#define TABLE_Z_AXIS(ROW,COLUMN)   table_z_axis[((ROW)*x_data_points) + (COLUMN)]
#endif

void update_3d(Table3D* table, Table3D* table_samples, const float x, const float y, const float z)
{
#ifdef _SAFE_MEMORY
	if ((table->x == NULL) || 
		(table->y == NULL) || 
		(table->z == NULL) || 
		(table->x_breakpoints == 0) || 
		(table->y_breakpoints == 0))
	{
		return;
	}
#endif

	float* x_axis = table->x;
	float* y_axis = table->y;
	float* z_axis = table->z;
	float* table_z_axis = table_samples->z;

	unsigned int x_data_points = table->x_breakpoints;
	unsigned int y_data_points = table->y_breakpoints;
	float A;
	float B;
	float C;
	float D;
	unsigned int A_row;
	unsigned int A_column;
	unsigned int B_row;
	unsigned int B_column;
	unsigned int C_row;
	unsigned int C_column;
	unsigned int D_row;
	unsigned int D_column;
	float A_value;
	float B_value;
	float C_value;
	float D_value;
	float column_f;
	float row_f;

	/* Update nominally four cells
	
	          column-1   column
	        +---------+---------+
	  row-1 |    A    |    B    |    
	        +---------+---------+
	   row  |    C    |    D    | 
	        +---------+---------+
	*/

	/* Determine column/row numbers */

	if (x <= x_axis[0])
	{
		A_column = B_column = 0;
	}
	else if (x >= x_axis[x_data_points - 1])
	{
		A_column = B_column = x_data_points - 1;
	}
	else
	{
		B_column = 1;
		while (x_axis[B_column] < x)
		{
			B_column++;
		}

		A_column = B_column - 1;
	}

	D_column = B_column;
	C_column = A_column;

	if (y <= y_axis[0])
	{
		A_row = C_row = 0;
	}
	else if (y >= y_axis[y_data_points - 1])
	{
		A_row = C_row = y_data_points - 1;
	}
	else
	{
		A_row = 1;
		while (y_axis[A_row] < x)
		{
			A_row++;
		}

		C_row = A_row - 1;
	}

	B_row = A_row;
	D_row = C_row;

	/* Calculate position within cells */

	column_f = (x - x_axis[A_column]) / (x_axis[B_column] - x_axis[A_column]);
	row_f = (y - y_axis[C_row]) / (y_axis[C_row] - y_axis[A_row]);

	/* A_value + B_value + C_value + D_value = z */

	float snap_threshold = 0.1f;
	float snap_threshold_inv = 1.0f - snap_threshold;

	if ((column_f < snap_threshold) && (row_f < snap_threshold))
	{
		A_value = 1.0f;
		B_value = 0.0f;
		C_value = 0.0f;
		D_value = 0.0f;
	}
	else if ((column_f > snap_threshold_inv) && (row_f < snap_threshold))
	{
		A_value = 0.0f;
		B_value = 1.0f;
		C_value = 0.0f;
		D_value = 0.0f;
	}
	else if ((column_f < snap_threshold) && (row_f > snap_threshold_inv))
	{
		A_value = 0.0f;
		B_value = 0.0f;
		C_value = 1.0f;
		D_value = 0.0f;
	}
	else if ((column_f > snap_threshold_inv) && (row_f > snap_threshold_inv))
	{
		A_value = 0.0f;
		B_value = 0.0f;
		C_value = 0.0f;
		D_value = 1.0f;
	}
	else
	{
		A_value = (1.0f - column_f) * (1.0f - row_f);
		B_value = column_f * (1.0f - row_f);
		C_value = (1.0f - column_f) * row_f;
		D_value = column_f * row_f;
	}

	A_value *= z;
	B_value *= z;
	C_value *= z;
	D_value *= z;

	/* Update cells */

	A = Z_AXIS(A_row, A_column);
	B = Z_AXIS(B_row, B_column);
	C = Z_AXIS(C_row, C_column);
	D = Z_AXIS(D_row, D_column);

	Z_AXIS(A_row, A_column) = (A*TABLE_Z_AXIS(A_row, A_column) + A_value) / (TABLE_Z_AXIS(A_row, A_column) + 1);
	TABLE_Z_AXIS(A_row, A_column) = TABLE_Z_AXIS(A_row, A_column) + 1;

	if (B_column > A_column)
	{
		Z_AXIS(B_row, B_column) = (A*TABLE_Z_AXIS(B_row, B_column) + B_value) / (TABLE_Z_AXIS(B_row, B_column) + 1);
		TABLE_Z_AXIS(B_row, B_column) = TABLE_Z_AXIS(B_row, B_column) + 1;
	}

	if (C_row > A_row)
	{
		Z_AXIS(C_row, C_column) = (A*TABLE_Z_AXIS(C_row, C_column) + C_value) / (TABLE_Z_AXIS(C_row, C_column) + 1);
		TABLE_Z_AXIS(C_row, C_column) = TABLE_Z_AXIS(C_row, C_column) + 1;

		if (D_column > C_column)
		{
			Z_AXIS(D_row, D_column) = (A*TABLE_Z_AXIS(D_row, D_column) + D_value) / (TABLE_Z_AXIS(D_row, D_column) + 1);
			TABLE_Z_AXIS(D_row, D_column) = TABLE_Z_AXIS(D_row, D_column) + 1;
		}
	}
}

#undef Z_AXIS
#undef TABLE_Z_AXIS

float sumf(const float* array, const size_t array_size)
{
	float sum = 0.0f;
	size_t i;

	if ((array == NULL) ||
		(array_size == 0))
	{
		return(0.0f);
	}

	for (i = 0; i < array_size; i++)
	{
		sum += array[i];
	}

	return(sum);
}

float minf(const float* array, const size_t array_size)
{
	float min;
	size_t start;
	size_t i;

	if ((array == NULL) ||
		(array_size == 0))
	{
		return(0.0f);
	}

	if (array_size % 2 == 1)
	{
		min = array[0];
		start = 1;
	}
	else
	{
		if (array[0] < array[1])
		{
			min = array[0];
		}
		else
		{
			min = array[1];
		}
		start = 2;
	}

	for (i = start; i < array_size; i += 2)
	{
		if (array[i] < array[i + 1])
		{
			if (array[i] < min)
			{
				min = array[i];
			}
		}
		else
		{
			if (array[i + 1] < min)
			{
				min = array[i + 1];
			}
		}
	}

	return(min);
}

float maxf(const float* array, const size_t array_size)
{
	float max;
	size_t start;
	size_t i;

	if ((array == NULL) ||
		(array_size == 0))
	{
		return(0.0f);
	}

	if (array_size % 2 == 1)
	{
		max = array[0];
		start = 1;
	}
	else
	{
		if (array[0] < array[1])
		{
			max = array[1];
		}
		else
		{
			max = array[0];
		}
		start = 2;
	}

	for (i = start; i < array_size; i += 2)
	{
		if (array[i] < array[i + 1])
		{
			if (array[i + 1] > max)
			{
				max = array[i + 1];
			}
		}
		else
		{
			if (array[i] > max)
			{
				max = array[i];
			}
		}
	}

	return(max);
}

void minmaxf(const float* array, const size_t array_size, float* minimum, float* maximum)
{
	float min;
	float max;
	size_t start;
	size_t i;

	if ((array == NULL) || 
		(array_size == 0))
	{
		if (minimum != NULL)
		{
			*minimum = 0.0f;
		}

		if (maximum != NULL)
		{
			*maximum = 0.0f;
		}

		return;
	}

	if (array_size % 2 == 1)
	{
		min = array[0];
		max = array[0];
		start = 1;
	}
	else
	{
		if (array[0] < array[1])
		{
			min = array[0];
			max = array[1];
		}
		else
		{
			min = array[1];
			max = array[0];
		}
		start = 2;
	}

	for (i = start; i<array_size; i += 2)
	{
		if (array[i] < array[i + 1])
		{
			if (array[i] < min)
			{
				min = array[i];
			}

			if (array[i + 1] > max)
			{
				max = array[i + 1];
			}
		}
		else
		{
			if (array[i + 1] < min)
			{
				min = array[i + 1];
			}

			if (array[i] > max)
			{
				max = array[i];
			}
		}
	}

	if (minimum != NULL)
	{
		*minimum = min;
	}

	if (maximum != NULL)
	{
		*maximum = max;
	}
}

void minmax(const double* array, const size_t array_size, double* minimum, double* maximum, const size_t offset, const size_t skip)
{
	double min;
	double max;
	size_t start;
	size_t i;

	if ((array == NULL) || 
		(array_size == 0))
	{
		if (minimum != NULL)
		{
			*minimum = 0.0;
		}

		if (maximum != NULL)
		{
			*maximum = 0.0;
		}

		return;
	}

	if (array_size % 2 == 1)
	{
		min = array[skip * 0 + offset];
		max = array[skip * 0 + offset];
		start = 1;
	}
	else
	{
		if (array[skip * 0 + offset] < array[skip * 1 + offset])
		{
			min = array[skip * 0 + offset];
			max = array[skip * 1 + offset];
		}
		else
		{
			min = array[skip * 1 + offset];
			max = array[skip * 0 + offset];
		}
		start = 2;
	}

	for (i = start; i<array_size; i += 2)
	{
		if (array[skip * i + offset] < array[skip * (i + 1) + offset])
		{
			if (array[skip * i + offset] < min)
			{
				min = array[skip * i + offset];
			}

			if (array[skip * (i + 1) + offset] > max)
			{
				max = array[skip * (i + 1) + offset];
			}
		}
		else
		{
			if (array[skip * (i + 1) + offset] < min)
			{
				min = array[skip * (i + 1) + offset];
			}

			if (array[skip * i + offset] > max)
			{
				max = array[skip * i + offset];
			}
		}
	}

	if (minimum != NULL)
	{
		*minimum = min;
	}

	if (maximum != NULL)
	{
		*maximum = max;
	}
}

float calculate_mean(const float* data, const unsigned int number_of_samples)
{
	unsigned int data_pointer;
	float sum = 0.0f;

	if ((data == NULL) || 
		(number_of_samples == 0))
	{
		return(0.0f);
	}

	for (data_pointer = 0; data_pointer < number_of_samples; data_pointer++)
	{
		sum += data[data_pointer];
	}

	return(sum / (float)number_of_samples);
}

int calculate_first_order_least_squares(const unsigned int number_of_data_points, const float* x_array_data, const float* y_array_data, float* a0, float* a1)
{
	float x_data;
	float y_data;
	float xy = 0;
	float xx = 0;
	float x = 0;
	float y = 0;
	float determinate;
	unsigned int data_point;

#ifdef _SAFE_MEMORY
	if ((a0 == NULL) || 
		(a1 == NULL))
	{
		return(0);
	}
#endif

	for (data_point = 0; data_point<number_of_data_points; data_point++)
	{
		x_data = x_array_data[data_point];
		y_data = y_array_data[data_point];

		xy += x_data * y_data;
		xx += x_data * x_data;
		x += x_data;
		y += y_data;
	}

	determinate = ((float)number_of_data_points*xx) - (x * x);

	if (determinate < FLT_EPSILON)
	{
		*a0 = 0.0f;
		*a1 = 0.0f;

		return(0);
	}

	*a0 = 1.0f / determinate * ((xx*y) - (x * xy));
	*a1 = 1.0f / determinate * (((float)number_of_data_points * xy) - (x * y));

	return(1);
}

int calculate_second_order_least_squares(const unsigned int number_of_data_points, const float* x_array_data, const float* y_array_data, float* a0, float* a1, float* a2)
{
	float x_data;
	float y_data;
	float determinate;
	float A;
	float B;
	float C;
	float D;
	float E;
	float F;
	float G;
	float H;
	float I;
	float xy = 0.0f;
	float xx = 0.0f;
	float xxy = 0.0f;
	float xxx = 0.0f;
	float xxxx = 0.0f;
	float x = 0.0f;
	float y = 0.0f;
	unsigned int data_point;

#ifdef _SAFE_MEMORY
	if ((a0 == NULL) || 
		(a1 == NULL) || 
		(a2 == NULL))
	{
		return(0);
	}
#endif

	for (data_point = 0; data_point<number_of_data_points; data_point++)
	{
		x_data = x_array_data[data_point];
		y_data = y_array_data[data_point];

		x += x_data;
		y += y_data;
		xy += x_data * y_data;
		xx += x_data * x_data;
		xxy += x_data * x_data*y_data;
		xxx += x_data * x_data*x_data;
		xxxx += x_data * x_data*x_data*x_data;
	}

	determinate = (float)number_of_data_points * (xx*xxxx - xxx * xxx) - x * (x*xxxx - xx * xxx) + xx * (x*xxx - xx * xx);

	if (determinate < FLT_EPSILON)
	{
		*a0 = 0.0f;
		*a1 = 0.0f;
		*a2 = 0.0f;

		return(0);
	}

	A = xx * xxxx - xxx * xxx;
	B = x * xxxx - xx * xxx;
	C = x * xxx - xx * xx;
	D = x * xxxx - xx * xxx;
	E = ((float)number_of_data_points * xxxx) - (xx * xx);
	F = ((float)number_of_data_points * xxx) - (x * xx);
	G = x * xxx - xx * xx;
	H = ((float)number_of_data_points * xxx) - (x * xx);
	I = ((float)number_of_data_points * xx) - (x * x);

	*a0 = 1.0f / determinate * (A * y + B * xy + C * xxy);
	*a1 = 1.0f / determinate * (D * y + E * xy + F * xxy);
	*a2 = 1.0f / determinate * (G * y + H * xy + I * xxy);

	return(1);
}

float calculate_coefficient_of_determination(const unsigned int number_of_data_points, const float* data, const float* model)
{
	unsigned int data_point;
	float total_sum_of_squares = 0.0f;
	float residual_sum_of_squares = 0.0f;
	float mean = 0.0f;

#ifdef _SAFE_MEMORY
	if ((data == NULL) || 
		(model == NULL) || 
		(number_of_data_points == 0))
	{
		return(0.0f);
	}
#endif

	for (data_point = 0; data_point < number_of_data_points; data_point++)
	{
		mean += data[data_point];
	}
	mean /= (float)number_of_data_points;

	for (data_point = 0; data_point < number_of_data_points; data_point++)
	{
		total_sum_of_squares += (data[data_point] - mean) * (data[data_point] - mean);
		residual_sum_of_squares += (data[data_point] - model[data_point])*(data[data_point] - model[data_point]);
	}

	return(1.0f - residual_sum_of_squares / total_sum_of_squares);
}

/* Calculate Mean, Max, Min, Standard Deviation, Coefficient of Variation (CoV),Least Normalized Value (LNV), Range, Skewness and Kurtosis */

void calculate_group_stats(Analysis* analysis, const unsigned int cycle_to_analyse, const unsigned int cycles_to_analyse, const unsigned int stats_calcs, const AnalysisRqst request_threshold, const unsigned int min_res_ppr, const unsigned int number_of_strokes, const unsigned int stats_type)
{
	unsigned int group_channel;
	unsigned int group;
	unsigned int analysis_channel;
	unsigned int channel;
	unsigned int cycle;
	unsigned int crank_angle;
	unsigned int samples_per_cycle;
	unsigned int adjusted_crank_angle;
	float cycles_minus_one;
	Statistics* stats = NULL;
	Statistics* data_ptr = NULL;
	StatisticsArray* stats_array = NULL;
	float value;
	float n;
	float delta;
	float min;
	float max;
	float mean;
	int theta;
	unsigned int maximum_theta = (unsigned int)((number_of_strokes * 180)*((float)min_res_ppr / 360.0f));
	float angle;
	unsigned int count;
	float ratio_modifier = 1.0f;

	if ((analysis == NULL) || 
		(cycles_to_analyse == 0) || 
		(min_res_ppr == 0) || 
		((number_of_strokes != 2) && (number_of_strokes != 4)))
	{
		return;
	}

	if (stats_type == STATS_PERCENT)
	{
		ratio_modifier = 100.0f;
	}

	cycles_minus_one = (float)cycles_to_analyse - 1.0f;

	for (group = 0; group<analysis->number_of_groups; group++)
	{
		for (analysis_channel = 0; analysis_channel<get_number_of_analysis_channels(); analysis_channel++)
		{
			if (analysis_channel == ANGULAR_TORQUE_SUM)
			{
				/* Skip this analysis type as it is only calculated for last cylinder */
			}
			else if (analysis->channel[analysis->group[group].channels[0]].analysis_rqst[analysis_channel] >= request_threshold)
			{
				if (analysis->channel[analysis->group[group].channels[0]].results[analysis_channel].abscissa.type == ABSCISSA_CYCLE)
				{
					/* Calculate cycle stats - does not rely on any other stats calculations*/

					if ((stats_calcs & STATS_CYCLE_BIT) > 0)
					{
						stats = &analysis->group[group].cycle_statistics[analysis_channel];

						stats->sum = analysis->channel[analysis->group[group].channels[0]].results[analysis_channel].data[cycle_to_analyse];
						stats->min = analysis->channel[analysis->group[group].channels[0]].results[analysis_channel].data[cycle_to_analyse];
						stats->max = analysis->channel[analysis->group[group].channels[0]].results[analysis_channel].data[cycle_to_analyse];
						count = 1;

						for (group_channel = 1; group_channel<analysis->group[group].number_of_channels; group_channel++)
						{
							if (analysis->channel[analysis->group[group].channels[group_channel]].analysis_rqst[analysis_channel] >= request_threshold)
							{
								value = analysis->channel[analysis->group[group].channels[group_channel]].results[analysis_channel].data[cycle_to_analyse];

								stats->sum += value;

								if (value < stats->min)
								{
									stats->min = value;
								}
								else if (value > stats->max)
								{
									stats->max = value;
								}
								else
								{
									/* Do Nothing */
								}

								count++;
							}
						}

						stats->range = stats->max - stats->min;
						stats->mean = stats->sum / (float)count;

						stats->stddev = 0.0f;
						stats->skewness = 0.0f;
						stats->kurtosis = 0.0f;

						for (group_channel = 0; group_channel<analysis->group[group].number_of_channels; group_channel++)
						{
							if (analysis->channel[analysis->group[group].channels[group_channel]].analysis_rqst[analysis_channel] >= request_threshold)
							{
								value = analysis->channel[analysis->group[group].channels[group_channel]].results[analysis_channel].data[cycle_to_analyse];

								delta = value - stats->mean;

								stats->stddev += delta * delta;
								stats->skewness += delta * delta * delta;
								stats->kurtosis += delta * delta * delta * delta;
							}
						}

						n = (float)(count - 1);

						if (n > FLT_EPSILON)
						{
							stats->stddev = sqrtf(stats->stddev / n);

							if (stats->stddev > FLT_EPSILON)
							{
								stats->skewness = stats->skewness / n / (stats->stddev * stats->stddev * stats->stddev);
								stats->kurtosis = (stats->kurtosis / n / (stats->stddev * stats->stddev * stats->stddev * stats->stddev)) - 3.0f;
							}
						}

						if ((stats->mean > FLT_EPSILON) || (stats->mean < -FLT_EPSILON))
						{
							stats->lnv = stats->min / stats->mean * ratio_modifier;
							stats->cov = stats->stddev / stats->mean * ratio_modifier;
						}
						else
						{
							stats->lnv = 1.0f * ratio_modifier;
							stats->cov = 0.0f;
						}

						/* Store Cycle Sum for statistical analysis */

						analysis->group[group].cycle_sum_data[analysis_channel].sum[cycle_to_analyse] = stats->sum;

						/* Calculate Cycle Sum statistics */

						stats = &analysis->group[group].cycle_sum_statistics[analysis_channel];

						stats->sum = 0.0f;
						stats->min = analysis->group[group].cycle_sum_data[analysis_channel].sum[0];
						stats->max = analysis->group[group].cycle_sum_data[analysis_channel].sum[0];

						for (cycle = 0; cycle<cycles_to_analyse; cycle++)
						{
							value = analysis->group[group].cycle_sum_data[analysis_channel].sum[cycle];

							stats->sum += value;

							if (value > stats->max)
							{
								stats->max = value;
							}
							else if (value < stats->min)
							{
								stats->min = value;
							}
							else
							{
								/* Do Nothing */
							}
						}

						stats->range = stats->max - stats->min;
						stats->mean = stats->sum / (float)cycles_to_analyse;

						stats->stddev = 0.0f;
						stats->skewness = 0.0f;
						stats->kurtosis = 0.0f;

						for (cycle = 0; cycle<cycles_to_analyse; cycle++)
						{
							delta = analysis->group[group].cycle_sum_data[analysis_channel].sum[cycle] - stats->mean;

							stats->stddev += delta * delta;
							stats->skewness += delta * delta * delta;
							stats->kurtosis += delta * delta * delta * delta;
						}

						if (cycles_minus_one > FLT_EPSILON)
						{
							stats->stddev = sqrtf(stats->stddev / cycles_minus_one);

							if (stats->stddev > FLT_EPSILON)
							{
								stats->skewness = stats->skewness / cycles_minus_one / (stats->stddev * stats->stddev * stats->stddev);
								stats->kurtosis = (stats->kurtosis / cycles_minus_one / (stats->stddev * stats->stddev * stats->stddev * stats->stddev)) - 3.0f;
							}
						}

						if ((stats->mean > FLT_EPSILON) || (stats->mean < -FLT_EPSILON))
						{
							stats->lnv = stats->min / stats->mean * ratio_modifier;
							stats->cov = stats->stddev / stats->mean * ratio_modifier;
						}
						else
						{
							stats->lnv = 1.0f * ratio_modifier;
							stats->cov = 0.0f;
						}

						analysis->group[group].cycle_sum_data[analysis_channel].min[cycle_to_analyse] = stats->min;
						analysis->group[group].cycle_sum_data[analysis_channel].max[cycle_to_analyse] = stats->max;
						analysis->group[group].cycle_sum_data[analysis_channel].mean[cycle_to_analyse] = stats->mean;
						analysis->group[group].cycle_sum_data[analysis_channel].stddev[cycle_to_analyse] = stats->stddev;
					}

					/* Calculate engine stats (engine mean) (aka RMS) - requires cylinder stats to have been calculated */

					if ((stats_calcs & STATS_CYLINDER_MEAN_BIT) > 0)
					{
						stats = &analysis->group[group].engine_mean_statistics[analysis_channel];

						stats->mean = 0.0f;
						stats->min = analysis->channel[analysis->group[group].channels[0]].statistics[analysis_channel].min;
						stats->max = analysis->channel[analysis->group[group].channels[0]].statistics[analysis_channel].max;

						for (group_channel = 0; group_channel<analysis->group[group].number_of_channels; group_channel++)
						{
							data_ptr = &analysis->channel[analysis->group[group].channels[group_channel]].statistics[analysis_channel];

							stats->mean += data_ptr->mean;

							if (data_ptr->min < stats->min)
							{
								stats->min = data_ptr->min;
							}
							else if (data_ptr->max > stats->max)
							{
								stats->max = data_ptr->max;
							}
							else
							{
								/* Do Nothing */
							}
						}

						stats->range = stats->max - stats->min;
						stats->mean /= analysis->group[group].number_of_channels;

						stats->stddev = 0.0f;
						stats->skewness = 0.0f;
						stats->kurtosis = 0.0f;

						for (group_channel = 0; group_channel<analysis->group[group].number_of_channels; group_channel++)
						{
							channel = analysis->group[group].channels[group_channel];

							for (cycle = 0; cycle<cycles_to_analyse; cycle++)
							{
								value = analysis->channel[channel].results[analysis_channel].data[cycle];

								delta = value - stats->mean;

								stats->stddev += delta * delta;
								stats->skewness += delta * delta * delta;
								stats->kurtosis += delta * delta * delta * delta;
							}
						}

						n = (analysis->group[group].number_of_channels * (float)cycles_to_analyse) - 1.0f;

						if (n > FLT_EPSILON)
						{
							stats->stddev = sqrtf(stats->stddev / n);

							if (stats->stddev > FLT_EPSILON)
							{
								stats->skewness = stats->skewness / n / (stats->stddev * stats->stddev * stats->stddev);
								stats->kurtosis = (stats->kurtosis / n / (stats->stddev * stats->stddev * stats->stddev * stats->stddev)) - 3.0f;
							}
						}

						if ((stats->mean > FLT_EPSILON) || (stats->mean < -FLT_EPSILON))
						{
							stats->lnv = stats->min / stats->mean * ratio_modifier;
							stats->cov = stats->stddev / stats->mean * ratio_modifier;
						}
						else
						{
							stats->lnv = 1.0f * ratio_modifier;
							stats->cov = 0.0f;
						}
					}

					/* Calculate engine stats (cylinder mean) (aka Lumped) - requires cylinder stats to have been calculated */

					if ((stats_calcs & STATS_ENGINE_MEAN_BIT) > 0)
					{
						stats = &analysis->group[group].cylinder_mean_statistics[analysis_channel];

						stats->stddev = 0.0f;
						stats->skewness = 0.0f;
						stats->kurtosis = 0.0f;
						stats->mean = 0.0f;

						for (group_channel = 0; group_channel<analysis->group[group].number_of_channels; group_channel++)
						{
							channel = analysis->group[group].channels[group_channel];

							stats->mean += analysis->channel[channel].statistics[analysis_channel].mean;

							for (cycle = 0; cycle<cycles_to_analyse; cycle++)
							{
								value = analysis->channel[channel].results[analysis_channel].data[cycle];

								delta = value - analysis->channel[channel].statistics[analysis_channel].mean;

								stats->stddev += delta * delta;
								stats->skewness += delta * delta * delta;
								stats->kurtosis += delta * delta * delta * delta;
							}
						}

						stats->mean /= analysis->group[group].number_of_channels;

						/* Note: Could have used stats->mean = analysis->group[group].engine_mean_statistics[analysis_channel].mean and not
						recalculated in above loop but then we would need to have run the RMS calcs to do Lumped */

						n = (analysis->group[group].number_of_channels * (float)cycles_to_analyse) - 1.0f;

						if (n > FLT_EPSILON)
						{
							stats->stddev = sqrtf(stats->stddev / n);

							if (stats->stddev > FLT_EPSILON)
							{
								stats->skewness = stats->skewness / n / (stats->stddev * stats->stddev * stats->stddev);
								stats->kurtosis = (stats->kurtosis / n / (stats->stddev * stats->stddev * stats->stddev * stats->stddev)) - 3.0f;
							}
						}

						if ((stats->mean > FLT_EPSILON) || (stats->mean < -FLT_EPSILON))
						{
							stats->cov = stats->stddev / stats->mean * ratio_modifier;
						}
						else
						{
							stats->cov = 0.0f;
						}
					}
				}
				else if (analysis->channel[analysis->group[group].channels[0]].results[analysis_channel].abscissa.type == ABSCISSA_CRANKANGLE)
				{
					if ((stats_calcs & STATS_CYCLE_BIT) > 0)
					{
						stats_array = &analysis->group[group].crank_angle_statistics[analysis_channel];

						samples_per_cycle = analysis->channel[analysis->group[group].channels[0]].results[analysis_channel].samples_per_cycle;

						/* TODO: This only works based on each channel in a group having the same measurement tables */

						for (crank_angle = 0; crank_angle<samples_per_cycle; crank_angle++)
						{
							stats_array->max[crank_angle] = analysis->channel[analysis->group[group].channels[0]].ca_statistics[analysis_channel].max[crank_angle];
							stats_array->min[crank_angle] = analysis->channel[analysis->group[group].channels[0]].ca_statistics[analysis_channel].min[crank_angle];
							stats_array->sum[crank_angle] = 0.0f;

							for (group_channel = 0; group_channel<analysis->group[group].number_of_channels; group_channel++)
							{
								if (analysis->channel[analysis->group[group].channels[group_channel]].analysis_rqst[analysis_channel] >= request_threshold)
								{
									angle = analysis->channel[analysis->group[group].channels[0]].results[analysis_channel].tdc_offset -
										analysis->channel[analysis->group[group].channels[group_channel]].results[analysis_channel].tdc_offset;

									theta = analysis->channel[analysis->group[group].channels[0]].results[analysis_channel].abscissa.ca_to_theta[crank_angle] +
										(unsigned int)(floor(angle) * ((float)min_res_ppr / 360.0f));

									while (theta < 0)
									{
										theta += maximum_theta;
									}

									while (theta >= (int)maximum_theta)
									{
										theta -= maximum_theta;
									}

									adjusted_crank_angle = analysis->channel[analysis->group[group].channels[group_channel]].results[analysis_channel].abscissa.theta_to_ca[theta];

									min = analysis->channel[analysis->group[group].channels[group_channel]].ca_statistics[analysis_channel].min[adjusted_crank_angle];
									max = analysis->channel[analysis->group[group].channels[group_channel]].ca_statistics[analysis_channel].max[adjusted_crank_angle];
									mean = analysis->channel[analysis->group[group].channels[group_channel]].ca_statistics[analysis_channel].mean[adjusted_crank_angle];

									stats_array->sum[crank_angle] += mean;

									if (min < stats_array->min[crank_angle])
									{
										stats_array->min[crank_angle] = min;
									}

									if (max > stats_array->max[crank_angle])
									{
										stats_array->max[crank_angle] = max;
									}
								}
							}

							stats_array->mean[crank_angle] = stats_array->sum[crank_angle] / (float)analysis->group[group].number_of_channels;

							stats_array->stddev[crank_angle] = 0.0f;		/* TODO: Calculate this */
						}
					}
				}
				else
				{
					/* Do Nothing */
				}
			}
		}
	}
}

void calculate_cycle_stats(Statistics* stats, const float* data, const unsigned int number_of_cycles, const unsigned int stats_type)
{
	unsigned int cycle;
	float delta;
	float value;
	float cycles_minus_one;
	float ratio_modifier = 1.0f;

	if ((stats == NULL) || 
		(data == NULL) || 
		(number_of_cycles == 0))
	{
		return;
	}

	if (stats_type == STATS_PERCENT)
	{
		ratio_modifier = 100.0f;
	}

	cycles_minus_one = (float)number_of_cycles - 1.0f;

	stats->sum = data[0];
	stats->min = data[0];
	stats->max = data[0];

	for (cycle = 1; cycle<number_of_cycles; cycle++)
	{
		value = data[cycle];

		stats->sum += value;

		if (value > stats->max)
		{
			stats->max = value;
		}
		else if (value < stats->min)
		{
			stats->min = value;
		}
		else
		{
			/* Do Nothing */
		}
	}

	stats->range = stats->max - stats->min;
	stats->mean = stats->sum / (float)number_of_cycles;

	stats->stddev = 0.0f;
	stats->skewness = 0.0f;
	stats->kurtosis = 0.0f;

	for (cycle = 0; cycle<number_of_cycles; cycle++)
	{
		delta = data[cycle] - stats->mean;

		stats->stddev += delta * delta;
		stats->skewness += delta * delta * delta;
		stats->kurtosis += delta * delta * delta * delta;
	}

	if (cycles_minus_one > FLT_EPSILON)
	{
		stats->stddev = sqrtf(stats->stddev / cycles_minus_one);

		if (stats->stddev > FLT_EPSILON)
		{
			stats->skewness = stats->skewness / cycles_minus_one / (stats->stddev * stats->stddev * stats->stddev);
			stats->kurtosis = (stats->kurtosis / cycles_minus_one / (stats->stddev * stats->stddev * stats->stddev * stats->stddev)) - 3.0f;
		}
		else
		{
			stats->skewness = 0.0f;
			stats->kurtosis = 0.0f;
		}
	}

	if ((stats->mean > FLT_EPSILON) || (stats->mean < -FLT_EPSILON))
	{
		stats->lnv = stats->min / stats->mean * ratio_modifier;
		stats->cov = stats->stddev / stats->mean * ratio_modifier;
	}
	else
	{
		stats->lnv = 1.0f * ratio_modifier;
		stats->cov = 0.0f;
	}
}

void calculate_ca_stats(StatisticsArray* stats, const float* data, const unsigned int samples_per_cycle, const unsigned int number_of_cycles)
{
	unsigned int cycle;
	unsigned int crank_angle;
	float value;
	float delta;
	float cycles_minus_one;
	unsigned int data_pointer;

	if ((stats == NULL) || 
		(data == NULL) || 
		(number_of_cycles == 0) || 
		(samples_per_cycle == 0))
	{
		return;
	}

	cycles_minus_one = (float)number_of_cycles - 1.0f;

	for (crank_angle = 0; crank_angle<samples_per_cycle; crank_angle++)
	{
		stats->sum[crank_angle] = 0.0f;
		stats->min[crank_angle] = data[crank_angle];
		stats->max[crank_angle] = data[crank_angle];

		data_pointer = crank_angle;

		for (cycle = 0; cycle<number_of_cycles; cycle++)
		{
			value = data[data_pointer];

			data_pointer += samples_per_cycle;

			stats->sum[crank_angle] += value;

			if (value > stats->max[crank_angle])
			{
				stats->max[crank_angle] = value;
			}
			else if (value < stats->min[crank_angle])
			{
				stats->min[crank_angle] = value;
			}
			else
			{
				/* Do Nothing */
			}
		}

		stats->mean[crank_angle] = stats->sum[crank_angle] / (float)number_of_cycles;

		stats->stddev[crank_angle] = 0.0f;

		data_pointer = crank_angle;

		for (cycle = 0; cycle<number_of_cycles; cycle++)
		{
			value = data[data_pointer];

			data_pointer += samples_per_cycle;

			delta = value - stats->mean[crank_angle];

			stats->stddev[crank_angle] += delta * delta;
		}

		stats->stddev[crank_angle] = sqrtf(stats->stddev[crank_angle] / cycles_minus_one);

		/*   Not worth the memory to calculate

		stats->range[crank_angle] = 0.0f;
		stats->cov[crank_angle] = 0.0f;
		stats->kurtosis[crank_angle] = 0.0f;
		stats->lnv[crank_angle] = 0.0f;
		stats->skewness[crank_angle] = 0.0f;

		*/
	}
}

void calculate_freq_stats(StatisticsArray* stats, const float* data, const float* frequency, const unsigned int unit, const unsigned int samples_per_cycle, const unsigned int number_of_cycles, float max_frequency, float resolution)
{
	unsigned int cycle;
	unsigned int crank_angle;
	unsigned int bin;
	unsigned int lower_bin;
	unsigned int upper_bin;
	float lower_value;
	float upper_value;
	float lower_freq;
	float upper_freq;
	float* min = NULL;
	float* max = NULL;
	unsigned int* n = NULL;
	float* sum = NULL;
	float** bin_values = NULL;
	unsigned int number_of_bins;
	unsigned int data_pointer;
	float slope;

	if ((stats == NULL) || 
		(data == NULL) || 
		(frequency == NULL) || 
		(samples_per_cycle == 0) || 
		(number_of_cycles == 0) || 
		(max_frequency < FLT_EPSILON) || 
		(resolution < FLT_EPSILON))
	{
		return;
	}

	if (unit == UNITS_HZ)
	{
		/* Configuration max_frequency and resolution are in kHz */

		max_frequency *= 50.0f;			/* 100kHz -> 5,000 Hz*/
		resolution *= 50.0f;			/* 0.25kHz -> 12.5 Hz */
	}

	number_of_bins = (unsigned int)(max_frequency / resolution) + 1;

	if (number_of_bins == 0)
	{
		return;
	}

	min = (float*)malloc(number_of_bins * sizeof(float));

	if (min == NULL)
	{
		logmessage(FATAL, "Out of memory\n");
	}

	max = (float*)malloc(number_of_bins * sizeof(float));

	if (max == NULL)
	{
		logmessage(FATAL, "Out of memory\n");
	}

	n = (unsigned int*)malloc(number_of_bins * sizeof(unsigned int));

	if (n == NULL)
	{
		logmessage(FATAL, "Out of memory\n");
	}

	sum = (float*)malloc(number_of_bins * sizeof(float));

	if (sum == NULL)
	{
		logmessage(FATAL, "Out of memory\n");
	}

	bin_values = (float**)malloc(number_of_bins * sizeof(float*));
	
	if (bin_values == NULL)
	{
		logmessage(FATAL, "Out of memory\n");
	}

	for (bin = 0; bin < number_of_bins; bin++)
	{
		min[bin] = FLT_MAX;
		max[bin] = -FLT_MAX;
		sum[bin] = 0.0f;
		n[bin] = 0U;
		bin_values[bin] = (float*)malloc(number_of_cycles * sizeof(float));
	}

	for (crank_angle = 0; crank_angle < samples_per_cycle; crank_angle++)
	{
		for (cycle = 0; cycle < number_of_cycles; cycle++)
		{
			data_pointer = cycle * samples_per_cycle + crank_angle;

			lower_value = data[data_pointer];
			lower_freq = frequency[data_pointer];

			if (crank_angle < samples_per_cycle - 1)
			{
				upper_value = data[data_pointer + 1];
				upper_freq = frequency[data_pointer + 1];

				slope = (upper_value - lower_value) / (upper_freq - lower_freq);
			}
			else
			{
				upper_freq = lower_freq;

				slope = 0.0f;
			}

			/* Calculate bins from freq (Hz)*/

			lower_bin = (unsigned int)(lower_freq / resolution);
			upper_bin = (unsigned int)(upper_freq / resolution) + 1;

			if ((lower_bin < number_of_bins) &&
				(upper_bin <= number_of_bins))

			{
				for (bin = lower_bin; bin < upper_bin; bin++)
				{
					float bin_freq = bin * resolution;

					if ((bin_freq >= lower_freq) && (bin_freq < upper_freq) && (bin < number_of_bins))
					{
						float value = lower_value + slope * (bin_freq - lower_freq);

						if (value < min[bin])
						{
							min[bin] = value;
						}

						if (value > max[bin])
						{
							max[bin] = value;
						}

						bin_values[bin][n[bin]] = value;
						sum[bin] += value;

						if (n[bin] < number_of_cycles - 1)
						{
							n[bin]++;
						}
					}
				}
			}
			else
			{
				logmessage(NOTICE, "Bin out of range (%f,%f,%f,%u,%u,%u)\n",
					lower_freq,
					upper_freq,
					resolution,
					lower_bin,
					upper_bin,
					number_of_bins);
			}
		}
	}

	for (bin = 0; bin < number_of_bins; bin++)
	{
		if (n[bin] > 0)
		{
			float mean = sum[bin] / (float)n[bin];
			float stddev = 0.0f;
			unsigned int i;

			for (i = 0; i < n[bin]; i++)
			{
				float delta = bin_values[bin][i] - mean;

				stddev += delta * delta;
			}

			stats->min[bin] = min[bin];
			stats->max[bin] = max[bin];
			stats->mean[bin] = mean;
			
			if (n[bin] > 1)
			{
				stats->stddev[bin] = sqrtf(stddev / (float)(n[bin] - 1U));
			}
			else
			{
				stats->stddev[bin] = 0.0f;
			}
		}
		else
		{
			stats->min[bin] = 0.0f;
			stats->max[bin] = 0.0f;
			stats->mean[bin] = 0.0f;
			stats->stddev[bin] = 0.0f;
		}

		free(bin_values[bin]);
		bin_values[bin] = NULL;
	}

	free(min);
	free(max);
	free(n);
	free(sum);
	free(bin_values);
}

void calculate_single_cycle_stats(Statistics* stats, const float* data, const unsigned int samples_per_cycle, const unsigned int stats_type)
{
	unsigned int crank_angle;
	float value;
	float delta;
	float ratio_modifier = 1.0f;

	if ((stats == NULL) || 
		(data == NULL) || 
		(samples_per_cycle == 0))
	{
		return;
	}

	if (stats_type == STATS_PERCENT)
	{
		ratio_modifier = 100.0f;
	}

	stats->sum = 0.0f;
	stats->min = data[0];
	stats->max = data[0];

	for (crank_angle = 0; crank_angle<samples_per_cycle; crank_angle++)
	{
		value = data[crank_angle];

		stats->sum += value;

		if (value > stats->max)
		{
			stats->max = value;
		}
		else if (value < stats->min)
		{
			stats->min = value;
		}
		else
		{
			/* Do Nothing */
		}
	}

	stats->range = stats->max - stats->min;

	stats->mean = stats->sum / (float)samples_per_cycle;

	stats->stddev = 0.0f;

	for (crank_angle = 0; crank_angle<samples_per_cycle; crank_angle++)
	{
		value = data[crank_angle];

		delta = value - stats->mean;

		stats->stddev += delta * delta;
	}

	if (samples_per_cycle > 1)
	{
		stats->stddev = sqrtf(stats->stddev / ((float)samples_per_cycle - 1.0f));
	}
	else
	{
		stats->stddev = 0.0f;
	}

	/* TODO: Not currently calculated */

	stats->skewness = 0.0f;
	stats->cov = 0.0f;
	stats->kurtosis = 0.0f;
	stats->lnv = 1.0f * ratio_modifier;
}

/* Ref: "General Least-Squares Smoothing and Differentiation by the Convolution (Savitzky-Golay) Method", Peter A. Gorry, Anal. Chem. 1990, pp 570-573 */

double GramPoly(int i, int m, int k, int s)
{
	double gp;

	if (k > 0)
	{
		gp = (4.0 * k - 2.0) / (k * (2.0 * m - k + 1.0)) * (i * GramPoly(i, m, k - 1, s) + s * GramPoly(i, m, k - 1, s - 1)) - ((k - 1.0) * (2.0 * m + k)) / (k * (2.0 * m - k + 1.0)) * GramPoly(i, m, k - 2, s);
	}
	else
	{
		if ((k == 0) && (s == 0))
		{
			gp = 1.0;
		}
		else
		{
			gp = 0.0;
		}
	}

	return (gp);
}

double GenFact(int a, int b)
{
	int j;
	double gf = 1.0;

	for (j = (a - b + 1); j <= a; j++)
	{
		gf = gf * j;
	}

	return (gf);
}

double Weight(int i, int t, int m, int n, int s)
{
	/* Calculate point i */
	/* t=0 gives Savitzky-Golay table values */
	/* of a 2m+1 point filter */
	/* with polynomial order n */
	/* calculating the s-th derivative (0 = smoothing filter) */

	int k;
	double sum = 0.0;

	for (k = 0; k <= n; k++)
	{
		sum += (2.0 * k + 1.0) * (GenFact(2 * m, k) / GenFact(2 * m + k + 1, k + 1)) * GramPoly(i, m, k, 0) * GramPoly(t, m, k, s);
	}

	return (sum);
}

void calc_sg_coefficients(const unsigned int order, float* coefficients)
{
	unsigned int z;
	unsigned int m;

	if ((coefficients == NULL) || 
		(order == 0))
	{
		return;
	}

	m = (order - 1) / 2;

	for (z = 0; z <= m; z++)
	{
		coefficients[m + z] = (float)Weight(z, 0, m, 2, 0);
		coefficients[m - z] = coefficients[m + z];
	}
}
