/* Combustion Analysis Tool (CAT)
   www.catool.org
   
   Filename: matfile.c

   Purpose:  Provide common programming interface for processing acquired engine data
  
   Author:   Ben Brown
   Version:  1.2
   Date:     19.10.2015

   Revision: GPL Release

   Copyright (C) Xarin Limited, 2000-2024

      This program is free software: you can redistribute it and/or modify
      it under the terms of version 2 of the GNU General Public License as
      published by the Free Software Foundation.

      This program is distributed in the hope that it will be useful,
      but WITHOUT ANY WARRANTY; without even the implied warranty of
      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
      GNU General Public License for more details.

      You should have received a copy of the GNU General Public License
      along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "cat.h"

#if defined(linux) || defined(__linux) || defined(__linux__) || defined (MGLNX86) || defined (ARCH_GLNX86)
    #if defined(__x86_64__) || defined(__amd64__) || defined(__ia64__) || defined(__ia64) || defined(_M_IA64)
        static char matlab_platform[20] = "GLNXA64";		/* Linux 64-bit */
    #else
        static char matlab_platform[20] = "GLNX86";      /* Linux 32-bit */
    #endif
#elif defined(__hppa) || defined(__hpux) || defined(_hpux) || defined(hpux) || defined (MHPUX) || defined (ARCH_HPUX)
static char matlab_platform[20] = "HPUX";			/* HP PA-RISC (HP-UX 11.00) */
#elif defined(__hp700) || defined(MHP700) || defined(ARCH_HP700)
static char matlab_platform[20] = "HP700";			/* HP 700 Unix */
#elif defined(Macintosh) || defined(macintosh) || defined(__APPLE__) || defined(__APPLE_CC__) || defined(__MACH__)
static char matlab_platform[20] = "MAC";             /* Macintosh OS X */
#elif defined(_WIN64) || defined(__WIN64__) || defined(WIN64) || defined(__WIN64)
static char matlab_platform[20] = "PCWIN64";         /* Microsoft Windows 64-bit */
#elif defined(_WIN32) || defined(__WIN32__) || defined(WIN32) || defined(__WIN32) || defined(_MSC_VER) || defined(__WINDOWS__) || defined(__TOS_WIN__)
static char matlab_platform[20] = "PCWIN";           /* Microsoft Windows 32-bit */
#elif defined(sun) || defined(__sun) || defined(MSOL2) || defined(ARCH_SOL2)
    #if defined(__x86_64__) || defined(__amd64__) || defined(__ia64__) || defined(__ia64) || defined(_M_IA64)
        static char matlab_platform[20] = "SOL64";		/* Sun Solaris 64-bit */
    #else
        static char matlab_platform[20] = "SOL2";		/* Sun Solaris 2 SPARC workstation */
    #endif
#elif defined (alpha) || defined (__alpha__) || defined(__alpha) || defined (MALPHA) || defined (ARCH_ALPHA)
static char matlab_platform[20] = "ALPHA";			/* Compaq Alpha */
#elif defined (__sgi) || defined (MSGI) || defined (ARCH_SGI)
    #if defined(SGI64)
        static char matlab_platform[20] = "SGI64";		/* SGI 64-bit */
    #else
        static char matlab_platform[20] = "SGI";			/* SGI */
    #endif
#elif defined (__vax) || defined(__VMS) || defined(__VMS__) || defined(VMS)
static char matlab_platform[20] = "VAX_VMSG";		/* VAX VMS */
#elif defined(IBM_RS)
static char matlab_platform[20] = "IBM_RS";			/* IBM RS6000 */
#elif defined(sun4) || defined(__sun4)
static char matlab_platform[20] = "SUN4";			/* Sun 4 */
#else
static char matlab_platform[20] = "PCWIN";			/* Not a native MATLAB platform, default to MS Windows */
#endif

extern unsigned int debug_level;

unsigned int sizeof_datatype(const unsigned int datatype)
{
	unsigned int number_of_bytes = 0;
	
	switch (datatype)
	{
	case miINT8:		/* 8 bit signed */
	{
		number_of_bytes = 1;
		break;
	}
	case miUINT8:		/* 8 bit unsigned */
	{
		number_of_bytes = 1;
		break;
	}
	case miINT16:		/* 16 bit signed */
	{
		number_of_bytes = 2;
		break;
	}
	case miUINT16:		/* 16 bit unsigned */
	{	
		number_of_bytes = 2;
		break;
	}
	case miINT32:		/* 32 bit signed */
	{
		number_of_bytes = 4;
		break;
	}
	case miUINT32:		/* 32 bit unsigned */
	{
		number_of_bytes = 4;
		break;
	}
	case miSINGLE:		/* IEEE 754 single format */
	{	
		number_of_bytes = 4;
		break;
	}
	case miDOUBLE:		/* IEEE 754 double format */
	{
		number_of_bytes = 8;
		break;
	}
	case miINT64:		/* 64-bit signed */
	{
		number_of_bytes = 8;
		break;
	}
	case miUINT64:		/* 64-bit unsigned */
	{
		number_of_bytes = 8;
		break;
	}
	case miMATRIX:		/* MATLAB array */
	{
		number_of_bytes = 0;
		break;
	}
	case miCOMPRESSED:	/* Compressed Data */
	{
		number_of_bytes = 0;
		break;
	}
	case miUTF8:		/* Unicode UTF-8 Encoded Character Data */
	{
		number_of_bytes = 1;
		break;
	}
	case miUTF16:		/* Unicode UTF-16 Encoded Character Data */
	{
		number_of_bytes = 2;
		break;
	}
	case miUTF32:		/* Unicode UTF-32 Encoded Character Data */
	{
		number_of_bytes = 4;
		break;
	}
	default:
	{
		number_of_bytes = 0;
		break;
	}
	}

	return (number_of_bytes);
}

void validate_variable_name(char* var_name)
{
	size_t var_name_length;
	size_t position;
	unsigned int keyword;
	const char* keywords[19] = { "break", "case", "catch", "classdef", "continue", "else", "elseif", "end", "for", "function", "global", "if", "otherwise", "parfor", "persistent", "return", "switch", "try", "while" };
	
	if (var_name == NULL)
	{
		return;
	}

	var_name_length = strlen(var_name);

	for (position=0;position<var_name_length;position++)
	{
		if (((var_name[position] >= 'A') && (var_name[position] <= 'Z')) ||
			((var_name[position] >= 'a') && (var_name[position] <= 'z')) ||
			((var_name[position] >= '0') && (var_name[position] <= '9')) ||
			(var_name[position] == '_'))
		{
			/* Character is OK */
		}
		else if (var_name[position] == '%')
		{
			var_name[position] = 'X';
		}
		else
		{
			var_name[position] = '_';
		}
	}

	if (((var_name[0] >= 'A') && (var_name[0] <= 'Z')) ||
		((var_name[0] >= 'a') && (var_name[0] <= 'z')))
	{
		/* Starting character is OK */
	}
	else
	{
		var_name[0] = 'z';
	}

	for (keyword=0;keyword<19;keyword++)
	{
		if (strncmp(var_name,keywords[keyword],11) == 0)
		{
			var_name[0] = toupper(var_name[0]);
		}
	}
}

void generate_matlab_channelnames(FileData* file,Analysis* analysis)
{
	unsigned int channel;
	unsigned int channel_2;
	unsigned int analysis_number;
	unsigned int analysis_channel;
	unsigned int analysis_number_2;
	unsigned int analysis_channel_2;
	char matlab_name[SIZEOF_CHANNEL_NAME];
	unsigned int number;
	unsigned int previous_number;

	if ((file != NULL) && (file->channel_data != NULL))
	{
		for (channel=0;channel<file->number_of_channels;channel++)
		{
			if (file->channel_data[channel].abscissa.type == ABSCISSA_TIME)
			{
				snprintf(file->channel_data[channel].matlab_name, SIZEOF_CHANNEL_NAME, "TM_%s", file->channel_data[channel].name);
			}
			else
			{
				strncpy(file->channel_data[channel].matlab_name, file->channel_data[channel].name, SIZEOF_CHANNEL_NAME);
			}

			file->channel_data[channel].matlab_name[SIZEOF_CHANNEL_NAME-1] = 0x00;
			validate_variable_name(file->channel_data[channel].matlab_name);
		}

		/* Now check all names are unique and add _# as necessary */

		for (channel = 0; channel < file->number_of_channels; channel++)
		{
			number = 1;

			for (channel_2 = channel + 1; channel_2 < file->number_of_channels; channel_2++)
			{
				previous_number = number;

				strncpy(matlab_name, file->channel_data[channel_2].matlab_name, SIZEOF_CHANNEL_NAME);
				matlab_name[SIZEOF_CHANNEL_NAME - 1] = 0x00;

				while (strncmp(file->channel_data[channel].matlab_name, matlab_name, SIZEOF_CHANNEL_NAME) == 0)
				{
					snprintf(matlab_name, SIZEOF_CHANNEL_NAME, "%s_%u", file->channel_data[channel_2].matlab_name, number);
					matlab_name[SIZEOF_CHANNEL_NAME - 1] = 0x00;
					number++;
				}

				if (number != previous_number)
				{
					logmessage(NOTICE, "%s: Renamed channel to %s to ensure unique MATLAB variable name\n", file->channel_data[channel_2].matlab_name, matlab_name);

					strncpy(file->channel_data[channel_2].matlab_name, matlab_name, SIZEOF_CHANNEL_NAME);
					file->channel_data[channel_2].matlab_name[SIZEOF_CHANNEL_NAME - 1] = 0x00;
				}
			}
		}
	}

	if ((analysis != NULL) && (analysis->channel != NULL))
	{
		for (analysis_number=0;analysis_number<analysis->number_of_channels;analysis_number++)
		{
			for (analysis_channel=0;analysis_channel<get_number_of_analysis_channels();analysis_channel++)
			{
				if (analysis->channel[analysis_number].analysis_rqst[analysis_channel] >= AnalysisRqst::CalculateAndStore)
				{
					strncpy(analysis->channel[analysis_number].results[analysis_channel].matlab_name, analysis->channel[analysis_number].results[analysis_channel].name, SIZEOF_CHANNEL_NAME);
					analysis->channel[analysis_number].results[analysis_channel].matlab_name[SIZEOF_CHANNEL_NAME - 1] = 0x00;

					validate_variable_name(analysis->channel[analysis_number].results[analysis_channel].matlab_name);
				}
			}
		}

		/* Now check all names are unique and add _# as necessary */

		for (analysis_number = 0; analysis_number < analysis->number_of_channels; analysis_number++)
		{
			for (analysis_channel = 0; analysis_channel < get_number_of_analysis_channels(); analysis_channel++)
			{
				number = 1;

				if (analysis->channel[analysis_number].analysis_rqst[analysis_channel] >= AnalysisRqst::CalculateAndStore)
				{
					for (analysis_number_2 = analysis_number + 1; analysis_number_2 < analysis->number_of_channels; analysis_number_2++)
					{
						for (analysis_channel_2 = 0; analysis_channel_2 < get_number_of_analysis_channels(); analysis_channel_2++)
						{
							previous_number = number;

							strncpy(matlab_name, analysis->channel[analysis_number_2].results[analysis_channel_2].matlab_name, SIZEOF_CHANNEL_NAME);
							matlab_name[SIZEOF_CHANNEL_NAME - 1] = 0x00;

							while (strncmp(analysis->channel[analysis_number].results[analysis_channel].matlab_name, matlab_name, SIZEOF_CHANNEL_NAME) == 0)
							{
								snprintf(matlab_name, SIZEOF_CHANNEL_NAME, "%s_%u", analysis->channel[analysis_number_2].results[analysis_channel_2].matlab_name, number);
								matlab_name[SIZEOF_CHANNEL_NAME - 1] = 0x00;
								number++;
							}

							if (number != previous_number)
							{
								logmessage(NOTICE, "%s: Renamed analysis channel to %s to ensure unique MATLAB variable name\n", analysis->channel[analysis_number_2].results[analysis_channel_2].matlab_name, matlab_name);

								strncpy(analysis->channel[analysis_number_2].results[analysis_channel_2].matlab_name, matlab_name, SIZEOF_CHANNEL_NAME);
								analysis->channel[analysis_number_2].results[analysis_channel_2].matlab_name[SIZEOF_CHANNEL_NAME - 1] = 0x00;
							}
						}
					}
				}
			}
		}
	}
}

void matlab_file_open(FILE* file_handle)
{
	char header_text[116];
	char subsystem_info[8];
    short version = 0x0100;
    short endian = 0x4D49;
	time_t t = time(NULL);
	struct tm current_time = *localtime(&t);
	size_t padding;
	size_t pad_start;
	size_t written;
	
	if (file_handle == NULL)
	{
		logmessage(WARNING, "MATLAB file is null\n");

		return;
	}

    memset(&header_text,0x20,116);
    memset(&subsystem_info,0x00,8);

	pad_start = snprintf(header_text,116,"MATLAB 5.0 MAT-file, Platform: %s, Created on: %s",matlab_platform,asctime(&current_time));
		
	if (pad_start > 115)
	{
		pad_start = 115;
	}
		
	for (padding=pad_start;padding<116;padding++)
	{
		header_text[padding] = 0x20;
	}

	written = fwrite(&header_text,116,1,file_handle);
	written += fwrite(&subsystem_info,8,1,file_handle);
	written += fwrite(&version,2,1,file_handle);
	written += fwrite(&endian,2,1,file_handle);

	if (written != 4)
	{
		logmessage(MSG_ERROR,"Did not successfully write MATLAB header\n");
	}
}

void matlab_file_close(FILE* file_handle)
{
	if (file_handle == NULL)
	{
		logmessage(WARNING, "MATLAB file is null\n");

		return;
	}


	fclose(file_handle);
}

void matlab_file_write(FILE* file_handle, const unsigned int data_type, const void* data, const unsigned int data_size)
{
	unsigned int pad_bytes = 0;
	unsigned int comp_header;
	
	if (file_handle == NULL)
	{
		logmessage(WARNING, "MATLAB file is null\n");

		return;
	}

	if (data_size <= 4)
	{
		pad_bytes = 4 - data_size;
		
		comp_header = data_type | data_size << 16;
		
		fwrite_int(comp_header, file_handle);
	}
	else
	{
		if (data_size < 8)
		{
			pad_bytes = 8 - data_size;
		}
		else if ((data_size % 8) > 0)
		{
			pad_bytes = 8 - (data_size % 8);
		}
		else
		{
			pad_bytes = 0;
		}

		fwrite_int(data_type, file_handle);
		fwrite_int(data_size, file_handle);
	}
		
	if ((data_size > 0) && (data != NULL))
	{
		fwrite(data, data_size, 1, file_handle);
	}
			
	if (pad_bytes > 0)
	{
		char zero = 0x00;
		for (unsigned int i = 0; i < pad_bytes; i++)
		{
			fwrite(&zero, 1, 1, file_handle);
		}
	}
}

void matlab_file_write_numeric_array(FILE* file_handle,const unsigned int data_type,const bool global,const bool logical,const char* array_name,const unsigned int number_of_array_dimensions,const int* dimensions_data,const void* real_data,const void* imaginary_data)
{
	unsigned int flags;
	int number_of_bytes = 0;
	unsigned int dimension;
	int array_name_size;
	int array_name_padding = 0;
	int array_dimension_size;
	int real_data_size;
	int imaginary_data_size;
	int sizeof_real_data = 0;
	int sizeof_imaginary_data = 0;
	unsigned int matlab_data_type = 0;
	unsigned int array_type = 0;
	bool complex = false;
	int array_flags_subelement;

	if (file_handle == NULL)
	{
		logmessage(WARNING, "MATLAB file is null\n");

		return;
	}

	if (imaginary_data != NULL)
	{
		complex = true;
	}

	switch (data_type)
	{
	case miINT8:
	{
		array_type = mxINT8_CLASS;
		matlab_data_type = data_type;
		break;
	}
	case miUINT8:
	{
		array_type = mxUINT8_CLASS;
		matlab_data_type = data_type;
		break;
	}
	case miINT16:
	{
		array_type = mxINT16_CLASS;
		matlab_data_type = data_type;
		break;
	}
	case miUINT16:
	{
		array_type = mxUINT16_CLASS;
		matlab_data_type = data_type;
		break;
	}
	case miINT32:
	{
		array_type = mxINT32_CLASS;
		matlab_data_type = data_type;
		break;
	}
	case miUINT32:
	{
		array_type = mxUINT32_CLASS;
		matlab_data_type = data_type;
		break;
	}
	case miINT64:
	{
		array_type = mxINT64_CLASS;
		matlab_data_type = data_type;
		break;
	}
	case miUINT64:
	{
		array_type = mxUINT64_CLASS;
		matlab_data_type = data_type;
		break;
	}
	case miSINGLE:
	{
		array_type = mxSINGLE_CLASS;
		matlab_data_type = data_type;
		break;
	}
	case miDOUBLE:
	{
		array_type = mxDOUBLE_CLASS;
		matlab_data_type = data_type;
		break;
	}
	case miUTF8:
	{
		array_type = mxCHAR_CLASS;
		matlab_data_type = miUINT8;
		break;
	}
	case miUTF16:
	{
		array_type = mxCHAR_CLASS;
		matlab_data_type = miUINT16;
		break;
	}
	case miUTF32:
	{
		array_type = mxCHAR_CLASS;
		matlab_data_type = miUINT32;
		break;
	}
	default:
	{
		array_type = 0;
		matlab_data_type = data_type;
	}
	}

	/* Array Flags */

	flags = mfNONE;
	if (complex == true)
	{
		flags |= mfCOMPLEX;
	}
	if (global == true)
	{
		flags |= mfGLOBAL;
	}
	if (logical == true)
	{
		flags |= mfLOGICAL;
	}
	
	/* Array Name */

	if (array_name == NULL)
	{
		array_name_size = 0;
	}
	else
	{
		array_name_size = (unsigned int)strlen(array_name);
	}

	if ((array_name_size % 8) > 0)
	{
		array_name_padding = 8 - (array_name_size % 8);
	}
	
	/* Array Dimensions */
	
	array_dimension_size = 4 * number_of_array_dimensions;
	
	if ((array_dimension_size % 8) > 0)
	{
		array_dimension_size += 8 - (array_dimension_size % 8);
	}
	
	/* Calculate Data Size */
	
	sizeof_real_data = 1;
	for (dimension=0;dimension<number_of_array_dimensions;dimension++)
	{
		sizeof_real_data *= dimensions_data[dimension];
	}
	
	sizeof_real_data *= sizeof_datatype(matlab_data_type);
	sizeof_imaginary_data = sizeof_real_data;
	
	if (real_data == NULL)
	{
		sizeof_real_data = 0;
	}
		
	if (imaginary_data == NULL)
	{
		sizeof_imaginary_data = 0;
	}
	
	if (sizeof_real_data == 0)
	{
		real_data_size = 0;
	}
	else if (sizeof_real_data <= 4)
	{
		real_data_size = 2 + 2 + 4;
	}
	else
	{
		real_data_size = sizeof_real_data;	
		if ((real_data_size % 8) > 0)
		{
			real_data_size += 8 - (real_data_size % 8);
		}
		real_data_size += 4 + 4;
	}
	
	if (sizeof_imaginary_data == 0)
	{
		imaginary_data_size = 0;
	}
	else if (sizeof_imaginary_data <= 4)
	{
		imaginary_data_size = 2 + 2 + 4;
	}
	else
	{
		imaginary_data_size = sizeof_imaginary_data;
		if ((imaginary_data_size % 8) > 0)
		{
			imaginary_data_size += 8 - (imaginary_data_size % 8);
		}
		imaginary_data_size += 4 + 4;
	}
			
	number_of_bytes = 4 + 4 + 
	                  4 + 4 +
					  4 + 4 +
					  array_dimension_size + 
					  4 + 4 + 
					  array_name_size +
					  array_name_padding;

	if (real_data_size > 0)
	{
		number_of_bytes += real_data_size;
	}

	if (imaginary_data_size > 0)
	{
		number_of_bytes += imaginary_data_size;
	}
				
	fwrite_int(miMATRIX,file_handle);
	fwrite_int(number_of_bytes,file_handle);
	
	fwrite_int(miUINT32,file_handle);
	fwrite_int(8,file_handle);
	
	array_flags_subelement = array_type | flags;

	fwrite_int(array_flags_subelement,file_handle);
	fwrite_int(0,file_handle);

	/* Dimensions Array */

	fwrite_int(miINT32,file_handle);
	fwrite_int(number_of_array_dimensions*4,file_handle);

	for (dimension=0;dimension<number_of_array_dimensions;dimension++)
	{
		fwrite_int(dimensions_data[dimension],file_handle);
	}

	fwrite_int(miINT8,file_handle);
	fwrite_int(array_name_size + array_name_padding,file_handle);
	
	if (array_name_size > 0)
	{
		fwrite(array_name, array_name_size, 1, file_handle);
	}

	for (int i = 0; i < array_name_padding; i++)
	{
		char zero = 0x00;
		fwrite(&zero, 1, 1, file_handle);
	}

	/* Real Part (pr) */

	if (real_data != NULL)
	{
		matlab_file_write(file_handle,matlab_data_type,real_data,sizeof_real_data);
	}

	/* Imaginary Part (pi) */

	if (imaginary_data != NULL)
	{
		matlab_file_write(file_handle,matlab_data_type,imaginary_data,sizeof_imaginary_data);
	}
}

size_t ascii_to_utf16(const char* ascii_in,char** utf16_out,const size_t maximum_length)
{
	size_t position;
	size_t length;
	short* utf16_out_ptr = NULL;
	
	if ((utf16_out == NULL) || (ascii_in == NULL))
	{
		return(0);
	}

	length = strnlen(ascii_in, 20);

	if (length > maximum_length)
	{
		length = maximum_length;
	}

	if (*utf16_out != NULL)
	{
		free(*utf16_out);
	}

	if (length == 0)
	{
		*utf16_out = (char*)malloc(1 * 2);
	}
	else
	{
		*utf16_out = (char*)malloc(length * 2);
	}

	if (*utf16_out == NULL)
	{
		logmessage(FATAL, "Memory could not be allocated\n");
	}

	utf16_out_ptr = (short*)*utf16_out;

	if (length == 0)
	{
		utf16_out_ptr[0] = 0x002D;	/* - */

		length = 1;
	}
	else
	{
		for (position = 0; position < length; position++)
		{
			utf16_out_ptr[position] = (short)ascii_in[position];
		}
	}

	return(length);
}

void matlab_file_write_utf16(FILE* file_handle, const char* string, size_t length)
{
	char* utf16_string = NULL;
	int dimensions[2];

	if ((file_handle == NULL) || (string == NULL))
	{
		return;
	}

	dimensions[0] = 1;
	dimensions[1] = (int)ascii_to_utf16(string, &utf16_string, length);

	matlab_file_write_numeric_array(file_handle, miUTF16, false, false, NULL, 2, dimensions, utf16_string, NULL);

	free(utf16_string);
	utf16_string = NULL;
}

void matlab_file_write_double(FILE* file_handle, const double* value)
{
	int dimensions[2];

	if (file_handle == NULL)
	{
		return;
	}

	dimensions[0] = 1;
	dimensions[1] = 1;

	matlab_file_write_numeric_array(file_handle, miDOUBLE, false, false, NULL, 2, dimensions, value, NULL);
}

void matlab_file_write_bool(FILE* file_handle, const bool* value)
{
	int dimensions[2];
	unsigned char uint8_value;

	if (file_handle == NULL)
	{
		return;
	}

	dimensions[0] = 1;
	dimensions[1] = 1;

	uint8_value = (*value == false) ? 0 : 1;

	matlab_file_write_numeric_array(file_handle, miUINT8, false, false, NULL, 2, dimensions, &uint8_value, NULL);
}

void matlab_file_write_float(FILE* file_handle, const float* value)
{
	int dimensions[2];

	if (file_handle == NULL)
	{
		return;
	}

	dimensions[0] = 1;
	dimensions[1] = 1;

	matlab_file_write_numeric_array(file_handle, miSINGLE, false, false, NULL, 2, dimensions, value, NULL);
}

void matlab_file_write_uint32(FILE* file_handle, const unsigned int* value)
{
	int dimensions[2];

	if (file_handle == NULL)
	{
		return;
	}

	dimensions[0] = 1;
	dimensions[1] = 1;

	matlab_file_write_numeric_array(file_handle, miUINT32, false, false, NULL, 2, dimensions, value, NULL);
}

void matlab_file_write_double_numeric_array(FILE* file_handle,const double value)
{
	int dimensions[2];
	dimensions[0] = 1;
	dimensions[1] = 1;

	matlab_file_write_numeric_array(file_handle,miDOUBLE,false,false,NULL,2,dimensions,&value,NULL);
}

void matlab_file_write_uint8_numeric_array(FILE* file_handle,const unsigned int value)
{
	int dimensions[2];
	dimensions[0] = 1;
	dimensions[1] = 1;

	matlab_file_write_numeric_array(file_handle,miUINT8,false,false,NULL,2,dimensions,&value,NULL);
}

void matlab_file_write_wchar_numeric_array(FILE* file_handle, const wchar_t* string, int max_length)
{
	int dimensions[2];

	dimensions[0] = 1;
	dimensions[1] = max_length;

	matlab_file_write_numeric_array(file_handle, miUTF16, false, false, NULL, 2, dimensions, string, NULL);
}

void matlab_file_write_string_numeric_array(FILE* file_handle,const char* string,int max_length)
{
	char* string_utf16 = NULL;
	int dimensions[2];

	dimensions[0] = 1;
	dimensions[1] = (int)ascii_to_utf16(string, &string_utf16, max_length);

	matlab_file_write_numeric_array(file_handle,miUTF16,false,false,NULL,2,dimensions,string_utf16,NULL);

	free(string_utf16);
}

void matlab_file_write_soc_config(FILE* file_handle, const SOCConfig* soc_config, char* array_name)
{
	unsigned int structure_size_file_position;
	unsigned int flags;
	unsigned int array_name_length;
	unsigned int field_name_length;
	unsigned int end_of_structure_file_position;
	unsigned int number_of_fields;
	unsigned int total_field_name_length;
	unsigned int structure_size;
	int array_flags_subelement;
	char array_type;
	char soc_type[20];

	if ((file_handle == NULL) || (soc_config == NULL) || (array_name == NULL))
	{
		return;
	}

	fwrite_int(miMATRIX, file_handle);

	structure_size_file_position = ftell(file_handle);

	fwrite_int(0, file_handle);			/* We'll go back later and fill in the number of bytes	*/

	/* Array Flags */

	array_type = mxSTRUCT_CLASS;
	flags = mfNONE;

	fwrite_int(miUINT32, file_handle);
	fwrite_int(8, file_handle);

	array_flags_subelement = (int)array_type | flags;

	fwrite_int(array_flags_subelement, file_handle);
	fwrite_int(0, file_handle);

	/* Dimensions Array */

	fwrite_int(miINT32, file_handle);
	fwrite_int(8, file_handle);
	fwrite_int(1, file_handle);
	fwrite_int(1, file_handle);

	/* Array Name */

	validate_variable_name(array_name);

	array_name_length = (unsigned int)strlen(array_name);

	if (array_name_length > SIZEOF_CHANNEL_NAME)
	{
		array_name_length = SIZEOF_CHANNEL_NAME;
	}

	matlab_file_write(file_handle, miINT8, array_name, array_name_length);

	/* Field Name Length */

	field_name_length = 24;

	matlab_file_write(file_handle, miINT32, &field_name_length, 4);

	/* Field Names */

	number_of_fields = 7;

	total_field_name_length = number_of_fields * field_name_length;

	fwrite_int(miINT8, file_handle);
	fwrite_int(total_field_name_length, file_handle);

	fwrite_padded_char("type", field_name_length, file_handle);
	fwrite_padded_char("start_window", field_name_length, file_handle);
	fwrite_padded_char("finish_window", field_name_length, file_handle);
	fwrite_padded_char("fixed_value", field_name_length, file_handle);
	fwrite_padded_char("channel", field_name_length, file_handle);
	fwrite_padded_char("aligned", field_name_length, file_handle);
	fwrite_padded_char("invert", field_name_length, file_handle);

	switch (soc_config->type)
	{
	case SOC_INVALID:				strncpy(soc_type, "INVALID", 20); break;
	case SOC_FIXED:					strncpy(soc_type, "FIXED", 20); break;
	case SOC_CA_CHANNEL_RISE:		strncpy(soc_type, "CA_CHANNEL_RISE", 20); break;
	case SOC_CA_CHANNEL_FALL:		strncpy(soc_type, "CA_CHANNEL_FALL", 20); break;
	case SOC_CA_CHANNEL_AVG:		strncpy(soc_type, "CA_CHANNEL_AVG", 20); break;
	case SOC_CYC_CHANNEL:			strncpy(soc_type, "CYC_CHANNEL", 20); break;
	case SOC_DIGITAL_RISING_EDGE:	strncpy(soc_type, "DIGITAL_RISING_EDGE", 20); break;
	case SOC_DIGITAL_FALLING_EDGE:	strncpy(soc_type, "DIGITAL_FALLING_EDGE", 20); break;
	default:						strncpy(soc_type, "UNKNOWN", 20); break;
	}

	matlab_file_write_utf16(file_handle, soc_type, 20);
	matlab_file_write_float(file_handle, &soc_config->start_window);
	matlab_file_write_float(file_handle, &soc_config->finish_window);
	matlab_file_write_float(file_handle, &soc_config->fixed_value);
	matlab_file_write_utf16(file_handle, soc_config->channel_name, SIZEOF_CHANNEL_NAME);
	matlab_file_write_bool(file_handle, &soc_config->aligned);
	matlab_file_write_bool(file_handle, &soc_config->invert);

	end_of_structure_file_position = ftell(file_handle);

	if ((structure_size_file_position >= 0) && (end_of_structure_file_position > structure_size_file_position))
	{
		structure_size = end_of_structure_file_position - structure_size_file_position - 4;

		if (structure_size >= 0)
		{
			if (fseek(file_handle, structure_size_file_position, SEEK_SET) == 0)
			{
				fwrite_int(structure_size, file_handle);

				fseek(file_handle, end_of_structure_file_position, SEEK_SET);
			}
		}
	}
	else
	{
		logmessage(MSG_ERROR, "Unable to determine end of MATLAB file\n%u\n%u\n",
			structure_size_file_position,
			end_of_structure_file_position);
	}
}

void matlab_file_write_offset_config(FILE* file_handle, const ChannelOffsetConfig* offset_config, char* array_name)
{
	int structure_size_file_position;
	unsigned int flags;
	unsigned int array_name_length;
	unsigned int field_name_length;
	int end_of_structure_file_position;
	unsigned int number_of_fields;
	unsigned int total_field_name_length;
	int structure_size;
	int array_flags_subelement;
	char array_type;
	char offset_type[20];

	if ((file_handle == NULL) || (offset_config == NULL) || (array_name == NULL))
	{
		return;
	}

	fwrite_int(miMATRIX, file_handle);

	structure_size_file_position = ftell(file_handle);

	fwrite_int(0, file_handle);			/* We'll go back later and fill in the number of bytes	*/

	/* Array Flags */

	array_type = mxSTRUCT_CLASS;
	flags = mfNONE;

	fwrite_int(miUINT32, file_handle);
	fwrite_int(8, file_handle);

	array_flags_subelement = (int)array_type | flags;

	fwrite_int(array_flags_subelement, file_handle);
	fwrite_int(0, file_handle);

	/* Dimensions Array */

	fwrite_int(miINT32, file_handle);
	fwrite_int(8, file_handle);
	fwrite_int(1, file_handle);
	fwrite_int(1, file_handle);

	/* Array Name */

	validate_variable_name(array_name);

	array_name_length = (unsigned int)strlen(array_name);

	if (array_name_length > SIZEOF_CHANNEL_NAME)
	{
		array_name_length = SIZEOF_CHANNEL_NAME;
	}

	matlab_file_write(file_handle, miINT8, array_name, array_name_length);

	/* Field Name Length */

	field_name_length = 24;

	matlab_file_write(file_handle, miINT32, &field_name_length, 4);

	/* Field Names */

	number_of_fields = 9;

	total_field_name_length = number_of_fields * field_name_length;

	fwrite_int(miINT8, file_handle);
	fwrite_int(total_field_name_length, file_handle);

	fwrite_padded_char("type", field_name_length, file_handle);
	fwrite_padded_char("polytropic_value", field_name_length, file_handle);
	fwrite_padded_char("start_window", field_name_length, file_handle);
	fwrite_padded_char("finish_window", field_name_length, file_handle);
	fwrite_padded_char("fixed_value", field_name_length, file_handle);
	fwrite_padded_char("channel", field_name_length, file_handle);
	fwrite_padded_char("calc_interval", field_name_length, file_handle);
	fwrite_padded_char("window_size", field_name_length, file_handle);
	fwrite_padded_char("truncate", field_name_length, file_handle);

	switch (offset_config->type)
	{
	case OFFSET_NONE:				strncpy(offset_type, "NONE", 20); break;
	case OFFSET_FIXED:				strncpy(offset_type, "FIXED", 20); break;
	case OFFSET_POLYTROPIC_2PT:		strncpy(offset_type, "POLYTROPIC_2PT", 20); break;
	case OFFSET_WINDOW:				strncpy(offset_type, "WINDOW", 20); break;
	case OFFSET_MEAN:				strncpy(offset_type, "MEAN", 20); break;
	case OFFSET_WINDOW_ABSOLUTE:	strncpy(offset_type, "WINDOW_ABSOLUTE", 20); break;
	case OFFSET_POLYTROPIC_3PT:		strncpy(offset_type, "POLYTROPIC_3PT", 20); break;
	case OFFSET_RESULT_CHANNEL:		strncpy(offset_type, "RESULT_CHANNEL", 20); break;
	default:						strncpy(offset_type, "UNKNOWN", 20); break;
	}

	matlab_file_write_utf16(file_handle, offset_type, 20);
	matlab_file_write_float(file_handle, &offset_config->polytropic_index);
	matlab_file_write_float(file_handle, &offset_config->start_window);
	matlab_file_write_float(file_handle, &offset_config->finish_window);
	matlab_file_write_float(file_handle, &offset_config->fixed_value);
	matlab_file_write_utf16(file_handle, offset_config->channel_name, SIZEOF_CHANNEL_NAME);
	unsigned int calc_interval = offset_config->window_size * 2 + 1;
	matlab_file_write_uint32(file_handle, &calc_interval);
	matlab_file_write_uint32(file_handle, &offset_config->window_size);
	matlab_file_write_bool(file_handle, &offset_config->truncate);

	end_of_structure_file_position = ftell(file_handle);

	if ((structure_size_file_position >= 0) && (end_of_structure_file_position > structure_size_file_position))
	{
		structure_size = end_of_structure_file_position - structure_size_file_position - 4;

		if (structure_size >= 0)
		{
			if (fseek(file_handle, structure_size_file_position, SEEK_SET) == 0)
			{
				fwrite_int(structure_size, file_handle);

				fseek(file_handle, end_of_structure_file_position, SEEK_SET);
			}
		}
	}
	else
	{
		logmessage(MSG_ERROR, "Unable to determine end of MATLAB file\n");
	}
}

void matlab_file_write_engine(FILE* file_handle, const FileData* file, char* array_name)
{
	int structure_size_file_position;
	unsigned int flags;
	unsigned int array_name_length;
	unsigned int field_name_length;
	int end_of_structure_file_position;
	unsigned int number_of_fields;
	unsigned int total_field_name_length;
	int structure_size;
	int array_flags_subelement;
	char array_type;
	int dimensions[2];

	if ((file_handle == NULL) || (file == NULL) || (array_name == NULL))
	{
		return;
	}

	fwrite_int(miMATRIX, file_handle);

	structure_size_file_position = ftell(file_handle);

	fwrite_int(0, file_handle);			/* We'll go back later and fill in the number of bytes	*/

	/* Array Flags */

	array_type = mxSTRUCT_CLASS;
	flags = mfNONE;

	fwrite_int(miUINT32, file_handle);
	fwrite_int(8, file_handle);

	array_flags_subelement = (int)array_type | flags;

	fwrite_int(array_flags_subelement, file_handle);
	fwrite_int(0, file_handle);

	/* Dimensions Array */

	fwrite_int(miINT32, file_handle);
	fwrite_int(8, file_handle);
	fwrite_int(1, file_handle);
	fwrite_int(1, file_handle);

	/* Array Name */

	validate_variable_name(array_name);

	array_name_length = (unsigned int)strlen(array_name);

	if (array_name_length > SIZEOF_CHANNEL_NAME)
	{
		array_name_length = SIZEOF_CHANNEL_NAME;
	}

	matlab_file_write(file_handle, miINT8, array_name, array_name_length);

	/* Field Name Length */

	field_name_length = 20;

	matlab_file_write(file_handle, miINT32, &field_name_length, 4);

	/* Field Names */

	number_of_fields = 8;

	total_field_name_length = number_of_fields * field_name_length;

	fwrite_int(miINT8, file_handle);
	fwrite_int(total_field_name_length, file_handle);

	fwrite_padded_char("name", field_name_length, file_handle);
	fwrite_padded_char("bore", field_name_length, file_handle);
	fwrite_padded_char("stroke", field_name_length, file_handle);
	fwrite_padded_char("conrod_length", field_name_length, file_handle);
	fwrite_padded_char("compression_ratio", field_name_length, file_handle);
	fwrite_padded_char("number_of_cylinders", field_name_length, file_handle);
	fwrite_padded_char("number_of_strokes", field_name_length, file_handle);
	fwrite_padded_char("pin_offset", field_name_length, file_handle);

	matlab_file_write_utf16(file_handle, file->engine.name, SIZEOF_ENGINE_NAME);
	matlab_file_write_float(file_handle, &file->engine.bore);
	matlab_file_write_float(file_handle, &file->engine.stroke);
	matlab_file_write_float(file_handle, &file->engine.conrod_length);
	matlab_file_write_float(file_handle, &file->engine.compression_ratio);
	matlab_file_write_uint32(file_handle, &file->engine.number_of_cylinders);
	matlab_file_write_uint32(file_handle, &file->engine.number_of_strokes);
	
	dimensions[0] = (int)file->engine.number_of_cylinders;
	dimensions[1] = 1;

	matlab_file_write_numeric_array(file_handle, miSINGLE, false, false, "pin_offset", 2, dimensions, &file->engine.pin_offset, NULL);
	
	end_of_structure_file_position = ftell(file_handle);

	if ((structure_size_file_position >= 0) && (end_of_structure_file_position > structure_size_file_position))
	{
		structure_size = end_of_structure_file_position - structure_size_file_position - 4;

		if (structure_size >= 0)
		{
			if (fseek(file_handle, structure_size_file_position, SEEK_SET) == 0)
			{
				fwrite_int(structure_size, file_handle);

				fseek(file_handle, end_of_structure_file_position, SEEK_SET);
			}
		}
	}
	else
	{
		logmessage(MSG_ERROR, "Unable to determine end of MATLAB file\n");
	}
}

void matlab_file_write_header(FILE* file_handle, const FileData* file, char* array_name)
{
	int structure_size_file_position;
	unsigned int flags;
	unsigned int array_name_length;
	unsigned int field_name_length;
	int end_of_structure_file_position;
	unsigned int number_of_fields;
	unsigned int total_field_name_length;
	int structure_size;
	int array_flags_subelement;
	char array_type;
	char date_output[24];

	if ((file_handle == NULL) || (file == NULL) || (array_name == NULL))
	{
		return;
	}

	fwrite_int(miMATRIX, file_handle);

	structure_size_file_position = ftell(file_handle);

	fwrite_int(0, file_handle);			/* We'll go back later and fill in the number of bytes	*/

	/* Array Flags */

	array_type = mxSTRUCT_CLASS;
	flags = mfNONE;

	fwrite_int(miUINT32, file_handle);
	fwrite_int(8, file_handle);

	array_flags_subelement = (int)array_type | flags;

	fwrite_int(array_flags_subelement, file_handle);
	fwrite_int(0, file_handle);

	/* Dimensions Array */

	fwrite_int(miINT32, file_handle);
	fwrite_int(8, file_handle);
	fwrite_int(1, file_handle);
	fwrite_int(1, file_handle);

	/* Array Name */

	validate_variable_name(array_name);

	array_name_length = (unsigned int)strlen(array_name);

	if (array_name_length > SIZEOF_CHANNEL_NAME)
	{
		array_name_length = SIZEOF_CHANNEL_NAME;
	}

	matlab_file_write(file_handle, miINT8, array_name, array_name_length);

	/* Field Name Length */

	field_name_length = 24;

	matlab_file_write(file_handle, miINT32, &field_name_length, 4);

	/* Field Names */

	number_of_fields = 3;

	total_field_name_length = number_of_fields * field_name_length;

	fwrite_int(miINT8, file_handle);
	fwrite_int(total_field_name_length, file_handle);

	fwrite_padded_char("filename", field_name_length, file_handle);
	fwrite_padded_char("comment", field_name_length, file_handle);
	fwrite_padded_char("date", field_name_length, file_handle);

	create_time(&file->creation_time, date_output);

#ifdef _CATOOL_UNICODE_
	matlab_file_write_wchar_numeric_array(file_handle, file->filename, 4096);
#else
	matlab_file_write_utf16(file_handle, file->filename, 4096);
#endif
 	matlab_file_write_utf16(file_handle, file->comment, SIZEOF_DESCRIPTION);
	matlab_file_write_utf16(file_handle, date_output, 24);

	end_of_structure_file_position = ftell(file_handle);

	if ((structure_size_file_position >= 0) && (end_of_structure_file_position > structure_size_file_position))
	{
		structure_size = end_of_structure_file_position - structure_size_file_position - 4;

		if (structure_size >= 0)
		{
			if (fseek(file_handle, structure_size_file_position, SEEK_SET) == 0)
			{
				fwrite_int(structure_size, file_handle);

				fseek(file_handle, end_of_structure_file_position, SEEK_SET);
			}
		}
	}
	else
	{
		logmessage(MSG_ERROR, "Unable to determine end of MATLAB file\n");
	}
}

void matlab_file_write_parameter(FILE* file_handle, const Parameter* parameter, char* array_name)
{
	int structure_size_file_position;
	unsigned int flags;
	unsigned int array_name_length;
	unsigned int field_name_length;
	int end_of_structure_file_position;
	unsigned int number_of_fields;
	unsigned int total_field_name_length;
	int structure_size;
	int array_flags_subelement;
	char array_type;

	if (file_handle == NULL)
	{
		return;
	}

	if (parameter == NULL)
	{
		return;
	}

	fwrite_int(miMATRIX, file_handle);

	structure_size_file_position = ftell(file_handle);

	fwrite_int(0, file_handle);			/* We'll go back later and fill in the number of bytes	*/

	/* Array Flags */

	array_type = mxSTRUCT_CLASS;
	flags = mfNONE;

	fwrite_int(miUINT32, file_handle);
	fwrite_int(8, file_handle);

	array_flags_subelement = (int)array_type | flags;

	fwrite_int(array_flags_subelement, file_handle);
	fwrite_int(0, file_handle);

	/* Dimensions Array */

	fwrite_int(miINT32, file_handle);
	fwrite_int(8, file_handle);
	fwrite_int(1, file_handle);
	fwrite_int(1, file_handle);

	/* Array Name */

	validate_variable_name(array_name);

	array_name_length = (unsigned int)strlen(array_name);

	if (array_name_length > SIZEOF_CHANNEL_NAME)
	{
		array_name_length = SIZEOF_CHANNEL_NAME;
	}

	matlab_file_write(file_handle, miINT8, array_name, array_name_length);

	/* Field Name Length */

	field_name_length = 20;

	matlab_file_write(file_handle, miINT32, &field_name_length, 4);

	/* Field Names */

	number_of_fields = 4;

	total_field_name_length = number_of_fields * field_name_length;

	fwrite_int(miINT8, file_handle);
	fwrite_int(total_field_name_length, file_handle);

	fwrite_padded_char("name", field_name_length, file_handle);
	fwrite_padded_char("value", field_name_length, file_handle);
	fwrite_padded_char("description", field_name_length, file_handle);
	fwrite_padded_char("units", field_name_length, file_handle);

	matlab_file_write_utf16(file_handle, parameter->name, SIZEOF_CHANNEL_NAME);
	matlab_file_write_double(file_handle, (double*)parameter->value);
	matlab_file_write_utf16(file_handle, parameter->description, SIZEOF_DESCRIPTION);
	matlab_file_write_utf16(file_handle, parameter->units, SIZEOF_UNITS);

	end_of_structure_file_position = ftell(file_handle);

	if ((structure_size_file_position >= 0) && (end_of_structure_file_position > structure_size_file_position))
	{
		structure_size = end_of_structure_file_position - structure_size_file_position - 4;

		if (structure_size >= 0)
		{
			if (fseek(file_handle, structure_size_file_position, SEEK_SET) == 0)
			{
				fwrite_int(structure_size, file_handle);

				fseek(file_handle, end_of_structure_file_position, SEEK_SET);
			}
		}
	}
	else
	{
		logmessage(MSG_ERROR, "Unable to determine end of MATLAB file\n");
	}
}

void matlab_file_write_parameters(FILE* file_handle, const FileData* file, char* array_name)
{
	int structure_size_file_position;
	unsigned int flags;
	unsigned int array_name_length;
	unsigned int field_name_length;
	int end_of_structure_file_position;
	unsigned int total_field_name_length;
	int structure_size;
	int array_flags_subelement;
	char array_type;
	unsigned int number_of_parameters = 0;
	char parameter_name[SIZEOF_CHANNEL_NAME];

	if (file_handle == NULL)
	{
		return;
	}

	Parameter* parameter = file->parameters;

	while (parameter != NULL)
	{
		if ((parameter->data_type == VARIABLE_DOUBLE) && (parameter->name[0] != 0x00))
		{
			number_of_parameters++;
		}

		parameter = parameter->next;
	}

	if (number_of_parameters == 0)
	{
		return;
	}

	fwrite_int(miMATRIX, file_handle);

	structure_size_file_position = ftell(file_handle);

	fwrite_int(0, file_handle);			/* We'll go back later and fill in the number of bytes	*/

	/* Array Flags */

	array_type = mxSTRUCT_CLASS;
	flags = mfNONE;

	fwrite_int(miUINT32, file_handle);
	fwrite_int(8, file_handle);

	array_flags_subelement = (int)array_type | flags;

	fwrite_int(array_flags_subelement, file_handle);
	fwrite_int(0, file_handle);

	/* Dimensions Array */

	fwrite_int(miINT32, file_handle);
	fwrite_int(8, file_handle);
	fwrite_int(1, file_handle);
	fwrite_int(1, file_handle);

	/* Array Name */

	validate_variable_name(array_name);

	array_name_length = (unsigned int)strlen(array_name);

	if (array_name_length > SIZEOF_CHANNEL_NAME)
	{
		array_name_length = SIZEOF_CHANNEL_NAME;
	}

	matlab_file_write(file_handle, miINT8, array_name, array_name_length);

	/* Field Name Length */

	field_name_length = 64;

	matlab_file_write(file_handle, miINT32, &field_name_length, 4);

	/* Field Names */

	total_field_name_length = number_of_parameters * field_name_length;

	fwrite_int(miINT8, file_handle);
	fwrite_int(total_field_name_length, file_handle);

	parameter = file->parameters;

	while (parameter != NULL)
	{
		if ((parameter->data_type == VARIABLE_DOUBLE) && (parameter->name[0] != 0x00))
		{
			strncpy(parameter_name, parameter->name, SIZEOF_CHANNEL_NAME);

			validate_variable_name(parameter_name);

			fwrite_padded_char(parameter_name, field_name_length, file_handle);
		}

		parameter = parameter->next;
	}

	parameter = file->parameters;

	while (parameter != NULL)
	{
		if ((parameter->data_type == VARIABLE_DOUBLE) && (parameter->name[0] != 0x00))
		{
			//matlab_file_write_double(file_handle,(double*)parameter->value);
			matlab_file_write_parameter(file_handle, parameter, parameter->name);
		}

		parameter = parameter->next;
	}

	end_of_structure_file_position = ftell(file_handle);

	if ((structure_size_file_position >= 0) && (end_of_structure_file_position > structure_size_file_position))
	{
		structure_size = end_of_structure_file_position - structure_size_file_position - 4;

		if (structure_size >= 0)
		{
			if (fseek(file_handle, structure_size_file_position, SEEK_SET) == 0)
			{
				fwrite_int(structure_size, file_handle);

				fseek(file_handle, end_of_structure_file_position, SEEK_SET);
			}
		}
	}
	else
	{
		logmessage(MSG_ERROR, "Unable to determine end of MATLAB file\n");
	}
}

void matlab_file_write_channel(FILE* file_handle, const FileData* file, const unsigned int number_of_cycles, ChannelData* channel_data)
{
	int structure_size_file_position;
	unsigned int flags;
	unsigned int array_name_length;
	unsigned int field_name_length;
	int end_of_structure_file_position;
	unsigned int number_of_fields;
	unsigned int total_field_name_length;
	int structure_size;
	unsigned int data_dimension_0;
	unsigned int data_dimension_1;
	unsigned int axis_dimension_0;
	unsigned int axis_dimension_1;
	int dimensions[2];
	int array_flags_subelement;
	char array_type;
	float* axis_f = NULL;
	double* axis_d = NULL;
	char axis_type[20];
	unsigned int data_pointer;

	if ((file == NULL) || (file_handle == NULL) || (channel_data == NULL))
	{
		return;
	}

	if (channel_data->samples_per_cycle == 0)
	{
		return;
	}

	/* Data */

	data_dimension_0 = channel_data->samples_per_cycle;

	if ((channel_data->abscissa.type == ABSCISSA_CRANKANGLE) || (channel_data->abscissa.type == ABSCISSA_FREQUENCY))
	{
		data_dimension_1 = number_of_cycles;
	}
	else
	{
		data_dimension_1 = 1;
	}

	/* Axis */

	axis_dimension_0 = data_dimension_0;
	axis_dimension_1 = 1;

	switch (channel_data->abscissa.type)
	{
	case ABSCISSA_CRANKANGLE:
	{
		axis_f = (float*)malloc(axis_dimension_0 * sizeof(float));

		if (axis_f == NULL)
		{
			fclose(file_handle);
			logmessage(FATAL, "Memory could not be allocated\n");
			return;
		}

		if (channel_data->abscissa.axis != NULL)
		{
			GetAxis(&channel_data->abscissa, VARIABLE_FLOAT, axis_f, axis_dimension_0);
		}
		else
		{
			for (data_pointer = 0; data_pointer < axis_dimension_0; data_pointer++)
			{
				axis_f[data_pointer] = channel_data->abscissa.ca_to_deg[data_pointer];
			}
		}
		
		strncpy(axis_type, "CRANKANGLE", 20);

		break;
	}
	case ABSCISSA_TIME:
	{
		axis_d = (double*)malloc(axis_dimension_0 * sizeof(double));

		if (axis_d == NULL)
		{
			fclose(file_handle);
			logmessage(FATAL, "Memory could not be allocated\n");
			return;
		}

		if (channel_data->abscissa.axis != NULL)
		{
			GetAxis(&channel_data->abscissa, VARIABLE_DOUBLE, axis_d, axis_dimension_0);
		}
		else
		{
			for (data_pointer = 0; data_pointer < axis_dimension_0; data_pointer++)
			{
				axis_d[data_pointer] = (double)data_pointer * file->time_resolution * (double)channel_data->abscissa.resolution[0];		/* ms */
			}
		}

		strncpy(axis_type, "TIME", 20);

		break;
	}
	case ABSCISSA_CYCLE:
	{
		axis_f = (float*)malloc(axis_dimension_0 * sizeof(float));

		if (axis_f == NULL)
		{
			fclose(file_handle);
			logmessage(FATAL, "Memory could not be allocated\n");
			return;
		}

		unsigned int start = channel_data->abscissa.start[0];

		for (data_pointer = 0; data_pointer < axis_dimension_0; data_pointer++)
		{
			axis_f[data_pointer] = (float)(data_pointer + start);
		}

		strncpy(axis_type, "CYCLE", 20);

		break;
	}
	case ABSCISSA_FREQUENCY:
	{
		axis_f = (float*)malloc(axis_dimension_0 * sizeof(float));

		if (axis_f == NULL)
		{
			fclose(file_handle);
			logmessage(FATAL, "Memory could not be allocated\n");
			return;
		}

		for (data_pointer = 0; data_pointer < axis_dimension_0; data_pointer++)
		{
			axis_f[data_pointer] = (float)data_pointer;
		}

		strncpy(axis_type, "FREQUENCY", 20);

		break;
	}
	default:
	{
		axis_f = (float*)malloc(axis_dimension_0 * sizeof(float));

		if (axis_f == NULL)
		{
			fclose(file_handle);
			logmessage(FATAL, "Memory could not be allocated\n");
			return;
		}

		for (data_pointer = 0; data_pointer < axis_dimension_0; data_pointer++)
		{
			axis_f[data_pointer] = (float)data_pointer;
		}

		strncpy(axis_type, "UNKNOWN", 20);

		break;
	}
	}

	axis_type[19] = 0x00;

	/* Output structure */

	fwrite_int(miMATRIX,file_handle);

	structure_size_file_position = ftell(file_handle);
	
	fwrite_int(0,file_handle);			/* We'll go back later and fill in the number of bytes	*/

	/* Array Flags */

	array_type = mxSTRUCT_CLASS;
	flags = mfNONE;

	fwrite_int(miUINT32,file_handle);
	fwrite_int(8,file_handle);
	
	array_flags_subelement = (int)array_type | flags;
	
	fwrite_int(array_flags_subelement,file_handle);
    fwrite_int(0,file_handle);
	
	/* Dimensions Array */

	fwrite_int(miINT32,file_handle);
	fwrite_int(8,file_handle);
	fwrite_int(1,file_handle);
	fwrite_int(1,file_handle);

	/* Array Name */

	validate_variable_name(channel_data->matlab_name);

	array_name_length = (unsigned int)strlen(channel_data->matlab_name);

	if (array_name_length > SIZEOF_CHANNEL_NAME)
	{
		array_name_length = SIZEOF_CHANNEL_NAME;
	}

	matlab_file_write(file_handle,miINT8, channel_data->matlab_name,array_name_length);

	/* Field Name Length */

	field_name_length = 32;
	
	matlab_file_write(file_handle,miINT32,&field_name_length,4);

	/* Field Names */

	number_of_fields = 12;

	total_field_name_length = number_of_fields * field_name_length;

	fwrite_int(miINT8,file_handle);
	fwrite_int(total_field_name_length,file_handle);

	fwrite_padded_char("data",field_name_length,file_handle);
	fwrite_padded_char("axis",field_name_length,file_handle);
	fwrite_padded_char("units",field_name_length,file_handle);
	fwrite_padded_char("description",field_name_length,file_handle);
	fwrite_padded_char("tdc_offset",field_name_length,file_handle);
	fwrite_padded_char("axis_units", field_name_length, file_handle);
	fwrite_padded_char("axis_type", field_name_length, file_handle);
	fwrite_padded_char("original_name", field_name_length, file_handle);
	fwrite_padded_char("type", field_name_length, file_handle);
	fwrite_padded_char("cylinder", field_name_length, file_handle);
	fwrite_padded_char("offset_config", field_name_length, file_handle);
	fwrite_padded_char("soc_config", field_name_length, file_handle);

	dimensions[0] = data_dimension_0;
	dimensions[1] = data_dimension_1;

	if (channel_data->data_d != NULL)
	{
		matlab_file_write_numeric_array(file_handle, miDOUBLE, false, false, NULL, 2, dimensions, channel_data->data_d, NULL);
	}
	else
	{
		matlab_file_write_numeric_array(file_handle, miSINGLE, false, false, NULL, 2, dimensions, channel_data->data, NULL);
	}

	dimensions[0] = axis_dimension_0;
	dimensions[1] = axis_dimension_1;

	if (axis_d != NULL)
	{
		matlab_file_write_numeric_array(file_handle, miDOUBLE, false, false, NULL, 2, dimensions, axis_d, NULL);

		free(axis_d);
		axis_d = NULL;
	}
	else if (axis_f != NULL)
	{
		matlab_file_write_numeric_array(file_handle, miSINGLE, false, false, NULL, 2, dimensions, axis_f, NULL);

		free(axis_f);
		axis_f = NULL;
	}
	else
	{
		logmessage(MSG_ERROR, "No axis data\n");
	}

	matlab_file_write_utf16(file_handle, channel_data->units, SIZEOF_UNITS);

	matlab_file_write_utf16(file_handle, channel_data->description, SIZEOF_DESCRIPTION);

	matlab_file_write_float(file_handle, &channel_data->tdc_offset);

	matlab_file_write_utf16(file_handle, channel_data->abscissa.units, SIZEOF_UNITS);

	matlab_file_write_utf16(file_handle, axis_type, 20);

	matlab_file_write_utf16(file_handle, channel_data->name, SIZEOF_DESCRIPTION);

	matlab_file_write_utf16(file_handle, channel_information[channel_data->type].name, SIZEOF_CHANNEL_NAME);

	matlab_file_write_uint32(file_handle, &channel_data->cylinder);

	matlab_file_write_offset_config(file_handle, &channel_data->offset_config, (char*)"offset_config");

	matlab_file_write_soc_config(file_handle, &channel_data->soc_config, (char*)"soc_config");

	end_of_structure_file_position = ftell(file_handle);
	
	if ((structure_size_file_position >= 0) && (end_of_structure_file_position > structure_size_file_position))
	{
		structure_size = end_of_structure_file_position - structure_size_file_position - 4;

		if (structure_size >= 0)
		{
			if (fseek(file_handle, structure_size_file_position, SEEK_SET) != 0)
			{
				logmessage(MSG_ERROR, "Unable to rewind file to structure size position\n");
			}
			else
			{
				fwrite_int(structure_size, file_handle);
				if (fseek(file_handle, end_of_structure_file_position, SEEK_SET) != 0)
				{
					logmessage(MSG_ERROR, "Unable to move file position to end of structure\n");
				}
			}
		}
	}
	else
	{
		logmessage(MSG_ERROR,"Unable to determine end of MATLAB file\n");
	}
}

#ifdef _CATOOL_UNICODE_
bool save_matlab_file(FileData* file, const wchar_t* filename, Analysis* analysis, const unsigned int ifile_output)
#else
bool save_matlab_file(FileData* file, const char* filename, Analysis* analysis, const unsigned int ifile_output)
#endif
{
	FILE* matfile = NULL;
	unsigned int channel = 0;
	unsigned int chan;
	unsigned int analysis_channel;
	unsigned int analysis_number;
	bool output_raw_channels = false;
	bool output_analysis = false;
	bool output_engine_information = true;
	bool output_cylinder_volume = true;
	bool output_ifile_parameters = true;
	bool output_header = true;
	bool found = false;
	ChannelData volume_channel;
	unsigned int data_pointer;
		
	if (((ifile_output & IFILE_CA_RAW) > 0) || 
		((ifile_output & IFILE_RESULTS_RAW) > 0) ||
		((ifile_output & IFILE_TIME) > 0))
	{
		output_raw_channels = true;
	}
	
	if ((analysis != NULL) && 
		(analysis->number_of_channels > 0) && 
		(((ifile_output & IFILE_CA) > 0) || ((ifile_output & IFILE_RESULTS) > 0)))
	{
		output_analysis = true;
	}
	
	if ((file->number_of_channels > 0) && (file->channel_data == NULL))
	{
		logmessage(MSG_ERROR,"Channels are configured but no data is present\n");

		return(false);
	}

	logmessage(DEBUG,"Generating MATLAB compatible channels names\n");
	generate_matlab_channelnames(file,analysis);

#ifdef _CATOOL_UNICODE_
	matfile = _wfopen(filename, L"wb");
#else
	matfile = fopen(filename,"wb");
#endif
	
	if ( matfile == NULL )
	{
		logmessage(MSG_ERROR,"MATLAB output file could not be opened\n");
		
		return(false);
	}
			
	matlab_file_open(matfile);

	if (output_header == true)
	{
		logmessage(DEBUG, "Outputting header\n");

		matlab_file_write_header(matfile, file, (char*)"header");
	}

	if (output_engine_information == true)
	{
		logmessage(DEBUG,"Outputting engine information\n");
		
		matlab_file_write_engine(matfile, file, (char*)"engine");
	}

	if (output_ifile_parameters == true)
	{
		logmessage(DEBUG, "Outputting parameter values\n");

		matlab_file_write_parameters(matfile, file, (char*)"parameters");
	}

	if (output_cylinder_volume == true)
	{
		logmessage(DEBUG, "Outputting cylinder volume information\n");

		for (chan = 0; chan < file->number_of_channels; chan++)
		{
			if ((found == false) && (file->channel_data[chan].loaded == true) && (file->channel_data[chan].abscissa.type == ABSCISSA_CRANKANGLE))
			{
				found = true;
				channel = chan;
			}
		}

		if (found == true)
		{
			strncpy(volume_channel.name, "V_ABS", SIZEOF_CHANNEL_NAME);
			volume_channel.name[SIZEOF_CHANNEL_NAME - 1] = 0x00;

			strncpy(volume_channel.short_name, "V_ABS", SIZEOF_SHORT_NAME);
			volume_channel.short_name[SIZEOF_SHORT_NAME - 1] = 0x00;

			strncpy(volume_channel.matlab_name, "V_ABS", SIZEOF_CHANNEL_NAME);
			volume_channel.matlab_name[SIZEOF_CHANNEL_NAME - 1] = 0x00;

			strncpy(volume_channel.units, "cm3", SIZEOF_UNITS);
			volume_channel.units[SIZEOF_UNITS - 1] = 0x00;

			strncpy(volume_channel.description, "Cylinder Volume", SIZEOF_DESCRIPTION);
			volume_channel.description[SIZEOF_DESCRIPTION - 1] = 0x00;

			strncpy(volume_channel.abscissa.units, "deg", SIZEOF_UNITS);
			volume_channel.name[SIZEOF_UNITS - 1] = 0x00;

			volume_channel.abscissa.type = ABSCISSA_CRANKANGLE;

			volume_channel.type = CHAN_VOLUME;

			volume_channel.abscissa.ca_to_theta = NULL;
			volume_channel.abscissa.ca_to_deg = file->theta_to_deg;
			volume_channel.abscissa.theta_to_ca = NULL;
			volume_channel.abscissa.theta_weighting = NULL;
			volume_channel.abscissa.theta_weighting_inv = NULL;
			volume_channel.abscissa.volume = NULL;
			volume_channel.abscissa.axis = NULL;
			volume_channel.abscissa.slope = 1.0;
			volume_channel.abscissa.offset = 0.0;
			volume_channel.abscissa.axis_type = VARIABLE_NONE;
			volume_channel.abscissa.number_of_measurement_tables = 0;
			volume_channel.abscissa.number_of_samples = NULL;
			volume_channel.abscissa.start = NULL;
			volume_channel.abscissa.resolution = NULL;

			volume_channel.cylinder = 1;
			volume_channel.tdc_offset = 0.0f;

			volume_channel.samples_per_cycle = MaximumTheta(file, channel);

			volume_channel.offset_config.window_size = 0;
			volume_channel.offset_config.channel_name[0] = 0x00;
			volume_channel.offset_config.finish_window = 0.0f;
			volume_channel.offset_config.fixed_value = 0.0f;
			volume_channel.offset_config.polytropic_index = 0.0f;
			volume_channel.offset_config.start_window = 0.0f;
			volume_channel.offset_config.truncate = false;
			volume_channel.offset_config.type = OFFSET_NONE;

			volume_channel.soc_config.aligned = false;
			volume_channel.soc_config.channel_name[0] = 0x00;
			volume_channel.soc_config.finish_window = 0.0f;
			volume_channel.soc_config.fixed_value = 0.0f;
			volume_channel.soc_config.invert = false;
			volume_channel.soc_config.start_window = 0.0f;
			volume_channel.soc_config.type = SOC_INVALID;

			volume_channel.data = (float*)malloc(volume_channel.samples_per_cycle * sizeof(float));

			if (volume_channel.data == NULL)
			{
				fclose(matfile);
				logmessage(FATAL, "Memory could not be allocated\n");
				return(false);
			}

			volume_channel.data_d = NULL;
			volume_channel.conversion_factor = 1.0f;
			volume_channel.offset = 0.0;
			volume_channel.slope = 1.0;
			volume_channel.min = 0.0;
			volume_channel.max = 0.0;
			volume_channel.duration = NULL;
			volume_channel.isoffset = false;
			volume_channel.loaded = true;
			volume_channel.file_flag = true;

			for (data_pointer = 0; data_pointer < volume_channel.samples_per_cycle; data_pointer++)
			{
				volume_channel.data[data_pointer] = file->channel_data[channel].abscissa.volume[data_pointer] * CUBIC_MM_TO_CM;
			}

			matlab_file_write_channel(matfile, file, 1, &volume_channel);

			strncpy(volume_channel.name, "V", SIZEOF_CHANNEL_NAME);
			volume_channel.name[SIZEOF_CHANNEL_NAME - 1] = 0x00;

			strncpy(volume_channel.matlab_name, "V", SIZEOF_CHANNEL_NAME);
			volume_channel.matlab_name[SIZEOF_CHANNEL_NAME - 1] = 0x00;

			strncpy(volume_channel.units, "", SIZEOF_UNITS);
			volume_channel.units[SIZEOF_UNITS - 1] = 0x00;

			for (data_pointer = 0; data_pointer < volume_channel.samples_per_cycle; data_pointer++)
			{
				volume_channel.data[data_pointer] = file->channel_data[channel].abscissa.volume[data_pointer] / file->engine.swept_volume;
			}

			matlab_file_write_channel(matfile, file, 1, &volume_channel);

			free(volume_channel.data);
		}
	}

	if (output_raw_channels == true)
	{
		logmessage(DEBUG, "Outputting raw channels\n");

		for (channel=0;channel<file->number_of_channels;channel++)
		{
			if (file->channel_data[channel].loaded == true)
			{
				matlab_file_write_channel(matfile, file, file->number_of_cycles, &file->channel_data[channel]);
			}
		}
	}
	
	if (output_analysis == true)
	{
		logmessage(DEBUG, "Outputting analysis\n");

		for (analysis_number=0;analysis_number<analysis->number_of_channels;analysis_number++)
		{
			for (analysis_channel=0;analysis_channel<get_number_of_analysis_channels();analysis_channel++)
			{
				if ((analysis->channel[analysis_number].analysis_rqst[analysis_channel] >= AnalysisRqst::CalculateAndStore) &&
					(((analysis->channel[analysis_number].results[analysis_channel].abscissa.type == ABSCISSA_CRANKANGLE) && ((ifile_output & IFILE_CA) > 0)) ||
					 ((analysis->channel[analysis_number].results[analysis_channel].abscissa.type == ABSCISSA_CYCLE) && ((ifile_output & IFILE_RESULTS) > 0)) ||
				     ((analysis->channel[analysis_number].results[analysis_channel].abscissa.type == ABSCISSA_TIME) && ((ifile_output & IFILE_RESULTS) > 0)) ||
				     ((analysis->channel[analysis_number].results[analysis_channel].abscissa.type == ABSCISSA_FREQUENCY) && ((ifile_output & IFILE_RESULTS) > 0))))
				{
					matlab_file_write_channel(matfile, file, file->number_of_cycles, &analysis->channel[analysis_number].results[analysis_channel]);
				}
			}
		}
	}

	logmessage(DEBUG, "Closing matfile\n");

	matlab_file_close(matfile);
	
	return(true);
}
