/* Combustion Analysis Tool (CAT)
   www.catool.org
   
   Filename: ifile.c

   Purpose:  Provide common programming interface for processing acquired engine data
  
   Author:   Ben Brown
   Version:  1.2
   Date:     19.10.2015

   Revision: GPL Release

   Copyright (C) Xarin Limited, 2000-2024

      This program is free software: you can redistribute it and/or modify
      it under the terms of version 2 of the GNU General Public License as
      published by the Free Software Foundation.

      This program is distributed in the hope that it will be useful,
      but WITHOUT ANY WARRANTY; without even the implied warranty of
      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
      GNU General Public License for more details.

      You should have received a copy of the GNU General Public License
      along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "cat.h"

extern unsigned int debug_level;

static short avl_signal_type(const unsigned int);
static void *float_to_short_thread(void* arguments);
static void float_to_short(const FileData* file,float* data,short* buffer,unsigned int number_of_datapoints,double kalfak_0,double kalfak_1);

unsigned int determine_cylinder_number(char* channel_name, unsigned int* max_cylinder_number)
{
	size_t signam_len;
	unsigned int position;
	unsigned int number_length;
	unsigned int cylinder_number;
	char cylinder_number_string[10];
	unsigned int max_cylinder = 0;

	if (channel_name == NULL)
	{
		return(1);
	}

	/* Determine cylinder number */

	signam_len = strnlen((char*)channel_name, SIZEOF_CHANNEL_NAME);

	position = (unsigned int)signam_len - 1;
	number_length = 0;

	while ((position > 0) && (channel_name[position] >= '0') && (channel_name[position] <= '9'))
	{
		number_length += 1;
		position -= 1;
	}

	if ((strncmp((char*)channel_name, "MFB", 3) == 0) && (number_length > 2))
	{
		if (signam_len == 5)
		{
			number_length = 1;
		}
		else if (signam_len == 6)
		{
			number_length = 1;
		}
		else
		{
			number_length = 2;
		}
	}

	if (number_length == 0)
	{
		cylinder_number = 1;
	}
	else
	{
		for (position = 0; position < number_length; position++)
		{
			cylinder_number_string[position] = channel_name[signam_len - number_length + position];
		}
		cylinder_number_string[number_length] = '\0';

		if (number_length > 2)
		{
			cylinder_number_string[0] = cylinder_number_string[number_length - 2];
			cylinder_number_string[1] = cylinder_number_string[number_length - 1];
			cylinder_number_string[2] = cylinder_number_string[number_length];
		}

		if (sscanf(cylinder_number_string, "%2u", &cylinder_number) == 1)
		{
			if ((cylinder_number > 0) && (cylinder_number <= MAX_NUMBER_OF_CYLINDERS))
			{
				if (cylinder_number > max_cylinder)
				{
					max_cylinder = cylinder_number;
				}
			}
			else
			{
				while (cylinder_number > 10)
				{
					cylinder_number -= 10;
				}
			}
		}
		else
		{
			cylinder_number = 1;
		}
	}

	if (max_cylinder_number != NULL)
	{
		if (max_cylinder > *max_cylinder_number)
		{
			*max_cylinder_number = max_cylinder;
		}
	}

	if (cylinder_number == 0)
	{
		cylinder_number = 1;
	}

	return(cylinder_number);
}

short avl_offset_type(const unsigned int offset_type)
{
	short return_value = 0;

	switch (offset_type)
	{
		case OFFSET_NONE:
		case OFFSET_RESULT_CHANNEL:
		{
			return_value = 0;
			break;
		}
		case OFFSET_MEAN:
		{
			return_value = 1;
			break;
		}
		case OFFSET_POLYTROPIC_2PT:
		case OFFSET_POLYTROPIC_3PT:
		{
			return_value = 2;
			break;
		}
		case OFFSET_FIXED:
		{
			return_value = 3;
			break;
		}
		case OFFSET_WINDOW:
		{
			return_value = 4;
			/* If referencing refzyl 4
			                  refsau 5
							  refaus 6 */
			break;
		}
		default:
		{
			return_value = 0;
			break;
		}
	}

	return(return_value);
}

static short avl_signal_type(const unsigned int channel_type)
{
	short return_value = 0;
	
	switch (channel_type)
	{
		case CHAN_CYL_PR:
		{
			return_value = 1;
			break;
		}
		case CHAN_INJ_PR:
		{
			return_value = 2;
			break;
		}
		case CHAN_NEEDLE_LIFT:
		{
			return_value = 3;
			break;
		}
		case CHAN_IGN_ANG:
		{
			return_value = 1;
			break;
		}
		case CHAN_BLOCK_ACC:
		{
			return_value = 14;
			break;
		}
		case CHAN_INLET_PR:
		{
			return_value = 16;
			break;
		}
		case CHAN_EXH_PR:
		{
			return_value = 17;
			break;
		}
		case CHAN_TOOTH_PERIODS:
		{
			return_value = 22;
			break;
		}
		default:
		{
			return_value = 0;
			break;
		}
	}
	
	return(return_value);
}

void generate_short_ifile_channelnames(FileData* file,Analysis* analysis)
{
	unsigned int channel;
	unsigned int analysis_number;
	unsigned int analysis_channel;
	unsigned int num;
	bool ok;
	unsigned int number;
	unsigned int chan;
	char short_name[SIZEOF_SHORT_NAME];
	char new_channel_name[SIZEOF_SHORT_NAME];

	if (file != NULL)
	{
		for (channel=0;channel<file->number_of_channels;channel++)
		{
			if ((file->channel_data[channel].file_flag == true) && (file->channel_data[channel].loaded == true) && (strlen(file->channel_data[channel].name) >= SIZEOF_SHORT_NAME))
			{
				if (file->channel_data[channel].short_name[0] == 0x00)
				{
					ok = false;
					number = 1;
					while (ok == false)
					{
						strncpy(short_name, file->channel_data[channel].name, SIZEOF_SHORT_NAME);

						if (number < 10)
						{
							short_name[7] = 0x00;
						}
						else if (number < 100)
						{
							short_name[6] = 0x00;
						}
						else if (number < 1000)
						{
							short_name[5] = 0x00;
						}
						else if (number < 10000)
						{
							short_name[4] = 0x00;
						}
						else if (number < 100000)
						{
							short_name[3] = 0x00;
						}
						else if (number < 1000000)
						{
							short_name[2] = 0x00;
						}
						else if (number < 10000000)
						{
							short_name[1] = 0x00;
						}
						else
						{
							short_name[0] = 0x00;
						}

						snprintf(new_channel_name,SIZEOF_SHORT_NAME,"%s~%u",short_name,number);
						new_channel_name[SIZEOF_SHORT_NAME-1] = 0x00;

						ok = true;
						for (chan=0;chan<file->number_of_channels;chan++)
						{
							if (strncmp(new_channel_name,file->channel_data[chan].short_name,SIZEOF_SHORT_NAME) == 0)
							{
								ok = false;
								number += 1;
							}
						}

						if (number > 9999)
						{
							ok = true;

							logmessage(WARNING,"Too many duplicate channel names");
						}
					}

					memset(file->channel_data[channel].short_name, 0, SIZEOF_SHORT_NAME);

					strncpy(file->channel_data[channel].short_name,new_channel_name, SIZEOF_SHORT_NAME);

					file->channel_data[channel].short_name[SIZEOF_SHORT_NAME - 1] = 0x00;
				}
			}
			else
			{
				memset(file->channel_data[channel].short_name, 0, SIZEOF_SHORT_NAME);

				strncpy(file->channel_data[channel].short_name,file->channel_data[channel].name, SIZEOF_SHORT_NAME);

				file->channel_data[channel].short_name[SIZEOF_SHORT_NAME - 1] = 0x00;
			}
		}
	}

	if (analysis != NULL)
	{
		for (analysis_number=0;analysis_number<analysis->number_of_channels;analysis_number++)
		{
			for (analysis_channel=0;analysis_channel<get_number_of_analysis_channels();analysis_channel++)
			{
				if (analysis->channel[analysis_number].analysis_rqst[analysis_channel] > AnalysisRqst::NotCalculated)
				{
					if (strlen(analysis->channel[analysis_number].results[analysis_channel].name) >= SIZEOF_SHORT_NAME)
					{
						if (analysis->channel[analysis_number].results[analysis_channel].short_name[0] == 0x00)
						{
							ok = false;
							number = 1;
							while (ok == false)
							{
								strncpy(short_name, analysis->channel[analysis_number].results[analysis_channel].name, SIZEOF_SHORT_NAME);

								if (number < 10)
								{
									short_name[7] = 0x00;
								}
								else if (number < 100)
								{
									short_name[6] = 0x00;
								}
								else if (number < 1000)
								{
									short_name[5] = 0x00;
								}
								else if (number < 10000)
								{
									short_name[4] = 0x00;
								}
								else if (number < 100000)
								{
									short_name[3] = 0x00;
								}
								else if (number < 1000000)
								{
									short_name[2] = 0x00;
								}
								else if (number < 10000000)
								{
									short_name[1] = 0x00;
								}
								else
								{
									short_name[0] = 0x00;
								}

								snprintf(new_channel_name, 10, "%s~%u", short_name, number);
								new_channel_name[SIZEOF_SHORT_NAME - 1] = 0x00;

								ok = true;
								for (num = 0; num < analysis->number_of_channels; num++)
								{
									for (chan = 0; chan < get_number_of_analysis_channels(); chan++)
									{
										if (strncmp(new_channel_name, analysis->channel[num].results[chan].short_name, SIZEOF_SHORT_NAME) == 0)
										{
											ok = false;
											number += 1;
										}
									}
								}

								if (number > 9999)
								{
									ok = true;

									logmessage(WARNING, "Too many duplicate channel names");
								}
							}

							memset(analysis->channel[analysis_number].results[analysis_channel].short_name, 0, SIZEOF_SHORT_NAME);

							strncpy(analysis->channel[analysis_number].results[analysis_channel].short_name, new_channel_name, SIZEOF_SHORT_NAME);

							analysis->channel[analysis_number].results[analysis_channel].short_name[SIZEOF_SHORT_NAME - 1] = 0x00;
						}
					}
					else
					{
						memset(analysis->channel[analysis_number].results[analysis_channel].short_name, 0, SIZEOF_SHORT_NAME);

						strncpy(analysis->channel[analysis_number].results[analysis_channel].short_name, analysis->channel[analysis_number].results[analysis_channel].name, SIZEOF_SHORT_NAME);

						analysis->channel[analysis_number].results[analysis_channel].short_name[SIZEOF_SHORT_NAME - 1] = 0x00;
					}
				}
			}
		}
	}
}

unsigned int ifile_read_object(FILE* file, char* result,const unsigned int maximum_length,const bool swap_endian)
{
	unsigned int length;
	size_t bytes_read;

	if ((file == NULL) || (result == NULL))
	{
		return(0);
	}

	fread(&length, 4, 1, file);

	if (swap_endian != false)
	{
		swap_endian_4(&length);
	}

	if (length <= 1)
	{
		result[0] = 0x00;
		return(0);
	}

	char* object = (char*)malloc(length);
	if (object == NULL)
	{
		logmessage(MSG_ERROR, "Memory could not be allocated\n");
		result[0] = 0x00;
		return(0);
	}

	bytes_read = fread(object, 1, length, file);

	if (bytes_read != length)
	{
		free(object);
		result[0] = 0x00;
		return(0);
	}
	else
	{
		if (length < maximum_length)
		{
			memcpy(result, object, length);
			result[length] = 0x00;
		}
		else
		{
			memcpy(result, object, maximum_length - 1);
			result[maximum_length - 1] = 0x00;
		}
	}

	free(object);
	return(length);
}

unsigned int datart_type(const short datart, const char* absein, const short zyklng)
{
	unsigned int abscissa_type;

	switch (datart)
	{
	case 1:
	{
		abscissa_type = ABSCISSA_CRANKANGLE;

		break;
	}
	case 2:
	{
		abscissa_type = ABSCISSA_TIME;
		break;
	}
	case 3:
	case 4:
	{
		abscissa_type = ABSCISSA_CYCLE;
		break;
	}
	case 5:
	{
		abscissa_type = ABSCISSA_ASYNC_UTC;
		break;
	}
	default:
	{
		if (strncmp(absein, "deg", 3) == 0)
		{
			abscissa_type = ABSCISSA_CRANKANGLE;
		}
		else if (strncmp(absein, "cycle", 5) == 0)
		{
			abscissa_type = ABSCISSA_CYCLE;
		}
		else if (strncmp(absein, "ms", 2) == 0)
		{
			abscissa_type = ABSCISSA_TIME;
		}
		else if (zyklng > 1)
		{
			abscissa_type = ABSCISSA_CRANKANGLE;
		}
		else
		{
			abscissa_type = ABSCISSA_UNKNOWN;
		}

		break;
	}
	}

	if (abscissa_type == ABSCISSA_UNKNOWN)
	{
		logmessage(WARNING, "datart: %d zyklng: %d absein: %s ", datart, zyklng, absein);
		
		/*switch (abscissa_type)
		{
		case ABSCISSA_CRANKANGLE: logmessage(WARNING, "CA\n"); break;
		case ABSCISSA_CYCLE: logmessage(WARNING, "CYC\n"); break;
		case ABSCISSA_TIME: logmessage(WARNING, "TM\n"); break;
		case ABSCISSA_ASYNC_UTC: logmessage(WARNING, "ASYNC\n"); break;
		default:
		case ABSCISSA_UNKNOWN: logmessage(WARNING, "Unknown\n"); break;
		}*/
	}

	return(abscissa_type);
}

bool valid_dgb(const short datart, const unsigned int grpflg, const char* absein, const short zyklng)
{
	unsigned int abscissa_type = datart_type(datart, absein, zyklng);

	return (((abscissa_type == ABSCISSA_CRANKANGLE) && ((grpflg & IFILE_CA) != 0)) ||
			((abscissa_type == ABSCISSA_TIME) && ((grpflg & IFILE_TIME) != 0)) ||
			((abscissa_type == ABSCISSA_CYCLE) && ((grpflg & IFILE_RTP_RESULTS) != 0)) ||
			((abscissa_type == ABSCISSA_CYCLE) && ((grpflg & IFILE_RESULTS) != 0)) ||
			((abscissa_type == ABSCISSA_ASYNC_UTC) && ((grpflg & IFILE_ASYNC_UTC) != 0)));
}

#ifdef _CATOOL_UNICODE_
bool is_valid_ifile(const wchar_t* filename)
#else
bool is_valid_ifile(const char* filename)
#endif
{
	FILE* input_file = NULL;
	long filesize;
	int16_t parlng;

	if (filename == NULL)
	{
		logmessage(WARNING, "AVL I-File empty filename\n");

		return(false);
	}
#ifdef _CATOOL_UNICODE_
	input_file = _wfopen(filename, L"rb");
#else
	input_file = fopen(filename, "rb");
#endif

	if (input_file == NULL)
	{
#ifdef _CATOOL_UNICODE_
		logmessage(WARNING, "AVL I-File (%ls) could not be opened\n", filename);
#else
		logmessage(WARNING, "AVL I-File (%s) could not be opened\n", filename);
#endif

		return(false);
	}

	filesize = fsize(input_file);

	if (filesize < 1024)
	{
		fclose(input_file);
#ifdef _CATOOL_UNICODE_
		logmessage(WARNING, "AVL I-File (%ls) is too small to be valid\n", filename);
#else
		logmessage(WARNING, "AVL I-File (%s) is too small to be valid\n", filename);
#endif

		return(false);
	}

	logmessage(NOTICE, "Reading header...");

	(void)fread(&parlng, 2, 1, input_file);

	fclose(input_file);

	if (parlng == 1024)
	{
		/* Endian swapping not required */

		return(true);
	}
	else if (parlng == 4)
	{
		/* Endian swapping required */

		return(true);
	}
	else
	{
		/* Not valid */
	}

	return(false);
}

void add_parameter(FileData* file, const char* name, const char* units, const char* description, const int parameter_number, const double value)
{
	Parameter* parameter = NULL;

	if (file == NULL)
	{
		return;
	}

	if (file->parameters == NULL)
	{
		file->parameters = (Parameter*)malloc(sizeof(Parameter));

		parameter = file->parameters;
	}
	else
	{
		parameter = file->parameters;

		while (parameter->next != NULL)
		{
			parameter = parameter->next;
		}

		parameter->next = (Parameter*)malloc(sizeof(Parameter));

		parameter = parameter->next;
	}

	if (parameter == NULL)
	{
		logmessage(FATAL, "Memory could not be allocated\n");
		return;
	}

	parameter->next = NULL;

	strncpy(parameter->name, name, SIZEOF_CHANNEL_NAME);
	parameter->name[SIZEOF_CHANNEL_NAME - 1] = 0x00;

	strncpy(parameter->units, units, SIZEOF_UNITS);
	parameter->units[SIZEOF_UNITS - 1] = 0x00;

	strncpy(parameter->description, description, SIZEOF_DESCRIPTION);
	parameter->description[SIZEOF_DESCRIPTION - 1] = 0x00;

	parameter->number = parameter_number;

	parameter->value = malloc(sizeof(double));

	if (parameter->value == NULL)
	{
		logmessage(FATAL, "Memory could not be allocated\n");
		return;
	}

	parameter->data_type = VARIABLE_DOUBLE;

	*((double*)parameter->value) = value;
}

bool process_ifile_extension(FILE* input_file, FileData* file, S_ExtensionObject* IFile_Object, const unsigned int pass)
{
	unsigned int channel;
	char* key_name = NULL;
	char short_channel_name[10];
	unsigned int object_size;
	unsigned int filesize;

	if ((input_file == NULL) || (file == NULL) || (IFile_Object == NULL))
	{
		return(false);
	}

	filesize = fsize(input_file);

	if (((file->ptr_APB->pExtensions +IFile_Object->lKeyFilePos) > filesize) ||
		((IFile_Object->lKeyFilePos) < 0) ||
		((file->ptr_APB->pExtensions + IFile_Object->lKeyFilePos + IFile_Object->ulKeyLength) > filesize))
	{
		logmessage(MSG_ERROR, "IFile IFile_Object points outside size of file\n");
		return(false);
	}

	if (IFile_Object->ulKeyLength > 1024)
	{
		logmessage(MSG_ERROR, "Key name too long\n");
		return(false);
	}

	if (fseek(input_file, file->ptr_APB->pExtensions + IFile_Object->lKeyFilePos, SEEK_SET) != 0)
	{
		logmessage(MSG_ERROR, "Could not set file position to start of data\n");

		return(false);
	}

	key_name = (char*)malloc(IFile_Object->ulKeyLength + 1);

	if (key_name == NULL)
	{
		fclose(input_file);
		logmessage(FATAL, "Memory could not be allocated\n");
		return(false);
	}

	(void)fread(key_name, IFile_Object->ulKeyLength, 1, input_file);
	key_name[IFile_Object->ulKeyLength] = 0x00;

	if (((file->ptr_APB->pExtensions + IFile_Object->lObjectFilePos) > filesize) ||
		((file->ptr_APB->pExtensions + IFile_Object->lObjectFilePos) < file->ptr_APB->pExtensions))
	{
		fclose(input_file);
		free(key_name);

		logmessage(MSG_ERROR, "IFile IFile_Object points outside size of file\n");

		return(false);
	}

	if (fseek(input_file, file->ptr_APB->pExtensions + IFile_Object->lObjectFilePos, SEEK_SET) != 0)
	{
		fclose(input_file);
		free(key_name);

		logmessage(MSG_ERROR, "Could not set file position to start of data\n");

		return(false);
	}

	switch (pass)
	{
	case 1:
	default:
	{
		switch (IFile_Object->eType)
		{
		case 5:		/* Extended Signal Name */
		{
			S_ExtendedSignalName ESN;
			S_ExtendedSignalName* extended_signal = &ESN;

			(void)fread(&extended_signal->ulNameLength, 4, 1, input_file);

			if (file->swap_endian != false)
			{
				swap_endian_4(&extended_signal->ulNameLength);
			}

			if (extended_signal->ulNameLength < SIZEOF_CHANNEL_NAME)
			{
				extended_signal->pcName = (uint8_t*)malloc(extended_signal->ulNameLength + 1);

				if (extended_signal->pcName == NULL)
				{
					fclose(input_file);
					logmessage(FATAL, "Memory could not be allocated\n");
					return(false);
				}

				(void)fread(extended_signal->pcName, extended_signal->ulNameLength, 1, input_file);
				extended_signal->pcName[extended_signal->ulNameLength] = 0x00;

				logmessage(DEBUG, "%s: %s\n", key_name, extended_signal->pcName);

				if ((IFile_Object->ulKeyLength > 10) && (IFile_Object->ulKeyLength < 20))
				{
					if (strncmp(&key_name[IFile_Object->ulKeyLength - 10], ".LongName", 10) == 0)
					{
						strncpy(short_channel_name, key_name, IFile_Object->ulKeyLength - 9);
						short_channel_name[IFile_Object->ulKeyLength - 10] = 0x00;

						for (channel = 0; channel < file->number_of_channels; channel++)
						{
							if (strncmp(short_channel_name, file->channel_data[channel].name, SIZEOF_CHANNEL_NAME) == 0)
							{
								logmessage(DEBUG, "Renaming channel '%s' as '%s'\n", file->channel_data[channel].name, extended_signal->pcName);

								Rename_Channel(file, channel, (char*)extended_signal->pcName);
							}
						}

						/* Can also be used to rename parameters */

						Parameter* parameter = file->parameters;

						while (parameter != NULL)
						{
							if (strncmp(short_channel_name, parameter->name, SIZEOF_CHANNEL_NAME) == 0)
							{
								logmessage(DEBUG, "Renaming parameter '%s' as '%s'\n", parameter->name, extended_signal->pcName);

								strncpy(parameter->name, (char*)extended_signal->pcName, SIZEOF_CHANNEL_NAME);
								parameter->name[SIZEOF_CHANNEL_NAME - 1] = 0x00;
							}

							parameter = parameter->next;
						}
					}
				}

				free(extended_signal->pcName);
			}
			else
			{
				logmessage(WARNING, "Extended signal %s too long\n", key_name);
			}

			break;
		}
		default:
		{
			/* Ignoring all other types on first pass */

			break;
		}
		}

		break;
	}
	case 2:
	{
		switch (IFile_Object->eType)
		{
		case 1:		/* RTP Type */
		case 1001:	/* ChannelType */
		{
			char key_string[256];
			int key_value;

			ifile_read_object(input_file, key_string, 256, file->swap_endian);

			(void)fread(&key_value, 4, 1, input_file);

			if (file->swap_endian != false)
			{
				swap_endian_4(&key_value);
			}

			logmessage(DEBUG, "%s: %i\n", key_name, key_value);

			if (IFile_Object->eType == 1001)
			{
				/* ChannelType */

				if ((key_value > 0) &&
					(key_value < NUMBER_OF_CHANNEL_TYPES) &&
					((channel_name_to_number(file, key_name, ABSCISSA_CRANKANGLE, &channel) == true) || (channel_name_to_number(file, key_name, ABSCISSA_CYCLE, &channel) == true)) &&
					((file->channel_data[channel].type == CHAN_UNKNOWN) || (file->channel_data[channel].type == CHAN_RESULTS)) &&
					((file->channel_data[channel].abscissa.type == ABSCISSA_CRANKANGLE) || (file->channel_data[channel].abscissa.type == ABSCISSA_CYCLE)) &&
					(file->channel_data[channel].type != (unsigned int)key_value))
				{
					logmessage(NOTICE, "Changing channel %s type to \"%s\"\n", file->channel_data[channel].name, channel_information[key_value].description);

					file->channel_data[channel].type = (unsigned int)key_value;
				}
			}

			break;
		}
		case 2:		/* Sensor Sensitivity, TDCValue, AmplifierRange */
		case 1003:	/* SOC Fixed Value */
		{
			char key_string[256];
			double key_value;

			ifile_read_object(input_file, key_string, 256, file->swap_endian);

			(void)fread(&key_value, 8, 1, input_file);

			if (file->swap_endian != false)
			{
				swap_endian_8(&key_value);
			}

			if (IFile_Object->eType == 1003)
			{
				if ((key_value >= (double)(-90 * (int)file->engine.number_of_strokes)) &&
					(key_value < (double)(90 * (int)file->engine.number_of_strokes)) &&
					(channel_name_to_number(file, key_name, ABSCISSA_CRANKANGLE, &channel) == true) &&
					((float)key_value != file->channel_data[channel].soc_config.fixed_value))
				{
					logmessage(NOTICE, "Changing channel %s SOC fixed value to \"%f\"\n", file->channel_data[channel].name, key_value);

					file->channel_data[channel].soc_config.fixed_value = (float)key_value;
				}
			}

			logmessage(DEBUG, "%s.%s: %f\n", key_name, key_string, key_value);

			break;
		}
		case 3:		/* Description, SourceSignal, AmplifierType, InputFilter, Filter, DriftCompensation, SensorSerialNumber, OutputOffset, AmplifierRangeUnit, SensorCalibrationDate, SensorSensitivityUnit */
		case 1002:	/* AnalysisType */
		{
			char key_string[256];
			char value_string[256];

			ifile_read_object(input_file, key_string, 256, file->swap_endian);

			ifile_read_object(input_file, value_string, 256, file->swap_endian);

			logmessage(DEBUG, "%s: %s\n", key_name, value_string);

			if (IFile_Object->eType == 1002)
			{
				/* AnalysisType */

				if ((channel_name_to_number(file, key_name, ABSCISSA_UNKNOWN, &channel) == true) &&
					((file->channel_data[channel].type == CHAN_UNKNOWN) || (file->channel_data[channel].type == CHAN_RESULTS)))
				{
					logmessage(DEBUG, "Channel %s is analysis type \"%s\"\n", file->channel_data[channel].name, value_string);
				}
			}

			char channel_type[10];
			char channel_name[SIZEOF_CHANNEL_NAME];
			unsigned int abscissa_type;

			if (strncmp(key_string, "Description", 11) == 0)
			{
				char* apos = strstr(key_name, "'");

				if (apos != NULL)
				{
					*apos = 0x00;

					strncpy(channel_type, key_name, 10);
					strncpy(channel_name, apos + 1, SIZEOF_CHANNEL_NAME);

					char* description = strstr(channel_name, ".Description");

					if (description != NULL)
					{
						*description = 0x00;
					}

					if (strncmp(channel_type, "CY", 2) == 0)
					{
						abscissa_type = ABSCISSA_CYCLE;
					}
					else if (strncmp(channel_type, "TM", 2) == 0)
					{
						abscissa_type = ABSCISSA_TIME;
					}
					else if (strncmp(channel_type, "CA", 2) == 0)
					{
						abscissa_type = ABSCISSA_CRANKANGLE;
					}
					else if (strncmp(channel_type, "ASYNC", 5) == 0)
					{
						abscissa_type = ABSCISSA_ASYNC_UTC;
					}
					else
					{
						abscissa_type = ABSCISSA_UNKNOWN;
					}
				}
				else
				{
					abscissa_type = ABSCISSA_UNKNOWN;
					strncpy(channel_name, key_name, SIZEOF_CHANNEL_NAME);
				}

				if (channel_name_to_number(file, channel_name, abscissa_type, &channel) == true)
				{
					strncpy(file->channel_data[channel].description, value_string, SIZEOF_DESCRIPTION);

					logmessage(DEBUG, "Set channel %s description to %s\n", file->channel_data[channel].name, file->channel_data[channel].description);
				}
			}

			break;
		}
		case 5:		/* Extended Signal Name */
		{
			/* Ignore on second pass */

			break;
		}
		case 7:		/* YRangeMin */
		case 8:		/* YRangeMax */
		{
			double key_value;

			(void)fread(&key_value, 8, 1, input_file);

			if (file->swap_endian != false)
			{
				swap_endian_8(&key_value);
			}

			logmessage(DEBUG, "%s: %g\n", key_name, key_value);

			break;
		}
		case 14:	/* mpax File */
		case 15:	/* Parameter File */
		{
			logmessage(DEBUG, "Embedded Parameter File (%s)\n", key_name);

			object_size = IFile_Object->ulObjectSize;

			if (object_size < 100 * 1024 * 1024)
			{
				if (file->parameter_file != NULL)
				{
					free(file->parameter_file);
				}

				file->parameter_file = (char*)malloc(object_size);
				if (file->parameter_file == NULL)
				{
					fclose(input_file);
					logmessage(FATAL, "Memory could not be allocated\n");
					return(false);
				}

				(void)fread(file->parameter_file, object_size, 1, input_file);
#ifdef _CATOOL_UNICODE_
				size_t convertedChars = 0;
				mbstowcs_s(&convertedChars, file->parameter_filename, key_name, 4095);
#else
				strncpy(file->parameter_filename, key_name, 4095);
#endif
				file->parameter_filesize = object_size;
			}
			else
			{
				logmessage(NOTICE, "Embedded parameter file not loaded as it is greater than 100Mb in size\n");
			}

			break;
		}
		case 16:	/* CA' CY' SYNC' CYF' CAF' - single cycle PAR' SYNC' - single value */
		case 20:	/* CA' CAO' CAF' - all cycles */
		{
			logmessage(DEBUG, "Extended Channel: %s (Type %u Size %u Position %i)\n", key_name, IFile_Object->eType, IFile_Object->ulObjectSize, IFile_Object->lObjectFilePos);

			unsigned int data_type;
			unsigned int samples_per_cycle;
			unsigned int axis_samples;
			unsigned int total_length;
			bool save_channel = true;
			char object_name[256] = { 0x00 };
			char description[256] = { 0x00 };
			char units[256] = { 0x00 };
			char category[256] = { 0x00 };
			char axis[256] = { 0x00 };
			unsigned int number_of_cycles;
			unsigned int unknown;
			unsigned int sizeof_data;
			unsigned int remaining;
			unsigned int data_pointer;
			char axis_units[256] = { 0x00 };
			long data_position;
			unsigned int element_size;
			double value;

			total_length = 4 + ifile_read_object(input_file, object_name, 256, file->swap_endian);
			total_length += 4 + ifile_read_object(input_file, description, 256, file->swap_endian);
			total_length += 4 + ifile_read_object(input_file, units, 256, file->swap_endian);
			total_length += 4 + ifile_read_object(input_file, category, 256, file->swap_endian);

			logmessage(DEBUG, "Name: %s Description: %s Units: %s Category: %s\n", object_name, description, units, category);

			if (IFile_Object->eType == 20)
			{
				total_length += 4 + ifile_read_object(input_file, axis, 256, file->swap_endian);
			}

			(void)fread(&data_type, 4, 1, input_file);
			total_length += 4;

			if (file->swap_endian != false)
			{
				swap_endian_4(&data_type);
			}

			if ((data_type != 2) && (data_type != 3) && (data_type != 4) && (data_type != 9))
			{
				logmessage(WARNING, "%s'%s: Don't know what to do with data type %u.  Contact support@catool.org for assistance.\n", category, object_name, data_type);

				save_channel = false;
			}

			(void)fread(&samples_per_cycle, 4, 1, input_file);
			total_length += 4;

			if (file->swap_endian != false)
			{
				swap_endian_4(&samples_per_cycle);
			}

			if (IFile_Object->eType == 20)
			{
				(void)fread(&axis_samples, 4, 1, input_file);
				total_length += 4;

				if (file->swap_endian != false)
				{
					swap_endian_4(&axis_samples);
				}

				(void)fread(&number_of_cycles, 4, 1, input_file);
				total_length += 4;

				if (file->swap_endian != false)
				{
					swap_endian_4(&number_of_cycles);
				}

				(void)fread(&unknown, 4, 1, input_file);
				total_length += 4;

				if (file->swap_endian != false)
				{
					swap_endian_4(&unknown);
				}

				if (unknown > 0)
				{
					logmessage(WARNING, "%s'%s: Unknown unknown #1: 0x%08X\n", category, object_name, unknown);
				}
			}
			else
			{
				number_of_cycles = 1;
				unknown = 0;
				axis_samples = 0;
			}

			sizeof_data = samples_per_cycle * number_of_cycles;
			element_size = 8;

			if (sizeof_data == 0)
			{
				save_channel = false;
			}
			else
			{
				data_position = ftell(input_file);

				if ((sizeof_data == 1) && (element_size == 8))
				{
					fread(&value, element_size, 1, input_file);
				}
				else
				{
					fseek(input_file, sizeof_data * element_size, SEEK_CUR);
				}

				total_length += sizeof_data * element_size;
			}

			if (IFile_Object->eType == 16)
			{
				(void)fread(&axis_samples, 4, 1, input_file);
				total_length += 4;

				if (file->swap_endian != false)
				{
					swap_endian_4(&axis_samples);
				}
			}

			logmessage(DEBUG, "%s'%s (%s) %u samples for %u cycles Data Type: %u Unknown: %u Total Length: %u\n", category, object_name, units, samples_per_cycle, number_of_cycles, data_type, unknown, total_length);

			remaining = IFile_Object->ulObjectSize - total_length;

			unsigned int remainder = remaining % 8U;

			if ((remainder != 0) && (remainder != 4))
			{
				if (remaining / 8U != axis_samples)
				{
					/* Ref: INDICOM.i: U_I_PFI_Inj01
					        INDICOM_2023-09-14.0001: U_I_PFI_Inj01
					        20140304_1500_ETH20_05bar_Lam09_SA2: KP_EV1, KFRQ1, IMEP1, AI50_1, KP_PK1
					        20140303_1500_CTL95_05bar_Lam09_SA3: KP_EV1, KFRQ1, IMEP1, AI50_1, KP_PK1
					        20140620_1500_CTL95_05bar_Lam09_SA3: KP_EV1, KFRQ1, IMEP1, AI50_1, KP_PK1
					        20140620_1500_Eth20_05bar_Lam09_SA2: KP_EV1, KFRQ1, IMEP1, AI50_1, KP_PK1
					        20140717_1500_Crude95_05bar_Lam09_SA5: KP_EV1, KFRQ1, IMEP1, AI50_1, KP_PK1
					        INDI_MPST1.115: I1, Q1
							TR19464ARun18.001: ValveA_Shift_CAO1
					*/

					logmessage(NOTICE, "%s'%s: %u samples remaining (%u double values, remainder %u)\n", category, object_name, remaining, remaining / 8U, remainder);
				}
				else if ((axis_samples == 0) && (remaining > 0))
				{
					logmessage(WARNING, "%s'%s: %u samples remaining but zero axis_samples\n", category, object_name, remaining);
				}
				else if ((axis_samples > 0) && (samples_per_cycle != axis_samples))
				{
					logmessage(WARNING, "%s'%s: axis_samples (%u) != samples_per_cycle (%u)\n", category, object_name, axis_samples, samples_per_cycle);
				}
				else
				{
					/* Do Nothing */
				}
			}

			double* axis_data = NULL;
			uint32_t int_axis_data;

			if (axis_samples > 0)
			{
				axis_data = (double*)malloc(axis_samples * sizeof(double));

				if (remaining < axis_samples * sizeof(double))
				{
					if (remaining >= axis_samples * sizeof(float))
					{
						for (data_pointer = 0; data_pointer < axis_samples; data_pointer++)
						{
							(void)fread(&int_axis_data, 1, 4, input_file);
							total_length += 4;

							if (data_type == 3)
							{
								axis_data[data_pointer] = (double)*(uint32_t*)&int_axis_data;
							}
							else
							{
								axis_data[data_pointer] = (double)*(float*)&int_axis_data;
							}
						}
					}
					else
					{
						logmessage(WARNING, "%s'%s: Axis too small to be float or double\n", category, object_name);
					}
				}
				else
				{
					(void)fread(axis_data, axis_samples, 8, input_file);
					total_length += axis_samples * 8;
				}
			}

			remaining = IFile_Object->ulObjectSize - total_length;

			if (IFile_Object->eType == 16)
			{
				if (remaining >= 4)
				{
					total_length += 4 + ifile_read_object(input_file, axis_units, 256, file->swap_endian);
				}
			}
			else /*if (IFile_Object->eType == 20)*/
			{
				if (remaining >= 8)
				{
					fread(&unknown, 4, 1, input_file);
					total_length += 4;

					if ((unknown & 0xFF000000) == 0x40000000)
					{
						total_length += 4 + ifile_read_object(input_file, axis_units, 256, file->swap_endian);
					}
					else
					{
						logmessage(WARNING, "%s'%s: Unknown unknown #2: 0x%08X", category, object_name, unknown);
					}

					logmessage(DEBUG, "Unknown 0x%08X (%f) Axis Units: %s\n", unknown, *(float*)&unknown, axis_units);
				}
			}

			if (total_length != IFile_Object->ulObjectSize)
			{
				logmessage(WARNING, "%s'%s: Total size does not match %u vs %u\n", category, object_name, total_length, IFile_Object->ulObjectSize);

				save_channel = false;
			}

			if ((samples_per_cycle == 1) && (number_of_cycles == 1)/* && (data_type == 4) && (strncmp(category, "PAR", 3) == 0)*/)
			{
				add_parameter(file, object_name, units, description, -1, value);
			}
			else if (save_channel == true)
			{
				char channel_name[SIZEOF_CHANNEL_NAME];

				strncpy(channel_name, object_name, SIZEOF_CHANNEL_NAME);
				channel_name[SIZEOF_CHANNEL_NAME - 1] = 0x00;

				if (Add_Channel(file, NULL, channel_name) == true)
				{
					logmessage(NOTICE, "Added channel %s\n", channel_name);
				}
				else
				{
					snprintf(channel_name, SIZEOF_CHANNEL_NAME, "%s'%s", category, object_name);
					channel_name[SIZEOF_CHANNEL_NAME - 1] = 0x00;

					if (Add_Channel(file, NULL, channel_name) == true)
					{
						logmessage(NOTICE, "Added channel %s\n", channel_name);
					}
					else
					{
						logmessage(WARNING, "Failed to add channel %s\n", channel_name);

						/* Force channel_name_to_number() to return false */

						channel_name[0] = 0x00;
					}
				}

				if (channel_name_to_number(file, channel_name, ABSCISSA_UNKNOWN, &channel) == true)
				{
					Empty_Abscissa(&file->channel_data[channel].abscissa);

					if (description[00] != 0x00)
					{
						strncpy(file->channel_data[channel].description, description, SIZEOF_DESCRIPTION);
					}
					else
					{
						strncpy(file->channel_data[channel].description, "Unknown", SIZEOF_DESCRIPTION);
					}
					strncpy(file->channel_data[channel].units, units, SIZEOF_UNITS);
					strncpy(file->channel_data[channel].abscissa.units, axis_units, SIZEOF_UNITS);

					unsigned int* data_info = (unsigned int*)malloc(5 * sizeof(unsigned int));

					if (data_info == NULL)
					{
						fclose(input_file);
						logmessage(FATAL, "Memory could not be allocated\n");
						return(false);
					}

					data_info[0] = 0x0CA70070;
					data_info[1] = number_of_cycles;
					data_info[2] = samples_per_cycle;
					data_info[3] = data_position;
					data_info[4] = element_size;

					file->channel_data[channel].data = (float*)data_info;

					if (data_type == 2)
					{
						/* Double Crank Angle Axis */

						/* Ref: INDICOM.i, INJ_DI_SIGD1 */

						file->channel_data[channel].abscissa.type = ABSCISSA_CRANKANGLE;
						file->channel_data[channel].abscissa.number_of_measurement_tables = 1;

						file->channel_data[channel].abscissa.number_of_samples = (unsigned int*)malloc(file->channel_data[channel].abscissa.number_of_measurement_tables * sizeof(unsigned int));
						if (file->channel_data[channel].abscissa.number_of_samples == NULL)
						{
							fclose(input_file);
							logmessage(FATAL, "Memory could not be allocated\n");
							return(false);
						}

						file->channel_data[channel].abscissa.start = (unsigned int*)malloc(file->channel_data[channel].abscissa.number_of_measurement_tables * sizeof(unsigned int));
						if (file->channel_data[channel].abscissa.start == NULL)
						{
							fclose(input_file);
							logmessage(FATAL, "Memory could not be allocated\n");
							return(false);
						}

						file->channel_data[channel].abscissa.resolution = (unsigned int*)malloc(file->channel_data[channel].abscissa.number_of_measurement_tables * sizeof(unsigned int));
						if (file->channel_data[channel].abscissa.resolution == NULL)
						{
							fclose(input_file);
							logmessage(FATAL, "Memory could not be allocated\n");
							return(false);
						}

						file->channel_data[channel].abscissa.number_of_samples[0] = samples_per_cycle;

						if ((samples_per_cycle <= 1) || (axis_data == NULL))
						{
							file->channel_data[channel].abscissa.start[0] = 0;
							file->channel_data[channel].abscissa.resolution[0] = 1;
						}
						else
						{
							file->channel_data[channel].abscissa.start[0] = DegreesToTheta(file, (float)axis_data[0]);
							file->channel_data[channel].abscissa.resolution[0] = 1;
						}

						file->channel_data[channel].abscissa.axis = axis_data;
						axis_data = NULL;

						file->channel_data[channel].abscissa.axis_type = VARIABLE_DOUBLE;

						file->channel_data[channel].samples_per_cycle = file->channel_data[channel].abscissa.number_of_samples[0];

						if (calculate_theta_lookups(file, channel) == false)
						{
							logmessage(WARNING, "process_ifile_extensions: Could not calculate theta lookups\n");
							return(false);
						}
					}
					else if (data_type == 3)
					{
						/* Double Cycle Axis */

						/* Ref: INDICOM.i, StartTime */

						file->channel_data[channel].abscissa.type = ABSCISSA_CYCLE;
						file->channel_data[channel].abscissa.number_of_measurement_tables = 1;

						file->channel_data[channel].abscissa.number_of_samples = (unsigned int*)malloc(file->channel_data[channel].abscissa.number_of_measurement_tables * sizeof(unsigned int));
						if (file->channel_data[channel].abscissa.number_of_samples == NULL)
						{
							fclose(input_file);
							logmessage(FATAL, "Memory could not be allocated\n");
							return(false);
						}

						file->channel_data[channel].abscissa.start = (unsigned int*)malloc(file->channel_data[channel].abscissa.number_of_measurement_tables * sizeof(unsigned int));
						if (file->channel_data[channel].abscissa.start == NULL)
						{
							fclose(input_file);
							logmessage(FATAL, "Memory could not be allocated\n");
							return(false);
						}

						file->channel_data[channel].abscissa.resolution = (unsigned int*)malloc(file->channel_data[channel].abscissa.number_of_measurement_tables * sizeof(unsigned int));
						if (file->channel_data[channel].abscissa.resolution == NULL)
						{
							fclose(input_file);
							logmessage(FATAL, "Memory could not be allocated\n");
							return(false);
						}

						file->channel_data[channel].abscissa.number_of_samples[0] = samples_per_cycle;
						file->channel_data[channel].abscissa.start[0] = 0;
						file->channel_data[channel].abscissa.resolution[0] = 1;

						file->channel_data[channel].abscissa.axis = axis_data;
						axis_data = NULL;

						file->channel_data[channel].abscissa.axis_type = VARIABLE_DOUBLE;

						file->channel_data[channel].samples_per_cycle = file->channel_data[channel].abscissa.number_of_samples[0];
					}
					else if ((data_type == 4) || (data_type == 9))
					{
						/* data_type 4 = Log Points
						   data_type 9 = Frequency */

						file->channel_data[channel].abscissa.type = ABSCISSA_FREQUENCY;
						file->channel_data[channel].abscissa.number_of_measurement_tables = 1;

						file->channel_data[channel].abscissa.number_of_samples = (unsigned int*)malloc(file->channel_data[channel].abscissa.number_of_measurement_tables * sizeof(unsigned int));
						if (file->channel_data[channel].abscissa.number_of_samples == NULL)
						{
							fclose(input_file);
							logmessage(FATAL, "Memory could not be allocated\n");
							return(false);
						}

						file->channel_data[channel].abscissa.start = (unsigned int*)malloc(file->channel_data[channel].abscissa.number_of_measurement_tables * sizeof(unsigned int));
						if (file->channel_data[channel].abscissa.start == NULL)
						{
							fclose(input_file);
							logmessage(FATAL, "Memory could not be allocated\n");
							return(false);
						}

						file->channel_data[channel].abscissa.resolution = (unsigned int*)malloc(file->channel_data[channel].abscissa.number_of_measurement_tables * sizeof(unsigned int));
						if (file->channel_data[channel].abscissa.resolution == NULL)
						{
							fclose(input_file);
							logmessage(FATAL, "Memory could not be allocated\n");
							return(false);
						}

						file->channel_data[channel].abscissa.number_of_samples[0] = samples_per_cycle;
						file->channel_data[channel].abscissa.start[0] = 0;
						file->channel_data[channel].abscissa.resolution[0] = 1;

						file->channel_data[channel].abscissa.axis = axis_data;
						axis_data = NULL;

						file->channel_data[channel].abscissa.axis_type = VARIABLE_DOUBLE;

						file->channel_data[channel].samples_per_cycle = file->channel_data[channel].abscissa.number_of_samples[0];
					}
					else
					{
						/* Do Nothing */
					}

					file->channel_data[channel].type = CHAN_RESULTS;
					file->channel_data[channel].cylinder = determine_cylinder_number(file->channel_data[channel].name, NULL);
					file->channel_data[channel].tdc_offset = file->engine.tdc_offset[file->channel_data[channel].cylinder - 1];
				}
				else
				{
					logmessage(WARNING, "Unable to add channel %s (channel_name_to_number() failed)\n", object_name);
				}
			}
			else
			{
				logmessage(WARNING, "Unable to add channel %s (save_channel false)\n", object_name);
			}

			if (axis_data != NULL)
			{
				free(axis_data);
			}

			break;
		}
		case 17:	/* SYS' IFileName Software Firmware */
					/* SYNC' UniversalTime */
		{
			char object_name[256];
			char description[256];
			char units[256];
			char category[256];
			char value[256];

			ifile_read_object(input_file, object_name, 256, file->swap_endian);

			ifile_read_object(input_file, description, 256, file->swap_endian);

			ifile_read_object(input_file, units, 256, file->swap_endian);

			ifile_read_object(input_file, category, 256, file->swap_endian);

			ifile_read_object(input_file, value, 256, file->swap_endian);

			//add_parameter(file, object_name, units, description, -1, value);

			logmessage(DEBUG, "%s: %s\n", key_name, value);

			break;
		}
		case 19:	/* Extended Parameter */
		{
			S_ParameterStructure PAR;

			(void)fread(&PAR.ulNameLength, 4, 1, input_file);

			if (file->swap_endian != false)
			{
				swap_endian_4(&PAR.ulNameLength);
			}

			if (PAR.ulNameLength > 1024)
			{
				fclose(input_file);
				logmessage(MSG_ERROR, "Parameter name too long\n");
				return(false);
			}

			PAR.pcName = (uint8_t*)malloc(PAR.ulNameLength + 1);
			if (PAR.pcName == NULL)
			{
				fclose(input_file);
				logmessage(FATAL, "Memory could not be allocated\n");
				return(false);
			}

			(void)fread(PAR.pcName, PAR.ulNameLength, 1, input_file);
			PAR.pcName[PAR.ulNameLength] = 0x00;

			(void)fread(&PAR.ulUnitLength, 4, 1, input_file);

			if (file->swap_endian != false)
			{
				swap_endian_4(&PAR.ulUnitLength);
			}

			if (PAR.ulUnitLength > 1024)
			{
				fclose(input_file);
				logmessage(MSG_ERROR, "Parameter unit too long\n");
				return(false);
			}

			PAR.pcUnit = (uint8_t*)malloc(PAR.ulUnitLength + 1);
			if (PAR.pcUnit == NULL)
			{
				fclose(input_file);
				logmessage(FATAL, "Memory could not be allocated\n");
				return(false);
			}

			(void)fread(PAR.pcUnit, PAR.ulUnitLength, 1, input_file);
			PAR.pcUnit[PAR.ulUnitLength] = 0x00;

			(void)fread(&PAR.dValue, 8, 1, input_file);

			if (file->swap_endian != false)
			{
				swap_endian_8(&PAR.dValue);
			}

			logmessage(DEBUG, "%s %s: %f (%s)\n", key_name, PAR.pcName, PAR.dValue, PAR.pcUnit);

			if (PAR.pcName[0] != 00)
			{
				int parameter_number;

				if (sscanf(key_name, "PAR'P%02d", &parameter_number) == 1)
				{
					if (parameter_number > 0)
					{
						parameter_number--;
					}
					else
					{
						parameter_number = -1;
					}
				}
				else
				{
					parameter_number = -1;
				}

				Parameter* parameter = file->parameters;

				bool found = false;

				while (parameter != NULL)
				{
					if ((parameter->number == parameter_number) && (parameter_number != -1))
					{
						found = true;
					}

					parameter = parameter->next;
				}

				if (found == false)
				{
					add_parameter(file, (char*)PAR.pcName, (char*)PAR.pcUnit, key_name, parameter_number, PAR.dValue);
				}
			}

			free(PAR.pcName);
			free(PAR.pcUnit);

			break;
		}
		case 22:	/* SYS' Options License Hardware SerialNr */
		{
			/* Value is as type 17 but the string is not human readable */

			char object_name[256];
			char description[256];
			char units[256];
			char category[256];
			char value[1024];
			unsigned int total_size;

			total_size = 4 + ifile_read_object(input_file, object_name, 256, file->swap_endian);

			total_size += 4 + ifile_read_object(input_file, description, 256, file->swap_endian);

			total_size += 4 + ifile_read_object(input_file, units, 256, file->swap_endian);

			total_size += 4 + ifile_read_object(input_file, category, 256, file->swap_endian);

			memset(value, 0xFF, 1024);

			unsigned int size = ifile_read_object(input_file, value, 1024, file->swap_endian);

			logmessage(DEBUG, "%s: Size: %d ObjectName: %s Description: %s Units: %s Category: %s Data size: %d Remainder: %d\n", key_name, IFile_Object->ulObjectSize, object_name, description, units, category, size, IFile_Object->ulObjectSize - (total_size + 4 + size));

			/*unsigned int i;
			for (i = 0; i < (size+8)/8; i++)
			{
				logmessage(DEBUG, "%02X %02X %02X %02X %02X %02X %02X %02X\n", value[i * 8 + 0] & 0xFF, value[i * 8 + 1] & 0xFF, value[i * 8 + 2] & 0xFF, value[i * 8 + 3] & 0xFF, value[i * 8 + 4] & 0xFF, value[i * 8 + 5] & 0xFF, value[i * 8 + 6] & 0xFF, value[i * 8 + 7] & 0xFF);
			}*/

			break;
		}
		case 23:	/* SYS'DigiSign */
		{
			logmessage(DEBUG, "%s: Digital Signature\n", key_name);

			break;
		}
		default:
		{
			logmessage(WARNING, "%s: Unknown Type (%u)\n", key_name, IFile_Object->eType);

			break;
		}
		}

		break;
	}
	}

	free(key_name);

	return(true);
}

#ifdef _CATOOL_UNICODE_
bool load_ifile_header(FileData* file, const wchar_t* filename, const unsigned int grpflg)
#else
bool load_ifile_header(FileData *file,const char* filename,const unsigned int grpflg)
#endif
{
	FILE* input_file = NULL;
	float geometric_conversion;
	int dgb_channel;
	unsigned int channel;
	unsigned int parameter_number;
	unsigned int dgb;
	unsigned int channel_no;
	unsigned int mt_number;
	unsigned int azt;
	unsigned int rzt;
	unsigned int start_channel;
	unsigned int number_of_data_groups;
	unsigned int max_cylinder_number = 0;
	unsigned int APB_end;
	unsigned int DGB_end;
	unsigned int header_end;
	unsigned int filesize;
	unsigned int mbllen;
	unsigned int aztlen;
	DATDIR* ptr_DI = NULL;
	MPTAB* ptr_MT = NULL;
	DATGRP* DGB_ptr = NULL;
	unsigned int measurement_table;
	unsigned int mt;
	unsigned int file_position;
	unsigned int chan;
	bool found_soc;
	S_IfileExtension IFE;
	S_ExtensionObject IFO;
	S_IfileExtension* IFile_Extension = &IFE;
	S_ExtensionObject* IFile_Object = &IFO;
	unsigned int object;
	unsigned int next_object;
	float pin_offset;
	float pin_offset_2;
	unsigned int cylinder;
	unsigned int unknown_channel_number = 0;
	unsigned int crank_angle;
	unsigned int offset_channel;
	double ref_time_slope = 1.0;
	double ref_time_offset = 0.0;
	char ref_time_units[SIZEOF_UNITS];
	bool found_ref_time = false;
	bool convert;

	if (file->number_of_channels > 0)
	{
		for (channel=0;channel<file->number_of_channels;channel++)
		{
			Delete_Channel(file,channel);
		}
	}

	if (filename == NULL)
	{
		logmessage(MSG_ERROR, "AVL I-File empty filename\n");

		return(false);
	}

	file->number_of_channels = 0;

#ifdef _CATOOL_UNICODE_
	wcsncpy(file->filename, filename, 4095);
	file->filename[4095] = 0x0000;

	input_file = _wfopen(file->filename, L"rb");
#else
	strncpy(file->filename,filename,4095);
	file->filename[4095] = 0x00;
	
	input_file = fopen(file->filename,"rb");
#endif

	if (input_file == NULL)
	{
#ifdef _CATOOL_UNICODE_
		logmessage(MSG_ERROR, "AVL I-File (%ls) could not be opened\n", file->filename);
#else
		logmessage(MSG_ERROR,"AVL I-File (%s) could not be opened\n",file->filename);
#endif
		
		return(false);
	}

	filesize = fsize(input_file);

	if (filesize < 1024)
	{
		fclose(input_file);
#ifdef _CATOOL_UNICODE_
		logmessage(MSG_ERROR, "AVL I-File (%ls) is too small to be valid\n", file->filename);
#else
		logmessage(MSG_ERROR, "AVL I-File (%s) is too small to be valid\n", file->filename);
#endif

		return(false);
	}

    file->number_of_cycles = 1;
    
	/* Load General Parameter Block (APB) */

	if (file->ptr_APB != NULL)
	{
		free(file->ptr_APB);
		file->ptr_APB = NULL;
	}

	file->ptr_APB = (PARBLK*)malloc(sizeof(PARBLK));

	if (file->ptr_APB == NULL)
	{
		fclose(input_file);
		
		logmessage(FATAL,"Memory could not be allocated\n");
		return(false);
	}

	logmessage(NOTICE,"Reading header...");

	(void)fread(&file->ptr_APB->parlng,2,1,input_file);
	(void)fread(&file->ptr_APB->grpanz,2,1,input_file);
	(void)fread(file->ptr_APB->filkom,80,1,input_file);
	(void)fread(file->ptr_APB->mesdat,24,1,input_file);
	(void)fread(file->ptr_APB->parfil,18,1,input_file);
	(void)fread(file->ptr_APB->pronam,18,1,input_file);
	(void)fread(file->ptr_APB->prfstd,10,1,input_file);
	(void)fread(file->ptr_APB->motnam,18,1,input_file);

	(void)fread(&file->ptr_APB->mottyp,2,1,input_file);
	(void)fread(&file->ptr_APB->tktanz,2,1,input_file);
	(void)fread(&file->ptr_APB->geoein,2,1,input_file);
	(void)fread(&file->ptr_APB->hublng,8,1,input_file);
	(void)fread(&file->ptr_APB->pleuel,8,1,input_file);
	(void)fread(&file->ptr_APB->bohrng,8,1,input_file);

	(void)fread(&file->ptr_APB->kompre,8,1,input_file);
	(void)fread(&file->ptr_APB->desaxi,8,1,input_file);
	(void)fread(&file->ptr_APB->grpflg,4,1,input_file);
	(void)fread(&file->ptr_APB->datahandle,4,1,input_file);
	(void)fread(&file->ptr_APB->version,2,1,input_file);
	(void)fread(&file->ptr_APB->desaxi2,8,1,input_file);
	(void)fread(&file->ptr_APB->pExtensions,4,1,input_file);
	
	file->ptr_APB->filkom[79] = 0x00;
	file->ptr_APB->mesdat[23] = 0x00;
	file->ptr_APB->parfil[17] = 0x00;
	file->ptr_APB->pronam[17] = 0x00;
	file->ptr_APB->prfstd[9] = 0x00;
	file->ptr_APB->motnam[17] = 0x00;

	for (parameter_number=0;parameter_number<28;parameter_number++)
	{
		(void)fread(file->ptr_APB->bpar[parameter_number].name,10,1,input_file);
		(void)fread(file->ptr_APB->bpar[parameter_number].einh,10,1,input_file);
		(void)fread(&file->ptr_APB->bpar[parameter_number].wert,8,1,input_file);
	}

	file->swap_endian = false;

	if (file->ptr_APB->parlng == 1024)
	{
		file->swap_endian = false;
    }
	else if (file->ptr_APB->parlng == 4)
	{
		file->swap_endian = true;
	}
	else if ((file->ptr_APB->parlng == 0x4B50) &&
	         (file->ptr_APB->grpanz == 0x0403))
	{
		fclose(input_file);		
		
		logmessage(MSG_ERROR,"Input file is a ZIP file.  Please uncompress before processing!\n");

		return(false);
	}
	else
	{
		fclose(input_file);
		
		logmessage(WARNING,"Unknown File Format!\n");
		
		return(false);
	}

	APB_end = ftell(input_file);

	if (file->swap_endian)
	{
		swap_endian_2(&file->ptr_APB->parlng);
		swap_endian_2(&file->ptr_APB->grpanz);
		swap_endian_2(&file->ptr_APB->mottyp);
		swap_endian_2(&file->ptr_APB->tktanz);
		swap_endian_2(&file->ptr_APB->geoein);
		
		swap_endian_8(&file->ptr_APB->hublng);
		swap_endian_8(&file->ptr_APB->pleuel);
		swap_endian_8(&file->ptr_APB->bohrng);
		swap_endian_8(&file->ptr_APB->kompre);
		swap_endian_8(&file->ptr_APB->desaxi);
		
		swap_endian_4(&file->ptr_APB->grpflg);
		swap_endian_4(&file->ptr_APB->datahandle);
		
		swap_endian_2(&file->ptr_APB->version);
		
		swap_endian_8(&file->ptr_APB->desaxi2);
		swap_endian_4(&file->ptr_APB->pExtensions);

		for (parameter_number=0;parameter_number<28;parameter_number++)
		{
			swap_endian_8(&file->ptr_APB->bpar[parameter_number].wert);
		}
	}

	logmessage(DEBUG,"done\n");
	logmessage(DEBUG,"Header size: %u\n",file->ptr_APB->parlng);
	if (file->swap_endian != false)
	{
		logmessage(DEBUG,"Endian swapping required\n");
	}
	logmessage(DEBUG,"No. of data groups in file: %i\n",file->ptr_APB->grpanz);
	logmessage(DEBUG,"File comment: %s\n",file->ptr_APB->filkom);
	logmessage(DEBUG,"Measurement date: %s\n",file->ptr_APB->mesdat);
	logmessage(DEBUG,"Measurement file used: %s\n",file->ptr_APB->parfil);
	logmessage(DEBUG,"Engine name: %s\n",file->ptr_APB->motnam);
	logmessage(DEBUG,"Engine type: %i\n",file->ptr_APB->mottyp);
	logmessage(DEBUG,"Number of strokes: %i\n",file->ptr_APB->tktanz);
	logmessage(DEBUG,"Geometric unit: %i\n",file->ptr_APB->geoein);
	logmessage(DEBUG,"Stroke: %f\n",file->ptr_APB->hublng);
	logmessage(DEBUG,"Conrod length: %f\n",file->ptr_APB->pleuel);
	logmessage(DEBUG,"Bore: %f\n",file->ptr_APB->bohrng);
	logmessage(DEBUG,"Compression Ratio: %f\n",file->ptr_APB->kompre);
	logmessage(DEBUG,"Offset 1st bank: %f\n",file->ptr_APB->desaxi);
	logmessage(DEBUG,"Offset 2nd bank: %f\n",file->ptr_APB->desaxi2);
	logmessage(DEBUG,"Software version: %i\n",file->ptr_APB->version);
	logmessage(DEBUG,"Data group description: %i\n",file->ptr_APB->grpflg);
	
	if ((file->ptr_APB->pExtensions > 0) && (file->ptr_APB->pExtensions < filesize))
	{
		logmessage(DEBUG,"IndiCom 1.3 file\n");
		logmessage(DEBUG,"Extensions: %u\n",file->ptr_APB->pExtensions);
	}

	if ((file->ptr_APB->parlng != 1024) || (APB_end != 1024))
	{
		logmessage(MSG_ERROR,"AVL I-File header is not 1024 bytes long\n");

		fclose(input_file);
		return(false);
	}
	
	strncpy(file->comment,(char*)file->ptr_APB->filkom,80);
	file->comment[255] = 0x00;
	file->creation_time = convert_time((char*)file->ptr_APB->mesdat);
	strncpy(file->engine.name,(char*)file->ptr_APB->motnam,18);
	file->engine.name[17] = 0x00;

	if (file->ptr_APB->mottyp == 0)
    {
        file->engine.type = ENGINE_CI_DI;
    }
    else if (file->ptr_APB->mottyp == 2)
    { 
        file->engine.type = ENGINE_DISI;
    }
    else
    {
        file->engine.type = ENGINE_SI;
    }

	if (file->ptr_APB->tktanz == 0)
	{
		file->engine.number_of_strokes = 4;
	}
	else
	{
		file->engine.number_of_strokes = 2;
	}

	if (file->ptr_APB->geoein == 1)
	{
		geometric_conversion = INCHES_TO_MM;
	}
	else
	{
		geometric_conversion = 1.0f;
	}

	//file->engine.volume_calculation_type = VOLUME_CATOOL;
	file->engine.stroke = (float)file->ptr_APB->hublng*geometric_conversion;
	file->engine.conrod_length = (float)file->ptr_APB->pleuel*geometric_conversion;
	file->engine.bore = (float)file->ptr_APB->bohrng*geometric_conversion;
	file->engine.compression_ratio = (float)file->ptr_APB->kompre;
	pin_offset = (float)file->ptr_APB->desaxi*geometric_conversion;
	pin_offset_2 = (float)file->ptr_APB->desaxi2*geometric_conversion;

	for (cylinder=0;cylinder<MAX_NUMBER_OF_CYLINDERS;cylinder++)
	{
		file->engine.pin_offset[cylinder] = 0.0f;
		file->engine.tdc_offset[cylinder] = 0.0f;
	}

	/* Guess some initial parameters */
	
	file->engine.number_of_cylinders = 4;

	if (file->engine.number_of_strokes == 4)
	{
		file->engine.ivc_angle = -140.0f;
		file->engine.evo_angle = 140.0f;
		file->engine.ivo_angle = 350.0f;
		file->engine.evc_angle = -355.0f;
	}
	else
	{
		file->engine.ivc_angle = -135.0f;
		file->engine.evo_angle = 120.0f;
		file->engine.ivo_angle = 135.0f;
		file->engine.evc_angle = -120.0f;
	}

	file->engine.tdc_error = 0.0f;
	file->engine.piston_factor = 1.2f;
	file->engine.firing_order = 0;

	Change_Minimum_Resolution(file,DEFAULT_MIN_RES_PPR);
	file->time_resolution = DEFAULT_TIME_RESOLUTION;

	/* Load Data Group Descriptions (DGB) */

	number_of_data_groups = 0;

	if (file->ptr_APB->grpanz < 1)
	{
		fclose(input_file);
		logmessage(MSG_ERROR,"Less than one data group");
		return(false);
	}
	else if (file->ptr_APB->grpanz > 100)
	{
		fclose(input_file);
		logmessage(MSG_ERROR,"Too many data groups found");
		return(false);
	}
	else
	{
		number_of_data_groups = (unsigned int)file->ptr_APB->grpanz;
	}

	logmessage(NOTICE,"Reading %u data group(s)...\n",number_of_data_groups);
	
	if (file->ptr_DGB != NULL)
	{
		free(file->ptr_DGB);
		file->ptr_DGB = NULL;
	}

	if (file->parameters != NULL)
	{
		Delete_File_Parameters(file);
	}

	for (parameter_number = 0; parameter_number < 28; parameter_number++)
	{
		if (file->ptr_APB->bpar[parameter_number].name[0] != 0x00)
		{
			char description[SIZEOF_DESCRIPTION];

			snprintf(description, SIZEOF_DESCRIPTION, "PAR'P%02u", parameter_number + 1);
			description[SIZEOF_DESCRIPTION - 1] = 0x00;

			add_parameter(file,
				(char*)file->ptr_APB->bpar[parameter_number].name,
				(char*)file->ptr_APB->bpar[parameter_number].einh,
				description,
				parameter_number,
				file->ptr_APB->bpar[parameter_number].wert);
		}
	}
	
	file->ptr_DGB = (DATGRP*)malloc(number_of_data_groups * sizeof(DATGRP)); 

	if (file->ptr_DGB == NULL)
	{
		fclose(input_file);
		logmessage(FATAL,"Memory could not be allocated\n");
		return(false);
	}

	for (dgb=0;dgb<number_of_data_groups;dgb++)
	{
		DGB_ptr = (DATGRP*)(file->ptr_DGB+dgb);
		
		DGB_ptr->diradr = NULL;
		DGB_ptr->absadr = NULL;
		DGB_ptr->mpladr = NULL;
		DGB_ptr->aztadr = NULL;
		DGB_ptr->rztadr = NULL;
		DGB_ptr->datadr = NULL;
		DGB_ptr->synadr = NULL;

		(void)fread(&DGB_ptr->datart,2,1,input_file);
		(void)fread(&DGB_ptr->absart,2,1,input_file);
		(void)fread(DGB_ptr->absein,10,1,input_file);
		(void)fread(&DGB_ptr->dltphi,8,1,input_file);
		(void)fread(&DGB_ptr->dltzei,8,1,input_file);
		(void)fread(&DGB_ptr->kananz,2,1,input_file);
		(void)fread(&DGB_ptr->absanf,4,1,input_file);
		(void)fread(&DGB_ptr->absend,4,1,input_file);
		(void)fread(&DGB_ptr->fortyp,2,1,input_file);
		(void)fread(&DGB_ptr->beranz,2,1,input_file);
		(void)fread(&DGB_ptr->blkanz,2,1,input_file);
		(void)fread(&DGB_ptr->zykanz,4,1,input_file);
		(void)fread(&DGB_ptr->zykanf,4,1,input_file);
		(void)fread(&DGB_ptr->zykend,4,1,input_file);
		(void)fread(&DGB_ptr->zyklng,4,1,input_file);
		(void)fread(&DGB_ptr->zyksta,4,1,input_file);
		(void)fread(&DGB_ptr->zyksum,4,1,input_file);
		(void)fread(&DGB_ptr->datfor,2,1,input_file);
		(void)fread(&DGB_ptr->mitanz,2,1,input_file);
		(void)fread(&DGB_ptr->mitkor,2,1,input_file);
		(void)fread(&DGB_ptr->datofs,2,1,input_file);
		(void)fread(&DGB_ptr->diradr,4,1,input_file);
		(void)fread(&DGB_ptr->absadr,4,1,input_file);
		(void)fread(&DGB_ptr->mpladr,4,1,input_file);
		(void)fread(&DGB_ptr->aztadr,4,1,input_file);
		(void)fread(&DGB_ptr->rztadr,4,1,input_file);
		(void)fread(&DGB_ptr->datadr,4,1,input_file);
		(void)fread(&DGB_ptr->refanf,4,1,input_file);
		(void)fread(&DGB_ptr->thekx1,8,1,input_file);
		(void)fread(&DGB_ptr->thekx2,8,1,input_file);
		(void)fread(&DGB_ptr->polexp,8,1,input_file);
		(void)fread(&DGB_ptr->thefac,8,1,input_file);
		(void)fread(&DGB_ptr->refzyl,10,1,input_file);
		(void)fread(&DGB_ptr->refsau,10,1,input_file);
		(void)fread(&DGB_ptr->refaus,10,1,input_file);
		(void)fread(&DGB_ptr->aztlen,4,1,input_file);
		(void)fread(&DGB_ptr->memhandle,4,1,input_file);
		(void)fread(&DGB_ptr->abslen,4,1,input_file);
		(void)fread(&DGB_ptr->refzylmit,2,1,input_file);
		(void)fread(&DGB_ptr->refsaumit,2,1,input_file);
		(void)fread(&DGB_ptr->refausmit,2,1,input_file);
		(void)fread(&DGB_ptr->azttyp,2,1,input_file);
		(void)fread(&DGB_ptr->aztofs,4,1,input_file);
		(void)fread(&DGB_ptr->blkofs,4,1,input_file);
		(void)fread(&DGB_ptr->blkdur,4,1,input_file);
		(void)fread(&DGB_ptr->blknum,4,1,input_file);
		(void)fread(&DGB_ptr->synadr,4,1,input_file);
		(void)fread(&DGB_ptr->synlen,4,1,input_file);
		(void)fread(DGB_ptr->reserve,44,1,input_file);
        		
		if (file->swap_endian != false)
		{
			swap_endian_2(&DGB_ptr->datart);
			swap_endian_2(&DGB_ptr->absart);
			swap_endian_8(&DGB_ptr->dltphi);
			swap_endian_8(&DGB_ptr->dltzei);
			swap_endian_2(&DGB_ptr->kananz);
			swap_endian_4(&DGB_ptr->absanf);
			swap_endian_4(&DGB_ptr->absend);
			swap_endian_2(&DGB_ptr->fortyp);
			swap_endian_2(&DGB_ptr->beranz);
			swap_endian_2(&DGB_ptr->blkanz);
			swap_endian_4(&DGB_ptr->zykanz);
			swap_endian_4(&DGB_ptr->zykanf);
			swap_endian_4(&DGB_ptr->zykend);
			swap_endian_4(&DGB_ptr->zyklng);
			swap_endian_4(&DGB_ptr->zyksta);
			swap_endian_4(&DGB_ptr->zyksum);
			swap_endian_2(&DGB_ptr->datfor);
			swap_endian_2(&DGB_ptr->mitanz);
			swap_endian_2(&DGB_ptr->mitkor);
			swap_endian_2(&DGB_ptr->datofs);
			swap_endian_4(&DGB_ptr->diradr);
			swap_endian_4(&DGB_ptr->absadr);
			swap_endian_4(&DGB_ptr->mpladr);
			swap_endian_4(&DGB_ptr->aztadr);
			swap_endian_4(&DGB_ptr->rztadr);
			swap_endian_4(&DGB_ptr->datadr);
			swap_endian_4(&DGB_ptr->refanf);
			swap_endian_8(&DGB_ptr->thekx1);
			swap_endian_8(&DGB_ptr->thekx2);
			swap_endian_8(&DGB_ptr->polexp);
			swap_endian_8(&DGB_ptr->thefac);
			swap_endian_4(&DGB_ptr->aztlen);
			swap_endian_4(&DGB_ptr->memhandle);
			swap_endian_4(&DGB_ptr->abslen);
			swap_endian_2(&DGB_ptr->refzylmit);
			swap_endian_2(&DGB_ptr->refsaumit);
			swap_endian_2(&DGB_ptr->refausmit);
			swap_endian_2(&DGB_ptr->azttyp);
			swap_endian_4(&DGB_ptr->aztofs);
			swap_endian_4(&DGB_ptr->blkofs);
			swap_endian_4(&DGB_ptr->blkdur);
			swap_endian_4(&DGB_ptr->blknum);
			swap_endian_4(&DGB_ptr->synadr);
			swap_endian_4(&DGB_ptr->synlen);
		}

		if (DGB_ptr->dltzei < DBL_EPSILON)
		{
			/* Some old Indimaster files have dltzei as zero, prevents calculation of cycle speed */

			DGB_ptr->dltzei = 0.01;

			logmessage(DEBUG, "IFile dltzei zero changed to %f\n",DGB_ptr->dltzei);
		}

		DGB_ptr->absein[9] = 0x00;

		logmessage(DEBUG,"Data Group:                 %u\n",dgb);
		logmessage(DEBUG,"Data Group Type:            ");

		switch (datart_type(DGB_ptr->datart, (char*)DGB_ptr->absein, DGB_ptr->zyklng))
		{
		case ABSCISSA_CRANKANGLE:
		{
			logmessage(DEBUG,"Crank Angle Related\n");
			break;
		}
		case ABSCISSA_TIME:
		{
			logmessage(DEBUG,"Time Related\n");
			break;
		}
		case ABSCISSA_CYCLE:
		{
			if (DGB_ptr->datart == 4)
			{
				logmessage(DEBUG, "Results\n");
			}
			else
			{
				logmessage(DEBUG, "Realtime Results (RTP)\n");
			}
			break;
		}
		case ABSCISSA_ASYNC_UTC:
		{
			logmessage(DEBUG,"Asynchronous UTC Data\n");
			break;
		}
		case ABSCISSA_UNKNOWN:
		default:
		{
			logmessage(DEBUG,"Unknown (%d)\n", DGB_ptr->datart);
			break;
		}
		}

		logmessage(DEBUG,"Number of Channels:         %i\n",DGB_ptr->kananz);
		logmessage(DEBUG,"Number of Cycles:           %i\n",DGB_ptr->zykanz);
		logmessage(DEBUG,"Abscissa Type:              %i\n",DGB_ptr->absart);
		logmessage(DEBUG,"Abscissa Length:            %i\n",DGB_ptr->abslen);
		logmessage(DEBUG,"Abscissa Unit:              %s\n",DGB_ptr->absein);
		logmessage(DEBUG,"Abscissa Tables:            %i\n",DGB_ptr->beranz);
		logmessage(DEBUG,"Data Directory Position:    0x%p\n",DGB_ptr->diradr);
		logmessage(DEBUG,"Absolute Time Address:      0x%p\n",DGB_ptr->absadr);
		logmessage(DEBUG,"Measurement Table Position: 0x%p\n",DGB_ptr->mpladr);
		logmessage(DEBUG,"AZT Address:                0x%p\n",DGB_ptr->aztadr);
		logmessage(DEBUG,"RZT Address:                0x%p\n",DGB_ptr->rztadr);
		logmessage(DEBUG,"AZT Length:                 %i\n",DGB_ptr->aztlen);
		logmessage(DEBUG,"CDM Address:                0x%p\n",DGB_ptr->synadr);
		logmessage(DEBUG,"CDM Length:                 %i\n",DGB_ptr->synlen);
		logmessage(DEBUG,"Data Length:                %i\n",DGB_ptr->datfor);
		logmessage(DEBUG,"fortyp: (Data Type)         %i\n",DGB_ptr->fortyp);
		logmessage(DEBUG,"zyklng: (Cycle Length)      %i\n",DGB_ptr->zyklng);

		if (datart_type(DGB_ptr->datart, (char*)DGB_ptr->absein,DGB_ptr->zyklng) == ABSCISSA_TIME)
		{
			logmessage(DEBUG, "dltphi: (ms)                %f\n", DGB_ptr->dltphi);
		}
		else
		{
			logmessage(DEBUG, "dltphi: (CA deg)            %f\n", DGB_ptr->dltphi);
		}
		
		logmessage(DEBUG,"dltzei: (ms)                %f\n",DGB_ptr->dltzei);
		logmessage(DEBUG,"Offset Correction\n");
		logmessage(DEBUG,"Angle 1:                    %f\n",DGB_ptr->thekx1);
		logmessage(DEBUG,"Angle 2:                    %f\n",DGB_ptr->thekx2);
		logmessage(DEBUG,"Polytropic exponent:        %f\n",DGB_ptr->polexp);
		logmessage(DEBUG,"Zero level channel:         %s (over %i samples)\n", DGB_ptr->refzyl, DGB_ptr->refzylmit);
		logmessage(DEBUG,"Intake channel:             %s (over %i samples)\n", DGB_ptr->refsau, DGB_ptr->refsaumit);
		logmessage(DEBUG,"Exhaust channel:            %s (over %i samples)\n", DGB_ptr->refaus, DGB_ptr->refausmit);
		logmessage(DEBUG,"CDMRNGST:                   %f\n",(float)DGB_ptr->refanf*DGB_ptr->dltphi);
	}
	
    logmessage(NOTICE,"done\n");

	DGB_end = ftell(input_file);
	
	for (dgb=0;dgb<number_of_data_groups;dgb++)
	{
		logmessage(NOTICE,"Datagroup: %u of %u (type %d)\n",dgb+1,number_of_data_groups,file->ptr_DGB[dgb].datart);
		
		if ((file->ptr_DGB[dgb].diradr != NULL) &&
			(file->ptr_DGB[dgb].datfor <= 16) &&
			(valid_dgb(file->ptr_DGB[dgb].datart, grpflg, (char*)(file->ptr_DGB[dgb].absein), file->ptr_DGB[dgb].zyklng) == true))
		{
			if (datart_type(file->ptr_DGB[dgb].datart, (char*)file->ptr_DGB[dgb].absein, file->ptr_DGB[dgb].zyklng) == ABSCISSA_CRANKANGLE)
			{
				if (file->ptr_DGB[dgb].dltphi > 0.0)
				{
					Change_Minimum_Resolution(file,(unsigned int)(1.0 / file->ptr_DGB[dgb].dltphi * 360.0 + 0.001));

					if (file->min_res_ppr != DEFAULT_MIN_RES_PPR)
					{
						logmessage(NOTICE,"Minimum resolution (%u) is not default (%u) pulses per rev!\n",file->min_res_ppr,DEFAULT_MIN_RES_PPR);
					}
				}
				else
				{
					logmessage(MSG_ERROR,"I-File resolution is zero\n");

					//fclose(input_file);
					//return(false);
				}
			}
			else if (datart_type(file->ptr_DGB[dgb].datart, (char*)file->ptr_DGB[dgb].absein, file->ptr_DGB[dgb].zyklng) == ABSCISSA_TIME)
			{
				/*file->time_resolution = file->ptr_DGB[dgb].dltphi;

				if (file->time_resolution < FLT_EPSILON)
				{
					file->time_resolution = DEFAULT_TIME_RESOLUTION;
				}*/
			}
			else
			{
				/* Do Nothing */
			}
		
			if (debug_level >= DEBUG)
			{
				logmessage(DEBUG,"Number of channels: %i\n",file->ptr_DGB[dgb].kananz);
				logmessage(DEBUG,"Number of cycles: %i\n",file->ptr_DGB[dgb].zykanz);
				logmessage(DEBUG,"Number of measurements per cycle: %i\n",file->ptr_DGB[dgb].beranz);
				logmessage(DEBUG,"Basic resolution: %f\n",file->ptr_DGB[dgb].dltphi);
				logmessage(DEBUG,"Polytropic index: %f\n",file->ptr_DGB[dgb].polexp);
				logmessage(DEBUG,"min_res_ppr: %d\n",file->min_res_ppr);
				logmessage(DEBUG,"time_resolution: %f (milliseconds)\n",file->time_resolution);
			}
			
			/* Load Data Directory (DI) */
			
			logmessage(NOTICE,"Loading DI...");
	
			/* The following line can give a compiler warning because ptr_DI is used as a pointer to DATDIR structure,
			   whereas in the file it is used as a file pointer. */
			
			if (file->ptr_DGB[dgb].diradr > (DATDIR*)(size_t)filesize)
			{
				fclose(input_file);

				logmessage(MSG_ERROR,"IFile ptr_DI points outside size of file\n");

				return(false);
			}

			if (fseek(input_file,(long)(size_t)file->ptr_DGB[dgb].diradr,SEEK_SET) != 0)
			{
				fclose(input_file);

				logmessage(MSG_ERROR, "Could not set file position to start of data\n");

				return(false);
			}

			file->ptr_DGB[dgb].diradr = (DATDIR*)malloc(file->ptr_DGB[dgb].kananz * sizeof(DATDIR));

			if (file->ptr_DGB[dgb].diradr == NULL)
			{
				fclose(input_file);
				logmessage(FATAL,"Memory could not be allocated\n");
				return(false);
			}
	
    		for (channel_no=0;channel_no<(unsigned int)file->ptr_DGB[dgb].kananz;channel_no++)
    		{
    			ptr_DI = (DATDIR*)&file->ptr_DGB[dgb].diradr[channel_no];
    			
                (void)fread(ptr_DI->signam,10,1,input_file);
    			(void)fread(ptr_DI->sigein,10,1,input_file);
				(void)fread(ptr_DI->sigkom,6,1,input_file);
    			(void)fread(&ptr_DI->minimum,4,1,input_file);
    			(void)fread(&ptr_DI->maximum,4,1,input_file);
    			(void)fread(&ptr_DI->dstov,8,1,input_file);
    			(void)fread(&ptr_DI->kstov,8,1,input_file);
    			(void)fread(&ptr_DI->abskor,4,1,input_file);
    			(void)fread(&ptr_DI->desori,2,1,input_file);
    			(void)fread(&ptr_DI->zykofs,4,1,input_file);
    			(void)fread(&ptr_DI->kaltyp,2,1,input_file);
    			(void)fread(&ptr_DI->nultyp,2,1,input_file);
    			(void)fread(&ptr_DI->sigtyp,2,1,input_file);
    			(void)fread(&ptr_DI->bitnum,2,1,input_file);
    			(void)fread(&ptr_DI->mitofs,4,1,input_file);
    			(void)fread(&ptr_DI->nulwrt,8,1,input_file);
    			(void)fread(&ptr_DI->kalfak[0],8,1,input_file);
    			(void)fread(&ptr_DI->kalfak[1],8,1,input_file);
    			(void)fread(&ptr_DI->intern[0],4,1,input_file);
    			(void)fread(&ptr_DI->intern[1],4,1,input_file);
    			(void)fread(&ptr_DI->intern[2],4,1,input_file);
    			(void)fread(&ptr_DI->intern[3],4,1,input_file);
    			(void)fread(&ptr_DI->intern[4],4,1,input_file);
    			(void)fread(&ptr_DI->intern[5],4,1,input_file);
    			(void)fread(&ptr_DI->refwrt,8,1,input_file);
    
    			if (file->swap_endian != false)
    			{
    				swap_endian_4(&ptr_DI->minimum);
    				swap_endian_4(&ptr_DI->maximum);
    				swap_endian_8(&ptr_DI->dstov);
    				swap_endian_8(&ptr_DI->kstov);
    				swap_endian_4(&ptr_DI->abskor);
    				swap_endian_2(&ptr_DI->desori);
    				swap_endian_4(&ptr_DI->zykofs);
    				swap_endian_2(&ptr_DI->kaltyp);
    				swap_endian_2(&ptr_DI->nultyp);
    				swap_endian_2(&ptr_DI->sigtyp);
    				swap_endian_2(&ptr_DI->bitnum);
    				swap_endian_4(&ptr_DI->mitofs);
    				swap_endian_8(&ptr_DI->nulwrt);
    				swap_endian_8(&ptr_DI->kalfak[0]);
    				swap_endian_8(&ptr_DI->kalfak[1]);
    				swap_endian_4(&ptr_DI->intern[0]);
    				swap_endian_4(&ptr_DI->intern[1]);
    				swap_endian_4(&ptr_DI->intern[2]);
    				swap_endian_4(&ptr_DI->intern[3]);
    				swap_endian_4(&ptr_DI->intern[4]);
    				swap_endian_4(&ptr_DI->intern[5]);
    				swap_endian_8(&ptr_DI->refwrt);
    			}

				ptr_DI->signam[9] = 0x00;
				ptr_DI->sigein[9] = 0x00;
    			ptr_DI->sigkom[5] = 0x00;
    		}

			logmessage(DEBUG,"done\n");
		
			logmessage(DEBUG,"+-----+------------+------------+------+----------------+----------------+---------+\n");
			logmessage(DEBUG,"| Chn | Name       | Units      | Type | Slope          | Offset         | Samples |\n");
			logmessage(DEBUG,"+-----+------------+------------+------+----------------+----------------+---------+\n");
				
			for (channel=0;channel<(unsigned int)file->ptr_DGB[dgb].kananz;channel++)
			{
				logmessage(DEBUG,"| %3u | %-10s | %-10s | %4u | %14.4f | %14.4f | %7i |\n",channel,file->ptr_DGB[dgb].diradr[channel].signam,file->ptr_DGB[dgb].diradr[channel].sigein,file->ptr_DGB[dgb].diradr[channel].sigtyp,file->ptr_DGB[dgb].diradr[channel].kalfak[1],file->ptr_DGB[dgb].diradr[channel].kalfak[0],file->ptr_DGB[dgb].diradr[channel].zykofs);
			}

			logmessage(DEBUG,"+-----+------------+------------+------+----------------+----------------+---------+\n");
		
			/* Load Measurement Table (MT) */
			
			logmessage(NOTICE,"Loading MT...");
			
			if (file->ptr_DGB[dgb].mpladr > (MPTAB*)(size_t)filesize)
			{
				fclose(input_file);

				logmessage(MSG_ERROR,"IFile mpladr points outside size of file\n");

				return(false);
			}

			/* The following line can give a compiler warning because mpladr is used as a pointer to MPTAB structure,
			   whereas in the file it is used as a file pointer. */

			if (fseek(input_file,(long)(size_t)file->ptr_DGB[dgb].mpladr,SEEK_SET) != 0)
			{
				fclose(input_file);

				logmessage(MSG_ERROR, "Could not set file position to start of data\n");

				return(false);
			}
				
			mbllen = file->ptr_DGB[dgb].kananz * file->ptr_DGB[dgb].beranz;
		
			file->ptr_DGB[dgb].mpladr = (MPTAB*)malloc(mbllen * sizeof(MPTAB));
		
			if (file->ptr_DGB[dgb].mpladr == NULL)
			{	
				fclose(input_file);
				logmessage(FATAL,"Memory could not be allocated\n");
				return(false);
			}
	
    		for (mt_number=0;mt_number<mbllen;mt_number++)
    		{
                (void)fread(&file->ptr_DGB[dgb].mpladr[mt_number].beranf,4,1,input_file);
    			(void)fread(&file->ptr_DGB[dgb].mpladr[mt_number].wrtanz,4,1,input_file);
    			(void)fread(&file->ptr_DGB[dgb].mpladr[mt_number].absint,4,1,input_file);
    			(void)fread(&file->ptr_DGB[dgb].mpladr[mt_number].beradr,4,1,input_file);
    			(void)fread(&file->ptr_DGB[dgb].mpladr[mt_number].adrint,4,1,input_file);

    			if (file->swap_endian != false)
    			{
    				swap_endian_4(&file->ptr_DGB[dgb].mpladr[mt_number].beranf);
    				swap_endian_4(&file->ptr_DGB[dgb].mpladr[mt_number].wrtanz);
    				swap_endian_4(&file->ptr_DGB[dgb].mpladr[mt_number].absint);
    				swap_endian_4(&file->ptr_DGB[dgb].mpladr[mt_number].beradr);
    				swap_endian_4(&file->ptr_DGB[dgb].mpladr[mt_number].adrint);
    			}
    		}
			
			logmessage(NOTICE,"done\n");
		
			/* Load Absolute Time Table (AZT) */
		
			if (file->ptr_DGB[dgb].aztadr != NULL)
			{
				logmessage(NOTICE,"Loading AZT...");
				
				if ((file->ptr_DGB[dgb].aztadr > (int32_t*)(size_t)filesize) ||
					(file->ptr_DGB[dgb].aztadr < (int32_t*)1024))
				{
					fclose(input_file);

					logmessage(MSG_ERROR,"IFile aztadr points outside size of file\n");

					return(false);
				}

				/* The following line can give a compiler warning because aztadr is used as a pointer to long, whereas
				   in the file it is used as a file pointer. */
	
				if (fseek(input_file,(long)(size_t)file->ptr_DGB[dgb].aztadr,SEEK_SET) != 0)
				{
					fclose(input_file);

					logmessage(MSG_ERROR, "Could not set file position to start of data\n");

					return(false);
				}

				aztlen = file->ptr_DGB[dgb].aztlen;

				file->ptr_DGB[dgb].aztadr = (int*)malloc(aztlen);
				
				if (file->ptr_DGB[dgb].aztadr == NULL)
				{
					fclose(input_file);
					logmessage(FATAL,"Memory could not be allocated\n");
					return(false);
				}
				
				(void)fread(file->ptr_DGB[dgb].aztadr,aztlen,1,input_file);
				
				if (file->swap_endian != false)
				{
					for (azt=0;azt<aztlen/4;azt++)
					{
						swap_endian_4(&file->ptr_DGB[dgb].aztadr[azt]);
					}
				}
				
				logmessage(NOTICE,"done\n");
			}
			
			/* Load Relative Time Table (RZT) */
		
			if (file->ptr_DGB[dgb].rztadr != NULL)
			{
				logmessage(NOTICE,"Loading RZT...");

				if ((file->ptr_DGB[dgb].rztadr > (int16_t*)(size_t)filesize) ||
					(file->ptr_DGB[dgb].rztadr < (int16_t*)1024))
				{
					fclose(input_file);

					logmessage(MSG_ERROR,"IFile rztadr points outside size of file\n");

					return(false);
				}

				/* The following line can give a compiler warning because rztadr is used as a pointer to short, whereas
				   in the file it is used as a file pointer. */
	
				if (fseek(input_file,(long)(size_t)file->ptr_DGB[dgb].rztadr,SEEK_SET) != 0)
				{
					fclose(input_file);

					logmessage(MSG_ERROR, "Could not set file position to start of data\n");

					return(false);
				}
				
				file->ptr_DGB[dgb].rztadr = (short*)malloc(file->ptr_DGB[dgb].zykanz * sizeof(short));
				
				if (file->ptr_DGB[dgb].rztadr == NULL)
				{
					fclose(input_file);
					logmessage(FATAL,"Memory could not be allocated\n");
					return(false);
				}
				
				(void)fread(file->ptr_DGB[dgb].rztadr,2,(size_t)file->ptr_DGB[dgb].zykanz,input_file);

				if (file->swap_endian != false)
				{
					for (rzt=0;rzt<(unsigned int)file->ptr_DGB[dgb].zykanz;rzt++)
					{
						swap_endian_2(&file->ptr_DGB[dgb].rztadr[rzt]);
					}
				}
	
				logmessage(NOTICE,"done\n");
			}
			
			file->number_of_channels += file->ptr_DGB[dgb].kananz;

	        if (datart_type(file->ptr_DGB[dgb].datart, (char*)file->ptr_DGB[dgb].absein, file->ptr_DGB[dgb].zyklng) == ABSCISSA_CRANKANGLE)
	        {
				if ((file->ptr_DGB[dgb].zykanz > 1) && ((file->ptr_DGB[dgb].zykanz < (int)file->number_of_cycles) || (file->number_of_cycles <= 1)))
				{
					file->number_of_cycles = file->ptr_DGB[dgb].zykanz;
				}
			}
		}
		else
		{
			logmessage(NOTICE,"Not configured to load this data group (datfor %d)\n", file->ptr_DGB[dgb].datfor);
				
			file->ptr_DGB[dgb].diradr = NULL;
			file->ptr_DGB[dgb].mpladr = NULL;
			file->ptr_DGB[dgb].aztadr = NULL;
			file->ptr_DGB[dgb].rztadr = NULL;
		}
	}
	
	header_end = ftell(input_file);

	/* Set Channel Data */
	
	logmessage(DEBUG,"Number of channels: %u\n",file->number_of_channels);
	logmessage(DEBUG,"End of APB filepos: %li\n",APB_end);
	logmessage(DEBUG,"End of DGB filepos: %li\n",DGB_end);
	logmessage(DEBUG,"End of DI/MT/AZT/RZT of CA data: %li\n",header_end);

	/* Allocate Channel data memory */
    
	file->channel_data = (ChannelData*)malloc(file->number_of_channels * sizeof(ChannelData));
	if (file->channel_data == NULL)
	{
		fclose(input_file);
		logmessage(FATAL,"Memory could not be allocated\n");
		return(false);
	}
	
	logmessage(NOTICE,"Configuring channels...");
	
	start_channel = 0;
	for (dgb=0;dgb<number_of_data_groups;dgb++)
	{
		if ((file->ptr_DGB[dgb].diradr != NULL) &&
			(file->ptr_DGB[dgb].datfor <= 16) && 
			(valid_dgb(file->ptr_DGB[dgb].datart, grpflg, (char*)(file->ptr_DGB[dgb].absein), file->ptr_DGB[dgb].zyklng) == true))
		{
			for (dgb_channel=0;dgb_channel<file->ptr_DGB[dgb].kananz;dgb_channel++)
			{
				channel = start_channel + dgb_channel;
				
                ptr_DI = (DATDIR*)&file->ptr_DGB[dgb].diradr[dgb_channel];

            	if (ptr_DI == NULL)
            	{
            		logmessage(MSG_ERROR,"AVL I-File channel %u has a null directory\n",dgb_channel);

            		fclose(input_file);
            		
            		return(false);
            	}
	
    			ptr_MT = (MPTAB*)file->ptr_DGB[dgb].mpladr + dgb_channel*file->ptr_DGB[dgb].beranz;
    
            	if (ptr_MT == NULL)
            	{
            		logmessage(MSG_ERROR,"AVL I-File channel %u has a null measurement table\n",dgb_channel);

            		fclose(input_file);
            		
            		return(false);
            	}

				if (ptr_DI->signam[0] == 0x00)
				{
					snprintf((char*)ptr_DI->signam, 10, "UNK_%u", unknown_channel_number);
					ptr_DI->signam[9] = 0x00;
					unknown_channel_number++;
				}

				Zero_Abscissa(&file->channel_data[channel].abscissa);
            	
				file->channel_data[channel].offset = ptr_DI->kalfak[0];
				file->channel_data[channel].slope = ptr_DI->kalfak[1];

				file->channel_data[channel].conversion_factor = 1.0f;
				file->channel_data[channel].tdc_offset = (float)(ptr_DI->abskor * file->ptr_DGB[dgb].dltphi);
				strncpy(file->channel_data[channel].name,(char*)ptr_DI->signam,10);
				file->channel_data[channel].name[9] = 0x00;
				strncpy(file->channel_data[channel].short_name,(char*)ptr_DI->signam,10);
				file->channel_data[channel].short_name[9] = 0x00;
				file->channel_data[channel].matlab_name[0] = 0x00;
				strncpy(file->channel_data[channel].units,(char*)ptr_DI->sigein,10);
				file->channel_data[channel].units[9] = 0x00;
				strncpy(file->channel_data[channel].abscissa.units,(char*)file->ptr_DGB[dgb].absein,10);
				file->channel_data[channel].abscissa.units[9] = 0x00;

				file->channel_data[channel].type = CHAN_UNKNOWN;

				file->channel_data[channel].cylinder = determine_cylinder_number(file->channel_data[channel].name, &max_cylinder_number);

				logmessage(DEBUG, "%s tdc_offset: %f\n", file->channel_data[channel].name, file->channel_data[channel].tdc_offset);

				if ((file->channel_data[channel].cylinder > 0) && (file->channel_data[channel].tdc_offset > file->engine.tdc_offset[file->channel_data[channel].cylinder - 1]))
				{
					file->engine.tdc_offset[file->channel_data[channel].cylinder - 1] = file->channel_data[channel].tdc_offset;
				}

				logmessage(DEBUG, "desori value (%i) for channel %s (%u) (cylinder %u)\n", ptr_DI->desori, file->channel_data[channel].name, channel, file->channel_data[channel].cylinder);

				if ((file->channel_data[channel].cylinder >= 1) && (file->channel_data[channel].cylinder <= MAX_NUMBER_OF_CYLINDERS))
				{
					switch (ptr_DI->desori)
					{
					default:
					{
						logmessage(DEBUG, "desori value (%i) invalid for channel %u (cylinder %u).  Using desaxi pin offset.\n", ptr_DI->desori, channel, file->channel_data[channel].cylinder);

						file->engine.pin_offset[file->channel_data[channel].cylinder - 1] = pin_offset;

						break;
					}
					case 0:
					{
						/* Do Nothing */
						break;
					}
					case 1:
					{
						file->engine.pin_offset[file->channel_data[channel].cylinder - 1] = pin_offset;
						break;
					}
					case -1:
					{
						file->engine.pin_offset[file->channel_data[channel].cylinder - 1] = -1.0f * pin_offset;
						break;
					}
					case 2:
					{
						file->engine.pin_offset[file->channel_data[channel].cylinder - 1] = pin_offset_2;
						break;
					}
					case -2:
					{
						file->engine.pin_offset[file->channel_data[channel].cylinder - 1] = -1.0f * pin_offset_2;
						break;
					}
					}
				}

				file->channel_data[channel].abscissa.type = datart_type(file->ptr_DGB[dgb].datart, (char*)file->ptr_DGB[dgb].absein, file->ptr_DGB[dgb].zyklng);

				switch (file->channel_data[channel].abscissa.type)
				{
				case ABSCISSA_CRANKANGLE:
				{
					if ((file->ptr_DGB[dgb].rztadr != NULL) || (file->ptr_DGB[dgb].aztadr != NULL))
					{
						file->ca_dgb = dgb;
					}

					/* Intentional fallthrough */					
				}
				case ABSCISSA_TIME:
				{		
					switch (ptr_DI->sigtyp)
					{
					default:
					case 0:		/* No Type Defined */
					case 3:		/* Misc */
					{
						if (file->channel_data[channel].abscissa.type == ABSCISSA_CRANKANGLE)
						{
							if ((case_insensitive_compare((char*)ptr_DI->signam,"CYLPR",5) == 1) ||
								(case_insensitive_compare((char*)ptr_DI->signam,"ZYLDR",5) == 1) ||
							    (case_insensitive_compare((char*)ptr_DI->signam,"PCYL",4) == 1) ||
							    (case_insensitive_compare((char*)ptr_DI->signam,"PZYL",4) == 1) ||
								(case_insensitive_compare((char*)ptr_DI->signam,"DZYL",4) == 1) ||
							    (case_insensitive_compare((char*)ptr_DI->signam,"P_CYL",5) == 1) ||
							    (case_insensitive_compare((char*)ptr_DI->signam,"P_ZYL",5) == 1) ||
								(case_insensitive_compare((char*)ptr_DI->signam,"D_ZYL", 5) == 1) ||
								(case_insensitive_compare((char*)ptr_DI->signam,"Cyl_",4) == 1))
							{
								file->channel_data[channel].type = CHAN_CYL_PR;
							}
							else if ((case_insensitive_compare((char*)ptr_DI->signam,"NLIFT",5) == 1) ||
							         (case_insensitive_compare((char*)ptr_DI->signam,"NEEDLE",6) == 1))
							{
								file->channel_data[channel].type = CHAN_NEEDLE_LIFT;
							}
							else if ((case_insensitive_compare((char*)ptr_DI->signam,"PLIN",4) == 1) ||
							         (case_insensitive_compare((char*)ptr_DI->signam,"INJ",3) == 1))
							{
								file->channel_data[channel].type = CHAN_INJ_PR;
							}
							else if ((case_insensitive_compare((char*)ptr_DI->signam,"SPARK",5) == 1) ||
							         (case_insensitive_compare((char*)ptr_DI->signam,"SPRK",4) == 1) ||
							         (case_insensitive_compare((char*)ptr_DI->signam,"SPK",3) == 1) ||
							         (case_insensitive_compare((char*)ptr_DI->signam,"IGN",3) == 1))
							{
								file->channel_data[channel].type = CHAN_IGN_ANG;
							}
							else if ((case_insensitive_compare((char*)ptr_DI->signam,"PMAN",4) == 1) ||
								     (case_insensitive_compare((char*)ptr_DI->signam, "PSAUG", 5) == 1))
							{
								file->channel_data[channel].type = CHAN_INLET_PR;
							}
							else if ((case_insensitive_compare((char*)ptr_DI->signam,"PEXH",4) == 1) ||
								     (case_insensitive_compare((char*)ptr_DI->signam, "PAUSP", 5) == 1))
							{
								file->channel_data[channel].type = CHAN_EXH_PR;
							}
							else if ((case_insensitive_compare((char*)ptr_DI->signam,"KNKB",4) == 1) ||
							         (case_insensitive_compare((char*)ptr_DI->signam,"KNKA",4) == 1) ||
								     (case_insensitive_compare((char*)ptr_DI->signam,"KNKM",4) == 1))
							{
								file->channel_data[channel].type = CHAN_BLOCK_ACC;
							}
							else if ((case_insensitive_compare((char*)ptr_DI->signam,"CAMIN",5) == 1) ||
							         (case_insensitive_compare((char*)ptr_DI->signam,"CAMEX",5) == 1))
							{
								file->channel_data[channel].type = CHAN_CID;
							}
							else
							{
								file->channel_data[channel].type = CHAN_UNKNOWN;
							}
						}

						break;
					}
					case 1:		/* Cylinder Pressure Signal */
					{
						file->channel_data[channel].type = CHAN_CYL_PR;

						break;
					}
					case 2:		/* Needle Lift Signal */
					{
						file->channel_data[channel].type = CHAN_NEEDLE_LIFT;
						break;
					}
					case 6:		/* dT ? (Time/degree) */
					{
						file->channel_data[channel].type = CHAN_UNKNOWN;
						break;
					}
					case 10:	/* Injection Pressure */
					{
						file->channel_data[channel].type = CHAN_INJ_PR;
						break;
					}
					case 11:	/* Ignition Angle */
					{
						file->channel_data[channel].type = CHAN_IGN_ANG;
						break;
					}
					case 14:	/* Knock boss acceleration */
					{
						file->channel_data[channel].type = CHAN_BLOCK_ACC;
						break;
					}
					case 16:	/* Intake manifold pressure */
					{
						file->channel_data[channel].type = CHAN_INLET_PR;
						break;
					}
					case 17:	/* Exhaust manifold pressure */
					{	
						file->channel_data[channel].type = CHAN_EXH_PR;
						break;
					}
					case 22:	/* Tooth periods */
					{
						file->channel_data[channel].type = CHAN_TOOTH_PERIODS;
						break;
					}
					case 33:	/* CDMTIME */
					{
						file->channel_data[channel].type = CHAN_TIME;
						break;
					}
					}
					break;
				}
				case ABSCISSA_CYCLE:
				{
					file->channel_data[channel].type = CHAN_RESULTS;
					break;
				}
				case ABSCISSA_ASYNC_UTC:
				{
					switch (ptr_DI->sigtyp)
					{
					case 4:		/* UTC bit channel */
					case 5:		/* UTC digital channel */
					{
						file->channel_data[channel].type = CHAN_UNKNOWN;
						break;
					}
					default:
					{
						file->channel_data[channel].type = CHAN_UNKNOWN;
						break;
					}
					}
					break;
				}
				case ABSCISSA_UNKNOWN:
				default:
				{
					file->channel_data[channel].type = CHAN_UNKNOWN;
					break;
				}
				}

				convert = false;

				if (case_insensitive_compare(file->channel_data[channel].units, "psi", SIZEOF_UNITS) == 1)
				{
					file->channel_data[channel].conversion_factor = PSI_TO_BAR;
					convert = true;
				}
				else if (case_insensitive_compare(file->channel_data[channel].units, "kPa", SIZEOF_UNITS) == 1)
				{
					file->channel_data[channel].conversion_factor = KPA_TO_BAR;
					convert = true;
				}
				else if (case_insensitive_compare(file->channel_data[channel].units, "Pa", SIZEOF_UNITS) == 1)
				{
					file->channel_data[channel].conversion_factor = PA_TO_BAR;
					convert = true;
				}
				else if (case_insensitive_compare(file->channel_data[channel].units, "MPa", SIZEOF_UNITS) == 1)
				{
					file->channel_data[channel].conversion_factor = MPA_TO_BAR;
					convert = true;
				}
				else
				{
					file->channel_data[channel].conversion_factor = 1.0f;

				}

				if ((file->channel_data[channel].type == CHAN_CYL_PR) && (convert == true))
				{
					file->channel_data[channel].offset *= (double)file->channel_data[channel].conversion_factor;
					file->channel_data[channel].slope *= (double)file->channel_data[channel].conversion_factor;

					file->channel_data[channel].conversion_factor = 1.0f;
				        
				    strncpy(file->channel_data[channel].units,"bar",10);
                }

				if (file->channel_data[channel].cylinder == 0)
				{
					strncpy(file->channel_data[channel].description, channel_information[file->channel_data[channel].type].description, SIZEOF_DESCRIPTION);
				}
				else
				{
					snprintf(file->channel_data[channel].description, SIZEOF_DESCRIPTION, "%s for Cylinder %u", channel_information[file->channel_data[channel].type].description, file->channel_data[channel].cylinder);
				}

				file->channel_data[channel].description[SIZEOF_DESCRIPTION-1] = 0x00;

				file->channel_data[channel].offset_config.type = OFFSET_NONE;
				file->channel_data[channel].offset_config.channel_name[0] = 0x00;
				file->channel_data[channel].offset_config.window_size = 5;
				file->channel_data[channel].offset_config.fixed_value = 0.0f;
				file->channel_data[channel].offset_config.start_window = (float)file->ptr_DGB[dgb].thekx1;
				file->channel_data[channel].offset_config.finish_window = (float)file->ptr_DGB[dgb].thekx2;
				file->channel_data[channel].offset_config.polytropic_index = (float)file->ptr_DGB[dgb].polexp;
				file->channel_data[channel].offset_config.truncate = false;

				if (file->channel_data[channel].offset_config.start_window > file->channel_data[channel].offset_config.finish_window)
				{
					float tmp = file->channel_data[channel].offset_config.start_window;
					file->channel_data[channel].offset_config.start_window = file->channel_data[channel].offset_config.finish_window;
					file->channel_data[channel].offset_config.finish_window = tmp;
				}

				if (file->channel_data[channel].offset_config.polytropic_index < 1.0f)
				{
					file->channel_data[channel].offset_config.polytropic_index = 1.0f;
				}

				if (file->channel_data[channel].offset_config.polytropic_index > 2.0f)
				{
					file->channel_data[channel].offset_config.polytropic_index = 2.0f;
				}

				file->channel_data[channel].soc_config.type = SOC_FIXED;
				file->channel_data[channel].soc_config.channel_name[0] = 0x00;
				file->channel_data[channel].soc_config.fixed_value = -20.0f;
				file->channel_data[channel].soc_config.start_window = -60.0f;
				file->channel_data[channel].soc_config.finish_window = 60.0f;
				file->channel_data[channel].soc_config.aligned = true;
				file->channel_data[channel].soc_config.invert = false;

				file->channel_data[channel].filter_config.type = FILTER_NONE;
				file->channel_data[channel].filter_config.filtered = false;
				file->channel_data[channel].filter_config.upper_frequency = 500.0f;
				file->channel_data[channel].filter_config.lower_frequency = 500.0f;

				file->channel_data[channel].digital_config.type = DIGITIZE_AUTO;
				file->channel_data[channel].digital_config.invert = false;
				file->channel_data[channel].digital_config.latch_high = 2.0f;
				file->channel_data[channel].digital_config.latch_low = 1.0f;
				file->channel_data[channel].digital_config.filter = 1.0f;

				file->channel_data[channel].cam_config.edge = EDGE_BOTH;
				file->channel_data[channel].cam_config.invert = false;
				file->channel_data[channel].cam_config.offset = 0.0f;
				file->channel_data[channel].cam_config.reference_angle = 0.0f;
				file->channel_data[channel].cam_config.type = CAMTYPE_NONE;

				file->channel_data[channel].isoffset = false;
				file->channel_data[channel].loaded = false;
				file->channel_data[channel].file_flag = false;
				file->channel_data[channel].data = NULL;
				file->channel_data[channel].data_d = NULL;
				file->channel_data[channel].duration = NULL;
				
            	file->channel_data[channel].abscissa.number_of_measurement_tables = 0;
            
            	file->channel_data[channel].samples_per_cycle = 0;
            
            	logmessage(DEBUG,"\n+----------------------------------------------+\n");
            	logmessage(DEBUG,"|        Channel %3u Measurement Tables        |\n",channel);
            	logmessage(DEBUG,"+-----+---------+---------+-------+------------+\n");
            	logmessage(DEBUG,"| No  | Start   | Samples | Res   | Data Start |\n");
            	logmessage(DEBUG,"+-----+---------+---------+-------+------------+\n");
            	
            	for (measurement_table=0;measurement_table<(unsigned int)file->ptr_DGB[dgb].beranz;measurement_table++)
            	{		
            		if ((ptr_MT[measurement_table].wrtanz > 0) && (ptr_MT[measurement_table].adrint > 0) && (ptr_MT[measurement_table].absint > 0))
            		{
						if (file->channel_data[channel].abscissa.type == ABSCISSA_CRANKANGLE)
						{
            				logmessage(DEBUG,"| %2u  | %7.2f | %7d | %5.2f | %10d |\n",measurement_table,ptr_MT[measurement_table].beranf*file->ptr_DGB[dgb].dltphi,ptr_MT[measurement_table].wrtanz,ptr_MT[measurement_table].absint*file->ptr_DGB[dgb].dltphi,ptr_MT[measurement_table].beradr);
						}
						else
						{
							logmessage(DEBUG,"| %2u  | %7d | %7d | %5d | %10d |\n",measurement_table,ptr_MT[measurement_table].beranf,ptr_MT[measurement_table].wrtanz,ptr_MT[measurement_table].absint,ptr_MT[measurement_table].beradr);
						}
            			file->channel_data[channel].abscissa.number_of_measurement_tables += 1;
            			file->channel_data[channel].samples_per_cycle += ptr_MT[measurement_table].wrtanz;
            		}
            		else
            		{
						if (file->channel_data[channel].abscissa.type == ABSCISSA_CRANKANGLE)
						{
							logmessage(DEBUG,"| %2u  | %7.2f | %7d | %5.2f | %10d | (Invalid)\n",measurement_table,ptr_MT[measurement_table].beranf*file->ptr_DGB[dgb].dltphi,ptr_MT[measurement_table].wrtanz,ptr_MT[measurement_table].absint*file->ptr_DGB[dgb].dltphi,ptr_MT[measurement_table].beradr);
						}
						else
						{
            				logmessage(DEBUG,"| %2u  | %7d | %7d | %5d | %10d | (Invalid)\n",measurement_table,ptr_MT[measurement_table].beranf,ptr_MT[measurement_table].wrtanz,ptr_MT[measurement_table].absint,ptr_MT[measurement_table].beradr);
						}	
            		}
            	}
            	
            	logmessage(DEBUG,"+-----+---------+---------+-------+------------+\n");
		        
            	file->channel_data[channel].abscissa.number_of_samples = (unsigned int*)malloc(file->channel_data[channel].abscissa.number_of_measurement_tables * sizeof(unsigned int));
            	if (file->channel_data[channel].abscissa.number_of_samples == NULL)
            	{
            		fclose(input_file);
            		logmessage(FATAL,"Memory could not be allocated\n");
					return(false);
            	}
            	
            	file->channel_data[channel].abscissa.start = (unsigned int*)malloc(file->channel_data[channel].abscissa.number_of_measurement_tables * sizeof(unsigned int));
            	if (file->channel_data[channel].abscissa.start == NULL)
            	{
            		fclose(input_file);
            		logmessage(FATAL,"Memory could not be allocated\n");
					return(false);
            	}
            
            	file->channel_data[channel].abscissa.resolution = (unsigned int*)malloc(file->channel_data[channel].abscissa.number_of_measurement_tables * sizeof(unsigned int));
            	if (file->channel_data[channel].abscissa.resolution == NULL)
            	{
            		fclose(input_file);
            		logmessage(FATAL,"Memory could not be allocated\n");
					return(false);
            	}
            
            	mt = 0;
            
            	/* TODO - for results data, if start in measurement table is < 0 then ignore data to provide number_of_cycle data points (Concerto Behaviour) */
            
            	file_position = 0;
            	for (measurement_table=0;measurement_table<(unsigned int)file->ptr_DGB[dgb].beranz;measurement_table++)
            	{
            		if ((ptr_MT[measurement_table].wrtanz > 0) && (ptr_MT[measurement_table].adrint > 0) && (ptr_MT[measurement_table].absint > 0))
            		{
            			file->channel_data[channel].abscissa.number_of_samples[mt] = ptr_MT[measurement_table].wrtanz;
            			
            			if (file->channel_data[channel].abscissa.type == ABSCISSA_CRANKANGLE)
            			{
            				file->channel_data[channel].abscissa.start[mt] = DegreesToTheta(file,(float)(ptr_MT[measurement_table].beranf * file->ptr_DGB[dgb].dltphi));
            				file->channel_data[channel].abscissa.resolution[mt] = DegreesToThetaAbs(file,(float)(ptr_MT[measurement_table].absint * file->ptr_DGB[dgb].dltphi));
            			}
						else if (file->channel_data[channel].abscissa.type == ABSCISSA_TIME)
						{
							unsigned int time_factor = (unsigned int)(file->ptr_DGB[dgb].dltphi / file->time_resolution);

							if (ptr_MT[measurement_table].beranf < 0)
							{
								file->channel_data[channel].abscissa.start[mt] = 0U;
							}
							else
							{
								file->channel_data[channel].abscissa.start[mt] = (unsigned int)(ptr_MT[measurement_table].beranf * time_factor);
							}

							file->channel_data[channel].abscissa.resolution[mt] = (unsigned int)(ptr_MT[measurement_table].absint * time_factor);
						}
						else
            			{
							if (ptr_MT[measurement_table].beranf < 0)
							{
								file->channel_data[channel].abscissa.start[mt] = 0U;
							}
							else
							{
								file->channel_data[channel].abscissa.start[mt] = (unsigned int)(ptr_MT[measurement_table].beranf);
							}

            				file->channel_data[channel].abscissa.resolution[mt] = (unsigned int)(ptr_MT[measurement_table].absint);
            			}
            
            			if (file_position == 0)
            			{
            				file_position = ptr_MT[measurement_table].beradr;
            			}
            			
            			mt += 1;
            		}
            	}

				if (mt == 0)
				{
					logmessage(DEBUG,"|       No measurement tables identified       |\n");
				}
				else
				{
					for (measurement_table=0;measurement_table<file->channel_data[channel].abscissa.number_of_measurement_tables;measurement_table++)
					{
						logmessage(DEBUG,"| %2u  | %7u | %7u | %5u |            |\n",measurement_table,file->channel_data[channel].abscissa.start[measurement_table],file->channel_data[channel].abscissa.number_of_samples[measurement_table],file->channel_data[channel].abscissa.resolution[measurement_table]);
					}
				}

				logmessage(DEBUG,"+-----+---------+---------+-------+------------+\n");

            	if ((file_position <= 1024U) || (file_position > (unsigned int)filesize))
				{
					logmessage(NOTICE,"Data start position points outside size of file\n");
				}

				if (file->channel_data[channel].abscissa.type == ABSCISSA_CRANKANGLE)
				{
					if (calculate_theta_lookups(file, channel) == false)
					{
						logmessage(WARNING, "load_ifile_header: Could not calculate theta lookups (%s)\n", file->channel_data[channel].name);

						file->channel_data[channel].abscissa.type = ABSCISSA_UNKNOWN;
					}
				}

				if ((ptr_DI->sigtyp == 33) && (found_ref_time == false))
				{
					found_ref_time = true;

					ref_time_offset = ptr_DI->kalfak[0];
					ref_time_slope = ptr_DI->kalfak[1];
					strncpy(ref_time_units, (char*)ptr_DI->sigein, 10);
					ref_time_units[9] = 0x00;
				}
			}
			start_channel += file->ptr_DGB[dgb].kananz;
		}
		else
		{
			/* Do Nothing */
		}
	}

	/* We can't do this above because channel_name_to_number() needs all channels to have been loaded into file structure first */

	start_channel = 0;
	for (dgb=0;dgb<number_of_data_groups;dgb++)
	{
		if ((file->ptr_DGB[dgb].diradr != NULL) &&
			(file->ptr_DGB[dgb].datfor <= 16) &&
			(valid_dgb(file->ptr_DGB[dgb].datart, grpflg, (char*)(file->ptr_DGB[dgb].absein), file->ptr_DGB[dgb].zyklng) == true))
		{
			for (dgb_channel=0;dgb_channel<file->ptr_DGB[dgb].kananz;dgb_channel++)
			{
				channel = start_channel + dgb_channel;
				ptr_DI = (DATDIR*)&file->ptr_DGB[dgb].diradr[dgb_channel];

				switch (file->channel_data[channel].abscissa.type)
				{
				case ABSCISSA_CRANKANGLE:
				{
					switch (ptr_DI->nultyp)
					{
					case -1:
					case 0:		/* None */
					{
						file->channel_data[channel].offset_config.type = OFFSET_NONE;
						file->channel_data[channel].offset_config.channel_name[0] = 0x00;

						break;
					}
					case 1:		/* Set mean to reference Value */
					{
						file->channel_data[channel].offset_config.type = OFFSET_MEAN;
						file->channel_data[channel].offset_config.fixed_value = (float)ptr_DI->refwrt;

						crank_angle = DegreesToCrankAngle(file, (float)ptr_DI->nulwrt, channel);

						/* Appears to be +/- 4 degrees but this does not align to refzylmit/refsaumit/refausmit */

						int16_t ref_offset = 4;

						file->channel_data[channel].offset_config.start_window = CrankAngleToDegrees(file, crank_angle, channel) - ref_offset;
						file->channel_data[channel].offset_config.finish_window = CrankAngleToDegrees(file, crank_angle, channel) + ref_offset;

						logmessage(DEBUG, "%s OFFSET_FIXED\n", file->channel_data[channel].name);
						logmessage(DEBUG, "nulwrt: %0.3f %s\n", ptr_DI->nulwrt, file->channel_data[channel].abscissa.units);
						logmessage(DEBUG, "refwrt: %0.3f %s\n", ptr_DI->refwrt, file->channel_data[channel].units);
						logmessage(DEBUG, "intern: [%d, %d, %d, %d, %d, %d]\n", ptr_DI->intern[0], ptr_DI->intern[1], ptr_DI->intern[2], ptr_DI->intern[3], ptr_DI->intern[4], ptr_DI->intern[5]);
						logmessage(DEBUG, "mitofs: %d\n", ptr_DI->mitofs);

						break;
					}
					case 2:		/* Thermodynamic */
					{
						file->channel_data[channel].offset_config.type = OFFSET_POLYTROPIC_2PT;
						file->channel_data[channel].offset_config.channel_name[0] = 0x00;

						file->channel_data[channel].offset_config.window_size = file->ptr_DGB[dgb].refsaumit;

						if (file->channel_data[channel].offset_config.window_size > 100)
						{
							file->channel_data[channel].offset_config.window_size = 5;
						}

						break;
					}
					case 3:		/* Fixed Value */
					{
						file->channel_data[channel].offset_config.type = OFFSET_FIXED;
						file->channel_data[channel].offset_config.fixed_value = (float)ptr_DI->refwrt;

						break;
					}
					case 4:		/* Manifold Pressure Signal (Auto) */
					{
						file->channel_data[channel].offset_config.type = OFFSET_WINDOW_ABSOLUTE;

						if (channel_name_to_number(file, (char*)file->ptr_DGB[dgb].refzyl, ABSCISSA_CRANKANGLE, &offset_channel) == false)
						{
							logmessage(DEBUG, "Can't use offset channel '%s' as it doesn't exist\n", file->ptr_DGB[dgb].refzyl);

							file->channel_data[channel].offset_config.type = OFFSET_NONE;
							file->channel_data[channel].offset_config.channel_name[0] = 0x00;
						}
						else if (file->channel_data[offset_channel].abscissa.type != ABSCISSA_CRANKANGLE)
						{
							logmessage(DEBUG, "Can't use channel %s as a offset window channel as it does not have a crank angle abscissa\n", file->channel_data[channel].offset_config.channel_name);

							file->channel_data[channel].offset_config.type = OFFSET_NONE;
							file->channel_data[channel].offset_config.channel_name[0] = 0x00;
						}
						else
						{
							strncpy(file->channel_data[channel].offset_config.channel_name, (char*)file->ptr_DGB[dgb].refzyl, SIZEOF_CHANNEL_NAME);

							crank_angle = DegreesToCrankAngle(file, (float)ptr_DI->nulwrt, channel);

							file->channel_data[channel].offset_config.start_window = CrankAngleToDegrees(file, crank_angle, channel) - file->ptr_DGB[dgb].refzylmit;
 							file->channel_data[channel].offset_config.finish_window = CrankAngleToDegrees(file, crank_angle, channel) + file->ptr_DGB[dgb].refzylmit;
						}

						break;
					}
					case 5:		/* Intake Pressure Ref. Signal */
					case 512:	/* Reference Signal */
					case 513:	/* Reference Signal */
					case 514:	/* Reference Signal */
					case 515:	/* Reference Signal */
					case 516:	/* Reference Signal */
					case 517:	/* Reference Signal */
					case 518:	/* Reference Signal */
					case 519:	/* Reference Signal */
					case 520:	/* Reference Signal */
					case 521:	/* Reference Signal */
					case 522:	/* Reference Signal */
					case 523:	/* Reference Signal */
					case 524:	/* Reference Signal */
					case 525:	/* Reference Signal */
					case 526:	/* Reference Signal */
					case 527:	/* Reference Signal */
					{
						file->channel_data[channel].offset_config.type = OFFSET_WINDOW_ABSOLUTE;

						if (channel_name_to_number(file, (char*)file->ptr_DGB[dgb].refsau, ABSCISSA_CRANKANGLE, &offset_channel) == false)
						{
							logmessage(DEBUG, "Can't use offset channel '%s' as it doesn't exist\n", file->ptr_DGB[dgb].refsau);

							file->channel_data[channel].offset_config.type = OFFSET_NONE;
							file->channel_data[channel].offset_config.channel_name[0] = 0x00;
						}
						else if (file->channel_data[offset_channel].abscissa.type != ABSCISSA_CRANKANGLE)
						{
							logmessage(DEBUG, "Can't use channel %s as a offset window channel as it does not have a crank angle abscissa\n", file->ptr_DGB[dgb].refsau);

							file->channel_data[channel].offset_config.type = OFFSET_NONE;
							file->channel_data[channel].offset_config.channel_name[0] = 0x00;
						}
						else
						{
							strncpy(file->channel_data[channel].offset_config.channel_name, (char*)file->ptr_DGB[dgb].refsau, SIZEOF_CHANNEL_NAME);

							crank_angle = DegreesToCrankAngle(file, (float)ptr_DI->nulwrt, channel);

							file->channel_data[channel].offset_config.start_window = CrankAngleToDegrees(file, crank_angle, channel) - file->ptr_DGB[dgb].refsaumit;
							file->channel_data[channel].offset_config.finish_window = CrankAngleToDegrees(file, crank_angle, channel) + file->ptr_DGB[dgb].refsaumit;
						}

						break;
					}
					case 6:		/* Exhaust Pressure Ref. Signal */
					{
						file->channel_data[channel].offset_config.type = OFFSET_WINDOW_ABSOLUTE;

						if (channel_name_to_number(file, (char*)file->ptr_DGB[dgb].refaus, ABSCISSA_CRANKANGLE, &offset_channel) == false)
						{
							logmessage(DEBUG, "Can't use offset channel '%s' as it doesn't exist\n", file->ptr_DGB[dgb].refaus);

							file->channel_data[channel].offset_config.type = OFFSET_NONE;
							file->channel_data[channel].offset_config.channel_name[0] = 0x00;
						}
						else if (file->channel_data[offset_channel].abscissa.type != ABSCISSA_CRANKANGLE)
						{
							logmessage(DEBUG, "Can't use channel %s as a offset window channel as it does not have a crank angle abscissa\n", file->ptr_DGB[dgb].refaus);

							file->channel_data[channel].offset_config.type = OFFSET_NONE;
							file->channel_data[channel].offset_config.channel_name[0] = 0;
						}
						else
						{
							strncpy(file->channel_data[channel].offset_config.channel_name, (char*)file->ptr_DGB[dgb].refaus, SIZEOF_CHANNEL_NAME);

							crank_angle = DegreesToCrankAngle(file, (float)ptr_DI->nulwrt, channel);

							file->channel_data[channel].offset_config.start_window = CrankAngleToDegrees(file, crank_angle, channel) - file->ptr_DGB[dgb].refausmit;
							file->channel_data[channel].offset_config.finish_window = CrankAngleToDegrees(file, crank_angle, channel) + file->ptr_DGB[dgb].refausmit;
						}

						break;
					}
					case 7:		/* Mean Value Ref. Signal */
					case 256:	/* Manifold Pressure Signal */
					case 257:	/* Manifold Pressure Signal */
					case 258:	/* Manifold Pressure Signal */
					case 259:	/* Manifold Pressure Signal */
					case 260:	/* Manifold Pressure Signal */
					case 261:	/* Manifold Pressure Signal */
					case 262:	/* Manifold Pressure Signal */
					case 263:	/* Manifold Pressure Signal */
					case 264:	/* Manifold Pressure Signal */
					case 265:	/* Manifold Pressure Signal */
					case 266:	/* Manifold Pressure Signal */
					case 267:	/* Manifold Pressure Signal */
					case 268:	/* Manifold Pressure Signal */
					case 269:	/* Manifold Pressure Signal */
					case 270:	/* Manifold Pressure Signal */
					case 271:	/* Manifold Pressure Signal */
					{
						/* TODO: Isn't this just referencing a results channel, not crank angle? */

						file->channel_data[channel].offset_config.type = OFFSET_WINDOW_ABSOLUTE;

						if (channel_name_to_number(file, (char*)file->ptr_DGB[dgb].refzyl, ABSCISSA_CRANKANGLE, &offset_channel) == false)
						{
							logmessage(DEBUG, "Can't use offset channel '%s' as it doesn't exist\n", file->ptr_DGB[dgb].refzyl);

							file->channel_data[channel].offset_config.type = OFFSET_NONE;
							file->channel_data[channel].offset_config.channel_name[0] = 0x00;
						}
						else if (file->channel_data[offset_channel].abscissa.type != ABSCISSA_CRANKANGLE)
						{
							logmessage(DEBUG, "Can't use channel %s as a offset window channel as it does not have a crank angle abscissa\n", file->ptr_DGB[dgb].refzyl);

							file->channel_data[channel].offset_config.type = OFFSET_NONE;
							file->channel_data[channel].offset_config.channel_name[0] = 0x00;
						}
						else
						{
							strncpy(file->channel_data[channel].offset_config.channel_name, (char*)file->ptr_DGB[dgb].refzyl, SIZEOF_CHANNEL_NAME);

							crank_angle = DegreesToCrankAngle(file, (float)ptr_DI->nulwrt, channel);

							file->channel_data[channel].offset_config.start_window = CrankAngleToDegrees(file, crank_angle, channel) - file->ptr_DGB[dgb].refzylmit;
							file->channel_data[channel].offset_config.finish_window = CrankAngleToDegrees(file, crank_angle, channel) + file->ptr_DGB[dgb].refzylmit;
						}

						break;
					}
					default:
					{
						file->channel_data[channel].offset_config.type = OFFSET_NONE;
						file->channel_data[channel].offset_config.channel_name[0] = 0x00;

						logmessage(WARNING, "Channel %u has unknown offset correction type (%d)\n", channel, ptr_DI->nultyp);

						break;
					}
					}

					if (file->channel_data[channel].offset_config.type != OFFSET_NONE)
					{
						logmessage(DEBUG, "+-------------------------------------------+\n");
						logmessage(DEBUG, "| Channel %4u (%10s) Offset          |\n", channel, file->channel_data[channel].name);
						logmessage(DEBUG, "+-------------------------------------------+\n");

						switch (file->channel_data[channel].offset_config.type)
						{
						case OFFSET_MEAN:
						{
							logmessage(DEBUG, "| Type: Mean                                |\n");
							logmessage(DEBUG, "| Value: %8.3f                           |\n", file->channel_data[channel].offset_config.fixed_value);
							logmessage(DEBUG, "| Start angle: %8.3f                     |\n", file->channel_data[channel].offset_config.start_window);
							logmessage(DEBUG, "| Finish angle: %8.3f                    |\n", file->channel_data[channel].offset_config.finish_window);
							break;
						}
						case OFFSET_FIXED:
						{
							logmessage(DEBUG, "| Type: Fixed                               |\n");
							logmessage(DEBUG, "| Value: %8.3f                           |\n", file->channel_data[channel].offset_config.fixed_value);
							break;
						}
						case OFFSET_WINDOW:
						case OFFSET_WINDOW_ABSOLUTE:
						{
							logmessage(DEBUG, "| Type: Reference Channel                   |\n");
							logmessage(DEBUG, "| Channel: %10s                       |\n", file->channel_data[channel].offset_config.channel_name);
							logmessage(DEBUG, "| Start angle: %8.3f                     |\n", file->channel_data[channel].offset_config.start_window);
							logmessage(DEBUG, "| Finish angle: %8.3f                    |\n", file->channel_data[channel].offset_config.finish_window);
							break;
						}
						case OFFSET_POLYTROPIC_2PT:
						case OFFSET_POLYTROPIC_3PT:
						{
							logmessage(DEBUG, "| Type: Polytropic Index                    |\n");
							logmessage(DEBUG, "| Start: %8.3f Finish: %8.3f          |\n", file->channel_data[channel].offset_config.start_window, file->channel_data[channel].offset_config.finish_window);
							logmessage(DEBUG, "| Polytropic index: %5.3f                   |\n", file->channel_data[channel].offset_config.polytropic_index);
							break;
						}
						default:
						{
							logmessage(DEBUG, "| Type: Unknown (%2u)                        |\n", file->channel_data[channel].offset_config.type);
						}
						}

						logmessage(DEBUG, "+-------------------------------------------+\n");
					}

					switch (file->channel_data[channel].type)
					{
					case CHAN_CYL_PR:
					{
						found_soc = false;

						for (chan = 0; chan < file->number_of_channels; chan++)
						{
							if ((file->channel_data[chan].type == CHAN_NEEDLE_LIFT) || (file->channel_data[chan].type == CHAN_IGN_ANG))
							{
								if ((file->channel_data[channel].cylinder == file->channel_data[chan].cylinder) && (found_soc == false))
								{
									file->channel_data[channel].soc_config.type = SOC_CYC_CHANNEL;
									strncpy(file->channel_data[channel].soc_config.channel_name, file->channel_data[chan].name, SIZEOF_CHANNEL_NAME);

									found_soc = true;
								}
							}
						}

						if (found_soc == false)
						{
							for (chan = 0; chan < file->number_of_channels; chan++)
							{
								if ((file->channel_data[chan].type == CHAN_NEEDLE_LIFT) || (file->channel_data[chan].type == CHAN_IGN_ANG))
								{
									if (found_soc == false)
									{
										file->channel_data[channel].soc_config.type = SOC_CYC_CHANNEL;
										strncpy(file->channel_data[channel].soc_config.channel_name, file->channel_data[chan].name, SIZEOF_CHANNEL_NAME);

										found_soc = true;
									}
								}
							}
						}

						if (found_soc == true)
						{
							logmessage(NOTICE, "Set cylinder pressure channel %s start of combustion to use channel %s\n", file->channel_data[channel].name, file->channel_data[channel].soc_config.channel_name);
						}
						else
						{
							logmessage(DEBUG, "Failed to find a SOC channel for channel %s\n", file->channel_data[channel].name);
						}

						break;
					}
					case CHAN_NEEDLE_LIFT:
					{
						file->channel_data[channel].soc_config.type = SOC_CA_CHANNEL_RISE;
						strncpy(file->channel_data[channel].soc_config.channel_name, file->channel_data[channel].name, SIZEOF_CHANNEL_NAME);

						break;
					}
					case CHAN_SPK_PRI_CURR:
					case CHAN_SPK_SEC_CURR:
					case CHAN_INJECTOR:
					case CHAN_INJECTOR_CURR:
					case CHAN_HIGH_T:
					case CHAN_LOW_T:
					case CHAN_IGN_ANG:
					{
						file->channel_data[channel].soc_config.type = SOC_DIGITAL_FALLING_EDGE;
						strncpy(file->channel_data[channel].soc_config.channel_name, file->channel_data[channel].name, SIZEOF_CHANNEL_NAME);

						break;
					}
					default:
					{
						break;
					}
					}

					if (file->channel_data[channel].soc_config.type != SOC_INVALID)
					{
						logmessage(DEBUG, "+-------------------------------------------+\n");
						logmessage(DEBUG, "|       Channel %4u (%s) SOC             |\n", channel, file->channel_data[channel].name);
						logmessage(DEBUG, "+-------------------------------------------+\n");
						switch (file->channel_data[channel].soc_config.type)
						{
						case SOC_INVALID:
						{
							logmessage(DEBUG, "| Type: None                                |\n", file->channel_data[channel].soc_config.type);
							break;
						}
						case SOC_FIXED:
						{
							logmessage(DEBUG, "| Type: Fixed                               |\n", file->channel_data[channel].soc_config.type);
							break;
						}
						case SOC_CA_CHANNEL_RISE:
						{
							logmessage(DEBUG, "| Type: Rise Rate                           |\n", file->channel_data[channel].soc_config.type);
							break;
						}
						case SOC_CA_CHANNEL_FALL:
						{
							logmessage(DEBUG, "| Type: Fall Rate                           |\n", file->channel_data[channel].soc_config.type);
							break;
						}
						case SOC_CA_CHANNEL_AVG:
						{
							logmessage(DEBUG, "| Type: Channel Average                     |\n", file->channel_data[channel].soc_config.type);
							break;
						}
						case SOC_CYC_CHANNEL:
						{
							logmessage(DEBUG, "| Type: Cycle Channel                       |\n", file->channel_data[channel].soc_config.type);
							break;
						}
						case SOC_DIGITAL_RISING_EDGE:
						{
							logmessage(DEBUG, "| Type: Digital Rising Edge                |\n", file->channel_data[channel].soc_config.type);
							break;
						}
						case SOC_DIGITAL_FALLING_EDGE:
						{
							logmessage(DEBUG, "| Type: Digital Falling Edge               |\n", file->channel_data[channel].soc_config.type);
							break;
						}
						default:
						{
							logmessage(DEBUG, "| Type: Unknown                             |\n", file->channel_data[channel].soc_config.type);
							break;
						}
						}
						logmessage(DEBUG, "| Channel: %s                            |\n", file->channel_data[channel].soc_config.channel_name);
						logmessage(DEBUG, "| Start Window: %3.3f                     |\n", file->channel_data[channel].soc_config.start_window);
						logmessage(DEBUG, "| Finish Window: %3.3f                     |\n", file->channel_data[channel].soc_config.finish_window);
						logmessage(DEBUG, "| Fixed Value: %3.3f                      |\n", file->channel_data[channel].soc_config.fixed_value);
						logmessage(DEBUG, "| Aligned: %u                                |\n", file->channel_data[channel].soc_config.aligned);
						logmessage(DEBUG, "| Inverted: %u                               |\n", file->channel_data[channel].soc_config.invert);
						logmessage(DEBUG, "+-------------------------------------------+\n");
					}

					break;
				}
				case ABSCISSA_TIME:
				{
					if (found_ref_time == true)
					{
						file->channel_data[channel].abscissa.slope = ref_time_slope;
						file->channel_data[channel].abscissa.offset = ref_time_offset;
						strncpy(file->channel_data[channel].abscissa.units, ref_time_units, SIZEOF_UNITS);
					}

					break;
				}
				default:
				{
					break;
				}
				}
			}

			start_channel += file->ptr_DGB[dgb].kananz;
		}
	}

	if ((max_cylinder_number > 0) && (max_cylinder_number <= MAX_NUMBER_OF_CYLINDERS))
	{
		file->engine.number_of_cylinders = max_cylinder_number;
	}

	logmessage(NOTICE,"done\n");

	logmessage(DEBUG, "+-------------------------------------+\n");
	logmessage(DEBUG, "|           Wrist Pin Offsets         |\n");
	logmessage(DEBUG, "+-------------------------------------+\n");
	for (channel = 0; channel < file->number_of_channels; channel++)
	{
		if (file->channel_data[channel].abscissa.type == ABSCISSA_CRANKANGLE)
		{
			logmessage(DEBUG, "| Channel %4u: Offset: %3.3f         |\n", channel, file->engine.pin_offset[cylinder - 1]);
		}
	}
	logmessage(DEBUG, "+-------------------------------------+\n");
	for (cylinder = 1; cylinder <= file->engine.number_of_cylinders; cylinder++)
	{
		logmessage(DEBUG, "| Cylinder %3u: Offset: %3.3f         |\n", cylinder, file->engine.pin_offset[cylinder - 1]);
	}
	logmessage(DEBUG, "+-------------------------------------+\n");

	for (channel = 0; channel < file->number_of_channels; channel++)
	{
		if ((file->channel_data[channel].abscissa.type == ABSCISSA_CYCLE) &&
			(file->channel_data[channel].cylinder > 0) &&
			(file->channel_data[channel].tdc_offset < FLT_EPSILON))
		{
			file->channel_data[channel].tdc_offset = file->engine.tdc_offset[file->channel_data[channel].cylinder - 1];

			logmessage(DEBUG, "%s: Changed TDC offset to %f\n", file->channel_data[channel].name, file->channel_data[channel].tdc_offset);
		}
	}

	if ((file->ptr_APB->pExtensions > 0) && (file->ptr_APB->pExtensions < filesize))
	{
		logmessage(NOTICE,"Loading IndiCom 1.3 extensions\n");

		if (fseek(input_file,(long)file->ptr_APB->pExtensions,SEEK_SET) != 0)
		{
			fclose(input_file);

			logmessage(MSG_ERROR, "Could not set file position to start of data\n");

			return(false);
		}
		
		(void)fread(&IFile_Extension->lNumberOfObjects,4,1,input_file);
		(void)fread(&IFile_Extension->lFirstExtensionObjectFilePos,4,1,input_file);
		(void)fread(&IFile_Extension->lLastExtensionObjectFilePos,4,1,input_file);
		
    	if (file->swap_endian != false)
    	{
    		swap_endian_4(&IFile_Extension->lNumberOfObjects);
    		swap_endian_4(&IFile_Extension->lFirstExtensionObjectFilePos);
    		swap_endian_4(&IFile_Extension->lLastExtensionObjectFilePos);
		}
		
		logmessage(DEBUG,"Number of objects: %u\n",IFile_Extension->lNumberOfObjects);
		logmessage(DEBUG,"First address: %u\n",IFile_Extension->lFirstExtensionObjectFilePos);
		logmessage(DEBUG,"Last address: %u\n",IFile_Extension->lLastExtensionObjectFilePos);
		
		unsigned int pass;

		for (pass = 1; pass <= 2; pass++)
		{
			next_object = IFile_Extension->lFirstExtensionObjectFilePos;
			uint32_t max_position = (uint32_t)(filesize - file->ptr_APB->pExtensions);

			if ((IFile_Extension->lFirstExtensionObjectFilePos > max_position) ||
				(IFile_Extension->lNumberOfObjects > 100000))
			{
				logmessage(WARNING, "IFile pExtensions do not look sensible\n%u\n%u\n%u\n%u\n%u\n%u\n",
					IFile_Extension->lFirstExtensionObjectFilePos,
					IFile_Extension->lLastExtensionObjectFilePos,
					IFile_Extension->lNumberOfObjects,
					max_position,
					filesize,
					file->ptr_APB->pExtensions);
			}
			else
			{
				for (object = 0; object < IFile_Extension->lNumberOfObjects; object++)
				{
					if ((file->ptr_APB->pExtensions + next_object) > filesize)
					{
						fclose(input_file);

						logmessage(MSG_ERROR, "IFile pExtensions points outside size of file\n");

						return(false);
					}

					if (fseek(input_file, file->ptr_APB->pExtensions + next_object, SEEK_SET) != 0)
					{
						fclose(input_file);

						logmessage(MSG_ERROR, "Could not set file position to start of data\n");

						return(false);
					}

					(void)fread(&IFile_Object->eType, 4, 1, input_file);
					(void)fread(&IFile_Object->ulKeyLength, 4, 1, input_file);
					(void)fread(&IFile_Object->lKeyFilePos, 4, 1, input_file);
					(void)fread(&IFile_Object->ulObjectSize, 4, 1, input_file);
					(void)fread(&IFile_Object->lObjectFilePos, 4, 1, input_file);
					(void)fread(&IFile_Object->lNextExtensionObjectFilePos, 4, 1, input_file);

					if (file->swap_endian != false)
					{
						swap_endian_4(&IFile_Object->eType);
						swap_endian_4(&IFile_Object->ulKeyLength);
						swap_endian_4(&IFile_Object->lKeyFilePos);
						swap_endian_4(&IFile_Object->ulObjectSize);
						swap_endian_4(&IFile_Object->lObjectFilePos);
						swap_endian_4(&IFile_Object->lNextExtensionObjectFilePos);
					}

					/*logmessage(DEBUG, "eType: %d\n", IFile_Object->eType);
					logmessage(DEBUG, "ulKeyLength: %d\n", IFile_Object->ulKeyLength);
					logmessage(DEBUG, "lKeyFilePos: %d\n", IFile_Object->lKeyFilePos);
					logmessage(DEBUG, "ulObjectSize: %d\n", IFile_Object->ulObjectSize);
					logmessage(DEBUG, "lObjectFilePos: %d\n", IFile_Object->lObjectFilePos);
					logmessage(DEBUG, "lNextExtensionObjectFilePos: %d\n", IFile_Object->lNextExtensionObjectFilePos);*/

					if (process_ifile_extension(input_file, file, IFile_Object, pass) == false)
					{
						fclose(input_file);
						return(false);
					}

					next_object = IFile_Object->lNextExtensionObjectFilePos;
				}
			}
		}

		logmessage(NOTICE,"done\n");
	}

	/* Make some checks now that channel name has been updated with .LongName */

	for (channel = 0; channel < file->number_of_channels; channel++)
	{
		if ((file->channel_data[channel].abscissa.type == ABSCISSA_CRANKANGLE) &&
			(file->channel_data[channel].type == CHAN_CYL_PR) &&
			(
				((case_insensitive_compare(file->channel_data[channel].name, "CDM_0", 5) == 1) && (case_insensitive_compare(file->channel_data[channel].units, "None", SIZEOF_UNITS) == 1)) ||
				((case_insensitive_compare(file->channel_data[channel].name, "I_", 2) == 1) && (case_insensitive_compare(file->channel_data[channel].units, "J", SIZEOF_UNITS) == 1)) ||
				((case_insensitive_compare(file->channel_data[channel].name, "PCYL_DER_", 9) == 1) && (case_insensitive_compare(file->channel_data[channel].units, "bar/deg", SIZEOF_UNITS) == 1)) ||
				((case_insensitive_compare(file->channel_data[channel].name, "PCYL_DE_", 8) == 1) && (case_insensitive_compare(file->channel_data[channel].units, "bar/deg", SIZEOF_UNITS) == 1)) ||
				((case_insensitive_compare(file->channel_data[channel].name, "PCYL_DER_", 9) == 1) && (case_insensitive_compare(file->channel_data[channel].units, "bar_p_Grd", SIZEOF_UNITS) == 1)) ||
				((case_insensitive_compare(file->channel_data[channel].name, "PVn_", 4) == 1) && (case_insensitive_compare(file->channel_data[channel].units, "bar", SIZEOF_UNITS) == 1)) ||
				((case_insensitive_compare(file->channel_data[channel].name, "Q_", 2) == 1) && (case_insensitive_compare(file->channel_data[channel].units, "J_p_Grd", SIZEOF_UNITS) == 1)) ||
				((case_insensitive_compare(file->channel_data[channel].name, "SPEED_CA_0", 10) == 1) && (case_insensitive_compare(file->channel_data[channel].units, "rpm", SIZEOF_UNITS) == 1)) ||
				((case_insensitive_compare(file->channel_data[channel].name, "TRIG_0", 6) == 1) && (case_insensitive_compare(file->channel_data[channel].units, "None", SIZEOF_UNITS) == 1))
				))
		{
			/* Kistler KiBox calculated channels */

			file->channel_data[channel].type = CHAN_RESULTS;

			snprintf(file->channel_data[channel].description, SIZEOF_DESCRIPTION, "%s for Cylinder %u", channel_information[file->channel_data[channel].type].description, file->channel_data[channel].cylinder);
		}
	}

	fclose(input_file);
	
	logmessage(DEBUG, "Generating MATLAB compatible channels names\n");

	generate_matlab_channelnames(file, NULL);

	logmessage(NOTICE,"done\n");

	file->preload = true;

	return(true);
}

bool load_ifile_channel(FileData* file,const unsigned int channel)
{
	unsigned int number_of_data_groups;
	unsigned int dgb;
	unsigned int start_channel;
	unsigned int avl_channel;
	unsigned int measurement_table;
	unsigned int data_pointer;
	unsigned int data_size;
	unsigned int samples_per_cycle;
	size_t samples_read;
	short* short_ptr = NULL;
	FILE* input_file = NULL;
	DATDIR* ptr_DI = NULL;
	MPTAB* ptr_MT = NULL;
	long filesize;
	unsigned int cycle;
	unsigned int file_position;

	if (file == NULL)
	{
		logmessage(MSG_ERROR,"Cannot load I-File into NULL file structure");
		
		return(false);
	}
	
	if (file->channel_data == NULL)
	{
		logmessage(MSG_ERROR,"Cannot load I-File into NULL file->channel_data structure\n");
		
		return(false);
	}
	
	if (channel >= file->number_of_channels)
	{
		logmessage(MSG_ERROR,"Channel number exceeds number of channels\n");
		
		return(false);
	}
	
	if (file->channel_data[channel].loaded == true)
	{
		logmessage(NOTICE, "Channel already loaded\n");

		return(true);
	}
	
	if ((file->preload == false) || (file->file_type != FILE_AVLIFILE))
	{
		logmessage(MSG_ERROR,"AVL I-File has not been pre-loaded.  You cannot load channels before the file header!\n");
		
		return(false);
    }
    
#ifdef _CATOOL_UNICODE_
	input_file = _wfopen(file->filename, L"rb");
#else
	input_file = fopen(file->filename, "rb");
#endif

	if (input_file == NULL)
	{
		logmessage(MSG_ERROR,"AVL I-File (%s) could not be opened\n",file->filename);

		return(false);
	}
	
	filesize = fsize(input_file);

	number_of_data_groups = file->ptr_APB->grpanz;	

	dgb = 0;
	start_channel = 0;
	while ((dgb < number_of_data_groups) && (channel >= start_channel + (unsigned int)file->ptr_DGB[dgb].kananz))
	{
		start_channel += file->ptr_DGB[dgb].kananz;
		dgb += 1;
	}

	if ((dgb == number_of_data_groups) || ((dgb < number_of_data_groups) && (file->ptr_DGB[dgb].diradr == NULL)))
	{
		unsigned int number_of_cycles;
		unsigned int data_samples;
		unsigned int sizeof_data;
		void* values = NULL;
		long data_position;
		unsigned int* data_info = (unsigned int*)file->channel_data[channel].data;
		unsigned int crank_angle;
		unsigned int element_size;

		if (data_info == NULL)
		{
			logmessage(WARNING, "Not loading channel %s due to NULL data\n", file->channel_data[channel].name);

			return(true);
		}

		if (data_info[0] != 0x0CA70070)
		{
			logmessage(WARNING, "Not loading channel %s due to invalid magic number\n", file->channel_data[channel].name);
			
			return(true);
		}

		number_of_cycles = data_info[1];
		data_samples = data_info[2];
		data_position = data_info[3];
		element_size = data_info[4];

		free(file->channel_data[channel].data);
		file->channel_data[channel].data = NULL;

		free(file->channel_data[channel].data_d);
		file->channel_data[channel].data_d = NULL;

		sizeof_data = data_samples * number_of_cycles;

		if (sizeof_data == 0)
		{
			logmessage(WARNING, "Not loading channel %s due to no data samples\n", file->channel_data[channel].name);

			return(true);
		}

		values = malloc(sizeof_data * element_size);

		if (values == NULL)
		{
			fclose(input_file);
			logmessage(FATAL, "Memory could not be allocated\n");
			return(false);
		}

		fseek(input_file, data_position, SEEK_SET);

		(void)fread(values, sizeof_data, element_size, input_file);

		if (file->channel_data[channel].abscissa.type == ABSCISSA_CRANKANGLE)
		{
			file->channel_data[channel].data = (float*)malloc(file->channel_data[channel].abscissa.number_of_samples[0] * file->number_of_cycles * sizeof(float));

			if (file->channel_data[channel].data == NULL)
			{
				fclose(input_file);
				logmessage(FATAL, "Memory could not be allocated\n");
				return(false);
			}

			/* data_samples may be more than samples_per_cycle -> channels seen -360 to +360 degrees inclusive */

			if (element_size == 4)
			{
				for (cycle = 0; cycle < file->number_of_cycles; cycle++)
				{
					for (crank_angle = 0; crank_angle < file->channel_data[channel].samples_per_cycle; crank_angle++)
					{
						data_pointer = cycle * data_samples + crank_angle;

						if (data_pointer < sizeof_data)
						{
							file->channel_data[channel].data[cycle * file->channel_data[channel].samples_per_cycle + crank_angle] = (float)((float*)values)[data_pointer];
						}
						else
						{
							file->channel_data[channel].data[cycle * file->channel_data[channel].samples_per_cycle + crank_angle] = 0.0f;
						}
					}
				}
			}
			else if (element_size == 8)
			{
				/* Ref: INDICOM.i, INJ_DI_SIGD1 */

				file->channel_data[channel].data_d = (double*)malloc(file->channel_data[channel].abscissa.number_of_samples[0] * file->number_of_cycles * sizeof(double));

				if (file->channel_data[channel].data_d == NULL)
				{
					fclose(input_file);
					logmessage(FATAL, "Memory could not be allocated\n");
					return(false);
				}

				for (cycle = 0; cycle < file->number_of_cycles; cycle++)
				{
					for (crank_angle = 0; crank_angle < file->channel_data[channel].samples_per_cycle; crank_angle++)
					{
						data_pointer = cycle * data_samples + crank_angle;

						if (data_pointer < sizeof_data)
						{

							file->channel_data[channel].data[cycle * file->channel_data[channel].samples_per_cycle + crank_angle] = (float)((double*)values)[data_pointer];
							file->channel_data[channel].data_d[cycle * file->channel_data[channel].samples_per_cycle + crank_angle] = ((double*)values)[data_pointer];
						}
						else
						{
							file->channel_data[channel].data[cycle * file->channel_data[channel].samples_per_cycle + crank_angle] = 0.0f;
							file->channel_data[channel].data_d[cycle * file->channel_data[channel].samples_per_cycle + crank_angle] = 0.0;
						}
					}
				}
			}
			else
			{
				logmessage(MSG_ERROR, "Unknown element size (%u)\n", element_size);
			}
		}
		else if (file->channel_data[channel].abscissa.type == ABSCISSA_CYCLE)
		{
			file->channel_data[channel].data = (float*)malloc(file->number_of_cycles * sizeof(float));

			if (file->channel_data[channel].data == NULL)
			{
				fclose(input_file);
				logmessage(FATAL, "Memory could not be allocated\n");
				return(false);
			}

			if (element_size == 4)
			{
				for (data_pointer = 0; data_pointer < file->number_of_cycles; data_pointer++)
				{
					if (data_pointer < file->channel_data[channel].samples_per_cycle)
					{
						file->channel_data[channel].data[data_pointer] = (float)((float*)values)[data_pointer];
					}
					else
					{
						file->channel_data[channel].data[data_pointer] = 0.0f;
					}
				}
			}
			else if (element_size == 8)
			{
				/* Ref: INDICOM.i, StartTime */

				file->channel_data[channel].data_d = (double*)malloc(file->number_of_cycles * sizeof(double));

				if (file->channel_data[channel].data_d == NULL)
				{
					fclose(input_file);
					logmessage(FATAL, "Memory could not be allocated\n");
					return(false);
				}

				for (data_pointer = 0; data_pointer < file->number_of_cycles; data_pointer++)
				{
					if (data_pointer < file->channel_data[channel].samples_per_cycle)
					{
						file->channel_data[channel].data[data_pointer] = (float)((double*)values)[data_pointer];
						file->channel_data[channel].data_d[data_pointer] = ((double*)values)[data_pointer];
					}
					else
					{
						file->channel_data[channel].data[data_pointer] = 0.0f;
						file->channel_data[channel].data_d[data_pointer] = 0.0;
					}
				}
			}
			else
			{
				logmessage(MSG_ERROR, "Unknown element size (%u)\n", element_size);
			}
		}
		else if (file->channel_data[channel].abscissa.type == ABSCISSA_FREQUENCY)
		{
			file->channel_data[channel].data = (float*)malloc(data_samples * file->number_of_cycles * sizeof(float));

			if (file->channel_data[channel].data == NULL)
			{
				fclose(input_file);
				logmessage(FATAL, "Memory could not be allocated\n");
				return(false);
			}

			if (element_size == 4)
			{
				for (data_pointer = 0; data_pointer < data_samples * file->number_of_cycles; data_pointer++)
				{
					if (data_pointer < sizeof_data)
					{
						file->channel_data[channel].data[data_pointer] = (float)((float*)values)[data_pointer];
					}
					else
					{
						file->channel_data[channel].data[data_pointer] = 0.0f;
					}
				}
			}
			else if (element_size == 8)
			{
				file->channel_data[channel].data_d = (double*)malloc(data_samples * file->number_of_cycles * sizeof(double));

				if (file->channel_data[channel].data_d == NULL)
				{
					fclose(input_file);
					logmessage(FATAL, "Memory could not be allocated\n");
					return(false);
				}

				for (data_pointer = 0; data_pointer < data_samples * file->number_of_cycles; data_pointer++)
				{
					if (data_pointer < sizeof_data)
					{

						file->channel_data[channel].data[data_pointer] = (float)((double*)values)[data_pointer];
						file->channel_data[channel].data_d[data_pointer] = ((double*)values)[data_pointer];
					}
					else
					{
						file->channel_data[channel].data[data_pointer] = 0.0f;
						file->channel_data[channel].data_d[data_pointer] = 0.0;
					}
				}
			}
			else
			{
				logmessage(MSG_ERROR, "Unknown element size (%u)\n", element_size);
			}
		}
		else
		{
			logmessage(MSG_ERROR, "Cannot handle loading this type of abscissa\n");
		}

		free(values);

		file->channel_data[channel].samples_per_cycle = file->channel_data[channel].abscissa.number_of_samples[0];

		fclose(input_file);

		file->channel_data[channel].loaded = (file->channel_data[channel].abscissa.type != ABSCISSA_UNKNOWN);

		return(true);
	}
	
	if ((file->channel_data[channel].data != NULL) && (*(unsigned int*)&file->channel_data[channel].data[0] == 0x0CA70070))
	{
		logmessage(WARNING, "%s: Magic number detected but diradr is not NULL\n", file->channel_data[channel].name);
	}

	if (file->ptr_DGB[dgb].datfor > 16)
	{
		logmessage(WARNING, "%s: Not loading channel due to invalid DGB (datfor = %u)\n", file->channel_data[channel].name, file->ptr_DGB[dgb].datfor);

		return(true);
	}

	avl_channel = channel - start_channel;

	logmessage(NOTICE, "%s: Loading channel %u of data group %u\n", file->channel_data[channel].name, avl_channel, dgb);

	ptr_DI = &file->ptr_DGB[dgb].diradr[avl_channel];

	if (file->ptr_DGB[dgb].mpladr == NULL)
	{
		logmessage(MSG_ERROR, "AVL I-File DGB has a null measurement table\n");

		fclose(input_file);

		return(false);
	}

	ptr_MT = &file->ptr_DGB[dgb].mpladr[avl_channel*file->ptr_DGB[dgb].beranz];
          	
	file_position = 0;
	for (measurement_table=0;measurement_table<(unsigned int)file->ptr_DGB[dgb].beranz;measurement_table++)
	{
		if ((ptr_MT[measurement_table].wrtanz > 0) && (ptr_MT[measurement_table].adrint > 0) && (ptr_MT[measurement_table].absint > 0))
		{
			if (file_position == 0)
			{
				file_position = ptr_MT[measurement_table].beradr;
			}
		}
	}
 	
	if (file->channel_data[channel].abscissa.type == ABSCISSA_CRANKANGLE)
	{
    	data_size = file->channel_data[channel].samples_per_cycle * file->number_of_cycles;
    }
    else
    {
        data_size = file->channel_data[channel].samples_per_cycle;
    }

	if ((file_position <= 1024) || 
		(file_position > (unsigned int)filesize) || 
		(file_position + data_size*(unsigned int)file->ptr_DGB[dgb].datfor > (unsigned int)filesize))
	{
		fclose(input_file);

		logmessage(NOTICE, "AVL I-File channel %u (%s) data points to invalid file position\n", avl_channel, file->channel_data[channel].name);
		logmessage(DEBUG, "file_position: %u\ndata_size : %u\ndatfor : %d\nfilesize : %u\n", file_position, data_size, file->ptr_DGB[dgb].datfor, filesize);

		return(false);
	}
				
	logmessage(DEBUG, "Loading channel %u of %u (%s) from data group %u\n", channel, file->number_of_channels, file->channel_data[channel].name, dgb);
	logmessage(DEBUG, "Samples per cycle: %u\n", file->channel_data[channel].samples_per_cycle);
	logmessage(DEBUG, "Number of measurement tables: %u\n", file->channel_data[channel].abscissa.number_of_measurement_tables);
	logmessage(DEBUG, "Data Size: %u items\n", data_size);
	logmessage(DEBUG, "Number of cycles: %u\n", file->number_of_cycles);

	if (file->channel_data[channel].abscissa.type == ABSCISSA_CYCLE)
	{
		/* Some cycle data does not cover the whole number of cycles, make sure enough memory is reserved here though */
		
		if (file->channel_data[channel].samples_per_cycle > file->number_of_cycles)
		{
			file->channel_data[channel].data = (float*)malloc(file->channel_data[channel].samples_per_cycle * sizeof(float));
		}
		else
		{
			file->channel_data[channel].data = (float*)malloc(file->number_of_cycles * sizeof(float));

			memset(file->channel_data[channel].data, 0, file->number_of_cycles * sizeof(float));
		}

		if (file->channel_data[channel].data == NULL)
		{
			fclose(input_file);
			logmessage(FATAL, "Memory could not be allocated\n");
			return(false);
		}
	}
	else
	{
		file->channel_data[channel].data = (float*)malloc(data_size * sizeof(float));

		if (file->channel_data[channel].data == NULL)
		{
			fclose(input_file);
			logmessage(FATAL, "Memory could not be allocated\n");
			return(false);
		}
	}

	file->channel_data[channel].duration = (float*)malloc(file->number_of_cycles * sizeof(float));
	if (file->channel_data[channel].duration == NULL)
	{
		fclose(input_file);
		logmessage(FATAL,"Memory could not be allocated\n");
		return(false);
	}

    if (file->swap_endian == true)
    {
        logmessage(DEBUG,"Loading %i cycles at %u samples per cycle (%u bytes per sample, endian swapping)\n",file->ptr_DGB[dgb].zykanz,file->channel_data[channel].samples_per_cycle,file->ptr_DGB[dgb].datfor);
    }
    else
    {
    	logmessage(DEBUG,"Loading %i cycles at %u samples per cycle (%u bytes per sample, native endian)\n",file->ptr_DGB[dgb].zykanz,file->channel_data[channel].samples_per_cycle,file->ptr_DGB[dgb].datfor);
    }

	if ((file->number_of_cycles == 0) || (file->channel_data[channel].samples_per_cycle == 0))
	{
		logmessage(NOTICE,"No data to load\n");
	}

	bool supported = false;
	bool tested = false;
	bool floating_point = (file->ptr_DGB[dgb].fortyp == 1) || (file->ptr_DGB[dgb].fortyp == 201);
	bool timestamps = (file->ptr_DGB[dgb].fortyp == 200) || (file->ptr_DGB[dgb].fortyp == 201);
	bool interleaved = (file->ptr_DGB[dgb].datfor == 2) && (ptr_DI->zykofs > (int32_t)file->channel_data[channel].samples_per_cycle);
	unsigned int abscissa_type = VARIABLE_NONE;
	unsigned int data_type = VARIABLE_NONE;

	switch (file->ptr_DGB[dgb].datfor)
	{
	default:
	{
		supported = false;
		tested = false;

		break;
	}
	case 2:
	{
		supported = (file->ptr_DGB[dgb].fortyp == 0) && (timestamps == false);
		abscissa_type = VARIABLE_NONE;
		data_type = VARIABLE_INT16;
		tested = (supported == true);

		if (file->ptr_DGB[dgb].fortyp != 0)
		{
			logmessage(WARNING, "%s: Cannot load 16-bit data that is not fixed point (datfor = %u fortyp = %u)\n", file->channel_data[channel].name, file->ptr_DGB[dgb].datfor, file->ptr_DGB[dgb].fortyp);
		}

		break;
	}
	case 4:
	{
		supported = (timestamps == false) && (interleaved == false);
		abscissa_type = VARIABLE_NONE;
		data_type = floating_point ? VARIABLE_FLOAT : VARIABLE_INT32;
		tested = (supported == true);

		break;
	}
	case 8:
	{
		if (timestamps == true)
		{
			supported = (interleaved == false);
			abscissa_type = VARIABLE_INT32;
			data_type = floating_point ? VARIABLE_FLOAT : VARIABLE_INT32;
			tested = false;

			/* Totally untested */

			logmessage(MSG_ERROR, "Untested data format (%s: 8-byte timestamped)\n", file->channel_data[channel].name);
		}
		else
		{
			supported = (interleaved == false);
			abscissa_type = VARIABLE_NONE;
			data_type = floating_point ? VARIABLE_DOUBLE: VARIABLE_INT64;
			tested = (supported == true);
		}

		break;
	}
	case 12:
	{
		supported = (timestamps == true) && (interleaved == false);
		abscissa_type = VARIABLE_INT64;
		data_type = floating_point ? VARIABLE_FLOAT : VARIABLE_INT32;
		tested = false;

		/* Totally untested */

		logmessage(MSG_ERROR, "Untested data format (%s: 12-byte)\n", file->channel_data[channel].name);

		break;
	}
	case 16:
	{
		supported = (timestamps == true) && (interleaved == false);
		abscissa_type = VARIABLE_INT64;
		data_type = floating_point ? VARIABLE_DOUBLE : VARIABLE_INT64;
		tested = (supported == true);

		break;
	}
	}

	if ((file->ptr_DGB[dgb].fortyp != 0) &&
		(file->ptr_DGB[dgb].fortyp != 1) &&
		(file->ptr_DGB[dgb].fortyp != 200) &&
		(file->ptr_DGB[dgb].fortyp != 201))
	{
		logmessage(WARNING, "Unsupported data format (%s: %u)\n",
			file->channel_data[channel].name,
			file->ptr_DGB[dgb].fortyp);
	}

	if ((supported == false) || (tested == false))
	{
		logmessage(WARNING, "Untested or unsupported data format (%s: %u %u %u %u %u %u %u %u %d %d %u %u)\n",
			file->channel_data[channel].name,
			file->ptr_DGB[dgb].datfor,
			abscissa_type, 
			data_type, 
			supported, 
			tested, 
			timestamps, 
			floating_point, 
			interleaved, 
			ptr_DI->zykofs,
			file->ptr_DGB[dgb].fortyp,
			file->channel_data[channel].samples_per_cycle, 
			file->number_of_cycles);
	}

	if (supported == false)
	{
		fclose(input_file);

		logmessage(WARNING, "%s: Unsupported data length (datfor = %u fortyp = %u)\n", file->channel_data[channel].name, file->ptr_DGB[dgb].datfor, file->ptr_DGB[dgb].fortyp);

		return(false);
	}

	char* data = (char*)malloc((size_t)data_size * file->ptr_DGB[dgb].datfor);

	if (data == NULL)
	{
		fclose(input_file);

		logmessage(FATAL, "Memory could not be allocated\n");

		return(false);
	}

	if (interleaved == true)
	{
		logmessage(DEBUG, "Channel data is interleaved (%u/%u)\n", ptr_DI->zykofs, file->channel_data[channel].samples_per_cycle);

		if (file->channel_data[channel].abscissa.type == ABSCISSA_CRANKANGLE)
		{
			samples_per_cycle = file->channel_data[channel].samples_per_cycle;
		}
		else
		{
			samples_per_cycle = 1;
		}

		for (cycle = 0; cycle < file->number_of_cycles; cycle++)
		{
			if (fseek(input_file, file_position + cycle * ptr_DI->zykofs * file->ptr_DGB[dgb].datfor, SEEK_SET) != 0)
			{
				fclose(input_file);

				logmessage(MSG_ERROR, "Could not set file position to start of data\n");

				return(false);
			}

			samples_read = fread((void*)((char*)file->channel_data[channel].data + cycle * samples_per_cycle * file->ptr_DGB[dgb].datfor), file->ptr_DGB[dgb].datfor, samples_per_cycle, input_file);

			if (samples_read != samples_per_cycle)
			{
				fclose(input_file);

				logmessage(MSG_ERROR, "%s: Could not load data (samples_read = %u samples_per_cycle = %u)\n", file->channel_data[channel].name, samples_read, samples_per_cycle);

				return(false);
			}
		}
	}
	else
	{
		/* Channel data is stored sequentially */

		logmessage(DEBUG, "Channel data is stored sequentially\n");

		if (fseek(input_file, file_position, SEEK_SET) != 0)
		{
			fclose(input_file);

			logmessage(MSG_ERROR, "Could not set file position to start of data\n");

			return(false);
		}

		samples_read = fread(data, file->ptr_DGB[dgb].datfor, data_size, input_file);

		if (samples_read != data_size)
		{
			fclose(input_file);

			logmessage(MSG_ERROR, "%s: Could not load data (samples_read = %u data_size = %u)\n", file->channel_data[channel].name, samples_read, data_size);

			free(data);
			return(false);
		}
	}

	if (file->channel_data[channel].abscissa.axis != NULL)
	{
		free(file->channel_data[channel].abscissa.axis);
		file->channel_data[channel].abscissa.axis = NULL;
	}

	size_t offset = 0;

	if (timestamps == true)
	{
		size_t factor = file->ptr_DGB[dgb].datfor;
		double scalar = file->channel_data[channel].slope * file->time_resolution;

		/* Extract abscissa data */

		/* TODO: Swap Endian */

		if (abscissa_type == VARIABLE_INT32)
		{
			logmessage(DEBUG, "Extracting 32-bit fixed point abscissa\n");

			file->channel_data[channel].abscissa.axis = malloc(data_size * sizeof(int32_t));

			if (file->channel_data[channel].abscissa.axis == NULL)
			{
				fclose(input_file);

				logmessage(FATAL, "Memory could not be allocated\n");

				return(false);
			}

			file->channel_data[channel].abscissa.axis_type = VARIABLE_FLOAT;

			offset = sizeof(int32_t);

			for (data_pointer = 0; data_pointer < data_size; data_pointer++)
			{
				int32_t value = *((int32_t*)&data[data_pointer * factor + 0]);

				((float*)file->channel_data[channel].abscissa.axis)[data_pointer] = (float)((double)value * scalar + file->channel_data[channel].abscissa.offset);
			}
		}
		else if (abscissa_type == VARIABLE_INT64)
		{
			logmessage(DEBUG, "Extracting 64-bit fixed point abscissa\n");

			file->channel_data[channel].abscissa.axis = malloc(data_size * sizeof(int64_t));

			if (file->channel_data[channel].abscissa.axis == NULL)
			{
				fclose(input_file);

				logmessage(FATAL, "Memory could not be allocated\n");

				return(false);
			}

			file->channel_data[channel].abscissa.axis_type = VARIABLE_DOUBLE;

			offset = sizeof(int64_t);

			for (data_pointer = 0; data_pointer < data_size; data_pointer++)
			{
				int64_t value = *((int64_t*)&data[data_pointer * factor + 0]);

				((double*)file->channel_data[channel].abscissa.axis)[data_pointer] = (double)value * scalar + file->channel_data[channel].abscissa.offset;
			}
		}
		else
		{
			logmessage(MSG_ERROR, "%s: Unsupported abscissa type (abscissa_type = %u)\n", abscissa_type);
		}
	}

	if (file->channel_data[channel].data_d != NULL)
	{
		free(file->channel_data[channel].data_d);
		file->channel_data[channel].data_d = NULL;
	}

	if ((data_type == VARIABLE_DOUBLE) ||
		(data_type == VARIABLE_INT64))
	{
		file->channel_data[channel].data_d = (double*)malloc(data_size * sizeof(double));

		if (file->channel_data[channel].data_d == NULL)
		{
			fclose(input_file);

			logmessage(FATAL, "Memory could not be allocated\n");

			return(false);
		}
	}

	/* Store data */

	switch (data_type)
	{
	case VARIABLE_INT16:
	{
		/* Swap endian */

		if (file->swap_endian == true)
		{
			for (data_pointer = 0; data_pointer < data_size; data_pointer++)
			{
				short_ptr = (short*)&data[(size_t)data_pointer * sizeof(short)];
				swap_endian_2(short_ptr);
			}
		}

		logmessage(DEBUG, "Converting 16-bit fixed point data to 32-bit floating point (Slope: %f Offset: %f)\n", file->channel_data[channel].slope, file->channel_data[channel].offset);

		size_t data_factor = file->ptr_DGB[dgb].datfor;
		for (data_pointer = 0; data_pointer < data_size; data_pointer++)
		{
			short int value = *((short int*)&data[data_pointer * data_factor + offset]);

			file->channel_data[channel].data[data_pointer] = (float)((double)value * file->channel_data[channel].slope + file->channel_data[channel].offset);
		}

		break;
	}
	case VARIABLE_INT32:
	{
		/* Swap endian */

		if (file->swap_endian == true)
		{
			for (data_pointer = 0; data_pointer < data_size; data_pointer++)
			{
				long int* ptr = (long int*)&data[(size_t)data_pointer * sizeof(long int)];
				swap_endian_4(ptr);
			}
		}

		logmessage(DEBUG, "Converting 32-bit fixed point data to 32-bit floating point\n");

		size_t data_factor = file->ptr_DGB[dgb].datfor;
		for (data_pointer = 0; data_pointer < data_size; data_pointer++)
		{
			long int value = *((long int*) & data[data_pointer * data_factor + offset]);

			file->channel_data[channel].data[data_pointer] = (float)((double)value*file->channel_data[channel].slope + file->channel_data[channel].offset);
		}

		break;
	}
	case VARIABLE_INT64:
	{
		/* Swap endian */

		if (file->swap_endian == true)
		{
			for (data_pointer = 0; data_pointer < data_size; data_pointer++)
			{
				long long int* ptr = (long long int*)&data[(size_t)data_pointer * sizeof(long long int)];
				swap_endian_8(ptr);
			}
		}

		logmessage(DEBUG, "Converting 64-bit fixed point data to 32-bit/64-bit floating point\n");

		size_t data_factor = file->ptr_DGB[dgb].datfor;
		for (data_pointer = 0; data_pointer < data_size; data_pointer++)
		{
			long long int value = *((long long int*)(&data[data_pointer * data_factor + offset]));

			file->channel_data[channel].data_d[data_pointer] = (double)value * file->channel_data[channel].slope + file->channel_data[channel].offset;
			file->channel_data[channel].data[data_pointer] = (float)file->channel_data[channel].data_d[data_pointer];
		}

		break;
	}
	case VARIABLE_FLOAT:
	{
		/* Swap endian */

		if (file->swap_endian == true)
		{
			for (data_pointer = 0; data_pointer < data_size; data_pointer++)
			{
				long int* ptr = (long int*)&data[(size_t)data_pointer * sizeof(float)];
				swap_endian_4(ptr);
			}
		}

		logmessage(DEBUG, "Applying slope and offset to 32-bit floating point data (Slope: %f Offset: %f)\n", file->channel_data[channel].slope, file->channel_data[channel].offset);
		
		size_t data_factor = file->ptr_DGB[dgb].datfor;
		for (data_pointer = 0; data_pointer < data_size; data_pointer++)
		{
			float value = *((float*)&data[data_pointer * data_factor + offset]);

			file->channel_data[channel].data[data_pointer] = (float)((double)value * file->channel_data[channel].slope + file->channel_data[channel].offset);
		}

		break;
	}
	case VARIABLE_DOUBLE:
	{
		/* Swap endian */

		if (file->swap_endian == true)
		{
			for (data_pointer = 0; data_pointer < data_size; data_pointer++)
			{
				long long int* ptr = (long long int*)&data[(size_t)data_pointer * sizeof(double)];
				swap_endian_8(ptr);
			}
		}

		logmessage(DEBUG, "Applying slope and offset to 64-bit floating point data (Slope: %f Offset: %f)\n", file->channel_data[channel].slope, file->channel_data[channel].offset);

		size_t data_factor = file->ptr_DGB[dgb].datfor;
		for (data_pointer = 0; data_pointer < data_size; data_pointer++)
		{
			double value = *((double*)&data[data_pointer * data_factor + offset]);

			file->channel_data[channel].data_d[data_pointer] = value * file->channel_data[channel].slope + file->channel_data[channel].offset;
			file->channel_data[channel].data[data_pointer] = (float)file->channel_data[channel].data_d[data_pointer];
		}

		break;
	}
	default:
	{
		logmessage(WARNING, "%s: Unsupported data type (datfor = %u fortyp = %u)\n", file->channel_data[channel].name, file->ptr_DGB[dgb].datfor, file->ptr_DGB[dgb].fortyp);

		break;
	}
	}

	free(data);

	fclose(input_file);

	logmessage(DEBUG,"Data loaded\n");

	file->channel_data[channel].loaded = (file->channel_data[channel].abscissa.type != ABSCISSA_UNKNOWN);
	
	logmessage(NOTICE,"done\n");
	
	return(true);
}

static void *float_to_short_thread(void* arguments)
{
	float_to_short_arguments* args = (float_to_short_arguments*)arguments;
	float* data = args->data;
	short* buffer = args->buffer;
	unsigned int number_of_datapoints = args->number_of_datapoints;
	double kalfak_0 = args->kalfak_0;
	double kalfak_1 = args->kalfak_1;
	unsigned int threads = args->threads;
	unsigned int data_pointer;

	for (data_pointer=0;data_pointer<number_of_datapoints;data_pointer++)
	{
		buffer[data_pointer] = (short)(( data[data_pointer] - kalfak_0 ) / kalfak_1);
	}

#if defined(_USE_PTHREADS)
	if (threads > 1)
	{
		pthread_exit((void*)0);
	}
#endif
	return((void*)0);
}


static void float_to_short(const FileData* file, float* data, short* buffer, unsigned int number_of_datapoints, double kalfak_0, double kalfak_1)
{
	unsigned int data_pointer;
	int datapoints_per_thread;
	int thread = 0;
	int t;
#if defined(_USE_PTHREADS)
	pthread_t* threads = NULL;
	pthread_attr_t attr;
	int rc;
#elif defined(_USE_STD_THREAD)
	std::thread* threads = NULL;
#else
	void* threads = NULL;
	file->threads = 0;
#endif

	float_to_short_arguments* float_to_short_configuration = NULL;

	if (file->threads > 1)
	{
#if defined(_USE_PTHREADS)
		threads = (pthread_t*)malloc(file->threads * sizeof(pthread_t));
#elif defined(_USE_STD_THREAD)
		threads = new std::thread[file->threads];
#endif
		if (threads == NULL)
		{
			logmessage(FATAL,"Memory could not be allocated\n");
			return;
		}

		float_to_short_configuration = (float_to_short_arguments*)malloc(file->threads * sizeof(float_to_short_arguments));
		if (float_to_short_configuration == NULL)
		{
#if defined(_USE_PTHREADS)
			free(threads);
#elif defined(_USE_STD_THREAD)
			delete[] threads;
#endif
			logmessage(FATAL,"Memory could not be allocated\n");
			return;
		}

#if defined(_USE_PTHREADS)
		pthread_attr_init(&attr);
		pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_JOINABLE);
		pthread_attr_setstacksize(&attr, THREAD_STACKSIZE);
#endif

        datapoints_per_thread = number_of_datapoints / file->threads;
		thread = 0;

		while (thread < (int)file->threads)
		{
			float_to_short_configuration[thread].data = data + thread*datapoints_per_thread;
			float_to_short_configuration[thread].buffer = buffer + thread*datapoints_per_thread;

			if (thread < (int)file->threads-1)
			{
				float_to_short_configuration[thread].number_of_datapoints = datapoints_per_thread;
			}			
			else
			{
				float_to_short_configuration[thread].number_of_datapoints = number_of_datapoints - (thread*datapoints_per_thread);
			}

			float_to_short_configuration[thread].kalfak_0 = kalfak_0;
			float_to_short_configuration[thread].kalfak_1 = kalfak_1;
			float_to_short_configuration[thread].threads = file->threads;

#if defined(_USE_PTHREADS)
			rc = pthread_create(&threads[thread],&attr,float_to_short_thread,(void*)&float_to_short_configuration[thread]);
			if (rc != 0)
			{
				logmessage(FATAL,"Thread could not be created\n");
			}
#elif defined(_USE_STD_THREAD)
			threads[thread] = std::thread(float_to_short_thread, (void*)&float_to_short_configuration[thread]);
#endif
			thread += 1;
		}

#if defined(_USE_PTHREADS)
		pthread_attr_destroy(&attr);

		for(t=0;t<thread;t++)
		{
			rc = pthread_join(threads[t],NULL);
			if (rc != 0)
			{
				free(float_to_short_configuration);
				free(threads);
				logmessage(FATAL,"Thread could not be rejoined\n");
				return;
			}
		}

		free(threads);
#else
		if (file->threads > 1)
		{
			for (t = 0; t < thread; t++)
			{
				threads[t].join();
			}
		}

		delete[] threads;
#endif
		threads = NULL;

		free(float_to_short_configuration);
		float_to_short_configuration = NULL;
	}
	else
	{
		for (data_pointer=0;data_pointer<number_of_datapoints;data_pointer++)
		{
			buffer[data_pointer] = (short)((data[data_pointer] - kalfak_0) / kalfak_1);
		}
	}
}

void calculate_pin_offsets(const FileData* file,float* pin_offset,float* pin_offset_2,int* desaxi)
{
	unsigned int channel;
	unsigned int cylinder;
	bool found_1;
	bool found_2;

	*pin_offset = 0.0f;
	*pin_offset_2 = 0.0f;

	cylinder = 0;
	found_1 = false;
	found_2 = false;
	while (cylinder < file->engine.number_of_cylinders)
	{
		if (fabs(file->engine.pin_offset[cylinder]) > 0.0f)
		{
			if (found_1 == false)
			{
				*pin_offset = file->engine.pin_offset[cylinder];
				found_1 = true;
			}
			else
			{
				if ((fabs(file->engine.pin_offset[cylinder] - *pin_offset) > 0.0f) && 
					(fabs(file->engine.pin_offset[cylinder] + *pin_offset) > 0.0f))
				{
					if (found_2 == false)
					{
						*pin_offset_2 = file->engine.pin_offset[cylinder];
						found_2 = true;
					}
					else if ((fabs(file->engine.pin_offset[cylinder] - *pin_offset_2) > 0.0f) && 
					         (fabs(file->engine.pin_offset[cylinder] + *pin_offset_2) > 0.0f))
					{
						logmessage(MSG_ERROR,"Found more than two different absolute wrist pin offsets");
					}
					else
					{
						/* Do Nothing - pin offset = pin_offset_2 or -pin_offset_2 */
					}
				}
			}
		}

		cylinder += 1;
	}

	for (channel=0;channel<file->number_of_channels;channel++)
	{
		cylinder = file->channel_data[channel].cylinder;

		if ((cylinder > 0) && (cylinder <= MAX_NUMBER_OF_CYLINDERS))
		{
			if (fabs(file->engine.pin_offset[cylinder-1] - *pin_offset) < 0.0001)
			{
				desaxi[channel] = 1;
			}
			else if (fabs(file->engine.pin_offset[cylinder-1] - (-1.0f*(*pin_offset))) < 0.0001)
			{
				desaxi[channel] = -1;
			}
			else if (fabs(file->engine.pin_offset[cylinder-1] - *pin_offset_2) < 0.0001)
			{
				desaxi[channel] = 2;
			}
			else if (fabs(file->engine.pin_offset[cylinder-1] - (-1.0f*(*pin_offset_2))) < 0.0001)
			{
				desaxi[channel] = -2;
			}
			else
			{
				desaxi[channel] = 0;
			}
		}
		else
		{
			desaxi[channel] = 0;
		}
	}
}

#ifdef _CATOOL_UNICODE_
bool save_ifile(FileData* file, const wchar_t* filename, Analysis* analysis, const unsigned int grpflg, const unsigned int grpflg_float)
#else
bool save_ifile(FileData* file, const char* filename, Analysis* analysis, const unsigned int grpflg, const unsigned int grpflg_float)
#endif
{
	FILE* output_file = NULL;
	float total_duration;
	float actual_value;
	unsigned int parameter_number;
	unsigned int analysis_channel;
	unsigned int cycle;
	unsigned int analysis_number;
	unsigned int channel = 0;
	unsigned int calc_file_size = 0;
	unsigned int file_position;
	unsigned int measurement_table;
	unsigned int channel_size;
	unsigned int timing_channel;
	unsigned int ca_data_start = 0;
	unsigned int ca_rtp_start = 0;
	unsigned int time_start = 0;
	unsigned int results_start = 0;
	char mesdat[24];
	short grpanz = 0;
	short value_to_write;
	double dltphi;
	double dltzei;
	double kalfak_0;
	double kalfak_1;
	double min;
	double max;
	double value;
	int start;
	int timebase_resolution;
	unsigned int ca_ptr_DI = 0;
	unsigned int res_ptr_DI = 0;
	unsigned int time_ptr_DI;
	unsigned int ca_mpladr = 0;
	unsigned int res_mpladr = 0;
	unsigned int time_mpladr;
	unsigned int ca_aztadr = 0;
	unsigned int ca_aztlen = 0;
	unsigned int ca_rztadr = 0;
	unsigned int ca_rztlen = 0;
	unsigned int number_ca_channels = 0;
	unsigned int number_ca_analysis_channels = 0;
	unsigned int number_cyc_analysis_channels = 0;
	unsigned int number_time_based_channels = 0;
	unsigned int number_time_based_results_channels = 0;
	unsigned int number_raw_analysis_channels = 0;
	unsigned int sizeof_header = 0;
	unsigned int sizeof_ca_data = 0;
	unsigned int sizeof_time_data = 0;
	unsigned int sizeof_rtp_results_data = 0;
	unsigned int sizeof_cyc_results_data = 0;
	unsigned int sizeof_utc_data = 0;
	unsigned int sizeof_apb = 0;
	unsigned int sizeof_dgb = 0;
	unsigned int sizeof_ca_headers = 0;
	unsigned int sizeof_rtp_results_headers = 0;
	unsigned int sizeof_cyc_results_headers = 0;
	unsigned int sizeof_time_headers = 0;
	unsigned int sizeof_utc_headers = 0;
	unsigned int number_of_datapoints = 0;
	unsigned int number_of_time_datapoints = 0;
	unsigned int ifile_grpflg = 0;
	bool output_ca = false;
	bool output_ca_analysis = false;
	bool output_time = false;
	bool output_time_results = false;
	bool output_cyc_results = false;
	bool output_raw_results = false;
	short* buffer = NULL;
	long pos = 0;
	unsigned int sizeof_ca_variable;
	unsigned int sizeof_time_variable;
	unsigned int sizeof_cyc_variable;
	unsigned int number_of_objects;
	int32_t lFirstExtensionObjectFilePos;
	int32_t lLastExtensionObjectFilePos;
	int32_t lNextExtensionObjectFilePos;
	char key[256];
	int32_t eType;
	uint32_t ulKeyLength;
	int32_t lKeyFilePos;
	uint32_t ulObjectSize;
	int32_t lObjectFilePos;
	float start_window = -100.0f;
	float finish_window = -65.0f;
	float polytropic_index = 1.32f;
	unsigned int offset_channel;
	float pin_offset;
	float pin_offset_2;
	int* desaxi = NULL;
	char ref_channel_cylpr[10] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
	char ref_channel_intake_manpr[10] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
	char ref_channel_exhaust_manpr[10] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
	short refzylmit = 0;
	short refsaumit = 0;
	short refausmit = 0;
	char unit[10];

	if (file == NULL)
	{
		logmessage(MSG_ERROR,"File structure is NULL\n");

		return(false);
	}

	dltphi = 360.0/(double)PulsesPerRev(file);
	dltzei = 0.01;

	generate_short_ifile_channelnames(file,analysis);

	if ((file->number_of_channels > 0) && (file->channel_data == NULL))
	{
		logmessage(MSG_ERROR,"Channels are configured but no data is present\n");

		return(false);
	}

	/* Calculate Min/Max of channels */
	
	Calculate_Min_Max(file, analysis);

#ifdef _CATOOL_UNICODE_
	output_file = _wfopen(filename, L"wb");
#else
	output_file = fopen(filename,"wb");
#endif
	
	if (output_file == NULL)
	{
		logmessage(MSG_ERROR,"I-File output file could not be opened\n");

		return(false);
	}

	if ((grpflg_float & IFILE_TIME) > 0)
	{
		sizeof_time_variable = 4;
	}
	else
	{
		sizeof_time_variable = 2;
	}

	if (((grpflg_float & IFILE_CA) > 0) || ((grpflg_float & IFILE_CA_RAW) > 0))
	{
		sizeof_ca_variable = 4;
	}
	else
	{
		sizeof_ca_variable = 2;
	}

	if (((grpflg_float & IFILE_RESULTS) > 0) || ((grpflg_float & IFILE_RESULTS_RAW) > 0))
	{
		sizeof_cyc_variable = 4;
	}
	else
	{
		sizeof_cyc_variable = 2;
	}

	if (((grpflg & IFILE_CA) > 0) || ((grpflg & IFILE_CA_RAW) > 0))
	{
		for (channel = 0; channel<file->number_of_channels; channel++)
		{
			if ((file->channel_data[channel].loaded != false) && (file->channel_data[channel].abscissa.type == ABSCISSA_CRANKANGLE))
			{
				if (file->channel_data[channel].offset_config.type == OFFSET_POLYTROPIC_2PT)
				{
					polytropic_index = file->channel_data[channel].offset_config.polytropic_index;

					refsaumit = file->channel_data[channel].offset_config.window_size;
				}

				if (file->channel_data[channel].offset_config.type != OFFSET_NONE)
				{
					start_window = file->channel_data[channel].offset_config.start_window;
					finish_window = file->channel_data[channel].offset_config.finish_window;
				}

				if ((file->channel_data[channel].offset_config.type == OFFSET_WINDOW) || 
					(file->channel_data[channel].offset_config.type == OFFSET_WINDOW_ABSOLUTE))
				{
					if (channel_name_to_number(file, file->channel_data[channel].offset_config.channel_name, ABSCISSA_UNKNOWN, &offset_channel) == true)
					{
						if ((offset_channel < file->number_of_channels) && (file->channel_data[offset_channel].loaded == true))
						{
							if (file->channel_data[offset_channel].offset_config.type == CHAN_INLET_PR)
							{
								strncpy(ref_channel_intake_manpr, file->channel_data[offset_channel].short_name, 10);
								ref_channel_intake_manpr[9] = 0x00;

								refsaumit = (short)((file->channel_data[channel].offset_config.finish_window - file->channel_data[channel].offset_config.start_window) / 2.0f);
							}
							else if (file->channel_data[offset_channel].offset_config.type == CHAN_EXH_PR)
							{
								strncpy(ref_channel_exhaust_manpr, file->channel_data[offset_channel].short_name, 10);
								ref_channel_exhaust_manpr[9] = 0x00;
								refausmit = (short)((file->channel_data[channel].offset_config.finish_window - file->channel_data[channel].offset_config.start_window) / 2.0f);
							}
							else
							{
								strncpy(ref_channel_cylpr, file->channel_data[offset_channel].short_name, 10);
								ref_channel_cylpr[9] = 0x00;
								refzylmit = (short)((file->channel_data[channel].offset_config.finish_window - file->channel_data[channel].offset_config.start_window) / 2.0f);
							}
						}
					}
				}
			}
		}

		if (((grpflg & IFILE_CA_RAW) != false) && (file->number_of_channels > 0))
		{		
			number_ca_channels = 0;
			for (channel=0;channel<file->number_of_channels;channel++)
			{
				if (file->channel_data[channel].abscissa.type == ABSCISSA_CRANKANGLE)
				{
					if (file->channel_data[channel].loaded != false)
					{
						logmessage(DEBUG, "IFILE_CA_RAW: %s\n", file->channel_data[channel].name);

						number_ca_channels += 1;
					}
					else
					{
						logmessage(DEBUG, "IFILE_CA_RAW: %s (Not loaded)\n", file->channel_data[channel].name);
					}
				}
			}
			
			if (number_ca_channels > 0)
			{
				logmessage(NOTICE,"Outputting raw crank angle data\n");
					
				output_ca = true;
			}
		}
					
		if (((grpflg & IFILE_CA) > 0) && (analysis != NULL) && (analysis->number_of_channels > 0))
		{
			number_ca_analysis_channels = 0;
			for (analysis_number=0;analysis_number<analysis->number_of_channels;analysis_number++)
			{
				if (file->channel_data[analysis->channel[analysis_number].channel].loaded == true)
				{
					for (analysis_channel = 0; analysis_channel < get_number_of_analysis_channels(); analysis_channel++)
					{
						if ((analysis->channel[analysis_number].analysis_rqst[analysis_channel] >= AnalysisRqst::CalculateAndStore) &&
							(analysis->channel[analysis_number].results[analysis_channel].abscissa.type == ABSCISSA_CRANKANGLE))
						{
							logmessage(DEBUG, "IFILE_CA: %s.%s\n", file->channel_data[analysis->channel[analysis_number].channel].name, analysis->channel[analysis_number].results[analysis_channel].name);

							number_ca_analysis_channels += 1;
						}
					}
				}
			}
			
			if (number_ca_analysis_channels > 0)
			{
				logmessage(NOTICE,"Outputting analysis crank angle data\n");
				
				output_ca_analysis = true;
			}
		}
		
		if ((output_ca != false) || (output_ca_analysis != false))
		{
			grpanz += 1;
		}
		else
		{
			logmessage(NOTICE,"There is no crank angle data to output\n");
		}
	}

	if (((grpflg & IFILE_TIME) > 0) && (file->number_of_channels > 0))
	{
		number_time_based_channels = 0;
		
		for (channel=0;channel<file->number_of_channels;channel++)
		{
			if ((file->channel_data[channel].loaded != false) && (file->channel_data[channel].abscissa.type == ABSCISSA_TIME))
			{
				logmessage(DEBUG, "IFILE_TIME: %s\n", file->channel_data[channel].name);

				number_time_based_channels++;
			}
		}
			
		if (number_time_based_channels > 0)
		{
			output_time = true;

			grpanz += 1;
			
			logmessage(NOTICE,"Outputting time based data\n");
		}
		else
		{
			logmessage(NOTICE,"There is no time based data to output\n");
		}
	}

	if ((grpflg & IFILE_RTP_RESULTS) > 0)
	{
		logmessage(NOTICE,"This version outputs RTP Results in the Results group\n");
	}

	if (((grpflg & IFILE_RESULTS_RAW) > 0) || ((grpflg & IFILE_RESULTS) > 0))
	{
		number_raw_analysis_channels = 0;

		if ((grpflg & IFILE_RESULTS_RAW) != false)
		{
			for (channel=0;channel<file->number_of_channels;channel++)
			{
				if ((file->channel_data[channel].loaded != false) && (file->channel_data[channel].abscissa.type == ABSCISSA_CYCLE))
				{
					logmessage(DEBUG, "IFILE_RESULTS_RAW: %s\n", file->channel_data[channel].name);

					number_raw_analysis_channels += 1;
				}
			}
		}
			
		number_time_based_results_channels = 0;
		number_cyc_analysis_channels = 0;

		if (((grpflg & IFILE_RESULTS) > 0) && (analysis != NULL) && (analysis->number_of_channels > 0))
		{
			for (analysis_number=0;analysis_number<analysis->number_of_channels;analysis_number++)
			{
				if (file->channel_data[analysis->channel[analysis_number].channel].loaded == true)
				{
					for (analysis_channel = 0; analysis_channel < get_number_of_analysis_channels(); analysis_channel++)
					{
						if (analysis->channel[analysis_number].analysis_rqst[analysis_channel] >= AnalysisRqst::CalculateAndStore)
						{
							if (analysis->channel[analysis_number].results[analysis_channel].abscissa.type == ABSCISSA_TIME)
							{
								logmessage(DEBUG, "IFILE_RESULTS (TIME): %s.%s\n", file->channel_data[analysis->channel[analysis_number].channel].name, analysis->channel[analysis_number].results[analysis_channel].name);

								number_time_based_results_channels += 1;
							}
							else if (analysis->channel[analysis_number].results[analysis_channel].abscissa.type == ABSCISSA_CYCLE)
							{
								logmessage(DEBUG, "IFILE_RESULTS (CYC): %s.%s\n", file->channel_data[analysis->channel[analysis_number].channel].name, analysis->channel[analysis_number].results[analysis_channel].name);

								number_cyc_analysis_channels += 1;
							}
							else
							{
								/* Do Nothing */
							}
						}
					}
				}
			}
		}

		if (number_time_based_results_channels > 0)
		{
			output_time_results = true;
			
			logmessage(NOTICE,"Outputting time based results data\n");
		}
		
		if (number_cyc_analysis_channels > 0)
		{
			output_cyc_results = true;

			logmessage(NOTICE,"Outputting results data\n");
		}
		
		if (number_raw_analysis_channels > 0)
		{
			output_raw_results = true;
		
			logmessage(NOTICE,"Outputting raw results data\n");
		}				
		
		if ((output_cyc_results == true) || (output_raw_results == true))
		{										
			grpanz += 1;
		}
			
		if ((output_time_results == true) && (output_time == false))
		{
			grpanz += 1;
		}
	}

	if ((grpflg & IFILE_ASYNC_UTC) != false)
	{
		logmessage(WARNING,"This version cannot output asynchronous UTC data\n");
	}

	if (grpanz == 0)
	{
		logmessage(MSG_ERROR,"There are no data groups to output to the I-File\n");
		
		fclose(output_file);
		
		return(false);
	}

	timing_channel = 0;
	while ((timing_channel < file->number_of_channels) && (file->channel_data[timing_channel].loaded == false))
	{
		timing_channel++;
	}

	if (file->channel_data[timing_channel].loaded == false)
	{
		logmessage(MSG_ERROR,"Cannot find any loaded channels to output to I-File\n");
		
		fclose(output_file);

		return(false);
	}

	logmessage(NOTICE,"Calculating file positions...");

	sizeof_apb = 1024;
	sizeof_dgb = 256 * grpanz;

	if ((output_ca != false) || (output_ca_analysis != false))
	{
		/* Crank Angle Data */
		
		ca_ptr_DI = sizeof_apb +
			        sizeof_dgb;

		ca_mpladr = ca_ptr_DI +
			   	    128 * (number_ca_channels + number_ca_analysis_channels);
				 
		ca_aztadr = ca_mpladr +
		            20 * 5 * (number_ca_channels + number_ca_analysis_channels);
		         
		ca_aztlen = 4 * (unsigned int)(file->number_of_cycles/16+1);
		
		ca_rztadr = ca_aztadr +
		            ca_aztlen;
		            
		ca_rztlen = 2 * file->number_of_cycles;

		sizeof_ca_headers = 128 * (number_ca_channels + number_ca_analysis_channels) +
				            20 * 5 * (number_ca_channels + number_ca_analysis_channels) +
							ca_aztlen +
							ca_rztlen;

		sizeof_ca_data = 0;

		if (output_ca != false)
		{
			for (channel=0;channel<file->number_of_channels;channel++)
			{
				if ((file->channel_data[channel].loaded != false) && (file->channel_data[channel].abscissa.type == ABSCISSA_CRANKANGLE))
				{
					sizeof_ca_data += file->number_of_cycles * file->channel_data[channel].samples_per_cycle * sizeof_ca_variable;
				}
			}
		}

		if ((output_ca_analysis != false) && (analysis != NULL))
		{
			for (analysis_number=0;analysis_number<analysis->number_of_channels;analysis_number++)
			{
				if (file->channel_data[analysis->channel[analysis_number].channel].loaded == true)
				{
					for (analysis_channel = 0; analysis_channel < get_number_of_analysis_channels(); analysis_channel++)
					{
						if ((analysis->channel[analysis_number].analysis_rqst[analysis_channel] >= AnalysisRqst::CalculateAndStore) && 
							(analysis->channel[analysis_number].results[analysis_channel].abscissa.type == ABSCISSA_CRANKANGLE))
						{
							sizeof_ca_data += file->number_of_cycles * analysis->channel[analysis_number].results[analysis_channel].samples_per_cycle * sizeof_ca_variable;
						}
					}
				}
			}
		}

		ifile_grpflg |= 0x02;
	}

	if ((output_time != false) || (output_time_results != false))
	{
		/* Time Based Data */

		time_ptr_DI = sizeof_apb +
			          sizeof_dgb +
				      sizeof_ca_headers;

		time_mpladr = time_ptr_DI +
				      128 * (number_time_based_channels + number_time_based_results_channels);

		sizeof_time_headers = 128 * (number_time_based_channels + number_time_based_results_channels) +
				              20 * 1 * (number_time_based_channels + number_time_based_results_channels);
		
		sizeof_time_data = 0;
		
		if (output_time != false)
		{
			for (channel=0;channel<file->number_of_channels;channel++)
			{
				if ((file->channel_data[channel].loaded != false) && (file->channel_data[channel].abscissa.type == ABSCISSA_TIME) && (output_time == true))
				{
					number_of_time_datapoints = file->channel_data[channel].samples_per_cycle;

					sizeof_time_data += number_of_time_datapoints * sizeof_time_variable;
				}
			}
		}

		if ((output_time_results != false) && (analysis != NULL))
		{
			for (analysis_number=0;analysis_number<analysis->number_of_channels;analysis_number++)
			{
				if (file->channel_data[analysis->channel[analysis_number].channel].loaded == true)
				{
					for (analysis_channel = 0; analysis_channel < get_number_of_analysis_channels(); analysis_channel++)
					{
						if ((analysis->channel[analysis_number].analysis_rqst[analysis_channel] >= AnalysisRqst::CalculateAndStore) && 
							(analysis->channel[analysis_number].results[analysis_channel].abscissa.type == ABSCISSA_TIME))
						{
							number_of_time_datapoints = analysis->channel[analysis_number].results[analysis_channel].samples_per_cycle;

							sizeof_time_data += number_of_time_datapoints * sizeof_time_variable;
						}
					}
				}
			}
		}

		ifile_grpflg |= 0x04;
	}

	if ((output_cyc_results != false) || (output_raw_results != false))
	{
		/* Results */

		res_ptr_DI = sizeof_apb +
			         sizeof_dgb +
				     sizeof_ca_headers +
				     sizeof_time_headers +
				     sizeof_rtp_results_headers;

		res_mpladr = res_ptr_DI +
				     128 * ( number_cyc_analysis_channels + number_raw_analysis_channels );

		sizeof_cyc_results_headers = 128 * ( number_cyc_analysis_channels + number_raw_analysis_channels ) +
				                     20 * 1 * ( number_cyc_analysis_channels + number_raw_analysis_channels );

		sizeof_cyc_results_data = 0;
		
		if (output_raw_results == true)
		{
			for (channel=0;channel<file->number_of_channels;channel++)
			{
				if (file->channel_data[channel].loaded != false)
				{
					if (file->channel_data[channel].abscissa.type == ABSCISSA_CYCLE)
					{
						sizeof_cyc_results_data += file->number_of_cycles * sizeof_cyc_variable;
					}
				}
			}
		}
			
		if ((output_cyc_results == true) && (analysis != NULL))
		{
			for (analysis_number=0;analysis_number<analysis->number_of_channels;analysis_number++)
			{
				if (file->channel_data[analysis->channel[analysis_number].channel].loaded == true)
				{
					for (analysis_channel = 0; analysis_channel < get_number_of_analysis_channels(); analysis_channel++)
					{
						if ((analysis->channel[analysis_number].analysis_rqst[analysis_channel] >= AnalysisRqst::CalculateAndStore) && 
							(analysis->channel[analysis_number].results[analysis_channel].abscissa.type == ABSCISSA_CYCLE))
						{
							sizeof_cyc_results_data += file->number_of_cycles * sizeof_cyc_variable;
						}
					}
				}
			}
		}

		ifile_grpflg |= 0x10;
	}
	
	sizeof_header = sizeof_apb +
			        sizeof_dgb +
				    sizeof_ca_headers +
				    sizeof_time_headers +
					sizeof_rtp_results_headers +
				    sizeof_cyc_results_headers +
					sizeof_utc_headers;

	calc_file_size = sizeof_header + sizeof_ca_data + sizeof_time_data + sizeof_rtp_results_data + sizeof_cyc_results_data;

	logmessage(NOTICE,"done\n");
    	
	if ((sizeof_ca_headers > 0) && (sizeof_ca_data == 0))
	{
	    logmessage(WARNING,"CA header calculated but no CA data\n");
	}
	    	
	if ((sizeof_time_headers > 0) && (sizeof_time_data == 0))
	{
	    logmessage(WARNING,"Time header calculated but no time data\n");
	}

	if ((sizeof_rtp_results_headers > 0) && (sizeof_rtp_results_data == 0))
	{
	    logmessage(WARNING,"RTP results header calculated but no RTP results data\n");
	}
	    
	if ((sizeof_cyc_results_headers > 0) && (sizeof_cyc_results_data == 0))
	{
	    logmessage(WARNING,"Cycle results header calculated but no cycle results data\n");
	}
	    	
	if ((sizeof_utc_headers > 0) && (sizeof_utc_data == 0))
	{
	    logmessage(WARNING,"UTC header calculated but no UTC data\n");
	}

	logmessage(NOTICE,"Saving APB...");

	desaxi = (int*)malloc(file->number_of_channels * sizeof(int));
	if (desaxi == NULL)
	{
		fclose(output_file);
		logmessage(FATAL,"Memory could not be allocated");
		return(false);
	}

	calculate_pin_offsets(file,&pin_offset,&pin_offset_2,desaxi);

	fwrite_short(1024,output_file);
	(void)fwrite(&grpanz,2,1,output_file);
	(void)fwrite(&file->comment,80,1,output_file);

	if (file->ptr_APB == NULL)
	{
		create_time(&file->creation_time,mesdat);

		(void)fwrite(&mesdat, 24, 1, output_file);
		fwrite_null(18, output_file);
		fwrite_null(18, output_file);
		fwrite_null(10, output_file);
	}
	else
	{
		if (file->ptr_APB->mesdat[0] != 0x00)
		{
			(void)fwrite(&file->ptr_APB->mesdat, 24, 1, output_file);
		}
		else
		{
			create_time(&file->creation_time,mesdat);
			(void)fwrite(&mesdat,24,1,output_file);
		}
			
		(void)fwrite(&file->ptr_APB->parfil, 18, 1, output_file);

		(void)fwrite(&file->ptr_APB->pronam, 18, 1, output_file);
		(void)fwrite(&file->ptr_APB->prfstd, 10, 1, output_file);
	}

	(void)fwrite(&file->engine.name,18,1,output_file);
	
	if ((file->engine.type == ENGINE_CI_DI) || (file->engine.type == ENGINE_CI_IDI))
	{
		fwrite_short(0,output_file);
	}
	else
	{
		fwrite_short(1,output_file);
	}

	if (file->engine.number_of_strokes == 2)
	{
		fwrite_short(1,output_file);
	}
	else
	{
		fwrite_short(0,output_file);
	}
	fwrite_short(0,output_file);
	fwrite_double((double)file->engine.stroke,output_file);
	fwrite_double((double)file->engine.conrod_length,output_file);
	fwrite_double((double)file->engine.bore,output_file);
	fwrite_double((double)file->engine.compression_ratio,output_file);
	fwrite_double((double)pin_offset,output_file);
	fwrite_int(ifile_grpflg,output_file);
	fwrite_int(0,output_file);
	fwrite_short(1908,output_file);						/* Software version number */
	fwrite_double((double)pin_offset_2,output_file);
	fwrite_uint(calc_file_size,output_file);			/* pExtensions */

	for (parameter_number = 0; parameter_number < 28; parameter_number++)
	{
		Parameter* parameter = file->parameters;
		bool found = false;

		while ((parameter != NULL) && (found == false))
		{
			if ((parameter->number == (int)parameter_number) && (parameter->data_type == VARIABLE_DOUBLE))
			{
				(void)fwrite(&parameter->name, 10, 1, output_file);
				(void)fwrite(&parameter->units, 10, 1, output_file);
				(void)fwrite(parameter->value, 8, 1, output_file);

				found = true;
			}

			parameter = parameter->next;
		}

		if (found == false)
		{
			fwrite_null(10, output_file);
			fwrite_null(10, output_file);
			fwrite_double(0.0, output_file);
		}
	}
	
	logmessage(NOTICE,"done\n");
	
	if (debug_level >= WARNING)
	{
		pos = ftell(output_file);
		 
		if (pos != (long)sizeof_apb)
		{
			logmessage(NOTICE,"Written APB is not %u bytes long\n",sizeof_apb);
		}
	}

	/* Save Data Group Descriptions (DGB) */
	
	if ((output_ca != false) || (output_ca_analysis != false))
	{
		/* Crank Angle Data */

		logmessage(NOTICE,"Saving Crank Angle Data DGB...");

		fwrite_short(1,output_file);
		fwrite_short(0,output_file);
		strncpy(unit, "deg", 10);
		fwrite_char(unit,10,output_file);
		(void)fwrite(&dltphi,8,1,output_file);
		(void)fwrite(&dltzei,8,1,output_file);
		fwrite_short(number_ca_channels+number_ca_analysis_channels,output_file);
		fwrite_int(0,output_file);
		fwrite_int(0,output_file);
		fwrite_short(sizeof_ca_variable == 2 ? 0 : 1, output_file);
		fwrite_short(5,output_file);
		fwrite_short(0,output_file);
		(void)fwrite(&file->number_of_cycles,4,1,output_file);
		fwrite_int(1,output_file);
		fwrite_int(file->number_of_cycles,output_file);
		fwrite_int(PulsesPerCycle(file),output_file);
		fwrite_int(0,output_file);
		fwrite_int(0,output_file);
		fwrite_short(sizeof_ca_variable, output_file);
		fwrite_short(0,output_file);
		fwrite_short(0,output_file);
		fwrite_short(0,output_file);
		(void)fwrite(&ca_ptr_DI,4,1,output_file);
		fwrite_int(0,output_file);
		(void)fwrite(&ca_mpladr,4,1,output_file);
		(void)fwrite(&ca_aztadr,4,1,output_file);
		(void)fwrite(&ca_rztadr,4,1,output_file);
		fwrite_int(0,output_file);
		fwrite_int((-1*(int)PulsesPerCycle(file)/2),output_file);
		fwrite_double((double)start_window,output_file);
		fwrite_double((double)finish_window,output_file);
		fwrite_double((double)polytropic_index,output_file);
		fwrite_double(0.0,output_file);
		(void)fwrite(ref_channel_cylpr, 1, 10, output_file);
		(void)fwrite(ref_channel_intake_manpr, 1, 10, output_file);
		(void)fwrite(ref_channel_exhaust_manpr, 1, 10, output_file);
		(void)fwrite(&ca_aztlen,4,1,output_file);
		fwrite_int(0,output_file);
		fwrite_int(0,output_file);
		fwrite_short(refzylmit, output_file);
		fwrite_short(refsaumit, output_file);
		fwrite_short(refausmit, output_file);
		fwrite_short(1,output_file);
		fwrite_int(0,output_file);
		fwrite_int(0,output_file);
		fwrite_int(0,output_file);
		fwrite_int(0,output_file);
		fwrite_int(0,output_file);
		fwrite_int(0,output_file);
		fwrite_null(44, output_file);
		
		logmessage(NOTICE,"done\n");
	}
	
	if ((output_time != false) || (output_time_results != false))
	{
		/* Time Based Data */
		
		logmessage(NOTICE,"Saving Time Based Data DGB...");

		fwrite_short(2,output_file);
		fwrite_short(0,output_file);
		strncpy(unit, "ms", 10);
		fwrite_char(unit,10,output_file);
		(void)fwrite(&file->time_resolution,8,1,output_file);
		(void)fwrite(&dltzei,8,1,output_file);
		fwrite_short((number_time_based_channels + number_time_based_results_channels),output_file);
		fwrite_int(0,output_file);
		fwrite_int(0,output_file);
		fwrite_short(sizeof_time_variable == 2 ? 0 : 1, output_file);
		fwrite_short(1,output_file);
		fwrite_short(0,output_file);
		fwrite_int(1,output_file);
		fwrite_int(1,output_file);
		fwrite_int(1,output_file);
		fwrite_int(0,output_file);
		fwrite_int(0,output_file);
		fwrite_int(0,output_file);
		fwrite_short(sizeof_time_variable, output_file);
		fwrite_short(0,output_file);
		fwrite_short(0,output_file);
		fwrite_short(0,output_file);
		(void)fwrite(&time_ptr_DI,4,1,output_file);
		fwrite_int(0,output_file);
		(void)fwrite(&time_mpladr,4,1,output_file);
		fwrite_int(0,output_file);
		fwrite_int(0,output_file);
		fwrite_int(0,output_file);
		fwrite_int(0,output_file);
		fwrite_double(0.0,output_file);
		fwrite_double(0.0,output_file);
		fwrite_double(0.0,output_file);
		fwrite_double(0.0,output_file);
		fwrite_null(10, output_file);
		fwrite_null(10, output_file);
		fwrite_null(10, output_file);
		fwrite_int(0,output_file);
		fwrite_int(0,output_file);
		fwrite_int(0,output_file);
		fwrite_short(0,output_file);
		fwrite_short(0,output_file);
		fwrite_short(0,output_file);
		fwrite_short(1,output_file);
		fwrite_int(0,output_file);
		fwrite_int(0,output_file);
		fwrite_int(0,output_file);
		fwrite_int(0,output_file);
		fwrite_int(0,output_file);
		fwrite_int(0,output_file);
		fwrite_null(44, output_file);

		if (debug_level >= NOTICE)
		{
			logmessage(NOTICE,"done\n");
		}
	}

	if ((output_cyc_results != false) || (output_raw_results != false))
	{
		/* Results */
		
		logmessage(NOTICE,"Saving Results DGB...");

		fwrite_short(4,output_file);
		fwrite_short(0,output_file);
		strncpy(unit, "cycle", 10);
		fwrite_char(unit,10,output_file);
		fwrite_double(1.0,output_file);
		fwrite_double(0.0,output_file);
		fwrite_short(number_cyc_analysis_channels + number_raw_analysis_channels,output_file);
		fwrite_int(0,output_file);
		fwrite_int(0,output_file);
		fwrite_short(sizeof_cyc_variable == 2 ? 0 : 1,output_file);
		fwrite_short(1,output_file);
		fwrite_short(0,output_file);
		fwrite_int(1,output_file);
		fwrite_int(1,output_file);
		fwrite_int(file->number_of_cycles,output_file);
		fwrite_int(0,output_file);
		fwrite_int(0,output_file);
		fwrite_int(0,output_file);
		fwrite_short(sizeof_cyc_variable,output_file);
		fwrite_short(0,output_file);
		fwrite_short(0,output_file);
		fwrite_short(0,output_file);
		fwrite_int(res_ptr_DI,output_file);
		fwrite_int(0,output_file);
		fwrite_int(res_mpladr,output_file);
		fwrite_int(0,output_file);
		fwrite_int(0,output_file);
		fwrite_int(0,output_file);
		fwrite_int(0,output_file);
		fwrite_double((double)start_window,output_file);
		fwrite_double((double)finish_window,output_file);
		fwrite_double((double)polytropic_index,output_file);
		fwrite_double(0.0,output_file);
		(void)fwrite(ref_channel_cylpr,1,10,output_file);
		(void)fwrite(ref_channel_intake_manpr,1,10,output_file);
		(void)fwrite(ref_channel_exhaust_manpr,1,10,output_file);
		fwrite_int(0,output_file);
		fwrite_int(0,output_file);
		fwrite_int(0,output_file);
		fwrite_short(0,output_file);
		fwrite_short(0,output_file);
		fwrite_short(0,output_file);
		fwrite_short(1,output_file);
		fwrite_int(0,output_file);
		fwrite_int(0,output_file);
		fwrite_int(0,output_file);
		fwrite_int(0,output_file);
		fwrite_int(0,output_file);
		fwrite_int(0,output_file);
		fwrite_null(44, output_file);

		logmessage(NOTICE,"done\n");
	}
	
	pos = ftell(output_file);
		
	if (pos != (long)(sizeof_apb + sizeof_dgb))
	{
		logmessage(WARNING,"File position after outputting APB and DGBs is not correct\n");
	}
	    	
	if ((output_ca != false) || (output_ca_analysis != false))
	{
		/* Crank Angle */

		/* Save Data Directory (DI) */

		pos = ftell(output_file);
			
		if (pos != (long)ca_ptr_DI)
		{
			logmessage(WARNING,"File position for start of crank angle DI not correct %li vs expected %u)\n",pos,ca_ptr_DI);
		}
		
		logmessage(NOTICE,"Saving Crank Angle DI...");
					
		if (output_ca != false)
		{
			for (channel=0;channel<file->number_of_channels;channel++)
			{
				if ((file->channel_data[channel].loaded != false) && (file->channel_data[channel].abscissa.type == ABSCISSA_CRANKANGLE))
				{
					(void)fwrite(file->channel_data[channel].short_name,10,1,output_file);
					(void)fwrite(file->channel_data[channel].units,10,1,output_file);
					fwrite_null(6, output_file);
					fwrite_float(file->channel_data[channel].min,output_file);
					fwrite_float(file->channel_data[channel].max,output_file);
					fwrite_double(0.0,output_file);
					fwrite_double(1.0,output_file);
					fwrite_int(DegreesToThetaAbs(file,file->channel_data[channel].tdc_offset),output_file);
					fwrite_short(desaxi[channel],output_file);
					fwrite_int(file->channel_data[channel].samples_per_cycle,output_file);
					fwrite_short(0,output_file);

					if (file->channel_data[channel].isoffset == true)
					{
						fwrite_short(0, output_file);
					}
					else
					{
						fwrite_short(avl_offset_type(file->channel_data[channel].offset_config.type), output_file);
					}

					fwrite_short(avl_signal_type(file->channel_data[channel].type),output_file);
					fwrite_short(0,output_file);
					fwrite_int(0,output_file);
					fwrite_double((double)file->channel_data[channel].offset_config.start_window,output_file);

					if (sizeof_ca_variable == 4)
					{
						kalfak_0 = 0.0;
						kalfak_1 = 1.0;
					}
					else
					{
						kalfak_0 = ((double)file->channel_data[channel].max + (double)file->channel_data[channel].min) / 2.0;
						kalfak_1 = ((double)file->channel_data[channel].max - (double)file->channel_data[channel].min) / 65534.0;

					}

					fwrite_double(kalfak_0,output_file);
					fwrite_double(kalfak_1,output_file);
					fwrite_int(0,output_file);
					fwrite_int(0,output_file);
					fwrite_int(0,output_file);
					fwrite_int(0,output_file);
					fwrite_int(0,output_file);
					fwrite_int(0,output_file);
					fwrite_double((double)file->channel_data[channel].offset_config.fixed_value,output_file);
				}
			}
		}

		if (output_ca_analysis != false)
		{
			for (analysis_number=0;analysis_number<analysis->number_of_channels;analysis_number++)
			{
				if (file->channel_data[analysis->channel[analysis_number].channel].loaded == true)
				{
					for (analysis_channel = 0; analysis_channel < get_number_of_analysis_channels(); analysis_channel++)
					{
						if ((analysis->channel[analysis_number].analysis_rqst[analysis_channel] >= AnalysisRqst::CalculateAndStore) && 
							(analysis->channel[analysis_number].results[analysis_channel].abscissa.type == ABSCISSA_CRANKANGLE))
						{
							(void)fwrite(analysis->channel[analysis_number].results[analysis_channel].short_name, 10, 1, output_file);
							(void)fwrite(analysis->channel[analysis_number].results[analysis_channel].units, 10, 1, output_file);
							fwrite_null(6, output_file);
							fwrite_float(analysis->channel[analysis_number].results[analysis_channel].min, output_file);
							fwrite_float(analysis->channel[analysis_number].results[analysis_channel].max, output_file);
							fwrite_double(0.0, output_file);
							fwrite_double(1.0, output_file);
							fwrite_int(DegreesToThetaAbs(file, file->channel_data[analysis->channel[analysis_number].channel].tdc_offset), output_file);
							fwrite_short(0, output_file);
							fwrite_int(analysis->channel[analysis_number].results[analysis_channel].samples_per_cycle, output_file);
							fwrite_short(0, output_file);
							fwrite_short(0, output_file);
							fwrite_short(0, output_file);
							fwrite_short(0, output_file);
							fwrite_int(0, output_file);
							fwrite_double(0.0, output_file);

							if (sizeof_ca_variable == 4)
							{
								kalfak_0 = 0.0;
								kalfak_1 = 1.0;
							}
							else
							{
								kalfak_0 = ((double)analysis->channel[analysis_number].results[analysis_channel].max + (double)analysis->channel[analysis_number].results[analysis_channel].min) / 2.0;
								kalfak_1 = ((double)analysis->channel[analysis_number].results[analysis_channel].max - (double)analysis->channel[analysis_number].results[analysis_channel].min) / 65534.0;
							}

							fwrite_double(kalfak_0, output_file);
							fwrite_double(kalfak_1, output_file);
							fwrite_int(0, output_file);
							fwrite_int(0, output_file);
							fwrite_int(0, output_file);
							fwrite_int(0, output_file);
							fwrite_int(0, output_file);
							fwrite_int(0, output_file);
							fwrite_double(0.0, output_file);
						}
					}
				}
			}
		}
				
		logmessage(NOTICE,"done\n");

		/* Save Measurement Table (MT) */

		pos = ftell(output_file);
			
		if (pos != (long)ca_mpladr)
		{
			logmessage(WARNING,"File position for start of crank angle MT not correct\n");
		}
			
		logmessage(NOTICE,"Saving Crank Angle MT...");

		file_position = sizeof_header;
		ca_data_start = file_position;

		if (output_ca != false)
		{
			for (channel=0;channel<file->number_of_channels;channel++)
			{
				if ((file->channel_data[channel].loaded != false) && (file->channel_data[channel].abscissa.type == ABSCISSA_CRANKANGLE))
				{
					channel_size = 0;
				
					for (measurement_table=0;measurement_table<file->channel_data[channel].abscissa.number_of_measurement_tables;measurement_table++)
					{
						//start = file->channel_data[channel].abscissa.start[measurement_table] - 1 * (int)PulsesPerCycle(file) / 2;

						start = (int)(ThetaToDegrees(file, file->channel_data[channel].abscissa.start[measurement_table]) * file->minimum_resolution);

						fwrite_int(start, output_file);
						fwrite_int(file->channel_data[channel].abscissa.number_of_samples[measurement_table],output_file);
						fwrite_int(file->channel_data[channel].abscissa.resolution[measurement_table],output_file);
						fwrite_int(file_position,output_file);
						fwrite_int(1,output_file);

						channel_size += file->channel_data[channel].abscissa.number_of_samples[measurement_table];
						file_position += file->channel_data[channel].abscissa.number_of_samples[measurement_table] * sizeof_ca_variable;
					}
					
					if (channel_size != file->channel_data[channel].samples_per_cycle)
					{
						logmessage(WARNING,"Mismatch abscissa size and channel size (%u vs. %u)\n",channel_size,file->channel_data[channel].samples_per_cycle);
					}
					
					if (file->channel_data[channel].abscissa.number_of_measurement_tables < 5)
					{
						for (measurement_table=file->channel_data[channel].abscissa.number_of_measurement_tables;measurement_table<5;measurement_table++)
						{
							fwrite_int(0,output_file);
							fwrite_int(0,output_file);
							fwrite_int(0,output_file);
							fwrite_int(file_position,output_file);
							fwrite_int(0,output_file);
						}
					}
					
					file_position += (file->number_of_cycles-1) * file->channel_data[channel].samples_per_cycle * sizeof_ca_variable;
				}
			}
		}
			
		ca_rtp_start = file_position;

		if (output_ca_analysis != false)
		{
			for (analysis_number=0;analysis_number<analysis->number_of_channels;analysis_number++)
			{
				if (file->channel_data[analysis->channel[analysis_number].channel].loaded == true)
				{
					for (analysis_channel = 0; analysis_channel < get_number_of_analysis_channels(); analysis_channel++)
					{
						if ((analysis->channel[analysis_number].analysis_rqst[analysis_channel] >= AnalysisRqst::CalculateAndStore) && 
							(analysis->channel[analysis_number].results[analysis_channel].abscissa.type == ABSCISSA_CRANKANGLE))
						{
							for (measurement_table = 0; measurement_table < analysis->channel[analysis_number].results[analysis_channel].abscissa.number_of_measurement_tables; measurement_table++)
							{
								//start = analysis->channel[analysis_number].results[analysis_channel].abscissa.start[measurement_table] - 1 * (int)PulsesPerCycle(file) / 2;
								
								start = (int)(ThetaToDegrees(file, analysis->channel[analysis_number].results[analysis_channel].abscissa.start[measurement_table]) * file->minimum_resolution);
								
								fwrite_int(start, output_file);
								fwrite_int(analysis->channel[analysis_number].results[analysis_channel].abscissa.number_of_samples[measurement_table], output_file);
								fwrite_int(analysis->channel[analysis_number].results[analysis_channel].abscissa.resolution[measurement_table], output_file);
								fwrite_int(file_position, output_file);
								fwrite_int(1, output_file);

								file_position += analysis->channel[analysis_number].results[analysis_channel].abscissa.number_of_samples[measurement_table] * sizeof_ca_variable;
							}

							if (analysis->channel[analysis_number].results[analysis_channel].abscissa.number_of_measurement_tables < 5)
							{
								for (measurement_table = analysis->channel[analysis_number].results[analysis_channel].abscissa.number_of_measurement_tables; measurement_table < 5; measurement_table++)
								{
									fwrite_int(0, output_file);
									fwrite_int(0, output_file);
									fwrite_int(0, output_file);
									fwrite_int(file_position, output_file);
									fwrite_int(0, output_file);
								}
							}

							file_position += (file->number_of_cycles - 1) * analysis->channel[analysis_number].results[analysis_channel].samples_per_cycle * sizeof_ca_variable;
						}
					}
				}
			}
		}
					
		logmessage(NOTICE,"done\n");
	
		if (file_position != (sizeof_header + sizeof_ca_data))
		{
			logmessage(WARNING,"File position and calculated file position after writing CA measurement tables do not agree (%u vs %u)\n",file_position,calc_file_size);
		}
			
		/* Save Absolute Time Table (AZT) */
		
		pos = ftell(output_file);
			
		if (pos != (long)ca_aztadr)
		{
			logmessage(WARNING,"File position for start of crank angle AZT not correct\n");
		}

		logmessage(NOTICE,"Saving Crank Angle AZT...");
		
		total_duration = 0.0f;

		for (cycle=0;cycle<file->number_of_cycles;cycle++)
		{
			if (cycle % 16 == 0)
			{
				fwrite_int((int)(total_duration/dltzei),output_file);
			}
			
			if (file->channel_data[timing_channel].duration != NULL)
			{
				total_duration += file->channel_data[timing_channel].duration[cycle];
			}
		}

		if (cycle % 16 == 0)
		{
			fwrite_int((int)(total_duration/dltzei),output_file);
		}
	
		logmessage(NOTICE,"done\n");
		
		/* Save Relative Time Table (RZT) */

		pos = ftell(output_file);
			
		if (pos != (long)ca_rztadr)
		{
			logmessage(WARNING,"File position for start of crank angle RZT not correct\n");
		}
		
		logmessage(NOTICE,"Saving Crank Angle RZT...");
		
		if (file->channel_data[timing_channel].duration != NULL)
		{
			for (cycle = 0; cycle < file->number_of_cycles; cycle++)
			{
				fwrite_short((short)(file->channel_data[timing_channel].duration[cycle] / dltzei), output_file);
			}
		}
		else
		{
			for (cycle = 0; cycle < file->number_of_cycles; cycle++)
			{
				fwrite_short(0, output_file);
			}
		}
				
		logmessage(NOTICE,"done\n");
		
		pos = ftell(output_file);
		if (pos != (long)(ca_rztadr + ca_rztlen))
		{
			logmessage(WARNING,"File position for end of crank angle RZT not correct\n");
		}
	}
	
	if (desaxi != NULL)
	{
		free(desaxi);
		desaxi = NULL;
	}

	if ((output_time != false) || (output_time_results != false))
	{
		/* Time based data */
		
		logmessage(NOTICE,"Saving Time Based DI...");

		if (output_time != false)
		{
			for (channel=0;channel<file->number_of_channels;channel++)
			{
				if ((file->channel_data[channel].loaded != false) && (file->channel_data[channel].abscissa.type == ABSCISSA_TIME))
				{
					number_of_time_datapoints = file->channel_data[channel].samples_per_cycle;
					
					fwrite(file->channel_data[channel].short_name,10,1,output_file);
					fwrite(file->channel_data[channel].units,10,1,output_file);
					fwrite_null(6, output_file);
					fwrite_float(file->channel_data[channel].min,output_file);
					fwrite_float(file->channel_data[channel].max,output_file);
					fwrite_double(0.0,output_file);
					fwrite_double(1.0,output_file);
					fwrite_int(0,output_file);
					fwrite_short(0,output_file);
					fwrite_int(number_of_time_datapoints,output_file);
					fwrite_short(0,output_file);
					fwrite_short(0,output_file);
					fwrite_short(0,output_file);
					fwrite_short(0,output_file);
					fwrite_int(0,output_file);
					fwrite_double(0.0,output_file);

					if (sizeof_time_variable == 4)
					{
						kalfak_0 = 0.0;
						kalfak_1 = 1.0;
					}
					else
					{
						kalfak_0 = ((double)file->channel_data[channel].max + (double)file->channel_data[channel].min) / 2.0;
						kalfak_1 = ((double)file->channel_data[channel].max - (double)file->channel_data[channel].min) / 65534.0;
					}

					fwrite_double(kalfak_0,output_file);
					fwrite_double(kalfak_1,output_file);
					fwrite_int(0,output_file);
					fwrite_int(0,output_file);
					fwrite_int(0,output_file);
					fwrite_int(0,output_file);
					fwrite_int(0,output_file);
					fwrite_int(0,output_file);
					fwrite_double(0.0,output_file);
				}
			}
		}
			
		if (output_time_results != false)
		{
			for (analysis_number=0;analysis_number<analysis->number_of_channels;analysis_number++)
			{
				if (file->channel_data[analysis->channel[analysis_number].channel].loaded == true)
				{
					for (analysis_channel = 0; analysis_channel < get_number_of_analysis_channels(); analysis_channel++)
					{
						if ((analysis->channel[analysis_number].analysis_rqst[analysis_channel] >= AnalysisRqst::CalculateAndStore) && 
							(analysis->channel[analysis_number].results[analysis_channel].abscissa.type == ABSCISSA_TIME))
						{
							number_of_time_datapoints = analysis->channel[analysis_number].results[analysis_channel].samples_per_cycle;

							fwrite(analysis->channel[analysis_number].results[analysis_channel].short_name, 10, 1, output_file);
							fwrite(analysis->channel[analysis_number].results[analysis_channel].units, 10, 1, output_file);
							fwrite_null(6, output_file);
							fwrite_float(analysis->channel[analysis_number].results[analysis_channel].min, output_file);
							fwrite_float(analysis->channel[analysis_number].results[analysis_channel].max, output_file);
							fwrite_double(0.0, output_file);
							fwrite_double(1.0, output_file);
							fwrite_int(0, output_file);
							fwrite_short(0, output_file);
							fwrite_int(number_of_time_datapoints, output_file);
							fwrite_short(0, output_file);
							fwrite_short(0, output_file);
							fwrite_short(0, output_file);
							fwrite_short(0, output_file);
							fwrite_int(0, output_file);
							fwrite_double(0.0, output_file);

							if (sizeof_time_variable == 4)
							{
								kalfak_0 = 0.0;
								kalfak_1 = 1.0;
							}
							else
							{
								kalfak_0 = ((double)analysis->channel[analysis_number].results[analysis_channel].max + (double)analysis->channel[analysis_number].results[analysis_channel].min) / 2.0;
								kalfak_1 = ((double)analysis->channel[analysis_number].results[analysis_channel].max - (double)analysis->channel[analysis_number].results[analysis_channel].min) / 65534.0;
							}

							fwrite_double(kalfak_0, output_file);
							fwrite_double(kalfak_1, output_file);
							fwrite_int(0, output_file);
							fwrite_int(0, output_file);
							fwrite_int(0, output_file);
							fwrite_int(0, output_file);
							fwrite_int(0, output_file);
							fwrite_int(0, output_file);
							fwrite_double(0.0, output_file);
						}
					}
				}
			}
		}
		
		logmessage(NOTICE,"done\n");

		/* Save Measurement Table (MT) */

		logmessage(NOTICE,"Saving Time Based MT...");

		file_position = sizeof_header + sizeof_ca_data;
		time_start = file_position;

		if (output_time != false)
		{
			for (channel=0;channel<file->number_of_channels;channel++)
			{
				if ((file->channel_data[channel].loaded != false) && (file->channel_data[channel].abscissa.type == ABSCISSA_TIME))
				{
					number_of_time_datapoints = file->channel_data[channel].samples_per_cycle;
					timebase_resolution = (int)file->channel_data[channel].abscissa.resolution[0];

					fwrite_int(0,output_file);
					fwrite_int(number_of_time_datapoints,output_file);
					fwrite_int(timebase_resolution,output_file);
					fwrite_int(file_position,output_file);
					fwrite_int(1,output_file);
				
					file_position += number_of_time_datapoints * sizeof_time_variable;
				}
			}
		}
			
		if (output_time_results != false)
		{
			for (analysis_number=0;analysis_number<analysis->number_of_channels;analysis_number++)
			{
				if (file->channel_data[analysis->channel[analysis_number].channel].loaded == true)
				{
					for (analysis_channel = 0; analysis_channel < get_number_of_analysis_channels(); analysis_channel++)
					{
						if ((analysis->channel[analysis_number].analysis_rqst[analysis_channel] >= AnalysisRqst::CalculateAndStore) && 
							(analysis->channel[analysis_number].results[analysis_channel].abscissa.type == ABSCISSA_TIME))
						{
							number_of_time_datapoints = analysis->channel[analysis_number].results[analysis_channel].samples_per_cycle;
							timebase_resolution = (int)analysis->channel[analysis_number].results[analysis_channel].abscissa.resolution[0];

							fwrite_int(0, output_file);
							fwrite_int(number_of_time_datapoints, output_file);
							fwrite_int(timebase_resolution, output_file);
							fwrite_int(file_position, output_file);
							fwrite_int(1, output_file);

							file_position += number_of_time_datapoints * sizeof_time_variable;
						}
					}
				}
			}
		}
										
		logmessage(NOTICE,"done\n");
		
		if (file_position != (sizeof_header + sizeof_ca_data + sizeof_time_data))
		{
			logmessage(WARNING,"File position and calculated file position after writing time based mesurement tables do not agree (%u vs %u)\n",file_position,calc_file_size);
		}
	}
		
	if ((output_cyc_results != false) || (output_raw_results != false))
	{
		/* Results */
		
		pos = ftell(output_file);
			
		if (pos != (long)res_ptr_DI)
		{
			logmessage(WARNING,"File position for start of Results DI not correct (%li vs %u)\n",pos,res_ptr_DI);
		}
		
		logmessage(NOTICE,"Saving Results DI...");

		if (output_raw_results != false)
		{
			for (channel=0;channel<file->number_of_channels;channel++)
			{
				if (file->channel_data[channel].loaded != false)
				{
					if (file->channel_data[channel].abscissa.type == ABSCISSA_CYCLE)
					{
						(void)fwrite(file->channel_data[channel].short_name,10,1,output_file);
						(void)fwrite(file->channel_data[channel].units,10,1,output_file);
						fwrite_null(6, output_file);
						fwrite_float(file->channel_data[channel].min,output_file);
						fwrite_float(file->channel_data[channel].max,output_file);
						fwrite_double(0.0,output_file);
						fwrite_double(1.0,output_file);
						fwrite_int(0,output_file);
						fwrite_short(0,output_file);
						fwrite_int(file->number_of_cycles,output_file);
						fwrite_short(0,output_file);
						fwrite_short(0,output_file);
						fwrite_short(0,output_file);
						fwrite_short(0,output_file);
						fwrite_int(0,output_file);
						fwrite_double(0.0,output_file);

						if (sizeof_cyc_variable == 4)
						{
							kalfak_0 = 0.0;
							kalfak_1 = 1.0;
						}
						else
						{
							kalfak_0 = ((double)file->channel_data[channel].max + (double)file->channel_data[channel].min) / 2.0;
							kalfak_1 = ((double)file->channel_data[channel].max - (double)file->channel_data[channel].min) / 65534.0;
						}

						fwrite_double(kalfak_0,output_file);
						fwrite_double(kalfak_1,output_file);
						fwrite_int(0,output_file);
						fwrite_int(0,output_file);
						fwrite_int(0,output_file);
						fwrite_int(0,output_file);
						fwrite_int(0,output_file);
						fwrite_int(0,output_file);
						fwrite_double(0.0,output_file);
					}
				}
			}
		}
	
		if (output_cyc_results != false)
		{
			for (analysis_number=0;analysis_number<analysis->number_of_channels;analysis_number++)
			{
				if (file->channel_data[analysis->channel[analysis_number].channel].loaded == true)
				{
					for (analysis_channel = 0; analysis_channel < get_number_of_analysis_channels(); analysis_channel++)
					{
						if ((analysis->channel[analysis_number].analysis_rqst[analysis_channel] >= AnalysisRqst::CalculateAndStore) && 
							(analysis->channel[analysis_number].results[analysis_channel].abscissa.type == ABSCISSA_CYCLE))
						{
							(void)fwrite(analysis->channel[analysis_number].results[analysis_channel].short_name, 10, 1, output_file);
							(void)fwrite(analysis->channel[analysis_number].results[analysis_channel].units, 10, 1, output_file);
							fwrite_null(6, output_file);
							fwrite_float(analysis->channel[analysis_number].results[analysis_channel].max, output_file);
							fwrite_float(analysis->channel[analysis_number].results[analysis_channel].min, output_file);
							fwrite_double(0.0, output_file);
							fwrite_double(1.0, output_file);
							fwrite_int(0, output_file);
							fwrite_short(0, output_file);
							fwrite_int(file->number_of_cycles, output_file);
							fwrite_short(0, output_file);
							fwrite_short(0, output_file);
							fwrite_short(0, output_file);
							fwrite_short(0, output_file);
							fwrite_int(0, output_file);
							fwrite_double(0.0, output_file);

							if (sizeof_cyc_variable == 4)
							{
								kalfak_0 = 0.0;
								kalfak_1 = 1.0;
							}
							else
							{
								kalfak_0 = ((double)analysis->channel[analysis_number].results[analysis_channel].max + (double)analysis->channel[analysis_number].results[analysis_channel].min) / 2.0;
								kalfak_1 = ((double)analysis->channel[analysis_number].results[analysis_channel].max - (double)analysis->channel[analysis_number].results[analysis_channel].min) / 65534.0;
							}

							fwrite_double(kalfak_0, output_file);
							fwrite_double(kalfak_1, output_file);
							fwrite_int(0, output_file);
							fwrite_int(0, output_file);
							fwrite_int(0, output_file);
							fwrite_int(0, output_file);
							fwrite_int(0, output_file);
							fwrite_int(0, output_file);
							fwrite_double(0, output_file);
						}
					}
				}
			}
		}
	
		logmessage(NOTICE,"done\n");
	
		/* Save Measurement Table (MT) */

		pos = ftell(output_file);
			
		if (pos != (long)res_mpladr)
		{
			logmessage(WARNING,"File position for start of Results MT not correct (%li vs %u)\n",pos,res_mpladr);
		}

		logmessage(NOTICE,"Saving Results MT...");

		file_position = sizeof_header + sizeof_ca_data + sizeof_time_data + sizeof_rtp_results_data;

		results_start = file_position;

		if (output_raw_results != false)
		{
			for (channel=0;channel<file->number_of_channels;channel++)
			{
				if (file->channel_data[channel].loaded != false)
				{
					if (file->channel_data[channel].abscissa.type == ABSCISSA_CYCLE)
					{
						fwrite_int(1,output_file);
						fwrite_int(file->number_of_cycles,output_file);
						fwrite_int(1,output_file);
						fwrite_int(file_position,output_file);
						fwrite_int(1,output_file);
						
						file_position += file->number_of_cycles * sizeof_cyc_variable;							
					}
				}
			}
		}
			
		if (output_cyc_results != false)
		{
			for (analysis_number=0;analysis_number<analysis->number_of_channels;analysis_number++)
			{
				if (file->channel_data[analysis->channel[analysis_number].channel].loaded == true)
				{
					for (analysis_channel = 0; analysis_channel < get_number_of_analysis_channels(); analysis_channel++)
					{
						if ((analysis->channel[analysis_number].analysis_rqst[analysis_channel] >= AnalysisRqst::CalculateAndStore) && 
							(analysis->channel[analysis_number].results[analysis_channel].abscissa.type == ABSCISSA_CYCLE))
						{
							fwrite_int(1, output_file);
							fwrite_int(file->number_of_cycles, output_file);
							fwrite_int(1, output_file);
							fwrite_int(file_position, output_file);
							fwrite_int(1, output_file);

							file_position += file->number_of_cycles * sizeof_cyc_variable;
						}
					}
				}
			}
		}
		
		logmessage(NOTICE,"done\n");
		
		if (file_position != (sizeof_header + sizeof_ca_data + sizeof_time_data + sizeof_rtp_results_data + sizeof_cyc_results_data))
		{
			logmessage(WARNING,"File position and calculated file position after writing results mesurement tables do not agree (%u vs %u)\n",file_position,calc_file_size);
		}
	}

	/* Save Channel Data */

	if ((output_ca != false) || (output_ca_analysis != false))
	{
		/* Crank Angle */

		pos = ftell(output_file);
			
		if (pos != (long)ca_data_start)
		{
			logmessage(WARNING,"File position for start of raw CA data not correct (%li vs %u)\n",pos,ca_data_start);
		}
		
		logmessage(NOTICE,"Saving Crank Angle Data...");
		
		if (output_ca != false)
		{			
			for (channel=0;channel<file->number_of_channels;channel++)
			{
				if ((file->channel_data[channel].loaded != false) && (file->channel_data[channel].abscissa.type == ABSCISSA_CRANKANGLE))
				{
					number_of_datapoints = file->number_of_cycles * file->channel_data[channel].samples_per_cycle;

					if (sizeof_ca_variable == 4)
					{
						(void)fwrite(file->channel_data[channel].data, 4, number_of_datapoints, output_file);
					}
					else
					{
						kalfak_0 = ((double)file->channel_data[channel].max + (double)file->channel_data[channel].min) / 2.0;
						kalfak_1 = ((double)file->channel_data[channel].max - (double)file->channel_data[channel].min) / 65534.0;

						buffer = (short*)malloc(number_of_datapoints * sizeof(short));
						if (buffer == NULL)
						{
							fclose(output_file);
							logmessage(FATAL, "Memory could not be allocated\n");
							return(false);
						}

						float_to_short(file, file->channel_data[channel].data, buffer, number_of_datapoints, kalfak_0, kalfak_1);

						(void)fwrite(buffer, 2, number_of_datapoints, output_file);

						free(buffer);
					}
				}
			}
		}
			
		if (output_ca_analysis != false)
		{
			pos = ftell(output_file);
				
			if (pos != (long)ca_rtp_start)
			{
				logmessage(WARNING,"File position for start of CA analysis data not correct (%li vs %u)\n",pos,ca_rtp_start);
			}
			
			for (analysis_number=0;analysis_number<analysis->number_of_channels;analysis_number++)
			{
				if (file->channel_data[analysis->channel[analysis_number].channel].loaded == true)
				{
					for (analysis_channel = 0; analysis_channel < get_number_of_analysis_channels(); analysis_channel++)
					{
						if ((analysis->channel[analysis_number].analysis_rqst[analysis_channel] >= AnalysisRqst::CalculateAndStore) && 
							(analysis->channel[analysis_number].results[analysis_channel].abscissa.type == ABSCISSA_CRANKANGLE))
						{
							number_of_datapoints = file->number_of_cycles * analysis->channel[analysis_number].results[analysis_channel].samples_per_cycle;

							if (sizeof_ca_variable == 4)
							{
								(void)fwrite(analysis->channel[analysis_number].results[analysis_channel].data, 4, number_of_datapoints, output_file);
							}
							else
							{
								kalfak_0 = ((double)analysis->channel[analysis_number].results[analysis_channel].max + (double)analysis->channel[analysis_number].results[analysis_channel].min) / 2.0;
								kalfak_1 = ((double)analysis->channel[analysis_number].results[analysis_channel].max - (double)analysis->channel[analysis_number].results[analysis_channel].min) / 65534.0;

								buffer = (short*)malloc(number_of_datapoints * sizeof(short));
								if (buffer == NULL)
								{
									fclose(output_file);
									logmessage(FATAL, "Memory could not be allocated\n");
									return(false);
								}

								float_to_short(file, analysis->channel[analysis_number].results[analysis_channel].data, buffer, number_of_datapoints, kalfak_0, kalfak_1);

								(void)fwrite(buffer, 2, number_of_datapoints, output_file);

								free(buffer);
							}
						}
					}
				}
			}
		}
				
		logmessage(NOTICE,"done\n");
	}
		
	if ((output_time != false) || (output_time_results != false))
	{
		/* Time based data */

		pos = ftell(output_file);
			
		if (pos != (long)time_start)
		{
			logmessage(WARNING,"File position for start of time data (%li vs %u)\n",pos,time_start);
		}
		
		logmessage(NOTICE,"Saving Time Based Data...");

		if (output_time != false)
		{
			for (channel=0;channel<file->number_of_channels;channel++)
			{
				if ((file->channel_data[channel].loaded != false) && (file->channel_data[channel].abscissa.type == ABSCISSA_TIME))
				{
					number_of_time_datapoints = file->channel_data[channel].samples_per_cycle;

					if (sizeof_time_variable == 4)
					{
						(void)fwrite(file->channel_data[channel].data, 4, number_of_time_datapoints, output_file);
					}
					else
					{
						kalfak_0 = ((double)file->channel_data[channel].max + (double)file->channel_data[channel].min) / 2.0;
						kalfak_1 = ((double)file->channel_data[channel].max - (double)file->channel_data[channel].min) / 65534.0;

						buffer = (short*)malloc(number_of_time_datapoints * sizeof(short));
						if (buffer == NULL)
						{
							fclose(output_file);
							logmessage(FATAL, "Memory could not be allocated\n");
							return(false);
						}

						float_to_short(file, file->channel_data[channel].data, buffer, number_of_time_datapoints, kalfak_0, kalfak_1);

						(void)fwrite(buffer, 2, (size_t)number_of_time_datapoints, output_file);

						free(buffer);
					}
				}
			}
		}

		if (output_time_results != false)
		{
			for (analysis_number=0;analysis_number<analysis->number_of_channels;analysis_number++)
			{
				if (file->channel_data[analysis->channel[analysis_number].channel].loaded == true)
				{
					for (analysis_channel = 0; analysis_channel < get_number_of_analysis_channels(); analysis_channel++)
					{
						if ((analysis->channel[analysis_number].analysis_rqst[analysis_channel] >= AnalysisRqst::CalculateAndStore) && 
							(analysis->channel[analysis_number].results[analysis_channel].abscissa.type == ABSCISSA_TIME))
						{
							number_of_time_datapoints = analysis->channel[analysis_number].results[analysis_channel].abscissa.number_of_samples[0];

							if (sizeof_time_variable == 4)
							{
								(void)fwrite(analysis->channel[analysis_number].results[analysis_channel].data, 4, number_of_time_datapoints, output_file);
							}
							else
							{
								kalfak_0 = ((double)analysis->channel[analysis_number].results[analysis_channel].max + (double)analysis->channel[analysis_number].results[analysis_channel].min) / 2.0;
								kalfak_1 = ((double)analysis->channel[analysis_number].results[analysis_channel].max - (double)analysis->channel[analysis_number].results[analysis_channel].min) / 65534.0;

								buffer = (short*)malloc(number_of_time_datapoints * sizeof(short));
								if (buffer == NULL)
								{
									fclose(output_file);
									logmessage(FATAL, "Memory could not be allocated\n");
									return(false);
								}

								float_to_short(file, analysis->channel[analysis_number].results[analysis_channel].data, buffer, number_of_time_datapoints, kalfak_0, kalfak_1);

								(void)fwrite(buffer, 2, (size_t)number_of_time_datapoints, output_file);

								free(buffer);
							}
						}
					}
				}
			}
		}
				
		logmessage(NOTICE,"done\n");
		
		pos = ftell(output_file);
			
		if (pos != (long)(time_start+sizeof_time_data))
		{
			logmessage(WARNING,"File position for end of time data (%li vs %u)\n",pos,time_start+sizeof_time_data);
		}
	}
	
	if ((output_cyc_results != false) || (output_raw_results != false))
	{
		/* Results */
		
		pos = ftell(output_file);
			
		if (pos != (long)results_start)
		{
			logmessage(WARNING,"File position for start of results data (%li vs %u)\n",pos,results_start);
		}
			
		logmessage(NOTICE,"Saving Results Data...");
		
		if (output_raw_results != false)
		{
			for (channel=0;channel<file->number_of_channels;channel++)
			{
				if (file->channel_data[channel].loaded != false)
				{
					if (file->channel_data[channel].abscissa.type == ABSCISSA_CYCLE)
					{
						if (sizeof_cyc_variable == 4)
						{
							(void)fwrite(file->channel_data[channel].data, 4, (size_t)file->number_of_cycles, output_file);
						}
						else
						{
							kalfak_0 = ((double)file->channel_data[channel].max + (double)file->channel_data[channel].min) / 2.0;
							kalfak_1 = ((double)file->channel_data[channel].max - (double)file->channel_data[channel].min) / 65534.0;
							
							number_of_datapoints = file->number_of_cycles;
							
							buffer = (short*)malloc(number_of_datapoints * sizeof(unsigned int));
							if (buffer == NULL)
							{
								fclose(output_file);
								logmessage(FATAL,"Memory could not be allocated\n");
								return(false);
							}
		
							float_to_short(file,file->channel_data[channel].data,buffer,number_of_datapoints,kalfak_0,kalfak_1);
		
							(void)fwrite(buffer,2,(size_t)number_of_datapoints,output_file);
		
							free(buffer);
						}
					}
				}
			}
		}
						
		if (output_cyc_results != false)
		{
			for (analysis_number=0;analysis_number<analysis->number_of_channels;analysis_number++)
			{
				if (file->channel_data[analysis->channel[analysis_number].channel].loaded == true)
				{
					for (analysis_channel = 0; analysis_channel < get_number_of_analysis_channels(); analysis_channel++)
					{
						if ((analysis->channel[analysis_number].analysis_rqst[analysis_channel] >= AnalysisRqst::CalculateAndStore) && 
							(analysis->channel[analysis_number].results[analysis_channel].abscissa.type == ABSCISSA_CYCLE))
						{
							if (sizeof_cyc_variable == 4)
							{
								fwrite(analysis->channel[analysis_number].results[analysis_channel].data, 4, (size_t)file->number_of_cycles, output_file);
							}
							else
							{
								max = analysis->channel[analysis_number].results[analysis_channel].max;
								min = analysis->channel[analysis_number].results[analysis_channel].min;

								kalfak_0 = ((double)max + (double)min) / 2.0;
								kalfak_1 = ((double)max - (double)min) / 65534.0;

								for (cycle = 0; cycle < file->number_of_cycles; cycle++)
								{
									actual_value = analysis->channel[analysis_number].results[analysis_channel].data[cycle];
									value = (actual_value - kalfak_0) / kalfak_1;
									value_to_write = (short)value;

									(void)fwrite(&value_to_write, 2, 1, output_file);
								}
							}
						}
					}
				}
			}
		}

		logmessage(NOTICE,"done\n");
	}
	
	pos = ftell(output_file);
		
	if ( pos != (long)calc_file_size)
	{
		logmessage(WARNING,"Final file size is not correct (%li vs %u)\n",pos,calc_file_size);
	}

	logmessage(NOTICE, "Saving extended channel name information...");

	lFirstExtensionObjectFilePos = 4 * 3;
	lLastExtensionObjectFilePos = lFirstExtensionObjectFilePos;
	lNextExtensionObjectFilePos = lFirstExtensionObjectFilePos;

	fwrite_uint(0, output_file);
	fwrite_int(0, output_file);
	fwrite_int(0, output_file);

	number_of_objects = 0;

	for (channel = 0; channel < file->number_of_channels; channel++)
	{
		if ((file->channel_data[channel].loaded != false) &&
			(((file->channel_data[channel].abscissa.type == ABSCISSA_CRANKANGLE) && (output_ca != false)) ||
			 ((file->channel_data[channel].abscissa.type == ABSCISSA_TIME) && (output_time != false)) ||
			 ((file->channel_data[channel].abscissa.type == ABSCISSA_CYCLE) && (output_raw_results != false))))
		{
			if (strncmp(file->channel_data[channel].name, file->channel_data[channel].short_name, SIZEOF_SHORT_NAME) != 0)
			{
				lLastExtensionObjectFilePos = lNextExtensionObjectFilePos;
				number_of_objects++;

				snprintf(key, 256, "%s.LongName", file->channel_data[channel].short_name);

				eType = 5;
				ulKeyLength = (uint32_t)strlen(key) + 1;
				key[ulKeyLength - 1] = 0x00;
				lKeyFilePos = lNextExtensionObjectFilePos + (4 * 6);
				ulObjectSize = 4 + (uint32_t)strlen(file->channel_data[channel].name) + 1;
				lObjectFilePos = lKeyFilePos + ulKeyLength;
				lNextExtensionObjectFilePos = lObjectFilePos + ulObjectSize;

				/* Data Object */

				fwrite_int(eType, output_file);
				fwrite_uint(ulKeyLength, output_file);
				fwrite_int(lKeyFilePos, output_file);
				fwrite_uint(ulObjectSize, output_file);
				fwrite_int(lObjectFilePos, output_file);
				fwrite_int(lNextExtensionObjectFilePos, output_file);

				/* Key */

				fwrite(key, ulKeyLength, 1, output_file);

				/* Object */

				fwrite_uint(ulObjectSize - 4, output_file);
				fwrite(file->channel_data[channel].name, ulObjectSize - 4, 1, output_file);
			}

			if (file->channel_data[channel].abscissa.type != ABSCISSA_TIME)
			{
				lLastExtensionObjectFilePos = lNextExtensionObjectFilePos;
				number_of_objects++;

				/* catool Channel Type */

				snprintf(key, 256, "%s", file->channel_data[channel].name);

				/* Data Object */

				eType = 1001;
				ulKeyLength = (uint32_t)strlen(key) + 1;
				key[ulKeyLength - 1] = 0x00;
				lKeyFilePos = lNextExtensionObjectFilePos + (4 * 6);
				ulObjectSize = 4 + 12 + 4;
				lObjectFilePos = lKeyFilePos + ulKeyLength;
				lNextExtensionObjectFilePos = lObjectFilePos + ulObjectSize;

				fwrite_int(eType, output_file);
				fwrite_uint(ulKeyLength, output_file);
				fwrite_int(lKeyFilePos, output_file);
				fwrite_uint(ulObjectSize, output_file);
				fwrite_int(lObjectFilePos, output_file);
				fwrite_int(lNextExtensionObjectFilePos, output_file);

				/* Key */

				fwrite(key, ulKeyLength, 1, output_file);

				fwrite_int(12, output_file);
				strncpy(key, "ChannelType", 256);
				key[11] = 0x00;
				fwrite(key, 12, 1, output_file);

				/* Object */

				fwrite_int(file->channel_data[channel].type, output_file);
			}

			if (file->channel_data[channel].description[0] != 0x00)
			{
				lLastExtensionObjectFilePos = lNextExtensionObjectFilePos;
				number_of_objects++;

				switch (file->channel_data[channel].abscissa.type)
				{

				case ABSCISSA_CRANKANGLE: snprintf(key, 256, "CA'%s.Description", file->channel_data[channel].name); break;
				case ABSCISSA_CYCLE: snprintf(key, 256, "CY'%s.Description", file->channel_data[channel].name); break;
				case ABSCISSA_TIME: snprintf(key, 256, "TM'%s.Description", file->channel_data[channel].name); break;
				case ABSCISSA_ASYNC_UTC: snprintf(key, 256, "ASYNC'%s.Description", file->channel_data[channel].name); break;
				default:
				case ABSCISSA_UNKNOWN:
				case ABSCISSA_FREQUENCY: snprintf(key, 256, "%s.Description", file->channel_data[channel].name); break;
				}

				/* Data Object */

				uint32_t description_length = (uint32_t)strlen(file->channel_data[channel].description);

				eType = 3;
				ulKeyLength = (uint32_t)strlen(key) + 1;
				key[ulKeyLength - 1] = 0x00;
				lKeyFilePos = lNextExtensionObjectFilePos + (4 * 6);
				ulObjectSize = 4 + 12 + 4 + description_length + 1;
				lObjectFilePos = lKeyFilePos + ulKeyLength;
				lNextExtensionObjectFilePos = lObjectFilePos + ulObjectSize;

				fwrite_int(eType, output_file);
				fwrite_uint(ulKeyLength, output_file);
				fwrite_int(lKeyFilePos, output_file);
				fwrite_uint(ulObjectSize, output_file);
				fwrite_int(lObjectFilePos, output_file);
				fwrite_int(lNextExtensionObjectFilePos, output_file);

				/* Key */

				fwrite(key, ulKeyLength, 1, output_file);

				fwrite_int(12, output_file);
				strncpy(key, "Description", 256);
				key[11] = 0x00;
				fwrite(key, 12, 1, output_file);

				/* Object */

				fwrite_int(description_length + 1, output_file);
				fwrite(file->channel_data[channel].description, description_length + 1, 1, output_file);
			}

			if (file->channel_data[channel].soc_config.type == SOC_FIXED)
			{
				lLastExtensionObjectFilePos = lNextExtensionObjectFilePos;
				number_of_objects++;

				/* catool SOC Fixed Value */

				snprintf(key, 256, "%s", file->channel_data[channel].name);

				/* Data Object */

				eType = 1003;
				ulKeyLength = (uint32_t)strlen(key) + 1;
				key[ulKeyLength - 1] = 0x00;
				lKeyFilePos = lNextExtensionObjectFilePos + (4 * 6);
				ulObjectSize = 4 + 12 + 4;
				lObjectFilePos = lKeyFilePos + ulKeyLength;
				lNextExtensionObjectFilePos = lObjectFilePos + ulObjectSize;

				fwrite_int(eType, output_file);
				fwrite_uint(ulKeyLength, output_file);
				fwrite_int(lKeyFilePos, output_file);
				fwrite_uint(ulObjectSize, output_file);
				fwrite_int(lObjectFilePos, output_file);
				fwrite_int(lNextExtensionObjectFilePos, output_file);

				/* Key */

				fwrite(key, ulKeyLength, 1, output_file);

				fwrite_int(13, output_file);
				strncpy(key, "SOCFixedValue", 256);
				key[13] = 0x00;
				fwrite(key, 13, 1, output_file);

				/* Object */

				fwrite_double(file->channel_data[channel].soc_config.fixed_value, output_file);
			}

			/*
			
				CYLPR1.ChargeAmplifierGain
				CYLPR1.TransducerGain
				CYLPR1.TransducerType
			
			*/
		}
	}

	if (analysis != NULL)
	{
		for (analysis_number = 0; analysis_number < analysis->number_of_channels; analysis_number++)
		{
			if ((file->channel_data[analysis->channel[analysis_number].channel].loaded == true) && (file->channel_data[analysis->channel[analysis_number].channel].abscissa.type != ABSCISSA_TIME))
			{
				for (analysis_channel = 0; analysis_channel < get_number_of_analysis_channels(); analysis_channel++)
				{
					if ((analysis->channel[analysis_number].analysis_rqst[analysis_channel] >= AnalysisRqst::CalculateAndStore) &&
						(((analysis->channel[analysis_number].results[analysis_channel].abscissa.type == ABSCISSA_CRANKANGLE) && (output_ca_analysis != false)) ||
							((analysis->channel[analysis_number].results[analysis_channel].abscissa.type == ABSCISSA_CYCLE) && (output_cyc_results != false))))
					{

						if (strncmp(analysis->channel[analysis_number].results[analysis_channel].name, analysis->channel[analysis_number].results[analysis_channel].short_name, SIZEOF_SHORT_NAME) != 0)
						{
							lLastExtensionObjectFilePos = lNextExtensionObjectFilePos;
							number_of_objects++;

							snprintf(key, 256, "%s.LongName", analysis->channel[analysis_number].results[analysis_channel].short_name);

							eType = 5;
							ulKeyLength = (uint32_t)strlen(key) + 1;
							key[ulKeyLength - 1] = 0x00;
							lKeyFilePos = lNextExtensionObjectFilePos + (4 * 6);
							ulObjectSize = 4 + (uint32_t)strlen(analysis->channel[analysis_number].results[analysis_channel].name) + 1;
							lObjectFilePos = lKeyFilePos + ulKeyLength;
							lNextExtensionObjectFilePos = lObjectFilePos + ulObjectSize;

							/* Data Object */

							fwrite_int(eType, output_file);
							fwrite_uint(ulKeyLength, output_file);
							fwrite_int(lKeyFilePos, output_file);
							fwrite_uint(ulObjectSize, output_file);
							fwrite_int(lObjectFilePos, output_file);
							fwrite_int(lNextExtensionObjectFilePos, output_file);

							/* Key */

							fwrite(key, ulKeyLength, 1, output_file);

							/* Object */

							fwrite_uint(ulObjectSize - 4, output_file);
							fwrite(analysis->channel[analysis_number].results[analysis_channel].name, ulObjectSize - 4, 1, output_file);
						}

						/* catool Analysis Type */

						lLastExtensionObjectFilePos = lNextExtensionObjectFilePos;
						number_of_objects++;

						snprintf(key, 256, "%s", analysis->channel[analysis_number].results[analysis_channel].name);

						char analysis_name[SIZEOF_CHANNEL_NAME];
						strncpy(analysis_name, get_analysis_unique_id(analysis_channel), SIZEOF_CHANNEL_NAME);
						analysis_name[SIZEOF_CHANNEL_NAME - 1] = 0x00;

						uint32_t name_length = (uint32_t)strlen(analysis_name);
						analysis_name[name_length] = 0x00;

						eType = 1002;
						ulKeyLength = (uint32_t)strlen(key) + 1;
						key[ulKeyLength - 1] = 0x00;
						lKeyFilePos = lNextExtensionObjectFilePos + (4 * 6);
						ulObjectSize = 4 + 13 + 4 + name_length + 1;
						lObjectFilePos = lKeyFilePos + ulKeyLength;
						lNextExtensionObjectFilePos = lObjectFilePos + ulObjectSize;

						/* Data Object */

						fwrite_int(eType, output_file);
						fwrite_uint(ulKeyLength, output_file);
						fwrite_int(lKeyFilePos, output_file);
						fwrite_uint(ulObjectSize, output_file);
						fwrite_int(lObjectFilePos, output_file);
						fwrite_int(lNextExtensionObjectFilePos, output_file);

						/* Key */

						fwrite(key, ulKeyLength, 1, output_file);

						fwrite_int(13, output_file);
						strncpy(key, "AnalysisType", 256);
						key[12] = 0x00;
						fwrite(key, 13, 1, output_file);

						/* Object */

						fwrite_uint(name_length + 1, output_file);
						fwrite(analysis_name, name_length + 1, 1, output_file);
					}
				}
			}
		}
	}

	fseek(output_file, pos, SEEK_SET);

	fwrite_uint(number_of_objects, output_file);
	fwrite_int(lFirstExtensionObjectFilePos, output_file);
	fwrite_int(lLastExtensionObjectFilePos, output_file);

	logmessage(NOTICE, "done\n");

	(void)fclose(output_file);

	logmessage(NOTICE, "File Saved\n");

	return(true);
}