/* Combustion Analysis Tool (CAT)
   www.catool.org
   
   Filename: filter.c

   Purpose:  Provide common programming interface for processing acquired engine data
  
   Author:   Ben Brown
   Version:  1.2
   Date:     19.10.2015

   Revision: GPL Release

   Copyright (C) Xarin Limited, 2000-2024

      This program is free software: you can redistribute it and/or modify
      it under the terms of version 2 of the GNU General Public License as
      published by the Free Software Foundation.

      This program is distributed in the hope that it will be useful,
      but WITHOUT ANY WARRANTY; without even the implied warranty of
      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
      GNU General Public License for more details.

      You should have received a copy of the GNU General Public License
      along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <math.h>
#include <stdlib.h>
#include <string.h>

#include "cat.h"

extern unsigned int debug_level;

float median_3pt(float a, float b, float c)
{
	float median;

	if ((a < b) && (a < c))
	{
		if (b < c)
		{
			median = b;
		}
		else
		{
			median = c;
		}
	}
	else if ((b < a) && (b < c))
	{
		if (a < c)
		{
			median = a;
		}
		else
		{
			median = c;
		}
	}
	else
	{
		if (a < b)
		{
			median = a;
		}
		else
		{
			median = b;
		}
	}

	return median;
}

bool filter_channel(FileData* file, const unsigned int channel, const float default_engine_speed)
{
	unsigned int resolution;
	unsigned int cycle;
	float engine_speed = 0.0f;
	float sampling_frequency;
	float* cycle_data = NULL;
	unsigned int mt;
	unsigned int offset;
	unsigned int number_of_samples;
	unsigned int result = 1;
	Filter filter;
	unsigned int number_of_cycles;

	if (file == NULL)
	{
		return(false);
	}

	if (channel >= file->number_of_channels)
	{
		return(false);
	}

	if ((file->channel_data[channel].data == NULL) ||
		(file->channel_data[channel].loaded == false) ||
		(file->channel_data[channel].filter_config.type == FILTER_NONE) ||
		((file->channel_data[channel].abscissa.type != ABSCISSA_CRANKANGLE) && (file->channel_data[channel].abscissa.type != ABSCISSA_TIME)))
	{
		return(false);
	}

	prepare_filter(&filter);

	if (file->channel_data[channel].abscissa.type == ABSCISSA_CRANKANGLE)
	{
		number_of_cycles = file->number_of_cycles;
	}
	else
	{
		number_of_cycles = 1;
	}

	for (cycle = 0; cycle < number_of_cycles; cycle++)
	{
		if (file->channel_data[channel].duration == NULL)
		{
			engine_speed = default_engine_speed;
		}
		else if (file->channel_data[channel].duration[cycle] > 0.0f)
		{
			engine_speed = (float)file->engine.number_of_strokes*30000.0f / file->channel_data[channel].duration[cycle];
		}
		else if (engine_speed < FLT_EPSILON)
		{
			engine_speed = default_engine_speed;
		}
		else
		{
			/* Do Nothing - Use previous good value */
		}

		offset = 0;
		for (mt = 0; mt < file->channel_data[channel].abscissa.number_of_measurement_tables; mt++)
		{
			resolution = file->min_res_ppr / file->channel_data[channel].abscissa.resolution[mt];

			sampling_frequency = (float)resolution * engine_speed / 60.0f;

			cycle_data = &file->channel_data[channel].data[cycle*file->channel_data[channel].samples_per_cycle + offset];

			number_of_samples = file->channel_data[channel].abscissa.number_of_samples[mt];

			offset += number_of_samples;

			if (file->channel_data[channel].filter_config.type == FILTER_FIR_BAND_PASS)
			{
				filter_calculate(number_of_samples,
					FILTER_FIR_LOW_PASS,
					sampling_frequency,
					FFT_WINDOW_HANN,
					file->channel_data[channel].filter_config.upper_frequency,
					file->channel_data[channel].filter_config.upper_frequency,		/* Not used in low-pass filter */
					&filter);

				result *= (int)filter_data(cycle_data, &filter);

				filter_calculate(number_of_samples,
					FILTER_FIR_HIGH_PASS,
					sampling_frequency,
					FFT_WINDOW_HANN,
					file->channel_data[channel].filter_config.lower_frequency,		/* Not used in high pass filter */
					file->channel_data[channel].filter_config.lower_frequency,
					&filter);

				result *= (int)filter_data(cycle_data, &filter);
			}
			else
			{
				filter_calculate(number_of_samples,
					file->channel_data[channel].filter_config.type,
					sampling_frequency,
					FFT_WINDOW_HANN,
					file->channel_data[channel].filter_config.lower_frequency,
					file->channel_data[channel].filter_config.upper_frequency,
					&filter);

				result *= (int)filter_data(cycle_data, &filter);
			}
		}
	}

	finish_filter(&filter);

	file->channel_data[channel].filter_config.filtered = true;

	return(result == 1);
}

/* Ref: http://www.mikroe.com/chapters/view/72/chapter-2-fir-filters/ */

bool prepare_filter(Filter* filter)
{
	if (filter == NULL)
	{
		return(false);
	}

	filter->data = NULL;
	filter->data_size = 0;

	filter->c = NULL;
	filter->c_size = 0;

	filter->h = NULL;
	filter->h_size = 0;

	filter->f = NULL;
	filter->f_size = 0;

	filter->number_of_samples = 0;
	filter->order = 0;
	filter->size = 0.0f;
	filter->type = FILTER_NONE;

	return(true);
}

bool finish_filter(Filter* filter)
{
	if (filter == NULL)
	{
		return(false);
	}

	if (filter->data != NULL)
	{
		free(filter->data);
		filter->data = NULL;
		filter->data_size = 0;
	}

	if (filter->c != NULL)
	{
		free(filter->c);
		filter->c = NULL;
		filter->c_size = 0;
	}

	if (filter->h != NULL)
	{
		free(filter->h);
		filter->h = NULL;
		filter->h_size = 0;
	}

	if (filter->f != NULL)
	{
		free(filter->f);
		filter->f = NULL;
		filter->f_size = 0;
	}

	filter->number_of_samples = 0;
	filter->order = 0;
	filter->size = 0.0f;
	filter->type = FILTER_NONE;

	return(true);
}

unsigned int filter_alloc(float** data, const unsigned int current_data_size, const unsigned int requested_data_size)
{
	if (data == NULL)
	{
		return(0);
	}

	if (requested_data_size <= current_data_size)
	{
		return(current_data_size);
	}

	if ((*data != NULL) && (current_data_size > 0))
	{
		free(*data);
	}

	*data = (float*)malloc(requested_data_size * sizeof(float));
	if (*data == NULL)
	{
		return(0);
	}

	return(requested_data_size);
}

bool filter_calculate(const unsigned int number_of_samples, const unsigned int filter_type, const float sampled_frequency, const unsigned int window_type, float lower_frequency, float upper_frequency, Filter* filter)
{
	unsigned int j;
	float factor = 0.5f;
	float nyquist_frequency;
	float omega_1;
	float omega_2;
	float sum;

	if (sampled_frequency < FLT_EPSILON)
	{
		return(false);
	}

	nyquist_frequency = sampled_frequency / 2.0f;
		
	if (upper_frequency > nyquist_frequency)
	{
		upper_frequency = nyquist_frequency;
	}
		
	if (lower_frequency > nyquist_frequency)
	{
		lower_frequency = nyquist_frequency;		
	}

	switch (filter_type)
	{
	case FILTER_MEAN_LOW_PASS:
	case FILTER_MEAN_HIGH_PASS:
	case FILTER_MEAN_LOW_PASS_SPIKE_ELIM:
	case FILTER_MEAN_HIGH_PASS_SPIKE_ELIM:
	{
		filter->order = 9;

		filter->data_size = filter_alloc(&filter->data, filter->data_size, number_of_samples + filter->order * 2);
#ifdef _SAFE_MEMORY
		if (filter->data_size == 0)
		{
			logmessage(FATAL, "filter_alloc() failed\n");
		}
#endif
		break;
	}
	case FILTER_FIR_LOW_PASS:
	{
		factor = sampled_frequency / lower_frequency;
		
		if (factor < 20.0f)
		{
			filter->order = 10;
		}
		else if (factor > 1000.0f)
		{
			filter->order = 500;
		}
		else
		{
			filter->order = (unsigned int)floor(factor / 2.0f);
		}

		filter->data_size = filter_alloc(&filter->data, filter->data_size, number_of_samples + filter->order * 2);
#ifdef _SAFE_MEMORY
		if (filter->data_size == 0)
		{
			logmessage(FATAL, "filter_alloc() failed\n");
		}
#endif
		filter->c_size = filter_alloc(&filter->c, filter->c_size, filter->order + 1);
#ifdef _SAFE_MEMORY
		if (filter->c_size == 0)
		{
			logmessage(FATAL, "filter_alloc() failed\n");
		}
#endif
		omega_1 = 2.0f*FLT_PI*lower_frequency / sampled_frequency;

		filter->c[0] = 2.0f*lower_frequency / sampled_frequency;

		for (j=1;j<=filter->order;j++)
		{
			filter->c[j] = sinf(omega_1*(float)j) / (FLT_PI*(float)j);
		}

		break;
	}
	case FILTER_FIR_HIGH_PASS:
	{
		factor = sampled_frequency / upper_frequency;

		if (factor < 20.0f)
		{
			filter->order = 10;
		}
		else if (factor > 1000.0f)
		{
			filter->order = 500;
		}
		else
		{
			filter->order = (unsigned int)floor(factor / 2.0f);
		}
		
		filter->data_size = filter_alloc(&filter->data, filter->data_size, number_of_samples + filter->order * 2);
#ifdef _SAFE_MEMORY
		if (filter->data_size == 0)
		{
			logmessage(FATAL, "filter_alloc() failed\n");
		}
#endif
		filter->c_size = filter_alloc(&filter->c, filter->c_size, filter->order + 1);
#ifdef _SAFE_MEMORY
		if (filter->c_size == 0)
		{
			logmessage(FATAL, "filter_alloc() failed\n");
		}
#endif
		omega_1 = 2.0f*FLT_PI*upper_frequency / sampled_frequency;

		filter->c[0] = 1.0f - 2.0f*upper_frequency / sampled_frequency;

		for (j=1;j<=filter->order;j++)
		{
			filter->c[j] = -1.0f*sinf(omega_1*(float)j) / (FLT_PI*(float)j);
		}

		break;
	}		
	case FILTER_FIR_BAND_PASS:
	{
		if (lower_frequency < upper_frequency)
		{
			factor = sampled_frequency / lower_frequency;
		}
		else
		{
			factor = sampled_frequency / upper_frequency;
		}
		
		if (factor < 20.0f)
		{
			filter->order = 10;
		}
		else if (factor > 1000.0f)
		{
			filter->order = 500;
		}
		else
		{
			filter->order = (unsigned int)floor(factor / 2.0f);
		}
		
		filter->data_size = filter_alloc(&filter->data, filter->data_size, number_of_samples + filter->order * 2);
#ifdef _SAFE_MEMORY
		if (filter->data_size == 0)
		{
			logmessage(FATAL, "filter_alloc() failed\n");
		}
#endif
		filter->c_size = filter_alloc(&filter->c, filter->c_size, filter->order + 1);
#ifdef _SAFE_MEMORY
		if (filter->c_size == 0)
		{
			logmessage(FATAL, "filter_alloc() failed\n");
		}
#endif
		omega_1 = 2.0f*FLT_PI*upper_frequency / sampled_frequency;
		omega_2 = 2.0f*FLT_PI*lower_frequency / sampled_frequency;

		filter->c[0] = 2.0f * (upper_frequency / sampled_frequency - lower_frequency / sampled_frequency);

		for (j=1;j<=filter->order;j++)
		{
			filter->c[j] = (sinf(omega_1*(float)j) - sinf(omega_2*(float)j)) / (FLT_PI*(float)j);
		}

		break;
	}
	case FILTER_FIR_BAND_BARRIER:
	{
		if (lower_frequency < upper_frequency)
		{
			factor = sampled_frequency / lower_frequency;
		}
		else
		{
			factor = sampled_frequency / upper_frequency;
		}
		
		if (factor < 20.0f)
		{
			filter->order = 10;
		}
		else if (factor > 1000.0f)
		{
			filter->order = 500;
		}
		else
		{
			filter->order = (unsigned int)floor(factor / 2.0f);
		}
		
		filter->data_size = filter_alloc(&filter->data, filter->data_size, number_of_samples + filter->order * 2);
#ifdef _SAFE_MEMORY
		if (filter->data_size == 0)
		{
			logmessage(FATAL, "filter_alloc() failed\n");
		}
#endif
		filter->c_size = filter_alloc(&filter->c, filter->c_size, filter->order + 1);
#ifdef _SAFE_MEMORY
		if (filter->c_size == 0)
		{
			logmessage(FATAL, "filter_alloc() failed\n");
		}
#endif
		omega_1 = 2.0f*FLT_PI*lower_frequency / sampled_frequency;
		omega_2 = 2.0f*FLT_PI*upper_frequency / sampled_frequency;

		filter->c[0] = 1.0f - 2.0f*(upper_frequency / sampled_frequency - lower_frequency / sampled_frequency);

		for (j=1;j<=filter->order;j++)
		{
			filter->c[j] = (sinf(omega_1*(float)j) - sinf(omega_2*(float)j)) / (FLT_PI*(float)j);
		}

		break;
	}
	case FILTER_MEDIAN:
	{
		filter->data_size = filter_alloc(&filter->data, filter->data_size, number_of_samples);
#ifdef _SAFE_MEMORY
		if (filter->data_size == 0)
		{
			logmessage(FATAL, "filter_alloc() failed\n");
		}
#endif
		filter->order = 1;

		break;
	}
	case FILTER_NONE:
	default:
	{
		filter->order = 0;

		break;
	}
	}

	switch (filter_type)
	{
	case FILTER_FIR_LOW_PASS:
	case FILTER_FIR_HIGH_PASS:
	case FILTER_FIR_BAND_PASS:
	case FILTER_FIR_BAND_BARRIER:
	{
		filter->h_size = filter_alloc(&filter->h, filter->h_size, filter->order * 2 + 1);
		filter->f_size = filter_alloc(&filter->f, filter->f_size, filter->order + 1);

		calculate_window(filter->h, filter->order * 2 + 1, window_type);

		/* calculate_window() produces a two-tailed window for FFT data, we need to only use the right hand half hence the addition of filter_order below */

		for (j = 0; j <= filter->order; j++)
		{
			filter->f[j] = filter->h[filter->order + j] * filter->c[j];
		}

		sum = filter->f[0];
		for (j = 1; j <= filter->order; j++)
		{
			sum += 2 * filter->f[j];
		}

		if (filter_type == FILTER_FIR_LOW_PASS)
		{
			for (j = 0; j <= filter->order; j++)
			{
				filter->f[j] /= sum;
			}
		}
		else if (filter_type == FILTER_FIR_HIGH_PASS)
		{
			filter->f[0] -= sum;
		}
		else
		{
			/* Do Nothing */
		}

		break;
	}
	default:
	{
		/* Do Nothing */

		break;
	}
	}

	filter->size = 1.0f + 2.0f * (float)filter->order;
	filter->number_of_samples = number_of_samples;
	filter->type = filter_type;

	return(true);
}

bool filter_data(float* data, Filter* filter)
{
	float filtered_value;
	unsigned int data_pointer;
	unsigned int j;
	unsigned int finish;

	switch (filter->type)
	{
	case FILTER_MEAN_LOW_PASS:
	case FILTER_MEAN_HIGH_PASS:
	case FILTER_MEAN_LOW_PASS_SPIKE_ELIM:
	case FILTER_MEAN_HIGH_PASS_SPIKE_ELIM:
	{
		/* Copy original data to temporary store */

		for (data_pointer = 0; data_pointer < filter->order; data_pointer++)
		{
			filter->data[data_pointer] = data[0];
		}

		memcpy(&filter->data[filter->order], data, filter->number_of_samples * sizeof(float));

		finish = filter->number_of_samples + 2 * filter->order;

		for (data_pointer = filter->number_of_samples + filter->order; data_pointer < finish; data_pointer++)
		{
			filter->data[data_pointer] = data[filter->number_of_samples - 1];
		}

		/* Filter data */

		finish = filter->number_of_samples + filter->order;

		for (data_pointer = filter->order; data_pointer < finish; data_pointer++)
		{
			filtered_value = filter->data[data_pointer];

			for (j = 1; j <= filter->order; j++)
			{
				filtered_value += filter->data[data_pointer - j];
				filtered_value += filter->data[data_pointer + j];
			}

			data[data_pointer - filter->order] = filtered_value * filter->size;
		}

		break;
	}
	case FILTER_FIR_LOW_PASS:
	case FILTER_FIR_HIGH_PASS:
	case FILTER_FIR_BAND_PASS:
	case FILTER_FIR_BAND_BARRIER:
	{
		/* Copy original data to temporary store */
		
		for (data_pointer = 0; data_pointer<filter->order; data_pointer++)
		{
			filter->data[data_pointer] = data[0];
		}
		
		memcpy(&filter->data[filter->order], data, filter->number_of_samples * sizeof(float));

		finish = filter->number_of_samples + 2 * filter->order;

		for (data_pointer = filter->number_of_samples + filter->order; data_pointer < finish; data_pointer++)
		{
			filter->data[data_pointer] = data[filter->number_of_samples - 1];
		}

		/* Filter data */

		finish = filter->number_of_samples + filter->order;

		for (data_pointer = filter->order; data_pointer < finish; data_pointer++)
		{
			filtered_value = filter->data[data_pointer] * filter->f[0];

			for (j = 1; j <= filter->order; j++)
			{
				filtered_value += filter->data[data_pointer - j] * filter->f[j];
				filtered_value += filter->data[data_pointer + j] * filter->f[j];
			}

			data[data_pointer - filter->order] = filtered_value;
		}

		break;
	}
	case FILTER_MEDIAN:
	{
		memcpy(filter->data, data, filter->number_of_samples * sizeof(float));

		finish = filter->number_of_samples - 1;

		for (data_pointer = 1; data_pointer<finish; data_pointer++)
		{
			data[data_pointer] = median_3pt(filter->data[data_pointer - 1], filter->data[data_pointer], filter->data[data_pointer + 1]);
		}

		data[0] = data[1];
		data[filter->number_of_samples - 1] = data[filter->number_of_samples - 2];

		break;
	}
	case FILTER_NONE:
	default:
	{
		break;
	}
	}

	return(true);
}

bool calculate_window(float* coefficient, const unsigned int N, const unsigned int window_type)
{
	unsigned int sample;
	bool result = true;
	float n_minus_one = (float)N - 1.0f;

	if ((coefficient == NULL) || (N <= 1))
	{
		return(false);
	}

	switch (window_type)
	{
	case FFT_WINDOW_HANN:
	{
		for (sample = 0; sample<N; sample++)
		{
			coefficient[sample] = 0.5f * (1.0f - cosf((2.0f * FLT_PI * (float)sample) / n_minus_one));
		}

		break;
	}
	case FFT_WINDOW_HAMMING:
	{
		for (sample = 0; sample<N; sample++)
		{
			coefficient[sample] = 0.54f - 0.46f * cosf((2.0f * FLT_PI * (float)sample) / n_minus_one);
		}

		break;
	}
	case FFT_WINDOW_RECTANGULAR:
	default:
	{
		for (sample = 0; sample<N; sample++)
		{
			coefficient[sample] = 1.0f;
		}

		break;
	}
	case FFT_WINDOW_TUKEY:
	case FFT_WINDOW_COSINE:
	case FFT_WINDOW_LANCZOS:
	case FFT_WINDOW_TRIANGULAR:
	case FFT_WINDOW_BARTLETT:
	case FFT_WINDOW_BARTLETT_HANN:
	case FFT_WINDOW_BLACKMAN:
	case FFT_WINDOW_KAISER:
	case FFT_WINDOW_NUTTALL:
	case FFT_WINDOW_BLACKMAN_HARRIS:
	case FFT_WINDOW_BLACKMAN_NUTTALL:
	case FFT_WINDOW_FLATTOP:
	{
		for (sample = 0; sample<N; sample++)
		{
			coefficient[sample] = 1.0f;
		}

		logmessage(WARNING, "Filter window type not supported\n");

		result = false;

		break;
	}
	}

	return(result);
}