/* Combustion Analysis Tool (CAT)
   www.catool.org
   
   Filename: fileprocessor.c

   Purpose:  Provide common programming interface for processing acquired engine data
  
   Author:   Ben Brown
   Version:  1.2
   Date:     19.10.2015

   Revision: GPL Release

   Copyright (C) Xarin Limited, 2000-2024

      This program is free software: you can redistribute it and/or modify
      it under the terms of version 2 of the GNU General Public License as
      published by the Free Software Foundation.

      This program is distributed in the hope that it will be useful,
      but WITHOUT ANY WARRANTY; without even the implied warranty of
      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
      GNU General Public License for more details.

      You should have received a copy of the GNU General Public License
      along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <stdlib.h>
#include <string.h>
#include <float.h>
#include <locale.h>

#if defined(_WIN32) || defined(_WIN64)
#include <Windows.h>
#endif

#include "cat.h"

extern unsigned int debug_level;

unsigned int set_double_parameter(const char* parameter,const char* value,const char* parameter_name,double* parameter_variable)
{
	double value_double;
	unsigned int result = 0;

	if ((parameter == NULL) || (value == NULL) || (parameter_name == NULL) || (parameter_variable == NULL))
	{
		return(0);
	}

	if (case_insensitive_compare(parameter, parameter_name, MAX_LINE_SIZE) == 1)
	{
		result = sscanf(value, "%20lf", &value_double);

		if (result == 1)
		{
			*(parameter_variable) = value_double;
		}
		else
		{
			result = 2;
		}
	}

	return(result);
}

unsigned int set_float_parameter(const char* parameter,const char* value,const char* parameter_name,float* parameter_variable)
{
	float value_float;
	unsigned int result = 0;

	if ((parameter == NULL) || (value == NULL) || (parameter_name == NULL) || (parameter_variable == NULL))
	{
		return(0);
	}

	if (case_insensitive_compare(parameter, parameter_name, MAX_LINE_SIZE) == 1)
	{
		result = sscanf(value, "%20f", &value_float);

		if (result == 1)
		{
			*(parameter_variable) = value_float;
		}
		else
		{
			result = 2;
		}
	}

	return(result);
}

unsigned int set_uint_parameter(const char* parameter,const char* value,const char* parameter_name,unsigned int* parameter_variable)
{
	unsigned int value_uint;
	unsigned int result = 0;
 	
	if ((parameter == NULL) || (value == NULL) || (parameter_name == NULL) || (parameter_variable == NULL))
	{
		return(0);
	}

	if (case_insensitive_compare(parameter,parameter_name,MAX_LINE_SIZE) == 1)
	{
		result = sscanf(value,"%6u",&value_uint);
         
		if (result == 1)
		{
			*(parameter_variable) = value_uint;
		}
	    else
	    {
			result = 2;
		}
	}
	
	return(result);
}

void set_analysis(const char* parameter, AnalysisRqst* analysis_rqst)
{
	unsigned int analysis_type;
	char negative_analysis_name[SIZEOF_CHANNEL_NAME+1];
	
	if ((parameter == NULL) || (analysis_rqst == NULL))
	{
		return;
	}

	for (analysis_type = 0; analysis_type < get_number_of_analysis_channels(); analysis_type++)
	{
		snprintf(negative_analysis_name, SIZEOF_CHANNEL_NAME, "-%s", get_analysis_unique_id(analysis_type));
		negative_analysis_name[SIZEOF_CHANNEL_NAME] = 0x00;

		if (case_insensitive_compare(get_analysis_unique_id(analysis_type), parameter, MAX_LINE_SIZE) == 1)
		{
			analysis_rqst[analysis_type] = AnalysisRqst::CalculateAndStore;
			return;
		}
		else if (case_insensitive_compare(negative_analysis_name, parameter, MAX_LINE_SIZE) == 1)
		{
			analysis_rqst[analysis_type] = AnalysisRqst::NotCalculated;
			return;
		}
		else
		{
			/* Do Nothing */
		}
	}
}

void analyse_channels(FileData* file, Analysis* analysis, const bool error_check)
{
	unsigned int analysis_structure;
	unsigned int channel;
	cycle_analysis_arguments* cycle_analysis_configuration = NULL;
	int thread = 0;
	int t;

	if ((file == NULL) || (analysis == NULL))
	{
		return;
	}

#if defined(_USE_PTHREADS)
	pthread_t* threads = NULL;
	pthread_attr_t attr;
	int rc;
#elif defined(_USE_STD_THREAD)
	std::thread* threads = NULL;
#else
	void* threads = NULL;
	file->threads = 0;
#endif

	logmessage(DEBUG,"Configuring analysis configuration\n");

	if (file->threads > 1)
	{
#if defined(_USE_PTHREADS)
		threads = (pthread_t*)malloc(file->number_of_channels * sizeof(pthread_t));
#elif defined (_USE_STD_THREAD)
		threads = new std::thread[file->number_of_channels];
#endif
		if (threads == NULL)
		{
			logmessage(FATAL,"Memory could not be allocated\n");
		}

		cycle_analysis_configuration = (cycle_analysis_arguments*)malloc(file->number_of_channels * sizeof(cycle_analysis_arguments));
		if (cycle_analysis_configuration == NULL)
		{
			logmessage(FATAL,"Memory could not be allocated\n");
		}

#if defined(_USE_PTHREADS)
		pthread_attr_init(&attr);
		pthread_attr_setdetachstate(&attr,PTHREAD_CREATE_JOINABLE);
		pthread_attr_setstacksize (&attr, THREAD_STACKSIZE);
#endif
	}
	else
	{
		cycle_analysis_configuration = (cycle_analysis_arguments*)malloc(sizeof(cycle_analysis_arguments));
		if (cycle_analysis_configuration == NULL)
		{
			logmessage(FATAL,"Memory could not be allocated\n");
			exit(EXIT_FAILURE);
		}						
	}	
	
	/* Run analysis */

	logmessage(DEBUG,"Running analysis...\n");

	thread = 0;
	analysis_structure = 0;

	for (channel=0;channel<file->number_of_channels;channel++)
	{
		if ((file->channel_data[channel].file_flag == true) && 
			(file->channel_data[channel].loaded == true))
		{ 
			if (file->channel_data[channel].type != CHAN_CYL_PR)
			{
				logmessage(NOTICE, "Running analysis for channel %s (Stage 1)\n", file->channel_data[channel].name);

				cycle_analysis_configuration[thread].file = file;
				cycle_analysis_configuration[thread].analysis = analysis;
				cycle_analysis_configuration[thread].channel = analysis_structure;
				cycle_analysis_configuration[thread].error_check = error_check;
				cycle_analysis_configuration[thread].status.min = 0.0f;
				cycle_analysis_configuration[thread].status.max = 100.0f;
				cycle_analysis_configuration[thread].status.ptr[0] = NULL;
				cycle_analysis_configuration[thread].status.ptr[1] = NULL;
				cycle_analysis_configuration[thread].thread_status = THREAD_STATUS_INITIALISING;

				if (file->threads > 1)
				{
#if defined(_USE_PTHREADS)
					rc = pthread_create(&threads[thread],&attr,cycle_analysis,(void*)&cycle_analysis_configuration[thread]);
					if (rc == 1)
					{
						logmessage(FATAL,"Thread could not be created\n");				
					}
#elif defined (_USE_STD_THREAD)
					threads[thread] = std::thread(cycle_analysis, (void*)&cycle_analysis_configuration[thread]);
#endif
					thread += 1;
				}
				else
				{
					cycle_analysis(&cycle_analysis_configuration[thread]);
				}
			}

			analysis_structure += 1;
		}
	}

#if defined(_USE_PTHREADS)
	if (file->threads > 1)
	{
		pthread_attr_destroy(&attr);

		for(t=0;t<thread;t++)
		{
			rc = pthread_join(threads[t],NULL);
			if (rc == 1)
			{
				logmessage(FATAL,"Thread could not be rejoined\n");
			}
		}
	}
#elif defined(_USE_STD_THREAD)
	if (file->threads > 1)
	{
		for (t = 0; t < thread; t++)
		{
			threads[t].join();
		}
	}
#endif

	thread = 0;
	analysis_structure = 0;

	for (channel=0;channel<file->number_of_channels;channel++)
	{
		if ((file->channel_data[channel].file_flag == true) && 
			(file->channel_data[channel].loaded == true))
		{
			if (file->channel_data[channel].type == CHAN_CYL_PR)
			{
				logmessage(NOTICE, "Running analysis for channel %s (Stage 2)\n", file->channel_data[channel].name);

				cycle_analysis_configuration[thread].file = file;
				cycle_analysis_configuration[thread].analysis = analysis;
				cycle_analysis_configuration[thread].channel = analysis_structure;
				cycle_analysis_configuration[thread].error_check = error_check;
				cycle_analysis_configuration[thread].status.min = 0.0f;
				cycle_analysis_configuration[thread].status.max = 100.0f;
				cycle_analysis_configuration[thread].status.ptr[0] = NULL;
				cycle_analysis_configuration[thread].status.ptr[1] = NULL;
				cycle_analysis_configuration[thread].thread_status = THREAD_STATUS_INITIALISING;


				if (file->threads > 1)
				{
#if defined(_USE_PTHREADS)
					rc = pthread_create(&threads[thread],&attr,cycle_analysis,(void*)&cycle_analysis_configuration[thread]);

					if (rc == 1)
					{
						logmessage(FATAL,"Thread could not be created\n");				
					}
#elif defined(_USE_STD_THREAD)
					threads[thread] = std::thread(cycle_analysis, (void*)&cycle_analysis_configuration[thread]);
#endif
					thread += 1;
				}
				else
				{
					cycle_analysis(&cycle_analysis_configuration[thread]);
				}
			}

			analysis_structure += 1;
		}
	}

#if defined(_USE_PTHREADS)
	if (file->threads > 1)
	{
		logmessage(DEBUG,"Waiting to rejoin threads...\n");

		pthread_attr_destroy(&attr);

		for(t=0;t<thread;t++)
		{
			rc = pthread_join(threads[t],NULL);
			if (rc == 1)
			{
				logmessage(FATAL,"Thread could not be rejoined\n");
			}
		}

		free(threads);
		threads = NULL;

		logmessage(DEBUG,"Threads rejoined\n");
	}
#elif defined(_USE_STD_THREAD)
	if (file->threads > 1)
	{
		for (t = 0; t < thread; t++)
		{
			threads[t].join();
		}

		delete[] threads;
		threads = NULL;
	}
#endif

	free(cycle_analysis_configuration);
	cycle_analysis_configuration = NULL;
}

void Copy_AbscissaBlocks(const AbscissaBlocks* src, AbscissaBlocks* dest)
{
	unsigned int block;

	if ((src == NULL) || (dest == NULL))
	{
		return;
	}

	dest->number_of_blocks = src->number_of_blocks;

	for (block = 0; block < 5; block++)
	{
		dest->start[block] = src->start[block];
		dest->finish[block] = src->finish[block];
		dest->resolution[block] = src->resolution[block];
	}
}

bool Initialise_Channel_Data(FileData* file, const AbscissaBlocks* channel_blocks, const unsigned int channel)
{
	AbscissaData abscissa;

	if (file == NULL)
	{
		return(false);
	}

	if (channel_blocks == NULL)
	{
		return(false);
	}

	if ((channel >= file->number_of_channels) || (file->number_of_channels == 0))
	{
		return(false);
	}

	if (channel_blocks[channel].number_of_blocks == 0)
	{
		return(false);
	}

	Zero_Abscissa(&abscissa);

	if (Build_Abscissa(file, &channel_blocks[channel], &abscissa) == true)
	{
		Empty_Abscissa(&file->channel_data[channel].abscissa);

		file->channel_data[channel].samples_per_cycle = Copy_Abscissa(&abscissa, &file->channel_data[channel].abscissa);

		if (calculate_theta_lookups(file, channel) == false)
		{
			logmessage(WARNING, "Failed to calculate theta lookups\n");
		}
		else
		{
			return(true);
		}
	}
	else
	{
		logmessage(WARNING, "Failed to build suitable abscissa\n");
	}

	return(false);
}

#ifdef _CATOOL_UNICODE_
bool load_config_file(FileData* file,const wchar_t* filename,const bool debug_level_fixed)
#else
bool load_config_file(FileData* file, const char* filename, const bool debug_level_fixed)
#endif
{
	Analysis* analysis = NULL;
	FILE* input_file = NULL;
	char input_line[MAX_LINE_SIZE];
#ifdef _CATOOL_UNICODE_
	wchar_t input_filename[4096];
	wchar_t config_filename[4096];
	wchar_t output_filename[4096];
#else
	char input_filename[4096];
	char config_filename[4096];
	char output_filename[4096];
#endif
	char channel_name[256];
	unsigned int channel;
	unsigned int output_file_type = FILE_CSV;
	unsigned int rqst;
	unsigned int number_of_analysis_structures = 0;
	unsigned int analysis_structure;
	unsigned int argument;
	unsigned int number_of_arguments;
	char** output = NULL;
	AnalysisRqst* analysis_rqst = NULL;
	unsigned int abscissa_type = ABSCISSA_UNKNOWN;
	unsigned int number_of_lines_to_skip = 0;
	float recording_length = 0.0f;
	bool done = false;
	AnalysisConfig analysis_config;
	unsigned int resolution = 1;
	unsigned int output_data = IFILE_RESULTS;
	unsigned int ifile_input = IFILE_CA | IFILE_RESULTS;
	unsigned int parameter_number = 0;
	int result;
	unsigned int cylinder;
	float pin_offset;
	unsigned int chan;
	unsigned int start_channel;
	unsigned int finish_channel;
	AbscissaBlocks* channel_blocks = NULL;
	bool loaded;
	unsigned int block;
	AbscissaData abscissa;
	bool process_all;
	unsigned int number_of_cycles;
	unsigned int* cycles_to_output = NULL;
	unsigned int cycle;
	unsigned int number_of_new_cycles;
	bool error_check = false;
	unsigned int number_of_channels;
	bool configured_ok;
	bool align;
	AbscissaBlocks* new_channel_blocks = NULL;
	bool interpolate;
	float frequency;
	float start;
	float finish;
	bool ca_to_time_warning;
	unsigned int channel_names_line = 0;
	unsigned int channel_units_line = 0;
	bool skip_columns[MAX_NUMBER_OF_COLUMNS];
	unsigned int column;
	unsigned int grpflg_float = IFILE_RESULTS | IFILE_RESULTS_RAW;
	unsigned int calc_interval;

	if (file == NULL)
	{
		return(false);
	}

	Zero_Abscissa(&abscissa);

#ifdef _CATOOL_UNICODE_
	wcsncpy(config_filename, filename, 4095);

	config_filename[4095] = 0x0000;
	output_filename[0] = 0x0000;
	input_filename[0] = 0x0000;

	/* Open file */

	if (config_filename[0] == L'-')
	{
		input_file = stdin;
	}
	else
	{
		input_file = _wfopen(config_filename, L"r");
	}
#else
	strncpy(config_filename,filename,4096);

	config_filename[4095] = 0x00;
	output_filename[0] = 0x00;
	input_filename[0] = 0x00;

	/* Open file */

	if (config_filename[0] == '-')
	{
		input_file = stdin;
	}
	else
	{
		input_file = fopen(config_filename, "r");
	}
#endif

	if (input_file == NULL)
	{
		logmessage(MSG_ERROR, "Configuration file could not be opened\n");
		return(false);
	}

	/* Initialise analysis */
	
	output = (char**)malloc(MAX_NUMBER_OF_COLUMNS * sizeof(char*));
	if (output == NULL)
	{
		logmessage(FATAL,"Memory could not be allocated\n");
	}
		
	for (argument=0;argument<MAX_NUMBER_OF_COLUMNS;argument++)
	{
		output[argument] = (char*)malloc(MAX_LINE_SIZE * sizeof(char));
		if (output[argument] == NULL)
		{
			logmessage(FATAL,"Memory could not be allocated\n");
		}
	}
	
	analysis_rqst = (AnalysisRqst*)malloc(get_number_of_analysis_channels() * sizeof(AnalysisRqst));

	if (analysis_rqst == NULL)
	{
		logmessage(FATAL, "Out of memory\n");
	}

	for (rqst=0;rqst<get_number_of_analysis_channels();rqst++)
	{
		analysis_rqst[rqst] = AnalysisRqst::NotCalculated;
	}

	for (column = 0; column < MAX_NUMBER_OF_COLUMNS; column++)
	{
		skip_columns[column] = false;
	}

	Initialise_Analysis(&analysis_config);
    
	/* Read and Process lines */
    
    done = false;

	while ((done == false) && (fgets(input_line, MAX_LINE_SIZE, input_file) != NULL))
	{
		input_line[MAX_LINE_SIZE - 1] = '\0';

		read_line(input_line, output, MAX_NUMBER_OF_COLUMNS, MAX_LINE_SIZE, &number_of_arguments, SEPARATOR_SPACE | SEPARATOR_TAB, 0x00);

		if (case_insensitive_compare(output[0], "locale", MAX_LINE_SIZE) == 1)
		{
			setlocale(LC_ALL, output[1]);
		}
		else if (case_insensitive_compare(output[0],"debug-level",MAX_LINE_SIZE) == 1)
	  	{
			if (debug_level_fixed == true)
			{
				logmessage(NOTICE,"Debug level cannot be changed if it has been passed as a command line parameter\n");
			}
			else if (set_uint_parameter(output[0],output[1],"debug-level",&debug_level) == 2)
			{
				if (case_insensitive_compare(output[1],"DEBUG",MAX_LINE_SIZE) == 1)
				{
					debug_level = DEBUG;
				}
	 			else if (case_insensitive_compare(output[1],"NOTICE",MAX_LINE_SIZE) == 1)
	 			{
					debug_level = NOTICE;
				}	 
				else if (case_insensitive_compare(output[1],"WARNING",MAX_LINE_SIZE) == 1)
				{
					debug_level = WARNING;
				}
				else if (case_insensitive_compare(output[1],"ERROR",MAX_LINE_SIZE) == 1)
				{
					debug_level = MSG_ERROR;
				}
				else if (case_insensitive_compare(output[1],"FATAL",MAX_LINE_SIZE) == 1)
				{
					debug_level = FATAL;
				}
				else if (case_insensitive_compare(output[1],"SILENT",MAX_LINE_SIZE) == 1)
				{
					debug_level = SILENT;
				}
				else
				{
					logmessage(WARNING,"Unknown debug level '%s'\n",output[1]);
				}
			}
					
			logmessage(NOTICE,"Debug level: %s\n",output[1]);
		}
		else if (case_insensitive_compare(output[0],"input-file-type",MAX_LINE_SIZE) == 1)
		{
			logmessage(NOTICE,"Set input filetype to %s\n",output[1]);
				
			if (set_uint_parameter(output[0],output[1],"input-file-type",&file->file_type) == 2)
			{
				if ((case_insensitive_compare(output[1],"AVL_IFILE",MAX_LINE_SIZE) == 1) || 
				    (case_insensitive_compare(output[1],"AVL-IFILE",MAX_LINE_SIZE) == 1) ||
					(case_insensitive_compare(output[1],"AVL",MAX_LINE_SIZE) == 1) ||
					(case_insensitive_compare(output[1],"IFILE",MAX_LINE_SIZE) == 1))
				{
					file->file_type = FILE_AVLIFILE;
					if (analysis_config.engine_speed_type == N_NOTHING)
					{
						analysis_config.engine_speed_type = N_AVLRZT;
					}
				}
	 			else if (case_insensitive_compare(output[1],"CSV",MAX_LINE_SIZE) == 1)
	 			{
					file->file_type = FILE_CSV;
				}	 
				else
				{
					logmessage(WARNING,"Unknown input filetype '%s'\n",output[1]);
				}
			}
		}
		else if (case_insensitive_compare(output[0],"input-file",MAX_LINE_SIZE) == 1)
		{
#ifdef _CATOOL_UNICODE_
			size_t newsize = strlen(output[1]) + 1;
			wchar_t* wcstring = new wchar_t[newsize];
			size_t convertedChars = 0;
			mbstowcs_s(&convertedChars, wcstring, newsize, output[1], _TRUNCATE);

			wcsncpy(input_filename,wcstring,4095);
			delete[] wcstring;

			input_filename[4095] = 0x0000;
#else
			strncpy(input_filename, output[1], 4095);
			input_filename[4095] = 0x00;
#endif
			
			logmessage(NOTICE,"Set input filename to %s\n",input_filename);	
			
			if (file->preload == true)
			{
				Delete_File(file);

				file->preload = false;
			}

			if (analysis != NULL)
			{
				logmessage(NOTICE, "Deleting %u old analysis structures\n", number_of_analysis_structures);

				Delete_Analysis(analysis);
				free(analysis);
				analysis = NULL;

				logmessage(DEBUG, "Analysis structures deleted\n");
			}
   			
		    switch (file->file_type)
		    {
		    case FILE_AVLIFILE:
		    {
				loaded = load_ifile_header(file, input_filename, ifile_input);

		        break;
		    }
		    case FILE_CSV:
			{
				loaded = load_csv_file_header(file, input_filename, NULL, &resolution, abscissa_type, &recording_length, &number_of_lines_to_skip,NULL);

				break;
			}
		    case FILE_CAS:
			case FILE_TDMS:
			case FILE_YOKOGAWA:
		    default:
		    {
		        logmessage(MSG_ERROR,"Input filetype not currently supported\n");

				loaded = false;

		        break;
		    }
		    }

			if (loaded == true)
			{
				if (channel_blocks != NULL)
				{
					free(channel_blocks);
					channel_blocks = NULL;
				}

				channel_blocks = (AbscissaBlocks*)malloc(file->number_of_channels*sizeof(AbscissaBlocks));

				if (channel_blocks == NULL)
				{
					logmessage(FATAL, "Memory could not be allocated\n");
				}

				for (channel = 0; channel < file->number_of_channels; channel++)
				{
					channel_blocks[channel].number_of_blocks = 0;
				}
			}
		}
		else if (case_insensitive_compare(output[0], "error-check", MAX_LINE_SIZE) == 1)
		{
			if (case_insensitive_compare(output[1], "on", MAX_LINE_SIZE) == 1)
			{
				error_check = true;
			}
			else if (case_insensitive_compare(output[1], "off", MAX_LINE_SIZE) == 1)
			{
				error_check = false;
			}
			else
			{
				logmessage(WARNING, "Syntax: error-check <on|off>\n");
			}
		}
		else if (case_insensitive_compare(output[0],"input-abscissa-type",MAX_LINE_SIZE) == 1)
		{
			logmessage(NOTICE,"Set input abscissa type to %s\n",output[1]);
				
			if (set_uint_parameter(output[0],output[1],"input-abscissa-type",&abscissa_type) == 2)
			{
				if ((case_insensitive_compare(output[1],"CRANK_ANGLE",MAX_LINE_SIZE) == 1) ||
					(case_insensitive_compare(output[1],"CRANKANGLE",MAX_LINE_SIZE) == 1))
				{
					abscissa_type = ABSCISSA_CRANKANGLE;
				}
	 			else if (case_insensitive_compare(output[1],"CYCLE",MAX_LINE_SIZE) == 1)
	 			{
					abscissa_type = ABSCISSA_CYCLE;
				}
				else if (case_insensitive_compare(output[1],"TIME",MAX_LINE_SIZE) == 1)
				{
					abscissa_type = ABSCISSA_TIME;
				}
				else
				{
					logmessage(WARNING,"Unknown input abscissa type '%s'\n",output[1]);		
				}
			}
		}
		else if (case_insensitive_compare(output[0],"number-of-channels",MAX_LINE_SIZE) == 1)
		{
			if (set_uint_parameter(output[0],output[1],"number-of-channels",&number_of_channels) == 1)
			{
				if (file->number_of_channels > 0)
				{
					for (channel = 0; channel<file->number_of_channels; channel++)
					{
						Delete_Channel(file, channel);
					}
				}

				file->number_of_channels = number_of_channels;

				if (channel_blocks != NULL)
				{
					free(channel_blocks);
				}

				channel_blocks = (AbscissaBlocks*)malloc(file->number_of_channels*sizeof(AbscissaBlocks));

				if (channel_blocks == NULL)
				{
					logmessage(FATAL, "Memory could not be allocated\n");
				}

				for (channel = 0; channel < file->number_of_channels; channel++)
				{
					channel_blocks[channel].number_of_blocks = 0;
				}

				logmessage(NOTICE,"Set number of channels to %u\n",file->number_of_channels);
			}
			else
			{
				logmessage(WARNING,"Syntax: number-of-channels <n>\n");
			}
		}
		else if (case_insensitive_compare(output[0],"number-of-cycles",MAX_LINE_SIZE) == 1)
		{
			if (set_uint_parameter(output[0],output[1],"number-of-cycles",&number_of_cycles) == 1)
			{
				if (number_of_cycles > 0)
				{
					if (file->preload == true)
					{
						/* Resize channels */

						Resize_Number_Of_Cycles(file, analysis, number_of_cycles);
					}
					else
					{
						file->number_of_cycles = number_of_cycles;
					}

					logmessage(NOTICE, "Set number of cycles to %u\n", file->number_of_cycles);
				}
				else
				{
					logmessage(WARNING, "Number of cycles must be greater than zero\n");
				}
			}
			else
			{
				logmessage(WARNING,"Syntax: number-of-cycles <n>\n");
			}
		}
		else if (case_insensitive_compare(output[0],"resolution",MAX_LINE_SIZE) == 1)
		{
			if (set_uint_parameter(output[0],output[1],"resolution",&resolution) == 1)
			{
				logmessage(NOTICE,"Set resolution to %u samples per degree/second\n",resolution);
			}
			else
			{
				logmessage(NOTICE,"Syntax: resolution <n>\n");
			}
		}
		else if (case_insensitive_compare(output[0],"recording-length",MAX_LINE_SIZE) == 1)
		{
			if (set_float_parameter(output[0],output[1],"recording-length",&recording_length) == 1)
			{
				logmessage(NOTICE,"Set recording length to %f seconds\n",recording_length);
			}
			else
			{
				logmessage(NOTICE,"Syntax: recording-length <n>\n");
			}
		}
		else if (case_insensitive_compare(output[0],"header-lines",MAX_LINE_SIZE) == 1)
		{
			if (set_uint_parameter(output[0],output[1],"header-lines",&number_of_lines_to_skip) == 1)
			{
				logmessage(NOTICE,"Set number of lines to skip to %u\n",number_of_lines_to_skip);
			}
			else
			{
				logmessage(NOTICE,"Syntax: header-lines <n>\n");
			}
		}
		else if (case_insensitive_compare(output[0], "skip-columns", MAX_LINE_SIZE) == 1)
		{
			if (number_of_arguments > 1)
			{
				for (argument = 1; argument < number_of_arguments; argument++)
				{
					if ((sscanf(output[argument], "%u", &column) == 1) && (column > 0) && (column <= MAX_NUMBER_OF_COLUMNS))
					{
						skip_columns[column-1] = true;

						logmessage(NOTICE, "Skipping column %u\n", column);
					}
				}
			}
		}
		else if (case_insensitive_compare(output[0], "reset-columns", MAX_LINE_SIZE) == 1)
		{
			for (column = 0; column < MAX_NUMBER_OF_COLUMNS; column++)
			{
				skip_columns[column] = false;
			}
		}
		else if (case_insensitive_compare(output[0], "channel-names-line", MAX_LINE_SIZE) == 1)
		{
			if (set_uint_parameter(output[0], output[1], "channel-names-line", &channel_names_line) == 1)
			{
				logmessage(NOTICE, "Set channel names line to %u\n", channel_names_line);
			}
			else
			{
				logmessage(NOTICE, "Syntax: channel-names-line <n>\n");
			}
		}
		else if (case_insensitive_compare(output[0], "channel-units-line", MAX_LINE_SIZE) == 1)
		{
			if (set_uint_parameter(output[0], output[1], "channel-units-line", &channel_units_line) == 1)
			{
				logmessage(NOTICE, "Set channel units line to %u\n", channel_units_line);
			}
			else
			{
				logmessage(NOTICE, "Syntax: channel-units-line <n>\n");
			}
		}
		else if (case_insensitive_compare(output[0],"load-channels",MAX_LINE_SIZE) == 1)
		{ 
		   	if (file->preload == false)
		   	{
 				logmessage(MSG_ERROR,"Specify the file to load with 'input-file' before loading channels\n");
			}
			else
			{
				if (number_of_arguments > 1)
				{
					for (argument=1;argument<number_of_arguments;argument++)
					{
						if (case_insensitive_compare(output[argument],"all",MAX_LINE_SIZE) == 1)
						{
							for (channel=0;channel<file->number_of_channels;channel++)
							{
						 		file->channel_data[channel].file_flag = true;
							}
						}
					 	else if (case_insensitive_compare(output[argument],"none",MAX_LINE_SIZE) == 1)
					 	{
							for (channel=0;channel<file->number_of_channels;channel++)
							{
						 		file->channel_data[channel].file_flag = false;
							}
						}
						else if (output[argument][0] == '-')
						{
							result = sscanf(&output[argument][1], "%255s", (char*)&channel_name);

							if ((result > 0) && (channel_name_to_number(file, channel_name, ABSCISSA_UNKNOWN, &channel) == true))
							{
								if (channel < file->number_of_channels)
								{
									file->channel_data[channel].file_flag = false;
								}
								else
								{
									logmessage(MSG_ERROR, "Unknown channel '%s'\n", output[argument][1]);
								}
							}
							else
							{
								result = sscanf(&output[argument][1], "#%6u", &channel);

								if ((result == 1) && (channel < file->number_of_channels))
								{
									file->channel_data[channel].file_flag = false;
								}
								else
								{
									logmessage(MSG_ERROR, "Unknown channel '%s'\n", output[argument][1]);
								}
							}
						}
						else
			   			{
							result = sscanf(output[argument],"%255s",(char*)&channel_name);

							if ((result > 0) && (channel_name_to_number(file, channel_name, ABSCISSA_UNKNOWN, &channel) == true))
							{
								if (channel < file->number_of_channels)
								{
									file->channel_data[channel].file_flag = true;
								}
								else
								{
									logmessage(MSG_ERROR,"Unknown channel '%s'\n",output[argument]);
								}
							}
							else
							{
								result = sscanf(output[argument],"#%6u",&channel);
				   	   
				   				if ((result == 1) && (channel < file->number_of_channels))
				   				{
		   	  						file->channel_data[channel].file_flag = true;
								}
								else
								{
									logmessage(MSG_ERROR,"Unknown channel '%s'\n",output[argument]);
								}
							}
						}
					}
				}
			}
		}
		else if (case_insensitive_compare(output[0],"channel",MAX_LINE_SIZE) == 1)
		{
			process_all = false;
			start_channel = 1;
			finish_channel = 0;

			if (file->preload == false)
			{
				logmessage(MSG_ERROR, "Specify the file to load with 'input-file' before modifying channels\n");
			}
			else if ((number_of_arguments >= 3) && (case_insensitive_compare(output[2], "rename", MAX_LINE_SIZE) == 1))
			{
				if (number_of_arguments == 4)
				{
					result = sscanf(output[1], "%255s", (char*)&channel_name);

					if ((result > 0) && (channel_name_to_number(file, channel_name, ABSCISSA_UNKNOWN, NULL) == true))
					{
						if (channel_name_to_number(file, output[3], ABSCISSA_UNKNOWN, &channel) == true)
						{
							logmessage(WARNING, "Channels can only be renamed to a unique name\n");
						}
						else
						{
							/* Rename channel */

							Rename_Channel(file, channel, (char*)output[3]);
						}
					}
					else
					{
						logmessage(WARNING, "Syntax: channel <name> rename <new-name>\n");
					}
				}
				else
				{
					logmessage(WARNING, "Syntax: channel <name> rename <new-name>\n");
				}
			}
			else if ((number_of_arguments >= 2) && (case_insensitive_compare(output[1], "info", MAX_LINE_SIZE) == 1))
			{
				logmessage(MSG_ERROR, "+-------------------------------------------+\n");
				logmessage(MSG_ERROR, "| Channel table                             |\n");
				logmessage(MSG_ERROR, "+-------------------------------------------+\n");
				
                if (file->channel_data != NULL)
	            {
                	for (channel=0;channel<file->number_of_channels;channel++)
	  	 			{
						if (file->channel_data[channel].loaded == true)
						{
							logmessage(MSG_ERROR, "%u: %s (%s) [%s] %u %s \"%s\" (loaded)\n", channel, file->channel_data[channel].name, file->channel_data[channel].matlab_name, file->channel_data[channel].units, file->channel_data[channel].samples_per_cycle, AbscissaType(file->channel_data[channel].abscissa.type), file->channel_data[channel].description);
						}
						else
						{
							logmessage(MSG_ERROR, "%u: %s (%s) [%s] %u %s \"%s\"\n", channel, file->channel_data[channel].name, file->channel_data[channel].matlab_name, file->channel_data[channel].units, file->channel_data[channel].samples_per_cycle, AbscissaType(file->channel_data[channel].abscissa.type), file->channel_data[channel].description);
						}
					}
				}

				logmessage(MSG_ERROR, "+-------------------------------------------+\n");
			}
			else if (((number_of_arguments == 3) && (case_insensitive_compare(output[2], "create", MAX_LINE_SIZE) == 1)) ||
				     ((number_of_arguments == 4) && (case_insensitive_compare(output[2], "copy", MAX_LINE_SIZE) == 1)))
			{
				result = sscanf(output[1], "%255s", (char*)&channel_name);

				if ((result > 0) && (channel_name_to_number(file, channel_name, ABSCISSA_UNKNOWN, NULL) == false))
				{
					if ((number_of_arguments == 4) && (channel_name_to_number(file, output[3], ABSCISSA_UNKNOWN, &channel) == false))
					{
						logmessage(WARNING, "Syntax: channel <name> copy [<existing-channel>]\n");
					}
					else
					{
						/* Add new channel */

						Add_Channel(file, analysis, channel_name);

						if (number_of_arguments == 4)
						{
							/* Create a copy of an existing channel */

							Copy_Channel(file, channel, file->number_of_channels - 1);
						}

						new_channel_blocks = (AbscissaBlocks*)malloc(file->number_of_channels*sizeof(AbscissaBlocks));

						if (new_channel_blocks == NULL)
						{
							logmessage(FATAL, "Memory could not be allocated\n");
						}

						if (channel_blocks != NULL)
						{
							for (channel = 0; channel < file->number_of_channels - 1; channel++)
							{
								Copy_AbscissaBlocks(&new_channel_blocks[channel], &channel_blocks[channel]);
							}

							new_channel_blocks[channel].number_of_blocks = 0;

							free(channel_blocks);
						}
						else
						{
							for (channel = 0; channel < file->number_of_channels; channel++)
							{
								new_channel_blocks[channel].number_of_blocks = 0;
							}
						}

						channel_blocks = new_channel_blocks;

						new_channel_blocks = NULL;
					}
				}
				else
				{
					if (case_insensitive_compare(output[2], "create", MAX_LINE_SIZE) == 1)
					{
						logmessage(WARNING, "Syntax: channel <name> create\n");
					}
					else
					{
						logmessage(WARNING, "Syntax: channel <name> copy [<existing-channel>]\n");
					}
				}
			}
			else if (case_insensitive_compare(output[1],"all",MAX_LINE_SIZE) == 1)
			{
				start_channel = 0;

				if (file->number_of_channels > 0)
				{
					finish_channel = file->number_of_channels-1;
					result = 1;
				}
				else
				{
					finish_channel = 0;
					result = 0;
				}

				process_all = true;
			}
			else
			{
				result = sscanf(output[1],"%255s",(char*)&channel_name);

				if ((result > 0) && (channel_name_to_number(file, channel_name, ABSCISSA_UNKNOWN, &channel) == true))
				{
					if (channel < file->number_of_channels)
					{
						start_channel = channel;
						finish_channel = channel;
					}
					else
					{
						logmessage(MSG_ERROR,"Unknown channel '%s'\n",output[1]);
					}
				}
				else
				{
					result = sscanf(output[1],"#%6u",&channel);

					if (result == 1)
					{
						start_channel = channel;
						finish_channel = channel;
					}
					else
					{
						logmessage(MSG_ERROR, "Unknown channel '%s'\n", output[1]);
					}
				}
			}

			for (chan=start_channel;chan<=finish_channel;chan++)
			{
				if ((file->channel_data != NULL) && 
					(chan < file->number_of_channels))
				{	
					if (case_insensitive_compare(output[2],"name",MAX_LINE_SIZE) == 1)
					{
						strncpy(file->channel_data[chan].name,output[3],SIZEOF_CHANNEL_NAME);
						file->channel_data[chan].name[SIZEOF_CHANNEL_NAME - 1] = 0x00;

						file->channel_data[chan].short_name[0] = 0x00;
						file->channel_data[chan].matlab_name[0] = 0x00;
					}
					else if (case_insensitive_compare(output[2],"description",MAX_LINE_SIZE) == 1)
					{
						strncpy(file->channel_data[chan].description,output[3],SIZEOF_DESCRIPTION);
						file->channel_data[chan].description[SIZEOF_DESCRIPTION - 1] = 0x00;
					}
					else if (case_insensitive_compare(output[2],"units",MAX_LINE_SIZE) == 1)
					{
						strncpy(file->channel_data[chan].units,output[3],SIZEOF_UNITS);
						file->channel_data[chan].units[SIZEOF_UNITS - 1] = 0x00;
					}
					else if (set_double_parameter(output[2],output[3],"offset",&file->channel_data[chan].offset) == 1)
					{
					}
					else if (set_double_parameter(output[2],output[3],"calibration_adder",&file->channel_data[chan].offset) == 1)
					{
						logmessage(WARNING,"'calibration_adder' parameter is depreciated.  Use 'offset' instead.\n");
					}
					else if (set_double_parameter(output[2],output[3],"slope",&file->channel_data[chan].slope) == 1)
					{
					}
					else if (set_double_parameter(output[2],output[3],"calibration_multiplier",&file->channel_data[chan].slope) == 1)
					{
						logmessage(WARNING,"'calibration_multiplier' parameter is depreciated.  Use 'slope' instead.\n");
					}
					else if (set_float_parameter(output[2],output[3],"tdc_offset",&file->channel_data[chan].tdc_offset) == 1)
					{
						logmessage(WARNING, "'tdc_offset' parameter is depreciated.  Use 'tdc-offset' instead.\n");
					}
					else if (set_float_parameter(output[2], output[3], "tdc-offset", &file->channel_data[chan].tdc_offset) == 1)
					{
					}
					else if (set_float_parameter(output[2],output[3],"filter_upper_frequency",&file->channel_data[chan].filter_config.upper_frequency) == 1)
					{
						logmessage(WARNING, "'filtering_type' parameter is depreciated.  Use 'filter upper-frequency' instead.\n");
					}
					else if (set_float_parameter(output[2],output[3],"filter_lower_frequency",&file->channel_data[chan].filter_config.lower_frequency) == 1)
					{
						logmessage(WARNING, "'filtering_type' parameter is depreciated.  Use 'filter lower-frequency' instead.\n");
					}
					else if (set_uint_parameter(output[2],output[3],"filtering_type",&file->channel_data[chan].filter_config.type) == 1)
					{
						logmessage(WARNING, "'filtering_type' parameter is depreciated.  Use 'filter type' instead.\n");
					}
					else if (set_uint_parameter(output[2],output[3],"cylinder",&file->channel_data[chan].cylinder) == 1)
					{
					}
					else if (case_insensitive_compare(output[2], "analysis-info", MAX_LINE_SIZE) == 1)
					{
						unsigned int analysis_number;
						unsigned int analysis_channel;
						unsigned int number_of_analysis_channels = get_number_of_analysis_channels();

						if (analysis == NULL)
						{
							logmessage(WARNING, "Call channel <channel-name/#channel-number> analysis-info once 'configure-analysis' or 'run-analysis' has been called\n");
						}
						else
						{
							for (analysis_number = 0; analysis_number < analysis->number_of_channels; analysis_number++)
							{
								if ((analysis->channel[analysis_number].channel == chan) && (file->channel_data[analysis->channel[analysis_number].channel].loaded == true))
								{
									logmessage(MSG_ERROR, "+-------------------------------------------+\n");
									logmessage(MSG_ERROR, "| %20s Analysis             |\n", file->channel_data[analysis->channel[analysis_number].channel].name);
									logmessage(MSG_ERROR, "+-------------------------------------------+\n");

									for (analysis_channel = 0; analysis_channel < number_of_analysis_channels; analysis_channel++)
									{
										if (analysis->channel[analysis_number].analysis_rqst[analysis_channel] > AnalysisRqst::NotCalculated)
										{
											logmessage(MSG_ERROR, "%u: %s %s (%s) [%s] %u %s \"%s\"\n", 
												analysis_channel, 
												analysis->channel[analysis_number].results[analysis_channel].name, 
												get_analysis_unique_id(analysis_channel), 
												analysis->channel[analysis_number].results[analysis_channel].matlab_name, 
												analysis->channel[analysis_number].results[analysis_channel].units, 
												analysis->channel[analysis_number].results[analysis_channel].samples_per_cycle, 
												AbscissaType(analysis->channel[analysis_number].results[analysis_channel].abscissa.type), 
												analysis->channel[analysis_number].results[analysis_channel].description);
										}
									}

									logmessage(MSG_ERROR, "+-------------------------------------------+\n");
								}
							}
						}
					}
					else if ((case_insensitive_compare(output[2], "convert-to-time", MAX_LINE_SIZE) == 1) || (case_insensitive_compare(output[2], "create-time", MAX_LINE_SIZE) == 1))
					{
						double conversion_resolution;

						if ((file->channel_data[chan].abscissa.type != ABSCISSA_CRANKANGLE) && (file->channel_data[chan].abscissa.type != ABSCISSA_CYCLE))
						{
							logmessage(NOTICE, "Cannot convert channel %s to time base as it is not crank angle or cycle based\n",file->channel_data[chan].name);
						}
						else if (file->channel_data[chan].loaded == false)
						{
							logmessage(NOTICE, "Channel %s must be loaded before it can be converted from crank angle to time\n", file->channel_data[chan].name);
						}
						else if (case_insensitive_compare(output[2], "create-time", MAX_LINE_SIZE) == 1)
						{
							if (sscanf(output[4], "%10lf", &conversion_resolution) == 1)
							{
								if ((Add_Channel(file, analysis, output[3]) == true) && ((file->channel_data[chan].abscissa.type == ABSCISSA_CRANKANGLE) || (file->channel_data[chan].abscissa.type == ABSCISSA_CYCLE)))
								{
									channel = file->number_of_channels - 1;

									if (Copy_Channel(file, chan, channel) == true)
									{
										strncpy(file->channel_data[channel].name, output[3], SIZEOF_CHANNEL_NAME);
										file->channel_data[channel].short_name[0] = 0x00;
										file->channel_data[channel].matlab_name[0] = 0x00;

										if (Convert_Channel_To_Time(file, channel, conversion_resolution) == false)
										{
											logmessage(WARNING, "Failed to convert channel %s from crank angle to time based abscissa\n", file->channel_data[chan].name);
										}

										new_channel_blocks = (AbscissaBlocks*)malloc(file->number_of_channels*sizeof(AbscissaBlocks));

										if (new_channel_blocks == NULL)
										{
											logmessage(FATAL, "Memory could not be allocated\n");
										}

										if (channel_blocks != NULL)
										{
											for (channel = 0; channel < file->number_of_channels - 1; channel++)
											{
												Copy_AbscissaBlocks(&new_channel_blocks[channel], &channel_blocks[channel]);
											}

											new_channel_blocks[channel].number_of_blocks = 0;

											free(channel_blocks);
										}
										else
										{
											for (channel = 0; channel < file->number_of_channels; channel++)
											{
												new_channel_blocks[channel].number_of_blocks = 0;
											}
										}

										channel_blocks = new_channel_blocks;

										new_channel_blocks = NULL;
									}
									else
									{
										logmessage(WARNING, "Failed to copy original channel data\n");
									}
								}
								else
								{
									logmessage(WARNING, "Channel %s not converted to new time based channel\n",file->channel_data[chan].name);
								}
							}
							else
							{
								logmessage(WARNING, "Syntax: channel <channel-name/#channel-number> create-time <new-channel> <resolution>\n");
							}
						}
						else /* convert-to-time */
						{
							if (sscanf(output[3], "%10lf", &conversion_resolution) == 1)
							{
								if (Convert_Channel_To_Time(file, chan, conversion_resolution) == false)
								{
									logmessage(WARNING, "Failed to convert channel %s from crank angle to time based abscissa\n", file->channel_data[chan].name);
								}
							}
							else
							{
								logmessage(WARNING, "Syntax: channel <channel-name/#channel-number> convert-to-time <resolution>\n");
							}
						}
					}
					else if (case_insensitive_compare(output[2], "delete", MAX_LINE_SIZE) == 1)
					{
						if (Remove_Channel(file, analysis, chan) == false)
						{
							logmessage(MSG_ERROR, "Failed to delete channel '%s'\n", file->channel_data[chan].name);
						}
						else
						{
							if (file->number_of_channels > 0)
							{
								new_channel_blocks = (AbscissaBlocks*)malloc(file->number_of_channels * sizeof(AbscissaBlocks));

								if (new_channel_blocks == NULL)
								{
									logmessage(FATAL, "Memory could not be allocated\n");
								}

								if (channel_blocks != NULL)
								{
									for (channel = 0; channel < chan; channel++)
									{
										Copy_AbscissaBlocks(&new_channel_blocks[channel], &channel_blocks[channel]);
									}

									for (channel = chan; channel < file->number_of_channels; channel++)
									{
										Copy_AbscissaBlocks(&new_channel_blocks[channel], &channel_blocks[channel + 1]);
									}

									free(channel_blocks);
								}
								else
								{
									for (channel = 0; channel < file->number_of_channels; channel++)
									{
										new_channel_blocks[channel].number_of_blocks = 0;
									}
								}

								channel_blocks = new_channel_blocks;

								new_channel_blocks = NULL;
							}
							else
							{
								if (channel_blocks != NULL)
								{
									free(channel_blocks);
									channel_blocks = NULL;
								}
							}
						}
					}
					else if (set_uint_parameter(output[2],output[3],"type",&file->channel_data[chan].type) == 2)
					{
						bool found = false;
						for (channel = 0; channel < NUMBER_OF_CHANNEL_TYPES; channel++)
						{
							if (case_insensitive_compare(output[3], channel_information[channel].name, SIZEOF_CHANNEL_NAME) == 1)
							{
								file->channel_data[chan].type = channel;
								found = true;
								break;
							}
						}

						if (found == false)
						{
							logmessage(WARNING, "Unknown channel type '%s'\n", output[3]);
						}
					}
					else if (case_insensitive_compare(output[2], "generate", MAX_LINE_SIZE) == 1)
					{
						if ((file->channel_data[chan].abscissa.type != ABSCISSA_CRANKANGLE) && (file->channel_data[chan].abscissa.type != ABSCISSA_TIME))
						{
							logmessage(WARNING, "Channel generation is only for crank angle or time based channels\n");
						}
						else if (case_insensitive_compare(output[3], "sin", MAX_LINE_SIZE) == 1)
						{
							if (sscanf(output[4], "%10f", &frequency) == 1)
							{
								if (file->channel_data[chan].data == NULL)
								{
									if (Initialise_Channel_Data(file, channel_blocks, chan) == false)
									{
										logmessage(WARNING, "Failed to build channel data\n");
									}
									else
									{
										file->channel_data[chan].data = (float*)malloc(file->channel_data[chan].samples_per_cycle * file->number_of_cycles * sizeof(float));
										if (file->channel_data[chan].data == NULL)
										{
											logmessage(FATAL, "Failed to allocated memory\n");
										}

										memset(file->channel_data[chan].data, 0, file->channel_data[chan].samples_per_cycle*file->number_of_cycles*sizeof(float));
									}
								}

								if (file->channel_data[chan].duration == NULL)
								{
									file->channel_data[chan].duration = (float*)malloc(file->number_of_cycles * sizeof(float));
									if (file->channel_data[chan].duration == NULL)
									{
										logmessage(FATAL, "Failed to allocated memory\n");
									}

									cycle_duration_analysis(file, chan, analysis_config.engine_speed_type, "SPEED", analysis_config.engine_speed);
								}

								if (file->channel_data[chan].duration == NULL)
								{
									logmessage(WARNING, "Failed to calculate channel duration\n");
								}
								else if (file->channel_data[chan].data == NULL)
								{
									/* Failed to build channel data */
								}
								else if (Generate_Sin_Channel(file, chan, frequency) == false)
								{
									logmessage(WARNING, "Failed to generate a sin channel for %s\n", file->channel_data[chan].name);
								}
								else
								{
									/* Channel generated OK */

									file->channel_data[chan].file_flag = true;
									file->channel_data[chan].loaded = true;
								}
							}
							else
							{
								logmessage(WARNING, "Syntax: channel <channel-name/#channel-number> generate sin <frequency>\n");
							}
						}
						else if (case_insensitive_compare(output[3], "block", MAX_LINE_SIZE) == 1)
						{
							if ((sscanf(output[4], "%10f", &start) == 1) && (sscanf(output[5], "%10f", &finish) == 1))
							{
								if (file->channel_data[chan].data == NULL)
								{
									if (Initialise_Channel_Data(file, channel_blocks, chan) == false)
									{
										logmessage(WARNING, "Failed to build channel data\n");
									}
									else
									{
										file->channel_data[chan].data = (float*)malloc(file->channel_data[chan].samples_per_cycle * file->number_of_cycles * sizeof(float));
										if (file->channel_data[chan].data == NULL)
										{
											logmessage(FATAL, "Failed to allocated memory\n");
										}

										memset(file->channel_data[chan].data, 0, file->channel_data[chan].samples_per_cycle*file->number_of_cycles*sizeof(float));
									}
								}

								if (file->channel_data[chan].duration == NULL)
								{
									file->channel_data[chan].duration = (float*)malloc(file->number_of_cycles * sizeof(float));
									if (file->channel_data[chan].duration == NULL)
									{
										logmessage(FATAL, "Failed to allocated memory\n");
									}

									cycle_duration_analysis(file, chan, analysis_config.engine_speed_type, "SPEED", analysis_config.engine_speed);
								}

								if (file->channel_data[chan].duration == NULL)
								{
									logmessage(WARNING, "Failed to calculate channel duration\n");
								}

								if (file->channel_data[chan].data == NULL)
								{
									/* Failed to build channel data */
								}
								else if (Generate_Block_Channel(file, chan, start, finish) == false)
								{
									logmessage(WARNING, "Failed to generate a block channel for %s\n", file->channel_data[chan].name);
								}
								else
								{
									/* Channel generated OK */

									file->channel_data[chan].file_flag = true;
									file->channel_data[chan].loaded = true;
								}
							}
							else
							{
								logmessage(WARNING, "Syntax: channel <channel-name/#channel-number> generate block <start-angle> <finish-angle>\n");
							}
						}
						else if (case_insensitive_compare(output[3], "cps", MAX_LINE_SIZE) == 1)
						{
							unsigned int pulses_per_rev;

							if (sscanf(output[4], "%6u", &pulses_per_rev))
							{
								if (file->channel_data[chan].data == NULL)
								{
									if (Initialise_Channel_Data(file, channel_blocks, chan) == false)
									{
										logmessage(WARNING, "Failed to build channel data\n");
									}
									else
									{
										file->channel_data[chan].data = (float*)malloc(file->channel_data[chan].samples_per_cycle * file->number_of_cycles * sizeof(float));
										if (file->channel_data[chan].data == NULL)
										{
											logmessage(FATAL, "Failed to allocated memory\n");
										}

										memset(file->channel_data[chan].data, 0, file->channel_data[chan].samples_per_cycle*file->number_of_cycles*sizeof(float));
									}
								}

								if (file->channel_data[chan].duration == NULL)
								{
									file->channel_data[chan].duration = (float*)malloc(file->number_of_cycles * sizeof(float));
									if (file->channel_data[chan].duration == NULL)
									{
										logmessage(FATAL, "Failed to allocated memory\n");
									}

									cycle_duration_analysis(file, chan, analysis_config.engine_speed_type, "SPEED", analysis_config.engine_speed);
								}

								if (file->channel_data[chan].duration == NULL)
								{
									logmessage(WARNING, "Failed to calculate channel duration\n");
								}

								if (file->channel_data[chan].data == NULL)
								{
									/* Failed to build channel data */
								}
								else if (Generate_CPS_Channel(file, chan, pulses_per_rev, 0.5f) == false)
								{
									logmessage(WARNING, "Failed to generate a CPS channel for %s\n", file->channel_data[chan].name);
								}
								else
								{
									/* Channel generated OK */

									file->channel_data[chan].file_flag = true;
									file->channel_data[chan].loaded = true;
								}
							}
							else
							{
								logmessage(WARNING, "Syntax: channel <channel-name/#channel-number> generate cps <pulses-per-rev>\n");
							}
						}
						else
						{
							logmessage(WARNING, "Unknown channel generation type '%s'\n", output[3]);
						}
					}
					else if (case_insensitive_compare(output[2], "sample", MAX_LINE_SIZE) == 1)
					{
						result = sscanf(output[3], "%255s", (char*)&channel_name);

						if ((result > 0) && (channel_name_to_number(file, channel_name, ABSCISSA_UNKNOWN, &channel) == true))
						{
							/* Populate channel "chan" using data from channel "channel" and abscissa "abscissa" */

							if (channel_blocks[chan].number_of_blocks > 0)
							{
								align = (case_insensitive_compare(output[4], "align-tdc", MAX_LINE_SIZE) == 1);

								if (Build_Abscissa(file, &channel_blocks[chan], &abscissa) == true)
								{
									Empty_Abscissa(&file->channel_data[chan].abscissa);

									file->channel_data[chan].samples_per_cycle = Copy_Abscissa(&abscissa, &file->channel_data[chan].abscissa);

									if (calculate_theta_lookups(file, chan) == false)
									{
										logmessage(WARNING, "Failed to calculate theta lookups\n");
									}
									else if (file->channel_data[channel].loaded == false)
									{
										logmessage(WARNING, "Cannot sample from channel %s as it is not loaded\n", file->channel_data[channel].name);
									}
									else
									{
										if (Sample_Channel(file, analysis, chan, channel, align) == false)
										{
											logmessage(WARNING, "Failed to sample channel %s\n", file->channel_data[channel].name);
										}
									}
								}
								else
								{
									logmessage(WARNING, "Failed to build suitable abscissa\n");
								}
							}
							else
							{
								logmessage(WARNING, "Before sampling a channel configure a new abscissa using 'channel %s abscissa add <start_angle> <finish_angle> <resolution>\n");
							}
						}
						else
						{
							logmessage(WARNING, "Syntax: channel <channel-name/#channel-number> sample <channel-name-to-sample> [align-tdc]\n");
						}
					}
					else if (case_insensitive_compare(output[2], "abscissa", MAX_LINE_SIZE) == 1)
					{
						if (file->channel_data[chan].abscissa.type != ABSCISSA_CRANKANGLE)
						{
							if (process_all == false)
							{
								logmessage(MSG_ERROR, "Channel to resample must have a crank angle abscissa\n");
							}
						}
						else if (case_insensitive_compare(output[3], "show", MAX_LINE_SIZE) == 1)
						{
							logmessage(MSG_ERROR, "Channel %s\n", file->channel_data[chan].name);

							if (channel_blocks[chan].number_of_blocks == 0)
							{
								logmessage(MSG_ERROR, "No abscissa blocks configured\n");
							}
							else
							{
								for (block = 0; block < channel_blocks[chan].number_of_blocks; block++)
								{
									logmessage(MSG_ERROR, "Block %u: %0.3f to %0.3f degrees at %0.3f resolution\n", block, channel_blocks[chan].start[block], channel_blocks[chan].finish[block], channel_blocks[chan].resolution[block]);
								}
							}
						}
						else if (case_insensitive_compare(output[3], "clear", MAX_LINE_SIZE) == 1)
						{
							channel_blocks[chan].number_of_blocks = 0;
						}
						else if (case_insensitive_compare(output[3], "resample", MAX_LINE_SIZE) == 1)
						{
							if (channel_blocks[chan].number_of_blocks > 0)
							{
								if (Build_Abscissa(file, &channel_blocks[chan], &abscissa) == true)
								{
									logmessage(NOTICE, "Resampling channel %s...", file->channel_data[chan].name);

									if (case_insensitive_compare(output[4], "no-interpolate", MAX_LINE_SIZE) == 1)
									{
										interpolate = false;
									}
									else
									{
										interpolate = true;
									}

									if (Resample_Channel(file, chan, &abscissa, interpolate) == false)
									{
										logmessage(WARNING, "Failed to resample channel %s\n", file->channel_data[chan].name);
									}

									logmessage(NOTICE, "done\n");

									Empty_Abscissa(&abscissa);
								}
								else
								{
									logmessage(WARNING, "Failed to build abscissa!  Out of memory?\n");
								}
							}
							else
							{
								logmessage(WARNING, "No abscissa blocks exist for channel %u\n", chan);
							}
						}
						else if (case_insensitive_compare(output[3], "add", MAX_LINE_SIZE) == 1)
						{
							if (number_of_arguments == 7)
							{
								float abscissa_start;
								float abscissa_finish;
								float abscissa_resolution;
								bool abscissa_ok;
								float minimum_angle;
								float maximum_angle;

								minimum_angle = -90.0f * (float)file->engine.number_of_strokes;
								maximum_angle = 90.0f * (float)file->engine.number_of_strokes;

								abscissa_ok = true;

								result = sscanf(output[4], "%10f", &abscissa_start);

								if (result == 1)
								{
									if ((abscissa_start < minimum_angle) || (abscissa_start > maximum_angle))
									{
										logmessage(WARNING, "Abscissa start angle of %0.2f is out-of-range\n", abscissa_start);
										abscissa_ok = false;
									}
								}
								else
								{
									logmessage(WARNING, "Abscissa start angle of '%s' is not valid\n", output[4]);
									abscissa_ok = false;
								}

								result = sscanf(output[5], "%10f", &abscissa_finish);

								if (result == 1)
								{
									if ((abscissa_finish < minimum_angle) || (abscissa_finish > maximum_angle))
									{
										logmessage(WARNING, "Abscissa finish angle of %0.2f is out-of-range\n", abscissa_finish);
										abscissa_ok = false;
									}
									else if (abscissa_finish - abscissa_start < FLT_EPSILON)
									{
										logmessage(WARNING, "Abscissa finish angle of %0.2f is less than or equal to start angle of %0.2f\n", abscissa_finish, abscissa_start);
										abscissa_ok = false;
									}
									else
									{
										/* Do Nothing */
									}
								}
								else
								{
									logmessage(WARNING, "Abscissa finish angle of '%s' is not valid\n", output[5]);
									abscissa_ok = false;
								}

								result = sscanf(output[6], "%10f", &abscissa_resolution);

								if (result == 1)
								{
									if ((abscissa_resolution < 1.0f/MinimumResolution(file)) || (abscissa_resolution > 360.0f))
									{
										logmessage(WARNING, "Abscissa resolution of %0.3f is out-of-range\n", abscissa_resolution);
										abscissa_ok = false;
									}
								}
								else
								{
									logmessage(WARNING, "Abscissa resolution of '%s' is not valid\n", output[6]);
									abscissa_ok = false;
								}

								if (abscissa_ok == true)
								{
									if (channel_blocks[chan].number_of_blocks < 3)
									{
										block = channel_blocks[chan].number_of_blocks;

										channel_blocks[chan].start[block] = abscissa_start;
										channel_blocks[chan].finish[block] = abscissa_finish;
										channel_blocks[chan].resolution[block] = abscissa_resolution;

										channel_blocks[chan].number_of_blocks++;
									}
									else
									{
										logmessage(MSG_ERROR, "Too many measurement tables\n");
									}
								}
							}
							else
							{
								logmessage(MSG_ERROR, "Configure a new abscissa using 'channel %s abscissa add <start_angle> <finish_angle> <resolution>\n", file->channel_data[chan].name);
							}
						}
						else
						{
							logmessage(WARNING, "Unknown channel abscissa argument '%s'\n", output[3]);
						}
					}
					else if (case_insensitive_compare(output[2], "digitize", MAX_LINE_SIZE) == 1)
					{
						if (set_uint_parameter(output[3], output[4], "type", &file->channel_data[chan].digital_config.type) == 2)
						{
							if (case_insensitive_compare(output[4], "NONE", MAX_LINE_SIZE) == 1)
							{
								file->channel_data[chan].digital_config.type = DIGITIZE_NONE;
							}
							else if (case_insensitive_compare(output[4], "AUTO", MAX_LINE_SIZE) == 1)
							{
								file->channel_data[chan].digital_config.type = DIGITIZE_AUTO;
							}
							else if (case_insensitive_compare(output[4], "THRESHOLD", MAX_LINE_SIZE) == 1)
							{
								file->channel_data[chan].digital_config.type = DIGITIZE_THRESHOLD;
							}
							else
							{
								logmessage(WARNING, "Unknown digitize type '%s'\n", output[4]);
							}
						}
						else if (set_float_parameter(output[3], output[4], "set-high", &file->channel_data[chan].digital_config.latch_high) == 1)
						{
						}
						else if (set_float_parameter(output[3], output[4], "clear-low", &file->channel_data[chan].digital_config.latch_low) == 1)
						{
						}
						else if (set_float_parameter(output[3], output[4], "filter", &file->channel_data[chan].digital_config.filter) == 1)
						{
						}
						else if (case_insensitive_compare(output[3], "invert", MAX_LINE_SIZE) == 1)
						{
							if ((case_insensitive_compare(output[4], "1", MAX_LINE_SIZE) == 1) ||
								(case_insensitive_compare(output[4], "true", MAX_LINE_SIZE) == 1) ||
								(case_insensitive_compare(output[4], "yes", MAX_LINE_SIZE) == 1))
							{
								file->channel_data[chan].digital_config.invert = true;
							}
							else if ((case_insensitive_compare(output[4], "0", MAX_LINE_SIZE) == 1) ||
								(case_insensitive_compare(output[4], "false", MAX_LINE_SIZE) == 1) ||
								(case_insensitive_compare(output[4], "no", MAX_LINE_SIZE) == 1))
							{
								file->channel_data[chan].digital_config.invert = false;
							}
							else
							{
								logmessage(WARNING, "Unknown \"invert %s\"", output[4]);
							}
						}
						else
						{
							logmessage(WARNING, "Unknown digitize configuration keyword '%s'\n", output[3]);
						}
					}
					else if (case_insensitive_compare(output[2], "filter", MAX_LINE_SIZE) == 1)
					{
						if (set_uint_parameter(output[3], output[4], "type", &file->channel_data[chan].filter_config.type) == 2)
						{
							if (case_insensitive_compare(output[4], "NONE", MAX_LINE_SIZE) == 1)
							{
								file->channel_data[chan].filter_config.type = FILTER_NONE;
							}
							else if (case_insensitive_compare(output[4], "MEDIAN", MAX_LINE_SIZE) == 1)
							{
								file->channel_data[chan].filter_config.type = FILTER_MEDIAN;
							}
							else if (case_insensitive_compare(output[4], "FIR_BAND_BARRIER", MAX_LINE_SIZE) == 1)
							{
								file->channel_data[chan].filter_config.type = FILTER_FIR_BAND_BARRIER;
							}
							else if (case_insensitive_compare(output[4], "FIR_BAND_PASS", MAX_LINE_SIZE) == 1)
							{
								file->channel_data[chan].filter_config.type = FILTER_FIR_BAND_PASS;
							}
							else if (case_insensitive_compare(output[4], "FIR_LOW_PASS", MAX_LINE_SIZE) == 1)
							{
								file->channel_data[chan].filter_config.type = FILTER_FIR_LOW_PASS;
							}
							else if (case_insensitive_compare(output[4], "FIR_HIGH_PASS", MAX_LINE_SIZE) == 1)
							{
								file->channel_data[chan].filter_config.type = FILTER_FIR_HIGH_PASS;
							}
							else if (case_insensitive_compare(output[4], "FIR_SPECIFIED", MAX_LINE_SIZE) == 1)
							{
								file->channel_data[chan].filter_config.type = FILTER_FIR_SPECIFIED;
							}
							else if (case_insensitive_compare(output[4], "FIR_SPECIFIED_RATE", MAX_LINE_SIZE) == 1)
							{
								file->channel_data[chan].filter_config.type = FILTER_FIR_SPECIFIED_RATE;
							}
							else
							{
								logmessage(WARNING, "Unknown filter type '%s'\n", output[4]);
							}
						}
						else if (set_float_parameter(output[3], output[4], "upper-frequency", &file->channel_data[chan].filter_config.upper_frequency) == 1)
						{
						}
						else if (set_float_parameter(output[3], output[4], "lower-frequency", &file->channel_data[chan].filter_config.lower_frequency) == 1)
						{
						}
						else if (set_float_parameter(output[3], output[4], "upper_frequency", &file->channel_data[chan].filter_config.upper_frequency) == 1)
						{
							logmessage(WARNING, "'upper_frequency' parameter is depreciated.  Use 'upper-frequency' instead.\n");
						}
						else if (set_float_parameter(output[3], output[4], "lower_frequency", &file->channel_data[chan].filter_config.lower_frequency) == 1)
						{
							logmessage(WARNING, "'lower_frequency' parameter is depreciated.  Use 'lower-frequency' instead.\n");
						}
						else if (case_insensitive_compare(output[3], "now", MAX_LINE_SIZE) == 1)
						{
							if (file->channel_data[chan].loaded == false)
							{
								logmessage(WARNING, "Cannot filter channel unless it has been loaded\n");
							}
							else if (file->channel_data[chan].filter_config.filtered == true)
							{
								logmessage(WARNING, "Cannot filter channel that has already been filtered\n");
							}
							else
							{
								filter_channel(file, chan, analysis_config.engine_speed);
							}
						}
						else if (case_insensitive_compare(output[3], "force", MAX_LINE_SIZE) == 1)
						{
							if (file->channel_data[chan].loaded == false)
							{
								logmessage(WARNING, "Cannot filter channel unless it has been loaded\n");
							}
							else
							{
								filter_channel(file, chan, analysis_config.engine_speed);
							}
						}
						else
						{
							logmessage(WARNING, "Unknown filter configuration keyword '%s'\n", output[3]);
						}
					}
					else if (case_insensitive_compare(output[2],"channel-offset",MAX_LINE_SIZE) == 1)
					{
						if (set_float_parameter(output[3],output[4],"fixed_value",&file->channel_data[chan].offset_config.fixed_value) == 1)
	    				{
						}
						else if (set_float_parameter(output[3],output[4],"start_window",&file->channel_data[chan].offset_config.start_window) == 1)
						{
							if (file->ptr_DGB != NULL)
							{
								file->ptr_DGB[file->ca_dgb].thekx1 = (double)file->channel_data[chan].offset_config.start_window;
							}
						}
						else if (set_float_parameter(output[3],output[4],"finish_window",&file->channel_data[chan].offset_config.finish_window) == 1)
						{
							if (file->ptr_DGB != NULL)
							{
								file->ptr_DGB[file->ca_dgb].thekx2 = (double)file->channel_data[chan].offset_config.finish_window;
							}
						}
						else if (set_float_parameter(output[3],output[4],"polytropic_index",&file->channel_data[channel].offset_config.polytropic_index) == 1)
						{
							if (file->ptr_DGB != NULL)
							{
								file->ptr_DGB[file->ca_dgb].polexp = (double)file->channel_data[chan].offset_config.polytropic_index;
							}
						}
						else if (set_uint_parameter(output[3], output[4], "calc_interval", &calc_interval) == 1)
						{
							file->channel_data[chan].offset_config.window_size = (calc_interval - 1) / 2;
						}
						else if (set_uint_parameter(output[3],output[4],"window_size",&file->channel_data[chan].offset_config.window_size) == 1)
						{
						}
						else if (case_insensitive_compare(output[3],"truncate",MAX_LINE_SIZE) == 1)
						{
							if ((case_insensitive_compare(output[4],"1",MAX_LINE_SIZE) == 1) ||
								(case_insensitive_compare(output[4], "on", MAX_LINE_SIZE) == 1) ||
								(case_insensitive_compare(output[4], "yes", MAX_LINE_SIZE) == 1))
							{
								file->channel_data[chan].offset_config.truncate = true;
							}
							else if ((case_insensitive_compare(output[4],"0",MAX_LINE_SIZE) == 1) ||
								(case_insensitive_compare(output[4], "off", MAX_LINE_SIZE) == 1) ||
								(case_insensitive_compare(output[4], "no", MAX_LINE_SIZE) == 1))
							{
								file->channel_data[chan].offset_config.truncate = false;
							}
							else
							{
								logmessage(WARNING,"Unknown \"truncate %s\"",output[4]);
							}
						}
						else if (set_uint_parameter(output[3],output[4],"type",&file->channel_data[chan].offset_config.type) == 2)
						{
							if ((case_insensitive_compare(output[4],"NOTHING",MAX_LINE_SIZE) == 1) || 
								(case_insensitive_compare(output[4],"NONE",MAX_LINE_SIZE) == 1))
							{
								file->channel_data[chan].offset_config.type = OFFSET_NONE;
							}
							else if (case_insensitive_compare(output[4],"FIXED",MAX_LINE_SIZE) == 1)
							{
								file->channel_data[chan].offset_config.type = OFFSET_FIXED;
							}
	 						else if (case_insensitive_compare(output[4],"POLYTROPIC",MAX_LINE_SIZE) == 1)
	 						{
								file->channel_data[chan].offset_config.type = OFFSET_POLYTROPIC_2PT;

								logmessage(NOTICE, "Note that you can also specify 'POLYTROPIC_2PT' or 'POLYTROPIC_3PT' offset types.\n", output[4]);
							}
							else if (case_insensitive_compare(output[4], "POLYTROPIC_2PT", MAX_LINE_SIZE) == 1)
							{
								file->channel_data[chan].offset_config.type = OFFSET_POLYTROPIC_2PT;
							}
							else if (case_insensitive_compare(output[4], "POLYTROPIC_3PT", MAX_LINE_SIZE) == 1)
							{
								file->channel_data[chan].offset_config.type = OFFSET_POLYTROPIC_3PT;
							}
							else if (case_insensitive_compare(output[4],"WINDOW",MAX_LINE_SIZE) == 1)
	 						{
								file->channel_data[chan].offset_config.type = OFFSET_WINDOW;
							}
							else if (case_insensitive_compare(output[4], "WINDOW_ABS", MAX_LINE_SIZE) == 1)
							{
								file->channel_data[chan].offset_config.type = OFFSET_WINDOW_ABSOLUTE;
							}
	 						else if (case_insensitive_compare(output[4],"MEAN",MAX_LINE_SIZE) == 1)
	 						{
								file->channel_data[chan].offset_config.type = OFFSET_MEAN;
							}
							else if (case_insensitive_compare(output[4], "RESULT", MAX_LINE_SIZE) == 1)
							{
								file->channel_data[chan].offset_config.type = OFFSET_RESULT_CHANNEL;
							}
							else if ((case_insensitive_compare(output[4],"OTHER_CHANNEL",MAX_LINE_SIZE) == 1) || 
								     (case_insensitive_compare(output[4],"OTHER",MAX_LINE_SIZE) == 1))
	 						{
								logmessage(WARNING,"'%s' offset type is depreciated.  Use 'WINDOW' instead.\n",output[4]);
							}
	 						else if (case_insensitive_compare(output[4],"IFILE",MAX_LINE_SIZE) == 1)
	 						{
								logmessage(WARNING,"'ifile' offset type is depreciated.  catool will automatically use the offset correction specified by the IFile.\n");
							}
							else
							{
								logmessage(WARNING,"Unknown offset type '%s'\n",output[4]);
							}
						}
						else if (case_insensitive_compare(output[3], "channel", MAX_LINE_SIZE) == 1)
						{
							result = sscanf(output[4], "%255s", (char*)&channel_name);

							if ((result > 0) && (channel_name_to_number(file, channel_name, ABSCISSA_UNKNOWN, &channel) == true))
							{
								strncpy(file->channel_data[chan].offset_config.channel_name, channel_name, SIZEOF_CHANNEL_NAME);
							}
							else
							{
								logmessage(WARNING, "Setting channel offset parameter %s to %s failed\n", output[3], output[4]);
							}
						}
						else
						{
							logmessage(WARNING,"Setting channel offset parameter %s to %s failed\n",output[3],output[4]);
						}
					}
					else if (case_insensitive_compare(output[2],"soc",MAX_LINE_SIZE) == 1)
					{
						if (set_float_parameter(output[3],output[4],"start_window",&file->channel_data[chan].soc_config.start_window) == 1)
						{
						}
						else if (set_float_parameter(output[3],output[4],"finish_window",&file->channel_data[chan].soc_config.finish_window) == 1)
						{
						}
						else if (set_float_parameter(output[3],output[4],"value",&file->channel_data[chan].soc_config.fixed_value) == 1)
						{
						}
						else if (case_insensitive_compare(output[3],"aligned",MAX_LINE_SIZE) == 1)
						{
							if ((case_insensitive_compare(output[4], "1", MAX_LINE_SIZE) == 1) ||
								(case_insensitive_compare(output[4], "true", MAX_LINE_SIZE) == 1) ||
								(case_insensitive_compare(output[4], "yes", MAX_LINE_SIZE) == 1))
							{
								file->channel_data[chan].soc_config.aligned = true;
							}
							else if ((case_insensitive_compare(output[4], "0", MAX_LINE_SIZE) == 1) ||
								(case_insensitive_compare(output[4], "false", MAX_LINE_SIZE) == 1) ||
								(case_insensitive_compare(output[4], "no", MAX_LINE_SIZE) == 1))
							{
								file->channel_data[chan].soc_config.aligned = false;
							}
							else
							{
								logmessage(WARNING, "Unknown \"aligned %s\"", output[4]);
							}
						}
						else if (case_insensitive_compare(output[3],"invert",MAX_LINE_SIZE) == 1)
						{
							if ((case_insensitive_compare(output[4], "1", MAX_LINE_SIZE) == 1) ||
								(case_insensitive_compare(output[4], "true", MAX_LINE_SIZE) == 1) ||
								(case_insensitive_compare(output[4], "yes", MAX_LINE_SIZE) == 1))
							{
								file->channel_data[chan].soc_config.invert = true;
							}
							else if ((case_insensitive_compare(output[4], "0", MAX_LINE_SIZE) == 1) ||
								(case_insensitive_compare(output[4], "false", MAX_LINE_SIZE) == 1) ||
								(case_insensitive_compare(output[4], "no", MAX_LINE_SIZE) == 1))
							{
								file->channel_data[chan].soc_config.invert = false;
							}
							else
							{
								logmessage(WARNING, "Unknown \"invert %s\"", output[4]);
							}
						}
						else if (set_uint_parameter(output[3],output[4],"type",&file->channel_data[chan].soc_config.type) == 2)
						{
							if ((case_insensitive_compare(output[4],"NOTHING",MAX_LINE_SIZE) == 1) || 
								(case_insensitive_compare(output[4],"NONE",MAX_LINE_SIZE) == 1))
							{
								file->channel_data[chan].soc_config.type = SOC_FIXED;
								file->channel_data[chan].soc_config.fixed_value = 0.0f;
							}
							else if ((case_insensitive_compare(output[4],"SPECIFY",MAX_LINE_SIZE) == 1) || 
							         (case_insensitive_compare(output[4], "FIXED", MAX_LINE_SIZE) == 1))
							{
								file->channel_data[chan].soc_config.type = SOC_FIXED;
							}
							else if (case_insensitive_compare(output[4], "CA_CHANNEL_RISE_RATE", MAX_LINE_SIZE) == 1)
							{
								file->channel_data[chan].soc_config.type = SOC_CA_CHANNEL_RISE;
							}
							else if (case_insensitive_compare(output[4], "CA_CHANNEL_FALL_RATE", MAX_LINE_SIZE) == 1)
							{
								file->channel_data[chan].soc_config.type = SOC_CA_CHANNEL_FALL;
							}
							else if (case_insensitive_compare(output[4], "CA_CHANNEL_AVERAGE", MAX_LINE_SIZE) == 1)
							{
								file->channel_data[chan].soc_config.type = SOC_CA_CHANNEL_AVG;
							}
							else if (case_insensitive_compare(output[4], "CYCLE_CHANNEL", MAX_LINE_SIZE) == 1)
							{
								file->channel_data[chan].soc_config.type = SOC_CYC_CHANNEL;
							}
							else if (case_insensitive_compare(output[4], "DIGITAL_FALLING_EDGE", MAX_LINE_SIZE) == 1)
							{
								file->channel_data[chan].soc_config.type = SOC_DIGITAL_FALLING_EDGE;
							}
							else if (case_insensitive_compare(output[4], "DIGITAL_RISING_EDGE", MAX_LINE_SIZE) == 1)
							{
								file->channel_data[chan].soc_config.type = SOC_DIGITAL_RISING_EDGE;
							}
							else if (case_insensitive_compare(output[4], "POSITIVE_HR", MAX_LINE_SIZE) == 1)
							{
								file->channel_data[chan].soc_config.type = SOC_POSITIVE_HR;
							}
							else
							{
								logmessage(WARNING, "Unknown start of combustion type '%s'\n", output[4]);
							}
						}
						else if (case_insensitive_compare(output[3], "analysis_type", MAX_LINE_SIZE) == 1)
						{
							logmessage(WARNING, "'analysis_type' parameter is depreciated.  The parameter is no longer required.\n");
						}
						else if (case_insensitive_compare(output[3], "channel", MAX_LINE_SIZE) == 1)
						{
							result = sscanf(output[4], "%255s", (char*)&channel_name);

							if ((result == 1) && (channel_name_to_number(file, channel_name, ABSCISSA_UNKNOWN, &channel) == true))
							{
								strncpy(file->channel_data[chan].soc_config.channel_name, channel_name, SIZEOF_CHANNEL_NAME);
							}
							else
							{
								logmessage(WARNING, "Setting channel offset parameter %s to %s failed\n", output[3], output[4]);
							}
						}
						else
						{
							logmessage(NOTICE, "Setting channel start of combustion parameter %s to %s failed\n", output[3], output[4]);
						}
					}
					else if (case_insensitive_compare(output[2], "camshaft", MAX_LINE_SIZE) == 1)
					{
						if (set_uint_parameter(output[3], output[4], "edge", &file->channel_data[chan].soc_config.type) == 2)
						{
							if (case_insensitive_compare(output[4], "RISING", MAX_LINE_SIZE) == 1)
							{
								file->channel_data[chan].cam_config.edge = EDGE_RISING;
							}
							else if (case_insensitive_compare(output[4], "FALLING", MAX_LINE_SIZE) == 1)
							{
								file->channel_data[chan].cam_config.edge = EDGE_FALLING;
							}
							else if (case_insensitive_compare(output[4], "BOTH", MAX_LINE_SIZE) == 1)
							{
								file->channel_data[chan].cam_config.edge = EDGE_BOTH;
							}
							else
							{
								logmessage(WARNING, "Unknown camshaft edge '%s'\n", output[4]);
							}
						}
						else if (set_float_parameter(output[3], output[4], "reference_angle", &file->channel_data[chan].cam_config.reference_angle) == 1)
						{
						}
						else if (set_float_parameter(output[3], output[4], "offset", &file->channel_data[chan].cam_config.offset) == 1)
						{
						}
						else if (case_insensitive_compare(output[3], "invert", MAX_LINE_SIZE) == 1)
						{
							if ((case_insensitive_compare(output[4], "1", MAX_LINE_SIZE) == 1) ||
								(case_insensitive_compare(output[4], "true", MAX_LINE_SIZE) == 1) ||
								(case_insensitive_compare(output[4], "yes", MAX_LINE_SIZE) == 1))
							{
								file->channel_data[chan].cam_config.invert = true;
							}
							else if ((case_insensitive_compare(output[4], "0", MAX_LINE_SIZE) == 1) ||
								(case_insensitive_compare(output[4], "false", MAX_LINE_SIZE) == 1) ||
								(case_insensitive_compare(output[4], "no", MAX_LINE_SIZE) == 1))
							{
								file->channel_data[chan].cam_config.invert = false;
							}
							else
							{
								logmessage(WARNING, "Unknown \"invert %s\"", output[4]);
							}
						}
						else if (case_insensitive_compare(output[3], "type", MAX_LINE_SIZE) == 1)
						{
							if (case_insensitive_compare(output[4], "NONE", MAX_LINE_SIZE) == 1)
							{
								file->channel_data[chan].cam_config.type = CAMTYPE_NONE;
							}
							else if (case_insensitive_compare(output[4], "INLET_CID", MAX_LINE_SIZE) == 1)
							{
								file->channel_data[chan].cam_config.type = CAMTYPE_INLET_CID;
							}
							else if (case_insensitive_compare(output[4], "INLET_CID", MAX_LINE_SIZE) == 1)
							{
								file->channel_data[chan].cam_config.type = CAMTYPE_INLET_CID;
							}
							else if (case_insensitive_compare(output[4], "EXHAUST_CID", MAX_LINE_SIZE) == 1)
							{
								file->channel_data[chan].cam_config.type = CAMTYPE_EXHAUST_CID;
							}
							else if (case_insensitive_compare(output[4], "4_PLUS_2", MAX_LINE_SIZE) == 1)
							{
								file->channel_data[chan].cam_config.type = CAMTYPE_4_PLUS_2;
							}
							else if (case_insensitive_compare(output[4], "BOSCH_QUICK_SYNC", MAX_LINE_SIZE) == 1)
							{
								file->channel_data[chan].cam_config.type = CAMTYPE_BOSCH_QUICK_SYNC;
							}
							else if (case_insensitive_compare(output[4], "4_MINUS_1", MAX_LINE_SIZE) == 1)
							{
								file->channel_data[chan].cam_config.type = CAMTYPE_4_MINUS_1;
							}
							else if (case_insensitive_compare(output[4], "ECOBOOST_INLET", MAX_LINE_SIZE) == 1)
							{
								file->channel_data[chan].cam_config.type = CAMTYPE_ECOBOOST_INLET;
							}
							else if (case_insensitive_compare(output[4], "ECOBOOST_EXHAUST", MAX_LINE_SIZE) == 1)
							{
								file->channel_data[chan].cam_config.type = CAMTYPE_ECOBOOST_EXHAUST;
							}
							else
							{
								logmessage(WARNING, "Unknown camshaft type '%s'\n", output[4]);
							}
						}
						else
						{
							logmessage(NOTICE, "Setting channel start of combustion parameter %s to %s failed\n", output[3], output[4]);
						}
					}
					else if (case_insensitive_compare(output[2], "analysis", MAX_LINE_SIZE) == 1)
					{
						if (case_insensitive_compare(output[5], "rename", MAX_LINE_SIZE) == 1)
						{
							if (analysis == NULL)
							{
								logmessage(WARNING, "You must call 'run-analysis' before renaming analysis channels\n");
							}
							else if (number_of_arguments != 6)
							{
								logmessage(WARNING, "Syntax: channel <channel-name/#channel-number> analysis <analysis-type> rename <new-channel-name>\n");
							}
							else
							{
								int analysis_channel = get_analysis_type_from_unique_id(output[4]);

								if (analysis_channel >= 0)
								{
									for (analysis_structure = 0; analysis_structure < analysis->number_of_channels; analysis_structure++)
									{
										if (analysis->channel[analysis_structure].channel == chan)
										{
											strncpy(analysis->channel[analysis_structure].results[analysis_channel].name, output[5], SIZEOF_CHANNEL_NAME);
											analysis->channel[analysis_structure].results[analysis_channel].name[SIZEOF_CHANNEL_NAME - 1] = 0x00;
										}
									}
								}
								else
								{
									logmessage(WARNING, "Unknown analysis type \"%s\"\n", output[4]);
								}
							}
						}
					}
					else
					{
						logmessage(WARNING, "Setting channel %u parameter %s to %s failed\n", chan, output[2], output[3]);
					}
				}
				else
				{
					logmessage(WARNING, "Channel number %u must be loaded before it can be specifically configured\n", chan);
				}
			}
		}
		else if (case_insensitive_compare(output[0], "engine", MAX_LINE_SIZE) == 1)
		{
			if (case_insensitive_compare(output[1], "name", MAX_LINE_SIZE) == 1)
			{
				strncpy(file->engine.name, output[2], SIZEOF_ENGINE_NAME);
				file->engine.name[SIZEOF_ENGINE_NAME - 1] = 0x00;
			}
			else if (set_float_parameter(output[1], output[2], "bore", &file->engine.bore) == 1)
			{
			}
			else if (set_float_parameter(output[1], output[2], "stroke", &file->engine.stroke) == 1)
			{
			}
			else if (set_float_parameter(output[1], output[2], "conrod_length", &file->engine.conrod_length) == 1)
			{
			}
			else if (set_float_parameter(output[1], output[2], "piston_factor", &file->engine.piston_factor) == 1)
			{
			}
			else if (case_insensitive_compare(output[1], "pin_offset", MAX_LINE_SIZE) == 1)
			{
				if (number_of_arguments == 4)
				{
					result = sscanf(output[3], "%6f", &pin_offset);

					if (result == 1)
					{
						if ((pin_offset < -100.0f) || (pin_offset > 100.0f))
						{
							logmessage(WARNING, "Pin offset of %f looks unlikely.\n");
						}
						else if (case_insensitive_compare(output[2], "all", MAX_LINE_SIZE) == 1)
						{
							for (cylinder = 0; cylinder < MAX_NUMBER_OF_CYLINDERS; cylinder++)
							{
								file->engine.pin_offset[cylinder] = pin_offset;
							}

							logmessage(DEBUG, "Set all pin offsets to %f\n", pin_offset);
						}
						else
						{
							result = sscanf(output[2], "%6u", &cylinder);

							if (result == 1)
							{
								if ((cylinder > 0) && (cylinder <= MAX_NUMBER_OF_CYLINDERS))
								{
									file->engine.pin_offset[cylinder - 1] = pin_offset;

									logmessage(DEBUG, "Set pin offset for cylinder %u to %f\n", cylinder, pin_offset);
								}
								else
								{
									logmessage(WARNING, "Pin offset cylinder must be between 1 and the number of cylinders\n");
								}
							}
							else
							{
								logmessage(WARNING, "New 'pin_offset' usage:  engine pin_offset <cylinder> <offset>.\n");
							}
						}
					}
					else
					{
						logmessage(WARNING, "New 'pin_offset' usage:  engine pin_offset <cylinder> <offset>.\n");
					}
				}
				else
				{
					logmessage(WARNING,"New 'pin_offset' usage:  engine pin_offset <cylinder> <offset>.\n");
				}
			}
			else if (case_insensitive_compare(output[1],"pin_offset_2",MAX_LINE_SIZE) == 1)
			{
				logmessage(WARNING,"New 'pin_offset' usage:  engine pin_offset <cylinder> <offset>.\n");
			}
			else if (set_float_parameter(output[1],output[2],"compression_ratio",&file->engine.compression_ratio) == 1)
			{
			}
			else if (set_uint_parameter(output[1],output[2],"cylinders",&file->engine.number_of_cylinders) == 1)
			{
			}
			else if (set_uint_parameter(output[1],output[2],"strokes",&file->engine.number_of_strokes) == 1)
			{
			}
			else if (set_float_parameter(output[1],output[2],"evo_ca",&file->engine.evo_angle) == 1)
			{
			}
			else if (set_float_parameter(output[1],output[2],"ivc_ca",&file->engine.ivc_angle) == 1)
			{
			}
			else if (set_float_parameter(output[1], output[2], "evc_ca", &file->engine.evc_angle) == 1)
			{
			}
			else if (set_float_parameter(output[1], output[2], "ivo_ca", &file->engine.ivo_angle) == 1)
			{
			}
			else if (set_uint_parameter(output[1],output[2],"type",&file->engine.type) == 2)
			{
				if (case_insensitive_compare(output[2],"SI",MAX_LINE_SIZE) == 1)
				{
					file->engine.type = ENGINE_SI;
				}
				else if (case_insensitive_compare(output[2],"DISI",MAX_LINE_SIZE) == 1)
				{
					file->engine.type = ENGINE_DISI;
				}
	 			else if (case_insensitive_compare(output[2],"CI_DI",MAX_LINE_SIZE) == 1)
	 			{
					file->engine.type = ENGINE_CI_DI;
				}
	 			else if (case_insensitive_compare(output[2],"CI_IDI",MAX_LINE_SIZE) == 1)
	 			{
					file->engine.type = ENGINE_CI_IDI;
				}
				else if (case_insensitive_compare(output[2],"SI_ROTARY",MAX_LINE_SIZE) == 1)
				{
					file->engine.type = ENGINE_SI_ROTARY;
				}
				else
				{
					logmessage(WARNING,"Unknown engine type '%s'\n",output[2]);
				}
			}
			else if (case_insensitive_compare(output[1],"info",MAX_LINE_SIZE) == 1)
			{
				logmessage(MSG_ERROR, "+-------------------------------------------+\n");
				logmessage(MSG_ERROR, "| Engine                                    |\n");
				logmessage(MSG_ERROR, "+-------------------------------------------+\n");

                logmessage(MSG_ERROR, "Name: %s\n",file->engine.name);
                logmessage(MSG_ERROR, "Bore: %0.3f mm\n",file->engine.bore);
                logmessage(MSG_ERROR, "Stroke: %0.3f mm\n",file->engine.stroke);
                logmessage(MSG_ERROR, "Condrod Length: %0.3f mm\n",file->engine.conrod_length);
                
				for (cylinder=0;cylinder<file->engine.number_of_cylinders;cylinder++)
                {
                	logmessage(MSG_ERROR, "Pin Offset %u: %0.3f mm\n",cylinder,file->engine.pin_offset[cylinder]);
				}
                
				logmessage(MSG_ERROR, "Number of Strokes: %u\n",file->engine.number_of_strokes);
                
				switch (file->engine.type)
				{
				default:
				case ENGINE_UNKNOWN:
				{
					logmessage(MSG_ERROR, "Type: Unknown\n");
					break;
				}
				case ENGINE_SI:
				{
					logmessage(MSG_ERROR, "Type: Spark Ignition\n");
					break;
				}
				case ENGINE_DISI:
				{
					logmessage(MSG_ERROR, "Type: DI Spark Ignition\n");
					break;
				}
				case ENGINE_CI_DI:
				{
					logmessage(MSG_ERROR, "Type: Compression Ignition Direct\n");
					break;
				}
				case ENGINE_CI_IDI:
				{
					logmessage(MSG_ERROR, "Type: Compression Ignition Indirect\n");
					break;
				}
				case ENGINE_SI_ROTARY:
				{
					logmessage(MSG_ERROR, "Type: Rotary\n");
					break;
				}
				}
				
                logmessage(MSG_ERROR, "Number of Cylinders: %u\n",file->engine.number_of_cylinders);
				logmessage(MSG_ERROR, "Intake Valve Opening: %0.1f\n", file->engine.ivo_angle);
                logmessage(MSG_ERROR, "Intake Valve Closing: %0.1f\n",file->engine.ivc_angle);
                logmessage(MSG_ERROR, "Exhaust Valve Opening: %0.1f\n",file->engine.evo_angle);
				logmessage(MSG_ERROR, "Exhaust Valve Closing: %0.1f\n", file->engine.evc_angle);

				if (file->engine.volume_calculation_type == VOLUME_USER)
				{
					logmessage(MSG_ERROR, "Volume: User\n  Number of samples: %u\n", file->engine.user.number_of_data_points);

					if (file->engine.user.crank_angle == NULL)
					{
						logmessage(MSG_ERROR, "  Crank angle: NULL\n");
					}

					if (file->engine.user.volume == NULL)
					{
						logmessage(MSG_ERROR, "  Volume: NULL\n");
					}

					if ((file->engine.user.crank_angle != NULL) && (file->engine.user.volume != NULL))
					{
						unsigned int data_pointer;

						logmessage(MSG_ERROR, "  CA (Deg),Volume (cc)\n");

						for (data_pointer = 0; data_pointer < 10; data_pointer++)
						{
							logmessage(MSG_ERROR, "  %g,%g\n", file->engine.user.crank_angle[data_pointer], file->engine.user.volume[data_pointer]);
						}

						logmessage(MSG_ERROR, "  ...\n");

						for (data_pointer = file->engine.user.number_of_data_points - 10; data_pointer < file->engine.user.number_of_data_points; data_pointer++)
						{
							logmessage(MSG_ERROR, "  %g,%g\n", file->engine.user.crank_angle[data_pointer], file->engine.user.volume[data_pointer]);
						}
					}
				}
				else
				{
					logmessage(MSG_ERROR, "Volume: catool\n");
				}

				logmessage(MSG_ERROR, "+-------------------------------------------+\n");
            }
			else if (case_insensitive_compare(output[1], "custom-volume", MAX_LINE_SIZE) == 1)
			{
#ifdef _CATOOL_UNICODE_
				size_t newsize = strlen(output[2]) + 1;
				wchar_t* wcstring = new wchar_t[newsize];
				size_t convertedChars = 0;
				mbstowcs_s(&convertedChars, wcstring, newsize, output[2], _TRUNCATE);
				if (Import_Custom_Volume(&file->engine, wcstring, 28800) == true)
#else
				if (Import_Custom_Volume(&file->engine, output[2], 28800) == true)
#endif
				{
					Calculate_Geometry(&file->engine);

					for (channel = 0; channel < file->number_of_channels; channel++)
					{
						if (file->channel_data[channel].abscissa.type == ABSCISSA_CRANKANGLE)
						{
							Calculate_Volume(file, channel, &file->channel_data[channel].abscissa.volume);
						}
					}

					logmessage(NOTICE, "Volume data imported successfully from file\n");
				}

#ifdef _CATOOL_UNICODE_
				delete[]wcstring;
#endif
			}
			else
			{
				logmessage(WARNING,"Setting engine parameter %s to %s failed\n",output[1],output[2]);
			}
		}
		else if (case_insensitive_compare(output[0],"load-file",MAX_LINE_SIZE) == 1)
		{
			/* Check for dependent channels, i.e. for offset correction and SOC */

			for (channel = 0; channel < file->number_of_channels; channel++)
			{
				if ((file->channel_data[channel].offset_config.type == OFFSET_WINDOW) ||
					(file->channel_data[channel].offset_config.type == OFFSET_WINDOW_ABSOLUTE))
				{
					unsigned int offset_channel;

					if (channel_name_to_number(file, file->channel_data[channel].offset_config.channel_name, ABSCISSA_CRANKANGLE, &offset_channel) == true)
					{
						if (file->channel_data[offset_channel].file_flag == false)
						{
							logmessage(WARNING, "Loading channel %s for use with %s offset\n", file->channel_data[offset_channel].name, file->channel_data[channel].name);

							file->channel_data[offset_channel].file_flag = true;
						}
						else
						{
							logmessage(DEBUG, "%s: Offset reference channel %s already loaded\n", file->channel_data[channel].name, file->channel_data[channel].offset_config.channel_name);
						}
					}
					else
					{
						logmessage(DEBUG, "%s: Offset reference channel %s not found\n", file->channel_data[channel].name, file->channel_data[channel].offset_config.channel_name);
					}
				}
				else
				{
					logmessage(DEBUG, "%s: Offset type does not require reference channel\n", file->channel_data[channel].name);
				}

				if ((file->channel_data[channel].soc_config.type == SOC_CA_CHANNEL_AVG) ||
					(file->channel_data[channel].soc_config.type == SOC_CA_CHANNEL_FALL) ||
					(file->channel_data[channel].soc_config.type == SOC_CA_CHANNEL_RISE) ||
					(file->channel_data[channel].soc_config.type == SOC_CYC_CHANNEL))
				{
					unsigned int soc_channel;

					if (channel_name_to_number(file, file->channel_data[channel].soc_config.channel_name, ABSCISSA_UNKNOWN, &soc_channel) == true)
					{
						if (file->channel_data[soc_channel].file_flag == false)
						{
							logmessage(WARNING, "Loading channel %s for use with %s start of combustion\n", file->channel_data[soc_channel].name, file->channel_data[channel].name);

							file->channel_data[soc_channel].file_flag = true;
						}
						else
						{
							logmessage(DEBUG, "%s: SOC reference channel %s already loaded\n", file->channel_data[channel].name, file->channel_data[channel].soc_config.channel_name);
						}
					}
					else
					{
						logmessage(DEBUG, "%s: SOC reference channel %s not found\n", file->channel_data[channel].name, file->channel_data[channel].soc_config.channel_name);
					}
				}
				else
				{
					logmessage(DEBUG, "%s: SOC type does not require reference channel\n", file->channel_data[channel].name);
				}
			}

			switch (file->file_type)
			{
			case FILE_AVLIFILE:
			{
	  	 		for (channel=0;channel<file->number_of_channels;channel++)
	  	 		{
			    	if (file->channel_data[channel].file_flag == true)
			    	{
						logmessage(DEBUG,"Loading AVL channel %u\n",channel);
						
						load_ifile_channel(file,channel);
					}
				}
				
				file->sync_checked = false;
			    break;
			}
			case FILE_CSV:
			{
				load_csv_file(file, resolution, number_of_lines_to_skip, channel_names_line, channel_units_line, skip_columns, 0, CSV_COLUMN_CHANNELS, true);

				file->sync_checked = false;
				break;
			}
			case FILE_CAS:
			case FILE_MATLAB:
			case FILE_TDMS:
			case FILE_YOKOGAWA:
			case FILE_SDF:
			case FILE_ATIMAT:
			default:
			{
				logmessage(WARNING,"Filetype not currently supported\n");

			    break;
			}
			}

			logmessage(NOTICE, "Running cycle duration analysis...");

			for (channel = 0; channel < file->number_of_channels; channel++)
			{
				if (file->channel_data[channel].loaded == true)
				{
					cycle_duration_analysis(file, channel, analysis_config.engine_speed_type, "SPEED", analysis_config.engine_speed);
				}
			}

			logmessage(NOTICE, "done\n");
		}
		else if (case_insensitive_compare(output[0],"channel-offset",MAX_LINE_SIZE) == 1)
		{
			logmessage(WARNING,"channel-offset parameter is depreciated.  Use channel <n> channel-offset instead\n");
		}
		else if (case_insensitive_compare(output[0],"set",MAX_LINE_SIZE) == 1)
		{
			if (case_insensitive_compare(output[1],"comment",MAX_LINE_SIZE) == 1)
			{
				strncpy(file->comment,output[2],SIZEOF_DESCRIPTION);
			}
			else if (case_insensitive_compare(output[1],"parameter-file",MAX_LINE_SIZE) == 1)
			{
				strncpy((char*)file->ptr_APB->parfil,output[2],18);
			}
			else if (case_insensitive_compare(output[1],"date",MAX_LINE_SIZE) == 1)
			{
				logmessage(NOTICE,"The 'set date' command is not yet implemented\n");
			}
			else if (case_insensitive_compare(output[1],"parameter",MAX_LINE_SIZE) == 1)
			{
				if (case_insensitive_compare(output[2], "show", MAX_LINE_SIZE) == 1)
				{
					Parameter* parameter = file->parameters;

					logmessage(DEBUG, "Outputting parameter values\n");

					logmessage(MSG_ERROR, "+---------------------------------------------+\n");
					logmessage(MSG_ERROR, "|                  Parameters                 |\n");
					logmessage(MSG_ERROR, "+---------------------------------------------+\n");

					while (parameter != NULL)
					{
						if ((parameter->data_type == VARIABLE_DOUBLE) && (parameter->name[0] != 0x00))
						{
							logmessage(MSG_ERROR, "| %2u | %10s | %12g | %10s |\n", parameter->number + 1, parameter->name, *(double*)(parameter->value), parameter->units);
						}

						parameter = parameter->next;
					}

					logmessage(MSG_ERROR, "+---------------------------------------------+\n");
				}
				else if (file->ptr_APB != NULL)
				{
					set_uint_parameter(output[1],output[2],"parameter",&parameter_number);
					
					if (parameter_number <= 27)
					{
						strncpy((char*)file->ptr_APB->bpar[parameter_number].name,output[3],10);
						strncpy((char*)file->ptr_APB->bpar[parameter_number].einh,output[4],10);
						set_double_parameter(output[1],output[5],"parameter",&file->ptr_APB->bpar[parameter_number].wert);
					}
					else
					{
						logmessage(NOTICE,"Parameter must be in range zero to 27\n");
					}
				}
				else
				{
					logmessage(NOTICE,"IFile block has not been initialised\n");
				}
			}
	    	else if (set_float_parameter(output[1],output[2],"eeoc_start_window",&analysis_config.eeoc_start_window) == 1)
	    	{
			}
			else if (set_float_parameter(output[1],output[2],"eeoc_finish_window",&analysis_config.eeoc_finish_window) == 1)
			{
			}
			else if (set_float_parameter(output[1],output[2],"fft_start_window",&analysis_config.fft_start_window) == 1)
	    	{
			}
			else if (set_float_parameter(output[1],output[2],"fft_finish_window",&analysis_config.fft_finish_window) == 1)
			{
			}
			else if (set_float_parameter(output[1],output[2],"eeoc_index",&analysis_config.eeoc_index) == 1)
			{
			}
			else if (set_float_parameter(output[1],output[2],"t_ivc",&analysis_config.t_ivc) == 1)
			{
			}
			else if (set_float_parameter(output[1],output[2],"temp_ref_ca",&analysis_config.temp_ref_ca) == 1)
			{
			}
			else if (set_float_parameter(output[1],output[2],"mfb_n",&analysis_config.mfb_n) == 1)
			{
			}
			else if (set_float_parameter(output[1],output[2],"annand_a",&analysis_config.annand_a) == 1)
			{
			}
			else if (set_float_parameter(output[1],output[2],"t_wall",&analysis_config.t_wall) == 1)
			{
			}
			else if (set_float_parameter(output[1],output[2],"R",&analysis_config.R) == 1)
			{
			}
			else if (set_float_parameter(output[1],output[2],"poly_exp_start_angle",&analysis_config.poly_exp_start_angle) == 1)
			{
			}
			else if (set_float_parameter(output[1],output[2],"poly_exp_finish_angle",&analysis_config.poly_exp_finish_angle) == 1)
			{
			}
			else if (set_float_parameter(output[1],output[2],"poly_comp_start_angle",&analysis_config.poly_comp_start_angle) == 1)
			{
			}
			else if (set_float_parameter(output[1],output[2],"poly_comp_finish_angle",&analysis_config.poly_comp_finish_angle) == 1)
			{
			}
			else if (set_float_parameter(output[1],output[2],"pkp_start_angle",&analysis_config.pkp_start_angle) == 1)
			{
			}
			else if (set_float_parameter(output[1],output[2],"pkp_finish_angle",&analysis_config.pkp_finish_angle) == 1)
			{
			}
			else if (set_float_parameter(output[1],output[2],"pkp_smoothing_range",&analysis_config.pkp_smoothing_range) == 1)
			{
			}
			else if (set_float_parameter(output[1], output[2], "pkp_smoothing_resolution", &analysis_config.pkp_smoothing_resolution) == 1)
			{
			}
			else if (set_float_parameter(output[1], output[2], "pressure_rise_start_angle", &analysis_config.pressure_rise_start_angle) == 1)
			{
			}
			else if (set_float_parameter(output[1], output[2], "pressure_rise_finish_angle", &analysis_config.pressure_rise_finish_angle) == 1)
			{
			}
			else if (set_float_parameter(output[1], output[2], "pressure_rise_range", &analysis_config.pressure_rise_range) == 1)
			{
			}
			else if (set_float_parameter(output[1],output[2],"cd_start_angle",&analysis_config.cd_start_angle) == 1)
			{
			}
			else if (set_float_parameter(output[1],output[2],"cd_finish_angle",&analysis_config.cd_finish_angle) == 1)
			{
			}
			else if (set_float_parameter(output[1],output[2],"engine_speed",&analysis_config.engine_speed) == 1)
			{
			}
			else if (set_float_parameter(output[1], output[2], "injector_start_window", &analysis_config.injector_start_window) == 1)
			{
			}
			else if (set_float_parameter(output[1], output[2], "injector_finish_window", &analysis_config.injector_finish_window) == 1)
			{
			}
			else if (set_uint_parameter(output[1], output[2], "max_number_of_injections", &analysis_config.max_number_of_injections) == 1)
			{
			}
			else if (case_insensitive_compare(output[3], "align_injections_to_tdc", MAX_LINE_SIZE) == 1)
			{
				if ((case_insensitive_compare(output[4], "1", MAX_LINE_SIZE) == 1) ||
					(case_insensitive_compare(output[4], "true", MAX_LINE_SIZE) == 1) ||
					(case_insensitive_compare(output[4], "yes", MAX_LINE_SIZE) == 1))
				{
					analysis_config.align_injections_to_tdc = true;
				}
				else if ((case_insensitive_compare(output[4], "0", MAX_LINE_SIZE) == 1) ||
					(case_insensitive_compare(output[4], "false", MAX_LINE_SIZE) == 1) ||
					(case_insensitive_compare(output[4], "no", MAX_LINE_SIZE) == 1))
				{
					analysis_config.align_injections_to_tdc = false;
				}
				else
				{
					logmessage(WARNING, "Unknown \"align_injections_to_tdc %s\"", output[4]);
				}
			}
			else if (case_insensitive_compare(output[3], "interpolate_mfb", MAX_LINE_SIZE) == 1)
			{
				if ((case_insensitive_compare(output[4], "1", MAX_LINE_SIZE) == 1) ||
					(case_insensitive_compare(output[4], "true", MAX_LINE_SIZE) == 1) ||
					(case_insensitive_compare(output[4], "yes", MAX_LINE_SIZE) == 1))
				{
					analysis_config.interpolate_mfb = true;
				}
				else if ((case_insensitive_compare(output[4], "0", MAX_LINE_SIZE) == 1) ||
					(case_insensitive_compare(output[4], "false", MAX_LINE_SIZE) == 1) ||
					(case_insensitive_compare(output[4], "no", MAX_LINE_SIZE) == 1))
				{
					analysis_config.interpolate_mfb = false;
				}
				else
				{
					logmessage(WARNING, "Unknown \"interpolate_mfb %s\"", output[4]);
				}
			}
			else if (case_insensitive_compare(output[3], "restrike_analysis", MAX_LINE_SIZE) == 1)
			{
				if ((case_insensitive_compare(output[4], "1", MAX_LINE_SIZE) == 1) ||
					(case_insensitive_compare(output[4], "true", MAX_LINE_SIZE) == 1) ||
					(case_insensitive_compare(output[4], "yes", MAX_LINE_SIZE) == 1))
				{
					analysis_config.restrike_analysis = true;
				}
				else if ((case_insensitive_compare(output[4], "0", MAX_LINE_SIZE) == 1) ||
					(case_insensitive_compare(output[4], "false", MAX_LINE_SIZE) == 1) ||
					(case_insensitive_compare(output[4], "no", MAX_LINE_SIZE) == 1))
				{
					analysis_config.restrike_analysis = false;
				}
				else
				{
					logmessage(WARNING, "Unknown \"restrike_analysis %s\"", output[4]);
				}
			}
			else if (set_float_parameter(output[1],output[2],"misfire_imep",&analysis_config.misfire_imep) == 1)
			{
			}
			else if (set_float_parameter(output[1],output[2],"slowburn_imep",&analysis_config.slowburn_imep) == 1)
			{
			}
			else if (set_float_parameter(output[1],output[2],"knock_pkp",&analysis_config.knock_pkp) == 1)
			{
			}
			else if (set_float_parameter(output[1], output[2], "mega_knock_pkp", &analysis_config.mega_knock_pkp) == 1)
			{
			}
			else if (set_float_parameter(output[1], output[2], "knock_kbf", &analysis_config.knock_kbf) == 1)
			{
			}
			else if (set_float_parameter(output[1], output[2], "mega_knock_kbf", &analysis_config.mega_knock_kbf) == 1)
			{
			}
			else if (set_float_parameter(output[1], output[2], "knock_knkbint", &analysis_config.knock_knkbint) == 1)
			{
			}
			else if (set_float_parameter(output[1], output[2], "knock_onset_threshold", &analysis_config.knock_onset_threshold) == 1)
			{
			}
			else if (set_float_parameter(output[1], output[2], "mega_knock_knkbint", &analysis_config.mega_knock_knkbint) == 1)
			{
			}
			else if (set_float_parameter(output[1],output[2],"tla_range",&analysis_config.tla_range) == 1)
			{
			}
			else if (set_float_parameter(output[1],output[2],"wiebe_a_start",&analysis_config.wiebe_a_start) == 1)
			{
			}
			else if (set_float_parameter(output[1],output[2],"wiebe_a_finish",&analysis_config.wiebe_a_finish) == 1)
			{
			}
			else if (set_float_parameter(output[1],output[2],"wiebe_a_step",&analysis_config.wiebe_a_step) == 1)
			{
			}
			else if (set_float_parameter(output[1],output[2],"wiebe_m_start",&analysis_config.wiebe_m_start) == 1)
			{
			}
			else if (set_float_parameter(output[1],output[2],"wiebe_m_finish",&analysis_config.wiebe_m_finish) == 1)
			{
			}
			else if (set_float_parameter(output[1],output[2],"wiebe_m_step",&analysis_config.wiebe_m_step) == 1)
			{
			}
			else if (set_float_parameter(output[1], output[2], "knock_hist_alarm_threshold_hist_factor", &analysis_config.knock_hist_alarm_threshold_hist_factor) == 1)
			{
			}
			else if (set_float_parameter(output[1], output[2], "knock_hist_alarm_threshold_pressure_factor", &analysis_config.knock_hist_alarm_threshold_pressure_factor) == 1)
			{
			}
			else if (set_float_parameter(output[1], output[2], "knock_hist_lower_threshold", &analysis_config.knock_hist_lower_threshold) == 1)
			{
			}
			else if (set_float_parameter(output[1], output[2], "knock_hist_shutdown_threshold_hist_factor", &analysis_config.knock_hist_shutdown_threshold_hist_factor) == 1)
			{
			}
			else if (set_float_parameter(output[1], output[2], "knock_hist_shutdown_threshold_pressure_factor", &analysis_config.knock_hist_shutdown_threshold_pressure_factor) == 1)
			{
			}
			else if (set_float_parameter(output[1], output[2], "knock_hist_time_cycle", &analysis_config.knock_hist_time_cycle) == 1)
			{
			}
			else if (set_uint_parameter(output[1], output[2], "knock_hist_min_cycles", &analysis_config.knock_hist_min_cycles) == 1)
			{
			}
			else if (set_uint_parameter(output[1],output[2],"knock_integral_type",&analysis_config.knock_integral_type) == 2)
			{
				if (case_insensitive_compare(output[2],"RECTIFIED_INTEGRAL",MAX_LINE_SIZE) == 1)
				{
					analysis_config.knock_integral_type = KNOCK_RECTIFIED_INTEGRAL;
				}
				else if (case_insensitive_compare(output[2],"SQUARED_INTEGRAL",MAX_LINE_SIZE) == 1)
				{
					analysis_config.knock_integral_type = KNOCK_SQUARED_INTEGRAL;
				}
				else
				{
					logmessage(WARNING,"Unknown knock integral type '%s'\n",output[2]);
				}
			}
			else if (set_uint_parameter(output[1],output[2],"engine_speed_type",&analysis_config.engine_speed_type) == 2)
			{
				if ((case_insensitive_compare(output[2],"NOTHING",MAX_LINE_SIZE) == 1) ||
				    (case_insensitive_compare(output[2],"NONE",MAX_LINE_SIZE) == 1))
				{
					analysis_config.engine_speed_type = N_NOTHING;
				}
				else if (case_insensitive_compare(output[2],"SPECIFY",MAX_LINE_SIZE) == 1)
				{
					analysis_config.engine_speed_type = N_SPECIFY;
				}
				else if (case_insensitive_compare(output[2],"CHANNEL_AVERAGE",MAX_LINE_SIZE) == 1)
				{
					analysis_config.engine_speed_type = N_AVGCHNL;
				}
	 			else if ((case_insensitive_compare(output[2],"AVL_RZT",MAX_LINE_SIZE) == 1) ||
				         (case_insensitive_compare(output[2],"IFILE",MAX_LINE_SIZE) == 1))
	 			{
					analysis_config.engine_speed_type = N_AVLRZT;
				}
				else
				{
					logmessage(WARNING,"Unknown engine speed type '%s'\n",output[2]);
				}
			}
			else if (set_uint_parameter(output[1], output[2], "heat_release_window", &analysis_config.heat_release_window) == 2)
			{
				if (case_insensitive_compare(output[2], "IVC_EVO", MAX_LINE_SIZE) == 1)
				{
					analysis_config.heat_release_window = HR_WINDOW_IVC_EVO;
				}
				else if (case_insensitive_compare(output[2], "SOC_EEOC", MAX_LINE_SIZE) == 1)
				{
					analysis_config.heat_release_window = HR_WINDOW_SOC_EEOC;
				}
				else
				{
					logmessage(WARNING, "Unknown heat release window type '%s'\n", output[2]);
				}
			}
			else if ((case_insensitive_compare(output[1],"start_of_combustion_start_window",MAX_LINE_SIZE) == 1) ||
					 (case_insensitive_compare(output[1],"start_of_combustion_finish_window",MAX_LINE_SIZE) == 1) ||
			         (case_insensitive_compare(output[1],"start_of_combustion_channel",MAX_LINE_SIZE) == 1) ||
			         (case_insensitive_compare(output[1],"start_of_combustion_type",MAX_LINE_SIZE) == 1) ||
 			         (case_insensitive_compare(output[1],"start_of_combustion",MAX_LINE_SIZE) == 1))
			{
				logmessage(WARNING,"Start of combustion no longer configured with analysis.  Use channel <n> soc * instead.\n");
			}
			else if (set_uint_parameter(output[1],output[2],"heat_release_model",&analysis_config.heat_release_model) == 2)
			{
				if (case_insensitive_compare(output[2],"FIRST_LAW",MAX_LINE_SIZE) == 1)
				{
					analysis_config.heat_release_model = HR_FIRSTLAW;
				}
				else if (case_insensitive_compare(output[2],"POLY_FIRST",MAX_LINE_SIZE) == 1)
				{
					analysis_config.heat_release_model = HR_POLYFIRST;
				}
				else if (case_insensitive_compare(output[2], "AVL_THERMO1", MAX_LINE_SIZE) == 1)
				{
					analysis_config.heat_release_model = HR_AVL_THERMO1;
				}
				else
				{
					logmessage(WARNING,"Unknown heat release model '%s'\n",output[2]);
				}
			}
			else if (set_uint_parameter(output[1],output[2],"heat_transfer_model",&analysis_config.heat_transfer_model) == 2)
			{
				if (case_insensitive_compare(output[2],"ANNAND",MAX_LINE_SIZE) == 1)
				{
					analysis_config.heat_transfer_model = HT_ANNAND;
				}
				else if (case_insensitive_compare(output[2],"WOSCHNI",MAX_LINE_SIZE) == 1)
				{
					analysis_config.heat_transfer_model = HT_WOSCHNI;
				}
	 			else if (case_insensitive_compare(output[2],"HOHENBERG",MAX_LINE_SIZE) == 1)
	 			{
					analysis_config.heat_transfer_model = HT_HOHENBERG;
				}
				else if (case_insensitive_compare(output[2],"EICHELBERG",MAX_LINE_SIZE) == 1)
				{
					analysis_config.heat_transfer_model = HT_EICHELBERG;
				}
	 			else if (case_insensitive_compare(output[2],"NUSSELT",MAX_LINE_SIZE) == 1)
	 			{
					analysis_config.heat_transfer_model = HT_NUSSELT;
				}
				else if (case_insensitive_compare(output[2],"BRILING",MAX_LINE_SIZE) == 1)
				{
					analysis_config.heat_transfer_model = HT_BRILING;
				}
				else if (case_insensitive_compare(output[2], "WOSCHNI_GT", MAX_LINE_SIZE) == 1)
				{
					analysis_config.heat_transfer_model = HT_WOSCHNI_GT;
				}
				else if (case_insensitive_compare(output[2], "ASSANIS", MAX_LINE_SIZE) == 1)
				{
					analysis_config.heat_transfer_model = HT_ASSANIS;
				}
				else
				{
					logmessage(WARNING,"Unknown heat transfer model '%s'\n",output[2]);
				}
			}
			else if (set_uint_parameter(output[1], output[2], "gas_temp_model", &analysis_config.gas_temp_model) == 2)
			{
				if (case_insensitive_compare(output[2], "REF_CA", MAX_LINE_SIZE) == 1)
				{
					analysis_config.gas_temp_model = GAS_TEMP_REF_CA;
				}
				else if (case_insensitive_compare(output[2], "IVC", MAX_LINE_SIZE) == 1)
				{
					analysis_config.gas_temp_model = GAS_TEMP_IVC;
				}
				else if (case_insensitive_compare(output[2], "AVL_THERMO1", MAX_LINE_SIZE) == 1)
				{
					analysis_config.gas_temp_model = GAS_TEMP_AVL_THERMO1;
				}
				else
				{
					logmessage(WARNING, "Unknown gas temperature model '%s'\n", output[2]);
				}
				}
			else if (case_insensitive_compare(output[1], "engine_speed_channel", MAX_LINE_SIZE) == 1)
			{
				result = sscanf(output[2], "%255s", (char*)&channel_name);

				if ((result > 0) && (channel_name_to_number(file, channel_name, ABSCISSA_UNKNOWN, &channel) == true))
				{
					analysis_config.engine_speed_channel = channel;
				}
				else if (set_uint_parameter(output[1], output[2], "engine_speed_channel", &analysis_config.engine_speed_channel) == 1)
				{
				}
				else
				{
					logmessage(WARNING, "Setting parameter %s to %s failed\n", output[1], output[2]);
				}
			}
			else if (set_float_parameter(output[1], output[2], "fft_reference_start", &analysis_config.fft_reference_start) == 1)
			{
			}
			else if (set_float_parameter(output[1], output[2], "fft_reference_finish", &analysis_config.fft_reference_start) == 1)
			{
			}
			else if (set_float_parameter(output[1], output[2], "crankcase_pressure", &analysis_config.crankcase_pressure) == 1)
			{
			}
			else if (set_float_parameter(output[1], output[2], "deactivation_delta_p", &analysis_config.deactivation_delta_p) == 1)
			{
			}
			else if (set_float_parameter(output[1], output[2], "d2p_window", &analysis_config.d2p_window) == 1)
			{
			}
			else if (set_float_parameter(output[1], output[2], "soc_threshold", &analysis_config.soc_threshold) == 1)
			{
			}
			else if (set_uint_parameter(output[1], output[2], "gamma_method", &analysis_config.gamma_method) == 2)
			{
				if (case_insensitive_compare(output[2], "BRUNT", MAX_LINE_SIZE) == 1)
				{
					analysis_config.gamma_method = GAMMA_BRUNT;
				}
				else if (case_insensitive_compare(output[2], "KULZER", MAX_LINE_SIZE) == 1)
				{
					analysis_config.gamma_method = GAMMA_KULZER;
				}
				else if (case_insensitive_compare(output[2], "MEASURED_MEAN", MAX_LINE_SIZE) == 1)
				{
					analysis_config.gamma_method = GAMMA_MEASURED_MEAN;
				}
				else if (case_insensitive_compare(output[2], "MEASURED", MAX_LINE_SIZE) == 1)
				{
					analysis_config.gamma_method = GAMMA_MEASURED;
				}
				else if (case_insensitive_compare(output[2], "AIR", MAX_LINE_SIZE) == 1)
				{
					analysis_config.gamma_method = GAMMA_AIR;
				}
				else if (case_insensitive_compare(output[2], "SPECIFIED", MAX_LINE_SIZE) == 1)
				{
					analysis_config.gamma_method = GAMMA_SPECIFIED;
				}
				else if (case_insensitive_compare(output[2], "INDOLENE", MAX_LINE_SIZE) == 1)
				{
					analysis_config.gamma_method = GAMMA_INDOLENE;
				}
				else if (case_insensitive_compare(output[2], "PROPANE", MAX_LINE_SIZE) == 1)
				{
					analysis_config.gamma_method = GAMMA_PROPANE;
				}
				else if (case_insensitive_compare(output[2], "HAYES", MAX_LINE_SIZE) == 1)
				{
					analysis_config.gamma_method = GAMMA_HAYES;
				}
				else if (case_insensitive_compare(output[2], "CHANG", MAX_LINE_SIZE) == 1)
				{
					analysis_config.gamma_method = GAMMA_CHANG;
				}
				else if (case_insensitive_compare(output[2], "AVL_THERM1_FIXED", MAX_LINE_SIZE) == 1)
				{
					analysis_config.gamma_method = GAMMA_AVL_THERM1_FIXED;
				}
				else if (case_insensitive_compare(output[2], "AVL_THERM1_CALC", MAX_LINE_SIZE) == 1)
				{
					analysis_config.gamma_method = GAMMA_AVL_THERM1_CALC;
				}
				else if (case_insensitive_compare(output[2], "AVL_THERM1_A", MAX_LINE_SIZE) == 1)
				{
					analysis_config.gamma_method = GAMMA_AVL_THERM1_A;
				}
				else
				{
					logmessage(WARNING, "Unknown gamma method '%s'\n", output[2]);
				}
			}
			else if (set_uint_parameter(output[1], output[2], "pressure_rise_method", &analysis_config.pressure_rise_method) == 2)
			{
				if (case_insensitive_compare(output[2], "DERIVATIVE", MAX_LINE_SIZE) == 1)
				{
					analysis_config.pressure_rise_method = PRR_DERIVATIVE;
				}
				else if (case_insensitive_compare(output[2], "SAVITZKY_GOLAY", MAX_LINE_SIZE) == 1)
				{
					analysis_config.pressure_rise_method = PRR_SAVITZKY_GOLAY;
				}
				else
				{
					logmessage(WARNING, "Unknown pressure rise method '%s'\n", output[2]);
				}
			}
			else if (set_uint_parameter(output[1], output[2], "imep_method", &analysis_config.imep_method) == 2)
			{
				if (case_insensitive_compare(output[2], "ACTUAL_PRESSURE", MAX_LINE_SIZE) == 1)
				{
					analysis_config.imep_method = IMEP_ACTUAL_PRESSURE;
				}
				else if (case_insensitive_compare(output[2], "MEAN_PRESSURE", MAX_LINE_SIZE) == 1)
				{
					analysis_config.imep_method = IMEP_MEAN_PRESSURE;
				}
				else if (case_insensitive_compare(output[2], "MEAN_PRESSURE_1DEG", MAX_LINE_SIZE) == 1)
				{
					analysis_config.imep_method = IMEP_MEAN_PRESSURE_1DEG;
				}
				else
				{
					logmessage(WARNING, "Unknown IMEP method '%s'\n", output[2]);
				}
			}
			else if (set_uint_parameter(output[1], output[2], "motored_pressure_method", &analysis_config.motored_pressure_method) == 2)
			{
				if (case_insensitive_compare(output[2], "FIXED_N", MAX_LINE_SIZE) == 1)
				{
					analysis_config.motored_pressure_method &= ~0x00000001;
				}
				else if (case_insensitive_compare(output[2], "POLY_COMP", MAX_LINE_SIZE) == 1)
				{
					analysis_config.motored_pressure_method |= 0x00000001;
				}
				else if (case_insensitive_compare(output[2], "IVC", MAX_LINE_SIZE) == 1)
				{
					analysis_config.motored_pressure_method &= ~0x00000006;
				}
				else if (case_insensitive_compare(output[2], "POLY_FINISH", MAX_LINE_SIZE) == 1)
				{
					analysis_config.motored_pressure_method &= ~0x00000004;
					analysis_config.motored_pressure_method |= 0x00000002;
				}
				else if (case_insensitive_compare(output[2], "BDC", MAX_LINE_SIZE) == 1)
				{
					analysis_config.motored_pressure_method &= ~0x00000002;
					analysis_config.motored_pressure_method |= 0x00000004;
				}
				else
				{
					logmessage(WARNING, "Unknown motored pressure method '%s'\n", output[2]);
				}
			}
			else if (set_uint_parameter(output[1], output[2], "smoothed_pressure_method", &analysis_config.smoothed_pressure_method) == 2)
			{
				if (case_insensitive_compare(output[2], "MOVING_AVERAGE", MAX_LINE_SIZE) == 1)
				{
					analysis_config.smoothed_pressure_method = SMOOTH_MOVING_AVERAGE;
				}
				else if (case_insensitive_compare(output[2], "SAVITZKY_GOLAY", MAX_LINE_SIZE) == 1)
				{
					analysis_config.smoothed_pressure_method = SMOOTH_SAVITZKY_GOLAY;
				}
				else
				{
					logmessage(WARNING, "Unknown smoothed pressure method '%s'\n", output[2]);
				}
			}
			else if (set_uint_parameter(output[1], output[2], "stats_type", &analysis_config.stats_type) == 2)
			{
				if (case_insensitive_compare(output[2], "RATIO", MAX_LINE_SIZE) == 1)
				{
					analysis_config.stats_type = STATS_RATIO;
				}
				else if (case_insensitive_compare(output[2], "PERCENT", MAX_LINE_SIZE) == 1)
				{
					analysis_config.stats_type = STATS_PERCENT;
				}
				else
				{
					logmessage(WARNING, "Unknown stats_type '%s'\n", output[2]);
				}
			}
			else if (set_uint_parameter(output[1], output[2], "misfire_cycles_f", &analysis_config.misfire_cycles_f) == 1)
			{
			}
			else if (set_uint_parameter(output[1], output[2], "misfire_cycles_m", &analysis_config.misfire_cycles_m) == 1)
			{
			}
			else if (set_uint_parameter(output[1], output[2], "misfire_cycles_s", &analysis_config.misfire_cycles_s) == 1)
			{
			}
			else if (set_float_parameter(output[1], output[2], "knock_boss_start_window", &analysis_config.knock_boss_start_window) == 1)
			{
			}
			else if (set_float_parameter(output[1], output[2], "knock_boss_finish_window", &analysis_config.knock_boss_finish_window) == 1)
			{
			}
			else if (set_float_parameter(output[1], output[2], "knock_boss_reference_start_window", &analysis_config.knock_boss_reference_start_window) == 1)
			{
			}
			else if (set_float_parameter(output[1], output[2], "knock_boss_reference_finish_window", &analysis_config.knock_boss_reference_finish_window) == 1)
			{
			}
			else if (set_float_parameter(output[1], output[2], "fft_stats_maximum", &analysis_config.fft_stats_maximum) == 1)
			{
			}
			else if (set_float_parameter(output[1], output[2], "fft_stats_resolution", &analysis_config.fft_stats_resolution) == 1)
			{
			}
			else if (set_uint_parameter(output[1], output[2], "eeoc_method", &analysis_config.eeoc_method) == 2)
			{
				if (case_insensitive_compare(output[2], "BRUNT", MAX_LINE_SIZE) == 1)
				{
					analysis_config.eeoc_method = EEOC_METHOD_BRUNT;
				}
				else if (case_insensitive_compare(output[2], "FIXED", MAX_LINE_SIZE) == 1)
				{
					analysis_config.eeoc_method = EEOC_METHOD_FIXED;
				}
				else
				{
					logmessage(WARNING, "Unknown smoothed pressure method '%s'\n", output[2]);
				}
			}
			else if (set_float_parameter(output[1], output[2], "eeoc_default", &analysis_config.eeoc_default) == 1)
			{
			}
			else if (set_float_parameter(output[1], output[2], "soc_default", &analysis_config.soc_default) == 1)
			{
			}
			else
			{
				logmessage(WARNING,"Setting parameter %s to %s failed\n",output[1],output[2]);
			}
		}
		else if (case_insensitive_compare(output[0],"output-file-type",MAX_LINE_SIZE) == 1)
		{
            logmessage(NOTICE,"Setting output filetype to %s\n",output[1]);
			
			if (set_uint_parameter(output[0],output[1],"output-file-type",&output_file_type) == 2)
			{
				if ((case_insensitive_compare(output[1],"AVL_IFILE",MAX_LINE_SIZE) == 1) || 
				    (case_insensitive_compare(output[1],"AVL-IFILE",MAX_LINE_SIZE) == 1) || 
					(case_insensitive_compare(output[1],"AVL",MAX_LINE_SIZE) == 1) || 
					(case_insensitive_compare(output[1],"IFILE",MAX_LINE_SIZE) == 1))
				{
					output_file_type = FILE_AVLIFILE;
				}
	 			else if (case_insensitive_compare(output[1],"CSV",MAX_LINE_SIZE) == 1)
	 			{
					output_file_type = FILE_CSV;
				}
				else if (case_insensitive_compare(output[1],"CAS",MAX_LINE_SIZE) == 1)
				{
					output_file_type = FILE_CAS;
				}
				else if (case_insensitive_compare(output[1],"MATLAB",MAX_LINE_SIZE) == 1)
				{
					output_file_type = FILE_MATLAB;
				}
				else if (case_insensitive_compare(output[1], "ATIMAT", MAX_LINE_SIZE) == 1)
				{
					output_file_type = FILE_ATIMAT;
				}
				else if (case_insensitive_compare(output[1],"TDMS",MAX_LINE_SIZE) == 1)
				{
					output_file_type = FILE_TDMS;
				}
	 			else if (case_insensitive_compare(output[1],"YOKOGAWA",MAX_LINE_SIZE) == 1)
	 			{
					output_file_type = FILE_YOKOGAWA;
				}
				else if (case_insensitive_compare(output[1], "SDF", MAX_LINE_SIZE) == 1)
				{
					output_file_type = FILE_SDF;
				}
				else
				{
					logmessage(WARNING,"Unknown output filetype '%s'\n",output[1]);
				}
			}
		}
		else if (case_insensitive_compare(output[0],"output-file",MAX_LINE_SIZE) == 1)
		{
#ifdef _CATOOL_UNICODE_
			size_t newsize = strlen(output[1]) + 1;
			wchar_t* wcstring = new wchar_t[newsize];
			size_t convertedChars = 0;
			mbstowcs_s(&convertedChars, wcstring, newsize, output[1], _TRUNCATE);
#endif

#ifdef _CATOOL_UNICODE_
 			if (wcsncmp(wcstring, input_filename, MAX_LINE_SIZE) == 0)
#else
			if (strncmp(output[1], input_filename, MAX_LINE_SIZE) == 0)
#endif
			{
				if (case_insensitive_compare(output[2], "force-overwrite", MAX_LINE_SIZE) == 1)
				{
#ifdef _CATOOL_UNICODE_
					wcsncpy(output_filename, wcstring, 4095);
					output_filename[4095] = 0x0000;
#else
					strncpy(output_filename, output[1], 4095);
					output_filename[4095] = 0x00;
#endif

					logmessage(NOTICE, "Set output filename to %s\n", output_filename);
				}
				else
				{
					logmessage(MSG_ERROR, "Cannot set input (%s) and output (%s) filenames the same.\nUse 'output-file <filename> force-overwrite' to force.\n", input_filename, output[1]);

#ifdef _CATOOL_UNICODE_
					output_filename[0] = 0x0000;
#else
					output_filename[0] = 0x00;
#endif
				}
			}
			else
			{
#ifdef _CATOOL_UNICODE_
				wcsncpy(output_filename, wcstring, 4095);
				output_filename[4095] = 0x0000;
#else
				strncpy(output_filename, output[1], 4095);
				output_filename[4095] = 0x00;
#endif
            
				logmessage(NOTICE,"Set output filename to %s\n",output_filename);
			}

#ifdef _CATOOL_UNICODE_
			delete[] wcstring;
#endif
		}
		else if (case_insensitive_compare(output[0],"output-data",MAX_LINE_SIZE) == 1)
		{
			if (number_of_arguments > 1)
			{
				output_data = 0;

				ca_to_time_warning = false;

				for (argument=1;argument<number_of_arguments;argument++)
				{
					logmessage(NOTICE,"Setting output-data to include %s\n",output[argument]);
					
					if (set_uint_parameter(output[0],output[argument],"output-data",&output_data) == 2)
					{
						if (case_insensitive_compare(output[argument],"CA_RAW",MAX_LINE_SIZE) == 1)
						{
							output_data |= IFILE_CA_RAW;
						}
						else if (case_insensitive_compare(output[argument], "-CA_RAW", MAX_LINE_SIZE) == 1)
						{
							output_data &= ~IFILE_CA_RAW;
						}
			 			else if (case_insensitive_compare(output[argument],"CA",MAX_LINE_SIZE) == 1)
			 			{
							output_data |= IFILE_CA;
						}
						else if (case_insensitive_compare(output[argument], "-CA", MAX_LINE_SIZE) == 1)
						{
							output_data &= ~IFILE_CA;
						}
						else if (case_insensitive_compare(output[argument], "CA_MEAN", MAX_LINE_SIZE) == 1)
						{
							output_data |= IFILE_CA_MEAN;
						}
						else if (case_insensitive_compare(output[argument], "-CA_MEAN", MAX_LINE_SIZE) == 1)
						{
							output_data &= ~IFILE_CA_MEAN;
						}
						else if (case_insensitive_compare(output[argument],"CA_TO_TIME",MAX_LINE_SIZE) == 1)
						{
							ca_to_time_warning = true;
						}
						else if (case_insensitive_compare(output[argument], "-CA_TO_TIME", MAX_LINE_SIZE) == 1)
						{
							ca_to_time_warning = true;
						}
						else if (case_insensitive_compare(output[argument],"TIME",MAX_LINE_SIZE) == 1)
						{
							output_data |= IFILE_TIME;
						}
						else if (case_insensitive_compare(output[argument], "-TIME", MAX_LINE_SIZE) == 1)
						{
							output_data &= ~IFILE_TIME;
						}
						else if (case_insensitive_compare(output[argument],"RTP_RESULTS",MAX_LINE_SIZE) == 1)
						{
							output_data |= IFILE_RTP_RESULTS;
						}
						else if (case_insensitive_compare(output[argument], "-RTP_RESULTS", MAX_LINE_SIZE) == 1)
						{
							output_data &= ~IFILE_RTP_RESULTS;
						}
						else if (case_insensitive_compare(output[argument],"RESULTS_RAW",MAX_LINE_SIZE) == 1)
						{
							output_data |= IFILE_RESULTS_RAW;
						}
						else if (case_insensitive_compare(output[argument], "-RESULTS_RAW", MAX_LINE_SIZE) == 1)
						{
							output_data &= ~IFILE_RESULTS_RAW;
						}
						else if (case_insensitive_compare(output[argument],"RESULTS",MAX_LINE_SIZE) == 1)
						{
							output_data |= IFILE_RESULTS;
						}
						else if (case_insensitive_compare(output[argument], "-RESULTS", MAX_LINE_SIZE) == 1)
						{
							output_data &= ~IFILE_RESULTS;
						}
						else if (case_insensitive_compare(output[argument], "RESULTS_STATS", MAX_LINE_SIZE) == 1)
						{
							output_data |= IFILE_RESULTS_STATS;
						}
						else if (case_insensitive_compare(output[argument], "-RESULTS_STATS", MAX_LINE_SIZE) == 1)
						{
							output_data &= ~IFILE_RESULTS_STATS;
						}
			 			else if (case_insensitive_compare(output[argument],"ASYNC_UTC",MAX_LINE_SIZE) == 1)
			 			{
							output_data |= IFILE_ASYNC_UTC;
						}
						else if (case_insensitive_compare(output[argument], "-ASYNC_UTC", MAX_LINE_SIZE) == 1)
						{
							output_data &= ~IFILE_ASYNC_UTC;
						}
						else if (case_insensitive_compare(output[argument],"ALL",MAX_LINE_SIZE) == 1)
			 			{
							output_data |= IFILE_CA_RAW;
							output_data |= IFILE_CA;
							output_data |= IFILE_TIME;
							output_data |= IFILE_RTP_RESULTS;
							output_data |= IFILE_RESULTS;
							output_data |= IFILE_RESULTS_RAW;
						}
						else
						{
							logmessage(WARNING,"Unknown output-data type '%s'\n",output[argument]);
						}
					}
				}

				logmessage(NOTICE,"Set output-data type to %u\n",output_data);

				if (ca_to_time_warning == true)
				{
					logmessage(WARNING, "CA_TO_TIME no longer supported.  Use channel <channel-name/#channel-number> ca-to-time <resolution> instead.\n");
				}
			}
			else
			{
				logmessage(WARNING,"No output-data type specified\n");
			}
		}
		else if (case_insensitive_compare(output[0],"input-data",MAX_LINE_SIZE) == 1)
		{
			if (number_of_arguments > 1)
			{
				ifile_input = 0;
				
				for (argument=1;argument<number_of_arguments;argument++)
				{
					logmessage(NOTICE,"Setting input-data to include %s\n",output[argument]);
					
					if (set_uint_parameter(output[0],output[argument],"input-data",&ifile_input) == 2)
					{
						if (case_insensitive_compare(output[argument],"CA",MAX_LINE_SIZE) == 1)
						{
							ifile_input |= IFILE_CA;
						}
						else if (case_insensitive_compare(output[argument], "-CA", MAX_LINE_SIZE) == 1)
						{
							ifile_input &= ~IFILE_CA;
						}
						else if (case_insensitive_compare(output[argument],"TIME",MAX_LINE_SIZE) == 1)
						{
							ifile_input |= IFILE_TIME;
						}
						else if (case_insensitive_compare(output[argument], "-TIME", MAX_LINE_SIZE) == 1)
						{
							ifile_input &= ~IFILE_TIME;
						}
						else if (case_insensitive_compare(output[argument],"RTP_RESULTS",MAX_LINE_SIZE) == 1)
						{
							ifile_input |= IFILE_RTP_RESULTS;
						}
						else if (case_insensitive_compare(output[argument], "-RTP_RESULTS", MAX_LINE_SIZE) == 1)
						{
							ifile_input &= ~IFILE_RTP_RESULTS;
						}
						else if (case_insensitive_compare(output[argument],"RESULTS_RAW",MAX_LINE_SIZE) == 1)
						{
							ifile_input |= IFILE_RESULTS_RAW;
						}
						else if (case_insensitive_compare(output[argument], "-RESULTS_RAW", MAX_LINE_SIZE) == 1)
						{
							ifile_input &= ~IFILE_RESULTS_RAW;
						}
						else if (case_insensitive_compare(output[argument], "RESULTS",MAX_LINE_SIZE) == 1)
						{
							ifile_input |= IFILE_RESULTS;
						}
						else if (case_insensitive_compare(output[argument], "-RESULTS", MAX_LINE_SIZE) == 1)
						{
							ifile_input &= ~IFILE_RESULTS;
						}
			 			else if (case_insensitive_compare(output[argument],"ASYNC_UTC",MAX_LINE_SIZE) == 1)
			 			{
							ifile_input |= IFILE_ASYNC_UTC;
						}
						else if (case_insensitive_compare(output[argument], "-ASYNC_UTC", MAX_LINE_SIZE) == 1)
						{
							ifile_input &= ~IFILE_ASYNC_UTC;
						}
						else if (case_insensitive_compare(output[argument],"ALL",MAX_LINE_SIZE) == 1)
						{
							ifile_input |= IFILE_CA;
							ifile_input |= IFILE_TIME;
							ifile_input |= IFILE_RTP_RESULTS;
							ifile_input |= IFILE_RESULTS;
							ifile_input |= IFILE_RESULTS_RAW;
							ifile_input |= IFILE_ASYNC_UTC;
						}
						else
						{
							logmessage(WARNING,"Unknown input-data type '%s'\n",output[argument]);
						}
					}
				}

				logmessage(NOTICE,"Set input-data type to %u\n",ifile_input);
			}
			else
			{
				logmessage(WARNING,"No input-data type specified\n");
			}
		}
		else if (case_insensitive_compare(output[0],"analyse",MAX_LINE_SIZE) == 1)
		{
			if (number_of_arguments > 1)
			{
				for (argument=1;argument<number_of_arguments;argument++)
				{
					if (case_insensitive_compare(output[argument],"all",MAX_LINE_SIZE) == 1)
					{
						for (rqst=0;rqst<get_number_of_analysis_channels();rqst++)
						{
					 		analysis_rqst[rqst] = AnalysisRqst::CalculateAndStore;
						}

						/* Disable FFT */

						analysis_rqst[FFT] = AnalysisRqst::NotCalculated;
						analysis_rqst[FFT_FREQUENCY] = AnalysisRqst::NotCalculated;

						analysis_rqst[FFT_REFERENCE] = AnalysisRqst::NotCalculated;
						analysis_rqst[FFT_REFERENCE_FREQUENCY] = AnalysisRqst::NotCalculated;

						analysis_rqst[FFT_KNOCK_PRESSURE] = AnalysisRqst::NotCalculated;
						analysis_rqst[FFT_KNOCK_PRESSURE_FREQUENCY] = AnalysisRqst::NotCalculated;

						analysis_rqst[FFT_TORQUE_SUM] = AnalysisRqst::NotCalculated;
						analysis_rqst[FFT_TORQUE_SUM_FREQUENCY] = AnalysisRqst::NotCalculated;

						/* Wiebe very slow */

						analysis_rqst[WIEBE_A] = AnalysisRqst::NotCalculated;
						analysis_rqst[WIEBE_M] = AnalysisRqst::NotCalculated;
						analysis_rqst[WIEBE_MFB] = AnalysisRqst::NotCalculated;
					}
				 	else if (case_insensitive_compare(output[argument],"none",MAX_LINE_SIZE) == 1)
				 	{
						for (rqst=0;rqst<get_number_of_analysis_channels();rqst++)
						{
							analysis_rqst[rqst] = AnalysisRqst::NotCalculated;
						}
					}
					else
					{
						set_analysis(output[argument], analysis_rqst);
					}
				}
			}
		}
		else if (case_insensitive_compare(output[0], "run-offset-correction", MAX_LINE_SIZE) == 1)
		{
			if (file->number_of_channels == 0)
			{
				logmessage(MSG_ERROR, "There are no channels loaded to perform any offset correction\n");
			}
			else if (file->preload == false)
			{
				logmessage(NOTICE, "Specify the file to load with 'load-file' before running offset correction\n");
			}
			else if (file->channel_data == NULL)
			{
				logmessage(NOTICE, "No channels have been loaded to perform offset correction\n");
			}
			else
			{
				for (channel = 0; channel < file->number_of_channels; channel++)
				{
					if ((file->channel_data[channel].file_flag == true) && (file->channel_data[channel].loaded == true))
					{
						if (file->channel_data[channel].abscissa.type == ABSCISSA_CRANKANGLE)
						{
							offset_correction_channel(file, NULL, channel);
						}
					}
				}
			}
		}
		else if ((case_insensitive_compare(output[0], "run-analysis",MAX_LINE_SIZE) == 1) ||
		         (case_insensitive_compare(output[0], "configure-analysis", MAX_LINE_SIZE) == 1))
		{
		    if (file->number_of_channels == 0)
		    {
				logmessage(MSG_ERROR,"There are no channels loaded to perform any analysis\n");
			}
			else if (file->preload == false)
			{
				logmessage(NOTICE,"Specify the file to load with 'input-file' before running analysis\n");
			}
			else if (file->channel_data == NULL)
			{
				logmessage(NOTICE,"No channels have been loaded to analyse\n");
			}
		    else
			{
				if (analysis != NULL)
				{
					logmessage(NOTICE, "Deleting %u old analysis structures\n", number_of_analysis_structures);

					Delete_Analysis(analysis);
					free(analysis);
					analysis = NULL;

					logmessage(DEBUG, "Analysis structures deleted\n");
				}

				Verify_File_Configuration(file,analysis);

				number_of_analysis_structures = 0;
		       
			   	for (channel=0;channel<file->number_of_channels;channel++)
			   	{
					if ((file->channel_data[channel].loaded == true) && (file->channel_data[channel].file_flag == true))
					{
						number_of_analysis_structures += 1;
					}
				}

				if (number_of_analysis_structures == 0)
				{
					logmessage(MSG_ERROR,"No channels are loaded requiring analysis\n");
				}
				else
				{
					logmessage(NOTICE,"Analysing %u channels\n",number_of_analysis_structures);

					analysis = (Analysis*)malloc(sizeof(Analysis));
					if (analysis == NULL)
					{
						logmessage(FATAL,"Memory could not be allocated\n");
					}

					analysis->number_of_channels = number_of_analysis_structures;
					analysis->channel = (AnalysisChannel*)malloc(number_of_analysis_structures * sizeof(AnalysisChannel));
					if (analysis->channel == NULL)
					{
						logmessage(FATAL,"Memory could not be allocated\n");
					}
					analysis->number_of_groups = 0;
					analysis->group = NULL;

					logmessage(DEBUG,"Configuring analysis structures\n");		
					
					analysis_structure = 0;
					configured_ok = true;
				    for (channel=0;channel<file->number_of_channels;channel++)
				  	{
						if ((file->channel_data[channel].file_flag == true) && (file->channel_data[channel].loaded == true))
						{
							/* Setup analysis structure */
						    
						    logmessage(DEBUG,"Configuring analysis structure %u\n",channel);
						    
							analysis->channel[analysis_structure].analysis_rqst = (AnalysisRqst*)malloc(get_number_of_analysis_channels() * sizeof(AnalysisRqst));

							if (analysis->channel[analysis_structure].analysis_rqst == NULL)
							{
								logmessage(FATAL, "Memory could not be allocated\n");
							}

						    analysis->channel[analysis_structure].number_of_cycles = file->number_of_cycles;
						    analysis->channel[analysis_structure].channel = channel;
						    
							analysis->channel[analysis_structure].results = NULL;
							analysis->channel[analysis_structure].cycle_classification = NULL;
							analysis->channel[analysis_structure].statistics = NULL;
							analysis->channel[analysis_structure].ca_statistics = NULL;
							Zero_Statistics(&analysis->channel[analysis_structure].raw_statistics);
							Zero_StatisticsArray(&analysis->channel[analysis_structure].raw_ca_statistics);

							if (file->channel_data[analysis->channel[analysis_structure].channel].abscissa.type == ABSCISSA_CRANKANGLE)
							{
								Allocate_StatisticsArray(&analysis->channel[analysis_structure].raw_ca_statistics, file->channel_data[channel].samples_per_cycle);
							}
							else if (file->channel_data[analysis->channel[analysis_structure].channel].abscissa.type == ABSCISSA_FREQUENCY)
							{
								unsigned int number_of_bins = (unsigned int)(analysis->channel[analysis_structure].config.fft_stats_maximum / analysis->channel[analysis_structure].config.fft_stats_resolution) + 1;

								Allocate_StatisticsArray(&analysis->channel[analysis_structure].raw_ca_statistics, number_of_bins);
							}
							else
							{
								/* Do Nothing */
							}

							for (rqst=0;rqst<get_number_of_analysis_channels();rqst++)
							{
								analysis->channel[analysis_structure].analysis_rqst[rqst] = analysis_rqst[rqst];
							}

							Copy_Analysis(&analysis_config, &analysis->channel[analysis_structure].config);

							validate_analysis_requests(file, &analysis->channel[analysis_structure]);

							logmessage(DEBUG,"Configuring analysis structure channels\n");
								
							if (configure_analysis_channels(file, &analysis->channel[analysis_structure]) == false)
							{
								memset(analysis->channel[analysis_structure].analysis_rqst, 0, get_number_of_analysis_channels() * sizeof(AnalysisRqst));

								configured_ok = false;
							}
							
							analysis_structure += 1;
						}
					}

					if (configured_ok == false)
					{
						logmessage(WARNING, "It was not possible to allocate memory for all the analysis calculations.  Some or all will be ommited.\n");
					}
			   	    
			   	    logmessage(DEBUG,"Configuring channel groups...\n");
			   	    
					calculate_channel_groups(file, analysis);
					
					logmessage(DEBUG,"done\n");

					logmessage(DEBUG, "Generating MATLAB compatible channels names\n");

					generate_matlab_channelnames(NULL, analysis);

					if (case_insensitive_compare(output[0], "run-analysis", MAX_LINE_SIZE) == 1)
					{
						analyse_channels(file, analysis, error_check);

						/* calculate_group_stats(analysis,file->number_of_cycles - 1,file->number_of_cycles,STATS_ALL,2,file->min_res_ppr,file->engine.number_of_strokes); */
					}
				}
			}
		}
		else if (case_insensitive_compare(output[0], "output-variable", MAX_LINE_SIZE) == 1)
		{
			if ((case_insensitive_compare(output[1], "CA", MAX_LINE_SIZE) == 1) || (case_insensitive_compare(output[1], "CA_RAW", MAX_LINE_SIZE) == 1))
			{
				if (case_insensitive_compare(output[2], "FLOAT", MAX_LINE_SIZE) == 1)
				{
					grpflg_float |= IFILE_CA;
				}
				else if (case_insensitive_compare(output[2], "INTEGER", MAX_LINE_SIZE) == 1)
				{
					grpflg_float &= ~IFILE_CA;
				}
				else
				{
					logmessage(MSG_ERROR, "Unknown output-variable type %s\n", output[2]);
				}
			}
			else if (case_insensitive_compare(output[1], "TIME", MAX_LINE_SIZE) == 1)
			{
				if (case_insensitive_compare(output[2], "FLOAT", MAX_LINE_SIZE) == 1)
				{
					grpflg_float |= IFILE_TIME;
				}
				else if (case_insensitive_compare(output[2], "INTEGER", MAX_LINE_SIZE) == 1)
				{
					grpflg_float &= ~IFILE_TIME;
				}
				else
				{
					logmessage(MSG_ERROR, "Unknown output-variable type %s\n", output[2]);
				}
			}
			else if ((case_insensitive_compare(output[1], "RESULTS", MAX_LINE_SIZE) == 1) || (case_insensitive_compare(output[1], "RESULTS_RAW", MAX_LINE_SIZE) == 1))
			{
				if (case_insensitive_compare(output[2], "FLOAT", MAX_LINE_SIZE) == 1)
				{
					grpflg_float |= IFILE_RESULTS;
				}
				else if (case_insensitive_compare(output[2], "INTEGER", MAX_LINE_SIZE) == 1)
				{
					grpflg_float &= ~IFILE_RESULTS;
				}
				else
				{
					logmessage(MSG_ERROR, "Unknown output-variable type %s\n", output[2]);
				}
			}
			else
			{
				logmessage(MSG_ERROR, "Unknown output-variable datagroup %s\n", output[1]);
			}
		}
		else if (case_insensitive_compare(output[0],"output",MAX_LINE_SIZE) == 1)
		{
	    	/* Output file */
	    	
	    	logmessage(NOTICE,"Outputting data to file\n");
	    	
			if (output_filename[0] == 0x00)
			{
				logmessage(MSG_ERROR,"No output filename specified\n");
			}
		    else if (file->number_of_channels == 0)
		    {
				logmessage(MSG_ERROR,"There are no channels loaded to output anything\n");
			}
			else
			{
				if ((number_of_arguments == 1) || (case_insensitive_compare(output[1], "all", MAX_LINE_SIZE) == 1))
				{
					/* Do Nothing - output all cycles */
				}
				else 
				{
					cycles_to_output = (unsigned int*)malloc(file->number_of_cycles*sizeof(unsigned int));
					if (cycles_to_output == NULL)
					{
						logmessage(FATAL, "Memory could not be allocated\n");
					}

					memset(cycles_to_output, 0, file->number_of_cycles*sizeof(unsigned int));

					if (determine_range(output[1], strlen(output[1]), cycles_to_output, file->number_of_cycles) == true)
					{
						number_of_new_cycles = 0;

						for (cycle = 0; cycle < file->number_of_cycles; cycle++)
						{
							number_of_new_cycles += cycles_to_output[cycle];
						}

						if (number_of_new_cycles > 0)
						{
							Select_Cycles(file, analysis, cycles_to_output);

							logmessage(NOTICE, "Number of cycles reduced to %u\n", number_of_new_cycles);
						}
					}
					else
					{
						logmessage(WARNING, "Could not determine cycles to output.  Outputting all cycles\n");
					}

					free(cycles_to_output);
					cycles_to_output = NULL;
				}

				switch (output_file_type)
				{
				case FILE_AVLIFILE:
				{
		        	save_ifile(file,output_filename,analysis,output_data,grpflg_float);
				    break;
				}
				case FILE_CSV:
				{
					save_csv_file(file,output_filename,analysis,output_data);
					break;
		 		}
		 		case FILE_MATLAB:
				{
					save_matlab_file(file,output_filename,analysis,output_data);
					break;
				}
				case FILE_ATIMAT:
				{
					save_atimat_file(file,output_filename,analysis,output_data);
					break;
				}
				case FILE_SDF:
				{
					save_sdf_file(file, output_filename);
					break;
				}
				case FILE_CAS:
				case FILE_TDMS:
				case FILE_YOKOGAWA:
				default:
				{
					logmessage(MSG_ERROR,"Output filetype not currently supported for output\n");

				    break;
				}
				}
			}
		}
		else if (case_insensitive_compare(output[0],"ifile-output",MAX_LINE_SIZE) == 1)
		{
			logmessage(MSG_ERROR,"'ifile-output' parameter is depreciated.  Use 'output-data' instead.\n");
		}
		else if (case_insensitive_compare(output[0],"output-cyc",MAX_LINE_SIZE) == 1)
		{
			logmessage(MSG_ERROR,"'output-cyc' parameter is depreciated.  Use 'analyse' instead.\n");
		}
		else if (case_insensitive_compare(output[0],"output-ca",MAX_LINE_SIZE) == 1)
		{
			logmessage(MSG_ERROR,"'output-ca' parameter is depreciated.  Use 'analyse' instead.\n");
		}
		else if (case_insensitive_compare(output[0],"print",MAX_LINE_SIZE) == 1)
		{
			logmessage(MSG_ERROR,"%s\n",output[1]);	
		}
		else if ((case_insensitive_compare(output[0],"stop",MAX_LINE_SIZE) == 1) || 
			     (case_insensitive_compare(output[0],"exit",MAX_LINE_SIZE) == 1) || 
				 (case_insensitive_compare(output[0],"quit",MAX_LINE_SIZE) == 1))
		{
			done = true;
		}
		else if (case_insensitive_compare(output[0],"threads",MAX_LINE_SIZE) == 1)
		{
			if (set_uint_parameter(output[0],output[1],"threads",&file->threads) == 1)
			{
				logmessage(NOTICE,"Set number of threads to use to %u\n",file->threads);
			}
			else
			{
				logmessage(NOTICE,"Syntax: threads <n>\n");
			}
		}
#if defined(_WIN32) || defined(_WIN64)
		else if (case_insensitive_compare(output[0], "plugin", MAX_LINE_SIZE) == 1)
		{
			char dll_filename[MAX_PATH];

			strncpy(dll_filename, output[1], MAX_PATH);
			dll_filename[MAX_PATH - 1] = 0x00;

			HINSTANCE plugin = LoadLibraryA(dll_filename);
			
			if (plugin != NULL)
			{
				RegisterAnalysisChannelsFunction RegisterAnalysisChannels = (RegisterAnalysisChannelsFunction)GetProcAddress(plugin, "RegisterAnalysisChannels");
				PluginFunctions functions;

				functions.InitialisePluginChannels = (InitialisePluginChannelsFunction)GetProcAddress(plugin, "InitialisePluginChannelsFunction");
				functions.CompletePluginChannels = (CompletePluginChannelsFunction)GetProcAddress(plugin, "CompletePluginChannels");
				functions.ReturnPluginAnalysis = (ReturnPluginAnalysisFunction)GetProcAddress(plugin, "ReturnPluginAnalysis");
				functions.SavePluginConfig = (SavePluginConfigFunction)GetProcAddress(plugin, "SavePluginConfig");
				functions.LoadPluginConfig = (LoadPluginConfigFunction)GetProcAddress(plugin, "LoadPluginConfig");
				functions.CalculateVolume = (CalculateVolumeFunction)GetProcAddress(plugin, "CalculateVolume");
				functions.ValidateAnalysisRequests = (ValidateAnalysisRequestsFunction)GetProcAddress(plugin, "ValidateAnalysisRequests");
				functions.logmessage = logmessage;

				size_t calc_interface_key =
					   (0x0D77C4EC +
						sizeof(FileData) * 2 +
						sizeof(AnalysisConf) * 4 +
						sizeof(Geometry) * 8 +
						sizeof(PluginFunctions) * 16 +
						sizeof(register_plugin_function) * 32 +
						sizeof(AnalysisConfig) * 64);

				if (RegisterAnalysisChannels(register_plugin, dll_filename, functions, calc_interface_key) == false)
				{
					logmessage(WARNING, "Failed to register plugin DLL (%s)\n", dll_filename);
				}
				else
				{
					AnalysisRqst* new_analysis_rqst = (AnalysisRqst*)malloc(get_number_of_analysis_channels() * sizeof(AnalysisRqst));

					if (new_analysis_rqst == NULL)
					{
						logmessage(FATAL, "Out of memory\n");
					}

					if (analysis_rqst != NULL)
					{
						for (rqst = 0; rqst < NUMBER_OF_ANALYSIS_CHANNELS; rqst++)
						{
							new_analysis_rqst[rqst] = analysis_rqst[rqst];
						}

						free(analysis_rqst);
					}
					else
					{
						for (rqst = 0; rqst < NUMBER_OF_ANALYSIS_CHANNELS; rqst++)
						{
							new_analysis_rqst[rqst] = AnalysisRqst::NotCalculated;
						}
					}

					for (rqst = NUMBER_OF_ANALYSIS_CHANNELS; rqst < get_number_of_analysis_channels(); rqst++)
					{
						new_analysis_rqst[rqst] = AnalysisRqst::TemporaryCalculation;
					}

					analysis_rqst = new_analysis_rqst;

					logmessage(NOTICE, "Registered %s as plugin DLL\n", dll_filename);
				}
			}
			else
			{
				logmessage(WARNING, "Failed to load plugin DLL (%s)\n", dll_filename);
			}
		}
#endif
		else
	    {
			/* Do Nothing */
		}
	}
      
	/* Close file */
	fclose(input_file); 

	logmessage(NOTICE,"Config file parsed\n");

	if (output != NULL)
	{
		logmessage(DEBUG,"Deleting argument structures\n");

		for (argument=0;argument<MAX_NUMBER_OF_COLUMNS;argument++)
		{
			if (output[argument] != NULL)
			{
				free(output[argument]);
				output[argument] = NULL;
			}
		}
		
		free(output);
		output = NULL;
	}

	free(analysis_rqst);

	if (analysis != NULL)
	{
		logmessage(DEBUG,"Deleting %u analysis structures\n",number_of_analysis_structures);
			
		Delete_Analysis(analysis);		
		free(analysis);
		analysis = NULL;
	}

	if (channel_blocks != NULL)
	{
		free(channel_blocks);
		channel_blocks = NULL;
	}

	return(true);
}

