/* Combustion Analysis Tool (CAT)
   www.catool.org
  
   Filename: filedata.c
  
   Purpose:  Provide common programming interface for processing acquired engine data
  
   Author:   Ben Brown
   Version:  1.2
   Date:     13.03.2017

   Revision: GPL Release

   Copyright (C) Xarin Limited, 2000-2024

      This program is free software: you can redistribute it and/or modify
      it under the terms of version 2 of the GNU General Public License as
      published by the Free Software Foundation.

      This program is distributed in the hope that it will be useful,
      but WITHOUT ANY WARRANTY; without even the implied warranty of
      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
      GNU General Public License for more details.

      You should have received a copy of the GNU General Public License
      along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* For Microsoft Visual C++ */
#ifdef _MSC_VER
#include "stdafx.h"
#endif

#include <stdlib.h>
#include <string.h>
#include <float.h>
#include <math.h>

#include "cat.h"

extern unsigned int debug_level;

void SetCAData(FileData* file, const unsigned int cycle, const unsigned int crank_angle, const unsigned int channel, const float value)
{
#ifdef _SAFE_MEMORY
	if (file == NULL)
	{
		logmessage(FATAL, "SetCAData: file NULL\n");
	}
	if (channel >= file->number_of_channels)
	{
		logmessage(FATAL, "SetCAData: Input channel number excessive (%u)\n", channel);
	}
#ifndef _SAFE_MEMORY_NO_CYCLE_CHECK
	if (cycle >= file->number_of_cycles)
	{
		logmessage(FATAL, "SetCAData: Input cycle excessive (%u)\n", cycle);
	}
#endif
	if (file->channel_data == NULL)
	{
		logmessage(FATAL, "SetCAData: No channel_data configured\n");
		exit(EXIT_FAILURE);
	}

	if (crank_angle >= file->channel_data[channel].samples_per_cycle)
	{
		logmessage(FATAL, "SetCAData: Input crank angle excessive (%u)\n", crank_angle);
	}

	if (file->channel_data[channel].data == NULL)
	{
		logmessage(FATAL, "SetCAData: No data configured in channel\n");
		exit(EXIT_FAILURE);
	}
#endif
	file->channel_data[channel].data[file->channel_data[channel].samples_per_cycle*cycle + crank_angle] = value;
}

float ReturnCAData(const FileData* file, const unsigned int cycle, const unsigned int crank_angle, const unsigned int channel)
{
#ifdef _SAFE_MEMORY
	if (channel >= file->number_of_channels)
	{
		logmessage(FATAL, "ReturnCAData: Input channel number excessive (%u)\n", channel);
	}
#ifndef _SAFE_MEMORY_NO_CYCLE_CHECK
	if (cycle >= file->number_of_cycles)
	{
		logmessage(FATAL, "ReturnCAData: Input cycle excessive (%u)\n", cycle);
	}
#endif
	if (file->channel_data == NULL)
	{
		logmessage(FATAL, "SetCAData: No channel_data configured\n");
		exit(EXIT_FAILURE);
	}

	if (crank_angle > file->channel_data[channel].samples_per_cycle)
	{
		logmessage(FATAL, "ReturnCAData: Input crank angle excessive (%u)\n", crank_angle);
	}
#endif
	return(file->channel_data[channel].data[file->channel_data[channel].samples_per_cycle*cycle + crank_angle]);
}

float ReturnAnalysisCAData(const FileData* file, const Analysis* analysis, const unsigned int cycle, const unsigned int crank_angle, const unsigned int channel, const unsigned int analysis_type)
{
	return(analysis->channel[channel].results[analysis_type].data[cycle * analysis->channel[channel].results[analysis_type].samples_per_cycle + crank_angle]);
}

unsigned int ReturnTheta(const FileData* file, const unsigned int crank_angle, const unsigned int channel)
{
#ifdef _SAFE_MEMORY
	if (channel >= file->number_of_channels)
	{
		logmessage(FATAL, "ReturnTheta: Input channel number excessive (%u)\n", channel);
	}

	if (file->channel_data == NULL)
	{
		logmessage(FATAL, "ReturnTheta: No channel_data configured\n");
		exit(EXIT_FAILURE);
	}

	if (crank_angle >= file->channel_data[channel].samples_per_cycle)
	{
		logmessage(FATAL, "ReturnTheta: Input crank angle excessive (%u)\n", crank_angle);
	}

	if (file->channel_data[channel].abscissa.ca_to_theta == NULL)
	{
		logmessage(FATAL, "ReturnTheta: Crank angle to Theta table not calculated for channel %u\n", channel);
		exit(EXIT_FAILURE);
	}
#endif
	return(file->channel_data[channel].abscissa.ca_to_theta[crank_angle]);
}

float Volume(const FileData* file, const unsigned int channel, const unsigned int theta)
{
#ifdef _SAFE_MEMORY
	if (channel >= file->number_of_channels)
	{
		logmessage(FATAL, "Volume: Input channel number excessive (%u)\n", channel);
	}

	if (file->channel_data[channel].abscissa.volume == NULL)
	{
		logmessage(FATAL, "Volume: Theta to cylinder volume not calculated for channel\n");
		exit(EXIT_FAILURE);
	}
#endif
	if (theta >= file->channel_data[channel].abscissa.ca_to_theta[0] + file->pulses_per_cycle)
	{
#ifdef _SAFE_MEMORY
		logmessage(FATAL, "Volume: Theta excessive (%u)\n", theta);
		exit(EXIT_FAILURE);
#else
		return(0.0f);
#endif
	}
	
	return(file->channel_data[channel].abscissa.volume[theta]);
}

unsigned int ReturnCrankAngle(const FileData* file, const unsigned int theta, const unsigned int channel)
{
#ifdef _SAFE_MEMORY
	if (channel >= file->number_of_channels)
	{
		logmessage(FATAL, "ReturnCrankAngle: Input channel number excessive (%u)\n", channel);
	}

	if (file->channel_data == NULL)
	{
		logmessage(FATAL, "ReturnCrankAngle: No channel_data configured\n");
		exit(EXIT_FAILURE);
	}

	if (file->channel_data[channel].abscissa.theta_to_ca == NULL)
	{
		logmessage(FATAL, "ReturnCrankAngle: Theta to Crank angle table not calculated for channel\n");
		exit(EXIT_FAILURE);
	}

	if (theta >= MaximumTheta(file, channel))
	{
		logmessage(FATAL, "ReturnCrankAngle: Input theta excessive (%u)\n", theta);
	}
#endif
	return(file->channel_data[channel].abscissa.theta_to_ca[theta]);
}

int ReturnReferencedCrankAngle(const FileData* file, const unsigned int crank_angle, const unsigned int channel, const unsigned int reference_channel)
{
	int theta;
	unsigned int maximum_theta;

#ifdef _SAFE_MEMORY
	if (channel >= file->number_of_channels)
	{
		logmessage(FATAL, "ReturnReferencedCrankAngle: Input channel number excessive (%u)\n", channel);
	}
	if (reference_channel >= file->number_of_channels)
	{
		logmessage(FATAL, "ReturnReferencedCrankAngle: Input reference channel excessive (%u)\n", reference_channel);
	}
	if (crank_angle >= file->channel_data[channel].samples_per_cycle)
	{
		logmessage(FATAL, "ReturnReferencedCrankAngle: Input crank angle excessive (%u)\n", crank_angle);
	}
	if (file->channel_data[channel].abscissa.ca_to_theta == NULL)
	{
		logmessage(FATAL, "ReturnReferencedCrankAngle: ca_to_theta is NULL\n");
	}
#endif

	theta = file->channel_data[channel].abscissa.ca_to_theta[crank_angle] + (unsigned int)(floor(file->channel_data[reference_channel].tdc_offset - file->channel_data[channel].tdc_offset)*MinimumResolution(file) + 0.5f);

	maximum_theta = MaximumTheta(file);

	while (theta < 0)
	{
		theta += maximum_theta;
	}

	while (theta >= (int)maximum_theta)
	{
		theta -= maximum_theta;
	}

#ifdef _SAFE_MEMORY
	if (theta < 0)
	{
		logmessage(FATAL, "ReturnReferencedCrankAngle: Output theta less than zero (%d)\n", theta);
	}
	else if (theta >= (int)maximum_theta)
	{
		logmessage(FATAL, "ReturnReferencedCrankAngle: Output theta excessive (%d)\n", theta);
	}
	else
	{
		/* Do Nothing */
	}
#endif

	return(ReturnCrankAngle(file, (unsigned int)theta, channel));
}

float ReturnReferencedCAData(const FileData* file, const unsigned int cycle, const unsigned int crank_angle, const unsigned int channel, const unsigned int reference_channel, const unsigned int max_cycle, const bool wraparound)
{
	float result;
	float ca;
	float max_ca;
	int cycle_no;
	int theta;
	int maximum_theta;
#ifdef _SAFE_MEMORY
	if (channel >= file->number_of_channels)
	{
		logmessage(FATAL, "ReturnReferencedCAData: Input channel number excessive (%u)\n", channel);
	}

	if (reference_channel >= file->number_of_channels)
	{
		logmessage(FATAL, "ReturnReferencedCAData: Input reference channel excessive (%u)\n", reference_channel);
	}
#ifndef _SAFE_MEMORY_NO_CYCLE_CHECK	
	if (cycle >= file->number_of_cycles)
	{
		logmessage(FATAL, "ReturnReferencedCAData: Input cycle excessive (%u)\n", cycle);
	}
#endif	
	if (crank_angle >= file->channel_data[channel].samples_per_cycle)
	{
		logmessage(FATAL, "ReturnReferencedCAData: Input crank angle excessive (%u)\n", crank_angle);
	}
#endif
	cycle_no = cycle;

	ca = CrankAngleToDegrees(file, crank_angle, channel) + file->channel_data[reference_channel].tdc_offset - file->channel_data[channel].tdc_offset;
	max_ca = (float)file->engine.number_of_strokes*90.0f;
	theta = (int)((ca + max_ca)*MinimumResolution(file));
	maximum_theta = (int)((2 * max_ca)*MinimumResolution(file));

	while (theta >= maximum_theta)
	{
		theta -= maximum_theta;
		cycle_no += 1;
	}

	while (theta < 0)
	{
		theta += maximum_theta;
		cycle_no -= 1;
	}

	//logmessage(WARNING, "%u: %u %f %f %d %d", reference_channel, crank_angle, ca, orig_ca, theta, cycle_no);

	if (cycle_no < 0)
	{
		if (wraparound == true)
		{
			cycle_no += max_cycle;

			result = ReturnThetaData(file, cycle_no, theta, channel);
		}
		else
		{
			result = ReturnCAData(file, 0, 0, channel);
		}
	}
	else if (cycle_no < (int)max_cycle)
	{
		result = ReturnThetaData(file, cycle_no, theta, channel);
	}
	else if (wraparound == false) /* && (cycle_no >= max_cycle) */
	{
		result = ReturnCAData(file, file->number_of_cycles - 1, file->channel_data[channel].samples_per_cycle - 1, channel);
	}
	else /* (wraparound == true) && (cycle_no >= max_cycle) */
	{
		cycle_no -= max_cycle;

		result = ReturnThetaData(file, cycle_no, theta, channel);
	}

	return(result);
}

float ReturnThetaDataEx(const FileData* file, unsigned int theta, const unsigned int channel, const float* data)
{
	unsigned int crank_angle;
	float crank_angle_data_1;
	float crank_angle_data_2;

#ifdef _SAFE_MEMORY
	if (channel >= file->number_of_channels)
	{
		logmessage(FATAL, "ReturnThetaData: Input channel number excessive (%u)\n", channel);
	}

	if (file->channel_data[channel].abscissa.theta_to_ca == NULL)
	{
		logmessage(FATAL, "ReturnThetaData: theta_to_ca NULL\n");
	}

	if (theta >= MaximumTheta(file, channel))
	{
		logmessage(FATAL, "ReturnThetaData: Input theta excessive (%u)\n", theta);
	}
#endif

	crank_angle = file->channel_data[channel].abscissa.theta_to_ca[theta];

	crank_angle_data_1 = data[crank_angle];

	if (crank_angle >= file->channel_data[channel].samples_per_cycle - 1)
	{
		return(crank_angle_data_1);
	}
	else
	{
		crank_angle_data_2 = data[crank_angle + 1];
	}

	return(crank_angle_data_1 * file->channel_data[channel].abscissa.theta_weighting[theta] + crank_angle_data_2 * file->channel_data[channel].abscissa.theta_weighting_inv[theta]);
}

float ReturnThetaData(const FileData* file, const unsigned int cycle, unsigned int theta, const unsigned int channel)
{
	return(ReturnThetaDataEx(file, theta, channel, &file->channel_data[channel].data[cycle * file->channel_data[channel].samples_per_cycle]));
}

float ReturnAnalysisThetaData(const FileData* file, const Analysis* analysis, const unsigned int cycle, unsigned int theta, const unsigned int channel, const unsigned int analysis_type)
{
	return(ReturnThetaDataEx(file, theta, channel, &analysis->channel[channel].results[analysis_type].data[cycle * analysis->channel[channel].results[analysis_type].samples_per_cycle]));
}

unsigned int ReturnCalculatedTheta(const AbscissaData* abscissa, const unsigned int crank_angle)
{
	unsigned int abscissa_number;
	unsigned int start_crank_angle;
	unsigned int theta;
	unsigned int result;

#ifdef _SAFE_MEMORY
	if (abscissa == NULL)
	{
		logmessage(FATAL, "ReturnCalculatedTheta: abscissa NULL\n");
	}
#endif

	abscissa_number = 0;
	start_crank_angle = 0;

	while (crank_angle >= start_crank_angle + abscissa->number_of_samples[abscissa_number])
	{
		start_crank_angle += abscissa->number_of_samples[abscissa_number];
		abscissa_number += 1;
	}

#ifdef _SAFE_MEMORY
	if (abscissa_number >= abscissa->number_of_measurement_tables)
	{
		logmessage(FATAL, "ReturnCalculatedTheta: Calculated abscissa measurement table excessive (%u)\n", abscissa_number);
	}
#endif

	theta = abscissa->start[abscissa_number];

	result = theta + abscissa->resolution[abscissa_number] * (crank_angle - start_crank_angle);

	return(result);
}

float ThetaToDegrees(const FileData* file, const unsigned int theta)
{
	return(file->theta_to_deg[theta]);
}

float ThetaAbsToDegrees(const FileData* file, const unsigned int theta)
{
	return((float)theta / MinimumResolution(file));
}

unsigned int DegreesToTheta(const FileData* file, const float degrees)
{
	float offset = (float)(MINIMUM_THETA_ANGLE_MULT * file->engine.number_of_strokes);
	float deg = degrees + offset;

	if (deg < 0.0f)
	{
		deg = 0.0f;
	}

	return((unsigned int)(floorf(deg*MinimumResolution(file) + 0.5f)));
}

unsigned int DegreesToThetaAbs(const FileData* file, const float degrees)
{
	float deg = degrees;

	if (deg < 0.0f)
	{
		deg = 0.0f;
	}

	return((unsigned int)(floorf(deg*MinimumResolution(file) + 0.5f)));
}

int DegreesToThetaRel(const FileData* file, const float degrees)
{
	return((int)(roundf(degrees * MinimumResolution(file))));
}

float MinimumResolution(const FileData* file)
{
	return(file->minimum_resolution);
}

unsigned int PulsesPerCycle(const FileData* file)
{
	return(file->pulses_per_cycle);
}

unsigned int PulsesPerRev(const FileData* file)
{
	return(file->min_res_ppr);
}

unsigned int MaximumTheta(const FileData* file)
{
	return(file->pulses_per_cycle*2);
}

unsigned int MaximumTheta(const FileData* file, const unsigned int channel)
{
	AbscissaData* abscissa = &file->channel_data[channel].abscissa;

	unsigned int mt = abscissa->number_of_measurement_tables - 1;

	return(abscissa->start[mt] + abscissa->number_of_samples[mt] * abscissa->resolution[mt]);
}

unsigned int DegreesToCrankAngle(const FileData* file, const float degrees, const unsigned int channel)
{
	unsigned int theta = DegreesToTheta(file, degrees);
	unsigned int maximum_theta = MaximumTheta(file, channel);

	if (theta >= maximum_theta)
	{
		theta = maximum_theta - 1;
	}

	return(ReturnCrankAngle(file, theta, channel));
}

float CrankAngleToDegrees(const FileData* file, const unsigned int crank_angle, const unsigned int channel)
{
#ifdef _SAFE_MEMORY
	if (crank_angle >= file->channel_data[channel].samples_per_cycle)
	{
		logmessage(FATAL, "CrankAngleToDegrees: crank_angle excessive\n");
	}
	if (file->channel_data[channel].abscissa.ca_to_deg == NULL)
	{
		logmessage(FATAL, "CrankAngleToDegrees: ca_to_deg is NULL\n");
		exit(EXIT_FAILURE);
	}
#endif
	return(file->channel_data[channel].abscissa.ca_to_deg[crank_angle]);
}

void clip_crank_angle(const FileData* file, unsigned int* crank_angle, const unsigned int channel)
{
#ifdef _SAFE_MEMORY
	if (channel >= file->number_of_channels)
	{
		logmessage(FATAL, "clip_crank_angle: Input channel number excessive (%u)\n", channel);
	}
#endif
	if (*crank_angle >= file->channel_data[channel].samples_per_cycle)
	{
		*crank_angle = file->channel_data[channel].samples_per_cycle - 1;
	}
}

float Return_PinOffset(const FileData* file, const unsigned int channel)
{
	if (file == NULL)
	{
		logmessage(MSG_ERROR, "Cannot determine pin offset for channel %u\n", channel);
		return(0.0f);
	}

	if ((file->channel_data == NULL) || (channel >= file->number_of_channels))
	{
		logmessage(MSG_ERROR, "Cannot determine pin offset for channel %u\n", channel);
		return(0.0f);
	}

	if ((file->channel_data[channel].cylinder < 1) || (file->channel_data[channel].cylinder > MAX_NUMBER_OF_CYLINDERS))
	{
		logmessage(WARNING, "Cannot determine pin offset for channel %s, cylinder number out of range\n", file->channel_data[channel].name);
		return(0.0f);
	}

	return(file->engine.pin_offset[file->channel_data[channel].cylinder - 1]);
}

bool channel_number_to_name_ex(const FileData* file, const unsigned int channel_number, char* channel_name, const bool short_name)
{
	if (channel_name == NULL)
	{
		return(false);
	}

	if (file == NULL)
	{
		channel_name[0] = 0x00;

		return(false);
	}

	if ((file->channel_data == NULL) || (channel_number >= file->number_of_channels))
	{
		channel_name[0] = 0x00;

		return(false);
	}

	if (short_name == true)
	{
		strncpy(channel_name, file->channel_data[channel_number].short_name, SIZEOF_CHANNEL_NAME);
	}
	else
	{
		strncpy(channel_name, file->channel_data[channel_number].name, SIZEOF_CHANNEL_NAME);
	}

	return(true);
}

bool channel_number_to_name(const FileData* file, const unsigned int channel_number, char* channel_name)
{
	return(channel_number_to_name_ex(file, channel_number, channel_name, false));
}

bool channel_name_to_number_ex(const FileData* file, const char* channel_name, const unsigned int abscissa_type, unsigned int* channel_number, const bool short_name)
{
	unsigned int channel;
	bool found = false;

	if (file == NULL)
	{
		if (channel_number != NULL)
		{
			*channel_number = 0;
		}

		return(false);
	}

	if (channel_name == NULL)
	{
		if (channel_number != NULL)
		{
			*channel_number = 0;
		}

		return(false);
	}

	if (channel_name[0] == 0x00)
	{
		if (channel_number != NULL)
		{
			*channel_number = 0;
		}

		return(false);
	}

	channel = 0;
	while ((channel<file->number_of_channels) && (found == false))
	{
		if ((((short_name == false) && (case_insensitive_compare(file->channel_data[channel].name, channel_name, SIZEOF_CHANNEL_NAME) == 1)) ||
			 ((short_name == true) && (case_insensitive_compare(file->channel_data[channel].short_name, channel_name, SIZEOF_SHORT_NAME) == 1))) &&
			((abscissa_type == ABSCISSA_UNKNOWN) || (abscissa_type == file->channel_data[channel].abscissa.type)))
		{
			if (channel_number != NULL)
			{
				*channel_number = channel;
			}

			found = true;
		}

		channel++;
	}

	if (found == true)
	{
		return(true);
	}
	else
	{
		if (channel_number != NULL)
		{
			*channel_number = 0;
		}

		return(false);
	}
}

bool channel_name_to_number(const FileData* file, const char* channel_name, const unsigned int abscissa_type, unsigned int* channel_number)
{
	return(channel_name_to_number_ex(file, channel_name, abscissa_type, channel_number, false));
}

float channel_average(const FileData* file, const unsigned int channel, const unsigned int cycle, const unsigned int reference_channel, bool quick)
{
	float average;
	unsigned int crank_angle;
	unsigned int step_size;
	unsigned int steps;

	if (file == NULL)
	{
		return(0.0f);
	}

	if ((channel >= file->number_of_channels) || (reference_channel >= file->number_of_channels) || (file->number_of_channels == 0))
	{
		return(0.0f);
	}

	if ((file->channel_data[channel].loaded == false) || (file->channel_data[reference_channel].loaded == false))
	{
		return(0.0f);
	}

	if (file->channel_data[channel].abscissa.type == ABSCISSA_CRANKANGLE)
	{
		if (quick == true)
		{
			step_size = (unsigned int)(file->channel_data[channel].samples_per_cycle / 10);
		}
		else
		{
			step_size = 1;
		}

		steps = 0;
		average = 0.0f;
		for (crank_angle = 0; crank_angle<file->channel_data[channel].samples_per_cycle; crank_angle += step_size)
		{
			average += ReturnReferencedCAData(file, cycle, crank_angle, channel, reference_channel, file->number_of_cycles, false);
			steps += 1;
		}

		if (steps > 0)
		{
			average /= steps;
		}
	}
	else if (file->channel_data[channel].abscissa.type == ABSCISSA_CYCLE)
	{
		average = file->channel_data[channel].data[cycle];
	}
	else
	{
		average = 0.0f;
	}

	return(average);
}

bool validate_angle(const FileData* file, const unsigned int channel, const float angle)
{
	unsigned int measurement_table;
	bool result;
	float start_angle;
	float finish_angle;
	unsigned int theta;
	unsigned int maximum_theta;

	if (file == NULL)
	{
		return(false);
	}

	if ((channel >= file->number_of_channels) || (file->number_of_channels == 0))
	{
		return(false);
	}

	if ((file->engine.number_of_strokes != 2) && (file->engine.number_of_strokes != 4))
	{
		return(false);
	}

	if (angle < -90.0f * (float)file->engine.number_of_strokes)
	{
		result = false;
	}
	else if (angle > 90.0f * (float)file->engine.number_of_strokes)
	{
		result = false;
	}
	else
	{
		/* Check angle is within an abscissa */

		result = false;
		maximum_theta = MaximumTheta(file, channel);

		for (measurement_table = 0; measurement_table < file->channel_data[channel].abscissa.number_of_measurement_tables; measurement_table++)
		{
			theta = file->channel_data[channel].abscissa.start[measurement_table];

			if (theta < maximum_theta)
			{
				start_angle = (float)ThetaToDegrees(file, theta);

				theta = file->channel_data[channel].abscissa.start[measurement_table] + (file->channel_data[channel].abscissa.number_of_samples[measurement_table] - 1) * file->channel_data[channel].abscissa.resolution[measurement_table];

				if (theta < maximum_theta)
				{
					finish_angle = (float)ThetaToDegrees(file, theta);

					if (((angle - start_angle) > -FLT_EPSILON) && ((finish_angle - angle) > -FLT_EPSILON))
					{
						result = true;
					}
				}
			}
		}
	}

	return(result);
}

int GetChannelFromChannelData(const FileData* file, const ChannelData* channel_data)
{
	unsigned int channel;

	if (file == NULL)
	{
		return(-1);
	}

	if (channel_data == NULL)
	{
		return(-1);
	}

	for (channel = 0; channel < file->number_of_channels; channel++)
	{
		if (&file->channel_data[channel] == channel_data)
		{
			return(channel);
		}
	}

	return(-1);
}

bool analysis_name_to_analysis_type(const FileData* file, const Analysis* analysis, const char* channel_name, const unsigned int abscissa_type, unsigned int* channel, unsigned int* analysis_type)
{
	unsigned int analysis_channel;
	unsigned int type;

	if ((file == NULL) || (analysis == NULL) || (channel_name == NULL))
	{
		return(false);
	}

	for (analysis_channel = 0; analysis_channel < analysis->number_of_channels; analysis_channel++)
	{
		for (type = 0; type < get_number_of_analysis_channels(); type++)
		{
			if (((analysis->channel[analysis_channel].results[type].abscissa.type == abscissa_type) || (abscissa_type == ABSCISSA_UNKNOWN)) &&
				(strncmp(channel_name, analysis->channel[analysis_channel].results[type].name, SIZEOF_CHANNEL_NAME) == 0) &&
				(analysis->channel[analysis_channel].analysis_rqst[type] > AnalysisRqst::NotCalculated))
			{
				if (channel != NULL)
				{
					*channel = analysis_channel;
				}

				if (analysis_type != NULL)
				{
					*analysis_type = type;
				}

				return(true);
			}
		}
	}

	return(false);
}