/* Combustion Analysis Tool (CAT)
   www.catool.org
   
   Filename: csvfile.c

   Purpose:  Provide common programming interface for processing acquired engine data
  
   Author:   Ben Brown
   Version:  1.2
   Date:     19.10.2015

   Revision: GPL Release

   Copyright (C) Xarin Limited, 2000-2024

      This program is free software: you can redistribute it and/or modify
      it under the terms of version 2 of the GNU General Public License as
      published by the Free Software Foundation.

      This program is distributed in the hope that it will be useful,
      but WITHOUT ANY WARRANTY; without even the implied warranty of
      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
      GNU General Public License for more details.

      You should have received a copy of the GNU General Public License
      along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* For Microsoft Visual C++ */
#ifdef _MSC_VER
#include "stdafx.h"
#endif

#include <float.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "cat.h"

extern unsigned int debug_level;

bool load_csv_file_header(
	FileData* file,
#ifdef _CATOOL_UNICODE_
	const wchar_t* filename,
#else
	const char* filename,
#endif
	AbscissaData* abscissa,
	unsigned int* resolution,
	const unsigned int abscissa_type,
	float* recording_length,
	unsigned int* number_of_lines_to_skip,
	const bool* skip_columns)
{
	char* input_line = NULL;
	char abscissa_units[SIZEOF_UNITS];
	unsigned int channel;
	unsigned int abscissa_resolution = 0;
	unsigned int samples_per_cycle = 0;
	unsigned int data_size = 0;
	unsigned int number_of_lines;
	FILE* input_file = NULL;
	float input_resolution;
	char** output = NULL;
	unsigned int arguments = 0;
	unsigned int column;
	unsigned int argument;
	unsigned int mt;
	bool result = true;

	if ((abscissa_type == ABSCISSA_CRANKANGLE) && (file->number_of_cycles == 0))
	{
		logmessage(MSG_ERROR, "You must specify the number of cycles to be greater than zero\n");

		return(false);
	}

	if ((abscissa_type != ABSCISSA_CRANKANGLE) && (abscissa_type != ABSCISSA_TIME))
	{
		logmessage(MSG_ERROR, "Abscissa type must be crank angle or time based\n");

		return(false);
	}

	output = (char**)malloc(MAX_NUMBER_OF_COLUMNS * sizeof(char*));

	if (output == NULL)
	{
		logmessage(FATAL, "Memory could not be allocated\n");
	}

	for (column = 0; column<MAX_NUMBER_OF_COLUMNS; column++)
	{
		output[column] = (char*)malloc(MAX_LINE_SIZE);

		if (output[column] == NULL)
		{
			logmessage(FATAL, "Memory could not be allocated\n");
		}
	}

	input_line = (char*)malloc(MAX_LINE_SIZE);

	if (input_line == NULL)
	{
		logmessage(FATAL, "Memory could not be allocated\n");
	}

#ifdef _CATOOL_UNICODE_
	wcsncpy(file->filename, filename, 4095);
#else
	strncpy(file->filename, filename, 4095);
#endif

	if (abscissa_type == ABSCISSA_CRANKANGLE)
	{
		if (abscissa != NULL)
		{
			samples_per_cycle = 0;
			for (mt = 0; mt < abscissa->number_of_measurement_tables; mt++)
			{
				samples_per_cycle += abscissa->number_of_samples[mt];
			}
		}
		else
		{
			samples_per_cycle = (180 * file->engine.number_of_strokes) * (*resolution);
		}

		data_size = samples_per_cycle * (file->number_of_cycles + 2);
		abscissa_resolution = DegreesToThetaAbs(file, 1 / (float)(*resolution));
		strncpy(abscissa_units, "deg", SIZEOF_UNITS);
		abscissa_units[SIZEOF_UNITS - 1] = 0x00;

		if (file->number_of_channels == 0)
		{
#ifdef _CATOOL_UNICODE_
			input_file = _wfopen(filename, L"rb");
#else
			input_file = fopen(filename, "rb");
#endif

			if (input_file == NULL)
			{
				free(input_line);

				logmessage(MSG_ERROR, "CSV File '%s' could not be opened\n", filename);

				return(false);
			}

			if (fgets(input_line, MAX_LINE_SIZE, input_file) != NULL)
			{
				read_line(input_line, output, MAX_NUMBER_OF_COLUMNS, MAX_LINE_SIZE, &arguments, SEPARATOR_COMMA | SEPARATOR_TAB, 0x00);

				argument = 0;

				while (argument < arguments)
				{
					if (skip_columns != NULL)
					{
						while ((argument < arguments) && (skip_columns[argument] == true))
						{
							logmessage(WARNING, "Skipping %s", output[argument]);

							argument++;
						}
					}

					if (argument < arguments)
					{
						logmessage(WARNING, "Adding %s", output[argument]);
						file->number_of_channels++;
					}

					argument++;
				}
			}
		}
	}
	else if (abscissa_type == ABSCISSA_TIME)
	{	
		if (file->number_of_channels == 0)
		{
			free(input_line);

			logmessage(MSG_ERROR, "You must specify the number of channels with the 'number-of-channels' keyword\n");

			return(false);
		}

#ifdef _CATOOL_UNICODE_
		input_file = _wfopen(filename, L"rb");
#else
		input_file = fopen(filename, "rb");
#endif
			
		if ( input_file == NULL )
		{
			free(input_line);

			logmessage(MSG_ERROR,"CSV File '%s' could not be opened\n",filename);
	
			return(false);
		}
		
		if (*resolution == 0)
		{
			if (fscanf(input_file, "%20f", &input_resolution) == 1)
			{
				abscissa_resolution = (unsigned int)input_resolution;
			}
			else
			{
				abscissa_resolution = 0;
			}
				
			if (abscissa_resolution == 0)
			{
				free(input_line);
				fclose(input_file);
				
				logmessage(MSG_ERROR, "Could not determine data acquisition resolution from first line of CSV file\n");

				return(false);
			}
				
			logmessage(NOTICE,"Recording resolution determined as %u Hz\n",abscissa_resolution);
					
			if (*number_of_lines_to_skip == 0)
			{
				*number_of_lines_to_skip = 1;
			}
					
			*resolution = abscissa_resolution;
		}
		else
		{
			abscissa_resolution = *resolution;
		}
		
		if (*recording_length < FLT_EPSILON)
		{
			number_of_lines = 0;
			while (fgets(input_line, MAX_LINE_SIZE, input_file) != NULL)
 			{
				number_of_lines += 1;
			}

			*recording_length = ((float)number_of_lines - (float)*number_of_lines_to_skip) / (float)abscissa_resolution; 	
		
			if (*recording_length < FLT_EPSILON)
			{
				free(input_line);
				fclose(input_file);

				logmessage(MSG_ERROR,"Could not determine recording length from number of lines of CSV file\n");

				return(false);
			}
				
			logmessage(NOTICE,"Recording length determined as %f seconds\n",*recording_length);
		}

		fclose(input_file);
		
		strncpy(abscissa_units,"Time",SIZEOF_UNITS);
		abscissa_units[SIZEOF_UNITS - 1] = 0x00;
		
		file->number_of_cycles = 1;
		samples_per_cycle = (unsigned int)(*recording_length * abscissa_resolution);
		data_size = samples_per_cycle;
	}
	else
	{
		logmessage(MSG_ERROR, "Unsupported abscissa type\n");
	}
	
#ifdef _CATOOL_UNICODE_
	snprintf(file->comment, SIZEOF_DESCRIPTION, "Loaded from CSV file");
#else
	snprintf(file->comment, SIZEOF_DESCRIPTION, "Loaded from CSV file %s", filename);
#endif

	/* The following could be replaced by some platform specific tests for the file creation date */
	
	file->creation_time = time(NULL);
	Change_Minimum_Resolution(file,DEFAULT_MIN_RES_PPR);
	file->ptr_APB = NULL;
	file->ptr_DGB = NULL;
	file->parameter_file = NULL;
	file->parameter_filename[0] = 0x00;
	file->parameter_filesize = 0;
	
	if (abscissa_resolution > 0)
	{
		file->time_resolution = 1.0 / (double)abscissa_resolution / 10.0;

		if (file->time_resolution < FLT_EPSILON)
		{
			file->time_resolution = DEFAULT_TIME_RESOLUTION;
		}
	}
	else
	{
		file->time_resolution = DEFAULT_TIME_RESOLUTION;
	}

	for (channel = 0; channel<file->number_of_channels; channel++)
	{
		Delete_Channel(file, channel);
	}

	file->channel_data = (ChannelData*)malloc(file->number_of_channels * sizeof(ChannelData));
	if (file->channel_data == NULL)
	{
		logmessage(FATAL, "Memory could not be allocated\n");
	}

	for (channel=0;channel<file->number_of_channels;channel++)
	{
		file->channel_data[channel].offset = 0.0;
		file->channel_data[channel].slope = 1.0;
		file->channel_data[channel].conversion_factor = 1.0f;
		file->channel_data[channel].tdc_offset = 0.0f;
		snprintf(file->channel_data[channel].name,SIZEOF_CHANNEL_NAME,"CHNL%u",channel+1);
		file->channel_data[channel].short_name[0] = 0x00;
		file->channel_data[channel].matlab_name[0] = 0x00;
		strncpy(file->channel_data[channel].description,"CSV Channel",SIZEOF_DESCRIPTION);
		file->channel_data[channel].description[SIZEOF_DESCRIPTION-1] = 0x00;
		strncpy(file->channel_data[channel].units,"",SIZEOF_UNITS);
		file->channel_data[channel].units[SIZEOF_UNITS-1] = 0x00;
		file->channel_data[channel].cylinder = 1;
		file->channel_data[channel].type = 0;
		file->channel_data[channel].filter_config.type = FILTER_NONE;
		file->channel_data[channel].filter_config.filtered = false;
		file->channel_data[channel].filter_config.upper_frequency = 500.0f;
		file->channel_data[channel].filter_config.lower_frequency = 500.0f;
		file->channel_data[channel].isoffset = false;
		file->channel_data[channel].loaded = false;
		file->channel_data[channel].file_flag = false;
		file->channel_data[channel].samples_per_cycle = samples_per_cycle;
		file->channel_data[channel].data_d = NULL;

		file->channel_data[channel].offset_config.window_size = 5;
		file->channel_data[channel].offset_config.fixed_value = 0.0f;
		file->channel_data[channel].offset_config.channel_name[0] = 0x00;
		file->channel_data[channel].offset_config.start_window = -100.0f;
		file->channel_data[channel].offset_config.finish_window = -65.0f;
		file->channel_data[channel].offset_config.type = OFFSET_NONE;
		file->channel_data[channel].offset_config.polytropic_index = 1.32f;
		file->channel_data[channel].offset_config.truncate = false;

		file->channel_data[channel].soc_config.fixed_value = -20.0f;
		file->channel_data[channel].soc_config.start_window = -60.0f;
		file->channel_data[channel].soc_config.finish_window = 60.0f;
		file->channel_data[channel].soc_config.aligned = true;
		file->channel_data[channel].soc_config.invert = false;
		file->channel_data[channel].soc_config.type = SOC_FIXED;
		file->channel_data[channel].soc_config.channel_name[0] = 0x00;

		file->channel_data[channel].digital_config.type = DIGITIZE_AUTO;
		file->channel_data[channel].digital_config.invert = false;
		file->channel_data[channel].digital_config.latch_high = 2.0f;
		file->channel_data[channel].digital_config.latch_low = 1.0f;
		file->channel_data[channel].digital_config.filter = 1.0f;

		file->channel_data[channel].cam_config.edge = EDGE_BOTH;
		file->channel_data[channel].cam_config.invert = false;
		file->channel_data[channel].cam_config.offset = 0.0f;
		file->channel_data[channel].cam_config.reference_angle = 0.0f;
		file->channel_data[channel].cam_config.type = CAMTYPE_NONE;

		file->channel_data[channel].min = 0.0f;
		file->channel_data[channel].max = 0.0f;

		file->channel_data[channel].data = (float*)malloc(data_size * sizeof(float));
		if (file->channel_data[channel].data == NULL)
		{
			logmessage(FATAL,"Memory could not be allocated");
		}
	
		if (abscissa_type == ABSCISSA_CRANKANGLE)
		{
			file->channel_data[channel].duration = (float*)malloc(file->number_of_cycles * sizeof(float));
			if (file->channel_data[channel].duration == NULL)
			{
				logmessage(FATAL, "Memory could not be allocated\n");
			}

			memset(file->channel_data[channel].duration, 0, file->number_of_cycles * sizeof(float));
		}
		else
		{
			file->channel_data[channel].duration = NULL;
		}
	
		Zero_Abscissa(&file->channel_data[channel].abscissa);

		if (abscissa != NULL)
		{
			Copy_Abscissa(abscissa, &file->channel_data[channel].abscissa);
		}
		else
		{
			file->channel_data[channel].abscissa.number_of_measurement_tables = 1;

			file->channel_data[channel].abscissa.number_of_samples = (unsigned int*)malloc(file->channel_data[channel].abscissa.number_of_measurement_tables * sizeof(unsigned int));
			if (file->channel_data[channel].abscissa.number_of_samples == NULL)
			{
				logmessage(FATAL, "Memory could not be allocated\n");
			}

			file->channel_data[channel].abscissa.start = (unsigned int*)malloc(file->channel_data[channel].abscissa.number_of_measurement_tables * sizeof(unsigned int));
			if (file->channel_data[channel].abscissa.start == NULL)
			{
				logmessage(FATAL, "Memory could not be allocated\n");
			}

			file->channel_data[channel].abscissa.resolution = (unsigned int*)malloc(file->channel_data[channel].abscissa.number_of_measurement_tables * sizeof(unsigned int));
			if (file->channel_data[channel].abscissa.resolution == NULL)
			{
				logmessage(FATAL, "Memory could not be allocated\n");
			}

			file->channel_data[channel].abscissa.number_of_samples[0] = samples_per_cycle;
			file->channel_data[channel].abscissa.start[0] = 0;
			if (abscissa_type == ABSCISSA_TIME)
			{
				file->channel_data[channel].abscissa.resolution[0] = 1;
			}
			else
			{
				file->channel_data[channel].abscissa.resolution[0] = abscissa_resolution;
			}
		}

		file->channel_data[channel].abscissa.type = abscissa_type;

		strncpy(file->channel_data[channel].abscissa.units, abscissa_units, SIZEOF_UNITS);

		memset(file->channel_data[channel].data, 0, data_size * sizeof(float));

		if ((abscissa_type == ABSCISSA_CRANKANGLE) && (calculate_theta_lookups(file, channel) == false))
		{
			logmessage(MSG_ERROR, "load_csv_file_header: Could not calculate theta lookups\n");

			result = false;
		}
	}

	file->preload = true;
	free(input_line);

	if (output != NULL)
	{
		for (column = 0; column<MAX_NUMBER_OF_COLUMNS; column++)
		{
			if (output[column] != NULL)
			{
				free(output[column]);
				output[column] = NULL;
			}
		}

		free(output);
		output = NULL;
	}

	logmessage(NOTICE,"CSV file configured\n");

	return(result);
}

unsigned int load_csv_file(
	FileData* file, 
	const unsigned int resolution, 
	const unsigned int number_of_lines_to_skip, 
	const unsigned int channel_names_line, 
	const unsigned int channel_units_line, 
	const bool* skip_columns, 
	const unsigned int start_cycle, 
	const unsigned int data_type,
	const bool align_tdc)
{
	FILE* input_file = NULL;
	char input_line[MAX_LINE_SIZE];
	char input[100];
	unsigned int channel;
	unsigned int crank_angle;
	unsigned int cycle;
	unsigned int tdc_offset;
	unsigned int abscissa_type;
	unsigned int data_pointer;
	unsigned int line;
	unsigned int data_size;
	unsigned int data_to_discard;
	unsigned int samples_per_cycle;
	unsigned int cycles_loaded;
	unsigned int minimum_cycles;
	unsigned int arguments = 0;
	double voltage_data;
	float actual_data;
	float first_sample;
	float mid_cycle_sample;
	CSVData* csv_input_data = NULL;
	bool sync;
	char** output = NULL;
	int result;
	unsigned int argument;
	unsigned int column;
	bool done;

	logmessage(NOTICE,"Loading CSV file\n");
	
	if ((file->preload == false) || (file->file_type != FILE_CSV))
	{
		logmessage(MSG_ERROR,"CSV file has not been pre-loaded.  You cannot load channels before the file header!\n");
		
		return(0);
    }

	if (skip_columns == NULL)
	{
		logmessage(MSG_ERROR,"Columns to skip is NULL\n");
	}

#ifdef _CATOOL_UNICODE_
	input_file = _wfopen(file->filename, L"rb");
#else
	input_file = fopen(file->filename, "rb");
#endif
	
	if ( input_file == NULL )
	{
		logmessage(MSG_ERROR,"CSV File '%s' could not be opened\n",file->filename);
		
		return(0);
	}
	
	/* Load upto file->number_of_cycles + 2

	   This will enable file->number_of_cycles cycles to be loaded even with a tdc offset and synchronisation error

	   For example a cylinder at 719 degrees tdc offset with a 360 sync error = 1079 degrees extra data required (near enough 1 1/2 cycles)

	   Therefore to ensure you acquire file->number_of_cycles cycles, the raw acquisition should be file->number_of_cycles + 2
	   otherwise we cannot take advantage of this algorithm. */

	/* Assuming all channels have same abscissa type, samples per cycle and number of cycles
	   This is not unreasonable for a CSV file */

	channel = 0;
	abscissa_type = file->channel_data[channel].abscissa.type;
	samples_per_cycle = file->channel_data[channel].samples_per_cycle;

	if (abscissa_type == ABSCISSA_CRANKANGLE)
	{
		if (data_type == CSV_COLUMN_CYCLES)
		{
			data_size = samples_per_cycle * file->number_of_cycles;
		}
		else
		{
			data_size = samples_per_cycle * (file->number_of_cycles + 2);
		}
	}
	else if (abscissa_type == ABSCISSA_TIME)
	{
		data_size = samples_per_cycle;
	}
	else if (abscissa_type == ABSCISSA_CYCLE)
	{
		data_size = file->number_of_cycles;
	}
	else
	{
		fclose(input_file);

		logmessage(MSG_ERROR,"Unknown abscissa type\n");
		
		return(0);
	}

	if (data_type == CSV_COLUMN_CHANNELS)
	{
		csv_input_data = (CSVData*)malloc(file->number_of_channels * sizeof(CSVData));

		if (csv_input_data == NULL)
		{
			logmessage(FATAL, "Memory could not be allocated\n");
		}

		for (channel = 0; channel < file->number_of_channels; channel++)
		{
			if (file->channel_data[channel].file_flag == true)
			{
				csv_input_data[channel].data = (float*)malloc(data_size * sizeof(float));

				if (csv_input_data[channel].data == NULL)
				{
					logmessage(FATAL, "Memory could not be allocated\n");
				}

				memset(csv_input_data[channel].data, 0, data_size * sizeof(float));
			}
			else
			{
				csv_input_data[channel].data = NULL;
			}
		}
	}
	else
	{
		csv_input_data = (CSVData*)malloc(sizeof(CSVData));

		if (csv_input_data == NULL)
		{
			logmessage(FATAL, "Memory could not be allocated\n");
		}

		csv_input_data->data = (float*)malloc(data_size * sizeof(float));

		if (csv_input_data->data == NULL)
		{
			logmessage(FATAL, "Memory could not be allocated\n");
		}

		memset(csv_input_data->data, 0, data_size * sizeof(float));
	}
	
	output = (char**)malloc(MAX_NUMBER_OF_COLUMNS * sizeof(char*));
	
	if (output == NULL)
	{
		logmessage(FATAL,"Memory could not be allocated");
	}
				
	for (column=0;column<MAX_NUMBER_OF_COLUMNS;column++)
	{
		output[column] = (char*)malloc(MAX_LINE_SIZE);
	
		if (output[column] == NULL)
		{
			logmessage(FATAL,"Memory could not be allocated");
		}
	}
	
	data_pointer = 0;
	crank_angle = 0;

	if (data_type == CSV_COLUMN_CHANNELS)
	{
		cycle = start_cycle;
	}
	else
	{
		cycle = 0;
	}

	logmessage(DEBUG,"Samples per cycle:  %u\n",samples_per_cycle);
	logmessage(DEBUG,"Number of cycles:   %u\n",file->number_of_cycles);
	logmessage(DEBUG,"Number of channels: %u\n",file->number_of_channels);
	logmessage(DEBUG,"Abscissa type:      %u\n",abscissa_type);
	logmessage(DEBUG,"Data size:          %u\n",data_size);
	logmessage(DEBUG,"Resolution:         %u\n",resolution);
	
	line = 1;
	done = false;
 	while ((fgets(input_line,MAX_LINE_SIZE,input_file) != NULL) && (done == false) )
 	{
		read_line(input_line,output,MAX_NUMBER_OF_COLUMNS,MAX_LINE_SIZE,&arguments,SEPARATOR_COMMA | SEPARATOR_TAB, 0x00);
		
		if (line <= number_of_lines_to_skip)
		{
			if ((line == channel_names_line) || (line == channel_units_line))
			{
				argument = 0;
				for (channel = 0; channel<file->number_of_channels; channel++)
				{
					if (file->channel_data[channel].file_flag == true)
					{
						if (skip_columns != NULL)
						{
							while ((argument < arguments) && (skip_columns[argument] == true))
							{
								argument++;
							}
						}

						if (argument < arguments)
						{
							strncpy(input, output[argument], sizeof(input));
							input[99] = 0x00;

							if (channel_names_line == channel_units_line)
							{
								char* pos_1 = strstr(input, "(");
								char* pos_2 = NULL;
								size_t units_length;

								if (pos_1 != NULL)
								{
									pos_2 = strstr(input, ")");

									if (pos_2 > pos_1 + 1)
									{
										units_length = pos_2 - pos_1 - 1;
										if (units_length >= SIZEOF_UNITS)
										{
											units_length = SIZEOF_UNITS - 1;
										}

										strncpy(file->channel_data[channel].units, pos_1 + 1, units_length);
										file->channel_data[channel].units[SIZEOF_UNITS - 1] = 0x00;

										logmessage(DEBUG, "Found channel units '%s'\n", file->channel_data[channel].units);
									}
								}
								else
								{
									pos_1 = strstr(input, "[");
									if (pos_1 != NULL)
									{
										pos_2 = strstr(input, "]");

										if (pos_2 > pos_1 + 1)
										{
											units_length = pos_2 - pos_1 - 1;
											if (units_length >= SIZEOF_UNITS)
											{
												units_length = SIZEOF_UNITS - 1;
											}

											strncpy(file->channel_data[channel].units, pos_1 + 1, units_length);
											file->channel_data[channel].units[SIZEOF_UNITS - 1] = 0x00;

											logmessage(DEBUG, "Found channel units '%s'\n", file->channel_data[channel].units);
										}
									}
								}

								/* Strip spaces between channel name and units */
								if (pos_1 != NULL)
								{
									*pos_1 = 0x00;
									pos_1--;
									while (*pos_1 == 0x20)
									{
										*pos_1 = 0x00;
										pos_1--;
									}
								}
								else
								{
									logmessage(DEBUG, "Could not find units");
								}

								strncpy(file->channel_data[channel].name, input, SIZEOF_CHANNEL_NAME);
								file->channel_data[channel].name[SIZEOF_CHANNEL_NAME - 1] = 0x00;

								logmessage(DEBUG, "Found channel name '%s'\n", file->channel_data[channel].name);
							}
							else if (line == channel_names_line)
							{
								strncpy(file->channel_data[channel].name, input, SIZEOF_CHANNEL_NAME);
								file->channel_data[channel].name[SIZEOF_CHANNEL_NAME - 1] = 0x00;

								logmessage(DEBUG, "Found channel name '%s'\n", file->channel_data[channel].name);
							}
							else
							{
								strncpy(file->channel_data[channel].units, input, SIZEOF_UNITS);
								file->channel_data[channel].units[SIZEOF_UNITS - 1] = 0x00;

								logmessage(DEBUG, "Found channel units '%s'\n", file->channel_data[channel].units);
							}

							argument++;
						}
					}
				}
			}

			line++;
		}
		else
		{
			switch (data_type)
			{
			case CSV_COLUMN_CYCLES:
			{
				channel = start_cycle;
				argument = 0;

				if (skip_columns != NULL)
				{
					while ((argument < arguments) && (skip_columns[argument] == true))
					{
						argument++;
					}
				}

				for (cycle = 0; cycle < file->number_of_cycles; cycle++)
				{
					if (argument < arguments)
					{
						result = sscanf(output[argument], "%20lf", &voltage_data);

						if (result == 1)
						{
							actual_data = (float)(voltage_data * file->channel_data[channel].slope + file->channel_data[channel].offset);
						}
						else
						{
							actual_data = 0.0f;
						}

						argument++;
					}
					else
					{
						actual_data = 0.0f;
					}

					csv_input_data->data[cycle * file->channel_data[channel].samples_per_cycle + data_pointer] = actual_data;
				}

				data_pointer++;

				done = data_pointer >= file->channel_data[channel].samples_per_cycle;

				break;
			}
			case CSV_COLUMN_CHANNELS:
			{
				argument = 0;
				for (channel = 0; channel < file->number_of_channels; channel++)
				{
					if (file->channel_data[channel].file_flag == true)
					{
						if (skip_columns != NULL)
						{
							while ((argument < arguments) && (skip_columns[argument] == true))
							{
								argument++;
							}
						}

						if (argument < arguments)
						{
							result = sscanf(output[argument], "%20lf", &voltage_data);

							if (result == 1)
							{
								actual_data = (float)(voltage_data * file->channel_data[channel].slope + file->channel_data[channel].offset);
							}
							else
							{
								actual_data = 0.0f;
							}

							argument++;
						}
						else
						{
							actual_data = 0.0f;
						}

						csv_input_data[channel].data[data_pointer] = actual_data;
					}
				}

				data_pointer++;

				crank_angle += 1;

				if (crank_angle >= samples_per_cycle)
				{
					cycle += 1;
					crank_angle = 0;
				}

				done = data_pointer >= data_size;

				break;
			}
			case CSV_SINGLE_CHANNEL:
			default:
			{
				argument = 0;

				if (file->channel_data[channel].file_flag == true)
				{
					if (skip_columns != NULL)
					{
						while ((argument < arguments) && (skip_columns[argument] == true))
						{
							argument++;
						}
					}

					if (argument < arguments)
					{
						result = sscanf(output[argument], "%20lf", &voltage_data);

						if (result == 1)
						{
							actual_data = (float)(voltage_data * file->channel_data[channel].slope + file->channel_data[channel].offset);
						}
						else
						{
							actual_data = 0.0f;
						}

						argument++;
					}
					else
					{
						actual_data = 0.0f;
					}

					csv_input_data->data[data_pointer] = actual_data;
				}

				data_pointer++;

				done = data_pointer >= data_size;

				break;
			}
			}
		}
	}

	fclose(input_file);
	
	/* Save how many data points have been loaded for each channel */
	
	if (data_type == CSV_COLUMN_CHANNELS)
	{
		for (channel = 0; channel < file->number_of_channels; channel++)
		{
			if (file->channel_data[channel].file_flag == true)
			{
				csv_input_data[channel].data_points = data_pointer;

				logmessage(DEBUG, "Channel %u loaded %u data points\n", channel, csv_input_data[channel].data_points);
			}
		}
	}
	else
	{
		csv_input_data->data_points = data_pointer * file->channel_data[channel].samples_per_cycle;

		logmessage(DEBUG, "Channel %u loaded %u data points\n", channel, csv_input_data[channel].data_points);
	}

	if (data_type != CSV_COLUMN_CHANNELS)
	{
		/* Check we have the right number of cycles loaded */

		if (cycle < file->number_of_cycles)
		{
			logmessage(WARNING, "There are only %u cycles available in the data versus the expected %u\n", cycle, file->number_of_cycles);
		}
	}

	if (abscissa_type == ABSCISSA_CRANKANGLE)
	{
		if (data_type == CSV_COLUMN_CHANNELS)
		{
			/* Check how many cycles we have loaded for each channel */

			minimum_cycles = cycle;

			for (channel = 0; channel < file->number_of_channels; channel++)
			{
				if (file->channel_data[channel].file_flag == true)
				{
					tdc_offset = (unsigned int)(file->channel_data[channel].tdc_offset*resolution);
					cycles_loaded = (csv_input_data[channel].data_points - tdc_offset) / file->channel_data[channel].samples_per_cycle;

					if (cycles_loaded < minimum_cycles)
					{
						minimum_cycles = cycles_loaded;
					}

					logmessage(NOTICE, "Channel %s loaded %u raw cycles\n", file->channel_data[channel].name, cycles_loaded);
				}
			}

			/* Discard the TDC offset so all channels are in-phase */

			if (align_tdc == true)
			{
				for (channel = 0; channel < file->number_of_channels; channel++)
				{
					if ((file->channel_data[channel].file_flag == true) && (file->channel_data[channel].tdc_offset > 0.0f))
					{
						data_to_discard = (unsigned int)(file->channel_data[channel].tdc_offset*resolution);

						if (data_to_discard <= csv_input_data[channel].data_points)
						{
							data_size = csv_input_data[channel].data_points - data_to_discard;

							memmove(csv_input_data[channel].data, &csv_input_data[channel].data[data_to_discard], data_size * sizeof(float));

							logmessage(DEBUG, "Discarding %u data points from channel %s\n", data_to_discard, file->channel_data[channel].name);

							csv_input_data[channel].data_points = data_size;
						}
					}
				}
			}

			/* Check for sync */

			sync = true;

			for (channel = 0; channel < file->number_of_channels; channel++)
			{
				if ((file->channel_data[channel].file_flag == true) && (file->channel_data[channel].type == CHAN_CYL_PR))
				{
					first_sample = csv_input_data[channel].data[0];
					mid_cycle_sample = csv_input_data[channel].data[file->engine.number_of_strokes * 90 * resolution];

					if (first_sample > mid_cycle_sample)
					{
						sync = false;

						logmessage(DEBUG, "Acquisition is mis-synchronised on channel %u\n", channel);
					}
				}
			}

			if (sync == false)
			{
				for (channel = 0; channel < file->number_of_channels; channel++)
				{
					if (file->channel_data[channel].file_flag == true)
					{
						data_to_discard = file->engine.number_of_strokes * 90 * resolution;

						if (data_to_discard <= csv_input_data[channel].data_points)
						{
							data_size = csv_input_data[channel].data_points - data_to_discard;

							memmove(csv_input_data[channel].data, &csv_input_data[channel].data[data_to_discard], data_size * sizeof(float));

							logmessage(DEBUG, "Discarding %u data points from channel %s\n", data_to_discard, file->channel_data[channel].name);

							csv_input_data[channel].data_points = data_size;
						}
					}
				}
			}

			/* Check how many cycles we have loaded for each channel */

			for (channel = 0; channel < file->number_of_channels; channel++)
			{
				if (file->channel_data[channel].file_flag == true)
				{
					cycles_loaded = csv_input_data[channel].data_points / file->channel_data[channel].samples_per_cycle;

					if (cycles_loaded < minimum_cycles)
					{
						minimum_cycles = cycles_loaded;
					}

					logmessage(NOTICE, "Channel %s loaded %u in-sync cycles\n", file->channel_data[channel].name, cycles_loaded);
				}
			}

			/* Crop the number of cycles loaded if necessary */

			if (minimum_cycles < file->number_of_cycles)
			{
				file->number_of_cycles = minimum_cycles;

				logmessage(NOTICE, "Number of cycles cropped to %u\n", file->number_of_cycles);
			}

			if (minimum_cycles > file->number_of_cycles)
			{
				logmessage(NOTICE, "Number of cycles cropped to %u\n", file->number_of_cycles);
			}

			/* Save to crank angle data */

			for (channel = 0; channel < file->number_of_channels; channel++)
			{
				if (file->channel_data[channel].file_flag == true)
				{
					logmessage(NOTICE, "Saving %u cycles of data to channel %s starting at cycle %u\n", file->number_of_cycles, file->channel_data[channel].name, start_cycle);

					memcpy(&file->channel_data[channel].data[start_cycle*file->channel_data[channel].samples_per_cycle], csv_input_data[channel].data, file->channel_data[channel].samples_per_cycle * file->number_of_cycles * sizeof(float));

					file->channel_data[channel].loaded = true;
				}
			}
		}
		else
		{
			/* Save to crank angle data */

			if (file->channel_data[channel].file_flag == true)
			{
				logmessage(NOTICE, "Saving channel %s as crank angle data\n", file->channel_data[channel].name);

				memcpy(file->channel_data[channel].data, csv_input_data->data, file->channel_data[channel].samples_per_cycle * file->number_of_cycles * sizeof(float));

				file->channel_data[channel].loaded = true;
			}
		}
	}
	else if (abscissa_type == ABSCISSA_TIME)
	{
		/* Save to time based data */
		
		for (channel=0;channel<file->number_of_channels;channel++)
		{
			if (file->channel_data[channel].file_flag == true)
			{
				logmessage(NOTICE,"Saving channel %u as time based data with %u data points\n",channel,data_size);	

				memcpy(file->channel_data[channel].data, csv_input_data[channel].data, data_size*sizeof(float));
				
				file->channel_data[channel].loaded = true;
			}
		}
	}
	else if (abscissa_type == ABSCISSA_CYCLE)
	{
		if (file->channel_data[channel].file_flag == true)
		{
			logmessage(NOTICE, "Saving channel %u as cycle data\n", channel);

			memcpy(file->channel_data[channel].data, csv_input_data->data, file->number_of_cycles * sizeof(float));

			file->channel_data[channel].loaded = true;
		}
	}
	else
	{
		/* Shouldn't be able to get this far! */
	}
	
	if (output != NULL)
	{
		for (column = 0; column<MAX_NUMBER_OF_COLUMNS; column++)
		{
			if (output[column] != NULL)
			{
				free(output[column]);
				output[column] = NULL;
			}
		}
			
		free(output);
		output = NULL;
	}
	
	if (csv_input_data != NULL)
	{
		if (data_type == CSV_COLUMN_CHANNELS)
		{
			for (channel = 0; channel < file->number_of_channels; channel++)
			{
				if (csv_input_data[channel].data != NULL)
				{
					free(csv_input_data[channel].data);
					csv_input_data[channel].data = NULL;
				}
			}
		}
		else
		{
			free(csv_input_data->data);
			csv_input_data->data = NULL;
		}

		free(csv_input_data);
		csv_input_data = NULL;
	}
		
	logmessage(NOTICE, "Loaded %u cycles of data\n", file->number_of_cycles);
		
	return(file->number_of_cycles);
}

bool save_csv_file(
	const FileData* file,
#ifdef _CATOOL_UNICODE_
	const wchar_t* filename,
#else
	const char* filename,
#endif
	const Analysis* analysis,
	const unsigned int output_data)
{
	unsigned int analysis_channel;
	unsigned int cycle;
	unsigned int crank_angle;
	unsigned int channel;
	unsigned int channel_to_analyse;
	unsigned int analysis_number;
	unsigned int abscissa_type;
	float* volume = NULL;
	FILE* output_file = NULL;
	bool output;

#ifdef _CATOOL_UNICODE_
    output_file = _wfopen(filename, L"w");
#else
	output_file = fopen(filename, "w");
#endif
	
	if (output_file == NULL)
	{
#ifdef _CATOOL_UNICODE_
		logmessage(MSG_ERROR, "Could not open '%ls' for output\n", filename);
#else
		logmessage(MSG_ERROR, "Could not open '%s' for output\n", filename);
#endif
		return(false);
	}
	
	if ((output_data & IFILE_RESULTS) && ((analysis == NULL) || (analysis->number_of_channels == 0)))
	{
		fclose(output_file);
		logmessage(NOTICE,"No analysis to output\n");
		
		return(false);
	}

	if ((output_data & IFILE_RTP_RESULTS) != false)
	{
		logmessage(WARNING,"RTP_RESULTS not implemented in CSV file output.  Use RESULTS for cycle based results and CA for crank angle based results\n");
	}
	
	if ((output_data & IFILE_ASYNC_UTC) != false)
	{
		logmessage(WARNING,"ASYNC_UTC not implemented in CSV file output\n");
	}
	
	if ((output_data & IFILE_TIME) > 0)
	{
        for (channel=0;channel<file->number_of_channels;channel++)
		{
			if ((file->channel_data[channel].abscissa.type == ABSCISSA_TIME) && (file->channel_data[channel].loaded == true))
			{
				fprintf(output_file,"Channel %s\n\n",file->channel_data[channel].name);
		    	
				logmessage(NOTICE,"Outputting raw time data for channel %u (%s)\n",channel+1,file->channel_data[channel].name);
				
    			for (crank_angle=0;crank_angle<file->channel_data[channel].samples_per_cycle;crank_angle++)
    	    	{
    				fprintf(output_file,"%f,%f\n",(float)crank_angle/file->channel_data[channel].abscissa.resolution[0]*1000.0f,ReturnCAData(file,0,crank_angle,channel));
    			}
    			fprintf(output_file,"\n");
            }
        }		
    }
    
	if ((output_data & IFILE_CA_RAW) > 0)
	{		
		for (channel=0;channel<file->number_of_channels;channel++)
		{
			if ((file->channel_data[channel].abscissa.type == ABSCISSA_CRANKANGLE) && (file->channel_data[channel].loaded == true))
			{
				fprintf(output_file,"Channel %s\n\n",file->channel_data[channel].name);
		    	
				logmessage(NOTICE,"Outputting raw crank angle data for channel %u (%s)\n",channel,file->channel_data[channel].name);
				
				for (cycle=0;cycle<file->number_of_cycles;cycle++)
				{
					fprintf(output_file,",%u",cycle+1);
				}
				fprintf(output_file,"\n");

				for (crank_angle=0;crank_angle<file->channel_data[channel].samples_per_cycle;crank_angle++)
		    	{
					fprintf(output_file,"%f",CrankAngleToDegrees(file,crank_angle,channel));

					for (cycle=0;cycle<file->number_of_cycles;cycle++)
					{
						fprintf(output_file,",%f",ReturnCAData(file,cycle,crank_angle,channel));
					}

					fprintf(output_file,"\n");
				}
				fprintf(output_file,"\n");
			}
		}
	}

	if ((output_data & IFILE_RESULTS_RAW) > 0)
	{
		output = false;

		for (channel = 0; channel < file->number_of_channels; channel++)
		{
			if ((file->channel_data[channel].abscissa.type == ABSCISSA_CYCLE) && (file->channel_data[channel].loaded == true))
			{
				output = true;
			}
		}

		if (output == true)
		{
			for (channel = 0; channel < file->number_of_channels; channel++)
			{
				if ((file->channel_data[channel].abscissa.type == ABSCISSA_CYCLE) && (file->channel_data[channel].loaded == true))
				{
					fprintf(output_file, ",%s", file->channel_data[channel].name);
				}
			}
			fprintf(output_file, "\n");

			for (channel = 0; channel < file->number_of_channels; channel++)
			{
				if ((file->channel_data[channel].abscissa.type == ABSCISSA_CYCLE) && (file->channel_data[channel].loaded == true))
				{
					fprintf(output_file, ",%s", file->channel_data[channel].units);
				}
			}
			fprintf(output_file, "\n");

			for (cycle = 0; cycle < file->number_of_cycles; cycle++)
			{
				fprintf(output_file, "%u", cycle + 1);

				for (channel = 0; channel < file->number_of_channels; channel++)
				{
					if ((file->channel_data[channel].abscissa.type == ABSCISSA_CYCLE) && (file->channel_data[channel].loaded == true))
					{
						fprintf(output_file, ",%f", file->channel_data[channel].data[cycle]);
					}
				}

				fprintf(output_file, "\n");
			}
		}
	}
	
	if ((analysis != NULL) && (((output_data & (IFILE_RESULTS | IFILE_RESULTS_STATS)) > 0)))
	{
		for (analysis_number=0;analysis_number<analysis->number_of_channels;analysis_number++)
		{
			channel_to_analyse = analysis->channel[analysis_number].channel;
	
			abscissa_type = file->channel_data[channel_to_analyse].abscissa.type;
			
			if (abscissa_type == ABSCISSA_CRANKANGLE)
			{			
		    	fprintf(output_file,"Cycle based analysis for channel %s\n",file->channel_data[channel_to_analyse].name);

		    	logmessage(NOTICE,"Cycle based analysis for channel %u\n",channel_to_analyse);
				
		        /* Cycle based output */
				
		    	for (analysis_channel=0;analysis_channel<get_number_of_analysis_channels();analysis_channel++)
		    	{
		            if ((analysis->channel[analysis_number].analysis_rqst[analysis_channel] >= AnalysisRqst::CalculateAndStore) && 
						(analysis->channel[analysis_number].results[analysis_channel].abscissa.type == ABSCISSA_CYCLE))
		            {
						fprintf(output_file,",%s",analysis->channel[analysis_number].results[analysis_channel].name);
					}
				}
		    	fprintf(output_file,"\n");
			    
		    	for (analysis_channel=0;analysis_channel<get_number_of_analysis_channels();analysis_channel++)
		    	{
		            if ((analysis->channel[analysis_number].analysis_rqst[analysis_channel] >= AnalysisRqst::CalculateAndStore) && 
						(analysis->channel[analysis_number].results[analysis_channel].abscissa.type == ABSCISSA_CYCLE))
		            {
						fprintf(output_file,",%s",analysis->channel[analysis_number].results[analysis_channel].units);
					}
				}
		    	fprintf(output_file,"\n");
				    
				if ((output_data & IFILE_RESULTS) > 0)
				{
					for (cycle = 0; cycle < file->number_of_cycles; cycle++)
					{
						fprintf(output_file, "%u", cycle + 1);

						for (analysis_channel = 0; analysis_channel < get_number_of_analysis_channels(); analysis_channel++)
						{
							if ((analysis->channel[analysis_number].analysis_rqst[analysis_channel] >= AnalysisRqst::CalculateAndStore) && 
								(analysis->channel[analysis_number].results[analysis_channel].abscissa.type == ABSCISSA_CYCLE))
							{
								fprintf(output_file, ",%f", analysis->channel[analysis_number].results[analysis_channel].data[cycle]);
							}
						}

						fprintf(output_file, "\n");
					}
				}
				
				if (file->number_of_cycles > 1)
				{
					fprintf(output_file,"Mean");
					for (analysis_channel=0;analysis_channel<get_number_of_analysis_channels();analysis_channel++)
	    			{
	    				if ((analysis->channel[analysis_number].analysis_rqst[analysis_channel] >= AnalysisRqst::CalculateAndStore) && 
							(analysis->channel[analysis_number].results[analysis_channel].abscissa.type == ABSCISSA_CYCLE))
	    				{
							fprintf(output_file,",%f",analysis->channel[analysis_number].statistics[analysis_channel].mean);
						}
					}
					fprintf(output_file,"\n");
	
					fprintf(output_file,"Max");
					for (analysis_channel=0;analysis_channel<get_number_of_analysis_channels();analysis_channel++)
	    			{
	    				if ((analysis->channel[analysis_number].analysis_rqst[analysis_channel] >= AnalysisRqst::CalculateAndStore) && 
							(analysis->channel[analysis_number].results[analysis_channel].abscissa.type == ABSCISSA_CYCLE))
	    				{
							fprintf(output_file,",%f",analysis->channel[analysis_number].statistics[analysis_channel].max);
						}
					}
					fprintf(output_file,"\n");
					
					fprintf(output_file,"Range");
					for (analysis_channel=0;analysis_channel<get_number_of_analysis_channels();analysis_channel++)
	    			{
	    				if ((analysis->channel[analysis_number].analysis_rqst[analysis_channel] >= AnalysisRqst::CalculateAndStore) && 
							(analysis->channel[analysis_number].results[analysis_channel].abscissa.type == ABSCISSA_CYCLE))
	    				{
							fprintf(output_file,",%f",analysis->channel[analysis_number].statistics[analysis_channel].range);
						}
					}
					fprintf(output_file,"\n");
	
					fprintf(output_file,"Min");
					for (analysis_channel=0;analysis_channel<get_number_of_analysis_channels();analysis_channel++)
	    			{
	    				if ((analysis->channel[analysis_number].analysis_rqst[analysis_channel] >= AnalysisRqst::CalculateAndStore) && 
							(analysis->channel[analysis_number].results[analysis_channel].abscissa.type == ABSCISSA_CYCLE))
	    				{
							fprintf(output_file,",%f",analysis->channel[analysis_number].statistics[analysis_channel].min);
						}
					}
					fprintf(output_file,"\n");
	
					fprintf(output_file,"Std. Dev.");
					for (analysis_channel=0;analysis_channel<get_number_of_analysis_channels();analysis_channel++)
	    			{
	    				if ((analysis->channel[analysis_number].analysis_rqst[analysis_channel] >= AnalysisRqst::CalculateAndStore) && 
							(analysis->channel[analysis_number].results[analysis_channel].abscissa.type == ABSCISSA_CYCLE))
	    				{
							fprintf(output_file,",%f",analysis->channel[analysis_number].statistics[analysis_channel].stddev);
						}
					}
					fprintf(output_file,"\n");
	
					fprintf(output_file,"CoV");
					for (analysis_channel=0;analysis_channel<get_number_of_analysis_channels();analysis_channel++)
	    			{
	    				if ((analysis->channel[analysis_number].analysis_rqst[analysis_channel] >= AnalysisRqst::CalculateAndStore) && 
							(analysis->channel[analysis_number].results[analysis_channel].abscissa.type == ABSCISSA_CYCLE))
	    				{
							fprintf(output_file,",%f",analysis->channel[analysis_number].statistics[analysis_channel].cov);
						}
					}
					fprintf(output_file,"\n");
	
					fprintf(output_file,"LNV");
					for (analysis_channel=0;analysis_channel<get_number_of_analysis_channels();analysis_channel++)
	    			{
	    				if ((analysis->channel[analysis_number].analysis_rqst[analysis_channel] >= AnalysisRqst::CalculateAndStore) && 
							(analysis->channel[analysis_number].results[analysis_channel].abscissa.type == ABSCISSA_CYCLE))
	    				{
							fprintf(output_file,",%f",analysis->channel[analysis_number].statistics[analysis_channel].lnv);
						}
					}
					fprintf(output_file,"\n");
	
					fprintf(output_file,"Skewness");
					for (analysis_channel=0;analysis_channel<get_number_of_analysis_channels();analysis_channel++)
	    			{
	    				if ((analysis->channel[analysis_number].analysis_rqst[analysis_channel] >= AnalysisRqst::CalculateAndStore) && 
							(analysis->channel[analysis_number].results[analysis_channel].abscissa.type == ABSCISSA_CYCLE))
	    				{
							fprintf(output_file,",%f",analysis->channel[analysis_number].statistics[analysis_channel].skewness);
						}
					}
					fprintf(output_file,"\n");
	
					fprintf(output_file,"Kurtosis");
					for (analysis_channel=0;analysis_channel<get_number_of_analysis_channels();analysis_channel++)
	    			{
	    				if ((analysis->channel[analysis_number].analysis_rqst[analysis_channel] >= AnalysisRqst::CalculateAndStore) && 
							(analysis->channel[analysis_number].results[analysis_channel].abscissa.type == ABSCISSA_CYCLE))
	    				{
							fprintf(output_file,",%f",analysis->channel[analysis_number].statistics[analysis_channel].kurtosis);
						}
					}
					fprintf(output_file,"\n");
				}

		    	fprintf(output_file,"\n");
			}
		}
		
		if ((analysis != NULL) && ((output_data & IFILE_CA) > 0))
		{
			for (analysis_number=0;analysis_number<analysis->number_of_channels;analysis_number++)
			{
				channel_to_analyse = analysis->channel[analysis_number].channel;

				abscissa_type = file->channel_data[channel_to_analyse].abscissa.type;

				if (abscissa_type == ABSCISSA_CRANKANGLE)
				{
			        /* Crank angle based output */
		        	
					volume = (float*)malloc(file->channel_data[channel_to_analyse].samples_per_cycle * sizeof(float));
		        	if (volume == NULL)
		        	{
		            	logmessage(FATAL,"Memory could not be allocated");
		        	}

			    	Return_Geometry_Data(file,channel_to_analyse,NULL,volume,NULL,NULL,NULL,NULL);

			    	fprintf(output_file,"Mean crank angle based analysis results for channel %s\n",file->channel_data[channel_to_analyse].name);
			    	
                    logmessage(NOTICE,"Mean crank angle based analysis results for channel %u\n",channel_to_analyse);

			    	fprintf(output_file,",Volume,%s",file->channel_data[channel_to_analyse].name);
			    	for (analysis_channel=0;analysis_channel<get_number_of_analysis_channels();analysis_channel++)
			    	{
			    		if ((analysis->channel[analysis_number].analysis_rqst[analysis_channel] >= AnalysisRqst::CalculateAndStore) &&
							(analysis->channel[analysis_number].results[analysis_channel].abscissa.type == ABSCISSA_CRANKANGLE))
			    		{
							fprintf(output_file,",%s",analysis->channel[analysis_number].results[analysis_channel].name);
						}
					}
			    	fprintf(output_file,"\n");

			    	fprintf(output_file,",cc,%s",file->channel_data[channel_to_analyse].units);
			    	for (analysis_channel=0;analysis_channel<get_number_of_analysis_channels();analysis_channel++)
			    	{
			    		if ((analysis->channel[analysis_number].analysis_rqst[analysis_channel] >= AnalysisRqst::CalculateAndStore) &&
							(analysis->channel[analysis_number].results[analysis_channel].abscissa.type == ABSCISSA_CRANKANGLE))
			    		{
							fprintf(output_file,",%s",analysis->channel[analysis_number].results[analysis_channel].units);
						}
					}
			    	fprintf(output_file,"\n");

			    	for (crank_angle=0;crank_angle<file->channel_data[channel_to_analyse].samples_per_cycle;crank_angle++)
			    	{
						fprintf(output_file, "%f,%f,%f", CrankAngleToDegrees(file, crank_angle, channel_to_analyse), volume[crank_angle]*CUBIC_MM_TO_CM, analysis->channel[analysis_number].raw_ca_statistics.mean[crank_angle]);

			    		for (analysis_channel=0;analysis_channel<get_number_of_analysis_channels();analysis_channel++)
			    		{
			    			if ((analysis->channel[analysis_number].analysis_rqst[analysis_channel] >= AnalysisRqst::CalculateAndStore) &&
								(analysis->channel[analysis_number].results[analysis_channel].abscissa.type == ABSCISSA_CRANKANGLE))
			    			{
								fprintf(output_file,",%f",analysis->channel[analysis_number].ca_statistics[analysis_channel].mean[crank_angle]);
							}
						}

						fprintf(output_file,"\n");
			    	}
			    	fprintf(output_file,"\n");

			    	free(volume);
				}
			}
		}
	}
	
	fclose(output_file);

	logmessage(NOTICE,"CSV file output complete\n");

	return(true);
}
