/* Combustion Analysis Tool (CAT)
   www.catool.org
   
   Filename: catool.c

   Purpose:  Provide common programming interface for processing acquired engine data
  
   Author:   Ben Brown
   Version:  1.2
   Date:     19.10.2015

   Revision: GPL Release

   Copyright (C) Xarin Limited, 2000-2024

      This program is free software: you can redistribute it and/or modify
      it under the terms of version 2 of the GNU General Public License as
      published by the Free Software Foundation.

      This program is distributed in the hope that it will be useful,
      but WITHOUT ANY WARRANTY; without even the implied warranty of
      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
      GNU General Public License for more details.

      You should have received a copy of the GNU General Public License
      along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* For Microsoft Visual C++ */
#ifdef _MSC_VER
#include "stdafx.h"
#endif

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "cat.h"

unsigned int debug_level = DEBUG;

#if !defined(_LIBCATOOL)
#if defined(_CATOOL_UNICODE_)
void ifile_to_matlab(FileData* file, wchar_t* input_filename, wchar_t* output_filename)
#else
void ifile_to_matlab(FileData* file, char* input_filename, char* output_filename)
#endif
{
	unsigned int channel;
	Analysis* analysis = NULL;

	unsigned int ifile_input = 0;
	unsigned int output_data = 0;

	ifile_input |= IFILE_CA;
	ifile_input |= IFILE_TIME;
	ifile_input |= IFILE_RTP_RESULTS;
	ifile_input |= IFILE_RESULTS;
	ifile_input |= IFILE_RESULTS_RAW;
	ifile_input |= IFILE_ASYNC_UTC;

	output_data |= IFILE_CA_RAW;
	output_data |= IFILE_CA;
	output_data |= IFILE_TIME;
	output_data |= IFILE_RTP_RESULTS;
	output_data |= IFILE_RESULTS;
	output_data |= IFILE_RESULTS_RAW;

	file->file_type = FILE_AVLIFILE;

	load_ifile_header(file,input_filename,ifile_input);

	for (channel=0;channel<file->number_of_channels;channel++)
	{
		logmessage(DEBUG,"Loading AVL channel %u\n",channel);
		
		load_ifile_channel(file,channel);
	}

	for (channel = 0; channel < file->number_of_channels; channel++)
	{
		if (file->channel_data[channel].loaded == true)
		{
			if (file->channel_data[channel].abscissa.type == ABSCISSA_CRANKANGLE)
			{
				cycle_duration_analysis(file, channel, N_AVLRZT, "SPEED", 3000.0f);

				offset_correction_channel(file, NULL, channel);
			}
		}
	}
				
	file->sync_checked = false;
	
	save_matlab_file(file,output_filename,analysis,output_data);
}

int main(int argc, char* argv[])
{
	unsigned int argument;
	size_t argument_length;
	size_t equals_position;
	unsigned int debug_level_tmp = debug_level;
	char option_argument[4096] = "\0";
	char option[4096] = "\0";
	char parameter[4096] = "\0";
#if defined(_CATOOL_UNICODE_)
	wchar_t config_filename[4096] = L"\0\0";
	wchar_t matfilename[4096] = L"\0\0";
	wchar_t ifilename[4096] = L"\0\0";
#else
	char config_filename[4096] = "\0";
	char matfilename[4096] = "\0";
	char ifilename[4096] = "\0";
#endif
	FileData* file = NULL;
	bool debug_level_fixed;
	bool fatal = false;

	if (argc < 2)
	{
		printf("\nUsage: catool [-h] [--ifile=<ifile_filename> --matfile=<matfile_filename>] [--debug-level=<debug_level>] [file ...]\n");
		return(1);
	}
	else
	{
		for (argument=1;argument<(unsigned int)argc;argument++)
		{
			if (strncmp(argv[argument],"--",2) == 0)
			{
				argument_length = strlen(argv[argument])-2;
				if (argument_length > 4095)
				{
					argument_length = 4095;
				}
					
				strncpy(option_argument,argv[argument]+2,argument_length);
				option_argument[argument_length] = '\0';
				option[0] = '\0';
				parameter[0] = '\0';
				
				if (strchr(option_argument,'=') == NULL)
				{
					equals_position = 0;
				}
				else
				{
					equals_position = strchr(option_argument,'=') - option_argument;
				}
					
				if ((equals_position > 0) && (equals_position < 4094))
				{
					strncpy(option,option_argument,equals_position);
					option[equals_position] = 0x00;
					strncpy(parameter,option_argument+equals_position+1,argument_length-equals_position);
					parameter[argument_length-equals_position] = 0x00;
				}
				else
				{
					strncpy(option,option_argument,4095);
					option[4095] = 0x00;
				}

				if (case_insensitive_compare(option,"ifile",4095) == 1)
				{
#if defined(_CATOOL_UNICODE_)
					size_t newsize = strlen(parameter) + 1;
					wchar_t* wcstring = new wchar_t[newsize];
					size_t convertedChars = 0;
					mbstowcs_s(&convertedChars, wcstring, newsize, parameter, _TRUNCATE);

					wcsncpy(ifilename, wcstring, 4095);
					delete[] wcstring;

					ifilename[4095] = 0x0000;
#else
					strncpy(ifilename,parameter,sizeof(ifilename));
					ifilename[4095] = 0x00;
#endif
					
				}
				else if (case_insensitive_compare(option,"matfile",4095) == 1)
				{
#if defined(_CATOOL_UNICODE_)
					size_t newsize = strlen(parameter) + 1;
					wchar_t* wcstring = new wchar_t[newsize];
					size_t convertedChars = 0;
					mbstowcs_s(&convertedChars, wcstring, newsize, parameter, _TRUNCATE);

					wcsncpy(matfilename, wcstring, 4095);
					delete[] wcstring;

					matfilename[4095] = 0x0000;
#else
					strncpy(matfilename,parameter,sizeof(matfilename));
					matfilename[4095 - 1] = 0x00;
#endif
					
				}
				else if (set_uint_parameter(option,parameter,"debug-level",&debug_level_tmp) == 2)
				{
					if (case_insensitive_compare(parameter,"DEBUG",4095) == 1)
					{
						debug_level_tmp = DEBUG + 100;
					}
		 			else if (case_insensitive_compare(parameter,"NOTICE",4095) == 1)
		 			{
						debug_level_tmp = NOTICE + 100;
					}
					else if (case_insensitive_compare(parameter,"WARNING",4095) == 1)
					{
						debug_level_tmp = WARNING + 100;
					}
					else if (case_insensitive_compare(parameter,"ERROR",4095) == 1)
					{
						debug_level_tmp = MSG_ERROR + 100;
					}
					else if (case_insensitive_compare(parameter,"FATAL",4095) == 1)
					{
						debug_level_tmp = FATAL + 100;
					}
					else if (case_insensitive_compare(parameter,"SILENT",4095) == 1)
					{
						debug_level_tmp = SILENT + 100;
					}
					else
					{
						printf("I do not understand the debug level '%s'\n",parameter);
					}
				}
			}
			else if ((strncmp(argv[argument],"-",1) == 0) && (strlen(argv[argument]) > 1))
			{
				strncpy(option,argv[argument]+1,4095);
				option[4095] = 0x00;
				
				switch (option[0])
				{
					case 'h':
					{
						printf("\nUsage: catool [-h] [--ifile=<ifile_filename> --matfile=<matfile_filename>] [--debug-level=<debug_level>] [file ...]\n");

						return(0);
					}
					default:
					{
						/* Do Nothing */
						break;
					}
				}
			}
			else
			{
#if defined(_CATOOL_UNICODE_)
				size_t newsize = strlen(argv[argument]) + 1;
				wchar_t* wcstring = new wchar_t[newsize];
				size_t convertedChars = 0;
				mbstowcs_s(&convertedChars, wcstring, newsize, argv[argument], _TRUNCATE);

				wcsncpy(config_filename, wcstring, 4095);
				delete[] wcstring;

				config_filename[4095] = 0x0000;
#else
				strncpy(config_filename,argv[argument],4095);
				config_filename[4095] = 0x00;
#endif
				
			}
		}
	}

	if (debug_level_tmp >= 100)
	{
		debug_level_fixed = true;
		debug_level = debug_level_tmp - 100;
	}
	else
	{
		debug_level_fixed = false;
		debug_level = debug_level_tmp;
	}
	
	initialise_catool();

	file = Initialise_File();
		
	logmessage(NOTICE,"\nCombustion Analysis Tool (catool)\n");
	logmessage(NOTICE,"Copyright (C) Xarin Limited, 2000-2025\n");
#if defined(_WIN64) || defined(__x86_64__) || defined(__amd64__) || defined(__ia64__) || defined(__ia64) || defined(_M_IA64)
	logmessage(NOTICE,"Version 2.2.1 64-bit\n");
#else
	logmessage(NOTICE,"Version 2.2.1\n");
#endif
	logmessage(NOTICE,"www.catool.org\n");
#if defined(_SAFE_MEMORY)
	logmessage(NOTICE,"\nCompiled with SAFE_MEMORY enabled\n");
#endif
#if defined(_USE_PTHREADS)
	logmessage(NOTICE, "\nCompiled with pthreads enabled\n");
#elif defined(_USE_STD_THREAD)
	logmessage(NOTICE, "\nCompiled with std::thread enabled\n");
#endif
	logmessage(NOTICE,"\ncatool incorporates code from the public domain FFTPACK library (www.netlib.org/fftpack)\n");
	
	logmessage(NOTICE,"\nThis program is distributed in the hope that it will be useful,\n");
	logmessage(NOTICE,"but WITHOUT ANY WARRANTY; without even the implied warranty of\n");
	logmessage(NOTICE,"MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n");
	logmessage(NOTICE,"GNU General Public License for more details.\n");
	
	if (sizeof(char)      != 1) { logmessage(MSG_ERROR,"sizeof(char) not 1\n"); fatal = true; }
	if (sizeof(short)     != 2) { logmessage(MSG_ERROR,"sizeof(short) not 2\n"); fatal = true; }
	if (sizeof(int)       != 4) { logmessage(MSG_ERROR,"sizeof(int) not 4\n"); fatal = true; }
	if (sizeof(float)     != 4) { logmessage(MSG_ERROR,"sizeof(float) not 4\n"); fatal = true; }
	if (sizeof(double)    != 8) { logmessage(MSG_ERROR,"sizeof(double) not 8\n"); fatal = true; }
	if (sizeof(long long) != 8) { logmessage(MSG_ERROR,"sizeof(long long) not 8\n"); fatal = true; }
	if (sizeof(uint8_t)   != 1) { logmessage(MSG_ERROR,"sizeof(uint8_t) not 1\n"); fatal = true; }
	if (sizeof(int8_t)    != 1) { logmessage(MSG_ERROR,"sizeof(int8_t) not 1\n"); fatal = true; }
	if (sizeof(uint16_t)  != 2) { logmessage(MSG_ERROR,"sizeof(uint16_t) not 2\n"); fatal = true; }
	if (sizeof(int16_t)   != 2) { logmessage(MSG_ERROR,"sizeof(int16_t) not 2\n"); fatal = true; }
	if (sizeof(uint32_t)  != 4) { logmessage(MSG_ERROR,"sizeof(uint32_t) not 4\n"); fatal = true; }
	if (sizeof(int32_t)   != 4) { logmessage(MSG_ERROR,"sizeof(int32_t) not 4\n"); fatal = true; }
	if (sizeof(uint64_t)  != 8) { logmessage(MSG_ERROR,"sizeof(uint64_t) not 8\n"); fatal = true; }
	if (sizeof(int64_t)   != 8) { logmessage(MSG_ERROR,"sizeof(int64_t) not 8\n"); fatal = true; }
	
	if (fatal == true)
	{
		logmessage(FATAL,"Variable size mismatch\n");
	}

	if ((ifilename[0] > 0) && (matfilename[0] > 0))
	{
		logmessage(NOTICE,"\nIFile: %s\n",ifilename);
		logmessage(NOTICE,"MATFile: %s\n",matfilename);

		ifile_to_matlab(file,ifilename,matfilename);
	}
	else
	{
#if defined(_CATOOL_UNICODE_)
		if (wcsncmp(config_filename, L"-", 4095) == 0)
#else
		if (strncmp(config_filename, "-", 4095) == 0)
#endif
		{
			logmessage(NOTICE,"\nConfig File: stdin\n\n");
		}
		else
		{
			logmessage(NOTICE, "\nConfig File: %s\n\n", config_filename);
		}

		(void)load_config_file(file,config_filename,debug_level_fixed);
	}

	(void)Delete_File(file);
	(void)Release_File(&file);
	
	logmessage(NOTICE,"\nAnalysis complete\n");

	uninitialise_catool();

 	return(0);
}
#endif

