/* Combustion Analysis Tool (CAT)
   www.catool.org
   
   Filename: cat.h

   Purpose:  Provide common programming interface for processing acquired engine data
  
   Author:   Ben Brown
   Version:  1.2
   Date:     19.10.2015

   Revision: GPL Release

   Copyright (C) Xarin Limited, 2000-2024

      This program is free software: you can redistribute it and/or modify
      it under the terms of version 2 of the GNU General Public License as
      published by the Free Software Foundation.

      This program is distributed in the hope that it will be useful,
      but WITHOUT ANY WARRANTY; without even the implied warranty of
      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
      GNU General Public License for more details.

      You should have received a copy of the GNU General Public License
      along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef __CAT_H__
#define __CAT_H__

#include <float.h>
#include <limits.h>
#include <math.h>
#include <stdlib.h>
#include <time.h>
#include "dat670.h"

#ifdef _MSC_VER
#include "stdafx.h"
#else
#include <stdbool.h>
#ifndef strnlen_s
#define strnlen_s(str, strsz) ((str) == NULL) ? 0 : strnlen((str),(strsz))
#endif
#endif

#if defined(_USE_PTHREADS)
#include <pthread.h>
#define THREAD_STACKSIZE					1048576
#elif defined(_USE_STD_THREAD)
#include <thread>
#endif

/* Mathematical constants */

constexpr float FLT_PI = 3.141592653589793238462643383280f;
constexpr float FLT_e = 2.718281828459045235360287471352f;

/* Conversion factors */

constexpr float BAR_TO_PA = 100000.0f;
constexpr float PA_TO_BAR = (1.0f / BAR_TO_PA);
constexpr float BAR_TO_KPA = 100.0f;
constexpr float KPA_TO_BAR = (1.0f / BAR_TO_KPA);
constexpr float BAR_TO_PSI = 14.5037743897283109469992923608f;
constexpr float PSI_TO_BAR = (1.0f / BAR_TO_PSI);
constexpr float MPA_TO_BAR = 10.0f;
constexpr float BAR_TO_MPA = (1.0f / MPA_TO_BAR);
constexpr float MILLIBAR_TO_KPA = 0.1f;
constexpr float LEN_MM_TO_M = 1E-3f;
constexpr float SQ_MM_TO_M = 1E-6f;
constexpr float CUBIC_MM_TO_M = 1E-9f;
constexpr float CUBIC_MM_TO_CM = 1E-3f;
constexpr float CUBIC_CM_TO_M = 1E-6f;
constexpr float INCHES_TO_MM = 25.4f;
constexpr float M_TO_MM = 1E3;
constexpr float JOULES_TO_KJ = 1E-3f;
constexpr float GRAD_TO_DEG = 0.9f;
constexpr float RAD_TO_DEG = 360.0f / 2.0f / FLT_PI;
constexpr float DEG_TO_RAD = 2.0f * FLT_PI / 360.0f;
constexpr float SQUARE_INCHES_TO_SQUARE_MM = INCHES_TO_MM * INCHES_TO_MM;
constexpr float CUBIC_INCHES_TO_CUBIC_MM = INCHES_TO_MM * INCHES_TO_MM * INCHES_TO_MM;
constexpr float RADIANS_PER_REV = 2 * FLT_PI;
constexpr float MILLIVOLTS_TO_VOLTS = 1000.0f;
constexpr float LITRES_TO_CUBIC_M = 0.001f;
constexpr float NM_TO_LBFT = 0.737562149f;
constexpr float KW_TO_HP = 1.3410220888f;
constexpr float KG_TO_G = 1000.0f;
constexpr float KG_TO_MG = 1000000.0f;
constexpr float C_TO_K = 273.15f;
constexpr float MILLISECONDS_TO_SECONDS = 1E-3f;

/* Definitions */

#define NUMBER_OF_ANALYSIS_CHANNELS			194U		/* Not a restriction, just the number in use */
#define DEFAULT_MIN_RES_PPR					7200U		/* Samples per rev, can be overridden */
#define MINIMUM_THETA_ANGLE_MULT			180			/* Multiplied by number of strokes to give minimum theta angle (90 = -360, 180 = -720) */
#define DEFAULT_TIME_RESOLUTION				0.0001		/* Milliseconds */
#define MAX_NUMBER_OF_CYLINDERS				16			/* For storing wrist pin offsets */
#define MAX_LINE_SIZE 						65536
#define MAX_NUMBER_OF_COLUMNS 				2000

constexpr size_t SIZEOF_CHANNEL_NAME = 64;
constexpr size_t SIZEOF_SHORT_NAME = 10;
constexpr size_t SIZEOF_UNITS = 10;
constexpr size_t SIZEOF_DESCRIPTION = 256;
constexpr size_t SIZEOF_ENGINE_NAME = 18;

/* Cycle Based */

enum {
	AVERAGE_EXHAUST_ABSOLUTE_PRESSURE = 0,
	BURN_ANGLE_1,
	BURN_ANGLE_2,
	BURN_ANGLE_5,
	BURN_ANGLE_10,
	BURN_ANGLE_20,
	BURN_ANGLE_25,
	BURN_ANGLE_50,
	BURN_ANGLE_75,
	BURN_ANGLE_80,
	BURN_ANGLE_90,
	BURN_ANGLE_95,
	BURN_ANGLE_98,
	BURN_ANGLE_99,
	BURN_DURATION_0_2,
	BURN_DURATION_0_5,
	BURN_DURATION_0_10,
	BURN_DURATION_0_90,
	BURN_DURATION_2_90,
	BURN_DURATION_5_90,
	BURN_DURATION_10_90,
	CENTRE_OF_GRAVITY,
	CAM_ADVANCE,
	CAM_ANGLE,
	CAM_EDGE_1,
	CAM_EDGE_2,
	CAM_EDGE_3,
	CAM_EDGE_4,
	CAM_EDGE_5,
	CAM_EDGE_6,
	CAM_EDGE_7,
	CAM_EDGE_8,
	CHANNEL_DELTA_MAX,
	CHANNEL_DELTA_MAX_CA,
	CID_ANGLE,
	DELTA_P,
	DIFFERENCE_PRESSURE_10,
	DIFFERENCE_PRESSURE_INTEGRAL,
	DILPAR,
	DWELL_TIME,
	EEOC,
	EEVO,
	EIVC,
	END_OF_INJECTION_1,
	END_OF_INJECTION_2,
	END_OF_INJECTION_3,
	END_OF_INJECTION_4,
	END_OF_INJECTION_5,
	END_OF_INJECTION_6,
	ENGINE_SPEED,
	EXHAUST_VALVE_CLOSING,
	EXHAUST_VALVE_OPENING,
	GROSS_IMEP,
	INDICATED_POWER,
	INDICATED_POWER_HP,
	INDICATED_TORQUE,
	INDICATED_TORQUE_LBFT,
	INDICATED_TORQUE_NET,
	INJECTOR_DURATION_1,
	INJECTOR_DURATION_2,
	INJECTOR_DURATION_3,
	INJECTOR_DURATION_4,
	INJECTOR_DURATION_5,
	INJECTOR_DURATION_6,
	INJECTOR_SEPARATION_1,
	INJECTOR_SEPARATION_2,
	INJECTOR_SEPARATION_3,
	INJECTOR_SEPARATION_4,
	INJECTOR_SEPARATION_5,
	INLET_VALVE_CLOSING,				/* Inlet or Intake?  "Internal Combustion Engine Fundamentals", John B. Heywood, pp. 224 */
	INLET_VALVE_OPENING,
	KNOCK_BOSS_FACTOR,
	KNOCK_BOSS_INTEGRAL,
	KNOCK_FACTOR,
	KNOCK_HISTOGRAM_INDEX,
	KNOCK_INTEGRAL,
	KNOCK_C_AND_D,
	KNOCK_C_AND_D_CA,
	KNOCK_ONSET_CA,
	KNOCK_ONSET_UMFB,
	KNOCK_SIGNAL_ENERGY,
	LOAD,
	LOWER_PUMPING_IMEP,
	MAXIMUM_VALUE,
	MAX_BURN_RATE,
	MAX_BURN_RATE_CA,
	MAX_COIL_CURRENT,
	MAX_HEAT_RELEASE_RATE,
	MAX_HEAT_RELEASE_RATE_CA,
	MAX_MEAN_GAS_TEMP,
	MAX_MEAN_GAS_TEMP_CA,
	MAX_PRESSURE,
	MAX_PRESSURE_CA,
	MAX_D2P,
	MAX_D2P_CA,
	MAX_PRESSURE_IVO,
	MAX_PRESSURE_IVO_CA,
	MAX_PRESSURE_RISE_RATE,
	MAX_PRESSURE_RISE_RATE_CA,
	MAX_VALVE_ACCEL,
	MAX_VALVE_ACCEL_CA,
	MAX_VALVE_LIFT,
	MAX_VALVE_LIFT_CA,
	MAX_VALVE_VELOCITY,
	MAX_VALVE_VELOCITY_CA,
	MEAN_VALUE,
	MINIMUM_VALUE,
	MIN_PRESSURE,
	MIN_PRESSURE_CA,
	MIN_PRESSURE_EVC,
	MIN_PRESSURE_EVC_CA,
	MISFIRE_F,
	MISFIRE_M,
	MISFIRE_S,
	MISSING_TOOTH_1,
	MISSING_TOOTH_2,
	MISSING_TOOTH_RATIO_MAX,
	MISSING_TOOTH_RATIO_MIN,
	NET_IMEP,
	NORMALISED_DIFF_PR_50,
	NUMBER_OF_INJECTIONS,
	PRESSURE_OFFSET,
	PRESSURE_RATIO_MGMT_10,
	PRESSURE_RATIO_MGMT_25,
	PEAK_KNOCKING_PRESSURE,
	PEAK_KNOCKING_PRESSURE_CA,
	POLY_COMP,
	POLY_EXP,
	PUMPING_IMEP,
	RESTRIKE_DELAY_TIME,
	RESTRIKE_DWELL_TIME,
	RESTRIKE_MAX_CURRENT,
	RESTRIKE_TIMING,
	START_OF_COMBUSTION,
	START_OF_INJECTION_1,
	START_OF_INJECTION_2,
	START_OF_INJECTION_3,
	START_OF_INJECTION_4,
	START_OF_INJECTION_5,
	START_OF_INJECTION_6,
	TLA,
	TOOTH_GAP_RATIO_MAX,
	TOOTH_GAP_RATIO_MIN,
	TOTAL_HEAT_RELEASE,
	TRAPPED_MASS,
	UPPER_PUMPING_IMEP,
	WIEBE_A,
	WIEBE_M,

	/* Frequency based */

	FFT,
	FFT_FREQUENCY,
	FFT_REFERENCE,
	FFT_REFERENCE_FREQUENCY,
	FFT_TORQUE_SUM,
	FFT_TORQUE_SUM_FREQUENCY,
	FFT_KNOCK_PRESSURE,
	FFT_KNOCK_PRESSURE_FREQUENCY,
	FFT_TOOTH_PERIOD,
	FFT_TOOTH_PERIOD_FREQUENCY,

	/* Crank angle based */

	ANGULAR_TORQUE,
	ANGULAR_TORQUE_SUM,
	D2P,
	CHANNEL_DELTA,
	CRANK_ANGLE_IMEP,
	DIFFERENCE_PRESSURE,
	DIGITAL_SIGNAL,
	EXTRAPOLATED_CYL_PRESSURE,
	GAMMA,
	GROSS_HEAT_RELEASE_RATE,
	H_COEFF,
	KNOCKING_PRESSURE,
	MEAN_GAS_TEMP,
	MFB,
	MOTORED_PRESSURE,
	MOVING_PRESSURE_AVERAGE,
	NET_HEAT_RELEASE_RATE,
	POLYFIT,
	POLYTROPIC_INDICES,
	PRESSURE_RATIO,
	PRESSURE_RATIO_FRACTION,
	PRESSURE_RISE_RATE,
	TOOTH_SPEED,
	VALVE_ACCEL,
	VALVE_VELOCITY,
	WIEBE_MFB,

	/* Time based */

	SQUARE_CPS,
	CPS_CRANK_ANGLE,

	/* Knock Pressure Histogram */

	KNOCK_PRESSURE_HISTOGRAM,
	KNOCK_PRESSURE_HIST_WARNING,
	KNOCK_PRESSURE_HIST_ALARM,
	KNOCK_PRESSURE_HIST_SHUTDOWN,

	/* Motor Power Analysis */

	MOTOR_ALPHA,
	MOTOR_BETA,
	MOTOR_D,
	MOTOR_Q,
};

enum {
	CYCLE_MISFIRE = (1 << 0),
	CYCLE_SLOW_BURN = (1 << 1),
	CYCLE_KNOCK = (1 << 2),
	CYCLE_MEGA_KNOCK = (1 << 3),
	CYCLE_DEACTIVATED = (1 << 4),
	CYCLE_INVALID = (1 << 5),
};

enum {
	STAT_SUM = 1,
	STAT_MEAN = 2,
	STAT_MAX = 3,
	STAT_MIN = 4,
	STAT_STD_DEV = 5,
	STAT_COV = 6,
	STAT_LNV = 7,
	STAT_SKEWNESS = 8,
	STAT_KURTOSIS = 9,
	STAT_RANGE = 10,
	STAT_RMS = 11,
};

enum {
	STATS_CYLINDER = 0,
	STATS_CYCLE = 20,
	STATS_ENGINE_MEAN = 40,
	STATS_CYLINDER_MEAN = 60,
	STATS_CRANKANGLE = 80,
	STATS_CYCLE_SUM = 100,
};

enum {
	STATS_CYLINDER_BIT = (1 << 0),
	STATS_CYCLE_BIT = (1 << 1),
	STATS_ENGINE_MEAN_BIT = (1 << 2),
	STATS_CYLINDER_MEAN_BIT = (1 << 3),
	STATS_CRANKANGLE_BIT = (1 << 4),
};

#define STATS_ALL							STATS_CYLINDER_BIT | STATS_CYCLE_BIT | STATS_ENGINE_MEAN_BIT | STATS_CYLINDER_MEAN_BIT

enum {
	HT_NONE = 0,
	HT_ANNAND = 1,
	HT_WOSCHNI = 2,
	HT_HOHENBERG = 3,
	HT_EICHELBERG = 4,
	HT_NUSSELT = 5,
	HT_BRILING = 6,
	HT_WOSCHNI_GT = 7,
	HT_ASSANIS = 8,
};

enum {
	HR_WINDOW_IVC_EVO = 1,
	HR_WINDOW_SOC_EEOC = 2,
};

enum {
	MFB_CATOOL = 1,
	MFB_KISTLER_RASSWEILER = 2,
	MFB_KISTLER_HOHENBERG = 3,
};

enum {
	GAMMA_BRUNT = 1,
	GAMMA_KULZER = 2,
	GAMMA_MEASURED_MEAN = 3,
	GAMMA_MEASURED = 4,
	GAMMA_AIR = 5,
	GAMMA_SPECIFIED = 6,
	GAMMA_INDOLENE = 7,
	GAMMA_PROPANE = 8,
	GAMMA_HAYES = 9,
	GAMMA_CHANG = 10,
	GAMMA_AVL_THERM1_FIXED = 11,
	GAMMA_AVL_THERM1_CALC = 12,
	GAMMA_AVL_THERM1_A = 13,
};

enum {
	SMOOTH_MOVING_AVERAGE = 1,
	SMOOTH_SAVITZKY_GOLAY = 2,
};

enum {
	HR_FIRSTLAW = 0,
	HR_POLYFIRST = 1,
	HR_AVL_THERMO1 = 2,
};

enum {
	N_NOTHING = 0,
	N_SPECIFY = 1,
	N_AVGCHNL = 2,
	N_AVLRZT = 3,
};

enum {
	SOC_INVALID = 0,
	SOC_FIXED = 1,
	SOC_CA_CHANNEL_RISE = 2,
	SOC_CA_CHANNEL_FALL = 3,
	SOC_CA_CHANNEL_AVG = 4,
	SOC_CYC_CHANNEL = 5,
	SOC_POSITIVE_HR = 6,
	SOC_DIGITAL_RISING_EDGE = 7,
	SOC_DIGITAL_FALLING_EDGE = 8,
};

enum {
	FILE_NONE = 0,
	FILE_AVLIFILE = 1,
	FILE_CSV = 2,
	FILE_CAS = 3,
	FILE_MATLAB = 4,
	FILE_TDMS = 5,
	FILE_YOKOGAWA = 6,
	FILE_TEK_WFM = 7,
	FILE_PICO_PSD = 8,
	FILE_PICO_PLW = 9,
	FILE_NICOLET_WFT = 10,
	FILE_DAQ = 11,
	FILE_ATIMAT = 12,
	FILE_SDF = 13,
	FILE_TFX_CSV = 14,
	FILE_MATLAB_COMPRESSED = 15,
	FILE_KISTLER_OPEN_V1 = 16,
	FILE_KISTLER_OPEN_V2 = 17,
};

enum {
	IFILE_CA_RAW = (1 << 0),
	IFILE_CA = (1 << 1),
	IFILE_CA_TO_TIME = (1 << 2),
	IFILE_TIME = (1 << 3),
	IFILE_RTP_RESULTS = (1 << 4),
	IFILE_RESULTS = (1 << 5),
	IFILE_ASYNC_UTC = (1 << 6),
	IFILE_RESULTS_RAW = (1 << 7),
	IFILE_CA_MEAN = (1 << 8),
	IFILE_RESULTS_STATS = (1 << 9),
};

enum {
	ENGINE_UNKNOWN = 0,
	ENGINE_SI = 1,
	ENGINE_DISI = 2,
	ENGINE_CI_DI = 3,
	ENGINE_CI_IDI = 4,
	ENGINE_SI_ROTARY = 5,
};

enum {
	OFFSET_NONE = 0,
	OFFSET_FIXED = 1,
	OFFSET_POLYTROPIC_2PT = 2,
	OFFSET_WINDOW = 3,
	OFFSET_MEAN = 4,
	OFFSET_WINDOW_ABSOLUTE = 5,
	OFFSET_POLYTROPIC_3PT = 6,
	OFFSET_RESULT_CHANNEL = 7,
};

enum {
	DIGITIZE_NONE = 0,
	DIGITIZE_AUTO = 1,
	DIGITIZE_THRESHOLD = 2,
};

enum {
	CAMTYPE_NONE = 0,
	CAMTYPE_INLET_CID = 1,
	CAMTYPE_EXHAUST_CID = 2,
	CAMTYPE_4_PLUS_2 = 3,
	CAMTYPE_BOSCH_QUICK_SYNC = 4,
	CAMTYPE_4_MINUS_1 = 5,
	CAMTYPE_ECOBOOST_INLET = 6,
	CAMTYPE_ECOBOOST_EXHAUST = 7,
};

enum {
	EDGE_FALLING = 1,
	EDGE_RISING = 2,
	EDGE_BOTH = 3,
};

enum {
	IMEP_ACTUAL_PRESSURE = 1,
	IMEP_MEAN_PRESSURE = 2,
	IMEP_MEAN_PRESSURE_1DEG = 3,
};

enum {
	MOTORED_POLY_COMP = 1,
	MOTORED_POLY_FINISH = 2,
	MOTORED_BDC = 4,
};

enum {
	TLA_POLYNOMIAL = 1,
	TLA_POLYNOMIAL_SMOOTHED = 2,
	TLA_SLICE = 3,
};

enum {
	STATS_RATIO = 1,
	STATS_PERCENT = 2,
};

enum {
	PRR_DERIVATIVE = 1,
	PRR_SAVITZKY_GOLAY = 2,
};

enum {
	GAS_TEMP_REF_CA = 1,
	GAS_TEMP_IVC = 2,
	GAS_TEMP_AVL_THERMO1 = 3,
};

#define NUMBER_OF_CHANNEL_TYPES				36

enum {
	CHAN_UNKNOWN = 0,			/* Unknown */
	CHAN_ENG_SPEED = 1,			/* Engine Speed (RPM) */
	CHAN_CYL_PR = 2,			/* Cylinder Pressure (bar) */
	CHAN_INJ_PR = 3,			/* Injection Line Pressure (bar) */
	CHAN_NEEDLE_LIFT = 4,		/* Needle Lift (mm) */
	CHAN_VALVE_LIFT = 5,		/* Valve Lift (mm) */
	CHAN_INLET_PR = 6,			/* Inlet Manifold Pressure (bar) */
	CHAN_EXH_PR = 7,			/* Exhaust Manifold Pressure (bar) */
	CHAN_LOW_T = 8,				/* Low Tension Coil Voltage (V) */
	CHAN_HIGH_T = 9,			/* High Tension Pickup Voltage (mV) */
	CHAN_SPK_PRI_CURR = 10,		/* Spark Primary Current (A) */
	CHAN_SPK_SEC_CURR = 11,		/* Spark Secondary Current (A) */
	CHAN_BLOCK_ACC = 12,		/* Engine Block Acceleration (g) */
	CHAN_IGN_ANG = 13,			/* Ignition Angle Meter (deg BTDC) */
	CHAN_CPS = 14,				/* Crank position sensor (V) */
	CHAN_CID = 15,				/* Camshaft position sensor (V) */
	CHAN_INJECTOR = 16,			/* Injector (V) */
	CHAN_TIME = 17,				/* Time (seconds) */
	CHAN_FRP = 18,				/* Fuel rail pressure (kPa) */
	CHAN_RESULTS = 19,			/* Results channel */
	CHAN_ATI_SCALAR = 20,		/* ATI Scalar */
	CHAN_ATI_STATEVAR = 21,		/* ATI State variable */
	CHAN_CAN = 22,				/* CAN signal */
	CHAN_DIGITAL = 23,			/* Digital signal */
	CHAN_STD_RESULTS = 24,		/* Standard Results */
	CHAN_LS_DAQ = 25,			/* Low speed DAQ */
	CHAN_TEMPERATURE = 26,		/* Temperature */
	CHAN_PRESSURE = 27,			/* Pressure */
	CHAN_OPTRAND_DIAG = 28,		/* Optrand Diagnostic */
	CHAN_TOOTH_PERIODS = 29,	/* Encoder Tooth Periods */
	CHAN_ATI_CALCULATED = 30,	/* ATI Virtual Scalar */
	CHAN_GENERIC = 31,			/* Generic Crankangle Channel */
	CHAN_INJECTOR_CURR = 32,	/* Injector current (A) */
	CHAN_VOLUME = 33,			/* Cylinder Volume (cm^3) */
	CHAN_PHASE_VOLTAGE = 34,	/* Phase Voltage */
	CHAN_PHASE_CURRENT = 35,	/* Phase Current */
};

#define NUMBER_OF_ABSCISSA_TYPES			10

enum {
	ABSCISSA_UNKNOWN = 0,
	ABSCISSA_CRANKANGLE = 1,
	ABSCISSA_CYCLE = 2,
	ABSCISSA_TIME = 3,
	ABSCISSA_HERTZ = 4,
	ABSCISSA_TOOTH = 5,
	ABSCISSA_FREQUENCY = 6,
	ABSCISSA_ASYNC_UTC = 7,
	ABSCISSA_EVENT = 8,
	ABSCISSA_HISTOGRAM = 9,
};

enum {
	UNITS_NONE = 0,				/* none*/
	UNITS_DEG = 1,				/* degrees */
	UNITS_DEG_REL = 2,			/* relative degrees */
	UNITS_MS = 3,				/* milliseconds */
	UNITS_NM = 4,				/* Newton metres */
	UNITS_RPM = 5,				/* revs per minute */
	UNITS_J = 6,				/* Joules */
	UNITS_J_DEG = 7,			/* Joules/deg*/
	UNITS_BAR = 8,				/* bar */
	UNITS_BAR_DEG = 9,			/* bar/deg */
	UNITS_BARDEG = 10,			/* bar*deg */
	UNITS_K = 11,				/* Kelvin */
	UNITS_PERCENT = 12,			/* percent */
	UNITS_GRAD = 13,			/* Grad */
	UNITS_MM = 14,				/* millimetres */
	UNITS_MM2 = 15,				/* millimetres^2 */
	UNITS_MM3 = 16,				/* millimetres^3 */
	UNITS_KPA = 17,				/* kilo Pascals */
	UNITS_GDEG = 18,			/* g*deg */
	UNITS_HZ = 19,				/* Hertz */
	UNITS_KHZ = 20,				/* kilo Hertz */
	UNITS_VOLTS = 21,			/* Volts */
	UNITS_AMP = 22,				/* Amps */
	UNITS_LBFT = 23,			/* pounds feet */
	UNITS_HP = 24,				/* horsepower */
	UNITS_MG = 25,				/* milligram */
	UNITS_AS_CHAN = 26,			/* same as source channel */
	UNITS_EO = 27,				/* engine order */
	UNITS_US = 28,				/* microseconds */
	UNITS_KW = 29,				/* kiloWatts */
	UNITS_BAR_DEG2 = 30,		/* bar/deg^2 */
	UNITS_BAR2_SEC = 31,		/* bar^2.sec */
	UNITS_MM_PER_DEG = 32,		/* millimetres/deg */
	UNITS_MM_PER_DEG2 = 33,		/* millimetres/deg^2 */
};

enum {
	KNOCK_RECTIFIED_INTEGRAL = 0,
	KNOCK_SQUARED_INTEGRAL = 1,
};

enum {
	FILTER_NONE = 0,
	FILTER_MEAN_LOW_PASS = 1,
	FILTER_MEAN_HIGH_PASS = 2,
	FILTER_MEAN_LOW_PASS_SPIKE_ELIM = 3,
	FILTER_MEAN_HIGH_PASS_SPIKE_ELIM = 4,
	FILTER_FIR_LOW_PASS = 5,
	FILTER_FIR_HIGH_PASS = 6,
	FILTER_FIR_BAND_PASS = 7,
	FILTER_FIR_BAND_BARRIER = 8,
	FILTER_FIR_SPECIFIED = 9,
	FILTER_FIR_SPECIFIED_RATE = 10,
	FILTER_MEDIAN = 11,
};

enum {
	FFT_WINDOW_RECTANGULAR = 0,
	FFT_WINDOW_HANN = 1,
	FFT_WINDOW_HAMMING = 2,
	FFT_WINDOW_TUKEY = 3,
	FFT_WINDOW_COSINE = 4,
	FFT_WINDOW_LANCZOS = 5,
	FFT_WINDOW_TRIANGULAR = 6,
	FFT_WINDOW_BARTLETT = 7,
	FFT_WINDOW_BARTLETT_HANN = 8,
	FFT_WINDOW_BLACKMAN = 9,
	FFT_WINDOW_KAISER = 10,
	FFT_WINDOW_NUTTALL = 11,
	FFT_WINDOW_BLACKMAN_HARRIS = 12,
	FFT_WINDOW_BLACKMAN_NUTTALL = 13,
	FFT_WINDOW_FLATTOP = 14,
};

enum {
	EEOC_METHOD_BRUNT = 1,
	EEOC_METHOD_FIXED = 2,
};

enum {
	VOLUME_CATOOL = 1,
	VOLUME_LUCID = 2,
	VOLUME_XLEAST = 3,
	VOLUME_USER = 4,
	VOLUME_KISTLER_KB1 = 5,
	VOLUME_KISTLER_KB2 = 6,
	VOLUME_KISTLER_FKFS = 7,
};

enum {
	CSV_COLUMN_CHANNELS = 1,
	CSV_COLUMN_CYCLES = 2,
	CSV_SINGLE_CHANNEL = 3,
};

enum {
	VARIABLE_NONE = 1,
	VARIABLE_FLOAT = 2,
	VARIABLE_UINT32 = 3,
	VARIABLE_INT32 = 4,
	VARIABLE_DOUBLE = 5,
	VARIABLE_UINT8 = 6,
	VARIABLE_INT8 = 7,
	VARIABLE_UINT16 = 8,
	VARIABLE_INT16 = 9,
	VARIABLE_UINT64 = 10,
	VARIABLE_INT64 = 11,
};

#define NUMBER_OF_CALIBRATION_TERMS			10

#define CALC_PRESSURE						(1ULL << 0)
#define CALC_PRESSURE_CA					(1ULL << 21)
#define CALC_PRESSURE_AVERAGE				(1ULL << 42)
#define CALC_PRESSURE_IVO					(1ULL << 33)
#define CALC_PRESSURE_EVC					(1ULL << 34)
#define CALC_PRESSURE_D2P					(1ULL << 44)
#define CALC_PRESSURE_D2P_CA				(1ULL << 45)
#define CALC_TLA							(1ULL << 22)
#define CALC_TLA_CA							(1ULL << 41)
#define CALC_KNOCK							(1ULL << 1)
#define CALC_KNOCK_EXTENDED					(1ULL << 23)
#define CALC_KNOCK_BOSS						(1ULL << 43)
#define CALC_KNOCK_CA						(1ULL << 24)
#define CALC_KNOCK_HISTOGRAM				(1ULL << 50)
#define CALC_MOTORED						(1ULL << 2)
#define CALC_EEOC							(1ULL << 3)
#define CALC_POLYTROPIC						(1ULL << 4)
#define CALC_POLYTROPIC_CA					(1ULL << 25)
#define CALC_SOC							(1ULL << 5)
#define CALC_HEAT_RELEASE					(1ULL << 6)
#define CALC_HEAT_RELEASE_CA				(1ULL << 26)
#define CALC_BURN_RATES						(1ULL << 7)
#define CALC_BURN_RATES_CA					(1ULL << 27)
#define CALC_BURN_ANGLES					(1ULL << 8)
#define CALC_WORK							(1ULL << 9)
#define CALC_WORK_CA						(1ULL << 28)
#define CALC_TEMP							(1ULL << 10)
#define CALC_TEMP_CA						(1ULL << 29)
#define CALC_FOURIER						(1ULL << 11)
#define CALC_FOURIER_REFERENCE				(1ULL << 36)
#define CALC_FOURIER_TQ_SUM					(1ULL << 37)
#define CALC_FOURIER_KNK_PRESSURE			(1ULL << 38)
#define CALC_WIEBE							(1ULL << 12)
#define CALC_WIEBE_CA						(1ULL << 30)
#define CALC_PRESSURE_RATIO					(1ULL << 31)
#define CALC_PRESSURE_RATIO_CA				(1ULL << 35)
#define CALC_VALVE_TIMING					(1ULL << 15)
#define CALC_INJECTOR_TIMING				(1ULL << 14)
#define CALC_IGNITION						(1ULL << 16)
#define CALC_IND							(1ULL << 17)
#define CALC_AIR_CHARGE						(1ULL << 18)
#define CALC_EST_VALVE_TIMING				(1ULL << 39)
#define CALC_EST_VALVE_TIMING_CA			(1ULL << 40)
#define CALC_CAMSHAFT						(1ULL << 46)
#define CALC_GENERIC						(1ULL << 32)
#define CALC_CYCLE_SPEED					(1ULL << 19)
#define CALC_GDI_INJECTOR					(1ULL << 20)
#define CALC_CUST_EXTRAS					(1ULL << 13)
#define CALC_MISFIRE						(1ULL << 47)
#define CALC_VALVE_LIFT						(1ULL << 48)
#define CALC_VALVE_LIFT_CA					(1ULL << 49)
#define CALC_MOTOR_POWER					(1ULL << 51)		/* MSB */

#define KHI_MAX_KNKMAX						40		/* bar */
#define KHI_RESOLUTION						20		/* samples/bar */
#define KHI_RESOLUTION_F					20.0f	/* KHI_RESOLUTION float */

#define NUMBER_OF_TOOTH_PERIOD_FFT_CYCLES	10

enum AnalysisRqst : unsigned char {
	NotCalculated = 0,
	TemporaryCalculation = 1,
	CalculateOnDemand = 2,
	CalculateAndStore = 3,
	RealtimeStats = 4,
};

enum {
	THREAD_STATUS_UNKNOWN = 0,
	THREAD_STATUS_IDLE = 1,
	THREAD_STATUS_INITIALISING = 2,
	THREAD_STATUS_RUNNING = 3,
	THREAD_STATUS_FINISHING = 4,
	THREAD_STATUS_DONE = 5,
};

typedef struct {
	float* dCA;
	float* volume;
	float* dVolume;
	float* heat_transfer_area;
	float* logVolume;
	float* pressure_to_torque;
} Geometry;

typedef struct {
	unsigned int terms;
	float factor[NUMBER_OF_CALIBRATION_TERMS];
} Calibration;

typedef struct {
	unsigned int number;
	char description[SIZEOF_DESCRIPTION];
} AbscissaInf;

typedef struct {
	unsigned int number;
	char name[SIZEOF_CHANNEL_NAME];
	unsigned int abscissa_type;
	char description[SIZEOF_DESCRIPTION];
	float min;
	float max;
	char units[SIZEOF_UNITS];
	unsigned int status;
} ChannelInf;

typedef struct {
	unsigned int number;
	char unique_id[SIZEOF_CHANNEL_NAME];
	unsigned long long int analysis_calcs;
	unsigned int abscissa_type;
	unsigned int output_abscissa;
	unsigned int abscissa_units;
	char name[SIZEOF_CHANNEL_NAME];
	char units[SIZEOF_UNITS];
	char description[SIZEOF_DESCRIPTION];
	float min;
	float max;
	unsigned int channel_validity[NUMBER_OF_CHANNEL_TYPES];
	float warning_min;
	float warning_max;
	float warning_stddev;
	float error_min;
	float error_max;
	float error_stddev;
	void* AnalysisFunction;
	void* InitialiseFunction;
	void* CompleteFunction;
} AnalysisConf;

static const AbscissaInf abscissa_information[NUMBER_OF_ABSCISSA_TYPES] = {
	{ ABSCISSA_UNKNOWN,			"UNKNOWN" },
	{ ABSCISSA_CRANKANGLE,		"CRANKANGLE" },
	{ ABSCISSA_CYCLE,			"CYCLE" },
	{ ABSCISSA_TIME,			"TIME" },
	{ ABSCISSA_HERTZ,			"HERTZ" },
	{ ABSCISSA_TOOTH,			"TOOTH" },
	{ ABSCISSA_FREQUENCY,		"FREQUENCY" },
	{ ABSCISSA_ASYNC_UTC,		"ASYNC_UTC" },
	{ ABSCISSA_EVENT,			"EVENT" },
	{ ABSCISSA_HISTOGRAM,		"HISTOGRAM" },
};

static const ChannelInf channel_information[NUMBER_OF_CHANNEL_TYPES] = {
	{ CHAN_UNKNOWN,			"UNKNOWN",			ABSCISSA_CRANKANGLE,	"Unknown",							0.0f,		100.0f,		"",		0 },
	{ CHAN_ENG_SPEED,		"ENG_SPEED",		ABSCISSA_CRANKANGLE,	"Engine Speed",						0.0f,		8000.0f,	"RPM",	0 },
	{ CHAN_CYL_PR,			"CYLPR",			ABSCISSA_CRANKANGLE,	"Cylinder Pressure",				0.0f,		100.0f,		"bar",	3 },
	{ CHAN_INJ_PR,			"INJPR",			ABSCISSA_CRANKANGLE,	"Injection Line Pressure",			0.0f,		250.0f,		"bar",	3 },
	{ CHAN_NEEDLE_LIFT,		"NEEDLELIFT",		ABSCISSA_CRANKANGLE,	"Needle Lift",						0.0f,		5.0f,		"mm",	7 },
	{ CHAN_VALVE_LIFT,		"VALVELIFT",		ABSCISSA_CRANKANGLE,	"Valve Lift",						0.0f,		20.0f,		"mm",	3 },
	{ CHAN_INLET_PR,		"INLETPR",			ABSCISSA_CRANKANGLE,	"Inlet Manifold Pressure",			0.0f,		2.0f,		"bar",	3 },
	{ CHAN_EXH_PR,			"EXHPR",			ABSCISSA_CRANKANGLE,	"Exhaust Manifold Pressure",		0.0f,		2.0f,		"bar",	3 },
	{ CHAN_LOW_T,			"LOWT",				ABSCISSA_CRANKANGLE,	"Ignition - Low Tension Signal",	0.0f,		20.0f,		"V",	7 },
	{ CHAN_HIGH_T,			"HIGHT",			ABSCISSA_CRANKANGLE,	"Ignition - High Tension Pickup",	0.0f,		20.0f,		"V",	7 },
	{ CHAN_SPK_PRI_CURR,	"SPKPRICURR",		ABSCISSA_CRANKANGLE,	"Ignition - Primary Current",		0.0f,		20.0f,		"A",	3 },
	{ CHAN_SPK_SEC_CURR,	"SPKSECCURR",		ABSCISSA_CRANKANGLE,	"Ignition - Secondary Current",		0.0f,		20.0f,		"A",	3 },
	{ CHAN_BLOCK_ACC,		"BLOCKACC",			ABSCISSA_CRANKANGLE,	"Engine Block Acceleration",		0.0f,		20.0f,		"g",	3 },
	{ CHAN_IGN_ANG,			"IGNANG",			ABSCISSA_CRANKANGLE,	"Ignition - Angle Meter",			-50.0f,		50.0f,		"deg",	3 },
	{ CHAN_CPS,				"CPS",				ABSCISSA_CRANKANGLE,	"Crank Position Sensor",			-10.0f,		10.0f,		"V",	7 },
	{ CHAN_CID,				"CID",				ABSCISSA_CRANKANGLE,	"Camshaft Position Sensor",			-10.0f,		10.0f,		"V"	,	7 },
	{ CHAN_INJECTOR,		"INJECTOR",			ABSCISSA_CRANKANGLE,	"Injector Voltage",					0.0f,		20.0f,		"V",	7 },
	{ CHAN_TIME,			"TIME",				ABSCISSA_CRANKANGLE,	"Time",								0.0f,		2.0f,		"s",	0 },
	{ CHAN_FRP,				"FRP",				ABSCISSA_CRANKANGLE,	"Fuel Rail Pressure",				0.0f,		5000.0f,	"kPa",	3 },
	{ CHAN_RESULTS,			"RESULTS",			ABSCISSA_CYCLE,			"Results",							0.0f,		10.0f,		"",		0 },
	{ CHAN_ATI_SCALAR,		"SCALAR",			ABSCISSA_CYCLE,			"ATI VISION Scalar",				-10.0f,		10.0f,		"",		0 },
	{ CHAN_ATI_STATEVAR,	"STATEVAR",			ABSCISSA_CYCLE,			"ATI VISION State Variable",		-10.0f,		10.0f,		"",		0 },
	{ CHAN_CAN,				"CAN",				ABSCISSA_CYCLE,			"CAN Signal",						-10.0f,		10.0f,		"",		0 },
	{ CHAN_DIGITAL,			"DIGITAL",			ABSCISSA_CYCLE,			"Digital Signal",					-10.0f,		10.0f,		"",		7 },
	{ CHAN_STD_RESULTS,		"STD_RESULTS",		ABSCISSA_CYCLE,			"Standard Results",					0.0f,		10.0f,		"",		0 },
	{ CHAN_LS_DAQ,			"LS_DAQ",			ABSCISSA_CYCLE,			"Low Speed DAQ",					-10.0f,		10.0f,		"",		7 },
	{ CHAN_TEMPERATURE,		"TEMPERATURE",		ABSCISSA_CYCLE,			"Temperature",						-40.0f,		1000.0f,	"degC",	3 },
	{ CHAN_PRESSURE, 		"PRESSURE",			ABSCISSA_CYCLE,			"Pressure",							0.0f,		500.0f,		"bar",	3 },
	{ CHAN_OPTRAND_DIAG,	"OPTRAND_DIAG",		ABSCISSA_CYCLE,			"Optrand Diagnostic",				0.0f,		5.0f,		"V",	3 },
	{ CHAN_TOOTH_PERIODS,	"TOOTH_PERIODS",	ABSCISSA_CRANKANGLE,	"Encoder Tooth Periods",			0.0f,		10000.0f,	"us",	0 },
	{ CHAN_ATI_CALCULATED,	"VIRTUAL",			ABSCISSA_CYCLE,			"ATI VISION Virtual Scalar",		-10.0f,		10.0f,		"",		0 },
	{ CHAN_GENERIC,			"GENERIC",			ABSCISSA_CRANKANGLE,	"Generic Channel",					-10.0f,		10.0f,		"",		7 },
	{ CHAN_INJECTOR_CURR,	"INJECTOR_CURR",	ABSCISSA_CRANKANGLE,	"Injector Current",					0.0f,		20.0f,		"A",	3 },
	{ CHAN_VOLUME,			"VOLUME",			ABSCISSA_CRANKANGLE,	"Cylinder Volume",					0.0f,		10000.0f,	"cm^3", 0 },
	{ CHAN_PHASE_VOLTAGE,	"PHASE_VOLTAGE",	ABSCISSA_CRANKANGLE,	"Phase Voltage",					-1000.0f,	1000.0f,	"V",	3 },
	{ CHAN_PHASE_CURRENT,	"PHASE_CURRENT",	ABSCISSA_CRANKANGLE,	"Phase Current",					-1000.0f,	1000.0f,	"A",	3 },
};

static const AnalysisConf analysis_configuration[NUMBER_OF_ANALYSIS_CHANNELS] = {
	
	/* Cycle Based */

	{ AVERAGE_EXHAUST_ABSOLUTE_PRESSURE,	"AVERAGE_EXHAUST_ABSOLUTE_PRESSURE",	CALC_PRESSURE_EVC,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_KPA,					"AEAP",			"kPa",			"Average Exhaust Absolute Pressure",					-360.0f,	360.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	0.0f,		2.0f,		FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ BURN_ANGLE_1,							"BURN_ANGLE_1",							CALC_BURN_ANGLES,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"MFB01_",		"deg",			"Mass Fraction Burned 1%",								-360.0f,	360.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-20.0f,		18.0f,		FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ BURN_ANGLE_2,							"BURN_ANGLE_2",							CALC_BURN_ANGLES,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"MFB02_",		"deg",			"Mass Fraction Burned 2%",								-360.0f,	360.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-13.0f,		21.0f,		FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ BURN_ANGLE_5,							"BURN_ANGLE_5",							CALC_BURN_ANGLES,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"MFB05_",		"deg",			"Mass Fraction Burned 5%",								-360.0f,	360.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-11.0f,		109.0f,		FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ BURN_ANGLE_10,						"BURN_ANGLE_10",						CALC_BURN_ANGLES,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"MFB10_",		"deg",			"Mass Fraction Burned 10%",								-360.0f,	360.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-9.0f,		110.0f,		FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ BURN_ANGLE_20,						"BURN_ANGLE_20",						CALC_BURN_ANGLES,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"MFB20_",		"deg",			"Mass Fraction Burned 20%",								-360.0f,	360.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-20.0f,		114.0f,		FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ BURN_ANGLE_25,						"BURN_ANGLE_25",						CALC_BURN_ANGLES,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"MFB25_",		"deg",			"Mass Fraction Burned 25%",								-360.0f,	360.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-5.0f,		117.0f,		FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ BURN_ANGLE_50,						"BURN_ANGLE_50",						CALC_BURN_ANGLES,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"MFB50_",		"deg",			"Mass Fraction Burned 50%",								-360.0f,	360.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	0.0f,		121.0f,		FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ BURN_ANGLE_75,						"BURN_ANGLE_75",						CALC_BURN_ANGLES,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"MFB75_",		"deg",			"Mass Fraction Burned 75%",								-360.0f,	360.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	0.0f,		132.0f,		FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ BURN_ANGLE_80,						"BURN_ANGLE_80",						CALC_BURN_ANGLES,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"MFB80_",		"deg",			"Mass Fraction Burned 80%",								-360.0f,	360.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	0.0f,		135.0f,		FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ BURN_ANGLE_90,						"BURN_ANGLE_90",						CALC_BURN_ANGLES,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"MFB90_",		"deg",			"Mass Fraction Burned 90%",								-360.0f,	360.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	0.0f,		138.0f,		FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ BURN_ANGLE_95,						"BURN_ANGLE_95",						CALC_BURN_ANGLES,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"MFB95_",		"deg",			"Mass Fraction Burned 95%",								-360.0f,	360.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	1.0f,		140.0f,		FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ BURN_ANGLE_98,						"BURN_ANGLE_98",						CALC_BURN_ANGLES,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"MFB98_",		"deg",			"Mass Fraction Burned 98%",								-360.0f,	360.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	1.0f,		140.0f,		FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ BURN_ANGLE_99,						"BURN_ANGLE_99",						CALC_BURN_ANGLES,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"MFB99_",		"deg",			"Mass Fraction Burned 99%",								-360.0f,	360.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	2.0f,		140.0f,		FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ BURN_DURATION_0_2,					"BURN_DURATION_0_2",					CALC_BURN_ANGLES,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG_REL,				"B0002_",		"deg",			"Burn Duration SOC to 2% MFB",							0.0f,		100.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ BURN_DURATION_0_5,					"BURN_DURATION_0_5",					CALC_BURN_ANGLES,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG_REL,				"B0005_",		"deg",			"Burn Duration SOC to 5% MFB",							0.0f,		100.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ BURN_DURATION_0_10,					"BURN_DURATION_0_10",					CALC_BURN_ANGLES,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG_REL,				"B0010_",		"deg",			"Burn Duration SOC to 10% MFB",							0.0f,		100.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ BURN_DURATION_0_90,					"BURN_DURATION_0_90",					CALC_BURN_ANGLES,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG_REL,				"B0090_",		"deg",			"Burn Duration SOC to 90% MFB",							0.0f,		100.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ BURN_DURATION_2_90,					"BURN_DURATION_2_90",					CALC_BURN_ANGLES,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG_REL,				"B0290_",		"deg",			"Burn Duration 2% to 90% MFB",							0.0f,		100.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ BURN_DURATION_5_90,					"BURN_DURATION_5_90",					CALC_BURN_ANGLES,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG_REL,				"B0590_",		"deg",			"Burn Duration 5% to 90% MFB",							0.0f,		100.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ BURN_DURATION_10_90,					"BURN_DURATION_10_90",					CALC_BURN_ANGLES,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG_REL,				"B1090_",		"deg",			"Burn Duration 10% to 90% MFB",							0.0f,		100.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ CENTRE_OF_GRAVITY,					"CENTRE_OF_GRAVITY",					CALC_BURN_ANGLES,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"COG",			"deg",			"MFB Centre of Gravity",								0.0f,		100.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ CAM_ADVANCE,							"CAM_ADVANCE",							CALC_CAMSHAFT,					ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG_REL,				"CAMADV",		"deg",			"Camshaft Advance",										-60.0f,		60.0f,		{ 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ CAM_ANGLE,							"CAM_ANGLE",							CALC_CAMSHAFT,					ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"CAMANG",		"deg",			"Camshaft Angle",										-60.0f,		60.0f,		{ 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ CAM_EDGE_1,							"CAM_EDGE_1",							CALC_CAMSHAFT,					ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"CAM_EDGE_R1",	"deg",			"Camshaft 1st Rising Edge",								-360.0f,	360.0f,		{ 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ CAM_EDGE_2,							"CAM_EDGE_2",							CALC_CAMSHAFT,					ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"CAM_EDGE_R2",	"deg",			"Camshaft 2nd Rising Edge",								-360.0f,	360.0f,		{ 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ CAM_EDGE_3,							"CAM_EDGE_3",							CALC_CAMSHAFT,					ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"CAM_EDGE_R3",	"deg",			"Camshaft 3rd Rising Edge",								-360.0f,	360.0f,		{ 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ CAM_EDGE_4,							"CAM_EDGE_4",							CALC_CAMSHAFT,					ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"CAM_EDGE_R4",	"deg",			"Camshaft 4th Rising Edge",								-360.0f,	360.0f,		{ 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ CAM_EDGE_5,							"CAM_EDGE_5",							CALC_CAMSHAFT,					ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"CAM_EDGE_F1",	"deg",			"Camshaft 1st Falling Edge",							-360.0f,	360.0f,		{ 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ CAM_EDGE_6,							"CAM_EDGE_6",							CALC_CAMSHAFT,					ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"CAM_EDGE_F2",	"deg",			"Camshaft 2nd Falling Edge",							-360.0f,	360.0f,		{ 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ CAM_EDGE_7,							"CAM_EDGE_7",							CALC_CAMSHAFT,					ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"CAM_EDGE_F3",	"deg",			"Camshaft 3rd Falling Edge",							-360.0f,	360.0f,		{ 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ CAM_EDGE_8,							"CAM_EDGE_8",							CALC_CAMSHAFT,					ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"CAM_EDGE_F4",	"deg",			"Camshaft 4th Falling Edge",							-360.0f,	360.0f,		{ 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ CHANNEL_DELTA_MAX,					"CHANNEL_DELTA_MAX",					0,								ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_NONE,					"MAX_DIFF",		"-",			"Maximum Difference versus Reference Channel",			-100.0f,	100.0f,		{ 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 1 , 1 , 1 , 1 , 1 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ CHANNEL_DELTA_MAX_CA,					"CHANNEL_DELTA_MAX_CA",					0,								ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_NONE,					"MAX_DIFF_CA",	"deg",			"Maximum Differemce versus Reference Channel CA",		-360.0f,	360.0f,		{ 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 1 , 1 , 1 , 1 , 1 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ CID_ANGLE,							"CID_ANGLE",							0,								ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"CID",			"deg",			"Camshaft ID Position",									-360.0f,	360.0f,		{ 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ DELTA_P,								"DELTA_P",								CALC_AIR_CHARGE,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_BAR,					"DELTA_P",		"bar",			"Delta-P during compression",							0.0f,		100.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ DIFFERENCE_PRESSURE_10,				"DIFFERENCE_PRESSURE_10",				CALC_PRESSURE_RATIO,			ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"DP10_",		"deg",			"Difference Pressure (10 bar) Crank Angle",				0.0f,		100.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ DIFFERENCE_PRESSURE_INTEGRAL,			"DIFFERENCE_PRESSURE_INTEGRAL",			CALC_PRESSURE_RATIO,			ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_BARDEG,				"DPI",			"bar*deg",		"Difference Pressure Integral",							0.0f,		5000.0f,	{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ DILPAR,								"DILPAR",								CALC_PRESSURE_RATIO,			ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_NONE,					"DILPAR",		"-",			"Dilution Parameter",									0.0f,		5.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ DWELL_TIME,							"DWELL_TIME",							CALC_IGNITION,					ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_MS,					"DWELL",		"ms",			"Coil Dwell Time",										0.0f,		10.0f,		{ 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 1 , 1 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ EEOC,									"EEOC",									CALC_EEOC,						ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"EEOC",			"deg",			"Estimated End Of Combustion",							-360.0f,	360.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	20.0f,		140.0f,		FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ EEVO,									"EEVO",									CALC_EST_VALVE_TIMING,			ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"EEVO",			"deg",			"Estimated Exhaust Valve Opening",						-360.0f,	360.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ EIVC,									"EIVC",									CALC_EST_VALVE_TIMING,			ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"EIVC",			"deg",			"Estimated Inlet Valve Closing",						-360.0f,	360.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ END_OF_INJECTION_1,					"END_OF_INJECTION_1",					CALC_INJECTOR_TIMING,			ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"EOI1_",		"deg",			"1st End of Injection",									-360.0f,	360.0f,		{ 0 , 0 , 0 , 1 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ END_OF_INJECTION_2,					"END_OF_INJECTION_2",					CALC_INJECTOR_TIMING,			ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"EOI2_",		"deg",			"2nd End of Injection",									-360.0f,	360.0f,		{ 0 , 0 , 0 , 1 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ END_OF_INJECTION_3,					"END_OF_INJECTION_3",					CALC_INJECTOR_TIMING,			ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"EOI3_",		"deg",			"3rd End of Injection",									-360.0f,	360.0f,		{ 0 , 0 , 0 , 1 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ END_OF_INJECTION_4,					"END_OF_INJECTION_4",					CALC_INJECTOR_TIMING,			ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"EOI4_",		"deg",			"4th End of Injection",									-360.0f,	360.0f,		{ 0 , 0 , 0 , 1 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ END_OF_INJECTION_5,					"END_OF_INJECTION_5",					CALC_INJECTOR_TIMING,			ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"EOI5_",		"deg",			"5th End of Injection",									-360.0f,	360.0f,		{ 0 , 0 , 0 , 1 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ END_OF_INJECTION_6,					"END_OF_INJECTION_6",					CALC_INJECTOR_TIMING,			ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"EOI6_",		"deg",			"6th End of Injection",									-360.0f,	360.0f,		{ 0 , 0 , 0 , 1 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ ENGINE_SPEED,							"ENGINE_SPEED",							CALC_CYCLE_SPEED,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_RPM,					"N",			"RPM",			"Engine Speed",											0.0f,		7000.0f,	{ 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 1 , 0 , 0 , 0 },	600.0f,		7000.0f,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ EXHAUST_VALVE_CLOSING,				"EXHAUST_VALVE_CLOSING",				0,								ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"EVC",			"deg",			"Exhaust Valve Closing",								-360.0f,	360.0f,		{ 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ EXHAUST_VALVE_OPENING,				"EXHAUST_VALVE_OPENING",				0,								ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"EVO",			"deg",			"Exhaust Valve Opening",								-360.0f,	360.0f,		{ 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ GROSS_IMEP,							"GROSS_IMEP",							CALC_WORK,						ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_BAR,					"GMEP",			"bar",			"Gross IMEP",											0.0f,		25.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	1.3f,		30.0f,		FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ INDICATED_POWER,						"INDICATED_POWER",						CALC_IND,						ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_KW,					"INDPWR",		"ikW",			"Indicated Power",										0.0f,		200.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ INDICATED_POWER_HP,					"INDICATED_POWER_HP",					CALC_IND,						ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_HP,					"INDHP",		"ihp",			"Indicated Horsepower",									0.0f,		200.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ INDICATED_TORQUE,						"INDICATED_TORQUE",						CALC_WORK,						ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_NM,					"TQIND",		"Nm",			"Indicated Torque (Gross)",								0.0f,		200.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ INDICATED_TORQUE_LBFT,				"INDICATED_TORQUE_LBFT",				CALC_IND,						ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_LBFT,					"TQLBFT",		"lb.ft",		"Indicated Torque",										0.0f,		200.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ INDICATED_TORQUE_NET,					"INDICATED_TORQUE_NET",					CALC_WORK,						ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_NM,					"TQINDN",		"Nm",			"Indicated Torque (Net)",								0.0f,		200.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ INJECTOR_DURATION_1,					"INJECTOR_DURATION_1",					CALC_INJECTOR_TIMING,			ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_MS,					"INJDUR1_",		"ms",			"1st Injection Duration",								0.0f,		1.0f,		{ 0 , 0 , 0 , 1 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ INJECTOR_DURATION_2,					"INJECTOR_DURATION_2",					CALC_INJECTOR_TIMING,			ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_MS,					"INJDUR2_",		"ms",			"2nd Injector Duration",								0.0f,		1.0f,		{ 0 , 0 , 0 , 1 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ INJECTOR_DURATION_3,					"INJECTOR_DURATION_3",					CALC_INJECTOR_TIMING,			ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_MS,					"INJDUR3_",		"ms",			"3rd Injector Duration",								0.0f,		1.0f,		{ 0 , 0 , 0 , 1 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ INJECTOR_DURATION_4,					"INJECTOR_DURATION_4",					CALC_INJECTOR_TIMING,			ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_MS,					"INJDUR4_",		"ms",			"4th Injector Duration",								0.0f,		1.0f,		{ 0 , 0 , 0 , 1 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ INJECTOR_DURATION_5,					"INJECTOR_DURATION_5",					CALC_INJECTOR_TIMING,			ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_MS,					"INJDUR5_",		"ms",			"5th Injector Duration",								0.0f,		1.0f,		{ 0 , 0 , 0 , 1 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ INJECTOR_DURATION_6,					"INJECTOR_DURATION_6",					CALC_INJECTOR_TIMING,			ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_MS,					"INJDUR6_",		"ms",			"6th Injector Duration",								0.0f,		1.0f,		{ 0 , 0 , 0 , 1 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ INJECTOR_SEPARATION_1,				"INJECTOR_SEPARATION_1",				CALC_INJECTOR_TIMING,			ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_MS,					"INJSEP1_",		"ms",			"1st Injector Separation",								0.0f,		1.0f,		{ 0 , 0 , 0 , 1 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ INJECTOR_SEPARATION_2,				"INJECTOR_SEPARATION_2",				CALC_INJECTOR_TIMING,			ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_MS,					"INJSEP2_",		"ms",			"2nd Injector Separation",								0.0f,		1.0f,		{ 0 , 0 , 0 , 1 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ INJECTOR_SEPARATION_3,				"INJECTOR_SEPARATION_3",				CALC_INJECTOR_TIMING,			ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_MS,					"INJSEP3_",		"ms",			"3rd Injector Separation",								0.0f,		1.0f,		{ 0 , 0 , 0 , 1 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ INJECTOR_SEPARATION_4,				"INJECTOR_SEPARATION_4",				CALC_INJECTOR_TIMING,			ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_MS,					"INJSEP4_",		"ms",			"4th Injector Separation",								0.0f,		1.0f,		{ 0 , 0 , 0 , 1 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ INJECTOR_SEPARATION_5,				"INJECTOR_SEPARATION_5",				CALC_INJECTOR_TIMING,			ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_MS,					"INJSEP5_",		"ms",			"5th Injector Separation",								0.0f,		1.0f,		{ 0 , 0 , 0 , 1 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ INLET_VALVE_CLOSING,					"INLET_VALVE_CLOSING",					0,								ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"IVC",			"deg",			"Inlet Valve Closing",									-360.0f,	360.0f,		{ 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ INLET_VALVE_OPENING,					"INLET_VALVE_OPENING",					0,								ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"IVO",			"deg",			"Inlet Valve Opening",									-360.0f,	360.0f,		{ 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ KNOCK_BOSS_FACTOR,					"KNOCK_BOSS_FACTOR",					CALC_KNOCK_BOSS,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_NONE,					"KBF",			"-",			"Knock Boss Factor",									0.0f,		10.0f,		{ 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ KNOCK_BOSS_INTEGRAL,					"KNOCK_BOSS_INTEGRAL",					CALC_KNOCK_BOSS,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_GDEG,					"KNKBINT",		"g.deg",		"Knock Boss Integral",									0.0f,		10.0f,		{ 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ KNOCK_FACTOR,							"KNOCK_FACTOR",							CALC_KNOCK_EXTENDED,			ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_NONE,					"KF",			"-",			"Knock Factor",											0.0f,		10.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	0.0f,		100.0f,		FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ KNOCK_HISTOGRAM_INDEX,				"KNOCK_HISTOGRAM_INDEX",				CALC_KNOCK_HISTOGRAM,			ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_NONE,					"KP_KHI",		"-",			"Knock Histogram Index",								0.0f,		4.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	0.0f,		100.0f,		FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL},
	{ KNOCK_INTEGRAL,						"KNOCK_INTEGRAL",						CALC_KNOCK,						ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_BARDEG,				"KNKINT",		"bar*deg",		"Knock Integral",										0.0f,		10.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	0.0f,		100.0f,		FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ KNOCK_C_AND_D,						"KNOCK_C_AND_D",						CALC_KNOCK_EXTENDED,			ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_NONE,					"CDKNK",		"bar/deg^3",	"Checkel & Dale Knock Intensity",						0.0f,		10.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ KNOCK_C_AND_D_CA,						"KNOCK_C_AND_D_CA",						CALC_KNOCK_EXTENDED,			ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"ACDKNK",		"deg",			"Maximum C&D KI Crank Angle",							-360.0f,	360.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ KNOCK_ONSET_CA,						"KNOCK_ONSET_CA",						CALC_KNOCK,						ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"AKO",			"deg",			"Knock Onset Angle",									-360.0f,	360.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ KNOCK_ONSET_UMFB,						"KNOCK_ONSET_UMFB",						CALC_BURN_ANGLES,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_PERCENT,				"KOUMFB",		"%",			"Unburnt MFB at Knock Onset",							0.0f,		100.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ KNOCK_SIGNAL_ENERGY,					"KNOCK_SIGNAL_ENERGY",					CALC_KNOCK,						ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_BAR2_SEC,				"KISEPO",		"bar^2.sec",	"Knock Intensity Signal Energy of Pressure Oscillation",0.0f,		10.0E-4f,	{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	0.0f,		8.2f,		FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ LOAD,									"LOAD",									CALC_AIR_CHARGE,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_NONE,					"LOAD",			"-",			"Normalised Load",										0.0f,		2.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ LOWER_PUMPING_IMEP,					"LOWER_PUMPING_IMEP",					CALC_WORK,						ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_BAR,					"LPMEP",		"bar",			"Lower Pumping IMEP",									0.0f,		25.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-1.0f,		0.9f,		FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ MAXIMUM_VALUE,						"MAXIMUM_VALUE",						CALC_GENERIC,					ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_AS_CHAN,				"MAX",			"",				"Maximum",												0.0f,		100.0f,		{ 1 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ MAX_BURN_RATE,						"MAX_BURN_RATE",						CALC_BURN_RATES,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_J_DEG,				"MXBRN",		"J/deg",		"Maximum Burn Rate",									0.0f,		10000.0f,	{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	540.0f,		54000.0f,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ MAX_BURN_RATE_CA,						"MAX_BURN_RATE_CA",						CALC_BURN_RATES,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"AMXBRN",		"deg",			"Maximum Burn Rate Crank Angle",						-360.0f,	360.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-20.0f,		116.0f,		FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ MAX_COIL_CURRENT,						"MAX_COIL_CURRENT",						CALC_IGNITION,					ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_AMP,					"MXCUR",		"A",			"Maximum Coil Current",									0.0f,		20.0f,		{ 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ MAX_HEAT_RELEASE_RATE,				"MAX_HEAT_RELEASE_RATE",				CALC_HEAT_RELEASE,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_J,					"QRTMX",		"J/deg",		"Maximum Heat Release Rate",							0.0f,		500.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	2.5f,		655.0f,		FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ MAX_HEAT_RELEASE_RATE_CA,				"MAX_HEAT_RELEASE_RATE_CA",				CALC_HEAT_RELEASE,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"AQRTMX",		"deg",			"Maximum Heat Release Rate Crank Angle",				-360.0f,	360.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-5.0f,		135.0f,		FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ MAX_MEAN_GAS_TEMP,					"MAX_MEAN_GAS_TEMP",					CALC_TEMP,						ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_K,					"MXT",			"K",			"Maximum Mean Gas Temperature",							0.0f,		2500.0f,	{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	1000.0f,	3000.0f,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ MAX_MEAN_GAS_TEMP_CA,					"MAX_MEAN_GAS_TEMP_CA",					CALC_TEMP,						ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"ATMAX",		"deg",			"Maximum Mean Gas Temperature Crank Angle",				-360.0f,	360.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-140.0f,	133.0f,		FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ MAX_PRESSURE,							"MAX_PRESSURE",							CALC_PRESSURE,					ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_BAR,					"PMAX",			"bar",			"Maximum Pressure",										0.0f,		200.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 1 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	5.5f,		180.0f,		FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ MAX_PRESSURE_CA,						"MAX_PRESSURE_CA",						CALC_PRESSURE,					ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"APMAX",		"deg",			"Maximum Pressure Crank Angle",							-360.0f,	360.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 1 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-5.0f,		40.0f,		FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ MAX_D2P,								"MAX_D2P",								CALC_PRESSURE_D2P,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_BAR_DEG2,				"D2PMAX",		"bar/deg^2",	"Maximum Second Derivative of Pressure",				-10.0f,		10.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ MAX_D2P_CA,							"MAX_D2P_CA",							CALC_PRESSURE_D2P,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"AD2PMAX",		"deg",			"Maximum Second Derivative of Pressure Crank Angle",	-360.0f,	360.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ MAX_PRESSURE_IVO,						"MAX_PRESSURE_IVO",						CALC_PRESSURE_IVO,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_BAR,					"PIVCMAX",		"bar",			"Maximum Pressure Around IVO",							0.0f,		20.0f,		{ 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ MAX_PRESSURE_IVO_CA,					"MAX_PRESSURE_IVO_CA",					CALC_PRESSURE_IVO,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"APIVCMAX",		"deg",			"Maximum Pressure Around IVO Crank Angle",				-360.0f,	360.0f,		{ 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ MAX_PRESSURE_RISE_RATE,				"MAX_PRESSURE_RISE_RATE",				CALC_PRESSURE,					ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_BAR_DEG,				"RPMAX",		"bar/deg",		"Maximum Pressure Rise Rate",							0.0f,		50.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 1 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	0.1f,		52.0f,		FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ MAX_PRESSURE_RISE_RATE_CA,			"MAX_PRESSURE_RISE_RATE_CA",			CALC_PRESSURE,					ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"ARPMX",		"deg",			"Maximum Pressure Rise Rate Crank Angle",				-360.0f,	360.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 1 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ MAX_VALVE_ACCEL,						"MAX_VALVE_ACCEL",						CALC_VALVE_LIFT,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_MM_PER_DEG2,			"V_AMX",		"mm/deg^2",		"Maximum Valve Lift Acceleration",						0.0f,		100.0f,		{ 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ MAX_VALVE_ACCEL_CA,					"MAX_VALVE_ACCEL_CA",					CALC_VALVE_LIFT,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"AV_AMX",		"deg",			"Maximum Valve Lift Acceleration Crank Angle",			-360.0f,	360.0f,		{ 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ MAX_VALVE_LIFT,						"MAX_VALVE_LIFT",						CALC_VALVE_LIFT,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_MM,					"V_LMX",		"mm",			"Maximum Valve Lift",									0.0f,		100.0f,		{ 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ MAX_VALVE_LIFT_CA,					"MAX_VALVE_LIFT_CA",					CALC_VALVE_LIFT,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"AV_LMX",		"deg",			"Maximum Valve Lift Crank Angle",						-360.0f,	360.0f,		{ 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ MAX_VALVE_VELOCITY,					"MAX_VALVE_VELOCITY",					CALC_VALVE_LIFT,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_MM_PER_DEG,			"V_VMX",		"mm/deg",		"Maximum Valve Lift Velocity",							0.0f,		100.0f,		{ 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ MAX_VALVE_VELOCITY_CA,				"MAX_VALVE_VELOCITY_CA",				CALC_VALVE_LIFT,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"AV_VMX",		"deg",			"Maximum Valve Lift Velocity Crank Angle",				-360.0f,	360.0f,		{ 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ MEAN_VALUE,							"MEAN_VALUE",							CALC_GENERIC,					ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_AS_CHAN,				"MEAN",			"",				"Mean",													0.0f,		100.0f,		{ 1 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ MINIMUM_VALUE,						"MINIMUM_VALUE",						CALC_GENERIC,					ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_AS_CHAN,				"MIN",			"",				"Minimum",												0.0f,		100.0f,		{ 1 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ MIN_PRESSURE,							"MIN_PRESSURE",							CALC_PRESSURE,					ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_BAR,					"PMIN",			"bar",			"Minimum Pressure",										0.0f,		2.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 1 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	0.0f,		1.7f,		FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ MIN_PRESSURE_CA,						"MIN_PRESSURE_CA",						CALC_PRESSURE,					ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"APMIN",		"deg",			"Minimum Pressure Crank Angle",							-360.0f,	360.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 1 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ MIN_PRESSURE_EVC,						"MIN_PRESSURE_EVC",						CALC_PRESSURE_EVC,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_BAR,					"PEVCMIN",		"bar",			"Minimum Pressure Around EVC",							0.0f,		2.0f,		{ 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	0.0f,		1.7f,		FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ MIN_PRESSURE_EVC_CA,					"MIN_PRESSURE_EVC_CA",					CALC_PRESSURE_EVC,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"APEVCMIN",		"deg",			"Minimum Pressure Around EVC Crank Angle",				-360.0f,	360.0f,		{ 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ MISFIRE_F,							"MISFIRE_F",							CALC_MISFIRE,					ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_PERCENT,				"MISFIRE_F",	"%",			"Percentage Misfire (Fast)",							0.0f,		100.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	1.3f,		30.0f,		FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ MISFIRE_M,							"MISFIRE_M",							CALC_MISFIRE,					ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_PERCENT,				"MISFIRE_M",	"%",			"Percentage Misfire (Medium)",							0.0f,		100.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	1.3f,		30.0f,		FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ MISFIRE_S,							"MISFIRE_S",							CALC_MISFIRE,					ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_PERCENT,				"MISFIRE_S",	"%",			"Percentage Misfire (Slow)",							0.0f,		100.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	1.3f,		30.0f,		FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ MISSING_TOOTH_1,						"MISSING_TOOTH_1",						0,								ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"MT1POS",		"deg",			"Missing Tooth 1 Position",								-360.0f,	-360.0f,	{ 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ MISSING_TOOTH_2,						"MISSING_TOOTH_2",						0,								ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"MT2POS",		"deg",			"Missing Tooth 2 Position",								-360.0f,	360.0f,		{ 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ MISSING_TOOTH_RATIO_MAX,				"MISSING_TOOTH_RATIO_MAX",				0,								ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_NONE,					"MTRMX",		"-",			"Maximum Missing Tooth Gap Ratio",						0.0f,		5.0f,		{ 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ MISSING_TOOTH_RATIO_MIN,				"MISSING_TOOTH_RATIO_MIN",				0,								ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_NONE,					"MTRMN",		"-",			"Minimum Missing Tooth Gap Ratio",						0.0f,		5.0f,		{ 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ NET_IMEP,								"NET_IMEP",								CALC_WORK,						ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_BAR,					"NMEP",			"bar",			"Net IMEP",												0.0f,		25.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ NORMALISED_DIFF_PR_50,				"NORMALISED_DIFF_PR_50",				CALC_PRESSURE_RATIO,			ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"NDP50_",		"deg",			"Normalised Difference Pressure 50% Crank Angle",		0.0f,		100.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ NUMBER_OF_INJECTIONS,					"NUMBER_OF_INJECTIONS",					CALC_INJECTOR_TIMING,			ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_NONE,					"NOI",			"-",			"Number of Injections",									0.0f,		6.0f,		{ 0 , 0 , 0 , 1 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ PRESSURE_OFFSET,						"PRESSURE_OFFSET",						0,								ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_BAR,					"POFF",			"bar",			"Absolute Pressure Offset",								-50.0f,		50.0f,		{ 1 , 0 , 1 , 0 , 0 , 0 , 1 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-800.0f,		800.0f,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ PRESSURE_RATIO_MGMT_10,				"PRESSURE_RATIO_MGMT_10",				CALC_PRESSURE_RATIO,			ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_NONE,					"PRM10_",		"-",			"Pressure Ratio Management 10",							0.0f,		5.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ PRESSURE_RATIO_MGMT_25,				"PRESSURE_RATIO_MGMT_25",				CALC_PRESSURE_RATIO,			ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_NONE,					"PRM25_",		"-",			"Pressure Ratio Management 25",							0.0f,		5.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ PEAK_KNOCKING_PRESSURE,				"PEAK_KNOCKING_PRESSURE",				CALC_KNOCK,						ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_BAR,					"KNKMAX",		"bar",			"Peak Knocking Pressure",								0.0f,		8.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	0.0f,		8.2f,		FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ PEAK_KNOCKING_PRESSURE_CA,			"PEAK_KNOCKING_PRESSURE_CA",			CALC_KNOCK,						ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"AKNKMX",		"deg",			"Peak Knocking Pressure Crank Angle",					-360.0f,	360.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ POLY_COMP,							"POLY_COMP",							CALC_POLYTROPIC,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_NONE,					"NCOMP",		"-",			"Polytropic Coefficient of Compression",				1.0f,		2.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	1.0f,		1.5f,		FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ POLY_EXP,								"POLY_EXP",								CALC_POLYTROPIC,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_NONE,					"NEXP",			"-",			"Polytropic Coefficient of Expansion",					1.0f,		2.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	1.0f,		1.5f,		FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ PUMPING_IMEP,							"PUMPING_IMEP",							CALC_WORK,						ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_BAR,					"PMEP",			"bar",			"Pumping IMEP",											0.0f,		20.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ RESTRIKE_DELAY_TIME,					"RESTRIKE_DELAY_TIME",					CALC_IGNITION,					ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_MS,					"RS_DLY",		"ms",			"Restrike Delay Time",									0.0f,		10.0f,		{ 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 1 , 1 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ RESTRIKE_DWELL_TIME,					"RESTRIKE_DWELL_TIME",					CALC_IGNITION,					ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_MS,					"RS_DWELL",		"ms",			"Restrike Dwell Time",									0.0f,		10.0f,		{ 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 1 , 1 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ RESTRIKE_MAX_CURRENT,					"RESTRIKE_MAX_CURRENT",					CALC_IGNITION,					ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_AMP,					"MXRCUR",		"A",			"Maximum Restrike Current",								-0.0f,		20.0f,		{ 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ RESTRIKE_TIMING,						"RESTRIKE_TIMING",						CALC_IGNITION,					ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"RESTR",		"deg",			"Restrike Timing",										-60.0f,		60.0f,		{ 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 1 , 1 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-60.0f,		40.0f,		FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ START_OF_COMBUSTION,					"START_OF_COMBUSTION",					CALC_SOC,						ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"SOC",			"deg",			"Start of Combustion",									-60.0f,		60.0f,		{ 0 , 0 , 1 , 1 , 1 , 0 , 0 , 0 , 1 , 1 , 1 , 1 , 0 , 1 , 0 , 0 , 1 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 },	-60.0f,		40.0f,		FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ START_OF_INJECTION_1,					"START_OF_INJECTION_1",					CALC_INJECTOR_TIMING,			ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"SOI1_",		"deg",			"1st Start of Injection",								-360.0f,	360.0f,		{ 0 , 0 , 0 , 1 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ START_OF_INJECTION_2,					"START_OF_INJECTION_2",					CALC_INJECTOR_TIMING,			ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"SOI2_",		"deg",			"2nd Start of Injection",								-360.0f,	360.0f,		{ 0 , 0 , 0 , 1 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ START_OF_INJECTION_3,					"START_OF_INJECTION_3",					CALC_INJECTOR_TIMING,			ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"SOI3_",		"deg",			"3rd Start of Injection",								-360.0f,	360.0f,		{ 0 , 0 , 0 , 1 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ START_OF_INJECTION_4,					"START_OF_INJECTION_4",					CALC_INJECTOR_TIMING,			ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"SOI4_",		"deg",			"4th Start of Injection",								-360.0f,	360.0f,		{ 0 , 0 , 0 , 1 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ START_OF_INJECTION_5,					"START_OF_INJECTION_5",					CALC_INJECTOR_TIMING,			ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"SOI5_",		"deg",			"5th Start of Injection",								-360.0f,	360.0f,		{ 0 , 0 , 0 , 1 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ START_OF_INJECTION_6,					"START_OF_INJECTION_6",					CALC_INJECTOR_TIMING,			ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG,					"SOI6_",		"deg",			"6th Start of Injection",								-360.0f,	360.0f,		{ 0 , 0 , 0 , 1 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ TLA,									"TLA",									CALC_TLA,						ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_DEG_REL,				"TLA",			"deg",			"Thermodynamic Loss Angle",								-50.0f,		50.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-10.0f,		45.0f,		FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ TOOTH_GAP_RATIO_MAX,					"TOOTH_GAP_RATIO_MAX",					0,								ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_NONE,					"GAPRMX",		"-",			"Maximum Tooth Gap Ratio",								0.0f,		2.0f,		{ 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ TOOTH_GAP_RATIO_MIN,					"TOOTH_GAP_RATIO_MIN",					0,								ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_NONE,					"GAPRMN",		"-",			"Minimum Tooth Gap Ratio",								0.0f,		2.0f,		{ 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ TOTAL_HEAT_RELEASE,					"TOTAL_HEAT_RELEASE",					CALC_HEAT_RELEASE,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_J,					"QMAX",			"J",			"Total Heat Release",									0.0f,		5000.0f,	{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	200.0f,		7200.0f,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ TRAPPED_MASS,							"TRAPPED_MASS",							CALC_AIR_CHARGE,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_MG,					"T_MASS",		"mg",			"Trapped Gas Mass During Compression",					0.0f,		1000.0f,	{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ UPPER_PUMPING_IMEP,					"UPPER_PUMPING_IMEP",					CALC_WORK,						ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_BAR,					"UPMEP",		"bar",			"Upper Pumping IMEP",									0.0f,		25.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-5.5f,		1.8f,		FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ WIEBE_A,								"WIEBE_A",								CALC_WIEBE,						ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_NONE,					"WIEBEA",		"-",			"Wiebe A Coefficient",									0.0f,		2.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ WIEBE_M,								"WIEBE_M",								CALC_WIEBE,						ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_NONE,					"WIEBEM",		"-",			"Wiebe M Coefficient",									0.0f,		20.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },

	/* Frequency Based */

	{ FFT,									"FFT",									CALC_FOURIER,					ABSCISSA_CRANKANGLE,	ABSCISSA_FREQUENCY,		UNITS_AS_CHAN,				"FFT",			"-",			"Fourier Analysis",										0.0f,		1.0f,		{ 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 1 , 1 , 0 , 1 , 1 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ FFT_FREQUENCY,						"FFT_FREQUENCY",						CALC_FOURIER,					ABSCISSA_CRANKANGLE,	ABSCISSA_FREQUENCY,		UNITS_KHZ,					"FFTFREQ",		"kHz",			"Fourier Analysis Frequency",							0.0f,		30.0f,		{ 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 1 , 1 , 0 , 1 , 1 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ FFT_REFERENCE,						"FFT_REFERENCE",						CALC_FOURIER_REFERENCE,			ABSCISSA_CRANKANGLE,	ABSCISSA_FREQUENCY,		UNITS_AS_CHAN,				"FFTR",			"-",			"Fourier Analysis (Reference Window)",					0.0f,		1.0f,		{ 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ FFT_REFERENCE_FREQUENCY,				"FFT_REFERENCE_FREQUENCY",				CALC_FOURIER_REFERENCE,			ABSCISSA_CRANKANGLE,	ABSCISSA_FREQUENCY,		UNITS_KHZ,					"FFTR_FQ",		"kHz",			"Fourier Analysis Frequency (Ref Window)",				0.0f,		30.0f,		{ 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ FFT_TORQUE_SUM,						"FFT_TORQUE_SUM",						CALC_FOURIER_TQ_SUM,			ABSCISSA_CRANKANGLE,	ABSCISSA_FREQUENCY,		UNITS_AS_CHAN,				"TQSUMFFT",		"-",			"Torque Sum Fourier Analysis",							0.0f,		1.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ FFT_TORQUE_SUM_FREQUENCY,				"FFT_TORQUE_SUM_FREQUENCY",				CALC_FOURIER_TQ_SUM,			ABSCISSA_CRANKANGLE,	ABSCISSA_FREQUENCY,		UNITS_EO,					"TQSUMFQ",		"Order",		"Torque Sum Fourier Analysis Frequency",				0.0f,		8.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ FFT_KNOCK_PRESSURE,					"FFT_KNOCK_PRESSURE",					CALC_FOURIER_KNK_PRESSURE,		ABSCISSA_CRANKANGLE,	ABSCISSA_FREQUENCY,		UNITS_AS_CHAN,				"KNKFFT",		"-",			"Knock Pressure Fourier Analysis",						0.0f,		30.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ FFT_KNOCK_PRESSURE_FREQUENCY,			"FFT_KNOCK_PRESSURE_FREQUENCY",			CALC_FOURIER_KNK_PRESSURE,		ABSCISSA_CRANKANGLE,	ABSCISSA_FREQUENCY,		UNITS_KHZ,					"KNKFFTFQ",		"kHz",			"Knock Pressure Fourier Analysis Frequency",			0.0f,		30.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ FFT_TOOTH_PERIOD,						"FFT_TOOTH_PERIOD",						CALC_FOURIER,					ABSCISSA_CRANKANGLE,	ABSCISSA_FREQUENCY,		UNITS_AS_CHAN,				"THFFT",		"-",			"Tooth Period Fourier Analysis",						0.0f,		30.0f,		{ 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ FFT_TOOTH_PERIOD_FREQUENCY,			"FFT_TOOTH_PERIOD_FREQUENCY",			CALC_FOURIER,					ABSCISSA_CRANKANGLE,	ABSCISSA_FREQUENCY,		UNITS_HZ,					"THFFTFQ",		"Hz",			"Tooth Period Fourier Analysis Frequency",				0.0f,		30.0f,		{ 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },

	/* Crank Angle Based */

	{ ANGULAR_TORQUE,						"ANGULAR_TORQUE",						CALC_WORK_CA,					ABSCISSA_CRANKANGLE,	ABSCISSA_CRANKANGLE,	UNITS_NM,					"TQANG",		"Nm",			"Angular Indicated Torque",								-500.0f,	1000.0f,	{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ ANGULAR_TORQUE_SUM,					"ANGULAR_TORQUE_SUM",					CALC_WORK_CA,					ABSCISSA_CRANKANGLE,	ABSCISSA_CRANKANGLE,	UNITS_NM,					"TQANGSUM",		"Nm",			"Angular Indicated Torque for Whole Engine",			-500.0f,	1000.0f,	{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ D2P,									"D2P",									CALC_PRESSURE_D2P_CA,			ABSCISSA_CRANKANGLE,	ABSCISSA_CRANKANGLE,	UNITS_BAR_DEG2,				"D2P",			"bar/deg^2",	"Second Derivative of Pressure",						-10.0f,		10.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ CHANNEL_DELTA,						"CHANNEL_DELTA",						0,								ABSCISSA_CRANKANGLE,	ABSCISSA_CRANKANGLE,	UNITS_NONE,					"DIFF",			"-",			"Maximum Difference versus Reference Channel",			-100.0f,	100.0f,		{ 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 1 , 1 , 1 , 1 , 1 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL},
	{ CRANK_ANGLE_IMEP,						"CRANK_ANGLE_IMEP",						CALC_WORK_CA,					ABSCISSA_CRANKANGLE,	ABSCISSA_CRANKANGLE,	UNITS_BAR,					"IMEPCA",		"bar",			"Crank Angle IMEP",										-10.0f,		10.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ DIFFERENCE_PRESSURE,					"DIFFERENCE_PRESSURE",					CALC_PRESSURE_RATIO_CA,			ABSCISSA_CRANKANGLE,	ABSCISSA_CRANKANGLE,	UNITS_BAR,					"DP",			"bar",			"Difference Pressure",									0.0f,		50.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ DIGITAL_SIGNAL,						"DIGITAL_SIGNAL",						0,								ABSCISSA_CRANKANGLE,	ABSCISSA_CRANKANGLE,	UNITS_VOLTS,				"DIGI",			"V",			"Digitised Signal",										0.0f,		1.0f,		{ 0 , 0 , 0 , 1 , 1 , 1 , 0 , 0 , 1 , 1 , 1 , 1 , 0 , 1 , 1 , 1 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 1 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ EXTRAPOLATED_CYL_PRESSURE,			"EXTRAPOLATED_CYL_PRESSURE",			CALC_EST_VALVE_TIMING_CA,		ABSCISSA_CRANKANGLE,	ABSCISSA_CRANKANGLE,	UNITS_BAR,					"ECP",			"bar",			"Extrapolated Cylinder Pressure",						0.0f,		200.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ GAMMA,								"GAMMA",								CALC_HEAT_RELEASE_CA,			ABSCISSA_CRANKANGLE,	ABSCISSA_CRANKANGLE,	UNITS_NONE,					"GAMMA",		"-",			"Ratio of Specific Heats",								0.0f,		720.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ GROSS_HEAT_RELEASE_RATE,				"GROSS_HEAT_RELEASE_RATE",				CALC_HEAT_RELEASE_CA,			ABSCISSA_CRANKANGLE,	ABSCISSA_CRANKANGLE,	UNITS_J_DEG,				"GHR",			"J/deg",		"Gross Heat Release Rate",								0.0f,		500.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ H_COEFF,								"H_COEFF",								CALC_HEAT_RELEASE_CA,			ABSCISSA_CRANKANGLE,	ABSCISSA_CRANKANGLE,	UNITS_NONE,					"H",			"W/(m^2.K)",	"Heat Transfer Coefficient",							0.0f,		720.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ KNOCKING_PRESSURE,					"KNOCKING_PRESSURE",					CALC_KNOCK_CA,					ABSCISSA_CRANKANGLE,	ABSCISSA_CRANKANGLE,	UNITS_BAR,					"KNKP",			"bar",			"Knocking Pressure",									0.0f,		8.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ MEAN_GAS_TEMP,						"MEAN_GAS_TEMP",						CALC_TEMP_CA,					ABSCISSA_CRANKANGLE,	ABSCISSA_CRANKANGLE,	UNITS_K,					"MGTEMP",		"K",			"Mean Gas Temperature",									0.0f,		2500.0f,	{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ MFB,									"MFB",									CALC_BURN_RATES_CA,				ABSCISSA_CRANKANGLE,	ABSCISSA_CRANKANGLE,	UNITS_PERCENT,				"MFB",			"%",			"Mass Fraction Burned",									0.0f,		100.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ MOTORED_PRESSURE,						"MOTORED_PRESSURE",						CALC_MOTORED,					ABSCISSA_CRANKANGLE,	ABSCISSA_CRANKANGLE,	UNITS_BAR,					"MOTPR",		"bar",			"Motored Cylinder Pressure",							0.0f,		200.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ MOVING_PRESSURE_AVERAGE,				"MOVING_PRESSURE_AVERAGE",				CALC_PRESSURE_AVERAGE,			ABSCISSA_CRANKANGLE,	ABSCISSA_CRANKANGLE,	UNITS_BAR,					"PRAVG",		"bar",			"Moving Pressure Average",								0.0f,		200.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 1 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ NET_HEAT_RELEASE_RATE,				"NET_HEAT_RELEASE_RATE",				CALC_HEAT_RELEASE_CA,			ABSCISSA_CRANKANGLE,	ABSCISSA_CRANKANGLE,	UNITS_J_DEG,				"NHR",			"J/deg",		"Net Heat Release Rate",								0.0f,		500.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ POLYFIT,								"POLYFIT",								CALC_TLA_CA,					ABSCISSA_CRANKANGLE,	ABSCISSA_CRANKANGLE,	UNITS_BAR,					"POLYFIT",		"bar",			"Polynomial fit of Cylinder Pressure",					0.0f,		200.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ POLYTROPIC_INDICES,					"POLYTROPIC_INDICES",					CALC_POLYTROPIC_CA,				ABSCISSA_CRANKANGLE,	ABSCISSA_CRANKANGLE,	UNITS_NM,					"POLY",			"-",			"Polytropic Index",										0.0f,		5.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ PRESSURE_RATIO,						"PRESSURE_RATIO",						CALC_PRESSURE_RATIO_CA,			ABSCISSA_CRANKANGLE,	ABSCISSA_CRANKANGLE,	UNITS_NONE,					"PR",			"-",			"Pressure Ratio",										0.0f,		5.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ PRESSURE_RATIO_FRACTION,				"PRESSURE_RATIO_FRACTION",				CALC_PRESSURE_RATIO_CA,			ABSCISSA_CRANKANGLE,	ABSCISSA_CRANKANGLE,	UNITS_NONE,					"PRF",			"-",			"Pressure Ratio Fraction",								0.0f,		1.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ PRESSURE_RISE_RATE,					"PRESSURE_RISE_RATE",					CALC_PRESSURE_CA,				ABSCISSA_CRANKANGLE,	ABSCISSA_CRANKANGLE,	UNITS_BAR_DEG,				"PRR",			"bar/deg",		"Pressure Rise Rate",									0.0f,		1000.0f,	{ 0 , 0 , 1 , 0 , 0 , 0 , 1 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ TOOTH_SPEED,							"TOOTH_SPEED",							0,								ABSCISSA_CRANKANGLE,	ABSCISSA_CRANKANGLE,	UNITS_US,					"THSPD",		"us",			"Normalised Tooth-to-Tooth Engine Speed",				0.0f,		4.0f,		{ 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ VALVE_ACCEL,							"VALVE_ACCEL",							CALC_VALVE_LIFT_CA,				ABSCISSA_CRANKANGLE,	ABSCISSA_CRANKANGLE,	UNITS_MM_PER_DEG2,			"V_ACCEL",		"mm/deg^2",		"Valve Lift Acceleration",								0.0f,		100.0f,		{ 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ VALVE_VELOCITY,						"VALVE_VELOCITY",						CALC_VALVE_LIFT_CA,				ABSCISSA_CRANKANGLE,	ABSCISSA_CRANKANGLE,	UNITS_MM_PER_DEG,			"V_VEL",		"mm/deg",		"Valve Lift Velocity",									0.0f,		100.0f,		{ 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ WIEBE_MFB,							"WIEBE_MFB",							CALC_WIEBE_CA,					ABSCISSA_CRANKANGLE,	ABSCISSA_CRANKANGLE,	UNITS_BAR,					"WIEBE",		"%",			"Mass Fraction Burned from Wiebe parameters",			0.0f,		100.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	
	/* Time Based */

	{ SQUARE_CPS,							"SQUARE_CPS",							0,								ABSCISSA_TIME,			ABSCISSA_TIME,			UNITS_NONE,					"SQCPS",		"-",			"Square CPS Signal",									0.0f,		1.0f,		{ 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 1 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ CPS_CRANK_ANGLE,						"CPS_CRANK_ANGLE",						0,								ABSCISSA_TIME,			ABSCISSA_TIME,			UNITS_DEG,					"CPSCA",		"deg",			"CPS Derived Crank Angle",								-360.0f,	360.0f,		{ 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },

	/* Knock Pressure Histogram */

	{ KNOCK_PRESSURE_HISTOGRAM,				"KNOCK_PRESSURE_HISTOGRAM",				CALC_KNOCK_HISTOGRAM,			ABSCISSA_CRANKANGLE,	ABSCISSA_HISTOGRAM,		UNITS_PERCENT,				"KP_HIS",		"%",			"Knock Pressure Histogram",								0.0f,		100.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ KNOCK_PRESSURE_HIST_WARNING,			"KNOCK_PRESSURE_HIST_WARNING",			CALC_KNOCK_HISTOGRAM,			ABSCISSA_CRANKANGLE,	ABSCISSA_HISTOGRAM,		UNITS_PERCENT,				"WARN_L",		"%",			"Knock Pressure Histogram - Warning Limit",				0.0f,		100.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ KNOCK_PRESSURE_HIST_ALARM,			"KNOCK_PRESSURE_HIST_ALARM",			CALC_KNOCK_HISTOGRAM,			ABSCISSA_CRANKANGLE,	ABSCISSA_HISTOGRAM,		UNITS_PERCENT,				"ALRM_L",		"%",			"Knock Pressure Histogram - Alarm Limit",				0.0f,		100.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ KNOCK_PRESSURE_HIST_SHUTDOWN,			"KNOCK_PRESSURE_HIST_SHUTDOWN",			CALC_KNOCK_HISTOGRAM,			ABSCISSA_CRANKANGLE,	ABSCISSA_HISTOGRAM,		UNITS_PERCENT,				"SHUT_L",		"%",			"Knock Pressure Histogram - Shutdown Limit",			0.0f,		100.0f,		{ 0 , 0 , 1 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },

	/* Motor Power */

	{ MOTOR_ALPHA,							"MOTOR_ALPHA",							CALC_MOTOR_POWER,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_AMP,					"ALPHA",		"A",			"Motor Alpha",											-1000.0f,	1000.0f,	{ 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ MOTOR_BETA,							"MOTOR_BETA",							CALC_MOTOR_POWER,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_AMP,					"BETA",			"A",			"Motor Beta",											-1000.0f,	1000.0f,	{ 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ MOTOR_D,								"MOTOR_D",								CALC_MOTOR_POWER,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_AMP,					"D",			"A",			"Motor D",												-1000.0f,	1000.0f,	{ 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
	{ MOTOR_Q,								"MOTOR_Q",								CALC_MOTOR_POWER,				ABSCISSA_CRANKANGLE,	ABSCISSA_CYCLE,			UNITS_AMP,					"Q",			"A",			"Motor Q",												-1000.0f,	1000.0f,	{ 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 1 },	-FLT_MAX,	FLT_MAX,	FLT_MAX, -FLT_MAX, FLT_MAX, FLT_MAX, NULL, NULL, NULL },
};

typedef struct {

	unsigned int number;
	AnalysisConf* config;

} AnalysisChannelConfig;


typedef struct {
	float sum;
	float mean;
	float min;
	float max;
	float stddev;
	float lnv;
	float cov;
	float skewness;
	float kurtosis;
	float range;
} Statistics;

typedef struct {
	float* sum;
	float* mean;
	float* min;
	float* max;
	float* stddev;
	float* lnv;
	float* cov;
	float* skewness;
	float* kurtosis;
	float* range;
} StatisticsArray;

typedef struct {
	char name[SIZEOF_CHANNEL_NAME];
	
	unsigned int number_of_channels;
	unsigned int* channels;

	Statistics* cycle_statistics;
	Statistics* cycle_sum_statistics;
	Statistics* cylinder_mean_statistics;
	Statistics* engine_mean_statistics;
	StatisticsArray* crank_angle_statistics;
	StatisticsArray* cycle_sum_data;
} ChannelGroup;

typedef struct
{
	unsigned int type;
	char units[SIZEOF_UNITS];
	unsigned int number_of_measurement_tables;

	unsigned int* number_of_samples;
	unsigned int* start;
	unsigned int* resolution;
	
	unsigned int* ca_to_theta;
	unsigned int* theta_to_ca;
	float* theta_weighting;
	float* theta_weighting_inv;
	float* volume;
	float* ca_to_deg;

	void* axis;
	unsigned int axis_type;

	double slope;
	double offset;
} AbscissaData;

typedef struct
{
	unsigned int number_of_blocks;
	float start[5];
	float finish[5];
	float resolution[5];
} AbscissaBlocks;

typedef struct 
{
	unsigned int type;
	float polytropic_index;
	float fixed_value;
	char channel_name[SIZEOF_CHANNEL_NAME];
	float start_window;
	float finish_window;
	unsigned int window_size;
	bool truncate;
} ChannelOffsetConfig;

typedef struct
{
	float start_window;
	float finish_window;
	float fixed_value;
	unsigned int type;
	char channel_name[SIZEOF_CHANNEL_NAME];
	bool aligned;
	bool invert;
} SOCConfig;

typedef struct
{
	unsigned int type;
	float lower_frequency;
	float upper_frequency;
	bool filtered;
} FilterConfig;

typedef struct
{
	unsigned int type;
	float latch_high;
	float latch_low;
	bool invert;
	float filter;
} DigitalConfig;

typedef struct
{
	unsigned int type;

	float reference_angle;
	float offset;

	unsigned int edge;
	bool invert;
} CamshaftConfig;

typedef struct 
{
	char name[SIZEOF_CHANNEL_NAME];
	char short_name[SIZEOF_SHORT_NAME];
	char matlab_name[SIZEOF_CHANNEL_NAME];

	char description[SIZEOF_DESCRIPTION];
	unsigned int type;
	char units[SIZEOF_UNITS];
	float tdc_offset;
	float conversion_factor;

	double offset;
	double slope;

	float min;
	float max;

	float* data;
	double* data_d;

	float* duration;
	
	unsigned int samples_per_cycle;
	unsigned int cylinder;

	AbscissaData abscissa;

	ChannelOffsetConfig offset_config;
	SOCConfig soc_config;
	FilterConfig filter_config;
	DigitalConfig digital_config;
	CamshaftConfig cam_config;

	bool isoffset;
	bool loaded;
	bool file_flag;
} ChannelData;

typedef struct 
{
	float annand_a;
	float cd_finish_angle;
	float cd_start_angle;
	float crankcase_pressure;
	float d2p_window;
	float deactivation_delta_p;
	float eeoc_default;
	float eeoc_finish_window;
	float eeoc_index;
	float eeoc_start_window;
	float engine_speed;
	float fft_finish_window;
	float fft_reference_finish;
	float fft_reference_start;
	float fft_start_window;
	float fft_stats_maximum;
	float fft_stats_resolution;
	float injector_finish_window;
	float injector_start_window;
	float knock_boss_reference_finish_window;
	float knock_boss_reference_start_window;
	float knock_boss_finish_window;
	float knock_boss_start_window;
	float knock_hist_alarm_threshold_hist_factor;
	float knock_hist_alarm_threshold_pressure_factor;
	float knock_hist_lower_threshold;
	float knock_hist_shutdown_threshold_hist_factor;
	float knock_hist_shutdown_threshold_pressure_factor;
	float knock_hist_time_cycle;
	float knock_kbf;
	float knock_knkbint;
	float knock_onset_threshold;
	float knock_pkp;
	float mega_knock_kbf;
	float mega_knock_knkbint;
	float mega_knock_pkp;
	float mfb_n;
	float misfire_imep;
	float pkp_finish_angle;
	float pkp_smoothing_range;
	float pkp_smoothing_resolution;
	float pkp_start_angle;
	float poly_comp_finish_angle;
	float poly_comp_start_angle;
	float poly_exp_finish_angle;
	float poly_exp_start_angle;
	float pressure_rise_finish_angle;
	float pressure_rise_range;
	float pressure_rise_start_angle;
	float R;
	float soc_default;
	float soc_threshold;
	float temp_ref_ca;
	float tla_range;
	float t_ivc;
	float t_wall;
	float slowburn_imep;
	float wiebe_a_finish;
	float wiebe_a_start;
	float wiebe_a_step;
	float wiebe_m_finish;
	float wiebe_m_start;
	float wiebe_m_step;
	
	unsigned int eeoc_method;
	unsigned int engine_speed_channel;
	unsigned int engine_speed_type;
	unsigned int knock_hist_min_cycles;
	unsigned int knock_integral_type;
	unsigned int heat_release_model;
	unsigned int heat_release_window;
	unsigned int heat_transfer_model;
	unsigned int max_number_of_injections;
	unsigned int gamma_method;
	unsigned int gas_temp_model;
	unsigned int imep_method;
	unsigned int mfb_model;
	unsigned int misfire_cycles_f;
	unsigned int misfire_cycles_m;
	unsigned int misfire_cycles_s;
	unsigned int motored_pressure_method;
	unsigned int tla_method;
	unsigned int smoothed_pressure_method;
	unsigned int stats_type;
	unsigned int pressure_rise_method;

	bool align_injections_to_tdc;
	bool interpolate_mfb;
	bool restrike_analysis;

} AnalysisConfig;

typedef struct 
{
	unsigned int channel;
	unsigned int number_of_cycles;
	AnalysisConfig config;
	AnalysisRqst* analysis_rqst;

	ChannelData* results;
	unsigned char* cycle_classification;

	Statistics* statistics;
	StatisticsArray* ca_statistics;
	Statistics raw_statistics;
	StatisticsArray raw_ca_statistics;
} AnalysisChannel;

typedef struct
{
	AnalysisChannel* channel;
	ChannelGroup* group;

	unsigned int number_of_channels;
	unsigned int number_of_groups;
} Analysis;

typedef struct
{
	unsigned int number_of_data_points;
	float* crank_angle;
	float* volume;
} UserDefinedEngineVolume;

typedef struct 
{
	char name[SIZEOF_ENGINE_NAME];
	float bore;
	float stroke;
	float conrod_length;
	float pin_offset[MAX_NUMBER_OF_CYLINDERS];
	float compression_ratio;
	float piston_factor;

	unsigned int number_of_strokes;
	unsigned int type;
	unsigned int number_of_cylinders;
	unsigned int volume_calculation_type;
	
	int firing_order;
	float tdc_offset[MAX_NUMBER_OF_CYLINDERS];

	float clearance_volume;
	float swept_area;
	float swept_volume;

	UserDefinedEngineVolume user;
	
	float ivo_angle;
	float ivc_angle;
	float evo_angle;
	float evc_angle;

	float tdc_error;
} Engine;

typedef struct 
{
	float* data;
	unsigned int data_points;
} CSVData;

typedef struct Parameter_S
{
	int number;

	char name[SIZEOF_CHANNEL_NAME];
	char units[SIZEOF_UNITS];
	char description[SIZEOF_DESCRIPTION];

	unsigned int data_type;
	void* value;

	struct Parameter_S* next;
} Parameter;

typedef struct 
{
	Engine engine;

	unsigned int number_of_channels;
	unsigned int number_of_cycles;

	char comment[SIZEOF_DESCRIPTION];

	ChannelData* channel_data;

#ifdef _CATOOL_UNICODE_
	wchar_t filename[4096];
#else
	char filename[4096];
#endif
	time_t creation_time;

	bool preload;
	unsigned int file_type;
	bool sync_checked;
	unsigned int threads;

	unsigned int min_res_ppr;
	double time_resolution;
	unsigned int pulses_per_cycle;
	float minimum_resolution;
	float* theta_to_deg;

	/* AVL I-File Variables */
	PARBLK* ptr_APB;
	DATGRP* ptr_DGB;
	unsigned int ca_dgb;
	bool swap_endian;
	char* parameter_file;
#ifdef _CATOOL_UNICODE_
	wchar_t parameter_filename[4096];
#else
	char parameter_filename[4096];
#endif
	size_t parameter_filesize;
	Parameter* parameters;
} FileData;

typedef struct {
	float min;
	float max;
	void* ptr[2];
} ProgressStatus;

typedef struct {
	FileData* file;
	Analysis* analysis;
	unsigned int channel;
	bool error_check;
	ProgressStatus status;
	unsigned int thread_status;
} cycle_analysis_arguments;

typedef struct  {
	double kalfak_0;
	double kalfak_1; 
	float* data;
	unsigned int number_of_datapoints;
	unsigned int threads;
	short* buffer;
} float_to_short_arguments;

typedef struct {
	float* buffer;
	float* wsave;
	float* window;
	int* ifac;
	unsigned int samples_per_cycle;
	float speed_to_freq;
	unsigned int start_crank_angle;
	unsigned int number_of_samples;
} FFTConfig;

typedef struct {
	unsigned int method;

	unsigned int averaging_range;
	unsigned int start_window_theta;
	unsigned int finish_window_theta;
	unsigned int step_size;
	
	float index;
	float default_value;
	float* values;
} EEOCConfig;

/* Prototype definitions */

float RawToEngineering(const float raw, const Calibration* calibration);

/* Engine Structure Functions */

void Copy_Engine(const Engine* src, Engine* dest);
void Calculate_Geometry(Engine* engine);
void Initialise_Engine(Engine* engine);
void Delete_Engine(Engine* engine);
void Copy_UserDefinedEngineVolume(const UserDefinedEngineVolume* src, UserDefinedEngineVolume* dest);
bool wrap_valve_angles(Engine* engine);

/* Statistics Structure Functions */

void Zero_Statistics(Statistics* statistics);
void Zero_StatisticsArray(StatisticsArray* statistics);
void Allocate_StatisticsArray(StatisticsArray* statistics, const unsigned int samples_per_cycle);
void Reset_StatisticsArray(StatisticsArray* statistics, const unsigned int samples_per_cycle);
void calculate_ca_stats(StatisticsArray* stats, const float* data, const unsigned int samples_per_cycle, const unsigned int number_of_cycles);
void calculate_cycle_stats(Statistics* stats, const float* data, const unsigned int number_of_cycles, const unsigned int stats_type);
void calculate_single_cycle_stats(Statistics* stats, const float* data, const unsigned int samples_per_cycle, const unsigned int stats_type);
void calculate_group_stats(Analysis* analysis, const unsigned int cycle_to_analyse, const unsigned int cycles_to_analyse, const unsigned int stats_calcs, const AnalysisRqst request_threshold, const unsigned int min_res_ppr, const unsigned int number_of_strokes, const unsigned int stats_type);
void calculate_freq_stats(StatisticsArray* stats, const float* data, const float* frequency, const unsigned int unit, const unsigned int samples_per_cycle, const unsigned int number_of_cycles, float max_frequency, float resolution);
void statistical_analysis(AnalysisChannel* analysis);

/* Analysis Structure Functions */

bool configure_results_channel(const FileData* file,AnalysisChannel* analysis,const unsigned int analysis_channel);
void Initialise_Analysis(AnalysisConfig* analysis_config);
bool configure_analysis_channels(const FileData* file, AnalysisChannel* analysis);
bool validate_analysis_requests(const FileData* file, AnalysisChannel* analysis);
bool calculate_dependencies(AnalysisRqst* analysis_rqst, const unsigned int abscissa_type, const unsigned int channel_type, const unsigned int cylinder, const unsigned int last_cylinder, const bool restrike_analysis);
void Delete_Analysis(Analysis* analysis);
void Delete_Analysis_Channel(AnalysisChannel* analysis);
void Delete_Analysis_Group(Analysis* analysis);
bool Copy_Analysis(const AnalysisConfig* src, AnalysisConfig* dest);
bool validate_analysis_config(const FileData* file, const AnalysisChannel* analysis);
void Move_Analysis_Channel(const AnalysisChannel* from, AnalysisChannel* to);
void Zero_Analysis(FileData* file, Analysis* analysis);
void Initialise_AnalysisChannel(AnalysisChannel* analysis);
bool AnalysisName(const FileData* file, const char* channel_name, const char* analysis_name, char* name, char* units);

/* File Structure Functions */

bool Delete_File(FileData* file);
void Delete_Channel(FileData* file,const unsigned int channel);
void Delete_Channel_Data(ChannelData* channel_data);
bool Release_File(FileData** file);
FileData* Initialise_File(void);
bool Delete_File_Parameters(FileData* file);
void Shift_ChannelData(ChannelData* channel_data, const unsigned int first, const unsigned int number_of_cycles);
void Offset_Channel(FileData* file, const unsigned int channel, const float tdc_offset, float* signal_offset);
void Zero_File(FileData* file);
float Return_PinOffset(const FileData* file, const unsigned int channel);
void Change_Minimum_Resolution(FileData* file, const unsigned int pulses_per_rev);
void Move_ChannelData(const ChannelData* from, ChannelData* to);
bool Copy_Channel(FileData* file, const unsigned int from_channel, const unsigned int to_channel);
bool Copy_Channel_Between(const FileData* src, FileData* dst, const unsigned int from_channel, const unsigned int to_channel);
bool Copy_Channel_Data(const FileData* from, FileData* to, const unsigned int from_channel, const unsigned int to_channel);
FileData* Build_FileDataFromChannel(const FileData* src, const unsigned int channel, const unsigned int cycle, const bool two_copies);
bool Build_Channels(FileData* file);
void Create_Measurement_Tables(FileData* file, const unsigned int channel, const unsigned int number_of_measurement_tables, const unsigned int start[5], const unsigned int resolution[5], const unsigned int number_of_samples[5]);
void Rename_Channel(FileData* file, const unsigned int channel, const char* name);
bool Remove_Channel(FileData* file, Analysis* analysis, const unsigned int channel_to_delete);
void Copy_Filter_Config(const FilterConfig* from, FilterConfig* to);
void Copy_Offset_Config(const ChannelOffsetConfig* from, ChannelOffsetConfig* to);
void Copy_SOC_Config(const SOCConfig* from, SOCConfig* to);
void Copy_Digital_Config(const DigitalConfig* from, DigitalConfig* to);
void Copy_Camshaft_Config(const CamshaftConfig* from, CamshaftConfig* to);
void Resize_Number_Of_Cycles(FileData* file, Analysis* analysis, const unsigned int number_of_cycles);
bool Resample_Channel(FileData* file, const unsigned int channel, AbscissaData* new_abscissa, const bool interpolate);
void Select_Cycles(FileData* file, Analysis* analysis, const unsigned int* cycles);
bool Add_Channel(FileData* file, Analysis* analysis, const char* channel_name, const bool force = false);
bool Sample_Channel(FileData* file, Analysis* analysis, const unsigned int channel, const unsigned int sample_channel, const bool align);
void Initialise_ChannelData(ChannelData* channel_data);
bool Generate_Sin_Channel(FileData* file, const unsigned int channel, const float frequency);
bool Generate_Block_Channel(FileData* file, const unsigned int channel, const float start, const float finish);
bool Generate_CPS_Channel(FileData* file, const unsigned int channel, const unsigned int pulses_per_rev, float tooth_ratio);
bool Convert_Channel_To_Time(FileData* file, const unsigned int channel, double conversion_resolution);
bool Verify_File_Configuration(FileData* file, Analysis* analysis);
void Calculate_Min_Max(FileData* file, Analysis* analysis);
const char* AbscissaType(const unsigned int abscissa_type);
bool ChannelUnits(const FileData* file, const char* channel_name, char* units);
int GetChannelFromChannelData(const FileData* file, const ChannelData* channel_data);

/* Analysis Functions */

void calculate_channel_groups(const FileData* file, Analysis* analysis);
int channel_to_group(const Analysis* analysis, const unsigned int channel);
float channel_average(const FileData* file, const unsigned int channel, const unsigned int cycle, const unsigned int reference_channel, bool quick);
bool offset_correction_channel(FileData* file, AnalysisChannel* analysis, const unsigned int channel);
void raw_statistical_analysis(const FileData* file, AnalysisChannel* analysis, const unsigned int channel);
void pressure_cycle_analysis(FileData* file, AnalysisChannel* analysis, Analysis* file_analysis, ProgressStatus* status);
void fft_analysis(FileData* file, AnalysisChannel* analysis, const unsigned int channel, ProgressStatus* status);
void engine_speed_analysis(const FileData* file, float* data, const unsigned int channel);
void start_of_combustion_analysis(FileData* file, AnalysisChannel* analysis, Analysis* file_analysis, const unsigned int channel, ProgressStatus* status);
void sync_analysis(FileData* file);
float Volume(const FileData* file, const unsigned int channel, const unsigned int theta);
void Calculate_Volume(const FileData* file, const unsigned int channel, float** volume);
#ifdef _CATOOL_UNICODE_
bool Import_Custom_Volume(Engine* engine, const wchar_t* filename, const unsigned int max_theta);
#else
bool Import_Custom_Volume(Engine* engine, const char* filename, const unsigned int max_theta);
#endif
bool calculate_theta_lookups(FileData* file, const unsigned int channel);
bool cycle_duration_analysis(FileData* file, const unsigned int channel, const unsigned int engine_speed_type, const char* engine_speed_channel, const float default_engine_speed);
void Set_Raw_Channel_Min_Max(FileData* file, const unsigned int channel);
void Set_Results_Channel_Min_Max(AnalysisChannel* analysis, const unsigned int channel);
void* cycle_analysis(void* arguments);
void Return_Geometry_Data(const FileData* file, const unsigned int channel, float* dCA_data, float* volume_data, float* dVolume_data, float* HTA_data, float* logVolume_data, float* pressure_to_torque_data);
void error_check_analysis(const FileData* file, const AnalysisChannel* analysis);
void cycle_classification(AnalysisChannel* analysis, const unsigned int number_of_cycles);

bool channel_name_to_number(const FileData* file, const char* channel_name, const unsigned int abscissa_type, unsigned int* channel);
bool channel_name_to_number_ex(const FileData* file, const char* channel_name, const unsigned int abscissa_type, unsigned int* channel_number,const bool short_name);
bool channel_number_to_name(const FileData* file, const unsigned int channel, char* channel_name);
bool channel_number_to_name_ex(const FileData* file, const unsigned int channel, char* channel_name, const bool short_name);
bool analysis_name_to_analysis_type(const FileData* file, const Analysis* analysis, const char* channel_name, const unsigned int abscissa_type, unsigned int* channel, unsigned int* analysis_type);
float MinimumResolution(const FileData* file);
unsigned int PulsesPerCycle(const FileData* file);
unsigned int PulsesPerRev(const FileData* file);

void SetCAData(FileData* file, const unsigned int cycle, const unsigned int crank_angle, const unsigned int channel, const float value);
float ReturnCAData(const FileData* file, const unsigned int cycle, const unsigned int crank_angle, const unsigned int channel);
unsigned int ReturnTheta(const FileData* file, const unsigned int crank_angle, const unsigned int channel);
float ThetaToDegrees(const FileData* file, const unsigned int theta);
float ThetaAbsToDegrees(const FileData* file, const unsigned int theta);
unsigned int ReturnCrankAngle(const FileData* file, const unsigned int theta, const unsigned int channel);
float ReturnReferencedCAData(const FileData* file, const unsigned int cycle, const unsigned int crank_angle, const unsigned int channel, const unsigned int reference_channel, const unsigned int max_cycle, const bool wraparound);
float ReturnThetaData(const FileData* file, const unsigned int cycle, unsigned int theta, const unsigned int channel);
float ReturnThetaDataEx(const FileData* file, unsigned int theta, const unsigned int channel, const float* data);
unsigned int ReturnCalculatedTheta(const AbscissaData* abscissa, const unsigned int crank_angle);
unsigned int DegreesToTheta(const FileData* file, const float degrees);
unsigned int DegreesToThetaAbs(const FileData* file, const float degrees);
int DegreesToThetaRel(const FileData* file, const float degrees);
unsigned int DegreesToCrankAngle(const FileData* file, const float degrees, const unsigned int channel);
float CrankAngleToDegrees(const FileData* file, const unsigned int crank_angle, const unsigned int channel);
void clip_crank_angle(const FileData* file, unsigned int* crank_angle, const unsigned int channel);
unsigned int MaximumTheta(const FileData* file);
unsigned int MaximumTheta(const FileData* file, const unsigned int channel);
bool validate_angle(const FileData* file, const unsigned int channel, const float angle);
float ReturnAnalysisThetaData(const FileData* file, const Analysis* analysis, const unsigned int cycle, unsigned int theta, const unsigned int channel, const unsigned int analysis_type);
float ReturnAnalysisCAData(const FileData* file, const Analysis* analysis, const unsigned int cycle, const unsigned int crank_angle, const unsigned int channel, const unsigned int analysis_type);

#include "abscissa.h"
#include "analysis.h"
#include "atimat.h"
#include "csvfile.h"
#include "dat670.h"
#include "fftpack.h"
#include "fileprocessor.h"
#include "filter.h"
#include "ifile.h"
#include "matfile.h"
#include "maths.h"
#include "plugin.h"
#include "sdf.h"
#include "utils.h"

#endif
