/* Combustion Analysis Tool (CAT)
   www.catool.org
  
   Filename: cat.c
  
   Purpose:  Provide common programming interface for processing acquired engine data
  
   Author:   Ben Brown
   Version:  1.2
   Date:     19.10.2015

   Revision: GPL Release

   Copyright (C) Xarin Limited, 2000-2024

      This program is free software: you can redistribute it and/or modify
      it under the terms of version 2 of the GNU General Public License as
      published by the Free Software Foundation.

      This program is distributed in the hope that it will be useful,
      but WITHOUT ANY WARRANTY; without even the implied warranty of
      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
      GNU General Public License for more details.

      You should have received a copy of the GNU General Public License
      along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* For Microsoft Visual C++ */
#ifdef _MSC_VER
#include "stdafx.h"
#endif

#include <stdlib.h>
#include <string.h>
#include <float.h>
#include <math.h>

#include "cat.h"

#ifdef __CATOOLRT__
#include "catool_closed.h"
#endif

extern unsigned int debug_level;

void Change_Minimum_Resolution(FileData* file, const unsigned int pulses_per_rev)
{
	unsigned int theta;
	float offset;
	unsigned int maximum_theta;
    
	if (file == NULL)
	{
		return;
	}

	offset = (float)(MINIMUM_THETA_ANGLE_MULT * file->engine.number_of_strokes);
	file->min_res_ppr = pulses_per_rev;
	file->pulses_per_cycle = file->min_res_ppr  * file->engine.number_of_strokes / 2;
	file->minimum_resolution = (float)file->min_res_ppr / 360.0f;

	if (file->theta_to_deg != NULL)
	{
		free(file->theta_to_deg);
	}

	maximum_theta = file->pulses_per_cycle * 2;
	
	file->theta_to_deg = (float*)malloc((maximum_theta + 1) * sizeof(float));

	if (file->theta_to_deg == NULL)
	{
		logmessage(FATAL,"Memory could not be allocated\n");
		exit(EXIT_FAILURE);
	}

	for (theta=0;theta<=maximum_theta;theta++)
	{
		file->theta_to_deg[theta] = ((float)theta/file->minimum_resolution) - offset;
	}
}

void Initialise_Analysis(AnalysisConfig* analysis_config)
{
	analysis_config->fft_start_window = 0.0f;
	analysis_config->fft_finish_window = 40.0f;
	analysis_config->eeoc_method = EEOC_METHOD_BRUNT;
	analysis_config->eeoc_start_window = -10.0f;				/* Brunt et al, SAE 970037 (TDC + 10) */
	analysis_config->eeoc_finish_window = 130.0f;				/* Brunt et al, SAE 970037 (EVO - 10) */
	analysis_config->eeoc_index = 1.15f;
	analysis_config->eeoc_default = 60.0f;
	analysis_config->t_ivc = 340.0f;							/* Brunt et al, SAE 1999-01-0187 */
	analysis_config->temp_ref_ca = -40.0f;
	analysis_config->mfb_n = 1.32f;
	analysis_config->annand_a = 0.45f;							/* Brunt et al, SAE 981052 */
	analysis_config->t_wall = 450.0f;
	analysis_config->R = 287.0f;
	analysis_config->poly_exp_start_angle = 60.0f;
	analysis_config->poly_exp_finish_angle = 100.0f;
	analysis_config->poly_comp_start_angle = -70.0f;			/* was -100.0f */
	analysis_config->poly_comp_finish_angle = -30.0f;			/* was -65.0f */
	analysis_config->pkp_start_angle = 0.0f;					/* Brunt et al, SAE */
	analysis_config->pkp_finish_angle = 40.0f;					/* Brunt et al, SAE */
	analysis_config->pkp_smoothing_range = 2.0f;				/* Brunt et al, SAE */
	analysis_config->pkp_smoothing_resolution = 0.2f;			/* Brunt et al, SAE */
	analysis_config->pressure_rise_start_angle = -90.0f;
	analysis_config->pressure_rise_finish_angle = 90.0f;
	analysis_config->pressure_rise_range = 2.0f;
	analysis_config->knock_pkp = 2.0f;
	analysis_config->mega_knock_pkp = 10.0f;
	analysis_config->knock_kbf = 10000.0f;
	analysis_config->knock_knkbint = 10000.0f;
	analysis_config->mega_knock_kbf = 10000.0f;
	analysis_config->mega_knock_knkbint = 10000.0f;
	analysis_config->misfire_imep = 0.2f;
	analysis_config->slowburn_imep = 5.0f;
	analysis_config->cd_start_angle = 0.0f;
	analysis_config->cd_finish_angle = 40.0f;
	analysis_config->engine_speed_type = N_AVLRZT;
	analysis_config->engine_speed = 1000.0f;
	analysis_config->engine_speed_channel = 0;
	analysis_config->heat_release_model = HR_FIRSTLAW;
	analysis_config->heat_transfer_model = HT_WOSCHNI;
	analysis_config->knock_integral_type = KNOCK_RECTIFIED_INTEGRAL;
	analysis_config->tla_range = 14.0f;
	analysis_config->wiebe_a_start = 0.1f;
	analysis_config->wiebe_a_finish = 30.0f;
	analysis_config->wiebe_a_step = 0.1f;
	analysis_config->wiebe_m_start = 0.1f;
	analysis_config->wiebe_m_finish = 10.0f;
	analysis_config->wiebe_m_step = 0.1f;
	analysis_config->injector_start_window = -360.0f;
	analysis_config->injector_finish_window = 360.0f;
	analysis_config->align_injections_to_tdc = false;
	analysis_config->max_number_of_injections = 6;
	analysis_config->interpolate_mfb = false;
	analysis_config->mfb_model = MFB_CATOOL;
	analysis_config->restrike_analysis = false;
	analysis_config->fft_reference_start = -40.0f;
	analysis_config->fft_reference_finish = 0.0f;
	analysis_config->crankcase_pressure = 1.0f;
	analysis_config->deactivation_delta_p = 5.0f;
	analysis_config->soc_threshold = 0.1f;
	analysis_config->gamma_method = GAMMA_BRUNT;
	analysis_config->pressure_rise_method = PRR_DERIVATIVE;
	analysis_config->imep_method = IMEP_MEAN_PRESSURE;
	analysis_config->motored_pressure_method = MOTORED_POLY_COMP | MOTORED_POLY_FINISH;
	analysis_config->smoothed_pressure_method = SMOOTH_MOVING_AVERAGE;
	analysis_config->d2p_window = 1.0f;
	analysis_config->knock_onset_threshold = 1.0f;
	analysis_config->heat_release_window = HR_WINDOW_SOC_EEOC;
	analysis_config->tla_method = TLA_POLYNOMIAL;
	analysis_config->misfire_cycles_f = 25;
	analysis_config->misfire_cycles_m = 100;
	analysis_config->misfire_cycles_s = 1000;
	analysis_config->knock_boss_start_window = 0.0f;
	analysis_config->knock_boss_finish_window = 40.0f;
	analysis_config->knock_boss_reference_start_window = -40.0f;
	analysis_config->knock_boss_reference_finish_window = 0.0f;
	analysis_config->fft_stats_maximum = 100.0f;
	analysis_config->fft_stats_resolution = 0.25f;
	analysis_config->knock_hist_alarm_threshold_hist_factor = 1.2f;
	analysis_config->knock_hist_alarm_threshold_pressure_factor = 1.5f;
	analysis_config->knock_hist_lower_threshold = 10.0f;
	analysis_config->knock_hist_shutdown_threshold_hist_factor = 1.4f;
	analysis_config->knock_hist_shutdown_threshold_pressure_factor = 2.0f;
	analysis_config->knock_hist_time_cycle = 1.5f;
	analysis_config->knock_hist_min_cycles = 30;
	analysis_config->gas_temp_model = GAS_TEMP_REF_CA;
	analysis_config->stats_type = STATS_RATIO;
}

void Copy_Offset_Config(const ChannelOffsetConfig* from, ChannelOffsetConfig* to)
{
	if ((from == NULL) || (to == NULL))
	{
		return;
	}

	to->window_size = from->window_size;
	to->fixed_value = from->fixed_value;
	strncpy(to->channel_name, from->channel_name, SIZEOF_CHANNEL_NAME);
	to->start_window = from->start_window;
	to->finish_window = from->finish_window;
	to->type = from->type;
	to->polytropic_index = from->polytropic_index;
	to->truncate = from->truncate;
}

void Copy_SOC_Config(const SOCConfig* from, SOCConfig* to)
{
	if ((from == NULL) || (to == NULL))
	{
		return;
	}

	to->type = from->type;
	strncpy(to->channel_name, from->channel_name, SIZEOF_CHANNEL_NAME);
	to->start_window = from->start_window;
	to->finish_window = from->finish_window;
	to->fixed_value = from->fixed_value;
	to->aligned = from->aligned;
	to->invert = from->invert;
}

void Copy_Camshaft_Config(const CamshaftConfig* from, CamshaftConfig* to)
{
	if ((from == NULL) || (to == NULL))
	{
		return;
	}

	to->edge = from->edge;
	to->invert = from->invert;
	to->offset = from->offset;
	to->reference_angle = from->reference_angle;
	to->type = from->type;
}

void Copy_Filter_Config(const FilterConfig* from, FilterConfig* to)
{
	if ((from == NULL) || (to == NULL))
	{
		return;
	}

	to->type = from->type;
	to->filtered = from->filtered;
	to->upper_frequency = from->upper_frequency;
	to->lower_frequency = from->lower_frequency;
}

void Copy_Digital_Config(const DigitalConfig* from, DigitalConfig* to)
{
	if ((from == NULL) || (to == NULL))
	{
		return;
	}

	to->type = from->type;
	to->invert = from->invert;
	to->latch_high = from->latch_high;
	to->latch_low = from->latch_low;
	to->filter = from->filter;
}

void Copy_UserDefinedEngineVolume(const UserDefinedEngineVolume* src, UserDefinedEngineVolume* dest)
{
	if ((src == NULL) || (dest == NULL))
	{
		return;
	}

	dest->number_of_data_points = src->number_of_data_points;

	if (dest->crank_angle != NULL)
	{
		free(dest->crank_angle);
		dest->crank_angle = NULL;
	}

	if (dest->volume != NULL)
	{
		free(dest->volume);
		dest->volume = NULL;
	}

	if (src->number_of_data_points > 0)
	{
		dest->crank_angle = (float*)malloc(src->number_of_data_points * sizeof(float));

		if (dest->crank_angle == NULL)
		{
			logmessage(FATAL, "Out of memory\n");
		}

		memcpy(dest->crank_angle, src->crank_angle, src->number_of_data_points * sizeof(float));

		dest->volume = (float*)malloc(src->number_of_data_points * sizeof(float));

		if (dest->volume == NULL)
		{
			logmessage(FATAL, "Out of memory\n");
		}

		memcpy(dest->volume, src->volume, src->number_of_data_points * sizeof(float));
	}
}

void Copy_Engine(const Engine* src, Engine* dest)
{
	unsigned int cylinder;

	if ((src == NULL) || (dest == NULL))
	{
		return;
	}

	strncpy(dest->name, src->name, SIZEOF_ENGINE_NAME);

	dest->bore = src->bore;
	dest->stroke = src->stroke;
	dest->conrod_length = src->conrod_length;
	for (cylinder = 0; cylinder < MAX_NUMBER_OF_CYLINDERS; cylinder++)
	{
		dest->pin_offset[cylinder] = src->pin_offset[cylinder];
		dest->tdc_offset[cylinder] = src->tdc_offset[cylinder];
	}
	dest->compression_ratio = src->compression_ratio;

	dest->number_of_strokes = src->number_of_strokes;
	dest->type = src->type;
	dest->number_of_cylinders = src->number_of_cylinders;

	dest->clearance_volume = src->clearance_volume;
	dest->swept_area = src->swept_area;
	dest->swept_volume = src->swept_volume;
	dest->ivc_angle = src->ivc_angle;
	dest->evo_angle = src->evo_angle;
	dest->ivo_angle = src->ivo_angle;
	dest->evc_angle = src->evc_angle;

	dest->volume_calculation_type = src->volume_calculation_type;
	dest->firing_order = src->firing_order;

	dest->piston_factor = src->piston_factor;
	dest->tdc_error = src->tdc_error;

	Copy_UserDefinedEngineVolume(&src->user, &dest->user);
}

void Copy_FileData(const FileData* src, FileData* dst)
{
	/* Creates new FileData structure, but does not copy ChannelData, parameters and sets 
	   number_of_cycles and number_of_channels to zero */

	if ((src == NULL) || (dst == NULL))
	{
		return;
	}

	dst->ca_dgb = src->ca_dgb;
	dst->channel_data = NULL;
	strncpy(dst->comment, src->comment, SIZEOF_DESCRIPTION);
	dst->creation_time = src->creation_time;
	Copy_Engine(&src->engine, &dst->engine);
#ifdef _CATOOL_UNICODE_
	wcsncpy(dst->filename, src->filename, 4095);
#else
	strncpy(dst->filename, src->filename, 4095);
#endif
	dst->file_type = src->file_type;
	dst->minimum_resolution = src->minimum_resolution;
	dst->min_res_ppr = src->min_res_ppr;
	dst->number_of_channels = 0;
	dst->number_of_cycles = 0;
	dst->parameters = NULL;
	dst->parameter_file = 0x00;
#ifdef _CATOOL_UNICODE_
	dst->parameter_filename[0] = 0x0000;
#else
	dst->parameter_filename[0] = 0x00;
#endif
	dst->parameter_filesize = 0;
	dst->preload = src->preload;
	dst->ptr_APB = NULL;
	dst->ptr_DGB = NULL;
	dst->pulses_per_cycle = src->pulses_per_cycle;
	dst->swap_endian = src->swap_endian;
	dst->sync_checked = src->sync_checked;
	dst->theta_to_deg = NULL;		/* Recalculated in Change_Minimum_Resolution() */
	dst->threads = src->threads;
	dst->time_resolution = src->time_resolution;

	Change_Minimum_Resolution(dst, src->min_res_ppr);
}

FileData* Build_FileDataFromChannel(const FileData* src, const unsigned int channel, const unsigned int cycle, const bool two_copies)
{
	unsigned int offset_channel;
	unsigned int chan;

	if (src == NULL)
	{
		return(NULL);
	}

	if (channel >= src->number_of_channels)
	{
		return(NULL);
	}

	if (cycle >= src->number_of_cycles)
	{
		return(NULL);
	}

	FileData* file = Initialise_File();

	if (file == NULL)
	{
		return(NULL);
	}

	Copy_FileData(src, file);

	file->number_of_channels = two_copies ? 2 : 1;
	file->number_of_cycles = src->number_of_cycles;

	switch (src->channel_data[channel].offset_config.type)
	{
	case OFFSET_RESULT_CHANNEL:
	case OFFSET_WINDOW:
	case OFFSET_WINDOW_ABSOLUTE:
	{
		if (channel_name_to_number(src, src->channel_data[channel].offset_config.channel_name, ABSCISSA_UNKNOWN, &offset_channel) == true)
		{
			file->number_of_channels++;
		}

		break;
	}
	default:
	{
		break;
	}
	}

	file->channel_data = (ChannelData*)malloc(sizeof(ChannelData) * file->number_of_channels);

	for (chan = 0; chan < (two_copies ? 2U : 1U); chan++)
	{
		file->channel_data[chan].data = NULL;
		file->channel_data[chan].data_d = NULL;
		file->channel_data[chan].duration = NULL;

		Zero_Abscissa(&file->channel_data[chan].abscissa);

		Copy_Channel_Between(src, file, channel, chan);
	}

	if (file->number_of_channels > (two_copies ? 2U : 1U))
	{
		file->channel_data[chan].data = NULL;
		file->channel_data[chan].data_d = NULL;
		file->channel_data[chan].duration = NULL;

		Zero_Abscissa(&file->channel_data[chan].abscissa);

		Copy_Channel_Between(src, file, offset_channel, chan);
	}

	if (two_copies == true)
	{
		snprintf(file->channel_data[1].name, SIZEOF_CHANNEL_NAME, "%s Original", file->channel_data[0].name);
	}

	return(file);
}

void Delete_Engine(Engine* engine)
{
	if (engine == NULL)
	{
		return;
	}

	engine->user.number_of_data_points = 0;

	if (engine->user.volume != NULL)
	{
		free(engine->user.volume);
		engine->user.volume = NULL;
	}

	if (engine->user.crank_angle != NULL)
	{
		free(engine->user.crank_angle);
		engine->user.crank_angle = NULL;
	}
}

const char* AbscissaType(const unsigned int abscissa_type)
{
	if (abscissa_type < NUMBER_OF_ABSCISSA_TYPES)
	{
		return(abscissa_information[abscissa_type].description);
	}

	return(abscissa_information[ABSCISSA_UNKNOWN].description);
}

bool AnalysisName(const FileData* file, const char* channel_name, const char* analysis_name, char* name, char* units)
{
	unsigned int channel;
	int analysis_type;

	if ((file == NULL) || (channel_name == NULL) || (analysis_name == NULL))
	{
		return(false);
	}

	if (file->channel_data == NULL)
	{
		return(false);
	}

	if (channel_name_to_number(file, channel_name, ABSCISSA_UNKNOWN, &channel) == false)
	{
		return(false);
	}
	
	analysis_type = get_analysis_type_from_unique_id(analysis_name);

	if (analysis_type == -1)
	{
		return(false);
	}

	if (get_analysis_channel_validity(analysis_type, file->channel_data[channel].type) == 0)
	{
		return(false);
	}

	if (get_analysis_output_abscissa(analysis_type) != ABSCISSA_CYCLE)
	{
		return(false);
	}

	if (name != NULL)
	{
		snprintf(name, SIZEOF_CHANNEL_NAME, "%s%u", get_analysis_name(analysis_type), file->channel_data[channel].cylinder);
	}
	
	if (units != NULL)
	{
		strncpy(units, get_analysis_units(analysis_type), SIZEOF_UNITS);
	}

	return(true);
}

bool ChannelUnits(const FileData* file, const char* channel_name, char* units)
{
	unsigned int channel;

	if ((file == NULL) || (channel_name == NULL))
	{
		return(false);
	}

	if (file->channel_data == NULL)
	{
		return(false);
	}

	if (channel_name_to_number(file, channel_name, ABSCISSA_UNKNOWN, &channel) == false)
	{
		return(false);
	}

	if (units != NULL)
	{
		strncpy(units, file->channel_data[channel].units, SIZEOF_UNITS);
	}

	return(true);
}

void Resize_Number_Of_Cycles(FileData* file, Analysis* analysis, const unsigned int number_of_cycles)
{
	unsigned int channel;
	float* original_data;
	unsigned int data_size;
	unsigned int new_data_size;
	unsigned int analysis_structure;
	unsigned int analysis_channel;

	if ((file == NULL) || (number_of_cycles == 0))
	{
		return;
	}

	if ((number_of_cycles == file->number_of_cycles) || (file->channel_data == NULL))
	{
		return;
	}

	analysis_structure = 0;
	for (channel = 0; channel < file->number_of_channels; channel++)
	{
		if ((file->channel_data[channel].file_flag == true) && (file->channel_data[channel].loaded == true))
		{
			original_data = file->channel_data[channel].data;

			if (original_data != NULL)
			{
				if ((file->channel_data[channel].abscissa.type == ABSCISSA_CRANKANGLE) ||
					(file->channel_data[channel].abscissa.type == ABSCISSA_FREQUENCY))
				{
					data_size = file->number_of_cycles * file->channel_data[channel].samples_per_cycle;
					new_data_size = number_of_cycles * file->channel_data[channel].samples_per_cycle;
				}
				else if (file->channel_data[channel].abscissa.type == ABSCISSA_CYCLE)
				{
					data_size = file->channel_data[channel].samples_per_cycle;
					new_data_size = number_of_cycles;

					file->channel_data[channel].samples_per_cycle = new_data_size;
				}
				else
				{
					data_size = 0;
					new_data_size = 0;
				}

				if (new_data_size > 0)
				{
					file->channel_data[channel].data = (float*)malloc(new_data_size * sizeof(float));
					if (file->channel_data[channel].data == NULL)
					{
						logmessage(FATAL, "Memory could not be allocated\n");
						exit(EXIT_FAILURE);
					}

					if (new_data_size > data_size)
					{
						memcpy((void*)file->channel_data[channel].data, (void*)original_data, data_size * sizeof(float));
						memset((void*)&file->channel_data[channel].data[data_size], 0, (new_data_size - data_size) * sizeof(float));
					}
					else
					{
						memcpy((void*)file->channel_data[channel].data, (void*)original_data, new_data_size * sizeof(float));
					}

					free(original_data);
					original_data = NULL;
				}
			}

			original_data = file->channel_data[channel].duration;

			if (original_data != NULL)
			{
				file->channel_data[channel].duration = (float*)malloc(number_of_cycles * sizeof(float));

				if (number_of_cycles > file->number_of_cycles)
				{
					memcpy((void*)file->channel_data[channel].duration, (void*)original_data, file->number_of_cycles * sizeof(float));
					memset((void*)&file->channel_data[channel].duration[file->number_of_cycles], 0, (number_of_cycles - file->number_of_cycles) * sizeof(float));
				}
				else
				{
					memcpy((void*)file->channel_data[channel].duration, (void*)original_data, number_of_cycles * sizeof(float));
				}

				free(original_data);
				original_data = NULL;
			}

			if (analysis != NULL)
			{
				for (analysis_channel = 0; analysis_channel < get_number_of_analysis_channels(); analysis_channel++)
				{
					original_data = analysis->channel[analysis_structure].results[analysis_channel].data;

					if ((analysis->channel[analysis_structure].results[analysis_channel].abscissa.type == ABSCISSA_CRANKANGLE) ||
						(analysis->channel[analysis_structure].results[analysis_channel].abscissa.type == ABSCISSA_FREQUENCY))
					{
						data_size = analysis->channel[analysis_structure].number_of_cycles * analysis->channel[analysis_structure].results[analysis_channel].samples_per_cycle;
						new_data_size = number_of_cycles * analysis->channel[analysis_structure].results[analysis_channel].samples_per_cycle;
					}
					else if (file->channel_data[channel].abscissa.type == ABSCISSA_CYCLE)
					{
						data_size = analysis->channel[analysis_structure].results[analysis_channel].samples_per_cycle;
						new_data_size = number_of_cycles;

						analysis->channel[analysis_structure].results[analysis_channel].samples_per_cycle = new_data_size;
					}
					else
					{
						data_size = 0;
						new_data_size = 0;
					}

					if (new_data_size > 0)
					{
						analysis->channel[analysis_structure].results[analysis_channel].data = (float*)malloc(new_data_size * sizeof(float));
						if (analysis->channel[analysis_structure].results[analysis_channel].data == NULL)
						{
							logmessage(FATAL, "Memory could not be allocated\n");
							exit(EXIT_FAILURE);
						}

						if (new_data_size > data_size)
						{
							memcpy((void*)analysis->channel[analysis_structure].results[analysis_channel].data, (void*)original_data, data_size * sizeof(float));
							memset((void*)&analysis->channel[analysis_structure].results[analysis_channel].data[data_size], 0, (new_data_size - data_size) * sizeof(float));
						}
						else
						{
							memcpy((void*)analysis->channel[analysis_structure].results[analysis_channel].data, (void*)original_data, new_data_size * sizeof(float));
						}

						free(original_data);
						original_data = NULL;
					}
				}
			}

			analysis_structure++;
		}
	}

	file->number_of_cycles = number_of_cycles;
}

void Select_Cycles(FileData* file, Analysis* analysis, const unsigned int* cycles)
{
	unsigned int channel;
	unsigned int cycle;
	unsigned int data_pointer;
	unsigned int new_number_of_cycles;
	unsigned int analysis_channel;
	unsigned int analysis_structure;

	if ((file == NULL) || (cycles == NULL))
	{
		return;
	}

	if (file->channel_data == NULL)
	{
		return;
	}

	new_number_of_cycles = 0;
	for (cycle = 0; cycle < file->number_of_cycles; cycle++)
	{
		new_number_of_cycles += cycles[cycle];
	}

	if (new_number_of_cycles == 0)
	{
		return;
	}

	analysis_structure = 0;

	for (channel = 0; channel < file->number_of_channels; channel++)
	{
		if ((file->channel_data[channel].file_flag == true) && (file->channel_data[channel].loaded == true))
		{
			data_pointer = 0;

			if (file->channel_data[channel].abscissa.type == ABSCISSA_CRANKANGLE)
			{
				for (cycle = 0; cycle < file->number_of_cycles; cycle++)
				{
					if (cycles[cycle] == 1)
					{
						memcpy(&file->channel_data[channel].data[data_pointer], &file->channel_data[channel].data[cycle*file->channel_data[channel].samples_per_cycle], file->channel_data[channel].samples_per_cycle*sizeof(float));
						data_pointer += file->channel_data[channel].samples_per_cycle;
					}
				}
			}
			else if (file->channel_data[channel].abscissa.type == ABSCISSA_CYCLE)
			{
				for (cycle = 0; cycle < file->number_of_cycles; cycle++)
				{
					if (cycles[cycle] == 1)
					{
						file->channel_data[channel].data[data_pointer] = file->channel_data[channel].data[cycle];
						data_pointer++;
					}
				}
			}
			else
			{
				/* Do Nothing */
			}

			if (analysis != NULL)
			{
				for (analysis_channel = 0; analysis_channel < get_number_of_analysis_channels(); analysis_channel++)
				{
					data_pointer = 0;

					if (analysis->channel[analysis_structure].results[analysis_channel].abscissa.type == ABSCISSA_CRANKANGLE)
					{
						for (cycle = 0; cycle < analysis->channel[analysis_structure].number_of_cycles; cycle++)
						{
							if (cycles[cycle] == 1)
							{
								memcpy(&analysis->channel[analysis_structure].results[analysis_channel].data[data_pointer], &analysis->channel[analysis_structure].results[analysis_channel].data[cycle*analysis->channel[analysis_structure].results[analysis_channel].samples_per_cycle], analysis->channel[analysis_structure].results[analysis_channel].samples_per_cycle*sizeof(float));
								data_pointer += analysis->channel[analysis_structure].results[analysis_channel].samples_per_cycle;
							}
						}
					}
					else if (analysis->channel[channel].results[analysis_channel].abscissa.type == ABSCISSA_CYCLE)
					{
						for (cycle = 0; cycle < analysis->channel[analysis_structure].number_of_cycles; cycle++)
						{
							if (cycles[cycle] == 1)
							{
								analysis->channel[analysis_structure].results[analysis_channel].data[data_pointer] = analysis->channel[analysis_structure].results[analysis_channel].data[cycle];
								data_pointer++;
							}
						}
					}
					else
					{
						/* Do Nothing */
					}
				}

				analysis->channel[analysis_structure].number_of_cycles = new_number_of_cycles;
			}

			analysis_structure++;
		}
	}

	file->number_of_cycles = new_number_of_cycles;
}

bool calculate_theta_lookups(FileData* file, const unsigned int channel)
{
	bool result;

	if (file == NULL)
	{
		return(false);
	}

	if ((file->channel_data == NULL) || (channel >= file->number_of_channels) || (file->number_of_channels == 0))
	{
		return(false);
	}

	if (file->channel_data[channel].abscissa.type != ABSCISSA_CRANKANGLE)
	{
		return(false);
	}

	result = Calculate_Abscissa_Lookups(&file->channel_data[channel].abscissa, PulsesPerCycle(file), file->engine.number_of_strokes, MinimumResolution(file));

	if (result == false)
	{
		logmessage(WARNING, "Calculate_Abscissa_Lookups: %s has excessive theta index\n", file->channel_data[channel].name);
	}

	unsigned int maximum_theta = MaximumTheta(file);// MaximumTheta(file, channel);
	unsigned int measurement_table;

	if (result == true)
	{
		for (measurement_table = 0; measurement_table < file->channel_data[channel].abscissa.number_of_measurement_tables; measurement_table++)
		{
			unsigned int start = file->channel_data[channel].abscissa.start[measurement_table];
			unsigned int samples = file->channel_data[channel].abscissa.number_of_samples[measurement_table];
			unsigned int resolution = file->channel_data[channel].abscissa.resolution[measurement_table];

			unsigned int finish = start + (samples - 1) * resolution;

			if (finish >= maximum_theta)
			{
				logmessage(WARNING, "%s: Error in measurement table %u of %u: Final sample (%u) greater or equal to maximum theta (%u)\n", file->channel_data[channel].name, measurement_table, file->channel_data[channel].abscissa.number_of_measurement_tables, finish, maximum_theta);
				return(false);
			}
			
			if (measurement_table < file->channel_data[channel].abscissa.number_of_measurement_tables - 1)
			{
				if (finish >= file->channel_data[channel].abscissa.start[measurement_table + 1])
				{
					logmessage(MSG_ERROR, "%s: Error in measurement table %u of %u: Tables overlap\n", file->channel_data[channel].name, measurement_table, file->channel_data[channel].abscissa.number_of_measurement_tables);
					return(false);
				}
			}
		}
	}

#ifdef _DEBUG
	unsigned int theta;
	unsigned int calculated_theta;
	unsigned int calculated_crank_angle;
	float weighting;
	unsigned int crank_angle;

	if (result == true)
	{
		//start_theta = file->channel_data[channel].abscissa.start[0];
		
		float offset = (float)file->engine.number_of_strokes * (float)MINIMUM_THETA_ANGLE_MULT;

		for (theta = 0; theta < maximum_theta; theta++)
		{
			crank_angle = file->channel_data[channel].abscissa.theta_to_ca[theta];

			if (crank_angle >= file->channel_data[channel].samples_per_cycle)
			{
				logmessage(MSG_ERROR, "%s: Calculated crank angle is greater than samples per cycle\ntheta: %u crank_angle: %u\n", file->channel_data[channel].name, theta, crank_angle);
				return(false);
			}

			calculated_theta = DegreesToTheta(file, ((float)theta) / MinimumResolution(file) - offset);

			if (calculated_theta != theta)
			{
				logmessage(MSG_ERROR, "%s: Error in DegreesToTheta calculation (theta = %u)\n", file->channel_data[channel].name, theta);
				return(false);
			}

			weighting = file->channel_data[channel].abscissa.theta_weighting[theta];

			if ((weighting < 0.0f) || (weighting > 1.0f))
			{
				logmessage(MSG_ERROR, "%s: Theta weighting outside of expected range\n", file->channel_data[channel].name);
				return(false);
			}

			weighting = file->channel_data[channel].abscissa.theta_weighting_inv[theta];

			if ((weighting < 0.0f) || (weighting > 1.0f))
			{
				logmessage(MSG_ERROR, "%s: Theta weighting outside of expected range\n", file->channel_data[channel].name);
				return(false);
			}

			weighting = file->channel_data[channel].abscissa.theta_weighting[theta] + file->channel_data[channel].abscissa.theta_weighting_inv[theta] - 1.0f;

			if ((weighting < -FLT_EPSILON) || (weighting > FLT_EPSILON))
			{
				logmessage(MSG_ERROR, "%s: Theta weightings do not sum to one\n", file->channel_data[channel].name);
				return(false);
			}

			if (theta > 0)
			{
				if (file->channel_data[channel].abscissa.theta_to_ca[theta - 1] > file->channel_data[channel].abscissa.theta_to_ca[theta])
				{
					logmessage(MSG_ERROR, "%s: Theta to crank angle lookup is not monotonically increasing\n", file->channel_data[channel].name);
					return(false);
				}
			}
		}
	}

	if (result == true)
	{
		if (maximum_theta > 360 * file->engine.number_of_strokes / 2 * 40 * 2)	/* 0.025 deg resolution (40) , *2 as start of cycle maybe not -360/-180 degrees */
		{
			logmessage(MSG_ERROR, "%s: Maximum theta greater than maximum possible\n", file->channel_data[channel].name);
			//return(false);
		}
	}

	if (result == true)
	{
		for (crank_angle = 0; crank_angle < file->channel_data[channel].samples_per_cycle; crank_angle++)
		{
			theta = ReturnTheta(file, crank_angle, channel);

			calculated_crank_angle = ReturnCrankAngle(file, theta, channel);

			if (crank_angle != calculated_crank_angle)
			{
				result = false;
			}
		}

		if (result == false)
		{
			logmessage(MSG_ERROR, "%s: Error in reverse crank_angle lookup (crank_angle = %u)\n", file->channel_data[channel].name, crank_angle);
		}
	}

	if (result == true)
	{
		for (crank_angle = 0; crank_angle < file->channel_data[channel].samples_per_cycle; crank_angle++)
		{
			theta = file->channel_data[channel].abscissa.ca_to_theta[crank_angle];

			if (theta >= maximum_theta)
			{
				result = false;
			}
		}

		if (result == false)
		{
			logmessage(MSG_ERROR, "%s: Theta greater than maximum possible\n", file->channel_data[channel].name);
			result = true;
		}
	}
#endif

	if (result == true)
	{
		Calculate_Geometry(&file->engine);

		Calculate_Volume(file, channel, &file->channel_data[channel].abscissa.volume);
	}

	return(result);
}

void Calculate_Volume(const FileData* file, const unsigned int channel, float** volume)
{
	double pin_offset;
	double crank_throw;
	double pin_offset_angle;
	float angle;
	double angle_rad;
	unsigned int theta;
	unsigned int maximum_theta;
	unsigned int volume_calculation_type;

	if (file == NULL)
	{
		logmessage(MSG_ERROR, "Cannot calculate cylinder volume for channel %u\n", channel);
		return;
	}

	if (volume == NULL)
	{
		logmessage(MSG_ERROR, "Nowhere to store cylinder volume for channel %u\n", channel);
		return;
	}

	if ((file->channel_data == NULL) || (channel >= file->number_of_channels))
	{
		logmessage(MSG_ERROR, "Cannot calculate cylinder volume for channel %u\n", channel);
		return;
	}

	if (file->channel_data[channel].abscissa.type != ABSCISSA_CRANKANGLE)
	{
		logmessage(MSG_ERROR, "Cannot calculate cylinder volume for channel %u\n", channel);
		return;
	}

	if (file->channel_data[channel].abscissa.ca_to_theta == NULL)
	{
		logmessage(MSG_ERROR, "Cannot calculate cylinder volume for channel %u\n", channel);
		return;
	}

	if (plugin_calculate_volume(file, channel, volume) == true)
	{
		return;
	}

	maximum_theta = MaximumTheta(file, channel);

	if (*volume != NULL)
	{
		free(*volume);
	}
	
	*volume = (float*)malloc(maximum_theta * sizeof(float));
	if (*volume == NULL)
	{
		logmessage(FATAL,"Memory could not be allocated\n");
		exit(EXIT_FAILURE);
	}

	volume_calculation_type = file->engine.volume_calculation_type;

	if (volume_calculation_type == VOLUME_USER)
	{
		if ((file->engine.user.number_of_data_points == 0) ||
			(file->engine.user.crank_angle == NULL) ||
			(file->engine.user.volume == NULL))
		{
			volume_calculation_type = VOLUME_CATOOL;
		}
	}
	
	switch (volume_calculation_type)
	{
	case VOLUME_KISTLER_KB1:
	case VOLUME_KISTLER_KB2:
	case VOLUME_KISTLER_FKFS:
#ifdef __CATOOLRT__
	{
		KistlerVolume(file, channel, *volume, volume_calculation_type);

		break;
	}
#endif
	default:
	case VOLUME_CATOOL:
	{
		pin_offset = (double)Return_PinOffset(file, channel);
		crank_throw = (double)file->engine.stroke / 2.0;

		/* Account for the fact that piston TDC and crank TDC are not at the same point with pin offset */
		pin_offset_angle = asin(pin_offset / (crank_throw + (double)file->engine.conrod_length));

		double swept_area = M_PI / 4.0 * (double)file->engine.bore * (double)file->engine.bore;
		double swept_volume = swept_area * (double)file->engine.stroke;
		double clearance_volume = swept_volume / ((double)file->engine.compression_ratio - 1.0);

		for (theta = 0; theta < maximum_theta; theta++)
		{
			angle = ThetaToDegrees(file, theta) + file->engine.tdc_error;
			angle_rad = (double)angle * (2.0 * M_PI / 360.0) + pin_offset_angle;

			(*volume)[theta] = (float)
				(
					clearance_volume +
					swept_area *
					(
						crank_throw -
						crank_throw * cos(angle_rad) +
						(double)file->engine.conrod_length -
						sqrt((double)file->engine.conrod_length * (double)file->engine.conrod_length - (crank_throw * sin(angle_rad) - pin_offset) * (crank_throw * sin(angle_rad) - pin_offset))
					)
				);
		}

		break;
	}
	case VOLUME_USER:
	{
		for (theta = 0; theta < maximum_theta; theta++)
		{
			angle = ThetaToDegrees(file, theta) + file->engine.tdc_error;

			(*volume)[theta] = lookup_2d(file->engine.user.crank_angle, file->engine.user.volume, file->engine.user.number_of_data_points, angle);
		}

		break;
	}
	}
}

#ifdef _CATOOL_UNICODE_
bool Import_Custom_Volume(Engine* engine, const wchar_t* filename, const unsigned int max_theta)
#else
bool Import_Custom_Volume(Engine* engine, const char* filename, const unsigned int max_theta)
#endif
{
	char buffer[1024];
	float previous_crank_angle = -FLT_MAX;
	float min_crank_angle = FLT_MAX;
	float max_crank_angle = -FLT_MAX;
	bool done = false;
	bool failed = false;
	float crank_angle;
	float volume;
	FILE* input_file = NULL;

	if ((engine == NULL) || (filename == NULL))
	{
		return(false);
	}
	
#ifdef _CATOOL_UNICODE_
	input_file = _wfopen(filename, L"r");
#else
	input_file = fopen(filename, "r");
#endif

	if (input_file == NULL)
	{
		logmessage(MSG_ERROR, "Could not open custom volume file\n");

		return(false);
	}

	if (engine->user.crank_angle != NULL)
	{
		free(engine->user.crank_angle);
		engine->user.crank_angle = NULL;
	}

	if (engine->user.volume != NULL)
	{
		free(engine->user.volume);
		engine->user.volume = NULL;
	}

	engine->user.number_of_data_points = 0;

	engine->user.crank_angle = (float*)malloc(max_theta * sizeof(float));

	if (engine->user.crank_angle == NULL)
	{
		fclose(input_file);

		logmessage(FATAL, "Out of memory\n");
	}

	engine->user.volume = (float*)malloc(max_theta * sizeof(float));

	if (engine->user.volume == NULL)
	{
		fclose(input_file);

		logmessage(FATAL, "Out of memory\n");
	}

	while ((fgets(buffer, 1024, input_file) != NULL) && (engine->user.number_of_data_points < max_theta) && (done == false))
	{
		if (sscanf(buffer, "%f,%f", &crank_angle, &volume) == 2)
		{
			if (crank_angle > previous_crank_angle)
			{
				engine->user.crank_angle[engine->user.number_of_data_points] = crank_angle;
				engine->user.volume[engine->user.number_of_data_points] = volume * 1000.0f;

				engine->user.number_of_data_points++;

				previous_crank_angle = crank_angle;

				if (crank_angle < min_crank_angle)
				{
					min_crank_angle = crank_angle;
				}

				if (crank_angle > max_crank_angle)
				{
					max_crank_angle = crank_angle;
				}
			}
			else
			{
				done = true;
				failed = true;
			}
		}
		else
		{
			done = true;
		}
	}

	fclose(input_file);

	if (engine->user.number_of_data_points == 0)
	{
		logmessage(MSG_ERROR, "An error occured whilst importing volume data from file - no data points loaded\n");

		return(false);
	}
	else if (failed == true)
	{
		logmessage(MSG_ERROR, "An error occured whilst importing volume data from file - crank angle points are not monotonically increasing\n");

		return(false);
	}
	else if ((max_crank_angle - min_crank_angle) < FLT_EPSILON)
	{
		logmessage(MSG_ERROR, "An error occured whilst importing volume data from file - crank angle values do not increase\n");

		return(false);
	}
	else if ((max_crank_angle < (float)engine->number_of_strokes * 90.0f) || (min_crank_angle > (float)engine->number_of_strokes * -90.0f))
	{
		logmessage(MSG_ERROR, "An error occured whilst importing volume data from file - crank angle points do not cover cycle range (-%d to %d deg)", engine->number_of_strokes * 90, engine->number_of_strokes * 90);
	
		return(false);
	}
	else
	{
		/* Imported OK */
	}

	engine->volume_calculation_type = VOLUME_USER;

	return(true);
}

void Initialise_Engine(Engine* engine)
{
	unsigned int cylinder;

	if (engine == NULL)
	{
		return;
	}

	strncpy(engine->name, "Engine", SIZEOF_ENGINE_NAME);

	engine->volume_calculation_type = VOLUME_CATOOL;
	engine->bore = 0.0f;
	engine->stroke = 0.0f;
	engine->conrod_length = 0.0f;
	engine->compression_ratio = 0.0f;
	for (cylinder = 0; cylinder < MAX_NUMBER_OF_CYLINDERS; cylinder++)
	{
		engine->pin_offset[cylinder] = 0.0f;
		engine->tdc_offset[cylinder] = 0.0f;
	}
	engine->number_of_strokes = 4;
	engine->number_of_cylinders = 1;
	engine->type = ENGINE_SI;
	engine->name[0] = 0x00;
	engine->ivc_angle = -140.0f;
	engine->evo_angle = 140.0f;
	engine->ivo_angle = 350.0f;
	engine->evc_angle = -355.0f;
	engine->tdc_error = 0.0f;
	engine->firing_order = 0;
	engine->piston_factor = 1.2f;

	engine->swept_area = 0.0f;
	engine->swept_volume = 0.0f;
	engine->clearance_volume = 0.0f;

	engine->user.number_of_data_points = 0;
	engine->user.crank_angle = NULL;
	engine->user.volume = NULL;
}

FileData* Initialise_File()
{
	FileData* file = (FileData*)malloc(sizeof(FileData));
	if (file == NULL)
	{
		logmessage(FATAL,"Memory could not be allocated\n");
	}
	
	memset(file, 0, sizeof(FileData));

	file->file_type = FILE_NONE;
	file->time_resolution = DEFAULT_TIME_RESOLUTION;

	Initialise_Engine(&file->engine);

	Change_Minimum_Resolution(file,DEFAULT_MIN_RES_PPR);

	file->threads = 2;

	return(file);
}

void unload_ifile(FileData *file)
{
	/* Delete dynamically allocated memory */

	unsigned int dgb;
	unsigned int number_of_data_groups;

	if ((file->ptr_APB != NULL) && (file->file_type == FILE_AVLIFILE))
	{
		logmessage(DEBUG, "Unloading I-File data\n");

		number_of_data_groups = (unsigned int)file->ptr_APB->grpanz;

		free(file->ptr_APB);
		file->ptr_APB = NULL;

		if (file->ptr_DGB != NULL)
		{
			for (dgb = 0; dgb<number_of_data_groups; dgb++)
			{
				if (file->ptr_DGB[dgb].diradr != NULL)
				{
					free(file->ptr_DGB[dgb].diradr);
					file->ptr_DGB[dgb].diradr = NULL;
				}
				if (file->ptr_DGB[dgb].mpladr != NULL)
				{
					free(file->ptr_DGB[dgb].mpladr);
					file->ptr_DGB[dgb].mpladr = NULL;
				}
				if (file->ptr_DGB[dgb].aztadr != NULL)
				{
					free(file->ptr_DGB[dgb].aztadr);
					file->ptr_DGB[dgb].aztadr = NULL;
				}
				if (file->ptr_DGB[dgb].rztadr != NULL)
				{
					free(file->ptr_DGB[dgb].rztadr);
					file->ptr_DGB[dgb].rztadr = NULL;
				}
			}

			free(file->ptr_DGB);
			file->ptr_DGB = NULL;

		}

		file->preload = false;

		logmessage(NOTICE, "I-File data unloaded\n");
	}
}

bool Delete_File(FileData *file)
{
	unsigned int channel;

	if (file == NULL)
	{
		return(false);
	}

	logmessage(DEBUG,"Cleaning up...\n");

	if (file->channel_data != NULL)
	{
		for (channel=0;channel<file->number_of_channels;channel++)
		{
			//if (file->channel_data[channel].loaded == true)
			{
				Delete_Channel(file,channel);
			}
		}
		
		file->number_of_channels = 0;
	
		free(file->channel_data);
		file->channel_data = NULL;
	}

	if (file->parameter_file != NULL)
	{
		free(file->parameter_file);
		file->parameter_file = NULL;
	}

	if (file->file_type == FILE_AVLIFILE)
	{
		unload_ifile(file);
	}
		
	file->preload = false;
	file->ca_dgb = 0;
	file->comment[0] = 0x00;
	file->filename[0] = 0x00;
	file->parameter_filename[0] = 0x00;
	file->parameter_filesize = 0;

	file->ptr_APB = NULL;
	file->ptr_DGB = NULL;

	file->swap_endian = false;
	file->sync_checked = false;
	if (file->theta_to_deg != NULL)
	{
		free(file->theta_to_deg);
		file->theta_to_deg = NULL;
	}

	Delete_File_Parameters(file);

	Delete_Engine(&file->engine);

	logmessage(DEBUG,"done\n");
			
	return(true);
}

bool Delete_File_Parameters(FileData* file)
{
	Parameter* parameter = NULL;

	if (file == NULL)
	{
		return(false);
	}

	while (file->parameters != NULL)
	{
		parameter = file->parameters;

		if (parameter->value != NULL)
		{
			free(parameter->value);
			parameter->value = NULL;
		}

		file->parameters = parameter->next;

		free(parameter);
	}

	return(true);
}

bool Release_File(FileData** file)
{
	if (file == NULL)
	{
		return(false);
	}

	if (*file == NULL)
	{
		return(false);
	}

   	free(*file);
	*file = NULL;

	return(true);
}

void Delete_Channel_Data(ChannelData* channel_data)
{
	if (channel_data == NULL)
	{
		return;
	}

	channel_data->abscissa.number_of_measurement_tables = 0;
	channel_data->loaded = false;
	channel_data->isoffset = false;
	channel_data->filter_config.filtered = false;

	if (channel_data->data != NULL)
	{
		free(channel_data->data);
		channel_data->data = NULL;
	}

	if (channel_data->data_d != NULL)
	{
		free(channel_data->data_d);
		channel_data->data_d = NULL;
	}

	if (channel_data->duration != NULL)
	{
		free(channel_data->duration);
		channel_data->duration = NULL;
	}

	Empty_Abscissa(&channel_data->abscissa);
}

void Initialise_ChannelData(ChannelData* channel_data)
{
	if (channel_data == NULL)
	{
		return;
	}

	memset(channel_data, 0, sizeof(ChannelData));

	channel_data->cylinder = 1;
	channel_data->conversion_factor = 1.0f;

	channel_data->offset_config.type = OFFSET_NONE;
	channel_data->offset_config.channel_name[0] = 0x00;
	channel_data->offset_config.window_size = 5;
	channel_data->offset_config.fixed_value = 0.0f;
	channel_data->offset_config.start_window = -100.0f;
	channel_data->offset_config.finish_window = -65.0f;
	channel_data->offset_config.polytropic_index = 1.32f;
	channel_data->offset_config.truncate = false;

	channel_data->soc_config.type = SOC_FIXED;
	channel_data->soc_config.channel_name[0] = 0x00;
	channel_data->soc_config.fixed_value = -20.0f;
	channel_data->soc_config.start_window = -60.0f;
	channel_data->soc_config.finish_window = 60.0f;
	channel_data->soc_config.aligned = true;
	channel_data->soc_config.invert = false;

	channel_data->filter_config.type = FILTER_NONE;
	channel_data->filter_config.filtered = false;
	channel_data->filter_config.upper_frequency = 500.0f;
	channel_data->filter_config.lower_frequency = 500.0f;

	channel_data->digital_config.type = DIGITIZE_AUTO;
	channel_data->digital_config.invert = false;
	channel_data->digital_config.latch_high = 2.0f;
	channel_data->digital_config.latch_low = 1.0f;
	channel_data->digital_config.filter = 1.0f;

	channel_data->cam_config.edge = EDGE_BOTH;
	channel_data->cam_config.invert = false;
	channel_data->cam_config.offset = 0.0f;
	channel_data->cam_config.reference_angle = 0.0f;
	channel_data->cam_config.type = CAMTYPE_NONE;
}

void Initialise_AnalysisChannel(AnalysisChannel* analysis)
{
	if (analysis == NULL)
	{
		return;
	}

	memset(analysis, 0, sizeof(AnalysisChannel));
}

bool Add_Channel(FileData* file, Analysis* analysis, const char* channel_name, const bool force)
{
	ChannelData* new_channel_data = NULL;
	AnalysisChannel* new_analysis_channel = NULL;
	unsigned int channel;

	if ((file == NULL) || (channel_name == NULL))
	{
		return(false);
	}

	if ((force == false) && (channel_name_to_number(file, channel_name, ABSCISSA_UNKNOWN, NULL) == true))
	{
		logmessage(NOTICE, "Cannot add channel '%s' as a channel already exists with that name\n", channel_name);

		return(false);
	}

	logmessage(NOTICE, "Adding channel %s...",channel_name);

	new_channel_data = (ChannelData*)malloc((file->number_of_channels + 1) * sizeof(ChannelData));

	if (new_channel_data == NULL)
	{
		logmessage(FATAL, "Memory could not be allocated\n");
	}

	for (channel = 0; channel < file->number_of_channels; channel++)
	{
		Move_ChannelData(&file->channel_data[channel], &new_channel_data[channel]);
	}

	free(file->channel_data);

	file->channel_data = new_channel_data;

	file->number_of_channels += 1;

	Initialise_ChannelData(&file->channel_data[channel]);

	strncpy(file->channel_data[channel].name, channel_name, SIZEOF_CHANNEL_NAME);
	file->channel_data[channel].name[SIZEOF_CHANNEL_NAME - 1] = 0x00;

	file->channel_data[channel].abscissa.type = ABSCISSA_CRANKANGLE;

	if ((analysis != NULL) && (analysis->channel != NULL))
	{
		new_analysis_channel = (AnalysisChannel*)malloc((file->number_of_channels + 1) * sizeof(AnalysisChannel));

		if (new_analysis_channel == NULL)
		{
			logmessage(FATAL, "Memory could not be allocated\n");
		}

		for (channel = 0; channel < analysis->number_of_channels; channel++)
		{
			new_analysis_channel[channel].analysis_rqst = (AnalysisRqst*)malloc(get_number_of_analysis_channels() * sizeof(AnalysisRqst));

			if (new_analysis_channel[channel].analysis_rqst == NULL)
			{
				logmessage(FATAL, "Memory could not be allocated\n");
			}

			Move_Analysis_Channel(&analysis->channel[channel], &new_analysis_channel[channel]);
		}

		free(analysis->channel);

		analysis->channel = new_analysis_channel;

		Initialise_AnalysisChannel(&analysis->channel[channel]);

		analysis->channel[channel].channel = channel;

		analysis->number_of_channels += 1;
	}

	logmessage(NOTICE, "done\n");

	return(true);
}

bool Sample_Channel(FileData* file, Analysis* analysis, const unsigned int channel, const unsigned int sample_channel, const bool align)
{
	unsigned int sizeof_data;
	unsigned int cycle;
	unsigned int crank_angle;
	float angle;
	float value;
	float offset;
	unsigned int data_cycle;

	if (file == NULL)
	{
		return(false);
	}

	if ((channel >= file->number_of_channels) || (sample_channel >= file->number_of_channels) || (file->number_of_cycles == 0))
	{
		return(false);
	}

	if ((file->channel_data[channel].abscissa.type != ABSCISSA_CRANKANGLE) || (file->channel_data[sample_channel].abscissa.type != ABSCISSA_CRANKANGLE))
	{
		return(false);
	}

	if ((file->channel_data[channel].samples_per_cycle == 0) || (file->channel_data[sample_channel].samples_per_cycle == 0))
	{
		return(false);
	}

	if (file->channel_data[channel].data != NULL)
	{
		free(file->channel_data[channel].data);
	}

	sizeof_data = file->channel_data[channel].samples_per_cycle * file->number_of_cycles;

	file->channel_data[channel].data = (float*)malloc(sizeof_data * sizeof(float));

	if (file->channel_data[channel].data == NULL)
	{
		return(false);
	}

	if (align == true)
	{
		offset = 0.0f;
	}
	else
	{
		offset = file->channel_data[channel].tdc_offset - file->channel_data[sample_channel].tdc_offset;
	}

	for (cycle = 0; cycle < file->number_of_cycles; cycle++)
	{
		for (crank_angle = 0; crank_angle < file->channel_data[channel].samples_per_cycle; crank_angle++)
		{
			angle = CrankAngleToDegrees(file,crank_angle,channel) + offset;

			data_cycle = cycle;

			if (angle > (float)file->engine.number_of_strokes*90.0f - 0.0001f)
			{
				angle -= file->engine.number_of_strokes*180.0f;
				data_cycle += 1;
			}

			if (data_cycle < file->number_of_cycles)
			{
				value = ReturnThetaData(file, data_cycle, DegreesToTheta(file, angle), sample_channel);
			}
			else
			{
				value = 0.0f;
			}

			SetCAData(file, cycle, crank_angle, channel, value);
		}
	}

	strncpy(file->channel_data[channel].units, file->channel_data[sample_channel].units, SIZEOF_UNITS);
	file->channel_data[channel].loaded = true;

	return(true);
}

void Delete_Channel(FileData* file,const unsigned int channel)
{
	if (file == NULL)
	{
		return;
	}

	if ((file->channel_data == NULL) || (file->number_of_channels == 0) || (channel >= file->number_of_channels))
	{
		return;
	}

	Delete_Channel_Data(&file->channel_data[channel]);
}

bool Copy_Channel_Data(const FileData* from, FileData* to, const unsigned int from_channel, const unsigned int to_channel)
{
	bool same;
	unsigned int mt;
	unsigned int data_size;
	unsigned int new_data_size;
	unsigned int cycle;
	unsigned int crank_angle;

	if ((to == NULL) || (from == NULL))
	{
		return(false);
	}

	if ((from_channel >= from->number_of_channels) || (to_channel >= to->number_of_channels))
	{
		return(false);
	}

	if (from->channel_data[from_channel].abscissa.type != to->channel_data[to_channel].abscissa.type)
	{
		return(false);
	}

	if (from->channel_data[from_channel].abscissa.number_of_measurement_tables == to->channel_data[to_channel].abscissa.number_of_measurement_tables)
	{
		same = true;

		if (from->channel_data[from_channel].abscissa.type != ABSCISSA_CYCLE)
		{
			for (mt = 0; mt < from->channel_data[from_channel].abscissa.number_of_measurement_tables; mt++)
			{
				if (from->channel_data[from_channel].abscissa.resolution[mt] != to->channel_data[to_channel].abscissa.resolution[mt])
				{
					same = false;
				}
				else if (from->channel_data[from_channel].abscissa.start[mt] != to->channel_data[to_channel].abscissa.start[mt])
				{
					same = false;
				}
				else if (from->channel_data[from_channel].abscissa.number_of_samples[mt] != to->channel_data[to_channel].abscissa.number_of_samples[mt])
				{
					same = false;
				}
				else
				{
					/* Do Nothing */
				}
			}

			if (same == true)
			{
				logmessage(DEBUG, "Channel abscissas are identical\n");
			}
			else
			{
				logmessage(DEBUG, "Channel abscissas are different\n");
			}
		}
	}
	else
	{
		same = false;
	}

	if (same == true)
	{
		if ((from->channel_data[from_channel].abscissa.type == ABSCISSA_CRANKANGLE) ||
			(from->channel_data[from_channel].abscissa.type == ABSCISSA_FREQUENCY))
		{
			data_size = from->number_of_cycles * from->channel_data[from_channel].samples_per_cycle;
			new_data_size = to->number_of_cycles * to->channel_data[to_channel].samples_per_cycle;
		}
		else if (from->channel_data[from_channel].abscissa.type == ABSCISSA_CYCLE)
		{
			/* IFile with samples_per_cycle = 100 and number_of_cycles = 101 (20190124_weilijiang_volllast_1000.dat) */

			data_size = from->channel_data[from_channel].samples_per_cycle; // from->number_of_cycles;
			new_data_size = to->channel_data[to_channel].samples_per_cycle; // to->number_of_cycles;
		}
		else if (from->channel_data[from_channel].abscissa.type == ABSCISSA_TIME)
		{
			data_size = from->channel_data[from_channel].samples_per_cycle;
			new_data_size = to->channel_data[to_channel].samples_per_cycle;
		}
		else
		{
			data_size = 0;
			new_data_size = 0;
		}

		if (new_data_size > 0)
		{
			logmessage(DEBUG, "Fast copying data\n");

			if (new_data_size > data_size)
			{
				memcpy((void*)to->channel_data[to_channel].data, (void*)from->channel_data[from_channel].data, data_size * sizeof(float));
				memset((void*)&to->channel_data[to_channel].data[data_size], 0, (new_data_size - data_size) * sizeof(float));
			}
			else
			{
				memcpy((void*)to->channel_data[to_channel].data, (void*)from->channel_data[from_channel].data, new_data_size * sizeof(float));
			}
		}
	}
	else if ((from->channel_data[from_channel].abscissa.type == ABSCISSA_CRANKANGLE) || (from->channel_data[from_channel].abscissa.type == ABSCISSA_FREQUENCY))
	{
		logmessage(DEBUG, "Slow copying crank angle/frequency data\n");

		if (to->number_of_cycles <= from->number_of_cycles)
		{
			for (cycle = 0; cycle < to->number_of_cycles; cycle++)
			{
				for (crank_angle = 0; crank_angle < to->channel_data[to_channel].samples_per_cycle; crank_angle++)
				{
					SetCAData(to, cycle, crank_angle, to_channel, ReturnThetaData(from, cycle, DegreesToTheta(from, CrankAngleToDegrees(to, crank_angle, to_channel)), from_channel));
				}
			}
		}
		else
		{
			for (cycle = 0; cycle < from->number_of_cycles; cycle++)
			{
				for (crank_angle = 0; crank_angle < to->channel_data[to_channel].samples_per_cycle; crank_angle++)
				{
					SetCAData(to, cycle, crank_angle, to_channel, ReturnThetaData(from, cycle, DegreesToTheta(from, CrankAngleToDegrees(to, crank_angle, to_channel)), from_channel));
				}
			}

			for (cycle = from->number_of_cycles; cycle < to->number_of_cycles; cycle++)
			{
				for (crank_angle = 0; crank_angle < to->channel_data[to_channel].samples_per_cycle; crank_angle++)
				{
					SetCAData(to, cycle, crank_angle, to_channel, 0.0f);
				}
			}
		}
	}
	else
	{
		logmessage(DEBUG, "Slow copying non-crank angle/frequency data\n");

		for (crank_angle = 0; crank_angle < to->channel_data[to_channel].samples_per_cycle; crank_angle++)
		{
			if (crank_angle < from->channel_data[from_channel].samples_per_cycle)
			{
				to->channel_data[to_channel].data[crank_angle] = from->channel_data[from_channel].data[crank_angle];
			}
			else
			{
				to->channel_data[to_channel].data[crank_angle] = 0.0f;
			}
		}
	}

	if ((to->channel_data[to_channel].duration != NULL) && (from->channel_data[from_channel].duration != NULL))
	{
		for (cycle = 0; cycle < to->number_of_cycles; cycle++)
		{
			if (cycle < from->number_of_cycles)
			{
				to->channel_data[to_channel].duration[cycle] = from->channel_data[from_channel].duration[cycle];
			}
			else
			{
				to->channel_data[to_channel].duration[cycle] = 0.0f;
			}
		}
	}

	return(true);
}

bool Copy_Channel(FileData* file, const unsigned int from_channel, const unsigned int to_channel)
{
	return(Copy_Channel_Between(file, file, from_channel, to_channel));
}

bool Copy_Channel_Between(const FileData* src, FileData* dst, const unsigned int from_channel, const unsigned int to_channel)
{
	unsigned int cycle;
	size_t data_size;
	char name[SIZEOF_CHANNEL_NAME];

	if ((src == NULL) || (dst == NULL))
	{
		return(false);
	}

	if ((from_channel >= src->number_of_channels) || (to_channel >= dst->number_of_channels))
	{
		return(false);
	}

	if ((src->channel_data == NULL) || (dst->channel_data == NULL))
	{
		return(false);
	}

	if (src->channel_data[from_channel].loaded == false)
	{
		return(false);
	}

	logmessage(DEBUG, "Copying channel %s to %s\n", src->channel_data[from_channel].name, dst->channel_data[to_channel].name);

	/* Save channel name as it will be overwritten by Move_ChannelData() */

	strncpy(name, src->channel_data[from_channel].name, SIZEOF_CHANNEL_NAME);
	name[SIZEOF_CHANNEL_NAME - 1] = 0x00;
	dst->channel_data[to_channel].short_name[0] = 0x00;
	dst->channel_data[to_channel].matlab_name[0] = 0x00;

	Move_ChannelData(&src->channel_data[from_channel], &dst->channel_data[to_channel]);

	/* Need to copy new name back again */

	strncpy(dst->channel_data[to_channel].name, name, SIZEOF_CHANNEL_NAME);
	dst->channel_data[to_channel].name[SIZEOF_CHANNEL_NAME - 1] = 0x00;

	/* Move_ChannelData does not create new copies of any data, pointers are copied so need to be recreated */

	/* Abscissa */

	/* Need to call Zero_Abscissa() before Copy_Abscissa() as it calls Empty_Abscissa() on the destination */

	Zero_Abscissa(&dst->channel_data[to_channel].abscissa);

	dst->channel_data[to_channel].samples_per_cycle = Copy_Abscissa(&src->channel_data[from_channel].abscissa, &dst->channel_data[to_channel].abscissa);

	if (dst->channel_data[to_channel].abscissa.type == ABSCISSA_CRANKANGLE)
	{
		if (calculate_theta_lookups(dst, to_channel) == false)
		{
			logmessage(WARNING, "Copy_Channel_Between: Could not calculate theta lookups\n");
		}
	}

	/* Duration */

	if (src->channel_data[from_channel].duration != NULL)
	{
		dst->channel_data[to_channel].duration = (float*)malloc(dst->number_of_cycles * sizeof(float));

		if (dst->channel_data[to_channel].duration == NULL)
		{
			logmessage(WARNING, "Memory could not be allocated\n");

			return(false);
		}

		for (cycle = 0; cycle < dst->number_of_cycles; cycle++)
		{
			dst->channel_data[to_channel].duration[cycle] = src->channel_data[from_channel].duration[cycle];
		}
	}

	/* Data */

	if ((dst->channel_data[to_channel].abscissa.type == ABSCISSA_CRANKANGLE) ||
		(dst->channel_data[to_channel].abscissa.type == ABSCISSA_FREQUENCY))
	{
		data_size = dst->number_of_cycles * dst->channel_data[to_channel].samples_per_cycle;
	}
	else
	{
		data_size = dst->channel_data[to_channel].samples_per_cycle;
	}

	dst->channel_data[to_channel].data = (float*)malloc(data_size * sizeof(float));

	if (dst->channel_data[to_channel].data == NULL)
	{
		logmessage(WARNING, "Memory could not be allocated\n");

		return(false);
	}

	return(Copy_Channel_Data(src, dst, from_channel, to_channel));
}

void Rename_Channel(FileData* file, const unsigned int channel, const char* name)
{
	unsigned int chan;

	if (file == NULL)
	{
		return;
	}

	if (channel >= file->number_of_channels)
	{
		return;
	}

	if (name == NULL)
	{
		return;
	}

	for (chan = 0; chan < file->number_of_channels; chan++)
	{
		if (strncmp(file->channel_data[chan].offset_config.channel_name, file->channel_data[channel].name, SIZEOF_CHANNEL_NAME) == 0)
		{
			strncpy(file->channel_data[chan].offset_config.channel_name, name, SIZEOF_CHANNEL_NAME);
			file->channel_data[chan].offset_config.channel_name[SIZEOF_CHANNEL_NAME - 1] = 0x00;
		}

		if (strncmp(file->channel_data[chan].soc_config.channel_name, file->channel_data[channel].name, SIZEOF_CHANNEL_NAME) == 0)
		{
			strncpy(file->channel_data[chan].soc_config.channel_name, name, SIZEOF_CHANNEL_NAME);
			file->channel_data[chan].soc_config.channel_name[SIZEOF_CHANNEL_NAME - 1] = 0x00;
		}
	}

	strncpy(file->channel_data[channel].name, name, SIZEOF_CHANNEL_NAME);
	file->channel_data[channel].name[SIZEOF_CHANNEL_NAME - 1] = 0x00;
	file->channel_data[channel].short_name[0] = 0x00;
	file->channel_data[channel].matlab_name[0] = 0x00;
}

bool Build_Channels(FileData* file)
{
	unsigned int channel;

	if (file == NULL)
	{
		return(false);
	}

	if (file->number_of_channels == 0)
	{
		return(false);
	}

	for (channel = 0; channel < file->number_of_channels; channel++)
	{
		Delete_Channel(file, channel);
	}

	if (file->channel_data != NULL)
	{
		free(file->channel_data);
	}

	file->channel_data = (ChannelData*)malloc(file->number_of_channels * sizeof(ChannelData));

	if (file->channel_data == NULL)
	{
		return(false);
	}

	for (channel = 0; channel < file->number_of_channels; channel++)
	{
		Initialise_ChannelData(&file->channel_data[channel]);

		file->channel_data[channel].conversion_factor = 1.0f;
		file->channel_data[channel].filter_config.upper_frequency = 500.0f;
		file->channel_data[channel].filter_config.lower_frequency = 500.0f;
		file->channel_data[channel].offset_config.window_size = 5;
		file->channel_data[channel].offset_config.finish_window = -65.0f;
		file->channel_data[channel].offset_config.polytropic_index = 1.32f;
		file->channel_data[channel].offset_config.start_window = -100.0f;
		file->channel_data[channel].offset = 0.0;
		file->channel_data[channel].slope = 1.0;
		file->channel_data[channel].soc_config.finish_window = 60.0f;
		file->channel_data[channel].soc_config.start_window = -60.0f;
	}

	return(true);
}

void Create_Measurement_Tables(FileData* file, const unsigned int channel, const unsigned int number_of_measurement_tables, const unsigned int start[5], const unsigned int resolution[5], const unsigned int number_of_samples[5])
{
	unsigned int mt;

	if (file == NULL)
	{
		return;
	}

	if (channel >= file->number_of_channels)
	{
		return;
	}

	if ((number_of_measurement_tables == 0) || (number_of_measurement_tables > 5))
	{
		return;
	}

	if (file->channel_data == NULL)
	{
		return;
	}

	if (file->channel_data[channel].abscissa.resolution != NULL)
	{
		free(file->channel_data[channel].abscissa.resolution);
	}

	if (file->channel_data[channel].abscissa.start != NULL)
	{
		free(file->channel_data[channel].abscissa.start);
	}

	if (file->channel_data[channel].abscissa.number_of_samples != NULL)
	{
		free(file->channel_data[channel].abscissa.number_of_samples);
	}

	file->channel_data[channel].abscissa.resolution = (unsigned int*)malloc(number_of_measurement_tables * sizeof(unsigned int));
	if (file->channel_data[channel].abscissa.resolution == NULL)
	{
		logmessage(FATAL, "Out of memory\n");
	}

	file->channel_data[channel].abscissa.start = (unsigned int*)malloc(number_of_measurement_tables * sizeof(unsigned int));
	if (file->channel_data[channel].abscissa.start == NULL)
	{
		logmessage(FATAL, "Out of memory\n");
	}

	file->channel_data[channel].abscissa.number_of_samples = (unsigned int*)malloc(number_of_measurement_tables * sizeof(unsigned int));
	if (file->channel_data[channel].abscissa.number_of_samples == NULL)
	{
		logmessage(FATAL, "Out of memory\n");
	}

	file->channel_data[channel].samples_per_cycle = 0;

	for (mt = 0; mt < number_of_measurement_tables; mt++)
	{
		file->channel_data[channel].abscissa.resolution[mt] = resolution[mt];
		file->channel_data[channel].abscissa.start[mt] = start[mt];
		file->channel_data[channel].abscissa.number_of_samples[mt] = number_of_samples[mt];

		file->channel_data[channel].samples_per_cycle += number_of_samples[mt];
	}

	file->channel_data[channel].abscissa.number_of_measurement_tables = number_of_measurement_tables;
}

void Move_ChannelData(const ChannelData* from, ChannelData* to)
{
	if ((to == NULL) || (from == NULL))
	{
		return;
	}

	to->abscissa.axis = from->abscissa.axis;
	to->abscissa.axis_type = from->abscissa.axis_type;
	to->abscissa.slope = from->abscissa.slope;
	to->abscissa.offset = from->abscissa.offset;

	to->abscissa.ca_to_deg = from->abscissa.ca_to_deg;
	to->abscissa.ca_to_theta = from->abscissa.ca_to_theta;
	to->abscissa.number_of_measurement_tables = from->abscissa.number_of_measurement_tables;
	to->abscissa.number_of_samples = from->abscissa.number_of_samples;
	to->abscissa.resolution = from->abscissa.resolution;
	to->abscissa.start = from->abscissa.start;
	to->abscissa.theta_to_ca = from->abscissa.theta_to_ca;
	to->abscissa.theta_weighting = from->abscissa.theta_weighting;
	to->abscissa.theta_weighting_inv = from->abscissa.theta_weighting_inv;
	to->abscissa.type = from->abscissa.type;
	strncpy(to->abscissa.units, from->abscissa.units, SIZEOF_UNITS);
	to->abscissa.volume = from->abscissa.volume;

	to->cylinder = from->cylinder;
	to->data = from->data;
	to->data_d = from->data_d;
	strncpy(to->description, from->description, SIZEOF_DESCRIPTION);
	to->duration = from->duration;
	to->file_flag = from->file_flag;

	to->isoffset = from->isoffset;
	to->loaded = from->loaded;
	strncpy(to->matlab_name, from->matlab_name, SIZEOF_CHANNEL_NAME);
	to->max = from->max;
	to->min = from->min;
	strncpy(to->name, from->name, SIZEOF_CHANNEL_NAME);
	to->offset = from->offset;
	to->samples_per_cycle = from->samples_per_cycle;
	strncpy(to->short_name, from->short_name, SIZEOF_SHORT_NAME);
	to->slope = from->slope;
	to->tdc_offset = from->tdc_offset;
	to->type = from->type;
	strncpy(to->units, from->units, SIZEOF_UNITS);
	to->conversion_factor = from->conversion_factor;

	Copy_Offset_Config(&from->offset_config, &to->offset_config);
	Copy_SOC_Config(&from->soc_config, &to->soc_config);
	Copy_Filter_Config(&from->filter_config, &to->filter_config);
	Copy_Digital_Config(&from->digital_config, &to->digital_config);
	Copy_Camshaft_Config(&from->cam_config, &to->cam_config);
}

void Move_Analysis_Channel(const AnalysisChannel* from, AnalysisChannel* to)
{
	if ((to == NULL) || (from == NULL))
	{
		return;
	}

	to->analysis_rqst = from->analysis_rqst;
	to->ca_statistics = from->ca_statistics;
	to->channel = from->channel;
	Copy_Analysis(&from->config, &to->config);
	to->cycle_classification = from->cycle_classification;
	to->number_of_cycles = from->number_of_cycles;

	to->raw_ca_statistics.max = from->raw_ca_statistics.max;
	to->raw_ca_statistics.mean = from->raw_ca_statistics.mean;
	to->raw_ca_statistics.min = from->raw_ca_statistics.min;
	to->raw_ca_statistics.stddev = from->raw_ca_statistics.stddev;
	to->raw_ca_statistics.sum = from->raw_ca_statistics.sum;
	to->raw_statistics.cov = from->raw_statistics.cov;
	to->raw_statistics.kurtosis = from->raw_statistics.kurtosis;
	to->raw_statistics.lnv = from->raw_statistics.lnv;
	to->raw_statistics.max = from->raw_statistics.max;
	to->raw_statistics.mean = from->raw_statistics.mean;
	to->raw_statistics.min = from->raw_statistics.min;
	to->raw_statistics.range = from->raw_statistics.range;
	to->raw_statistics.skewness = from->raw_statistics.skewness;
	to->raw_statistics.stddev = from->raw_statistics.stddev;
	to->raw_statistics.sum = from->raw_statistics.sum;

	to->results = from->results;
	to->statistics = from->statistics;
}

bool Remove_Channel(FileData* file, Analysis* analysis, const unsigned int channel_to_delete)
{
	unsigned int channel;
	ChannelData* new_channel_data = NULL;
	AnalysisChannel* new_analysis_channel = NULL;

	if (file == NULL)
	{
		return(false);
	}

	if ((channel_to_delete >= file->number_of_channels) || (file->number_of_channels == 0))
	{
		return(false);
	}

	logmessage(NOTICE, "Removing channel %s...", file->channel_data[channel_to_delete].name);

	Delete_Channel(file, channel_to_delete);

	new_channel_data = (ChannelData*)malloc((file->number_of_channels - 1) * sizeof(ChannelData));

	if (new_channel_data == NULL)
	{
		logmessage(FATAL, "Memory could not be allocated\n");
	}

	for (channel = 0; channel < channel_to_delete; channel++)
	{
		Move_ChannelData(&file->channel_data[channel], &new_channel_data[channel]);
	}

	for (channel = channel_to_delete; channel < file->number_of_channels - 1; channel++)
	{
		Move_ChannelData(&file->channel_data[channel + 1], &new_channel_data[channel]);
	}

	free(file->channel_data);

	file->channel_data = new_channel_data;

	if ((analysis != NULL) && (analysis->channel != NULL))
	{
		new_analysis_channel = (AnalysisChannel*)malloc((file->number_of_channels - 1) * sizeof(AnalysisChannel));

		if (new_analysis_channel == NULL)
		{
			logmessage(FATAL, "Memory could not be allocated\n");
		}

		Delete_Analysis_Channel(&analysis->channel[channel_to_delete]);

		for (channel = 0; channel < channel_to_delete; channel++)
		{
			new_analysis_channel[channel].analysis_rqst = (AnalysisRqst*)malloc(get_number_of_analysis_channels() * sizeof(AnalysisRqst));

			if (new_analysis_channel[channel].analysis_rqst == NULL)
			{
				logmessage(FATAL, "Memory could not be allocated\n");
			}

			Move_Analysis_Channel(&analysis->channel[channel], &new_analysis_channel[channel]);
		}

		for (channel = channel_to_delete; channel < file->number_of_channels - 1; channel++)
		{
			Move_Analysis_Channel(&analysis->channel[channel + 1], &new_analysis_channel[channel]);
			new_analysis_channel[channel].channel -= 1;
		}

		free(analysis->channel);

		analysis->channel = new_analysis_channel;

		analysis->number_of_channels -= 1;
	}

	file->number_of_channels -= 1;

	logmessage(NOTICE, "done\n");

	return(true);
}

bool Resample_Channel(FileData* file, const unsigned int channel, AbscissaData* new_abscissa, const bool interpolate)
{
	unsigned int samples_per_cycle = 0;
	unsigned int mt;
	float* data = NULL;
	unsigned int cycle;
	unsigned int crank_angle;
	unsigned int theta;
	unsigned int new_crank_angle;

	if ((file == NULL) || (new_abscissa == NULL))
	{
		return(false);
	}

	if ((channel >= file->number_of_channels) || (file->channel_data == NULL) || (file->number_of_cycles == 0))
	{
		return(false);
	}

	if ((file->channel_data[channel].data == NULL) || (file->channel_data[channel].abscissa.ca_to_theta == NULL) || (file->channel_data[channel].abscissa.type != ABSCISSA_CRANKANGLE) || (file->channel_data[channel].loaded == false))
	{
		return(false);
	}

	for (mt = 0; mt < new_abscissa->number_of_measurement_tables; mt++)
	{
		samples_per_cycle += new_abscissa->number_of_samples[mt];
	}

	data = (float*)malloc(samples_per_cycle * file->number_of_cycles * sizeof(float));

	if (data == NULL)
	{
		logmessage(MSG_ERROR, "Memory could not be allocated\n");

		return(false);
	}

	/* Calculate new_abscissa lookup tables */

	if (Calculate_Abscissa_Lookups(new_abscissa, PulsesPerCycle(file), file->engine.number_of_strokes, MinimumResolution(file)) == true)
	{
		if (interpolate == true)
		{
			for (cycle = 0; cycle < file->number_of_cycles; cycle++)
			{
				for (crank_angle = 0; crank_angle < samples_per_cycle; crank_angle++)
				{
					data[cycle*samples_per_cycle + crank_angle] = ReturnThetaData(file, cycle, DegreesToTheta(file, new_abscissa->ca_to_deg[crank_angle]), channel);
				}
			}
		}
		else
		{
			for (cycle = 0; cycle < file->number_of_cycles; cycle++)
			{
				for (crank_angle = 0; crank_angle < samples_per_cycle; crank_angle++)
				{
					theta = DegreesToTheta(file, new_abscissa->ca_to_deg[crank_angle]);

					new_crank_angle = file->channel_data[channel].abscissa.theta_to_ca[theta];

					data[cycle*samples_per_cycle + crank_angle] = ReturnCAData(file, cycle, new_crank_angle, channel);
				}
			}
			
		}

		free(file->channel_data[channel].data);

		file->channel_data[channel].data = data;

		file->channel_data[channel].samples_per_cycle = Copy_Abscissa(new_abscissa, &file->channel_data[channel].abscissa);

		if (calculate_theta_lookups(file, channel) == false)
		{
			logmessage(WARNING, "Resample_Channel: Could not calculate theta lookups\n");
		}
	}
	else
	{
		free(data);

		return(false);
	}

	return(true);
}

bool Generate_Sin_Channel(FileData* file, const unsigned int channel, const float frequency)
{
	unsigned int cycle;
	unsigned int crank_angle;
	float value;
	float angle;
	float duration;
	float fixed_modifier;
	float modifier;

	if (file == NULL)
	{
		return(false);
	}

	if (channel >= file->number_of_channels)
	{
		return(false);
	}

	if (file->channel_data == NULL)
	{
		return(false);
	}

	if (frequency < FLT_EPSILON)
	{
		return(false);
	}

	if (file->channel_data[channel].data == NULL)
	{
		return(false);
	}

	if (file->channel_data[channel].duration == NULL)
	{
		return(false);
	}

	if ((file->engine.number_of_strokes != 2) && (file->engine.number_of_strokes != 4))
	{
		return(false);
	}

	fixed_modifier = FLT_PI * frequency * 2.0f / 1000.0f / 360.0f * 2.0f / file->engine.number_of_strokes;

	for (cycle = 0; cycle < file->number_of_cycles; cycle++)
	{
		duration = file->channel_data[channel].duration[cycle];

		if (duration < FLT_EPSILON)
		{
			modifier = 1.0f;
		}
		else
		{
			modifier = fixed_modifier * duration ;
		}

		for (crank_angle = 0; crank_angle < file->channel_data[channel].samples_per_cycle; crank_angle++)
		{
			angle = CrankAngleToDegrees(file, crank_angle, channel) * modifier;

			// value = ReturnCAData(file, cycle, crank_angle, channel);
			// value += sinf(angle) * file->channel_data[channel].slope + file->channel_data[channel].offset;

			value = sinf(angle) * 10.0f;

			SetCAData(file, cycle, crank_angle, channel, value);
		}
	}

	return(true);
}

bool Generate_Block_Channel(FileData* file, const unsigned int channel, const float start, const float finish)
{
	unsigned int cycle;
	unsigned int crank_angle;
	unsigned int start_crank_angle;
	unsigned int finish_crank_angle;
	float value;
	float offset;

	if (file == NULL)
	{
		return(false);
	}

	if (channel >= file->number_of_channels)
	{
		return(false);
	}

	if (file->channel_data == NULL)
	{
		return(false);
	}

	if (file->channel_data[channel].data == NULL)
	{
		return(false);
	}

	if ((validate_angle(file, channel, start) == false) || (validate_angle(file, channel, finish) == false))
	{
		return(false);
	}

	start_crank_angle = DegreesToCrankAngle(file, start, channel);
	finish_crank_angle = DegreesToCrankAngle(file, finish, channel);

	offset = (float)(1.0 * file->channel_data[channel].slope + file->channel_data[channel].offset);

	for (cycle = 0; cycle < file->number_of_cycles; cycle++)
	{
		for (crank_angle = start_crank_angle; crank_angle <= finish_crank_angle; crank_angle++)
		{
			value = ReturnCAData(file, cycle, crank_angle, channel);

			value += offset;

			SetCAData(file, cycle, crank_angle, channel, value);
		}
	}

	return(true);
}

bool Generate_CPS_Channel(FileData* file, const unsigned int channel, const unsigned int pulses_per_rev, float tooth_ratio)
{
	unsigned int crank_angle;
	unsigned int cycle;
	float value;
	float angle;

	if (file == NULL)
	{
		return(false);
	}

	if (channel >= file->number_of_channels)
	{
		return(false);
	}

	if (file->channel_data == NULL)
	{
		return(false);
	}

	if (file->channel_data[channel].data == NULL)
	{
		return(false);
	}

	if ((tooth_ratio < 0.0f) || (tooth_ratio > 1.0f))
	{
		tooth_ratio = 0.5f;
	}

	float tooth_size = 360.0f / (float)pulses_per_rev;
	float offset = 90.0f * (float)file->engine.number_of_strokes;

	float off_value = (float)file->channel_data[channel].offset;
	float on_value = (float)(1.0 * file->channel_data[channel].slope + file->channel_data[channel].offset);

	for (crank_angle = 0; crank_angle < file->channel_data[channel].samples_per_cycle; crank_angle++)
	{
		angle = CrankAngleToDegrees(file, crank_angle, channel) + offset;

		float tooth_percentage = angle / tooth_size;
		tooth_percentage -= floorf(tooth_percentage);

		if (tooth_percentage < tooth_ratio)
		{
			value = off_value;
		}
		else
		{
			value = on_value;
		}

		for (cycle = 0; cycle < file->number_of_cycles; cycle++)
		{
			SetCAData(file, cycle, crank_angle, channel, value);
		}
	}

	return(true);
}

bool Convert_Channel_To_Time(FileData* file, const unsigned int channel, double conversion_resolution)
{
	unsigned int cycle;
	double total_duration;
	unsigned int total_samples;
	float* data = NULL;
	unsigned int sample;
	double t;
	double total_t;
	float angle;
	unsigned int theta;
	float range;
	float offset;
	unsigned int maximum_theta;
	unsigned int resolution;
	float value = 0.0f;
	double cycle_duration;
	unsigned int skipped_samples;
	unsigned int lookup_cycle;
	unsigned int abscissa_type;

	if (file == NULL)
	{
		return(false);
	}

	if ((channel >= file->number_of_channels) || (file->number_of_channels == 0))
	{
		return(false);
	}

	if (file->channel_data == NULL)
	{
		return(false);
	}

	if (file->channel_data[channel].samples_per_cycle == 0)
	{
		return(false);
	}

	if (file->channel_data[channel].data == NULL)
	{
		return(false);
	}

	if ((conversion_resolution < DBL_EPSILON) || (conversion_resolution > 1000.0))
	{
		return(false);
	}

	logmessage(WARNING, "Converting channel %s to time base\n", file->channel_data[channel].name);

	abscissa_type = file->channel_data[channel].abscissa.type;

	if ((abscissa_type != ABSCISSA_CRANKANGLE) && (abscissa_type != ABSCISSA_CYCLE))
	{
		return(false);
	}

	resolution = (unsigned int)((conversion_resolution + 0.00001) / file->time_resolution);

	conversion_resolution = file->time_resolution*(double)resolution;

	if (conversion_resolution < DBL_EPSILON)
	{
		conversion_resolution = file->time_resolution;
	}

	if (abscissa_type == ABSCISSA_CRANKANGLE)
	{
		maximum_theta = MaximumTheta(file, channel);
	}
	else
	{
		maximum_theta = 0;
	}

	total_duration = 0.0;

	for (cycle = 0; cycle<file->number_of_cycles; cycle++)
	{
		total_duration += (double)file->channel_data[channel].duration[cycle];
	}

	total_samples = (unsigned int)floor(total_duration / conversion_resolution);

	data = (float*)malloc(total_samples * sizeof(float));
	if (data == NULL)
	{
		logmessage(WARNING, "Could not allocate memory\n");
		return(false);
	}

	range = 180.0f * (float)file->engine.number_of_strokes;
	offset = -90.0f * (float)file->engine.number_of_strokes;

	sample = 0;
	cycle = 0;
	t = 0.0;
	total_t = 0.0;
	cycle_duration = (double)file->channel_data[channel].duration[0];

	skipped_samples = 0;
	while (sample < total_samples)
	{
		if (t >= total_t + cycle_duration)
		{
			total_t += cycle_duration;
			cycle++;

			if (cycle < file->number_of_cycles)
			{
				cycle_duration = (double)file->channel_data[channel].duration[cycle];
			}
		}

		if (cycle < file->number_of_cycles)
		{
			angle = range * (float)((t - total_t) / cycle_duration) + offset - file->channel_data[channel].tdc_offset;

			if (angle < offset)
			{
				if (cycle > 0)
				{
					lookup_cycle = cycle - 1;
				}
				else
				{
					lookup_cycle = 0;
				}

				angle += range;
			}
			else
			{
				lookup_cycle = cycle;
			}
			
			if (abscissa_type == ABSCISSA_CRANKANGLE)
			{
				theta = DegreesToTheta(file, angle);

				if (theta >= maximum_theta)
				{
					theta = maximum_theta - 1;
				}

				value = ReturnThetaData(file, lookup_cycle, theta, channel);
			}
			else /* if (abscissa_type == ABSCISSA_CYCLE) */
			{
				value = file->channel_data[channel].data[lookup_cycle];
			}
		}
		else
		{
			skipped_samples++;
		}

		data[sample] = value;

		sample++;
		t += conversion_resolution;
	}

	if (skipped_samples > 0)
	{
		logmessage(NOTICE, "Skipped %u samples\n", skipped_samples);
	}

	/* Release abscissa */

	Empty_Abscissa(&file->channel_data[channel].abscissa);

	file->channel_data[channel].abscissa.type = ABSCISSA_TIME;

	file->channel_data[channel].abscissa.number_of_measurement_tables = 1;
	
	file->channel_data[channel].abscissa.start = (unsigned int*)malloc(sizeof(unsigned int));
	if (file->channel_data[channel].abscissa.start == NULL)
	{
		logmessage(FATAL, "Failed to allocate memory\n");
	}

	file->channel_data[channel].abscissa.number_of_samples = (unsigned int*)malloc(sizeof(unsigned int));
	if (file->channel_data[channel].abscissa.number_of_samples == NULL)
	{
		logmessage(FATAL, "Failed to allocate memory\n");
	}

	file->channel_data[channel].abscissa.resolution = (unsigned int*)malloc(sizeof(unsigned int));
	if (file->channel_data[channel].abscissa.resolution == NULL)
	{
		logmessage(FATAL, "Failed to allocate memory\n");
	}

	file->channel_data[channel].abscissa.start[0] = 0;
	file->channel_data[channel].abscissa.number_of_samples[0] = total_samples;
	file->channel_data[channel].abscissa.resolution[0] = resolution;

	file->channel_data[channel].samples_per_cycle = total_samples;

	/* Release data */

	free(file->channel_data[channel].data);

	file->channel_data[channel].slope = 1.0;
	file->channel_data[channel].offset = 0.0;

	file->channel_data[channel].data = data;

	return(true);
}

bool Verify_File_Configuration(FileData* file,Analysis* analysis)
{
	unsigned int channel;
	unsigned int soc_channel;
	unsigned int offset_channel;

	if (file == NULL)
	{
		return(false);
	}

	if (file->channel_data == NULL)
	{
		return(false);
	}

	for (channel = 0; channel < file->number_of_channels; channel++)
	{
		if ((file->channel_data[channel].offset_config.type == OFFSET_WINDOW) ||
			(file->channel_data[channel].offset_config.type == OFFSET_WINDOW_ABSOLUTE))
		{
			if (channel_name_to_number(file, file->channel_data[channel].offset_config.channel_name, ABSCISSA_UNKNOWN, &offset_channel) == false)
			{
				logmessage(NOTICE, "Channel %s invalid offset channel %s\n", file->channel_data[channel].name, file->channel_data[channel].offset_config.channel_name);

				file->channel_data[channel].offset_config.type = OFFSET_NONE;
			}
			else if (file->channel_data[offset_channel].abscissa.type != ABSCISSA_CRANKANGLE)
			{
				logmessage(NOTICE, "Channel %s offset channel %s not crank angle based\n", file->channel_data[channel].name, file->channel_data[channel].offset_config.channel_name);

				file->channel_data[channel].offset_config.type = OFFSET_NONE;
			}
			else
			{
				/* Do Nothing */
			}
		}

		if ((file->channel_data[channel].soc_config.type == SOC_CA_CHANNEL_RISE) ||
			(file->channel_data[channel].soc_config.type == SOC_CA_CHANNEL_FALL) ||
			(file->channel_data[channel].soc_config.type == SOC_DIGITAL_FALLING_EDGE) ||
			(file->channel_data[channel].soc_config.type == SOC_DIGITAL_RISING_EDGE) ||
			(file->channel_data[channel].soc_config.type == SOC_CA_CHANNEL_AVG))
		{
			if (channel_name_to_number(file, file->channel_data[channel].soc_config.channel_name, ABSCISSA_UNKNOWN, &soc_channel) == false)
			{
				logmessage(NOTICE, "Channel %s changing soc_config.type to SOC_FIXED due to unknown SOC channel %s\n", file->channel_data[channel].name, file->channel_data[channel].soc_config.channel_name);

				file->channel_data[channel].soc_config.type = SOC_FIXED;
				file->channel_data[channel].soc_config.fixed_value = 0.0f;

			}
			else if (file->channel_data[soc_channel].abscissa.type != ABSCISSA_CRANKANGLE)
			{
				logmessage(WARNING, "Channel %s changing soc_config.type to SOC_FIXED due to invalid channel\n", file->channel_data[channel].name);

				file->channel_data[channel].soc_config.type = SOC_FIXED;
				file->channel_data[channel].soc_config.fixed_value = 0.0f;
			}
			else
			{
				/* Do Nothing */
			}
		}
		else if (file->channel_data[channel].soc_config.type == SOC_CYC_CHANNEL)
		{
			if (channel_name_to_number(file, file->channel_data[channel].soc_config.channel_name, ABSCISSA_UNKNOWN, &soc_channel) == false)
			{
				logmessage(NOTICE, "Channel %s invalid SOC channel %s\n", file->channel_data[channel].name, file->channel_data[channel].soc_config.channel_name);

				file->channel_data[channel].soc_config.type = SOC_FIXED;
				file->channel_data[channel].soc_config.fixed_value = 0.0f;
			}
			else if (file->channel_data[soc_channel].abscissa.type == ABSCISSA_CYCLE)
			{
				/* Channel OK */

				logmessage(NOTICE, "Channel %s SOC_CYC_CHANNEL valid\n", file->channel_data[channel].name);
			}
			else if (file->channel_data[soc_channel].abscissa.type == ABSCISSA_CRANKANGLE)
			{
				if ((file->channel_data[channel].type == CHAN_CYL_PR) && (soc_channel == channel))
				{
					logmessage(WARNING, "Channel %s cannot perform SOC analysis on cylinder pressure channel\n", file->channel_data[channel].name);

					file->channel_data[channel].soc_config.type = SOC_FIXED;
					file->channel_data[channel].soc_config.fixed_value = 0.0f;
				}
				else if (analysis != NULL)
				{
					if (analysis->channel[soc_channel].analysis_rqst[START_OF_COMBUSTION] >= AnalysisRqst::TemporaryCalculation)
					{
						/* Channel OK */

						logmessage(NOTICE, "Channel %s SOC_CYC_CHANNEL valid\n", file->channel_data[channel].name);
					}
					else
					{
						logmessage(WARNING, "Channel %s changing soc_config.type to SOC_FIXED due to SOC not requested for calculation\n", file->channel_data[channel].name);

						file->channel_data[channel].soc_config.type = SOC_FIXED;
						file->channel_data[channel].soc_config.fixed_value = 0.0f;
					}
				}
				else
				{
					logmessage(NOTICE, "Channel %s SOC_CYC_CHANNEL valid (no analysis structure)\n", file->channel_data[channel].name);
				}
			}
			else
			{
				logmessage(WARNING, "Channel %s changing soc_config.type to SOC_FIXED due to invalid channel\n", file->channel_data[channel].name);

				file->channel_data[channel].soc_config.type = SOC_FIXED;
				file->channel_data[channel].soc_config.fixed_value = 0.0f;
			}
		}
		else
		{
			/* Do Nothing */
		}
	}

	return(true);
}

void calculate_channel_groups(const FileData* file,Analysis* analysis)
{
	unsigned int channel;
	unsigned int group;
	unsigned int number_of_groups;
	unsigned int unique_channel;
	unsigned int position;
	unsigned int analysis_type;
	unsigned int* unique_channel_types = NULL;
	char char_position;
	char group_name[SIZEOF_CHANNEL_NAME];
	bool found;
	bool same;

	if ((file == NULL) || (analysis == NULL))
	{
		return;
	}

	if ((file->number_of_channels == 0) || (file->channel_data == NULL))
	{
		return;
	}

	/* Delete existing group structure */

	Delete_Analysis_Group(analysis);

	/* Calculate number of channel groups */

	unique_channel_types = (unsigned int*)malloc(file->number_of_channels * sizeof(unsigned int));
	if (unique_channel_types == NULL)
	{
		logmessage(FATAL,"Memory could not be allocated\n");
	}
	unique_channel_types[0] = file->channel_data[0].type;
	number_of_groups = 0;

	for (channel=0;channel<file->number_of_channels;channel++)
	{
		if (file->channel_data[channel].loaded == true)
		{
			found = false;
			for (unique_channel = 0; unique_channel < number_of_groups; unique_channel++)
			{
				if ((file->channel_data[channel].abscissa.type == ABSCISSA_CRANKANGLE) && (file->channel_data[channel].type == unique_channel_types[unique_channel]))
				{
					found = true;
				}
			}

			if ((file->channel_data[channel].abscissa.type == ABSCISSA_CRANKANGLE) && (found == false))
			{
				unique_channel_types[number_of_groups] = file->channel_data[channel].type;
				number_of_groups += 1;
			}
		}
	}

	if (number_of_groups == 0)
	{
		free(unique_channel_types);

		return;
	}

	/* Allocate number of channel groups */
	
	analysis->group = (ChannelGroup*)malloc(number_of_groups * sizeof(ChannelGroup));
	if (analysis->group == NULL)
	{
		logmessage(FATAL,"Memory could not be allocated\n");
	}
	analysis->number_of_groups = number_of_groups;

	for (group = 0; group < number_of_groups; group++)
	{
		analysis->group[group].number_of_channels = 0;

		for (channel = 0; channel < file->number_of_channels; channel++)
		{
			if ((file->channel_data[channel].loaded == true) &&
				(file->channel_data[channel].abscissa.type == ABSCISSA_CRANKANGLE) && 
				(file->channel_data[channel].type == unique_channel_types[group]))
			{
				analysis->group[group].number_of_channels += 1;
			}
		}

		analysis->group[group].channels = (unsigned int*)malloc(analysis->group[group].number_of_channels * sizeof(unsigned int));
		if (analysis->group[group].channels == NULL)
		{
			logmessage(FATAL, "Memory could not be allocated\n");
		}

		/* Populate each channel group with channels */

		position = 0;
		for (channel = 0; channel < file->number_of_channels; channel++)
		{
			if ((file->channel_data[channel].loaded == true) && 
				(file->channel_data[channel].abscissa.type == ABSCISSA_CRANKANGLE) && 
				(file->channel_data[channel].type == unique_channel_types[group]))
			{
				analysis->group[group].channels[position] = channel;
				position += 1;
			}
		}

		/* Determine group name */

		if (analysis->group[group].number_of_channels > 1)
		{
			position = 0;
			same = true;
			while ((position < SIZEOF_CHANNEL_NAME) && (same == true))
			{
				char_position = (char)file->channel_data[analysis->group[group].channels[0]].name[position];

				for (channel = 1; channel < analysis->group[group].number_of_channels; channel++)
				{
					if ((char)file->channel_data[analysis->group[group].channels[channel]].name[position] != char_position)
					{
						same = false;
					}
				}

				if (same == true)
				{
					position += 1;
				}
			}

			if ((position == 0) || (position >= 64))
			{
				snprintf(group_name, SIZEOF_CHANNEL_NAME, "GROUP%u", group);
			}
			else
			{
				snprintf(group_name, SIZEOF_CHANNEL_NAME, "%s", file->channel_data[analysis->group[group].channels[0]].name);
				group_name[position] = 0x00;
			}
		}
		else
		{
			strncpy(group_name, file->channel_data[analysis->group[group].channels[0]].name, SIZEOF_CHANNEL_NAME);
		}

		group_name[SIZEOF_CHANNEL_NAME - 1] = 0x00;

		strncpy(analysis->group[group].name, group_name, SIZEOF_CHANNEL_NAME);

		analysis->group[group].name[SIZEOF_CHANNEL_NAME - 1] = 0x00;

		/* Allocate statistics memory */

		analysis->group[group].cycle_statistics = (Statistics*)malloc(get_number_of_analysis_channels() * sizeof(Statistics));
		if (analysis->group[group].cycle_statistics == NULL)
		{
			logmessage(FATAL, "Memory could not be allocated\n");
		}

		analysis->group[group].cycle_sum_statistics = (Statistics*)malloc(get_number_of_analysis_channels() * sizeof(Statistics));
		if (analysis->group[group].cycle_sum_statistics == NULL)
		{
			logmessage(FATAL, "Memory could not be allocated\n");
		}

		analysis->group[group].cylinder_mean_statistics = (Statistics*)malloc(get_number_of_analysis_channels() * sizeof(Statistics));
		if (analysis->group[group].cylinder_mean_statistics == NULL)
		{
			logmessage(FATAL, "Memory could not be allocated\n");
		}

		analysis->group[group].engine_mean_statistics = (Statistics*)malloc(get_number_of_analysis_channels() * sizeof(Statistics));
		if (analysis->group[group].engine_mean_statistics == NULL)
		{
			logmessage(FATAL, "Memory could not be allocated\n");
		}

		analysis->group[group].crank_angle_statistics = (StatisticsArray*)malloc(get_number_of_analysis_channels() * sizeof(StatisticsArray));
		if (analysis->group[group].crank_angle_statistics == NULL)
		{
			logmessage(FATAL, "Memory could not be allocated\n");
		}

		analysis->group[group].cycle_sum_data = (StatisticsArray*)malloc(get_number_of_analysis_channels() * sizeof(StatisticsArray));
		if (analysis->group[group].cycle_sum_data == NULL)
		{
			logmessage(FATAL, "Memory could not be allocated\n");
		}

		/* Zero all statistics */

		for (analysis_type = 0; analysis_type < get_number_of_analysis_channels(); analysis_type++)
		{
			Zero_Statistics(&analysis->group[group].cycle_statistics[analysis_type]);
			Zero_Statistics(&analysis->group[group].cycle_sum_statistics[analysis_type]);
			Zero_Statistics(&analysis->group[group].cylinder_mean_statistics[analysis_type]);
			Zero_Statistics(&analysis->group[group].engine_mean_statistics[analysis_type]);
			Zero_StatisticsArray(&analysis->group[group].crank_angle_statistics[analysis_type]);
			Zero_StatisticsArray(&analysis->group[group].cycle_sum_data[analysis_type]);
		}

		/* Allocate crank angle statistics data */

		for (analysis_type = 0; analysis_type < get_number_of_analysis_channels(); analysis_type++)
		{
			if (get_analysis_output_abscissa(analysis_type) == ABSCISSA_CRANKANGLE)
			{
				Allocate_StatisticsArray(&analysis->group[group].crank_angle_statistics[analysis_type], file->channel_data[analysis->group[group].channels[0]].samples_per_cycle);
			}
			else if (get_analysis_output_abscissa(analysis_type) == ABSCISSA_FREQUENCY)
			{
				unsigned int number_of_bins = (unsigned int)(analysis->channel[analysis->group[group].channels[0]].config.fft_stats_maximum / analysis->channel[analysis->group[group].channels[0]].config.fft_stats_resolution) + 1;
				
				Allocate_StatisticsArray(&analysis->group[group].crank_angle_statistics[analysis_type], number_of_bins);
			}
			else if (get_analysis_output_abscissa(analysis_type) == ABSCISSA_CYCLE)
			{
				Allocate_StatisticsArray(&analysis->group[group].cycle_sum_data[analysis_type], file->number_of_cycles);
			}
			else
			{
				/* Do Nothing */
			}
		}
	}

	free(unique_channel_types);
}

int channel_to_group(const Analysis* analysis,const unsigned int channel)
{
	unsigned int group;
	unsigned int return_value = -1;
	unsigned int group_channel;
	bool found = false;

	if (analysis == NULL)
	{
		return(-1);
	}

	if (channel >= analysis->number_of_channels)
	{
		return(-1);
	}

	group = 0;

	while ((found == false) && (group < analysis->number_of_groups))
	{
		for (group_channel=0;group_channel<analysis->group[group].number_of_channels;group_channel++)
		{
			if (analysis->group[group].channels[group_channel] == channel)
			{
				return_value = group;
				found = true;
			}
		}

		group += 1;
	}

	if (found == false)
	{
		logmessage(DEBUG,"Number of groups: %u\n",analysis->number_of_groups);

		for (group=0;group<analysis->number_of_groups;group++)
		{
			for (group_channel=0;group_channel<analysis->group[group].number_of_channels;group_channel++)
			{
				logmessage(DEBUG,"Group %u channel %u: %u\n",group,group_channel,analysis->group[group].channels[group_channel]);
			}
		}

		logmessage(NOTICE,"Failed to convert channel %u into valid group\n",channel);
	}

	return(return_value);
}

void Shift_ChannelData(ChannelData* channel_data,const unsigned int first,const unsigned int number_of_cycles)
{
	/* For time based data number_of_cycles is the total size of the data, i.e. sample length in seconds * acquisition rate in Hertz */

	float* new_data = NULL;
	float* tmp_data = NULL;
	size_t data_size;
	/*size_t lower_half_size;
	size_t upper_half_size;*/

	if (channel_data == NULL)
	{
		return;
	}

	if (channel_data->data == NULL)
	{
		return;
	}
	
	if ((first < number_of_cycles) && (first != 0))
	{
		if ((channel_data->abscissa.type == ABSCISSA_CRANKANGLE) ||
			(channel_data->abscissa.type == ABSCISSA_FREQUENCY))
		{
			/*lower_half_size = first;
			upper_half_size = number_of_cycles - first;

			if (upper_half_size < lower_half_size)
			{
				new_data = (float*)malloc(upper_half_size * channel_data->samples_per_cycle * sizeof(float));
				if (new_data == NULL)
				{
					logmessage(FATAL, "Memory could not be allocated\n");
					exit(EXIT_FAILURE);
				}

				memcpy(new_data, &channel_data->data[lower_half_size], upper_half_size * channel_data->samples_per_cycle * sizeof(float));

				memmove(&channel_data->data[upper_half_size], channel_data->data, lower_half_size * channel_data->samples_per_cycle * sizeof(float));

				memcpy(channel_data->data, new_data, upper_half_size * channel_data->samples_per_cycle * sizeof(float));

				free(new_data);
			}
			else
			{
				new_data = (float*)malloc(lower_half_size * channel_data->samples_per_cycle * sizeof(float));
				if (new_data == NULL)
				{
					logmessage(FATAL, "Memory could not be allocated\n");
					exit(EXIT_FAILURE);
				}

				memcpy(new_data, channel_data->data, lower_half_size * channel_data->samples_per_cycle * sizeof(float));

				memmove(channel_data->data, &channel_data->data[lower_half_size], upper_half_size * channel_data->samples_per_cycle * sizeof(float));

				memcpy(&channel_data->data[upper_half_size], new_data, lower_half_size * channel_data->samples_per_cycle * sizeof(float));

				free(new_data);
			}*/

			new_data = (float*)malloc(number_of_cycles * channel_data->samples_per_cycle * sizeof(float));
			if (new_data == NULL)
			{
				logmessage(FATAL, "Memory could not be allocated\n");
				exit(EXIT_FAILURE);
			}
			data_size = (number_of_cycles - first) * channel_data->samples_per_cycle;

			memcpy((void*)new_data, (void*)&channel_data->data[first * channel_data->samples_per_cycle], data_size * sizeof(float));

			memcpy((void*)&new_data[data_size], (void*)channel_data->data, (first * channel_data->samples_per_cycle * sizeof(float)));

			tmp_data = channel_data->data;
			channel_data->data = new_data;
			free(tmp_data);
		}
		else if ((channel_data->abscissa.type == ABSCISSA_CYCLE) ||
			     (channel_data->abscissa.type == ABSCISSA_TIME))
		{
			new_data = (float*)malloc(number_of_cycles * sizeof(float));
			if (new_data == NULL)
			{
				logmessage(FATAL, "Memory could not be allocated\n");
				exit(EXIT_FAILURE);
			}

			data_size = (number_of_cycles - first);

			memcpy((void*)new_data, (void*)&channel_data->data[first], data_size * sizeof(float));

			memcpy((void*)&new_data[data_size], (void*)channel_data->data, (first * sizeof(float)));

			tmp_data = channel_data->data;
			channel_data->data = new_data;
			free(tmp_data);
		}
		else if (channel_data->abscissa.type == ABSCISSA_HISTOGRAM)
		{
			unsigned int samples_per_cycle = KHI_MAX_KNKMAX * KHI_RESOLUTION;

			new_data = (float*)malloc(number_of_cycles * samples_per_cycle * sizeof(float));
			if (new_data == NULL)
			{
				logmessage(FATAL, "Memory could not be allocated\n");
				exit(EXIT_FAILURE);
			}
			data_size = (number_of_cycles - first) * samples_per_cycle;

			memcpy((void*)new_data, (void*)&channel_data->data[first * samples_per_cycle], data_size * sizeof(float));

			memcpy((void*)&new_data[data_size], (void*)channel_data->data, (first * samples_per_cycle * sizeof(float)));

			tmp_data = channel_data->data;
			channel_data->data = new_data;
			free(tmp_data);
		}
		else
		{
			/* Do Nothing */
		}

		if ((channel_data->abscissa.type == ABSCISSA_CRANKANGLE) && (channel_data->duration != NULL))
		{
			new_data = (float*)malloc(number_of_cycles * sizeof(float));
			if (new_data == NULL)
			{
				logmessage(FATAL, "Memory could not be allocated\n");
				exit(EXIT_FAILURE);
			}

			data_size = (number_of_cycles - first);

			memcpy((void*)new_data, (void*)&channel_data->duration[first], data_size * sizeof(float));

			memcpy((void*)&new_data[data_size], (void*)channel_data->duration, (first * sizeof(float)));

			tmp_data = channel_data->duration;
			channel_data->duration = new_data;
			free(tmp_data);
		}
	}
}

void Offset_Channel(FileData* file, const unsigned int channel, const float tdc_offset, float* signal_offset)
{
	if (file == NULL)
	{
		return;
	}

	if (channel >= file->number_of_channels)
	{
		return;
	}

	if (file->channel_data == NULL)
	{
		return;
	}

	if (file->channel_data[channel].data == NULL)
	{
		return;
	}

	if (file->channel_data[channel].abscissa.type != ABSCISSA_CRANKANGLE)
	{
		logmessage(MSG_ERROR, "Offset_ChannelData() is for crank angle data only\n");

		return;
	}

	unsigned int cycle;
	unsigned int cycle_reversed;
	unsigned int crank_angle;
	unsigned int crank_angle_reversed;
	float value;
	float offset;
	int theta_offset = DegreesToThetaRel(file, tdc_offset);
	unsigned int theta;
	unsigned int maximum_theta = MaximumTheta(file, channel);
	int cycle_mod;

	if (theta_offset == 0)
	{
		/* Nothing to do */

		return;
	}

	while (theta_offset >= (int)maximum_theta)
	{
		theta_offset -= maximum_theta;
	}

	while (theta_offset <= -(int)maximum_theta)
	{
		theta_offset += maximum_theta;
	}

	if (theta_offset < 0)
	{
		/* theta will be ahead of current crank_angle (data shifted to left) */

		for (cycle = 0; cycle < file->number_of_cycles; cycle++)
		{
			for (crank_angle = 0; crank_angle < file->channel_data[channel].samples_per_cycle; crank_angle++)
			{
				theta = ReturnTheta(file, crank_angle, channel);
				
				if (theta - theta_offset < maximum_theta)
				{
					cycle_mod = cycle;
					theta = theta - theta_offset;
				}
				else
				{
					cycle_mod = cycle + 1;
					theta = theta - theta_offset - maximum_theta;
				}

				if (cycle_mod < (int)file->number_of_cycles)
				{
					if ((signal_offset != NULL) && (cycle_mod != cycle))
					{
						offset = signal_offset[cycle] - signal_offset[cycle_mod];
					}
					else
					{
						offset = 0.0f;
					}

					value = ReturnThetaData(file, (unsigned int)cycle_mod, theta, channel) + offset;
				}
				else
				{
					value = ReturnThetaData(file, file->number_of_cycles-1, maximum_theta -1, channel);
				}

				SetCAData(file, cycle, crank_angle, channel, value);
			}
		}
	}
	else
	{
		/* theta will be behind current crank_angle (data shifted to right) */

		for (cycle_reversed = 0; cycle_reversed < file->number_of_cycles; cycle_reversed++)
		{
			cycle = file->number_of_cycles - (cycle_reversed + 1);

			for (crank_angle_reversed = 0; crank_angle_reversed < file->channel_data[channel].samples_per_cycle; crank_angle_reversed++)
			{
				crank_angle = file->channel_data[channel].samples_per_cycle - (crank_angle_reversed + 1);

				theta = ReturnTheta(file, crank_angle, channel);

				if ((int)theta >= theta_offset)
				{
					cycle_mod = cycle;
					theta = theta - theta_offset;
				}
				else
				{
					cycle_mod = cycle - 1;
					theta = theta - theta_offset + maximum_theta;
				}

				if (cycle_mod >= 0)
				{
					if ((signal_offset != NULL) && (cycle_mod != cycle))
					{
						offset = signal_offset[cycle] - signal_offset[cycle_mod];
					}
					else
					{
						offset = 0.0f;
					}

					value = ReturnThetaData(file, (unsigned int)cycle_mod, theta, channel) + offset;
				}
				else
				{
					value = ReturnThetaData(file, 0, 0, channel);
				}

				SetCAData(file, cycle, crank_angle, channel, value);
			}
		}
	}
}

void Zero_Statistics(Statistics* statistics)
{
	if (statistics == NULL)
	{
		return;
	}

	statistics->sum = 0.0f;
	statistics->mean = 0.0f;
	statistics->min = 0.0f;
	statistics->max = 0.0f;
	statistics->stddev = 0.0f;
	statistics->cov = 0.0f;
	statistics->lnv = 0.0f;
	statistics->skewness = 0.0f;
	statistics->kurtosis = 0.0f;
	statistics->range = 0.0f;
}

void Zero_StatisticsArray(StatisticsArray* statistics)
{
	if (statistics == NULL)
	{
		return;
	}

	statistics->sum = NULL;
	statistics->mean = NULL;
	statistics->min = NULL;
	statistics->max = NULL;
	statistics->stddev = NULL;
	statistics->cov = NULL;
	statistics->lnv = NULL;
	statistics->skewness = NULL;
	statistics->kurtosis = NULL;
	statistics->range = NULL;
}

void Allocate_StatisticsArray(StatisticsArray* statistics,const unsigned int samples_per_cycle)
{
	if ((statistics == NULL) || (samples_per_cycle == 0))
	{
		logmessage(DEBUG,"Invalid call to Allocate_StatisticsArray()\n");
		return;
	}
	
	statistics->max = (float*)malloc(samples_per_cycle * sizeof(float));
	if (statistics->max == NULL)
	{
		logmessage(FATAL,"Memory could not be allocated\n");
		return;
	}

	statistics->mean = (float*)malloc(samples_per_cycle * sizeof(float));
	if (statistics->mean == NULL)
	{
		logmessage(FATAL,"Memory could not be allocated\n");
		return;
	}

	statistics->min = (float*)malloc(samples_per_cycle * sizeof(float));
	if (statistics->min == NULL)
	{
		logmessage(FATAL,"Memory could not be allocated\n");
		return;
	}

	statistics->stddev = (float*)malloc(samples_per_cycle * sizeof(float));
	if (statistics->stddev == NULL)
	{
		logmessage(FATAL,"Memory could not be allocated\n");
		return;
	}

	statistics->sum = (float*)malloc(samples_per_cycle * sizeof(float));
	if (statistics->sum == NULL)
	{
		logmessage(FATAL,"Memory could not be allocated\n");
		return;
	}

	Reset_StatisticsArray(statistics, samples_per_cycle);
}

void Reset_StatisticsArray(StatisticsArray* statistics, const unsigned int samples_per_cycle)
{
	if ((statistics == NULL) || (samples_per_cycle == 0))
	{
		return;
	}

	if (statistics->max != NULL)
	{
		memset((void*)statistics->max, 0, samples_per_cycle * sizeof(float));
	}
	
	if (statistics->mean != NULL)
	{
		memset((void*)statistics->mean, 0, samples_per_cycle * sizeof(float));
	}
	
	if (statistics->min != NULL)
	{
		memset((void*)statistics->min, 0, samples_per_cycle * sizeof(float));
	}

	if (statistics->stddev != NULL)
	{
		memset((void*)statistics->stddev, 0, samples_per_cycle * sizeof(float));
	}
	
	if (statistics->sum != NULL)
	{
		memset((void*)statistics->sum, 0, samples_per_cycle * sizeof(float));
	}
}

void Zero_File(FileData* file)
{
	unsigned int channel;
	size_t channel_size;

	if (file == NULL)
	{
		return;
	}

	for (channel=0;channel<file->number_of_channels;channel++)
	{
		if (file->channel_data[channel].loaded == true)
		{
			if ((file->channel_data[channel].abscissa.type == ABSCISSA_CRANKANGLE) ||
				(file->channel_data[channel].abscissa.type == ABSCISSA_FREQUENCY))
			{
				channel_size = file->number_of_cycles * file->channel_data[channel].samples_per_cycle * sizeof(float);
			}
			else if (file->channel_data[channel].abscissa.type == ABSCISSA_CYCLE)
			{
				channel_size = file->number_of_cycles * sizeof(float);
			}
			else
			{
				channel_size = file->channel_data[channel].samples_per_cycle * sizeof(float);
			}

			if (file->channel_data[channel].data != NULL)
			{
				memset(file->channel_data[channel].data,0,channel_size);
			}

			if (file->channel_data[channel].duration != NULL)
			{
				memset(file->channel_data[channel].duration, 0, file->number_of_cycles * sizeof(float));
			}
		}
	}
}

void Zero_Analysis(FileData* file, Analysis* analysis)
{
	unsigned int analysis_structure;
	unsigned int analysis_channel;
	size_t channel_size;
	unsigned int group;
	unsigned int samples_per_cycle;

	if ((file == NULL) || (analysis == NULL))
	{
		return;
	}

	if (file->channel_data == NULL)
	{
		return;
	}

	if (analysis->group != NULL)
	{
		for (group = 0; group < analysis->number_of_groups; group++)
		{
			for (analysis_channel = 0; analysis_channel < get_number_of_analysis_channels(); analysis_channel++)
			{
				if (analysis->group[group].cycle_statistics != NULL)
				{
					Zero_Statistics(&analysis->group[group].cycle_statistics[analysis_channel]);
				}
				
				if (analysis->group[group].cycle_sum_statistics != NULL)
				{
					Zero_Statistics(&analysis->group[group].cycle_sum_statistics[analysis_channel]);
				}
				
				if (analysis->group[group].cylinder_mean_statistics != NULL)
				{
					Zero_Statistics(&analysis->group[group].cylinder_mean_statistics[analysis_channel]);
				}

				if (analysis->group[group].engine_mean_statistics != NULL)
				{
					Zero_Statistics(&analysis->group[group].engine_mean_statistics[analysis_channel]);
				}
				
				if (analysis->group[group].crank_angle_statistics != NULL)
				{
					if (get_analysis_output_abscissa(analysis_channel) == ABSCISSA_FREQUENCY)
					{
						unsigned int number_of_bins = (unsigned int)(analysis->channel[analysis->group[group].channels[0]].config.fft_stats_maximum / analysis->channel[analysis->group[group].channels[0]].config.fft_stats_resolution) + 1;

						Reset_StatisticsArray(&analysis->group[group].crank_angle_statistics[analysis_channel], number_of_bins);
					}
					else
					{
						Reset_StatisticsArray(&analysis->group[group].crank_angle_statistics[analysis_channel], file->channel_data[analysis->group[group].channels[0]].samples_per_cycle);
					}
				}
				
				if (analysis->group[group].cycle_sum_data != NULL)
				{
					Reset_StatisticsArray(&analysis->group[group].cycle_sum_data[analysis_channel], file->number_of_cycles);
				}
			}
		}
	}

	for (analysis_structure=0;analysis_structure<analysis->number_of_channels;analysis_structure++)
	{
		Zero_Statistics(&analysis->channel[analysis_structure].raw_statistics);

		Reset_StatisticsArray(&analysis->channel[analysis_structure].raw_ca_statistics, file->channel_data[analysis->channel[analysis_structure].channel].samples_per_cycle);

		for (analysis_channel=0;analysis_channel<get_number_of_analysis_channels();analysis_channel++)
		{
			if (analysis->channel[analysis_structure].analysis_rqst[analysis_channel] > AnalysisRqst::NotCalculated)
			{
				samples_per_cycle = analysis->channel[analysis_structure].results[analysis_channel].samples_per_cycle;

				if ((analysis->channel[analysis_structure].results[analysis_channel].abscissa.type == ABSCISSA_CRANKANGLE) ||
					(analysis->channel[analysis_structure].results[analysis_channel].abscissa.type == ABSCISSA_FREQUENCY))
				{
					if (analysis->channel[analysis_structure].analysis_rqst[analysis_channel] >= AnalysisRqst::CalculateAndStore)
					{
						channel_size = (size_t)analysis->channel[analysis_structure].number_of_cycles * (size_t)samples_per_cycle * sizeof(float);
					}
					else
					{
						channel_size = (size_t)samples_per_cycle * sizeof(float);
					}
				}
				else if (analysis->channel[analysis_structure].results[analysis_channel].abscissa.type == ABSCISSA_CYCLE)
				{
					channel_size = (size_t)analysis->channel[analysis_structure].number_of_cycles * sizeof(float);
				}
				else
				{
					channel_size = (size_t)samples_per_cycle * sizeof(float);
				}

				if (analysis->channel[analysis_structure].results[analysis_channel].data != NULL)
				{
					memset(analysis->channel[analysis_structure].results[analysis_channel].data,0,channel_size);
				}

				if (analysis->channel[analysis_structure].statistics != NULL)
				{
					Zero_Statistics(&analysis->channel[analysis_structure].statistics[analysis_channel]);
				}

				if (analysis->channel[analysis_structure].ca_statistics != NULL)
				{
					if (analysis->channel[analysis_structure].results[analysis_channel].abscissa.type == ABSCISSA_FREQUENCY)
					{
						unsigned int number_of_bins = (unsigned int)(analysis->channel[analysis_structure].config.fft_stats_maximum / analysis->channel[analysis_structure].config.fft_stats_resolution) + 1;

						Reset_StatisticsArray(&analysis->channel[analysis_structure].ca_statistics[analysis_channel], number_of_bins);
					}
					else
					{
						Reset_StatisticsArray(&analysis->channel[analysis_structure].ca_statistics[analysis_channel], samples_per_cycle);
					}
				}
			}
		}
	}
}

bool configure_results_channel(const FileData* file,AnalysisChannel* analysis,const unsigned int analysis_channel)
{
	unsigned int measurement_table;
	unsigned int sizeof_fft_analysis = 0;
	unsigned int channel;
	bool result = true;
	unsigned int data_pointer;
	float min;
	float range;
	float value;
	unsigned int cycle;
	unsigned int sizeof_histogram_analysis = 0;
	unsigned int number_of_results_cycles;

	if ((analysis == NULL) || (analysis_channel >= get_number_of_analysis_channels()))
	{
		return(false);
	}

	if (analysis->results == NULL)
	{
		return(false);
	}

	Zero_Statistics(&analysis->statistics[analysis_channel]);
	Zero_StatisticsArray(&analysis->ca_statistics[analysis_channel]);

	analysis->results[analysis_channel].data = NULL;
	analysis->results[analysis_channel].data_d = NULL;
	analysis->results[analysis_channel].abscissa.number_of_samples = NULL;
	analysis->results[analysis_channel].abscissa.start = NULL;
	analysis->results[analysis_channel].abscissa.resolution = NULL;
	analysis->results[analysis_channel].abscissa.theta_to_ca = NULL;
	analysis->results[analysis_channel].abscissa.theta_weighting = NULL;
	analysis->results[analysis_channel].abscissa.theta_weighting_inv = NULL;
	analysis->results[analysis_channel].abscissa.ca_to_theta = NULL;
	analysis->results[analysis_channel].abscissa.volume = NULL;
	analysis->results[analysis_channel].abscissa.ca_to_deg = NULL;
	analysis->results[analysis_channel].abscissa.axis = NULL;
	analysis->results[analysis_channel].abscissa.slope = 1.0;
	analysis->results[analysis_channel].abscissa.offset = 0.0;
	analysis->results[analysis_channel].abscissa.axis_type = VARIABLE_NONE;
	analysis->results[analysis_channel].duration = NULL;

	if (file == NULL)
	{
		return(false);
	}

	if (file->channel_data == NULL)
	{
		return(false);
	}

	channel = analysis->channel;

	if ((channel >= file->number_of_channels) || (file->number_of_channels == 0))
	{
		return(false);
	}

	analysis->results[analysis_channel].abscissa.type = ABSCISSA_UNKNOWN;
	analysis->results[analysis_channel].abscissa.number_of_measurement_tables = 0;
	analysis->results[analysis_channel].abscissa.units[0] = 0x00;
	analysis->results[analysis_channel].loaded = true;
	analysis->results[analysis_channel].max = 0.0f;
	analysis->results[analysis_channel].min = 0.0f;
	analysis->results[analysis_channel].isoffset = false;
	analysis->results[analysis_channel].type = CHAN_RESULTS;
	analysis->results[analysis_channel].offset = 0.0;
	analysis->results[analysis_channel].slope = 1.0;
	analysis->results[analysis_channel].conversion_factor = 1.0f;
	analysis->results[analysis_channel].cylinder = file->channel_data[analysis->channel].cylinder;
	analysis->results[analysis_channel].samples_per_cycle = 0;

	analysis->results[analysis_channel].file_flag = true;
	analysis->results[analysis_channel].short_name[0] = 0x00;
	analysis->results[analysis_channel].matlab_name[0] = 0x00;

	Copy_Filter_Config(&file->channel_data[channel].filter_config, &analysis->results[analysis_channel].filter_config);
	Copy_Offset_Config(&file->channel_data[channel].offset_config, &analysis->results[analysis_channel].offset_config);
	Copy_SOC_Config(&file->channel_data[channel].soc_config, &analysis->results[analysis_channel].soc_config);
	Copy_Digital_Config(&file->channel_data[channel].digital_config, &analysis->results[analysis_channel].digital_config);
	Copy_Camshaft_Config(&file->channel_data[channel].cam_config, &analysis->results[analysis_channel].cam_config);

	if ((analysis_channel == ANGULAR_TORQUE_SUM) ||
		(analysis_channel == FFT_TORQUE_SUM) ||
		(analysis_channel == FFT_TORQUE_SUM_FREQUENCY) ||
		(analysis_channel == FFT_TOOTH_PERIOD_FREQUENCY))
	{
		snprintf(analysis->results[analysis_channel].name, SIZEOF_CHANNEL_NAME, "%s", get_analysis_name(analysis_channel));
		snprintf(analysis->results[analysis_channel].description, SIZEOF_DESCRIPTION, "%s", get_analysis_description(analysis_channel));
		analysis->results[analysis_channel].tdc_offset = 0.0f;
	}
	else
	{
		snprintf(analysis->results[analysis_channel].name, SIZEOF_CHANNEL_NAME, "%s%u", get_analysis_name(analysis_channel), file->channel_data[analysis->channel].cylinder);
		snprintf(analysis->results[analysis_channel].description, SIZEOF_DESCRIPTION, "%s for Cylinder %u", get_analysis_description(analysis_channel), file->channel_data[analysis->channel].cylinder);
		analysis->results[analysis_channel].tdc_offset = file->channel_data[analysis->channel].tdc_offset;
	}
	
	if (get_analysis_abscissa_units(analysis_channel) == UNITS_AS_CHAN)
	{
		strncpy(analysis->results[analysis_channel].units, file->channel_data[analysis->channel].units, SIZEOF_UNITS);
	}
	else if ((analysis_channel == NET_HEAT_RELEASE_RATE) && (analysis->config.heat_release_model == HR_AVL_THERMO1))
	{
		strncpy(analysis->results[analysis_channel].units, "kJ/m3deg", SIZEOF_UNITS);		/* kJ/m^3/deg but SIZEOF_UNITS = 10, units inline with AVL */
	}
	else if ((analysis_channel == TOTAL_HEAT_RELEASE) && (analysis->config.heat_release_model == HR_AVL_THERMO1))
	{
		strncpy(analysis->results[analysis_channel].units, "kJ/m3", SIZEOF_UNITS);			/* Units inline with AVL */
	}
	else
	{
		strncpy(analysis->results[analysis_channel].units, get_analysis_units(analysis_channel), SIZEOF_UNITS);
	}
	
	analysis->results[analysis_channel].units[SIZEOF_UNITS-1] = 0x00;

	switch (get_analysis_output_abscissa(analysis_channel))
	{
	case ABSCISSA_CYCLE:
	{
		strncpy(analysis->results[analysis_channel].abscissa.units, "cycle", SIZEOF_UNITS);
		analysis->results[analysis_channel].abscissa.type = ABSCISSA_CYCLE;

		break;
	}
	case ABSCISSA_CRANKANGLE:
	{
		strncpy(analysis->results[analysis_channel].abscissa.units, "deg", SIZEOF_UNITS);
		analysis->results[analysis_channel].abscissa.type = ABSCISSA_CRANKANGLE;

		break;
	}
	case ABSCISSA_TIME:
	{
		strncpy(analysis->results[analysis_channel].abscissa.units, "ms", SIZEOF_UNITS);
		analysis->results[analysis_channel].abscissa.type = ABSCISSA_TIME;

		break;
	}
	case ABSCISSA_FREQUENCY:
	{
		strncpy(analysis->results[analysis_channel].abscissa.units, "Hertz", SIZEOF_UNITS);
		analysis->results[analysis_channel].abscissa.type = ABSCISSA_FREQUENCY;

		break;
	}
	case ABSCISSA_HISTOGRAM:
	{
		strncpy(analysis->results[analysis_channel].abscissa.units, "%", SIZEOF_UNITS);
		analysis->results[analysis_channel].abscissa.type = ABSCISSA_HISTOGRAM;

		break;
	}
	default:
	case ABSCISSA_UNKNOWN:
	{
		strncpy(analysis->results[analysis_channel].abscissa.units, "Unknown", SIZEOF_UNITS);
		analysis->results[analysis_channel].abscissa.type = ABSCISSA_UNKNOWN;

		break;
	}
	}

	unsigned int output_abscissa = get_analysis_output_abscissa(analysis_channel);

	if (analysis->analysis_rqst[analysis_channel] == AnalysisRqst::NotCalculated)
	{
		return(true);
	}

	if (file->channel_data[analysis->channel].loaded == false)
	{
		return(true);
	}

	switch (output_abscissa)
	{
	case ABSCISSA_CYCLE:
	{
		analysis->results[analysis_channel].abscissa.number_of_measurement_tables = 1;

		analysis->results[analysis_channel].abscissa.number_of_samples = (unsigned int*)malloc(sizeof(unsigned int));
		if (analysis->results[analysis_channel].abscissa.number_of_samples == NULL)
		{
			Delete_Channel_Data(&analysis->results[analysis_channel]);

			logmessage(WARNING, "Memory could not be allocated\n");
			return(false);
		}

		analysis->results[analysis_channel].abscissa.start = (unsigned int*)malloc(sizeof(unsigned int));
		if (analysis->results[analysis_channel].abscissa.start == NULL)
		{
			Delete_Channel_Data(&analysis->results[analysis_channel]);

			logmessage(WARNING, "Memory could not be allocated\n");
			return(false);
		}

		analysis->results[analysis_channel].abscissa.resolution = (unsigned int*)malloc(sizeof(unsigned int));
		if (analysis->results[analysis_channel].abscissa.resolution == NULL)
		{
			Delete_Channel_Data(&analysis->results[analysis_channel]);

			logmessage(WARNING, "Memory could not be allocated\n");
			return(false);
		}

		analysis->results[analysis_channel].abscissa.number_of_samples[0] = file->number_of_cycles;
		analysis->results[analysis_channel].abscissa.start[0] = 1;
		analysis->results[analysis_channel].abscissa.resolution[0] = 1;

		analysis->results[analysis_channel].data = (float*)malloc(file->number_of_cycles * sizeof(float));

		if (analysis->results[analysis_channel].data == NULL)
		{
			Delete_Channel_Data(&analysis->results[analysis_channel]);

			logmessage(WARNING, "Memory could not be allocated\n");
			return(false);
		}

		memset((void*)analysis->results[analysis_channel].data,0,file->number_of_cycles * sizeof(float));

		analysis->results[analysis_channel].samples_per_cycle = file->number_of_cycles;

		break;
	}
	case ABSCISSA_CRANKANGLE:
	{
		analysis->results[analysis_channel].abscissa.number_of_measurement_tables = file->channel_data[analysis->channel].abscissa.number_of_measurement_tables;

		analysis->results[analysis_channel].abscissa.number_of_samples = (unsigned int*)malloc(file->channel_data[analysis->channel].abscissa.number_of_measurement_tables * sizeof(unsigned int));
		if (analysis->results[analysis_channel].abscissa.number_of_samples == NULL)
		{
			Delete_Channel_Data(&analysis->results[analysis_channel]);

			logmessage(WARNING, "Memory could not be allocated\n");
			return(false);
		}
	
		analysis->results[analysis_channel].abscissa.start = (unsigned int*)malloc(file->channel_data[analysis->channel].abscissa.number_of_measurement_tables * sizeof(unsigned int));
		if (analysis->results[analysis_channel].abscissa.start == NULL)
		{
			Delete_Channel_Data(&analysis->results[analysis_channel]);

			logmessage(WARNING, "Memory could not be allocated\n");
			return(false);
		}
	
		analysis->results[analysis_channel].abscissa.resolution = (unsigned int*)malloc(file->channel_data[analysis->channel].abscissa.number_of_measurement_tables * sizeof(unsigned int));
		if (analysis->results[analysis_channel].abscissa.resolution == NULL)
		{
			Delete_Channel_Data(&analysis->results[analysis_channel]);

			logmessage(WARNING, "Memory could not be allocated\n");
			return(false);
		}
	
		for (measurement_table=0;measurement_table<file->channel_data[analysis->channel].abscissa.number_of_measurement_tables;measurement_table++)
		{
			analysis->results[analysis_channel].abscissa.number_of_samples[measurement_table] = file->channel_data[analysis->channel].abscissa.number_of_samples[measurement_table];
			analysis->results[analysis_channel].abscissa.start[measurement_table] = file->channel_data[analysis->channel].abscissa.start[measurement_table];
			analysis->results[analysis_channel].abscissa.resolution[measurement_table] = file->channel_data[analysis->channel].abscissa.resolution[measurement_table];
		}
	
		if (analysis->analysis_rqst[analysis_channel] >= AnalysisRqst::CalculateAndStore)
		{
			number_of_results_cycles = file->number_of_cycles;
		} 
		else
		{
			number_of_results_cycles = 1;
		}

		analysis->results[analysis_channel].data = (float*)malloc(number_of_results_cycles * file->channel_data[analysis->channel].samples_per_cycle * sizeof(float));

		if (analysis->results[analysis_channel].data == NULL)
		{
			Delete_Channel_Data(&analysis->results[analysis_channel]);

 			logmessage(WARNING, "Memory could not be allocated\n");
			return(false);
		}

		analysis->results[analysis_channel].abscissa.ca_to_theta = file->channel_data[analysis->channel].abscissa.ca_to_theta;
		analysis->results[analysis_channel].abscissa.theta_to_ca = file->channel_data[analysis->channel].abscissa.theta_to_ca;
		analysis->results[analysis_channel].abscissa.ca_to_deg = file->channel_data[analysis->channel].abscissa.ca_to_deg;
		analysis->results[analysis_channel].abscissa.axis = file->channel_data[analysis->channel].abscissa.axis;
		analysis->results[analysis_channel].abscissa.slope = 1.0;
		analysis->results[analysis_channel].abscissa.offset = 0.0;
		analysis->results[analysis_channel].abscissa.axis_type = file->channel_data[analysis->channel].abscissa.axis_type;
		analysis->results[analysis_channel].abscissa.theta_weighting = file->channel_data[analysis->channel].abscissa.theta_weighting;
		analysis->results[analysis_channel].abscissa.theta_weighting_inv = file->channel_data[analysis->channel].abscissa.theta_weighting_inv;
		analysis->results[analysis_channel].abscissa.volume = file->channel_data[analysis->channel].abscissa.volume;

		memset((void*)analysis->results[analysis_channel].data,0, number_of_results_cycles * file->channel_data[analysis->channel].samples_per_cycle * sizeof(float));

		analysis->results[analysis_channel].samples_per_cycle = file->channel_data[analysis->channel].samples_per_cycle;
	
		break;
	}
	case ABSCISSA_TIME:
	{
		analysis->results[analysis_channel].abscissa.number_of_measurement_tables = 1;

		analysis->results[analysis_channel].abscissa.number_of_samples = (unsigned int*)malloc(sizeof(unsigned int));
		if (analysis->results[analysis_channel].abscissa.number_of_samples == NULL)
		{
			Delete_Channel_Data(&analysis->results[analysis_channel]);

			logmessage(WARNING, "Memory could not be allocated\n");
			return(false);
		}

		analysis->results[analysis_channel].abscissa.start = (unsigned int*)malloc(sizeof(unsigned int));
		if (analysis->results[analysis_channel].abscissa.start == NULL)
		{
			Delete_Channel_Data(&analysis->results[analysis_channel]);

			logmessage(WARNING, "Memory could not be allocated\n");
			return(false);
		}

		analysis->results[analysis_channel].abscissa.resolution = (unsigned int*)malloc(sizeof(unsigned int));
		if (analysis->results[analysis_channel].abscissa.resolution == NULL)
		{
			Delete_Channel_Data(&analysis->results[analysis_channel]);

			logmessage(WARNING, "Memory could not be allocated\n");
			return(false);
		}

		analysis->results[analysis_channel].abscissa.number_of_samples[0] = file->channel_data[analysis->channel].abscissa.number_of_samples[0];
		analysis->results[analysis_channel].abscissa.start[0] = file->channel_data[analysis->channel].abscissa.start[0];
		analysis->results[analysis_channel].abscissa.resolution[0] = file->channel_data[analysis->channel].abscissa.resolution[0];

		analysis->results[analysis_channel].data = (float*)malloc(file->channel_data[analysis->channel].samples_per_cycle * sizeof(float));

		if (analysis->results[analysis_channel].data == NULL)
		{
			Delete_Channel_Data(&analysis->results[analysis_channel]);

			logmessage(WARNING, "Memory could not be allocated\n");
			return(false);
		}

		memset((void*)analysis->results[analysis_channel].data, 0, file->channel_data[analysis->channel].samples_per_cycle * sizeof(float));

		analysis->results[analysis_channel].samples_per_cycle = file->channel_data[analysis->channel].samples_per_cycle;

		break;
	}
	case ABSCISSA_FREQUENCY:
	{
		analysis->results[analysis_channel].abscissa.number_of_measurement_tables = 1;

		analysis->results[analysis_channel].abscissa.number_of_samples = (unsigned int*)malloc(sizeof(unsigned int));
		if (analysis->results[analysis_channel].abscissa.number_of_samples == NULL)
		{
			Delete_Channel_Data(&analysis->results[analysis_channel]);

			logmessage(WARNING, "Memory could not be allocated\n");
			return(false);
		}

		analysis->results[analysis_channel].abscissa.start = (unsigned int*)malloc(sizeof(unsigned int));
		if (analysis->results[analysis_channel].abscissa.start == NULL)
		{
			Delete_Channel_Data(&analysis->results[analysis_channel]);

			logmessage(WARNING, "Memory could not be allocated\n");
			return(false);
		}

		analysis->results[analysis_channel].abscissa.resolution = (unsigned int*)malloc(sizeof(unsigned int));
		if (analysis->results[analysis_channel].abscissa.resolution == NULL)
		{
			Delete_Channel_Data(&analysis->results[analysis_channel]);

			logmessage(WARNING, "Memory could not be allocated\n");
			return(false);
		}
		
		switch (analysis_channel)
		{
		case FFT:
		case FFT_FREQUENCY:
		{
			sizeof_fft_analysis = DegreesToCrankAngle(file, analysis->config.fft_finish_window, analysis->channel) - DegreesToCrankAngle(file, analysis->config.fft_start_window, analysis->channel) + 1;

			break;
		}
		case FFT_REFERENCE:
		case FFT_REFERENCE_FREQUENCY:
		{
			sizeof_fft_analysis = DegreesToCrankAngle(file, analysis->config.fft_reference_finish, analysis->channel) - DegreesToCrankAngle(file, analysis->config.fft_reference_start, analysis->channel) + 1;

			break;
		}
		case FFT_KNOCK_PRESSURE:
		case FFT_KNOCK_PRESSURE_FREQUENCY:
		{
			sizeof_fft_analysis = DegreesToCrankAngle(file, analysis->config.pkp_finish_angle, analysis->channel) - DegreesToCrankAngle(file, analysis->config.pkp_start_angle, analysis->channel) + 1;

			break;
		}
		case FFT_TORQUE_SUM:
		case FFT_TORQUE_SUM_FREQUENCY:
		{
			sizeof_fft_analysis = file->channel_data[analysis->channel].samples_per_cycle;

			break;
		}
		case FFT_TOOTH_PERIOD:
		case FFT_TOOTH_PERIOD_FREQUENCY:
		{
			sizeof_fft_analysis = file->channel_data[analysis->channel].samples_per_cycle * NUMBER_OF_TOOTH_PERIOD_FFT_CYCLES;

			break;
		}
		default:
		{
			logmessage(FATAL, "configure_results_channel(): Unknown frequency analysis channel\n");

			break;
		}
		}

		if (sizeof_fft_analysis % 2 == 0)
		{
			sizeof_fft_analysis = sizeof_fft_analysis / 2 - 1;
		}
		else
		{
			sizeof_fft_analysis = (sizeof_fft_analysis - 1) / 2;
		}

		if (sizeof_fft_analysis < 2)
		{
			sizeof_fft_analysis = 2;
		}

		/* We ignore the first (DC) data point so lose one point */

		sizeof_fft_analysis -= 1;

		analysis->results[analysis_channel].abscissa.number_of_samples[0] = sizeof_fft_analysis;
		analysis->results[analysis_channel].abscissa.start[0] = 1;
		analysis->results[analysis_channel].abscissa.resolution[0] = 1;

		if (analysis->analysis_rqst[analysis_channel] >= AnalysisRqst::CalculateAndStore)
		{
			number_of_results_cycles = file->number_of_cycles;
		}
		else
		{
			number_of_results_cycles = 1;
		}

		analysis->results[analysis_channel].data = (float*)malloc(number_of_results_cycles * sizeof_fft_analysis * sizeof(float));

		if (analysis->results[analysis_channel].data == NULL)
		{
			Delete_Channel_Data(&analysis->results[analysis_channel]);

			logmessage(WARNING, "Memory could not be allocated\n");
			return(false);
		}

		if (get_analysis_abscissa_units(analysis_channel) == UNITS_KHZ)
		{
			min = get_analysis_min(analysis_channel);
			range = get_analysis_max(analysis_channel) - get_analysis_min(analysis_channel);

			for (data_pointer = 0; data_pointer < sizeof_fft_analysis; data_pointer++)
			{
				value = min + (float)data_pointer / (float)sizeof_fft_analysis*range;

				for (cycle = 0; cycle < number_of_results_cycles; cycle++)
				{
					analysis->results[analysis_channel].data[cycle*sizeof_fft_analysis + data_pointer] = value;
				}
			}
		}
		else
		{
			memset((void*)analysis->results[analysis_channel].data, 0, number_of_results_cycles* sizeof_fft_analysis * sizeof(float));
		}

		analysis->results[analysis_channel].samples_per_cycle = sizeof_fft_analysis;

		break;
	}
	case ABSCISSA_HISTOGRAM:
	{
		analysis->results[analysis_channel].abscissa.number_of_measurement_tables = 1;

		analysis->results[analysis_channel].abscissa.number_of_samples = (unsigned int*)malloc(sizeof(unsigned int));
		if (analysis->results[analysis_channel].abscissa.number_of_samples == NULL)
		{
			Delete_Channel_Data(&analysis->results[analysis_channel]);

			logmessage(WARNING, "Memory could not be allocated\n");
			return(false);
		}

		analysis->results[analysis_channel].abscissa.start = (unsigned int*)malloc(sizeof(unsigned int));
		if (analysis->results[analysis_channel].abscissa.start == NULL)
		{
			Delete_Channel_Data(&analysis->results[analysis_channel]);

			logmessage(WARNING, "Memory could not be allocated\n");
			return(false);
		}

		analysis->results[analysis_channel].abscissa.resolution = (unsigned int*)malloc(sizeof(unsigned int));
		if (analysis->results[analysis_channel].abscissa.resolution == NULL)
		{
			Delete_Channel_Data(&analysis->results[analysis_channel]);

			logmessage(WARNING, "Memory could not be allocated\n");
			return(false);
		}

		sizeof_histogram_analysis = KHI_MAX_KNKMAX * KHI_RESOLUTION;

		analysis->results[analysis_channel].abscissa.number_of_samples[0] = sizeof_histogram_analysis;
		analysis->results[analysis_channel].abscissa.start[0] = 0;
		analysis->results[analysis_channel].abscissa.resolution[0] = 1;

		if (analysis->analysis_rqst[analysis_channel] >= AnalysisRqst::CalculateAndStore)
		{
			number_of_results_cycles = file->number_of_cycles;
		}
		else
		{
			number_of_results_cycles = 1;
		}

		analysis->results[analysis_channel].data = (float*)malloc(number_of_results_cycles * sizeof_histogram_analysis * sizeof(float));

		if (analysis->results[analysis_channel].data == NULL)
		{
			Delete_Channel_Data(&analysis->results[analysis_channel]);

			logmessage(WARNING, "Memory could not be allocated\n");
			return(false);
		}

		memset((void*)analysis->results[analysis_channel].data, 0, number_of_results_cycles * sizeof_histogram_analysis * sizeof(float));

		analysis->results[analysis_channel].samples_per_cycle = sizeof_histogram_analysis;

		break;
	}
	default:
	case ABSCISSA_UNKNOWN:
	{
		logmessage(WARNING, "Don't know how to initialise unknown abscissa type (%u)\n", output_abscissa);

		result = false;

		break;
	}
	}

	return(result);
}

bool configure_analysis_channels(const FileData* file, AnalysisChannel* analysis)
{
	unsigned int analysis_channel;
	unsigned int number_of_analysis_channels = get_number_of_analysis_channels();
	bool result = true;

	if ((file == NULL) || (analysis == NULL))
	{
		return(false);
	}

	analysis->cycle_classification = NULL;

	logmessage(DEBUG, "Allocating analysis channel memory (%s)...\n", file->channel_data[analysis->channel].name);

	analysis->results = (ChannelData*)malloc(number_of_analysis_channels * sizeof(ChannelData));
	if (analysis->results == NULL)
	{
		logmessage(MSG_ERROR, "Memory could not be allocated\n");
		return(false);
	}

	analysis->statistics = (Statistics*)malloc(number_of_analysis_channels * sizeof(Statistics));
	if (analysis->statistics == NULL)
	{
		free(analysis->results);
		analysis->results = NULL;

		logmessage(MSG_ERROR, "Memory could not be allocated\n");
		return(false);
	}

	analysis->ca_statistics = (StatisticsArray*)malloc(number_of_analysis_channels * sizeof(StatisticsArray));
	if (analysis->ca_statistics == NULL)
	{
		free(analysis->results);
		analysis->results = NULL;

		free(analysis->statistics);
		analysis->statistics = NULL;

		logmessage(MSG_ERROR, "Memory could not be allocated\n");
		return(false);
	}

	logmessage(DEBUG, "Configuring analysis channels...\n");

	for (analysis_channel = 0; analysis_channel < number_of_analysis_channels; analysis_channel++)
	{
		if (configure_results_channel(file, analysis, analysis_channel) == false)
		{
			/* It has not been possible to allocated enough memory for this analysis so prevent it from being calculated */

			logmessage(WARNING, "Cannot analyse channel %s.%s\n", file->channel_data[analysis->channel].name, get_analysis_name(analysis_channel));

			result = false;
		}
	}

	if (file->channel_data[analysis->channel].abscissa.type == ABSCISSA_CRANKANGLE)
	{
		logmessage(DEBUG, "Configuring cycle classifications...\n");

		analysis->cycle_classification = (unsigned char*)malloc(file->number_of_cycles * sizeof(unsigned char));
		if (analysis->cycle_classification == NULL)
		{
			logmessage(MSG_ERROR, "Memory could not be allocated\n");
			return(false);
		}

		memset((void*)analysis->cycle_classification, 0, file->number_of_cycles * sizeof(unsigned char));

		logmessage(DEBUG, "Configuring crank angle statistics...\n");

		for (analysis_channel = 0; analysis_channel < number_of_analysis_channels; analysis_channel++)
		{
			if (get_analysis_output_abscissa(analysis_channel) == ABSCISSA_CRANKANGLE)
			{
				Allocate_StatisticsArray(&analysis->ca_statistics[analysis_channel], file->channel_data[analysis->channel].samples_per_cycle);
			}
			else if (get_analysis_output_abscissa(analysis_channel) == ABSCISSA_FREQUENCY)
			{
				unsigned int number_of_bins = (unsigned int)(analysis->config.fft_stats_maximum / analysis->config.fft_stats_resolution) + 1;

				Allocate_StatisticsArray(&analysis->ca_statistics[analysis_channel], number_of_bins);
			}
			else
			{
				/* Do Nothing */
			}
		}
	}

	logmessage(DEBUG, "Configuration complete\n");

	return(result);
}

//#define CALCULATE_CHANNEL(analysis_rqst, analysis_type) if ((get_analysis_output_abscissa(analysis_type) == ABSCISSA_CRANKANGLE) || (get_analysis_output_abscissa(analysis_type) == ABSCISSA_FREQUENCY)) { if (analysis_rqst[(analysis_type)] < 1) { analysis_rqst[(analysis_type)] = 1; } } else { if (analysis_rqst[(analysis_type)] < 2) { analysis_rqst[(analysis_type)] = 2; } }
#define CALCULATE_CHANNEL(analysis_rqst, analysis_type) if (analysis_rqst[(analysis_type)] == AnalysisRqst::NotCalculated) { analysis_rqst[(analysis_type)] = AnalysisRqst::TemporaryCalculation; }

bool calculate_dependencies(AnalysisRqst* analysis_rqst, const unsigned int abscissa_type, const unsigned int channel_type, const unsigned int cylinder, const unsigned int last_cylinder, const bool restrike_analysis)
{
	if (analysis_rqst == NULL)
	{
		return(false);
	}

	/* Check for analysis dependencies */

	if (abscissa_type == ABSCISSA_CRANKANGLE)
	{
		if (get_analysis_channel_validity(PRESSURE_OFFSET, channel_type) == 1)
		{
			//analysis_rqst[PRESSURE_OFFSET] = AnalysisRqst::CalculateAndStore;
		}

		if (channel_type == CHAN_TOOTH_PERIODS)
		{
			analysis_rqst[TOOTH_SPEED] = AnalysisRqst::CalculateAndStore;

			analysis_rqst[ENGINE_SPEED] = AnalysisRqst::CalculateAndStore;
			analysis_rqst[FFT] = AnalysisRqst::CalculateAndStore;
			analysis_rqst[FFT_TOOTH_PERIOD] = AnalysisRqst::CalculateAndStore;
		}
	}

	/* Check for analysis dependencies */

#ifdef __CATOOLRT__

	if ((analysis_rqst[NORMALISED_DIFF_PR_50]) ||
		(analysis_rqst[PRESSURE_RATIO_MGMT_10]) ||
		(analysis_rqst[PRESSURE_RATIO_MGMT_25]) ||
		(analysis_rqst[DILPAR]) ||
		(analysis_rqst[DIFFERENCE_PRESSURE_10]) ||
		(analysis_rqst[DIFFERENCE_PRESSURE_INTEGRAL]) ||
		(analysis_rqst[DIFFERENCE_PRESSURE]) ||
		(analysis_rqst[PRESSURE_RATIO]) ||
		(analysis_rqst[PRESSURE_RATIO_FRACTION]))
	{
		CALCULATE_CHANNEL(analysis_rqst, NORMALISED_DIFF_PR_50);
		CALCULATE_CHANNEL(analysis_rqst, PRESSURE_RATIO_MGMT_10);
		CALCULATE_CHANNEL(analysis_rqst, PRESSURE_RATIO_MGMT_25);
		CALCULATE_CHANNEL(analysis_rqst, DILPAR);
		CALCULATE_CHANNEL(analysis_rqst, DIFFERENCE_PRESSURE_10);
		CALCULATE_CHANNEL(analysis_rqst, DIFFERENCE_PRESSURE_INTEGRAL);
		CALCULATE_CHANNEL(analysis_rqst, DIFFERENCE_PRESSURE);
		CALCULATE_CHANNEL(analysis_rqst, PRESSURE_RATIO);
		CALCULATE_CHANNEL(analysis_rqst, PRESSURE_RATIO_FRACTION);

		/* Dependencies */

		CALCULATE_CHANNEL(analysis_rqst, MOTORED_PRESSURE);
		CALCULATE_CHANNEL(analysis_rqst, MOVING_PRESSURE_AVERAGE);
	}

	if ((analysis_rqst[EEVO]) ||
		(analysis_rqst[EIVC]) ||
		(analysis_rqst[EXTRAPOLATED_CYL_PRESSURE]))
	{
		CALCULATE_CHANNEL(analysis_rqst, EEVO);
		CALCULATE_CHANNEL(analysis_rqst, EIVC);
		CALCULATE_CHANNEL(analysis_rqst, EXTRAPOLATED_CYL_PRESSURE);

		/* Dependencies */

		CALCULATE_CHANNEL(analysis_rqst, POLYTROPIC_INDICES);
		CALCULATE_CHANNEL(analysis_rqst, POLY_COMP);
		CALCULATE_CHANNEL(analysis_rqst, POLY_EXP);
		CALCULATE_CHANNEL(analysis_rqst, MOVING_PRESSURE_AVERAGE);
	}

	if ((analysis_rqst[DELTA_P]) ||
		(analysis_rqst[TRAPPED_MASS]) ||
		(analysis_rqst[LOAD]))
	{

		CALCULATE_CHANNEL(analysis_rqst, DELTA_P);
		CALCULATE_CHANNEL(analysis_rqst, TRAPPED_MASS);
		CALCULATE_CHANNEL(analysis_rqst, LOAD);

		/* Dependencies */

		CALCULATE_CHANNEL(analysis_rqst, POLY_COMP);
		CALCULATE_CHANNEL(analysis_rqst, MOVING_PRESSURE_AVERAGE);
	}

	if ((analysis_rqst[MAX_VALVE_ACCEL]) ||
		(analysis_rqst[MAX_VALVE_ACCEL_CA]) ||
		(analysis_rqst[MAX_VALVE_LIFT]) ||
		(analysis_rqst[MAX_VALVE_LIFT_CA]) ||
		(analysis_rqst[MAX_VALVE_VELOCITY]) ||
		(analysis_rqst[MAX_VALVE_VELOCITY_CA]))
	{
		CALCULATE_CHANNEL(analysis_rqst, MAX_VALVE_ACCEL);
		CALCULATE_CHANNEL(analysis_rqst, MAX_VALVE_ACCEL_CA);
		CALCULATE_CHANNEL(analysis_rqst, MAX_VALVE_LIFT);
		CALCULATE_CHANNEL(analysis_rqst, MAX_VALVE_LIFT_CA);
		CALCULATE_CHANNEL(analysis_rqst, MAX_VALVE_VELOCITY);
		CALCULATE_CHANNEL(analysis_rqst, MAX_VALVE_VELOCITY_CA);

		/* Dependencies */

		CALCULATE_CHANNEL(analysis_rqst, VALVE_ACCEL);
		CALCULATE_CHANNEL(analysis_rqst, VALVE_VELOCITY);
	}

	if ((analysis_rqst[VALVE_ACCEL]) ||
		(analysis_rqst[VALVE_VELOCITY]))
	{
		CALCULATE_CHANNEL(analysis_rqst, VALVE_ACCEL);
		CALCULATE_CHANNEL(analysis_rqst, VALVE_VELOCITY);
	}

	if ((analysis_rqst[MOTOR_ALPHA]) ||
		(analysis_rqst[MOTOR_BETA]) ||
		(analysis_rqst[MOTOR_D]) ||
		(analysis_rqst[MOTOR_Q]))
	{
		CALCULATE_CHANNEL(analysis_rqst, MOTOR_ALPHA);
		CALCULATE_CHANNEL(analysis_rqst, MOTOR_BETA);
		CALCULATE_CHANNEL(analysis_rqst, MOTOR_D);
		CALCULATE_CHANNEL(analysis_rqst, MOTOR_Q);
	}
#else
		analysis_rqst[NORMALISED_DIFF_PR_50] = AnalysisRqst::NotCalculated;
		analysis_rqst[PRESSURE_RATIO_MGMT_10] = AnalysisRqst::NotCalculated;
		analysis_rqst[PRESSURE_RATIO_MGMT_25] = AnalysisRqst::NotCalculated;
		analysis_rqst[DILPAR] = AnalysisRqst::NotCalculated;
		analysis_rqst[DIFFERENCE_PRESSURE_10] = AnalysisRqst::NotCalculated;
		analysis_rqst[DIFFERENCE_PRESSURE_INTEGRAL] = AnalysisRqst::NotCalculated;

		analysis_rqst[DIFFERENCE_PRESSURE] = AnalysisRqst::NotCalculated;
		analysis_rqst[PRESSURE_RATIO] = AnalysisRqst::NotCalculated;
		analysis_rqst[PRESSURE_RATIO_FRACTION] = AnalysisRqst::NotCalculated;

		analysis_rqst[EEVO] = AnalysisRqst::NotCalculated;
		analysis_rqst[EIVC] = AnalysisRqst::NotCalculated;
		analysis_rqst[EXTRAPOLATED_CYL_PRESSURE] = AnalysisRqst::NotCalculated;

		analysis_rqst[DELTA_P] = AnalysisRqst::NotCalculated;
		analysis_rqst[TRAPPED_MASS] = AnalysisRqst::NotCalculated;
		analysis_rqst[LOAD] = AnalysisRqst::NotCalculated;

		analysis_rqst[MAX_VALVE_ACCEL] = AnalysisRqst::NotCalculated;
		analysis_rqst[MAX_VALVE_ACCEL_CA] = AnalysisRqst::NotCalculated;
		analysis_rqst[MAX_VALVE_LIFT] = AnalysisRqst::NotCalculated;
		analysis_rqst[MAX_VALVE_LIFT_CA] = AnalysisRqst::NotCalculated;
		analysis_rqst[MAX_VALVE_VELOCITY] = AnalysisRqst::NotCalculated;
		analysis_rqst[MAX_VALVE_VELOCITY_CA] = AnalysisRqst::NotCalculated;

		analysis_rqst[VALVE_ACCEL] = AnalysisRqst::NotCalculated;
		analysis_rqst[VALVE_VELOCITY] = AnalysisRqst::NotCalculated;

		analysis_rqst[MOTOR_ALPHA] = AnalysisRqst::NotCalculated;
		analysis_rqst[MOTOR_BETA] = AnalysisRqst::NotCalculated;
		analysis_rqst[MOTOR_D] = AnalysisRqst::NotCalculated;
		analysis_rqst[MOTOR_Q] = AnalysisRqst::NotCalculated;
#endif

	if ((analysis_rqst[INDICATED_POWER]) ||
		(analysis_rqst[INDICATED_POWER_HP]) ||
		(analysis_rqst[INDICATED_TORQUE_LBFT]))
	{
		CALCULATE_CHANNEL(analysis_rqst, INDICATED_POWER);
		CALCULATE_CHANNEL(analysis_rqst, INDICATED_POWER_HP);
		CALCULATE_CHANNEL(analysis_rqst, INDICATED_TORQUE_LBFT);

		CALCULATE_CHANNEL(analysis_rqst, INDICATED_TORQUE);
	}

	if ((analysis_rqst[ANGULAR_TORQUE_SUM]) ||
		(analysis_rqst[FFT_TORQUE_SUM]))
	{
		if (cylinder == last_cylinder)
		{
			CALCULATE_CHANNEL(analysis_rqst, ANGULAR_TORQUE_SUM);
			CALCULATE_CHANNEL(analysis_rqst, FFT_TORQUE_SUM);

			/* Dependencies */

			CALCULATE_CHANNEL(analysis_rqst, ANGULAR_TORQUE);
		}
		else
		{
			analysis_rqst[ANGULAR_TORQUE_SUM] = AnalysisRqst::NotCalculated;
			analysis_rqst[FFT_TORQUE_SUM] = AnalysisRqst::NotCalculated;
		}
	}

	if ((analysis_rqst[MISFIRE_F]) ||
		(analysis_rqst[MISFIRE_M]) ||
		(analysis_rqst[MISFIRE_S]))
	{
		CALCULATE_CHANNEL(analysis_rqst, MISFIRE_F);
		CALCULATE_CHANNEL(analysis_rqst, MISFIRE_M);
		CALCULATE_CHANNEL(analysis_rqst, MISFIRE_S);

		/* Dependenices */

		CALCULATE_CHANNEL(analysis_rqst, GROSS_IMEP);
	}

	if ((analysis_rqst[GROSS_IMEP]) ||
		(analysis_rqst[NET_IMEP]) ||
		(analysis_rqst[UPPER_PUMPING_IMEP]) ||
		(analysis_rqst[LOWER_PUMPING_IMEP]) ||
		(analysis_rqst[PUMPING_IMEP]) ||
		(analysis_rqst[INDICATED_TORQUE]) ||
		(analysis_rqst[INDICATED_TORQUE_NET]) ||
		(analysis_rqst[ANGULAR_TORQUE]) ||
		(analysis_rqst[CRANK_ANGLE_IMEP]))
	{
		/* No additional dependencies but if one is set then the others must */

		CALCULATE_CHANNEL(analysis_rqst, GROSS_IMEP);
		CALCULATE_CHANNEL(analysis_rqst, NET_IMEP);
		CALCULATE_CHANNEL(analysis_rqst, UPPER_PUMPING_IMEP);
		CALCULATE_CHANNEL(analysis_rqst, LOWER_PUMPING_IMEP);
		CALCULATE_CHANNEL(analysis_rqst, PUMPING_IMEP);
		CALCULATE_CHANNEL(analysis_rqst, INDICATED_TORQUE);
		CALCULATE_CHANNEL(analysis_rqst, INDICATED_TORQUE_NET);
		CALCULATE_CHANNEL(analysis_rqst, ANGULAR_TORQUE);
		CALCULATE_CHANNEL(analysis_rqst, CRANK_ANGLE_IMEP);
	}

	if ((analysis_rqst[KNOCK_BOSS_FACTOR]) ||
		(analysis_rqst[KNOCK_BOSS_INTEGRAL]))
	{
		/* No additional dependencies but if one is set then the other must */

		CALCULATE_CHANNEL(analysis_rqst, KNOCK_BOSS_FACTOR);
		CALCULATE_CHANNEL(analysis_rqst, KNOCK_BOSS_INTEGRAL);
	}

	if ((analysis_rqst[TLA]) ||
		(analysis_rqst[POLYFIT]))
	{
		CALCULATE_CHANNEL(analysis_rqst, TLA);
		CALCULATE_CHANNEL(analysis_rqst, POLYFIT);

		/* Dependencies */

		CALCULATE_CHANNEL(analysis_rqst, MAX_PRESSURE_CA);
		CALCULATE_CHANNEL(analysis_rqst, MOVING_PRESSURE_AVERAGE);
	}

	if ((analysis_rqst[KNOCK_C_AND_D]) ||
		(analysis_rqst[KNOCK_C_AND_D_CA]))
	{
		CALCULATE_CHANNEL(analysis_rqst, KNOCK_C_AND_D);
		CALCULATE_CHANNEL(analysis_rqst, KNOCK_C_AND_D_CA);
	}

	if (analysis_rqst[KNOCK_FACTOR])
	{
		/* Dependencies */

		CALCULATE_CHANNEL(analysis_rqst, MAX_PRESSURE_CA);
		CALCULATE_CHANNEL(analysis_rqst, KNOCKING_PRESSURE);
	}

	if ((analysis_rqst[KNOCK_PRESSURE_HISTOGRAM]) ||
		(analysis_rqst[KNOCK_PRESSURE_HIST_WARNING]) ||
		(analysis_rqst[KNOCK_PRESSURE_HIST_ALARM]) ||
		(analysis_rqst[KNOCK_PRESSURE_HIST_SHUTDOWN]) ||
		(analysis_rqst[KNOCK_HISTOGRAM_INDEX]))
	{
		CALCULATE_CHANNEL(analysis_rqst, KNOCK_PRESSURE_HISTOGRAM);
		CALCULATE_CHANNEL(analysis_rqst, KNOCK_PRESSURE_HIST_WARNING);
		CALCULATE_CHANNEL(analysis_rqst, KNOCK_PRESSURE_HIST_ALARM);
		CALCULATE_CHANNEL(analysis_rqst, KNOCK_PRESSURE_HIST_SHUTDOWN);
		CALCULATE_CHANNEL(analysis_rqst, KNOCK_HISTOGRAM_INDEX);

		/* Dependencies */

		CALCULATE_CHANNEL(analysis_rqst, KNOCKING_PRESSURE);
	}

	if (analysis_rqst[KNOCK_ONSET_UMFB])
	{
		CALCULATE_CHANNEL(analysis_rqst, KNOCK_ONSET_CA);
	}

	if (analysis_rqst[KNOCK_SIGNAL_ENERGY])
	{
		CALCULATE_CHANNEL(analysis_rqst, KNOCKING_PRESSURE);
		CALCULATE_CHANNEL(analysis_rqst, KNOCK_ONSET_CA);
	}

	if (analysis_rqst[FFT_KNOCK_PRESSURE])
	{
		/* Dependencies */

		CALCULATE_CHANNEL(analysis_rqst, KNOCKING_PRESSURE);
	}

	if (analysis_rqst[KNOCKING_PRESSURE])
	{
		/* Dependencies */

		CALCULATE_CHANNEL(analysis_rqst, MOVING_PRESSURE_AVERAGE);
		CALCULATE_CHANNEL(analysis_rqst, PEAK_KNOCKING_PRESSURE);
	}

	if ((analysis_rqst[PEAK_KNOCKING_PRESSURE]) ||
		(analysis_rqst[PEAK_KNOCKING_PRESSURE_CA]) ||
		(analysis_rqst[KNOCK_INTEGRAL]) ||
		(analysis_rqst[KNOCK_ONSET_CA]))
	{
		CALCULATE_CHANNEL(analysis_rqst, PEAK_KNOCKING_PRESSURE);
		CALCULATE_CHANNEL(analysis_rqst, PEAK_KNOCKING_PRESSURE_CA);
		CALCULATE_CHANNEL(analysis_rqst, KNOCK_INTEGRAL);
		CALCULATE_CHANNEL(analysis_rqst, KNOCK_ONSET_CA);
		CALCULATE_CHANNEL(analysis_rqst, FFT_KNOCK_PRESSURE);

		/* Dependencies */

		CALCULATE_CHANNEL(analysis_rqst, MOVING_PRESSURE_AVERAGE);
		CALCULATE_CHANNEL(analysis_rqst, KNOCKING_PRESSURE);
	}

	if (analysis_rqst[MOTORED_PRESSURE])
	{
		/* Dependencies */

		CALCULATE_CHANNEL(analysis_rqst, POLY_COMP);
		CALCULATE_CHANNEL(analysis_rqst, MOVING_PRESSURE_AVERAGE);
	}

	if (analysis_rqst[CENTRE_OF_GRAVITY])
	{
		/* Dependencies */

		CALCULATE_CHANNEL(analysis_rqst, BURN_ANGLE_10);
		CALCULATE_CHANNEL(analysis_rqst, BURN_ANGLE_90);
		CALCULATE_CHANNEL(analysis_rqst, MFB);
	}

	if ((analysis_rqst[WIEBE_A]) ||
		(analysis_rqst[WIEBE_M]) ||
		(analysis_rqst[WIEBE_MFB]))
	{
		CALCULATE_CHANNEL(analysis_rqst, WIEBE_A);
		CALCULATE_CHANNEL(analysis_rqst, WIEBE_M);
		CALCULATE_CHANNEL(analysis_rqst, WIEBE_MFB);

		/* Dependencies */

		CALCULATE_CHANNEL(analysis_rqst, MFB);
		CALCULATE_CHANNEL(analysis_rqst, BURN_ANGLE_5);
		CALCULATE_CHANNEL(analysis_rqst, BURN_ANGLE_10);
		CALCULATE_CHANNEL(analysis_rqst, BURN_ANGLE_50);
		CALCULATE_CHANNEL(analysis_rqst, BURN_ANGLE_90);
		CALCULATE_CHANNEL(analysis_rqst, BURN_ANGLE_95);
		CALCULATE_CHANNEL(analysis_rqst, START_OF_COMBUSTION);
		CALCULATE_CHANNEL(analysis_rqst, EEOC);
	}

	if ((analysis_rqst[BURN_ANGLE_1]) ||
		(analysis_rqst[BURN_ANGLE_2]) ||
		(analysis_rqst[BURN_ANGLE_5]) ||
		(analysis_rqst[BURN_ANGLE_10]) ||
		(analysis_rqst[BURN_ANGLE_20]) ||
		(analysis_rqst[BURN_ANGLE_25]) ||
		(analysis_rqst[BURN_ANGLE_50]) ||
		(analysis_rqst[BURN_ANGLE_75]) ||
		(analysis_rqst[BURN_ANGLE_80]) ||
		(analysis_rqst[BURN_ANGLE_90]) ||
		(analysis_rqst[BURN_ANGLE_95]) ||
		(analysis_rqst[BURN_ANGLE_98]) ||
		(analysis_rqst[BURN_ANGLE_99]) ||
		(analysis_rqst[BURN_DURATION_0_2]) ||
		(analysis_rqst[BURN_DURATION_0_5]) ||
		(analysis_rqst[BURN_DURATION_0_10]) ||
		(analysis_rqst[BURN_DURATION_0_90]) ||
		(analysis_rqst[BURN_DURATION_2_90]) ||
		(analysis_rqst[BURN_DURATION_5_90]) ||
		(analysis_rqst[BURN_DURATION_10_90]) ||
		(analysis_rqst[KNOCK_ONSET_UMFB]))
	{
		CALCULATE_CHANNEL(analysis_rqst, BURN_ANGLE_1);
		CALCULATE_CHANNEL(analysis_rqst, BURN_ANGLE_2);
		CALCULATE_CHANNEL(analysis_rqst, BURN_ANGLE_5);
		CALCULATE_CHANNEL(analysis_rqst, BURN_ANGLE_10);
		CALCULATE_CHANNEL(analysis_rqst, BURN_ANGLE_20);
		CALCULATE_CHANNEL(analysis_rqst, BURN_ANGLE_25);
		CALCULATE_CHANNEL(analysis_rqst, BURN_ANGLE_50);
		CALCULATE_CHANNEL(analysis_rqst, BURN_ANGLE_75);
		CALCULATE_CHANNEL(analysis_rqst, BURN_ANGLE_80);
		CALCULATE_CHANNEL(analysis_rqst, BURN_ANGLE_90);
		CALCULATE_CHANNEL(analysis_rqst, BURN_ANGLE_95);
		CALCULATE_CHANNEL(analysis_rqst, BURN_ANGLE_98);
		CALCULATE_CHANNEL(analysis_rqst, BURN_ANGLE_99);
		CALCULATE_CHANNEL(analysis_rqst, BURN_DURATION_0_2);
		CALCULATE_CHANNEL(analysis_rqst, BURN_DURATION_0_5);
		CALCULATE_CHANNEL(analysis_rqst, BURN_DURATION_0_10);
		CALCULATE_CHANNEL(analysis_rqst, BURN_DURATION_0_90);
		CALCULATE_CHANNEL(analysis_rqst, BURN_DURATION_2_90);
		CALCULATE_CHANNEL(analysis_rqst, BURN_DURATION_5_90);
		CALCULATE_CHANNEL(analysis_rqst, BURN_DURATION_10_90);
		CALCULATE_CHANNEL(analysis_rqst, KNOCK_ONSET_UMFB);

		/* Dependencies */

		CALCULATE_CHANNEL(analysis_rqst, MFB);
		CALCULATE_CHANNEL(analysis_rqst, EEOC);
		CALCULATE_CHANNEL(analysis_rqst, START_OF_COMBUSTION);
		CALCULATE_CHANNEL(analysis_rqst, MOVING_PRESSURE_AVERAGE);
		CALCULATE_CHANNEL(analysis_rqst, KNOCK_ONSET_CA);
	}

	if ((analysis_rqst[MAX_BURN_RATE]) ||
		(analysis_rqst[MAX_BURN_RATE_CA]) ||
		(analysis_rqst[MFB]))
	{
		CALCULATE_CHANNEL(analysis_rqst, MAX_BURN_RATE);
		CALCULATE_CHANNEL(analysis_rqst, MAX_BURN_RATE_CA);
		CALCULATE_CHANNEL(analysis_rqst, MFB);

		/* Dependencies */

		CALCULATE_CHANNEL(analysis_rqst, EEOC);
		CALCULATE_CHANNEL(analysis_rqst, START_OF_COMBUSTION);
		CALCULATE_CHANNEL(analysis_rqst, MOVING_PRESSURE_AVERAGE);
	}

	if ((analysis_rqst[MAX_HEAT_RELEASE_RATE]) ||
		(analysis_rqst[MAX_HEAT_RELEASE_RATE_CA]) ||
		(analysis_rqst[TOTAL_HEAT_RELEASE]) ||
		(analysis_rqst[NET_HEAT_RELEASE_RATE]) ||
		(analysis_rqst[GROSS_HEAT_RELEASE_RATE]) ||
		(analysis_rqst[H_COEFF]))
	{
		CALCULATE_CHANNEL(analysis_rqst, MAX_HEAT_RELEASE_RATE);
		CALCULATE_CHANNEL(analysis_rqst, MAX_HEAT_RELEASE_RATE_CA);
		CALCULATE_CHANNEL(analysis_rqst, TOTAL_HEAT_RELEASE);
		CALCULATE_CHANNEL(analysis_rqst, NET_HEAT_RELEASE_RATE);
		CALCULATE_CHANNEL(analysis_rqst, GROSS_HEAT_RELEASE_RATE);
		CALCULATE_CHANNEL(analysis_rqst, H_COEFF);

		/* Dependencies */

		CALCULATE_CHANNEL(analysis_rqst, MEAN_GAS_TEMP);
		CALCULATE_CHANNEL(analysis_rqst, GAMMA);
		CALCULATE_CHANNEL(analysis_rqst, EEOC);
		CALCULATE_CHANNEL(analysis_rqst, START_OF_COMBUSTION);
		CALCULATE_CHANNEL(analysis_rqst, POLY_COMP);
		CALCULATE_CHANNEL(analysis_rqst, POLY_EXP);
	}

	if (analysis_rqst[GAMMA])
	{
		/* Dependencies */

		CALCULATE_CHANNEL(analysis_rqst, MEAN_GAS_TEMP);
		CALCULATE_CHANNEL(analysis_rqst, MOVING_PRESSURE_AVERAGE);
		CALCULATE_CHANNEL(analysis_rqst, POLYTROPIC_INDICES);
		CALCULATE_CHANNEL(analysis_rqst, POLY_COMP);
		CALCULATE_CHANNEL(analysis_rqst, POLY_EXP);
	}

	analysis_rqst[FFT_FREQUENCY] = analysis_rqst[FFT];
	analysis_rqst[FFT_REFERENCE_FREQUENCY] = analysis_rqst[FFT_REFERENCE];
	analysis_rqst[FFT_KNOCK_PRESSURE_FREQUENCY] = analysis_rqst[FFT_KNOCK_PRESSURE];
	analysis_rqst[FFT_TORQUE_SUM_FREQUENCY] = analysis_rqst[FFT_TORQUE_SUM];
	analysis_rqst[FFT_TOOTH_PERIOD_FREQUENCY] = analysis_rqst[FFT_TOOTH_PERIOD];

	if (analysis_rqst[TLA])
	{
		CALCULATE_CHANNEL(analysis_rqst, MAX_PRESSURE_CA);
	}

	if ((analysis_rqst[CAM_ADVANCE]) ||
		(analysis_rqst[CAM_ANGLE]))
	{
		CALCULATE_CHANNEL(analysis_rqst, CAM_ADVANCE);
		CALCULATE_CHANNEL(analysis_rqst, CAM_ANGLE);

		/* Dependencies */

		CALCULATE_CHANNEL(analysis_rqst, CAM_EDGE_1);
	}

	if ((analysis_rqst[START_OF_INJECTION_1]) ||
		(analysis_rqst[START_OF_INJECTION_2]) ||
		(analysis_rqst[START_OF_INJECTION_3]) ||
		(analysis_rqst[START_OF_INJECTION_4]) ||
		(analysis_rqst[START_OF_INJECTION_5]) ||
		(analysis_rqst[START_OF_INJECTION_6]) ||
		(analysis_rqst[END_OF_INJECTION_1]) ||
		(analysis_rqst[END_OF_INJECTION_2]) ||
		(analysis_rqst[END_OF_INJECTION_3]) ||
		(analysis_rqst[END_OF_INJECTION_4]) ||
		(analysis_rqst[END_OF_INJECTION_5]) ||
		(analysis_rqst[END_OF_INJECTION_6]) ||
		(analysis_rqst[NUMBER_OF_INJECTIONS]) ||
		(analysis_rqst[INJECTOR_DURATION_1]) ||
		(analysis_rqst[INJECTOR_DURATION_2]) ||
		(analysis_rqst[INJECTOR_DURATION_3]) ||
		(analysis_rqst[INJECTOR_DURATION_4]) ||
		(analysis_rqst[INJECTOR_DURATION_5]) ||
		(analysis_rqst[INJECTOR_DURATION_6]) ||
		(analysis_rqst[INJECTOR_SEPARATION_1]) ||
		(analysis_rqst[INJECTOR_SEPARATION_2]) ||
		(analysis_rqst[INJECTOR_SEPARATION_3]) ||
		(analysis_rqst[INJECTOR_SEPARATION_4]) ||
		(analysis_rqst[INJECTOR_SEPARATION_5]))
	{
		CALCULATE_CHANNEL(analysis_rqst, START_OF_INJECTION_1);
		CALCULATE_CHANNEL(analysis_rqst, START_OF_INJECTION_2);
		CALCULATE_CHANNEL(analysis_rqst, START_OF_INJECTION_3);
		CALCULATE_CHANNEL(analysis_rqst, START_OF_INJECTION_4);
		CALCULATE_CHANNEL(analysis_rqst, START_OF_INJECTION_5);
		CALCULATE_CHANNEL(analysis_rqst, START_OF_INJECTION_6);
		CALCULATE_CHANNEL(analysis_rqst, END_OF_INJECTION_1);
		CALCULATE_CHANNEL(analysis_rqst, END_OF_INJECTION_2);
		CALCULATE_CHANNEL(analysis_rqst, END_OF_INJECTION_3);
		CALCULATE_CHANNEL(analysis_rqst, END_OF_INJECTION_4);
		CALCULATE_CHANNEL(analysis_rqst, END_OF_INJECTION_5);
		CALCULATE_CHANNEL(analysis_rqst, END_OF_INJECTION_6);
		CALCULATE_CHANNEL(analysis_rqst, NUMBER_OF_INJECTIONS);
		CALCULATE_CHANNEL(analysis_rqst, INJECTOR_DURATION_1);
		CALCULATE_CHANNEL(analysis_rqst, INJECTOR_DURATION_2);
		CALCULATE_CHANNEL(analysis_rqst, INJECTOR_DURATION_3);
		CALCULATE_CHANNEL(analysis_rqst, INJECTOR_DURATION_4);
		CALCULATE_CHANNEL(analysis_rqst, INJECTOR_DURATION_5);
		CALCULATE_CHANNEL(analysis_rqst, INJECTOR_DURATION_6);
		CALCULATE_CHANNEL(analysis_rqst, INJECTOR_SEPARATION_1);
		CALCULATE_CHANNEL(analysis_rqst, INJECTOR_SEPARATION_2);
		CALCULATE_CHANNEL(analysis_rqst, INJECTOR_SEPARATION_3);
		CALCULATE_CHANNEL(analysis_rqst, INJECTOR_SEPARATION_4);
		CALCULATE_CHANNEL(analysis_rqst, INJECTOR_SEPARATION_5);

		/* Dependencies */

		CALCULATE_CHANNEL(analysis_rqst, DIGITAL_SIGNAL);
		CALCULATE_CHANNEL(analysis_rqst, START_OF_COMBUSTION);
	}

	if (analysis_rqst[POLYTROPIC_INDICES])
	{
		CALCULATE_CHANNEL(analysis_rqst, MOVING_PRESSURE_AVERAGE);
	}

	if ((analysis_rqst[MINIMUM_VALUE]) ||
		(analysis_rqst[MAXIMUM_VALUE]) ||
		(analysis_rqst[MEAN_VALUE]))
	{
		CALCULATE_CHANNEL(analysis_rqst, MINIMUM_VALUE);
		CALCULATE_CHANNEL(analysis_rqst, MAXIMUM_VALUE);
		CALCULATE_CHANNEL(analysis_rqst, MEAN_VALUE);
	}

	if ((analysis_rqst[MIN_PRESSURE]) ||
		(analysis_rqst[MIN_PRESSURE_CA]) ||
		(analysis_rqst[MAX_PRESSURE]) ||
		(analysis_rqst[MAX_PRESSURE_CA]) ||
		(analysis_rqst[MAX_PRESSURE_RISE_RATE]) ||
		(analysis_rqst[MAX_PRESSURE_RISE_RATE_CA]) ||
		(analysis_rqst[PRESSURE_RISE_RATE]))
	{
		/* If one is set then the rest must */

		CALCULATE_CHANNEL(analysis_rqst, MIN_PRESSURE);
		CALCULATE_CHANNEL(analysis_rqst, MIN_PRESSURE_CA);
		CALCULATE_CHANNEL(analysis_rqst, MAX_PRESSURE);
		CALCULATE_CHANNEL(analysis_rqst, MAX_PRESSURE_CA);
		CALCULATE_CHANNEL(analysis_rqst, MAX_PRESSURE_RISE_RATE);
		CALCULATE_CHANNEL(analysis_rqst, MAX_PRESSURE_RISE_RATE_CA);
		CALCULATE_CHANNEL(analysis_rqst, PRESSURE_RISE_RATE);
	}

	if ((analysis_rqst[MAX_MEAN_GAS_TEMP]) ||
		(analysis_rqst[MAX_MEAN_GAS_TEMP_CA]) ||
		(analysis_rqst[MEAN_GAS_TEMP]))
	{
		CALCULATE_CHANNEL(analysis_rqst, MAX_MEAN_GAS_TEMP);
		CALCULATE_CHANNEL(analysis_rqst, MAX_MEAN_GAS_TEMP_CA);
		CALCULATE_CHANNEL(analysis_rqst, MEAN_GAS_TEMP);

		/* Dependencies */

		CALCULATE_CHANNEL(analysis_rqst, MOVING_PRESSURE_AVERAGE);
		CALCULATE_CHANNEL(analysis_rqst, POLY_COMP);
	}

	if ((analysis_rqst[MISSING_TOOTH_1]) ||
		(analysis_rqst[MISSING_TOOTH_2]) ||
		(analysis_rqst[MISSING_TOOTH_RATIO_MAX]) ||
		(analysis_rqst[MISSING_TOOTH_RATIO_MIN]) ||
		(analysis_rqst[TOOTH_GAP_RATIO_MAX]) ||
		(analysis_rqst[TOOTH_GAP_RATIO_MIN]))
	{
		CALCULATE_CHANNEL(analysis_rqst, MISSING_TOOTH_1);
		CALCULATE_CHANNEL(analysis_rqst, MISSING_TOOTH_2);
		CALCULATE_CHANNEL(analysis_rqst, MISSING_TOOTH_RATIO_MAX);
		CALCULATE_CHANNEL(analysis_rqst, MISSING_TOOTH_RATIO_MIN);
		CALCULATE_CHANNEL(analysis_rqst, TOOTH_GAP_RATIO_MAX);
		CALCULATE_CHANNEL(analysis_rqst, TOOTH_GAP_RATIO_MIN);

		/* Dependencies */

		CALCULATE_CHANNEL(analysis_rqst, DIGITAL_SIGNAL);
	}

	if (restrike_analysis == false)
	{
		analysis_rqst[RESTRIKE_DELAY_TIME] = AnalysisRqst::NotCalculated;
		analysis_rqst[RESTRIKE_DWELL_TIME] = AnalysisRqst::NotCalculated;
		analysis_rqst[RESTRIKE_MAX_CURRENT] = AnalysisRqst::NotCalculated;
		analysis_rqst[RESTRIKE_TIMING] = AnalysisRqst::NotCalculated;
	}

	if ((analysis_rqst[DWELL_TIME]) ||
		(analysis_rqst[MAX_COIL_CURRENT]) ||
		(analysis_rqst[RESTRIKE_DELAY_TIME]) ||
		(analysis_rqst[RESTRIKE_DWELL_TIME]) ||
		(analysis_rqst[RESTRIKE_MAX_CURRENT]) ||
		(analysis_rqst[RESTRIKE_TIMING]))
	{
		CALCULATE_CHANNEL(analysis_rqst, DWELL_TIME);
		CALCULATE_CHANNEL(analysis_rqst, MAX_COIL_CURRENT);

		if (restrike_analysis == true)
		{
			CALCULATE_CHANNEL(analysis_rqst, RESTRIKE_DELAY_TIME);
			CALCULATE_CHANNEL(analysis_rqst, RESTRIKE_DWELL_TIME);
			CALCULATE_CHANNEL(analysis_rqst, RESTRIKE_MAX_CURRENT);
			CALCULATE_CHANNEL(analysis_rqst, RESTRIKE_TIMING);
		}

		/* Dependencies */

		CALCULATE_CHANNEL(analysis_rqst, DIGITAL_SIGNAL);
	}

	if ((analysis_rqst[MAX_PRESSURE_IVO]) ||
		(analysis_rqst[MAX_PRESSURE_IVO_CA]))
	{
		CALCULATE_CHANNEL(analysis_rqst, MAX_PRESSURE_IVO);
		CALCULATE_CHANNEL(analysis_rqst, MAX_PRESSURE_IVO_CA);
	}

	if ((analysis_rqst[MIN_PRESSURE_EVC]) ||
		(analysis_rqst[MIN_PRESSURE_EVC_CA]))
	{
		CALCULATE_CHANNEL(analysis_rqst, MIN_PRESSURE_EVC);
		CALCULATE_CHANNEL(analysis_rqst, MIN_PRESSURE_EVC_CA);
	}

	if (analysis_rqst[START_OF_COMBUSTION])
	{
		CALCULATE_CHANNEL(analysis_rqst, DIGITAL_SIGNAL);
	}

	/* EEOC has no dependencies */

	if ((analysis_rqst[POLY_COMP]) ||
		(analysis_rqst[POLY_EXP]))
	{
		CALCULATE_CHANNEL(analysis_rqst, MOVING_PRESSURE_AVERAGE);
	}

	if ((analysis_rqst[MAX_D2P]) ||
		(analysis_rqst[MAX_D2P_CA]) ||
		(analysis_rqst[D2P]))
	{
		CALCULATE_CHANNEL(analysis_rqst, MAX_D2P);
		CALCULATE_CHANNEL(analysis_rqst, MAX_D2P_CA);

		CALCULATE_CHANNEL(analysis_rqst, D2P);
	}

	if ((analysis_rqst[CAM_EDGE_1]) ||
		(analysis_rqst[CAM_EDGE_2]) ||
		(analysis_rqst[CAM_EDGE_3]) ||
		(analysis_rqst[CAM_EDGE_4]) ||
		(analysis_rqst[CAM_EDGE_5]) ||
		(analysis_rqst[CAM_EDGE_6]) ||
		(analysis_rqst[CAM_EDGE_7]) ||
		(analysis_rqst[CAM_EDGE_8]))
	{
		CALCULATE_CHANNEL(analysis_rqst, CAM_EDGE_1);
		CALCULATE_CHANNEL(analysis_rqst, CAM_EDGE_2);
		CALCULATE_CHANNEL(analysis_rqst, CAM_EDGE_3);
		CALCULATE_CHANNEL(analysis_rqst, CAM_EDGE_4);
		CALCULATE_CHANNEL(analysis_rqst, CAM_EDGE_5);
		CALCULATE_CHANNEL(analysis_rqst, CAM_EDGE_6);
		CALCULATE_CHANNEL(analysis_rqst, CAM_EDGE_7);
		CALCULATE_CHANNEL(analysis_rqst, CAM_EDGE_8);

		/* Dependencies */

		CALCULATE_CHANNEL(analysis_rqst, DIGITAL_SIGNAL);
	}

	if ((analysis_rqst[CHANNEL_DELTA]) ||
		(analysis_rqst[CHANNEL_DELTA_MAX]) ||
		(analysis_rqst[CHANNEL_DELTA_MAX_CA]))
	{
		CALCULATE_CHANNEL(analysis_rqst, CHANNEL_DELTA);
		CALCULATE_CHANNEL(analysis_rqst, CHANNEL_DELTA_MAX);
		CALCULATE_CHANNEL(analysis_rqst, CHANNEL_DELTA_MAX_CA);
	}

	/* No Dependencies */

	/* AVERAGE_EXHAUST_ABSOLUTE_PRESSURE */
	/* MOVING_PRESSURE_AVERAGE */

	/* Not Implemented */

	analysis_rqst[CHANNEL_DELTA] = AnalysisRqst::NotCalculated;
	analysis_rqst[CHANNEL_DELTA_MAX] = AnalysisRqst::NotCalculated;
	analysis_rqst[CHANNEL_DELTA_MAX_CA] = AnalysisRqst::NotCalculated;

	return(true);
}

bool validate_analysis_requests(const FileData* file,AnalysisChannel* analysis)
{
	unsigned int rqst;
	unsigned int channel_type;
	unsigned int abscissa_type;
	AnalysisRqst* analysis_rqst = NULL;
	AnalysisRqst* original_rqst = NULL;
	unsigned int cylinder;
	bool result;
	unsigned int last_cylinder;
	float max_offset;

	if ((file == NULL) || (analysis == NULL))
	{
		return(false);
	}

	if (file->channel_data == NULL)
	{
		return(false);
	}

	if ((analysis->channel >= file->number_of_channels) || (file->number_of_channels == 0))
	{
		logmessage(WARNING, "Channel number of analysis structure (%u) is invalid\n", analysis->channel);

		return(false);
	}

	analysis_rqst = analysis->analysis_rqst;

	original_rqst = (AnalysisRqst*)malloc(get_number_of_analysis_channels() * sizeof(AnalysisRqst));

	if (original_rqst == NULL)
	{
		logmessage(FATAL, "Out of memory\n");
	}

	for (rqst = 0; rqst < get_number_of_analysis_channels(); rqst++)
	{
		original_rqst[rqst] = analysis_rqst[rqst];
	}

	channel_type = file->channel_data[analysis->channel].type;

	if (channel_type >= NUMBER_OF_CHANNEL_TYPES)
	{
		channel_type = CHAN_UNKNOWN;

		logmessage(NOTICE, "Channel type %u on channel %u is not recognised.  Assuming channel type is unknown.\n", channel_type, analysis->channel);
	}

	abscissa_type = file->channel_data[analysis->channel].abscissa.type;

	if (abscissa_type >= NUMBER_OF_ABSCISSA_TYPES)
	{
		abscissa_type = ABSCISSA_UNKNOWN;

		logmessage(NOTICE, "Abscissa type %u on channel %u is not recognised.  Assuming abscissa type is unknown.\n", abscissa_type, analysis->channel);
	}

	last_cylinder = 1;
	max_offset = 0.0f;
	for (cylinder = 0; cylinder < file->engine.number_of_cylinders; cylinder++)
	{
		if (file->engine.tdc_offset[cylinder] > max_offset)
		{
			last_cylinder = cylinder + 1;
			max_offset = file->engine.tdc_offset[cylinder];
		}
	}

	cylinder = file->channel_data[analysis->channel].cylinder;

	/* The following analysis routines are unimplemented */

	analysis_rqst[SQUARE_CPS] = AnalysisRqst::NotCalculated;

	/* Don't need to store engine speed */

	if (analysis_rqst[ENGINE_SPEED] == AnalysisRqst::NotCalculated)
	{
		analysis_rqst[ENGINE_SPEED] = AnalysisRqst::TemporaryCalculation;
	}

	//analysis_rqst[ENGINE_SPEED] = AnalysisRqst::NotCalculated;
	
	/* Check requests are valid for the channel type */

	for (rqst = 0; rqst<get_number_of_analysis_channels(); rqst++)
	{
		if (analysis_rqst[rqst] > AnalysisRqst::NotCalculated)
		{
			if (get_analysis_channel_validity(rqst, channel_type) == false)
			{
				analysis_rqst[rqst] = AnalysisRqst::NotCalculated;

				logmessage(DEBUG,"Analysis type %u is not permitted on channel type %u\n", rqst, channel_type);
			}
			else if ((get_analysis_abscissa_type(rqst) != ABSCISSA_UNKNOWN) && (get_analysis_abscissa_type(rqst) != abscissa_type))
			{
				analysis_rqst[rqst] = AnalysisRqst::NotCalculated;

				logmessage(DEBUG,"Channel with abscissa type %u is not permitted on analysis type %u\n",abscissa_type,rqst);
			}
		}
	}

	/* Validate plugin requests - do this before calculate_dependencies() so that plugin can add base analysis types */

	plugin_validate_analysis_requests(file, analysis);

	calculate_dependencies(analysis_rqst, abscissa_type, channel_type, cylinder, last_cylinder, analysis->config.restrike_analysis);

#ifdef _SAFE_MEMORY
	/* Now double-check requests are valid for the channel type */

	for (rqst = 0; rqst<get_number_of_analysis_channels(); rqst++)
	{
		if (analysis_rqst[rqst] > AnalysisRqst::NotCalculated)
		{
			if (get_analysis_channel_validity(rqst, channel_type) == false)
			{
				logmessage(NOTICE,"Analysis type %s is not permitted on channel type %u\n", get_analysis_name(rqst), channel_type);
			}
			else if ((get_analysis_abscissa_type(rqst) != ABSCISSA_UNKNOWN) && (get_analysis_abscissa_type(rqst) != abscissa_type))
			{
				logmessage(NOTICE, "Channel with abscissa type %u is not permitted on analysis type %s\n", abscissa_type, get_analysis_name(rqst));
			}
		}
	}
#endif

	result = true;// validate_analysis_config(file, analysis);

	logmessage(DEBUG, "The following analysis types were removed: ");
	for (rqst = 0; rqst < get_number_of_analysis_channels(); rqst++)
	{
		if ((original_rqst[rqst] != analysis_rqst[rqst]) && (analysis_rqst[rqst] == AnalysisRqst::NotCalculated))
		{
			logmessage(DEBUG, "%s ", get_analysis_name(rqst));
		}
	}
	logmessage(DEBUG, "\n");

	logmessage(DEBUG, "The following analysis types were added: ");
	for (rqst = 0; rqst < get_number_of_analysis_channels(); rqst++)
	{
		if ((original_rqst[rqst] != analysis_rqst[rqst]) && (analysis_rqst[rqst] > AnalysisRqst::NotCalculated))
		{
			logmessage(DEBUG, "%s ", get_analysis_name(rqst));
		}
	}
	logmessage(DEBUG, "\n");

	/* Check analysis config is OK for the requested analysis */

	free(original_rqst);

	return(result);
}

bool validate_analysis_config(const FileData* file,const AnalysisChannel* analysis)
{
	bool result = true;
	unsigned int channel;

    if ((file == NULL) || (analysis == NULL))
    {
        return(false);
    }

	if (file->channel_data == NULL)
	{
		return(false);
	}
    
    channel = analysis->channel;

	if ((channel >= file->number_of_channels) || (file->number_of_channels == 0))
	{
		return(false);
	}

	if (file->channel_data[channel].abscissa.type != ABSCISSA_CRANKANGLE)
	{
		return(true);
	}

	if (analysis->analysis_rqst[FFT] > AnalysisRqst::NotCalculated)
	{
		/* FFT */

		if (validate_angle(file,channel,analysis->config.fft_start_window) == false)
		{
			logmessage(NOTICE,"%s: FFT Start Window invalid (%f)\n",file->channel_data[channel].name,analysis->config.fft_start_window);
			result = false;

			analysis->analysis_rqst[FFT] = AnalysisRqst::NotCalculated;
			analysis->analysis_rqst[FFT_FREQUENCY] = AnalysisRqst::NotCalculated;
		}

		if (validate_angle(file,channel,analysis->config.fft_finish_window) == false)
		{
			logmessage(NOTICE, "%s: FFT Finish Window invalid (%f)\n", file->channel_data[channel].name, analysis->config.fft_finish_window);
			result = false;

			analysis->analysis_rqst[FFT] = AnalysisRqst::NotCalculated;
			analysis->analysis_rqst[FFT_FREQUENCY] = AnalysisRqst::NotCalculated;
		}
	
		if (DegreesToCrankAngle(file,analysis->config.fft_start_window,channel) >= DegreesToCrankAngle(file,analysis->config.fft_finish_window,channel) + 5)
		{
			logmessage(NOTICE, "%s: FFT Start Window (%f) must be less than FFT Finish Window (%f)\n", file->channel_data[channel].name, analysis->config.fft_start_window, analysis->config.fft_finish_window);
			result = false;

			analysis->analysis_rqst[FFT] = AnalysisRqst::NotCalculated;
			analysis->analysis_rqst[FFT_FREQUENCY] = AnalysisRqst::NotCalculated;
		}
	}

	if (analysis->analysis_rqst[FFT_REFERENCE] > AnalysisRqst::NotCalculated)
	{
		/* FFT Reference */

		if (validate_angle(file, channel, analysis->config.fft_reference_start) == false)
		{
			logmessage(NOTICE, "%s: FFT Reference Start Window (%f) invalid\n", file->channel_data[channel].name, analysis->config.fft_reference_start);
			result = false;

			analysis->analysis_rqst[FFT_REFERENCE] = AnalysisRqst::NotCalculated;
			analysis->analysis_rqst[FFT_REFERENCE_FREQUENCY] = AnalysisRqst::NotCalculated;
		}

		if (validate_angle(file, channel, analysis->config.fft_reference_finish) == false)
		{
			logmessage(NOTICE, "%s: FFT Reference Finish Window (%f) invalid\n", file->channel_data[channel].name, analysis->config.fft_reference_finish);
			result = false;

			analysis->analysis_rqst[FFT_REFERENCE] = AnalysisRqst::NotCalculated;
			analysis->analysis_rqst[FFT_REFERENCE_FREQUENCY] = AnalysisRqst::NotCalculated;
		}

		if (DegreesToCrankAngle(file, analysis->config.fft_reference_start, channel) >= DegreesToCrankAngle(file, analysis->config.fft_reference_finish, channel) + 5)
		{
			logmessage(NOTICE, "%s: FFT Reference Start Window (%f) must be less than FFT Reference Finish Window (%f)\n", file->channel_data[channel].name, analysis->config.fft_reference_start, analysis->config.fft_reference_finish);
			result = false;

			analysis->analysis_rqst[FFT_REFERENCE] = AnalysisRqst::NotCalculated;
			analysis->analysis_rqst[FFT_REFERENCE_FREQUENCY] = AnalysisRqst::NotCalculated;
		}
	}

	if (analysis->analysis_rqst[FFT_KNOCK_PRESSURE] > AnalysisRqst::NotCalculated)
	{
		/* FFT Knock Pressure */

		if (validate_angle(file, channel, analysis->config.pkp_start_angle) == false)
		{
			logmessage(NOTICE, "%s: FFT Knock Pressure Start Window (%f) invalid\n", file->channel_data[channel].name, analysis->config.pkp_start_angle);
			result = false;

			analysis->analysis_rqst[FFT_KNOCK_PRESSURE] = AnalysisRqst::NotCalculated;
			analysis->analysis_rqst[FFT_KNOCK_PRESSURE_FREQUENCY] = AnalysisRqst::NotCalculated;
		}

		if (validate_angle(file, channel, analysis->config.pkp_finish_angle) == false)
		{
			logmessage(NOTICE, "%s: FFT Knock Pressure Finish Window (%f) invalid\n", file->channel_data[channel].name, analysis->config.pkp_finish_angle);
			result = false;

			analysis->analysis_rqst[FFT_KNOCK_PRESSURE] = AnalysisRqst::NotCalculated;
			analysis->analysis_rqst[FFT_KNOCK_PRESSURE_FREQUENCY] = AnalysisRqst::NotCalculated;
		}

		if (DegreesToCrankAngle(file, analysis->config.pkp_start_angle, channel) >= DegreesToCrankAngle(file, analysis->config.pkp_finish_angle, channel) + 5)
		{
			logmessage(NOTICE, "%s: FFT Knock Pressure Start Start Window (%f) must be less than FFT Knock Pressure Finish Window (%f)\n", file->channel_data[channel].name, analysis->config.pkp_start_angle, analysis->config.pkp_finish_angle);
			result = false;

			analysis->analysis_rqst[FFT_KNOCK_PRESSURE] = AnalysisRqst::NotCalculated;
			analysis->analysis_rqst[FFT_KNOCK_PRESSURE_FREQUENCY] = AnalysisRqst::NotCalculated;
		}
	}

	if (analysis->analysis_rqst[EEOC] > AnalysisRqst::NotCalculated)
	{
		/* EEOC */

		if (analysis->config.eeoc_method == EEOC_METHOD_BRUNT)
		{

			if (validate_angle(file, channel, analysis->config.eeoc_start_window) == false)
			{
				logmessage(NOTICE, "%s: EEOC Start Window invalid (%f)\n", file->channel_data[channel].name, analysis->config.eeoc_start_window);
				result = false;

				analysis->analysis_rqst[EEOC] = AnalysisRqst::NotCalculated;
			}

			if (validate_angle(file, channel, analysis->config.eeoc_finish_window) == false)
			{
				logmessage(NOTICE, "%s: EEOC Finish Window invalid (%f)\n", file->channel_data[channel].name, analysis->config.eeoc_finish_window);
				result = false;

				analysis->analysis_rqst[EEOC] = AnalysisRqst::NotCalculated;
			}

			if (DegreesToCrankAngle(file, analysis->config.eeoc_start_window, channel) >= DegreesToCrankAngle(file, analysis->config.eeoc_finish_window, channel))
			{
				logmessage(NOTICE, "%s: EEOC Start Window (%f) must be less than EEOC Finish Window (%f)\n", file->channel_data[channel].name, analysis->config.eeoc_start_window, analysis->config.eeoc_finish_window);
				result = false;

				analysis->analysis_rqst[EEOC] = AnalysisRqst::NotCalculated;
			}

			if ((analysis->config.eeoc_index < 1.0f) || (analysis->config.eeoc_index > 2.0f))
			{
				logmessage(NOTICE, "%s: EEOC Polytropic Index (%f) out-of-range\n", file->channel_data[channel].name, analysis->config.eeoc_index);
				result = false;

				analysis->analysis_rqst[EEOC] = AnalysisRqst::NotCalculated;
			}
		}
		else if (analysis->config.eeoc_method == EEOC_METHOD_FIXED)
		{
			if ((analysis->config.eeoc_default < 0.0f) || (analysis->config.eeoc_default > 180.0f))
			{
				logmessage(NOTICE, "%s: EEOC Default Angle (%f) out-of-range\n", file->channel_data[channel].name, analysis->config.eeoc_default);
				result = false;

				analysis->analysis_rqst[EEOC] = AnalysisRqst::NotCalculated;
			}
		}
		else
		{
			/* Do Nothing */
		}
	}

	if (analysis->analysis_rqst[POLY_EXP] > AnalysisRqst::NotCalculated)
	{
		/* Polytropic Indices */

		if (validate_angle(file,channel,analysis->config.poly_exp_start_angle) == false)
		{
			logmessage(NOTICE, "%s: Polytropic Expansion Start Window (%f) invalid\n", file->channel_data[channel].name, analysis->config.poly_exp_start_angle);
			result = false;

			analysis->analysis_rqst[POLY_EXP] = AnalysisRqst::NotCalculated;
		}

		if (validate_angle(file,channel,analysis->config.poly_exp_finish_angle) == false)
		{
			logmessage(NOTICE, "%s: Polytropic Expansion Finish Window (%f) invalid\n", file->channel_data[channel].name, analysis->config.poly_exp_finish_angle);
			result = false;

			analysis->analysis_rqst[POLY_EXP] = AnalysisRqst::NotCalculated;
		}
	
		if (DegreesToCrankAngle(file,analysis->config.poly_exp_start_angle,channel) >= DegreesToCrankAngle(file,analysis->config.poly_exp_finish_angle,channel))
		{
			logmessage(NOTICE, "%s: Polytropic Expansion Start Window (%f) must be less than Polytropic Expansion Finish Window (%f)\n", file->channel_data[channel].name, analysis->config.poly_exp_start_angle, analysis->config.poly_exp_finish_angle);
			result = false;

			analysis->analysis_rqst[POLY_EXP] = AnalysisRqst::NotCalculated;
		}
	}

	if (analysis->analysis_rqst[POLY_COMP] > AnalysisRqst::NotCalculated)
	{
		/* Polytropic Indices */

		if (validate_angle(file,channel,analysis->config.poly_comp_start_angle) == false)
		{
			logmessage(NOTICE, "%s: Polytropic Compression Start Window (%f) invalid\n", file->channel_data[channel].name, analysis->config.poly_comp_start_angle);
			result = false;

			analysis->analysis_rqst[POLY_COMP] = AnalysisRqst::NotCalculated;
		}

		if (validate_angle(file,channel,analysis->config.poly_comp_finish_angle) == false)
		{
			logmessage(NOTICE, "%s: Polytropic Compression Finish Window (%f) invalid\n", file->channel_data[channel].name, analysis->config.poly_comp_finish_angle);
			result = false;

			analysis->analysis_rqst[POLY_COMP] = AnalysisRqst::NotCalculated;
		}
	
		if (DegreesToCrankAngle(file,analysis->config.poly_comp_start_angle,channel) >= DegreesToCrankAngle(file,analysis->config.poly_comp_finish_angle,channel))
		{
			logmessage(NOTICE, "%s: Polytropic Compression Start Window (%f) must be less than Polytropic Compression Finish Window (%f)\n", file->channel_data[channel].name, analysis->config.poly_comp_start_angle, analysis->config.poly_comp_finish_angle);
			result = false;

			analysis->analysis_rqst[POLY_COMP] = AnalysisRqst::NotCalculated;
		}
	}

	if (analysis->analysis_rqst[PRESSURE_RISE_RATE] > AnalysisRqst::NotCalculated)
	{
		/* Pressure Rise Rate */

		if (validate_angle(file, channel, analysis->config.pressure_rise_start_angle) == false)
		{
			logmessage(NOTICE, "%s: Pressure Rise Rate Start Window (%f) invalid\n", file->channel_data[channel].name, analysis->config.pressure_rise_start_angle);
			result = false;

			analysis->analysis_rqst[PRESSURE_RISE_RATE] = AnalysisRqst::NotCalculated;
		}

		if (validate_angle(file, channel, analysis->config.pressure_rise_finish_angle) == false)
		{
			logmessage(NOTICE, "%s: Pressure Rise Rate Finish Window (%f) invalid\n", file->channel_data[channel].name, analysis->config.pressure_rise_finish_angle);
			result = false;

			analysis->analysis_rqst[PRESSURE_RISE_RATE] = AnalysisRqst::NotCalculated;
		}

		if (DegreesToCrankAngle(file, analysis->config.pressure_rise_start_angle, channel) >= DegreesToCrankAngle(file, analysis->config.pressure_rise_finish_angle, channel))
		{
			logmessage(NOTICE, "%s: Pressure Rise Rate Start Window (%f) must be less than Pressure Rise Rate Finish Window (%f)\n", file->channel_data[channel].name, analysis->config.pressure_rise_start_angle, analysis->config.pressure_rise_finish_angle);
			result = false;

			analysis->analysis_rqst[PRESSURE_RISE_RATE] = AnalysisRqst::NotCalculated;
		}
	}

	if (analysis->analysis_rqst[MOVING_PRESSURE_AVERAGE] > AnalysisRqst::NotCalculated)
	{
		/* Knock */

		if (validate_angle(file,channel,analysis->config.pkp_start_angle) == false)
		{
			logmessage(NOTICE, "%s: Knock Start Window (%f) invalid\n", file->channel_data[channel].name, analysis->config.pkp_start_angle);
			result = false;

			analysis->analysis_rqst[MOVING_PRESSURE_AVERAGE] = AnalysisRqst::NotCalculated;
		}

		if (validate_angle(file,channel,analysis->config.pkp_finish_angle) == false)
		{
			logmessage(NOTICE, "%s: Knock Finish Window (%f) invalid\n", file->channel_data[channel].name, analysis->config.pkp_finish_angle);
			result = false;

			analysis->analysis_rqst[MOVING_PRESSURE_AVERAGE] = AnalysisRqst::NotCalculated;
		}
	
		if (DegreesToCrankAngle(file,analysis->config.pkp_start_angle,channel) >= DegreesToCrankAngle(file,analysis->config.pkp_finish_angle,channel))
		{
			logmessage(NOTICE, "%s: Knock Start Window (%f) must be less than Knock Finish Window (%f)\n", file->channel_data[channel].name, analysis->config.pkp_start_angle, analysis->config.pkp_finish_angle);
			result = false;

			analysis->analysis_rqst[MOVING_PRESSURE_AVERAGE] = AnalysisRqst::NotCalculated;
		}

		/*if (analysis->config.pkp_smoothing_range < FLT_EPSILON)
		{
			logmessage(NOTICE,"%s: Knock smoothing range (%f) must be greater than zero\n",file->channel_data[channel].name,analysis->config.pkp_smoothing_range);
			result = false;
		}*/

		if (analysis->config.pkp_smoothing_resolution < FLT_EPSILON)
		{
			logmessage(NOTICE, "%s: Knock smoothing range (%f) must be greater than zero\n", file->channel_data[channel].name, analysis->config.pkp_smoothing_resolution);
			result = false;

			analysis->analysis_rqst[MOVING_PRESSURE_AVERAGE] = AnalysisRqst::NotCalculated;
		}
	}

	if (analysis->analysis_rqst[KNOCK_C_AND_D] > AnalysisRqst::NotCalculated)
	{
		/* C&D Knock */

		if (validate_angle(file,channel,analysis->config.cd_start_angle) == false)
		{
			logmessage(NOTICE, "%s: C&D Knock Start Window (%f) invalid\n", file->channel_data[channel].name, analysis->config.cd_start_angle);
			result = false;

			analysis->analysis_rqst[KNOCK_C_AND_D] = AnalysisRqst::NotCalculated;
		}

		if (validate_angle(file,channel,analysis->config.cd_finish_angle) == false)
		{
			logmessage(NOTICE, "%s: C&D Knock Finish Window (%f) invalid\n", file->channel_data[channel].name, analysis->config.cd_finish_angle);
			result = false;

			analysis->analysis_rqst[KNOCK_C_AND_D] = AnalysisRqst::NotCalculated;
		}

		if (DegreesToCrankAngle(file,analysis->config.cd_start_angle,channel) >= DegreesToCrankAngle(file,analysis->config.cd_finish_angle,channel))
		{
			logmessage(NOTICE, "%s: C&D Knock Start Window (%f) must be less than C&D Knock Finish Window (%f)\n", file->channel_data[channel].name, analysis->config.cd_start_angle, analysis->config.cd_finish_angle);
			result = false;

			analysis->analysis_rqst[KNOCK_C_AND_D] = AnalysisRqst::NotCalculated;
		}
	}

	/* Engine Speed */

	if (analysis->config.engine_speed < 10.0f)
	{
		logmessage(NOTICE, "%s: Default engine speed (%f) must be greater than 10 RPM\n", file->channel_data[channel].name, analysis->config.engine_speed);
	}

	return(result);
}

bool Copy_Analysis(const AnalysisConfig* src,AnalysisConfig* dest)
{
	bool return_code = true;
	
	if (src == NULL)
	{
		return(false);
	}

	if (dest == NULL)
	{
		return(false);
	}

	dest->fft_start_window = src->fft_start_window;
	dest->fft_finish_window = src->fft_finish_window;
	dest->eeoc_start_window = src->eeoc_start_window;
	dest->eeoc_finish_window = src->eeoc_finish_window;
	dest->eeoc_index = src->eeoc_index;
	dest->t_ivc = src->t_ivc;
	dest->mfb_n = src->mfb_n;
	dest->annand_a = src->annand_a;
	dest->t_wall = src->t_wall;
	dest->R = src->R;
	dest->engine_speed = src->engine_speed;
	dest->poly_exp_start_angle = src->poly_exp_start_angle;
	dest->poly_exp_finish_angle = src->poly_exp_finish_angle;
	dest->poly_comp_start_angle = src->poly_comp_start_angle;
	dest->poly_comp_finish_angle = src->poly_comp_finish_angle;
	dest->pkp_start_angle = src->pkp_start_angle;
	dest->pkp_finish_angle = src->pkp_finish_angle;
	dest->pkp_smoothing_range = src->pkp_smoothing_range;
	dest->pkp_smoothing_resolution = src->pkp_smoothing_resolution;
	dest->pressure_rise_start_angle = src->pressure_rise_start_angle;
	dest->pressure_rise_finish_angle = src->pressure_rise_finish_angle;
	dest->pressure_rise_range = src->pressure_rise_range;
	dest->temp_ref_ca = src->temp_ref_ca;
	dest->cd_start_angle = src->cd_start_angle;
	dest->cd_finish_angle = src->cd_finish_angle;
	dest->misfire_imep = src->misfire_imep;
	dest->slowburn_imep = src->slowburn_imep;
	dest->knock_pkp = src->knock_pkp;
	dest->mega_knock_pkp = src->mega_knock_pkp;
	dest->knock_kbf = src->knock_kbf;
	dest->knock_knkbint = src->knock_knkbint;
	dest->mega_knock_kbf = src->mega_knock_kbf;
	dest->mega_knock_knkbint = src->mega_knock_knkbint;
	dest->wiebe_a_start = src->wiebe_a_start;
	dest->wiebe_a_finish = src->wiebe_a_finish;
	dest->wiebe_a_step = src->wiebe_a_step;
	dest->wiebe_m_start = src->wiebe_m_start;
	dest->wiebe_m_finish = src->wiebe_m_finish;
	dest->wiebe_m_step = src->wiebe_m_step;
	dest->tla_range = src->tla_range;
	dest->injector_start_window = src->injector_start_window;
	dest->injector_finish_window = src->injector_finish_window;
	dest->align_injections_to_tdc = src->align_injections_to_tdc;
	dest->max_number_of_injections = src->max_number_of_injections;
	dest->gamma_method = src->gamma_method;
	dest->imep_method = src->imep_method;
	dest->motored_pressure_method = src->motored_pressure_method;
	dest->soc_threshold = src->soc_threshold;
	dest->tla_method = src->tla_method;
	dest->smoothed_pressure_method = src->smoothed_pressure_method;
	dest->stats_type = src->stats_type;
	dest->pressure_rise_method = src->pressure_rise_method;
	dest->fft_reference_finish = src->fft_reference_finish;
	dest->fft_reference_start = src->fft_reference_start;
	dest->crankcase_pressure = src->crankcase_pressure;
	dest->deactivation_delta_p = src->deactivation_delta_p;
	dest->d2p_window = src->d2p_window;
	dest->knock_onset_threshold = src->knock_onset_threshold;
	dest->heat_release_window = src->heat_release_window;

	dest->engine_speed_type = src->engine_speed_type;
	dest->engine_speed_channel = src->engine_speed_channel;
	dest->gas_temp_model = src->gas_temp_model;
	dest->heat_release_model = src->heat_release_model;
	dest->heat_transfer_model = src->heat_transfer_model;
	dest->knock_integral_type = src->knock_integral_type;
	dest->misfire_cycles_f = src->misfire_cycles_f;
	dest->misfire_cycles_m = src->misfire_cycles_m;
	dest->misfire_cycles_s = src->misfire_cycles_s;

	dest->interpolate_mfb = src->interpolate_mfb;
	dest->mfb_model = src->mfb_model;
	dest->restrike_analysis = src->restrike_analysis;

	dest->knock_boss_finish_window = src->knock_boss_finish_window;
	dest->knock_boss_reference_finish_window = src->knock_boss_reference_finish_window;
	dest->knock_boss_reference_start_window = src->knock_boss_reference_start_window;
	dest->knock_boss_start_window = src->knock_boss_start_window;

	dest->fft_stats_maximum = src->fft_stats_maximum;
	dest->fft_stats_resolution = src->fft_stats_resolution;

	dest->knock_hist_alarm_threshold_hist_factor = src->knock_hist_alarm_threshold_hist_factor;
	dest->knock_hist_alarm_threshold_pressure_factor = src->knock_hist_alarm_threshold_pressure_factor;
	dest->knock_hist_lower_threshold = src->knock_hist_lower_threshold;
	dest->knock_hist_shutdown_threshold_hist_factor = src->knock_hist_shutdown_threshold_hist_factor;
	dest->knock_hist_shutdown_threshold_pressure_factor = src->knock_hist_shutdown_threshold_pressure_factor;
	dest->knock_hist_time_cycle = src->knock_hist_time_cycle;
	dest->knock_hist_min_cycles = src->knock_hist_min_cycles;

	dest->eeoc_default = src->eeoc_default;
	dest->eeoc_method = src->eeoc_method;
	dest->soc_default = src->soc_default;

	return(return_code);
}

void Delete_Analysis_Group(Analysis* analysis)
{
	unsigned int group;
	unsigned int analysis_type;

	if (analysis == NULL)
	{
		return;
	}

	if (analysis->number_of_groups == 0)
	{
		return;
	}

	if (analysis->group == NULL)
	{
		return;
	}

	logmessage(DEBUG, "Releasing %u group stats\n", analysis->number_of_groups);

	for (group = 0; group < analysis->number_of_groups; group++)
	{
		if (analysis->group[group].channels != NULL)
		{
			free(analysis->group[group].channels);
			analysis->group[group].channels = NULL;
		}

		if (analysis->group[group].cycle_statistics != NULL)
		{
			free(analysis->group[group].cycle_statistics);
			analysis->group[group].cycle_statistics = NULL;
		}

		if (analysis->group[group].cycle_sum_statistics != NULL)
		{
			free(analysis->group[group].cycle_sum_statistics);
			analysis->group[group].cycle_sum_statistics = NULL;
		}

		if (analysis->group[group].engine_mean_statistics != NULL)
		{
			free(analysis->group[group].engine_mean_statistics);
			analysis->group[group].engine_mean_statistics = NULL;
		}

		if (analysis->group[group].cylinder_mean_statistics != NULL)
		{
			free(analysis->group[group].cylinder_mean_statistics);
			analysis->group[group].cylinder_mean_statistics = NULL;
		}

		if (analysis->group[group].crank_angle_statistics != NULL)
		{
			for (analysis_type = 0; analysis_type < get_number_of_analysis_channels(); analysis_type++)
			{
				if (analysis->group[group].crank_angle_statistics[analysis_type].max != NULL)
				{
					free(analysis->group[group].crank_angle_statistics[analysis_type].max);
					analysis->group[group].crank_angle_statistics[analysis_type].max = NULL;
				}

				if (analysis->group[group].crank_angle_statistics[analysis_type].mean != NULL)
				{
					free(analysis->group[group].crank_angle_statistics[analysis_type].mean);
					analysis->group[group].crank_angle_statistics[analysis_type].mean = NULL;
				}

				if (analysis->group[group].crank_angle_statistics[analysis_type].min != NULL)
				{
					free(analysis->group[group].crank_angle_statistics[analysis_type].min);
					analysis->group[group].crank_angle_statistics[analysis_type].min = NULL;
				}

				if (analysis->group[group].crank_angle_statistics[analysis_type].stddev != NULL)
				{
					free(analysis->group[group].crank_angle_statistics[analysis_type].stddev);
					analysis->group[group].crank_angle_statistics[analysis_type].stddev = NULL;
				}

				if (analysis->group[group].crank_angle_statistics[analysis_type].sum != NULL)
				{
					free(analysis->group[group].crank_angle_statistics[analysis_type].sum);
					analysis->group[group].crank_angle_statistics[analysis_type].sum = NULL;
				}
			}

			free(analysis->group[group].crank_angle_statistics);
			analysis->group[group].crank_angle_statistics = NULL;
		}

		if (analysis->group[group].cycle_sum_data != NULL)
		{
			for (analysis_type = 0; analysis_type < get_number_of_analysis_channels(); analysis_type++)
			{
				if (analysis->group[group].cycle_sum_data[analysis_type].sum != NULL)
				{
					free(analysis->group[group].cycle_sum_data[analysis_type].sum);
					analysis->group[group].cycle_sum_data[analysis_type].sum = NULL;
				}

				if (analysis->group[group].cycle_sum_data[analysis_type].min != NULL)
				{
					free(analysis->group[group].cycle_sum_data[analysis_type].min);
					analysis->group[group].cycle_sum_data[analysis_type].min = NULL;
				}

				if (analysis->group[group].cycle_sum_data[analysis_type].max != NULL)
				{
					free(analysis->group[group].cycle_sum_data[analysis_type].max);
					analysis->group[group].cycle_sum_data[analysis_type].max = NULL;
				}

				if (analysis->group[group].cycle_sum_data[analysis_type].mean != NULL)
				{
					free(analysis->group[group].cycle_sum_data[analysis_type].mean);
					analysis->group[group].cycle_sum_data[analysis_type].mean = NULL;
				}

				if (analysis->group[group].cycle_sum_data[analysis_type].stddev != NULL)
				{
					free(analysis->group[group].cycle_sum_data[analysis_type].stddev);
					analysis->group[group].cycle_sum_data[analysis_type].stddev = NULL;
				}
			}

			free(analysis->group[group].cycle_sum_data);
			analysis->group[group].cycle_sum_data = NULL;
		}
	}

	free(analysis->group);
	analysis->group = NULL;
	analysis->number_of_groups = 0;
}

void Delete_Analysis(Analysis* analysis)
{
	unsigned int channel;

	if (analysis == NULL)
	{
		return;
	}

	if (analysis->number_of_channels == 0)
	{
		return;
	}

	if (analysis->channel != NULL)
	{
		for (channel=0;channel<analysis->number_of_channels;channel++)
		{
			Delete_Analysis_Channel(&analysis->channel[channel]);
		}

		free(analysis->channel);
		analysis->channel = NULL;
		analysis->number_of_channels = 0;
	}

	Delete_Analysis_Group(analysis);
}

void Delete_Analysis_Channel(AnalysisChannel* analysis)
{
	unsigned int analysis_channel;

	if (analysis == NULL)
	{
		return;
	}

	if (analysis->cycle_classification != NULL)
	{
		free(analysis->cycle_classification);
		analysis->cycle_classification = NULL;
	}
							
	if (analysis->results != NULL)
	{				
	    for (analysis_channel=0;analysis_channel<get_number_of_analysis_channels();analysis_channel++)
		{
			/* These reference the file channel arrays.  Don't release them here */

			analysis->results[analysis_channel].abscissa.ca_to_deg = NULL;
			analysis->results[analysis_channel].abscissa.axis = NULL;
			analysis->results[analysis_channel].abscissa.volume = NULL;
			analysis->results[analysis_channel].abscissa.ca_to_theta = NULL;
			analysis->results[analysis_channel].abscissa.theta_to_ca = NULL;
			analysis->results[analysis_channel].abscissa.theta_weighting = NULL;
			analysis->results[analysis_channel].abscissa.theta_weighting_inv = NULL;

			Delete_Channel_Data(&analysis->results[analysis_channel]);
		}
				
		free(analysis->results);
		analysis->results = NULL;
	}

	if (analysis->statistics != NULL)
	{
		free(analysis->statistics);
		analysis->statistics = NULL;
	}

	if (analysis->ca_statistics != NULL)
	{
		for (analysis_channel=0;analysis_channel<get_number_of_analysis_channels();analysis_channel++)
		{
			if (analysis->ca_statistics[analysis_channel].max != NULL)
			{
				free(analysis->ca_statistics[analysis_channel].max);
				analysis->ca_statistics[analysis_channel].max = NULL;
			}

			if (analysis->ca_statistics[analysis_channel].mean != NULL)
			{
				free(analysis->ca_statistics[analysis_channel].mean);
				analysis->ca_statistics[analysis_channel].mean = NULL;
			}

			if (analysis->ca_statistics[analysis_channel].min != NULL)
			{
				free(analysis->ca_statistics[analysis_channel].min);
				analysis->ca_statistics[analysis_channel].min = NULL;
			}

			if (analysis->ca_statistics[analysis_channel].stddev != NULL)
			{
				free(analysis->ca_statistics[analysis_channel].stddev);
				analysis->ca_statistics[analysis_channel].stddev = NULL;
			}

			if (analysis->ca_statistics[analysis_channel].sum != NULL)
			{
				free(analysis->ca_statistics[analysis_channel].sum);
				analysis->ca_statistics[analysis_channel].sum = NULL;
			}
		}

		free(analysis->ca_statistics);
		analysis->ca_statistics = NULL;
	}

	if (analysis->raw_ca_statistics.max != NULL)
	{
		free(analysis->raw_ca_statistics.max);
		analysis->raw_ca_statistics.max = NULL;
	}

	if (analysis->raw_ca_statistics.mean != NULL)
	{
		free(analysis->raw_ca_statistics.mean);
		analysis->raw_ca_statistics.mean = NULL;
	}

	if (analysis->raw_ca_statistics.min != NULL)
	{
		free(analysis->raw_ca_statistics.min);
		analysis->raw_ca_statistics.min = NULL;
	}

	if (analysis->raw_ca_statistics.stddev != NULL)
	{
		free(analysis->raw_ca_statistics.stddev);
		analysis->raw_ca_statistics.stddev = NULL;
	}

	if (analysis->raw_ca_statistics.sum != NULL)
	{
		free(analysis->raw_ca_statistics.sum);
		analysis->raw_ca_statistics.sum = NULL;
	}

	if (analysis->analysis_rqst != NULL)
	{
		free(analysis->analysis_rqst);
		analysis->analysis_rqst = NULL;
	}
}

void Calculate_Geometry(Engine* engine)
{
	double swept_area;
	double swept_volume;
	double clearance_volume;

	if (engine == NULL)
	{
		return;
	}

	if ((engine->compression_ratio-1.0f) < FLT_EPSILON)
	{
		logmessage(WARNING,"Modifying compression ratio to 10.0:1 to prevent division by zero\n");

		engine->compression_ratio = 10.0f;				/* Prevent division by zero */
	}
	
	if (engine->bore < FLT_EPSILON)
	{
		logmessage(WARNING,"Bore is zero\n");
	}
		
	if (engine->stroke < FLT_EPSILON)
	{
		logmessage(WARNING,"Stroke is zero\n");
	}
		
	if (engine->conrod_length < FLT_EPSILON)
	{
	  logmessage(WARNING,"Conrod length is zero\n");
    }

	swept_area = M_PI / 4.0 * (double)engine->bore * (double)engine->bore;
	swept_volume = (double)swept_area * (double)engine->stroke;
	clearance_volume = (double)swept_volume / ((double)engine->compression_ratio - 1.0);

	engine->swept_area = (float)swept_area;
	engine->swept_volume = (float)swept_volume;
	engine->clearance_volume = (float)clearance_volume;
}

bool offset_correction_channel(FileData* file,AnalysisChannel* analysis,const unsigned int channel)
{
	double average_cylinder_pressure;
	double average_other_channel;
	float p1;
	float p2;
	float p3;
	float v1 = 0.0f;
	float v2;
	float v3 = 0.0f;
	float offset_start_angle = 0.0f;
	float offset_middle_angle = 0.0f;
	float offset_finish_angle = 0.0f;
	float offset = 0.0f;
	float offset_polytropic_index = 1.32f;
	unsigned int crank_angle;
	unsigned int offset_start_crank_angle=0;
	unsigned int offset_finish_crank_angle=0;
	unsigned int cycle;
	unsigned int offset_start_other_crank_angle = 0;
	unsigned int offset_finish_other_crank_angle = 0;
	unsigned int offset_calc_interval = 0;
	unsigned int* theta_1 = NULL;
	unsigned int* theta_2 = NULL;
	unsigned int* theta_3 = NULL;
	unsigned int angle;
	unsigned int offset_config_type;
	unsigned int data_pointer;
	unsigned int sizeof_data;
	float v_ratio_n = 1.0f;
	unsigned int offset_channel;
	float J;
	float K;
	float Z2 = 0.0f;
	float Z1 = 0.0f;
	float n;
	unsigned int samples_per_cycle;

	if (file == NULL)
	{
		return(false);
	}

	if ((channel >= file->number_of_channels) || (file->number_of_channels == 0))
	{
		return(false);
	}

	if (file->channel_data == NULL)
	{
		return(false);
	}

	if (file->channel_data[channel].abscissa.type != ABSCISSA_CRANKANGLE)
	{
		return(false);
	}

	offset_config_type = file->channel_data[channel].offset_config.type;
	samples_per_cycle = file->channel_data[channel].samples_per_cycle;

	switch (offset_config_type)
	{
	case OFFSET_NONE:
	case OFFSET_FIXED:
	default:
	{
		break;
	}
	case OFFSET_POLYTROPIC_2PT:
	case OFFSET_POLYTROPIC_3PT:
	{
		float window_size = (float)file->channel_data[channel].offset_config.window_size;
		offset_calc_interval = file->channel_data[channel].offset_config.window_size * 2 + 1;

		if (window_size == 0)
		{
			window_size = 1;
		}

		logmessage(DEBUG, "%s: offset_window_size %u\n", file->channel_data[channel].name, window_size);

		offset_start_angle = file->channel_data[channel].offset_config.start_window - window_size;
		offset_middle_angle = (file->channel_data[channel].offset_config.start_window + file->channel_data[channel].offset_config.finish_window) / 2.0f - window_size;
		offset_finish_angle = file->channel_data[channel].offset_config.finish_window - window_size;
		
		offset_polytropic_index = file->channel_data[channel].offset_config.polytropic_index;

		if ((offset_start_angle < (-90.0f * file->engine.number_of_strokes)) || 
			(offset_finish_angle < (-90.0f * file->engine.number_of_strokes)) || 
			(offset_start_angle > (90.0f * file->engine.number_of_strokes)) ||
			(offset_finish_angle > (90.0f * file->engine.number_of_strokes)) ||
			(offset_polytropic_index < 0.0f) ||
			(offset_polytropic_index > 10.0f) ||
			(offset_finish_angle - offset_start_angle < FLT_EPSILON))
		{
			logmessage(WARNING, "Offset correction: Reference or polytropic windows out of range\n");

			offset_config_type = OFFSET_NONE;
		}
		else
		{
			/* Fixed polytropic index

			   This is the only use of the Volume function */

			theta_1 = (unsigned int*)malloc(offset_calc_interval * sizeof(unsigned int));
			if (theta_1 == NULL)
			{
				logmessage(FATAL, "Memory could not be allocated\n");
			}

			theta_2 = (unsigned int*)malloc(offset_calc_interval * sizeof(unsigned int));
			if (theta_2 == NULL)
			{
				logmessage(FATAL, "Memory could not be allocated\n");
			}

			theta_3 = (unsigned int*)malloc(offset_calc_interval * sizeof(unsigned int));
			if (theta_3 == NULL)
			{
				logmessage(FATAL, "Memory could not be allocated\n");
			}

			for (angle = 0; angle < offset_calc_interval; angle++)
			{
				theta_1[angle] = DegreesToTheta(file, offset_start_angle + (float)angle);
				theta_2[angle] = DegreesToTheta(file, offset_middle_angle + (float)angle);
				theta_3[angle] = DegreesToTheta(file, offset_finish_angle + (float)angle);
			}

			v1 = Volume(file, channel, DegreesToTheta(file, file->channel_data[channel].offset_config.start_window));
			v2 = Volume(file, channel, DegreesToTheta(file, (file->channel_data[channel].offset_config.start_window + file->channel_data[channel].offset_config.finish_window) / 2.0f));
			v3 = Volume(file, channel, DegreesToTheta(file, file->channel_data[channel].offset_config.finish_window));

			v_ratio_n = powf(v1 / v3, offset_polytropic_index) - 1.0f;

			/* 3-pt offset correction - Ref SAE 900170 */

			J = v1 / v2;
			K = v1 / v3;
			Z2 = (powf(J, offset_polytropic_index) - 1.0f) / (powf(K, offset_polytropic_index) - 1.0f);
			Z1 = (powf(K, offset_polytropic_index) - 1.0f) / (powf(J, offset_polytropic_index) * logf(J) - Z2 * powf(K, offset_polytropic_index) * logf(K));
		}

		break;
	}
	case OFFSET_RESULT_CHANNEL:
	{
		if (channel_name_to_number(file, file->channel_data[channel].offset_config.channel_name, ABSCISSA_UNKNOWN, &offset_channel) == false)
		{
			logmessage(WARNING, "Offset correction: Cannot find reference channel %s\n", file->channel_data[channel].offset_config.channel_name);

			offset_config_type = OFFSET_NONE;
		}
		else if (file->channel_data[offset_channel].loaded == false)
		{
			logmessage(WARNING, "Offset correction: Reference channel %s is not loaded\n", file->channel_data[channel].offset_config.channel_name);

			offset_config_type = OFFSET_NONE;
		}
		else
		{
			/* Do Nothing */
		}

		break;
	}
	case OFFSET_WINDOW:
	case OFFSET_WINDOW_ABSOLUTE:
	{
		if (channel_name_to_number(file, file->channel_data[channel].offset_config.channel_name, ABSCISSA_UNKNOWN, &offset_channel) == false)
		{
			logmessage(WARNING, "Offset correction: Cannot find reference channel %s\n", file->channel_data[channel].offset_config.channel_name);

			offset_config_type = OFFSET_NONE;
		}
		else if (file->channel_data[offset_channel].loaded == false)
		{
			logmessage(WARNING, "Offset correction: Reference channel %s is not loaded\n", file->channel_data[channel].offset_config.channel_name);

			offset_config_type = OFFSET_NONE;
		}
		else
		{
			offset_start_angle = file->channel_data[channel].offset_config.start_window;
			offset_finish_angle = file->channel_data[channel].offset_config.finish_window;

			offset_start_crank_angle = DegreesToCrankAngle(file, offset_start_angle, channel);
			offset_finish_crank_angle = DegreesToCrankAngle(file, offset_finish_angle, channel);

			float tdc_delta = 0.0f;

			if (offset_config_type == OFFSET_WINDOW)
			{
				tdc_delta = file->channel_data[channel].tdc_offset - file->channel_data[offset_channel].tdc_offset;
			}

			offset_start_other_crank_angle = DegreesToCrankAngle(file, offset_start_angle + tdc_delta, offset_channel);
			offset_finish_other_crank_angle = DegreesToCrankAngle(file, offset_finish_angle + tdc_delta, offset_channel);

			if ((offset_finish_crank_angle < offset_start_crank_angle) ||
				(offset_finish_other_crank_angle < offset_start_other_crank_angle))
			{
				offset_config_type = OFFSET_NONE;
			}
		}

		break;
	}
	case OFFSET_MEAN:
	{
		offset_start_angle = file->channel_data[channel].offset_config.start_window;
		offset_finish_angle = file->channel_data[channel].offset_config.finish_window;

		offset_start_crank_angle = DegreesToCrankAngle(file, offset_start_angle, channel);
		offset_finish_crank_angle = DegreesToCrankAngle(file, offset_finish_angle, channel);

		if (offset_finish_crank_angle < offset_start_crank_angle)
		{
			offset_config_type = OFFSET_NONE;
		}

		break;
	}
	}

	logmessage(DEBUG, "Performing offset correction on channel %u type %u (originally %u)\n", channel, offset_config_type, file->channel_data[channel].offset_config.type);

	switch (offset_config_type)
	{
	case OFFSET_FIXED:
	{
		/* Add a fixed value */

		for (cycle=0;cycle<file->number_of_cycles;cycle++)
		{
			offset = file->channel_data[channel].offset_config.fixed_value;

			for (crank_angle=0;crank_angle<samples_per_cycle;crank_angle++)
			{
				file->channel_data[channel].data[samples_per_cycle * cycle + crank_angle] += offset;
			}

			if ((analysis != NULL) && (analysis->analysis_rqst[PRESSURE_OFFSET] > AnalysisRqst::NotCalculated))
			{
				analysis->results[PRESSURE_OFFSET].data[cycle] = offset;
			}
		}

		break;
	}
	case OFFSET_POLYTROPIC_2PT:
	case OFFSET_POLYTROPIC_3PT:
	{
		for (cycle = 0; cycle < file->number_of_cycles; cycle++)
		{
			if (offset_config_type == OFFSET_POLYTROPIC_3PT)
			{
				/* 3-pt offset correction - Ref SAE 900170 */

				p1 = 0.0f;
				p2 = 0.0f;
				p3 = 0.0f;

				for (angle = 0; angle < offset_calc_interval; angle++)
				{
					p1 += ReturnThetaData(file, cycle, theta_1[angle], channel);
					p2 += ReturnThetaData(file, cycle, theta_2[angle], channel);
					p3 += ReturnThetaData(file, cycle, theta_3[angle], channel);
				}

				/* offset_calc_interval is guaranteed to be non-zero above */

				p1 /= (float)offset_calc_interval;
				p2 /= (float)offset_calc_interval;
				p3 /= (float)offset_calc_interval;

				n = offset_polytropic_index + Z1 * ((p2 - p1) / (p3 - p1) - Z2);

				v_ratio_n = powf(v1 / v3, n) - 1.0f;
			}
			else
			{
				p1 = 0.0f;
				p3 = 0.0f;

				for (angle = 0; angle < offset_calc_interval; angle++)
				{
					p1 += ReturnThetaData(file, cycle, theta_1[angle], channel);
					p3 += ReturnThetaData(file, cycle, theta_3[angle], channel);
				}

				/* offset_calc_interval is guaranteed to be non-zero above */

				p1 /= (float)offset_calc_interval;
				p3 /= (float)offset_calc_interval;
			}

			offset = (p3 - p1) / v_ratio_n - p1;

			if (cycle == 0)
			{
				logmessage(DEBUG, "Cycle 0: %0.6f bar offset\n", offset);
			}

			for (crank_angle = 0; crank_angle < samples_per_cycle; crank_angle++)
			{
				file->channel_data[channel].data[samples_per_cycle * cycle + crank_angle] += offset;
			}

			if ((analysis != NULL) && (analysis->analysis_rqst[PRESSURE_OFFSET] > AnalysisRqst::NotCalculated))
			{
				analysis->results[PRESSURE_OFFSET].data[cycle] = offset;
			}
		}

		break;
	}
	case OFFSET_RESULT_CHANNEL:
	{
		for (cycle = 0; cycle < file->number_of_cycles; cycle++)
		{
			offset = file->channel_data[offset_channel].data[cycle];

			for (crank_angle = 0; crank_angle < samples_per_cycle; crank_angle++)
			{
				file->channel_data[channel].data[samples_per_cycle * cycle + crank_angle] += offset;
			}

			if ((analysis != NULL) && (analysis->analysis_rqst[PRESSURE_OFFSET] > AnalysisRqst::NotCalculated))
			{
				analysis->results[PRESSURE_OFFSET].data[cycle] = offset;
			}
		}

		file->channel_data[channel].offset_config.type = OFFSET_NONE;

		break;
	}
	case OFFSET_WINDOW:
	case OFFSET_WINDOW_ABSOLUTE:
	{
		/* Equal to average of other channel over fixed window */

		for (cycle=0;cycle<file->number_of_cycles;cycle++)
		{
			average_cylinder_pressure = 0.0;
			for (crank_angle = offset_start_crank_angle; crank_angle <= offset_finish_crank_angle; crank_angle++)
			{
				average_cylinder_pressure += ReturnCAData(file, cycle, crank_angle, channel);
			}
			average_cylinder_pressure /= (double)(offset_finish_crank_angle-offset_start_crank_angle+1);

			average_other_channel = 0.0;
			for (crank_angle = offset_start_other_crank_angle; crank_angle <= offset_finish_other_crank_angle; crank_angle++)
			{
				average_other_channel += ReturnCAData(file, cycle, crank_angle, offset_channel);
			}
			average_other_channel /= (double)(offset_finish_other_crank_angle - offset_start_other_crank_angle + 1);

			/* Convert other channel to bar if necessary */

			average_other_channel *= file->channel_data[offset_channel].conversion_factor;

			offset = (float)(average_other_channel - average_cylinder_pressure);

			for (crank_angle = 0; crank_angle < samples_per_cycle; crank_angle++)
			{
				file->channel_data[channel].data[samples_per_cycle * cycle + crank_angle] += offset;
			}

			if ((analysis != NULL) && (analysis->analysis_rqst[PRESSURE_OFFSET] > AnalysisRqst::NotCalculated))
			{
				analysis->results[PRESSURE_OFFSET].data[cycle] = offset;
			}
		}

		break;
	}
	case OFFSET_MEAN:
	{
		/* Set mean of channel equal to a fixed value */

		for (cycle=0;cycle<file->number_of_cycles;cycle++)
		{
			average_cylinder_pressure = 0.0;
			for (crank_angle = offset_start_crank_angle; crank_angle <= offset_finish_crank_angle; crank_angle++)
			{
				average_cylinder_pressure += ReturnCAData(file, cycle, crank_angle, channel);
			}
			average_cylinder_pressure /= (double)(offset_finish_crank_angle-offset_start_crank_angle+1);

			offset = file->channel_data[channel].offset_config.fixed_value - (float)average_cylinder_pressure;

			for (crank_angle = 0; crank_angle < samples_per_cycle; crank_angle++)
			{
				file->channel_data[channel].data[samples_per_cycle * cycle + crank_angle] += offset;
			}

			if ((analysis != NULL) && (analysis->analysis_rqst[PRESSURE_OFFSET] > AnalysisRqst::NotCalculated))
			{
				analysis->results[PRESSURE_OFFSET].data[cycle] = offset;
			}
		}

		break;
	}
	case OFFSET_NONE:
	default:
	{
		/* Do nothing */

		if ((analysis != NULL) && (analysis->analysis_rqst[PRESSURE_OFFSET] > AnalysisRqst::NotCalculated))
		{
			for (cycle = 0; cycle < file->number_of_cycles; cycle++)
			{
				analysis->results[PRESSURE_OFFSET].data[cycle] = 0.0f;
			}
		}

		break;
	}
	}

	if (file->channel_data[channel].offset_config.truncate == true)
	{
		sizeof_data = file->number_of_cycles * samples_per_cycle;

		for (data_pointer = 0; data_pointer < sizeof_data; data_pointer++)
		{
			if (file->channel_data[channel].data[data_pointer] < 0.0f)
			{
				file->channel_data[channel].data[data_pointer] = 0.0f;
			}
		}
	}

	if (theta_1 != NULL)
	{
		free(theta_1);
		theta_1 = NULL;
	}

	if (theta_2 != NULL)
	{
		free(theta_2);
		theta_2 = NULL;
	}

	if (theta_3 != NULL)
	{
		free(theta_3);
		theta_3 = NULL;
	}

	file->channel_data[channel].isoffset = true;

	return (true);
}

void Set_Raw_Channel_Min_Max(FileData* file,const unsigned int channel)
{
	size_t number_data_points = 0;
	float min = 0.0f;
	float max = 0.0f;

	if (file == NULL)
	{
		return;
	}
	
	if ((file->channel_data == NULL) || (channel >= file->number_of_channels) || (file->number_of_channels == 0))
	{
		return;
	}
	
	if (file->channel_data[channel].data == NULL)
	{
		return;
	}

	if ((file->channel_data[channel].abscissa.type == ABSCISSA_CRANKANGLE) || 
		(file->channel_data[channel].abscissa.type == ABSCISSA_FREQUENCY))
	{
		number_data_points = file->channel_data[channel].samples_per_cycle * file->number_of_cycles;
	}
	else
	{
		number_data_points = file->channel_data[channel].samples_per_cycle;
	}

	minmaxf(file->channel_data[channel].data,number_data_points,&min,&max);

	file->channel_data[channel].min = min;
	file->channel_data[channel].max = max;
}

void Set_Results_Channel_Min_Max(AnalysisChannel* analysis,const unsigned int channel)
{
	size_t number_data_points;
	float min = 0.0f;
	float max = 0.0f;

	if ((analysis == NULL) || (channel >= get_number_of_analysis_channels()))
	{
		return;
	}
	
	if (analysis->results == NULL)
	{
		return;
	}

	if (analysis->analysis_rqst[channel] < AnalysisRqst::CalculateAndStore)
	{
		return;
	}

	if (analysis->results[channel].data == NULL)
	{
		return;
	}

	if (analysis->results[channel].abscissa.type == ABSCISSA_CRANKANGLE)
	{
		number_data_points = analysis->results[channel].samples_per_cycle * analysis->number_of_cycles;
	}
	else if (analysis->results[channel].abscissa.type == ABSCISSA_FREQUENCY)
	{
		number_data_points = analysis->results[channel].samples_per_cycle * analysis->number_of_cycles;
	}
	else
	{
		number_data_points = analysis->results[channel].samples_per_cycle;
	}
		
	minmaxf(analysis->results[channel].data, number_data_points, &min, &max);

	analysis->results[channel].min = min;
	analysis->results[channel].max = max;
}

void Calculate_Min_Max(FileData* file, Analysis* analysis)
{
	unsigned int channel;
	unsigned int analysis_number;

	if (file != NULL)
	{
		for (channel = 0; channel < file->number_of_channels; channel++)
		{
			if (file->channel_data[channel].loaded == true)
			{
				Set_Raw_Channel_Min_Max(file, channel);
			}
		}

		if (analysis != NULL)
		{
			for (analysis_number = 0; analysis_number < analysis->number_of_channels; analysis_number++)
			{
				if (file->channel_data[analysis->channel[analysis_number].channel].loaded == true)
				{
					for (channel = 0; channel < get_number_of_analysis_channels(); channel++)
					{
						Set_Results_Channel_Min_Max(&analysis->channel[analysis_number], channel);
					}
				}
			}
		}
	}
}

void update_status(ProgressStatus* status, char* text, const float pct)
{
#ifdef __CATOOLRT__
	if (status != NULL)
	{
		int percentage = status->min + pct * (status->max - status->min);

		if (status->ptr[0] != NULL)
		{
			((wxStatusBarProgress*)status->ptr[0])->Update(percentage, wxString::Format(_("Analysing Channel %s"), text));
		}
	}
#else
	logmessage(DEBUG, "%s: %0.0f%%                    \r", text, pct * 100.0f);
#endif
}

#define CONFIGURE_CA_RESULTS(NAME,name) 	float* name;										\
	if ((analysis->analysis_rqst[NAME] == AnalysisRqst::TemporaryCalculation) ||				\
	    (analysis->analysis_rqst[NAME] == AnalysisRqst::CalculateOnDemand)) {					\
		name = (float*)malloc(samples_per_cycle * sizeof(float));								\
		if (name == NULL) {																		\
			logmessage(FATAL, "Memory could not be allocated\n"); }								\
	} else {																					\
		name = NULL;																			\
	}

#define CONFIGURE_FFT_RESULTS(NAME,name,config)	float* name;									\
	if ((analysis->analysis_rqst[NAME] == AnalysisRqst::TemporaryCalculation) ||				\
	    (analysis->analysis_rqst[NAME] == AnalysisRqst::CalculateOnDemand)) {					\
		name = (float*)malloc(config.number_of_samples * sizeof(float));						\
		if (name == NULL) {																		\
			logmessage(FATAL, "Memory could not be allocated\n"); }								\
	} else {																					\
		name = NULL;																			\
	}

#define FINISH_RESULTS(NAME,name)																\
	if ((analysis->analysis_rqst[NAME] == AnalysisRqst::TemporaryCalculation) ||				\
	    (analysis->analysis_rqst[NAME] == AnalysisRqst::CalculateOnDemand)) {					\
		free(name);																				\
		name = NULL; }

void start_of_combustion_analysis(FileData* file, AnalysisChannel* analysis, Analysis* file_analysis, const unsigned int channel, ProgressStatus* status)
{
	unsigned int cycle;
	unsigned int samples_per_cycle;
	float* volume = NULL;
	float* dVolume = NULL;
	float* dCA = NULL;
	float* heat_transfer_area = NULL;
	float* logVolume = NULL;
	float* pressure_to_torque = NULL;
	bool calculate_coil_current;
	unsigned int smoothing_range;
	unsigned int smoothing_resolution;
	unsigned int smoothing_points;
	float* smoothing_coefficients = NULL;
	float ivo;
	float evc;
	unsigned int previous_cycle;
	unsigned int soc_config_channel = 0;
	void** plugin_config = NULL;
	unsigned int number_of_plugin_channels;
	Geometry geometry;
	ReturnPluginAnalysisFunction ReturnPluginAnalysis = NULL;
	CompletePluginChannelsFunction CompletePluginChannels = NULL;
	InitialisePluginChannelsFunction InitialisePluginChannels = NULL;
	unsigned int analysis_type;
	FFTConfig fft_tooth_config;
	float start_angle;
	float finish_angle;

	if ((file == NULL) || (analysis == NULL) || (file_analysis == NULL))
	{
		return;
	}

	if ((channel >= file->number_of_channels) || (file->number_of_channels == 0) || (file->channel_data == NULL))
	{
		return;
	}

	if (file->engine.number_of_strokes == 2)
	{
		start_angle = -180.0f;
		finish_angle = 180.0f;

		ivo = -140.0f;
		evc = -120.0f;
	}
	else
	{
		start_angle = -360.0f;
		finish_angle = 360.0f;

		ivo = 320.0f;
		evc = -320.0f;
	}

	samples_per_cycle = file->channel_data[channel].samples_per_cycle;

	volume = (float*)malloc(samples_per_cycle * sizeof(float));
	if (volume == NULL)
	{
		logmessage(FATAL, "Memory could not be allocated\n");
	}

	dVolume = (float*)malloc(samples_per_cycle * sizeof(float));
	if (dVolume == NULL)
	{
		logmessage(FATAL, "Memory could not be allocated\n");
	}

	dCA = (float*)malloc(samples_per_cycle * sizeof(float));
	if (dCA == NULL)
	{
		logmessage(FATAL, "Memory could not be allocated\n");
	}

	heat_transfer_area = (float*)malloc(samples_per_cycle * sizeof(float));
	if (heat_transfer_area == NULL)
	{
		logmessage(FATAL, "Memory could not be allocated\n");
	}

	logVolume = (float*)malloc(samples_per_cycle * sizeof(float));
	if (logVolume == NULL)
	{
		logmessage(FATAL, "Memory could not be allocated\n");
	}

	pressure_to_torque = (float*)malloc(samples_per_cycle * sizeof(float));
	if (pressure_to_torque == NULL)
	{
		logmessage(FATAL, "Memory could not be allocated\n");
	}

	Init_FFTConfig(&fft_tooth_config);

	Return_Geometry_Data(file, channel, dCA, volume, dVolume, heat_transfer_area, logVolume, pressure_to_torque);

	geometry.volume = volume;
	geometry.dVolume = dVolume;
	geometry.dCA = dCA;
	geometry.heat_transfer_area = heat_transfer_area;
	geometry.logVolume = logVolume;
	geometry.pressure_to_torque = pressure_to_torque;

	number_of_plugin_channels = get_number_of_plugin_channels();

	if (number_of_plugin_channels > 0)
	{
		plugin_config = (void**)malloc(number_of_plugin_channels * sizeof(void*));

		if (plugin_config == NULL)
		{
			logmessage(FATAL, "Memory could not be allocated\n");
		}

		memset(plugin_config, 0, number_of_plugin_channels * sizeof(void*));
	}

	if ((file->channel_data[channel].soc_config.type == SOC_CA_CHANNEL_AVG) ||
		(file->channel_data[channel].soc_config.type == SOC_CA_CHANNEL_FALL) ||
		(file->channel_data[channel].soc_config.type == SOC_CA_CHANNEL_RISE) ||
		(file->channel_data[channel].soc_config.type == SOC_CYC_CHANNEL))
	{
		if (channel_name_to_number(file, file->channel_data[channel].soc_config.channel_name, ABSCISSA_UNKNOWN, &soc_config_channel) == false)
		{
			file->channel_data[channel].soc_config.type = SOC_FIXED;
			file->channel_data[channel].soc_config.fixed_value = 0.0f;
		}
	}

	if (analysis->analysis_rqst[MOVING_PRESSURE_AVERAGE] > AnalysisRqst::NotCalculated)
	{
		Prepare_SmoothedPressure_Data(file, analysis->config.smoothed_pressure_method, analysis->config.pkp_smoothing_range, analysis->config.pkp_smoothing_resolution, &smoothing_range, &smoothing_resolution, &smoothing_points, &smoothing_coefficients);
	}

	if (analysis->analysis_rqst[FFT_TOOTH_PERIOD] > AnalysisRqst::NotCalculated)
	{
		Prepare_FFT(file, channel, UNITS_HZ, start_angle, finish_angle, FFT_WINDOW_HAMMING, NUMBER_OF_TOOTH_PERIOD_FFT_CYCLES, &fft_tooth_config);
	}

	for (analysis_type = NUMBER_OF_ANALYSIS_CHANNELS; analysis_type < get_number_of_analysis_channels(); analysis_type++)
	{
		InitialisePluginChannels = (InitialisePluginChannelsFunction)get_initialise_function(analysis_type);

		if ((InitialisePluginChannels != NULL) && (analysis->analysis_rqst[analysis_type] >= AnalysisRqst::CalculateAndStore))
		{
			InitialisePluginChannels(file, analysis, channel, analysis_type, geometry, &plugin_config[analysis_type - NUMBER_OF_ANALYSIS_CHANNELS]);
		}
	}

	memset(analysis->cycle_classification, 0, analysis->number_of_cycles * sizeof(unsigned char));

	CONFIGURE_CA_RESULTS(MOVING_PRESSURE_AVERAGE, moving_pressure_average);
	CONFIGURE_CA_RESULTS(PRESSURE_RISE_RATE, pressure_rise_rate);
	CONFIGURE_CA_RESULTS(DIGITAL_SIGNAL, digital_signal);
	CONFIGURE_CA_RESULTS(VALVE_VELOCITY, valve_velocity);
	CONFIGURE_CA_RESULTS(VALVE_ACCEL, valve_accel);
	CONFIGURE_CA_RESULTS(TOOTH_SPEED, tooth_speed);

	CONFIGURE_FFT_RESULTS(FFT_TOOTH_PERIOD, fft_tooth_period, fft_tooth_config);
	CONFIGURE_FFT_RESULTS(FFT_TOOTH_PERIOD_FREQUENCY, fft_tooth_period_frequency, fft_tooth_config);

	if (analysis->analysis_rqst[ENGINE_SPEED] > AnalysisRqst::NotCalculated)
	{
		engine_speed_analysis(file, analysis->results[ENGINE_SPEED].data, channel);
	}

	for (cycle = 0; cycle < file->number_of_cycles; cycle++)
	{
#ifdef __CATOOLRT__
		if (analysis->analysis_rqst[AVERAGE_EXHAUST_ABSOLUTE_PRESSURE] > AnalysisRqst::NotCalculated)
		{
			Return_Average_Exhaust_Absolute_Pressure(file,
				cycle,
				channel,
				240.0f,
				300.0f,
				&analysis->results[AVERAGE_EXHAUST_ABSOLUTE_PRESSURE].data[cycle]);
		}
#endif

		if (analysis->analysis_rqst[MOVING_PRESSURE_AVERAGE] > AnalysisRqst::NotCalculated)
		{
			if (analysis->analysis_rqst[MOVING_PRESSURE_AVERAGE] >= AnalysisRqst::CalculateAndStore)
			{
				moving_pressure_average = &analysis->results[MOVING_PRESSURE_AVERAGE].data[cycle * samples_per_cycle];
			}

			Return_SmoothedPressure_Data(file,
				cycle,
				channel,
				smoothing_range,
				smoothing_resolution,
				smoothing_points,
				smoothing_coefficients,
				moving_pressure_average);
		}

		if (analysis->analysis_rqst[MAX_PRESSURE] > AnalysisRqst::NotCalculated)
		{
			if (analysis->analysis_rqst[PRESSURE_RISE_RATE] >= AnalysisRqst::CalculateAndStore)
			{
				pressure_rise_rate = &analysis->results[PRESSURE_RISE_RATE].data[cycle * samples_per_cycle];
			}

			Return_Pressure_Analysis(file,
				cycle,
				channel,
				analysis->config.pressure_rise_start_angle,
				analysis->config.pressure_rise_finish_angle,
				analysis->config.pressure_rise_range,
				&analysis->results[MIN_PRESSURE].data[cycle],
				&analysis->results[MIN_PRESSURE_CA].data[cycle],
				&analysis->results[MAX_PRESSURE].data[cycle],
				&analysis->results[MAX_PRESSURE_CA].data[cycle],
				&analysis->results[MAX_PRESSURE_RISE_RATE].data[cycle],
				&analysis->results[MAX_PRESSURE_RISE_RATE_CA].data[cycle],
				pressure_rise_rate);
		}

		if (analysis->analysis_rqst[DIGITAL_SIGNAL] > AnalysisRqst::NotCalculated)
		{
			if (analysis->analysis_rqst[DIGITAL_SIGNAL] >= AnalysisRqst::CalculateAndStore)
			{
				digital_signal = &analysis->results[DIGITAL_SIGNAL].data[cycle * samples_per_cycle];
			}

			Return_Digital_Channel(file,
				cycle,
				channel,
				file->channel_data[channel].digital_config.type,
				file->channel_data[channel].digital_config.latch_high,
				file->channel_data[channel].digital_config.latch_low,
				file->channel_data[channel].digital_config.filter,
				digital_signal);
		}

		if (analysis->analysis_rqst[START_OF_COMBUSTION] > AnalysisRqst::NotCalculated)
		{
			if (analysis->analysis_rqst[DIGITAL_SIGNAL] >= AnalysisRqst::CalculateAndStore)
			{
				digital_signal = &analysis->results[DIGITAL_SIGNAL].data[cycle * samples_per_cycle];
			}

			Return_SOC(file,
				file_analysis,
				cycle,
				channel,
				digital_signal,
				soc_config_channel,
				file->channel_data[channel].soc_config.type,
				file->channel_data[channel].soc_config.start_window,
				file->channel_data[channel].soc_config.finish_window,
				file->channel_data[channel].soc_config.fixed_value,
				START_OF_COMBUSTION,
				file->channel_data[channel].soc_config.aligned,
				file->channel_data[channel].soc_config.invert,
				false,
				file->number_of_cycles,
				dCA,
				&analysis->results[START_OF_COMBUSTION].data[cycle]);
		}

		if (analysis->analysis_rqst[START_OF_INJECTION_1] > AnalysisRqst::NotCalculated)
		{
			if (analysis->analysis_rqst[DIGITAL_SIGNAL] >= AnalysisRqst::CalculateAndStore)
			{
				digital_signal = &analysis->results[DIGITAL_SIGNAL].data[cycle * samples_per_cycle];
			}

			Return_Injection_Timing(file,
				channel,
				analysis->results[ENGINE_SPEED].data[cycle],
				digital_signal,
				analysis->config.injector_start_window,
				analysis->config.injector_finish_window,
				analysis->config.align_injections_to_tdc,
				analysis->config.max_number_of_injections,
				&analysis->results[START_OF_INJECTION_1].data[cycle],
				&analysis->results[START_OF_INJECTION_2].data[cycle],
				&analysis->results[START_OF_INJECTION_3].data[cycle],
				&analysis->results[START_OF_INJECTION_4].data[cycle],
				&analysis->results[START_OF_INJECTION_5].data[cycle],
				&analysis->results[START_OF_INJECTION_6].data[cycle],
				&analysis->results[END_OF_INJECTION_1].data[cycle],
				&analysis->results[END_OF_INJECTION_2].data[cycle],
				&analysis->results[END_OF_INJECTION_3].data[cycle],
				&analysis->results[END_OF_INJECTION_4].data[cycle],
				&analysis->results[END_OF_INJECTION_5].data[cycle],
				&analysis->results[END_OF_INJECTION_6].data[cycle],
				&analysis->results[NUMBER_OF_INJECTIONS].data[cycle],
				&analysis->results[INJECTOR_DURATION_1].data[cycle],
				&analysis->results[INJECTOR_DURATION_2].data[cycle],
				&analysis->results[INJECTOR_DURATION_3].data[cycle],
				&analysis->results[INJECTOR_DURATION_4].data[cycle],
				&analysis->results[INJECTOR_DURATION_5].data[cycle],
				&analysis->results[INJECTOR_DURATION_6].data[cycle],
				&analysis->results[INJECTOR_SEPARATION_1].data[cycle],
				&analysis->results[INJECTOR_SEPARATION_2].data[cycle],
				&analysis->results[INJECTOR_SEPARATION_3].data[cycle],
				&analysis->results[INJECTOR_SEPARATION_4].data[cycle],
				&analysis->results[INJECTOR_SEPARATION_5].data[cycle]);
		}

		if (analysis->analysis_rqst[CAM_EDGE_1] > AnalysisRqst::NotCalculated)
		{
			if (analysis->analysis_rqst[DIGITAL_SIGNAL] >= AnalysisRqst::CalculateAndStore)
			{
				digital_signal = &analysis->results[DIGITAL_SIGNAL].data[cycle * samples_per_cycle];
			}

			Return_Cam_Edges(file,
				channel,
				digital_signal,
				&analysis->results[CAM_EDGE_1].data[cycle],
				&analysis->results[CAM_EDGE_2].data[cycle],
				&analysis->results[CAM_EDGE_3].data[cycle],
				&analysis->results[CAM_EDGE_4].data[cycle],
				&analysis->results[CAM_EDGE_5].data[cycle],
				&analysis->results[CAM_EDGE_6].data[cycle],
				&analysis->results[CAM_EDGE_7].data[cycle],
				&analysis->results[CAM_EDGE_8].data[cycle]);
		}

		if (analysis->analysis_rqst[CAM_ADVANCE] > AnalysisRqst::NotCalculated)
		{
			Return_Camshaft_Analysis(file,
				channel,
				file->channel_data[channel].cam_config.type,
				file->channel_data[channel].cam_config.edge,
				file->channel_data[channel].cam_config.reference_angle,
				file->channel_data[channel].cam_config.offset,
				analysis->results[CAM_EDGE_1].data[cycle],
				analysis->results[CAM_EDGE_2].data[cycle],
				analysis->results[CAM_EDGE_3].data[cycle],
				analysis->results[CAM_EDGE_4].data[cycle],
				analysis->results[CAM_EDGE_5].data[cycle],
				analysis->results[CAM_EDGE_6].data[cycle],
				analysis->results[CAM_EDGE_7].data[cycle],
				analysis->results[CAM_EDGE_8].data[cycle],
				&analysis->results[CAM_ANGLE].data[cycle],
				&analysis->results[CAM_ADVANCE].data[cycle]);
		}

		if (analysis->analysis_rqst[KNOCK_BOSS_INTEGRAL] > AnalysisRqst::NotCalculated)
		{
			Return_Accelerometer_Data(file,
				cycle,
				channel,
				dCA,
				analysis->config.knock_boss_reference_start_window,
				analysis->config.knock_boss_reference_finish_window,
				analysis->config.knock_boss_start_window,
				analysis->config.knock_boss_finish_window,
				&analysis->results[KNOCK_BOSS_INTEGRAL].data[cycle],
				&analysis->results[KNOCK_BOSS_FACTOR].data[cycle]);

			if (analysis->results[KNOCK_BOSS_FACTOR].data[cycle] > analysis->config.mega_knock_kbf)
			{
				analysis->cycle_classification[cycle] |= CYCLE_MEGA_KNOCK & 0xFF;
			}
			else if (analysis->results[KNOCK_BOSS_FACTOR].data[cycle] > analysis->config.knock_kbf)
			{
				analysis->cycle_classification[cycle] |= CYCLE_KNOCK & 0xFF;
			}
			else
			{
				/* Do Nothing */
			}

			if (analysis->results[KNOCK_BOSS_INTEGRAL].data[cycle] > analysis->config.mega_knock_knkbint)
			{
				analysis->cycle_classification[cycle] |= CYCLE_MEGA_KNOCK & 0xFF;
			}
			else if (analysis->results[KNOCK_BOSS_INTEGRAL].data[cycle] > analysis->config.knock_knkbint)
			{
				analysis->cycle_classification[cycle] |= CYCLE_KNOCK & 0xFF;
			}
			else
			{
				/* Do Nothing */
			}
		}

		if (analysis->analysis_rqst[MISSING_TOOTH_1] > AnalysisRqst::NotCalculated)
		{
			if (analysis->analysis_rqst[DIGITAL_SIGNAL] >= AnalysisRqst::CalculateAndStore)
			{
				digital_signal = &analysis->results[DIGITAL_SIGNAL].data[cycle * samples_per_cycle];
			}

			Return_Crank_Sensor_Analysis(file,
				channel,
				digital_signal,
				&analysis->results[MISSING_TOOTH_1].data[cycle],
				&analysis->results[MISSING_TOOTH_2].data[cycle],
				&analysis->results[MISSING_TOOTH_RATIO_MIN].data[cycle],
				&analysis->results[MISSING_TOOTH_RATIO_MAX].data[cycle],
				&analysis->results[TOOTH_GAP_RATIO_MIN].data[cycle],
				&analysis->results[TOOTH_GAP_RATIO_MAX].data[cycle]);
		}

		if (analysis->analysis_rqst[DWELL_TIME] > AnalysisRqst::NotCalculated)
		{
			calculate_coil_current = (file->channel_data[channel].type == CHAN_SPK_PRI_CURR);

			if (analysis->analysis_rqst[DIGITAL_SIGNAL] >= AnalysisRqst::CalculateAndStore)
			{
				digital_signal = &analysis->results[DIGITAL_SIGNAL].data[cycle * samples_per_cycle];
			}

			Return_Coil_Analysis(file,
				cycle,
				channel,
				calculate_coil_current,
				analysis->config.restrike_analysis,
				digital_signal,
				analysis->results[ENGINE_SPEED].data[cycle],
				&analysis->results[DWELL_TIME].data[cycle],
				&analysis->results[MAX_COIL_CURRENT].data[cycle],
				&analysis->results[RESTRIKE_MAX_CURRENT].data[cycle],
				&analysis->results[RESTRIKE_TIMING].data[cycle],
				&analysis->results[RESTRIKE_DELAY_TIME].data[cycle],
				&analysis->results[RESTRIKE_DWELL_TIME].data[cycle]);
		}

		if (analysis->analysis_rqst[MAX_PRESSURE_IVO] > AnalysisRqst::NotCalculated)
		{
			if (cycle > 0)
			{
				previous_cycle = cycle - 1;
			}
			else
			{
				previous_cycle = 0;
			}

			if (analysis->analysis_rqst[MOVING_PRESSURE_AVERAGE] >= AnalysisRqst::CalculateAndStore)
			{
				moving_pressure_average = &analysis->results[MOVING_PRESSURE_AVERAGE].data[cycle * samples_per_cycle];
			}

			Return_Intake_Pressure_Analysis(file,
				cycle,
				previous_cycle,
				channel,
				moving_pressure_average,
				ivo,
				&analysis->results[MAX_PRESSURE_IVO].data[cycle],
				&analysis->results[MAX_PRESSURE_IVO_CA].data[cycle]);
		}

		if (analysis->analysis_rqst[MIN_PRESSURE_EVC] > AnalysisRqst::NotCalculated)
		{
			if (cycle > 0)
			{
				previous_cycle = cycle - 1;
			}
			else
			{
				previous_cycle = 0;
			}

			if (analysis->analysis_rqst[MOVING_PRESSURE_AVERAGE] >= AnalysisRqst::CalculateAndStore)
			{
				moving_pressure_average = &analysis->results[MOVING_PRESSURE_AVERAGE].data[cycle * samples_per_cycle];
			}

			Return_Exhaust_Pressure_Analysis(file,
				cycle,
				previous_cycle,
				channel,
				moving_pressure_average,
				evc,
				&analysis->results[MIN_PRESSURE_EVC].data[cycle],
				&analysis->results[MIN_PRESSURE_EVC_CA].data[cycle]);
		}

		if (analysis->analysis_rqst[MINIMUM_VALUE] > AnalysisRqst::NotCalculated)
		{
			Return_MinMaxMean_Analysis(file,
				cycle,
				channel,
				&analysis->results[MINIMUM_VALUE].data[cycle],
				&analysis->results[MAXIMUM_VALUE].data[cycle],
				&analysis->results[MEAN_VALUE].data[cycle]);
		}

#ifdef __CATOOLRT__
		if (analysis->analysis_rqst[TOOTH_SPEED] > AnalysisRqst::NotCalculated)
		{
			if (analysis->analysis_rqst[TOOTH_SPEED] >= AnalysisRqst::CalculateAndStore)
			{
				tooth_speed = &analysis->results[TOOTH_SPEED].data[cycle * samples_per_cycle];
			}

			Return_Tooth_Period_Analysis(file,
				cycle,
				channel,
				tooth_speed);
		}

		if (analysis->analysis_rqst[FFT_TOOTH_PERIOD] > AnalysisRqst::NotCalculated)
		{
			if (cycle < file->number_of_cycles - NUMBER_OF_TOOTH_PERIOD_FFT_CYCLES)
			{
				if (analysis->analysis_rqst[TOOTH_SPEED] >= AnalysisRqst::CalculateAndStore)
				{
					tooth_speed = &analysis->results[TOOTH_SPEED].data[cycle * samples_per_cycle];
				}

				if (analysis->analysis_rqst[FFT_TOOTH_PERIOD_FREQUENCY] >= AnalysisRqst::CalculateAndStore)
				{
					fft_tooth_period_frequency = &analysis->results[FFT_TOOTH_PERIOD_FREQUENCY].data[cycle * analysis->results[FFT_TOOTH_PERIOD_FREQUENCY].samples_per_cycle];
				}

				if (analysis->analysis_rqst[FFT_TOOTH_PERIOD] >= AnalysisRqst::CalculateAndStore)
				{
					fft_tooth_period = &analysis->results[FFT_TOOTH_PERIOD].data[cycle * analysis->results[FFT_TOOTH_PERIOD].samples_per_cycle];
				}

				Return_FFT(&fft_tooth_config,
					tooth_speed,
					analysis->results[ENGINE_SPEED].data[cycle],
					false,
					fft_tooth_period_frequency,
					fft_tooth_period);
			}
		}

		if (analysis->analysis_rqst[VALVE_ACCEL] > AnalysisRqst::NotCalculated)
		{
			if (analysis->analysis_rqst[VALVE_VELOCITY] >= AnalysisRqst::CalculateAndStore)
			{
				valve_velocity = &analysis->results[VALVE_VELOCITY].data[cycle * file->channel_data[channel].samples_per_cycle];
			}

			if (analysis->analysis_rqst[VALVE_ACCEL] >= AnalysisRqst::CalculateAndStore)
			{
				valve_accel = &analysis->results[VALVE_ACCEL].data[cycle * file->channel_data[channel].samples_per_cycle];
			}

			Return_ValveLift_Analysis(file,
				cycle,
				previous_cycle,
				channel,
				&analysis->results[MAX_VALVE_LIFT].data[cycle],
				&analysis->results[MAX_VALVE_LIFT_CA].data[cycle],
				&analysis->results[MAX_VALVE_VELOCITY].data[cycle],
				&analysis->results[MAX_VALVE_VELOCITY_CA].data[cycle],
				&analysis->results[MAX_VALVE_ACCEL].data[cycle],
				&analysis->results[MAX_VALVE_ACCEL_CA].data[cycle],
				valve_velocity,
				valve_accel);
		}
#endif

		for (analysis_type = NUMBER_OF_ANALYSIS_CHANNELS; analysis_type < get_number_of_analysis_channels(); analysis_type++)
		{
			ReturnPluginAnalysis = (ReturnPluginAnalysisFunction)get_analysis_function(analysis_type);

			if ((ReturnPluginAnalysis != NULL) && (analysis->analysis_rqst[analysis_type] >= AnalysisRqst::CalculateAndStore))
			{
				ReturnPluginAnalysis(file, analysis, cycle, channel, analysis_type, geometry, &plugin_config[analysis_type - NUMBER_OF_ANALYSIS_CHANNELS], cycle + 1, 0, analysis->results[ENGINE_SPEED].data[cycle], file->number_of_cycles, 0);
			}
		}

		update_status(status, file->channel_data[channel].name, (float)cycle / (float)file->number_of_cycles);
	}

	if (analysis->analysis_rqst[MOVING_PRESSURE_AVERAGE] > AnalysisRqst::NotCalculated)
	{
		Complete_SmoothedPressure_Data(&smoothing_coefficients);
	}

	if (analysis->analysis_rqst[FFT_TOOTH_PERIOD] > AnalysisRqst::NotCalculated)
	{
		Complete_FFT(&fft_tooth_config);
	}

	for (analysis_type = NUMBER_OF_ANALYSIS_CHANNELS; analysis_type < get_number_of_analysis_channels(); analysis_type++)
	{
		CompletePluginChannels = (CompletePluginChannelsFunction)get_complete_function(analysis_type);

		if ((CompletePluginChannels != NULL) && (analysis->analysis_rqst[analysis_type] >= AnalysisRqst::CalculateAndStore))
		{
			CompletePluginChannels(file, analysis, channel, analysis_type, geometry, &plugin_config[analysis_type - NUMBER_OF_ANALYSIS_CHANNELS]);
		}
	}

	FINISH_RESULTS(MOVING_PRESSURE_AVERAGE, moving_pressure_average);
	FINISH_RESULTS(PRESSURE_RISE_RATE, pressure_rise_rate);
	FINISH_RESULTS(DIGITAL_SIGNAL, digital_signal);
	FINISH_RESULTS(VALVE_ACCEL, valve_accel);
	FINISH_RESULTS(VALVE_VELOCITY, valve_velocity);
	FINISH_RESULTS(TOOTH_SPEED, valve_velocity);

	FINISH_RESULTS(FFT_TOOTH_PERIOD, fft_tooth_period);
	FINISH_RESULTS(FFT_TOOTH_PERIOD_FREQUENCY, fft_tooth_period_frequency);

    free(dCA);
}

void fft_analysis(FileData* file,AnalysisChannel* analysis,const unsigned int channel, ProgressStatus* status)
{
	unsigned int cycle;
	FFTConfig fft_config;
	FFTConfig fft_ref_config;

	if ((file == NULL) || (analysis == NULL))
	{
		return;
	}

	if (channel >= file->number_of_channels)
	{
		return;
	}

	if (analysis->analysis_rqst[FFT] > AnalysisRqst::NotCalculated)
	{
		Init_FFTConfig(&fft_config);

		Prepare_FFT(file, channel, UNITS_KHZ, analysis->config.fft_start_window, analysis->config.fft_finish_window, FFT_WINDOW_HAMMING, 1, &fft_config);
	}

	if (analysis->analysis_rqst[FFT_REFERENCE] > AnalysisRqst::NotCalculated)
	{
		Init_FFTConfig(&fft_ref_config);

		Prepare_FFT(file, channel, UNITS_KHZ, analysis->config.fft_reference_start, analysis->config.fft_reference_finish, FFT_WINDOW_HAMMING, 1, &fft_ref_config);
	}

	if (analysis->analysis_rqst[ENGINE_SPEED] > AnalysisRqst::NotCalculated)
	{
		engine_speed_analysis(file, analysis->results[ENGINE_SPEED].data, channel);
	}

	CONFIGURE_FFT_RESULTS(FFT, fft, fft_config);
	CONFIGURE_FFT_RESULTS(FFT_FREQUENCY, fft_frequency, fft_config);
	CONFIGURE_FFT_RESULTS(FFT_REFERENCE, fft_reference, fft_ref_config);
	CONFIGURE_FFT_RESULTS(FFT_REFERENCE_FREQUENCY, fft_reference_frequency, fft_ref_config);

	/* Calculate Fourier Analysis */

	if (analysis->analysis_rqst[FFT] > AnalysisRqst::NotCalculated)
	{
		logmessage(NOTICE,"FFT\n");

		for (cycle=0;cycle<file->number_of_cycles;cycle++)
		{
			if (analysis->analysis_rqst[FFT_FREQUENCY] >= AnalysisRqst::CalculateAndStore)
			{
				fft_frequency = &analysis->results[FFT_FREQUENCY].data[cycle * analysis->results[FFT_FREQUENCY].samples_per_cycle];
			}

			if (analysis->analysis_rqst[FFT] >= AnalysisRqst::CalculateAndStore)
			{
				fft = &analysis->results[FFT].data[cycle * analysis->results[FFT].samples_per_cycle];
			}

			Return_FFT(&fft_config,
					   &file->channel_data[channel].data[cycle*file->channel_data[channel].samples_per_cycle],
			           analysis->results[ENGINE_SPEED].data[cycle],
				       false,
					   fft_frequency,
					   fft);
		}
	}

	if (analysis->analysis_rqst[FFT_REFERENCE] > AnalysisRqst::NotCalculated)
	{
		logmessage(NOTICE, "FFT (Reference Window)\n");

		for (cycle = 0; cycle<file->number_of_cycles; cycle++)
		{
			if (analysis->analysis_rqst[FFT_REFERENCE_FREQUENCY] >= AnalysisRqst::CalculateAndStore)
			{
				fft_reference_frequency = &analysis->results[FFT_REFERENCE_FREQUENCY].data[cycle * analysis->results[FFT_REFERENCE_FREQUENCY].samples_per_cycle];
			}

			if (analysis->analysis_rqst[FFT_REFERENCE] >= AnalysisRqst::CalculateAndStore)
			{
				fft_reference = &analysis->results[FFT_REFERENCE].data[cycle * analysis->results[FFT_REFERENCE].samples_per_cycle];
			}

			Return_FFT(&fft_ref_config,
				&file->channel_data[channel].data[cycle*file->channel_data[channel].samples_per_cycle],
				analysis->results[ENGINE_SPEED].data[cycle],
				false,
				fft_reference_frequency,
				fft_reference);
		}
	}

	if (analysis->analysis_rqst[FFT] > AnalysisRqst::NotCalculated)
	{
		Complete_FFT(&fft_config);
	}

	if (analysis->analysis_rqst[FFT_REFERENCE] > AnalysisRqst::NotCalculated)
	{
		Complete_FFT(&fft_ref_config);
	}

	FINISH_RESULTS(FFT, fft);
	FINISH_RESULTS(FFT_FREQUENCY, fft_frequency);
	FINISH_RESULTS(FFT_REFERENCE, fft_reference);
	FINISH_RESULTS(FFT_REFERENCE_FREQUENCY, fft_reference_frequency);
}

void pressure_cycle_analysis(FileData* file,AnalysisChannel* analysis,Analysis* file_analysis,ProgressStatus* status)
{
	unsigned int cycle;
	unsigned int samples_per_cycle;
	float* volume = NULL;
	float* dVolume = NULL;
	float* dCA = NULL;
	float* heat_transfer_area = NULL;
	float* logVolume = NULL;
	float* pressure_to_torque = NULL;
	float* tla_crank_angle_data = NULL;
	float* tla_pressure_data = NULL;
	//float* eeoc_value = NULL;
	float* cd_first = NULL;
	float* cd_second = NULL;
	float* cd_filt_sec = NULL;
	unsigned int poly_comp_number_of_points;
	unsigned int poly_comp_start_ca;
	float* poly_comp_logPressure = NULL;
	unsigned int poly_exp_number_of_points;
	unsigned int poly_exp_start_ca;
	float* poly_exp_logPressure = NULL;
	float total_angular_indicated_torque;
	FFTConfig fft_config;
	FFTConfig fft_knk_config;
	unsigned int smoothing_range;
	unsigned int smoothing_resolution;
	unsigned int smoothing_points;
	float* smoothing_coefficients = NULL;
	unsigned int channel;
	float k;
	unsigned int soc_config_channel = 0;
	ReturnPluginAnalysisFunction ReturnPluginAnalysis = NULL;
	CompletePluginChannelsFunction CompletePluginChannels = NULL;
	InitialisePluginChannelsFunction InitialisePluginChannels = NULL;
	unsigned int analysis_type;
	bool cylinder_deactivated;
	void** plugin_config = NULL;
	unsigned int number_of_plugin_channels;
	Geometry geometry;
	EEOCConfig eeoc_config;

	if ((file == NULL) || (analysis == NULL))
	{
		return;
	}

	channel = analysis->channel;

	if ((channel >= file->number_of_channels) || (file->channel_data == NULL) || (file->number_of_channels == 0))
	{
		return;
	}

	if ((file->channel_data[channel].samples_per_cycle == 0) || (file->channel_data[channel].data == NULL))
	{
		return;
	}

	samples_per_cycle = file->channel_data[channel].samples_per_cycle;

	volume = (float*)malloc(samples_per_cycle * sizeof(float));
	if (volume == NULL)
	{
		logmessage(FATAL,"Memory could not be allocated\n");
	}

	dVolume = (float*)malloc(samples_per_cycle * sizeof(float));
	if (dVolume == NULL)
	{
		logmessage(FATAL,"Memory could not be allocated\n");
	}

	dCA = (float*)malloc(samples_per_cycle * sizeof(float));
	if (dCA == NULL)
	{
		logmessage(FATAL,"Memory could not be allocated\n");
	}

	heat_transfer_area = (float*)malloc(samples_per_cycle * sizeof(float));
	if (heat_transfer_area == NULL)
	{
		logmessage(FATAL,"Memory could not be allocated\n");
	}

	logVolume = (float*)malloc(samples_per_cycle * sizeof(float));
	if (logVolume == NULL)
	{
		logmessage(FATAL,"Memory could not be allocated\n");
	}

	pressure_to_torque = (float*)malloc(samples_per_cycle * sizeof(float));
	if (pressure_to_torque == NULL)
	{
		logmessage(FATAL, "Memory could not be allocated\n");
	}
	
	Return_Geometry_Data(file, channel, dCA, volume, dVolume, heat_transfer_area, logVolume, pressure_to_torque);

	geometry.volume = volume;
	geometry.dVolume = dVolume;
	geometry.dCA = dCA;
	geometry.heat_transfer_area = heat_transfer_area;
	geometry.logVolume = logVolume;
	geometry.pressure_to_torque = pressure_to_torque;

	number_of_plugin_channels = get_number_of_plugin_channels();

	if (number_of_plugin_channels > 0)
	{
		plugin_config = (void**)malloc(number_of_plugin_channels * sizeof(void*));

		if (plugin_config == NULL)
		{
			logmessage(FATAL, "Memory could not be allocated\n");
		}

		memset(plugin_config, 0, number_of_plugin_channels * sizeof(void*));
	}

	if ((file->channel_data[channel].soc_config.type == SOC_CA_CHANNEL_AVG) ||
		(file->channel_data[channel].soc_config.type == SOC_CA_CHANNEL_FALL) ||
		(file->channel_data[channel].soc_config.type == SOC_CA_CHANNEL_RISE) ||
		(file->channel_data[channel].soc_config.type == SOC_CYC_CHANNEL))
	{
		if (channel_name_to_number(file, file->channel_data[channel].soc_config.channel_name, ABSCISSA_UNKNOWN, &soc_config_channel) == false)
		{
			file->channel_data[channel].soc_config.type = SOC_FIXED;
			file->channel_data[channel].soc_config.fixed_value = 0.0f;
		}
		else if (file->channel_data[soc_config_channel].loaded == false)
		{
			file->channel_data[channel].soc_config.type = SOC_FIXED;
			file->channel_data[channel].soc_config.fixed_value = 0.0f;

			logmessage(WARNING, "Not using channel %s for SOC as it is not loaded\n", file->channel_data[soc_config_channel].name);
		}
		else
		{
			/* Do Nothing */
		}
	}

	if (analysis->analysis_rqst[TLA] > AnalysisRqst::NotCalculated)
	{
		Prepare_TLA(file, analysis->config.tla_range, analysis->config.tla_method, &tla_crank_angle_data, &tla_pressure_data);
	}
	
	if (analysis->analysis_rqst[EEOC] > AnalysisRqst::NotCalculated)
	{
		eeoc_config.values = NULL;

		Prepare_EEOC(file, 
			channel, 
			analysis->config.eeoc_start_window,
			analysis->config.eeoc_finish_window,
			analysis->config.eeoc_index,
			analysis->config.eeoc_default,
			analysis->config.eeoc_method,
			&eeoc_config);
	}

	if (analysis->analysis_rqst[FFT] > AnalysisRqst::NotCalculated)
	{
		Init_FFTConfig(&fft_config);

		Prepare_FFT(file, channel, UNITS_KHZ, analysis->config.fft_start_window, analysis->config.fft_finish_window, FFT_WINDOW_HAMMING, 1, &fft_config);
	}
	
	if (analysis->analysis_rqst[FFT_KNOCK_PRESSURE] > AnalysisRqst::NotCalculated)
	{
		Init_FFTConfig(&fft_knk_config);

		Prepare_FFT(file, channel, UNITS_KHZ, analysis->config.pkp_start_angle, analysis->config.pkp_finish_angle, FFT_WINDOW_HAMMING, 1, &fft_knk_config);
	}

	if (analysis->analysis_rqst[KNOCK_C_AND_D] > AnalysisRqst::NotCalculated)
	{
		Prepare_CandD(file, channel, &cd_first, &cd_second, &cd_filt_sec);
	}
	
	if (analysis->analysis_rqst[POLY_COMP] > AnalysisRqst::NotCalculated)
	{
		Prepare_Polytropic_Index(file, channel, analysis->config.poly_comp_start_angle, analysis->config.poly_comp_finish_angle, &poly_comp_number_of_points, &poly_comp_start_ca, &poly_comp_logPressure);
	}
	
	if (analysis->analysis_rqst[POLY_EXP] > AnalysisRqst::NotCalculated)
	{
		Prepare_Polytropic_Index(file, channel, analysis->config.poly_exp_start_angle, analysis->config.poly_exp_finish_angle, &poly_exp_number_of_points, &poly_exp_start_ca, &poly_exp_logPressure);
	}
	
	if (analysis->analysis_rqst[MOVING_PRESSURE_AVERAGE] > AnalysisRqst::NotCalculated)
	{
		Prepare_SmoothedPressure_Data(file, analysis->config.smoothed_pressure_method, analysis->config.pkp_smoothing_range, analysis->config.pkp_smoothing_resolution, &smoothing_range, &smoothing_resolution, &smoothing_points, &smoothing_coefficients);
	}

	for (analysis_type = NUMBER_OF_ANALYSIS_CHANNELS; analysis_type < get_number_of_analysis_channels(); analysis_type++)
	{
		InitialisePluginChannels = (InitialisePluginChannelsFunction)get_initialise_function(analysis_type);

		if ((InitialisePluginChannels != NULL) && (analysis->analysis_rqst[analysis_type] >= AnalysisRqst::CalculateAndStore))
		{
			InitialisePluginChannels(file, analysis, channel, analysis_type, geometry, &plugin_config[analysis_type-NUMBER_OF_ANALYSIS_CHANNELS]);
		}
	}

	memset(analysis->cycle_classification, 0, analysis->number_of_cycles * sizeof(unsigned char));

	CONFIGURE_CA_RESULTS(NET_HEAT_RELEASE_RATE, net_heat_release_rate);
	CONFIGURE_CA_RESULTS(GROSS_HEAT_RELEASE_RATE, gross_heat_release_rate);
	CONFIGURE_CA_RESULTS(H_COEFF, h_coeff);
	CONFIGURE_CA_RESULTS(ANGULAR_TORQUE, angular_torque);
	CONFIGURE_CA_RESULTS(KNOCKING_PRESSURE, knocking_pressure);
	CONFIGURE_CA_RESULTS(MEAN_GAS_TEMP, mean_gas_temp);
	CONFIGURE_CA_RESULTS(GAMMA, gamma);
	CONFIGURE_CA_RESULTS(MFB, mfb);
	CONFIGURE_CA_RESULTS(MOTORED_PRESSURE, motored_pressure);
	CONFIGURE_CA_RESULTS(MOVING_PRESSURE_AVERAGE, moving_pressure_average);
	CONFIGURE_CA_RESULTS(POLYFIT, polyfit);
	CONFIGURE_CA_RESULTS(POLYTROPIC_INDICES, polytropic_indices);
	CONFIGURE_CA_RESULTS(PRESSURE_RISE_RATE, pressure_rise_rate);
	CONFIGURE_CA_RESULTS(WIEBE_MFB, wiebe_mfb);
	CONFIGURE_CA_RESULTS(DIFFERENCE_PRESSURE, difference_pressure);
	CONFIGURE_CA_RESULTS(PRESSURE_RATIO, pressure_ratio);
	CONFIGURE_CA_RESULTS(PRESSURE_RATIO_FRACTION, pressure_ratio_fraction);
	CONFIGURE_CA_RESULTS(EXTRAPOLATED_CYL_PRESSURE, extrapolated_cyl_pressure);
	CONFIGURE_CA_RESULTS(D2P, d2p);
	CONFIGURE_CA_RESULTS(CRANK_ANGLE_IMEP, crank_angle_imep);

	CONFIGURE_FFT_RESULTS(FFT_KNOCK_PRESSURE, fft_knock_pressure, fft_knk_config);
	CONFIGURE_FFT_RESULTS(FFT_KNOCK_PRESSURE_FREQUENCY, fft_knock_pressure_frequency, fft_knk_config);
	CONFIGURE_FFT_RESULTS(FFT, fft, fft_config);
	CONFIGURE_FFT_RESULTS(FFT_FREQUENCY, fft_frequency, fft_config);

	/* Calculate engine speed for each cycle */
	if (analysis->analysis_rqst[ENGINE_SPEED] > AnalysisRqst::NotCalculated)
	{
		engine_speed_analysis(file, analysis->results[ENGINE_SPEED].data, channel);
	}

	for (cycle = 0; cycle < file->number_of_cycles; cycle++)
	{
#ifdef __CATOOLRT__
		if (analysis->analysis_rqst[AVERAGE_EXHAUST_ABSOLUTE_PRESSURE] > AnalysisRqst::NotCalculated)
		{
			Return_Average_Exhaust_Absolute_Pressure(file,
				cycle,
				channel,
				240.0f,
				300.0f,
				&analysis->results[AVERAGE_EXHAUST_ABSOLUTE_PRESSURE].data[cycle]);
		}
#endif

		/* Calculate smoothed pressure curve for knock analysis */
		if (analysis->analysis_rqst[MOVING_PRESSURE_AVERAGE] > AnalysisRqst::NotCalculated)
		{
			if (analysis->analysis_rqst[MOVING_PRESSURE_AVERAGE] >= AnalysisRqst::CalculateAndStore)
			{
				moving_pressure_average = &analysis->results[MOVING_PRESSURE_AVERAGE].data[cycle * samples_per_cycle];
			}

			Return_SmoothedPressure_Data(file,
				cycle,
				channel,
				smoothing_range,
				smoothing_resolution,
				smoothing_points,
				smoothing_coefficients,
				moving_pressure_average);
		}

		/* Polytropic indices */
		if (analysis->analysis_rqst[POLY_EXP] > AnalysisRqst::NotCalculated)
		{
			if (analysis->analysis_rqst[MOVING_PRESSURE_AVERAGE] >= AnalysisRqst::CalculateAndStore)
			{
				moving_pressure_average = &analysis->results[MOVING_PRESSURE_AVERAGE].data[cycle * samples_per_cycle];
			}

			Return_Polytropic_Index(file,
				cycle,
				channel,
				poly_exp_start_ca,
				poly_exp_number_of_points,
				logVolume,
				poly_exp_logPressure,
				moving_pressure_average,
				&analysis->results[POLY_EXP].data[cycle],
				&k);
		}

		if (analysis->analysis_rqst[POLY_COMP] > AnalysisRqst::NotCalculated)
		{
			if (analysis->analysis_rqst[MOVING_PRESSURE_AVERAGE] >= AnalysisRqst::CalculateAndStore)
			{
				moving_pressure_average = &analysis->results[MOVING_PRESSURE_AVERAGE].data[cycle * samples_per_cycle];
			}

			Return_Polytropic_Index(file,
				cycle,
				channel,
				poly_comp_start_ca,
				poly_comp_number_of_points,
				logVolume,
				poly_comp_logPressure,
				moving_pressure_average,
				&analysis->results[POLY_COMP].data[cycle],
				&k);
		}

		/* Calculate generic min/max/mean */
		if (analysis->analysis_rqst[MINIMUM_VALUE] > AnalysisRqst::NotCalculated)
		{
			Return_MinMaxMean_Analysis(file,
				cycle,
				channel,
				&analysis->results[MINIMUM_VALUE].data[cycle],
				&analysis->results[MAXIMUM_VALUE].data[cycle],
				&analysis->results[MEAN_VALUE].data[cycle]);
		}

		/* Determine EEOC */
		if (analysis->analysis_rqst[EEOC] > AnalysisRqst::NotCalculated)
		{
			Return_EEOC(file,
				cycle,
				channel,
				&eeoc_config,
				&analysis->results[EEOC].data[cycle]);
		}

		/* Calculate motored pressure data */

		if (analysis->analysis_rqst[MOTORED_PRESSURE] > AnalysisRqst::NotCalculated)
		{
			if (analysis->analysis_rqst[MOVING_PRESSURE_AVERAGE] >= AnalysisRqst::CalculateAndStore)
			{
				moving_pressure_average = &analysis->results[MOVING_PRESSURE_AVERAGE].data[cycle * samples_per_cycle];
			}

			if (analysis->analysis_rqst[MOTORED_PRESSURE] >= AnalysisRqst::CalculateAndStore)
			{
				motored_pressure = &analysis->results[MOTORED_PRESSURE].data[cycle * samples_per_cycle];
			}

			Return_Motored_Pressure_Data(file,
				channel,
				analysis->config.motored_pressure_method,
				analysis->config.poly_comp_finish_angle,
				analysis->results[POLY_COMP].data[cycle],
				moving_pressure_average,
				volume,
				motored_pressure);
		}

#ifdef __CATOOLRT__
		/* Pressure Ratio*/
		if (analysis->analysis_rqst[PRESSURE_RATIO] > AnalysisRqst::NotCalculated)
		{
			if (analysis->analysis_rqst[MOVING_PRESSURE_AVERAGE] >= AnalysisRqst::CalculateAndStore)
			{
				moving_pressure_average = &analysis->results[MOVING_PRESSURE_AVERAGE].data[cycle * samples_per_cycle];
			}

			if (analysis->analysis_rqst[MOTORED_PRESSURE] >= AnalysisRqst::CalculateAndStore)
			{
				motored_pressure = &analysis->results[MOTORED_PRESSURE].data[cycle * samples_per_cycle];
			}

			if (analysis->analysis_rqst[DIFFERENCE_PRESSURE] >= AnalysisRqst::CalculateAndStore)
			{
				difference_pressure = &analysis->results[DIFFERENCE_PRESSURE].data[cycle * samples_per_cycle];
			}

			if (analysis->analysis_rqst[PRESSURE_RATIO] >= AnalysisRqst::CalculateAndStore)
			{
				pressure_ratio = &analysis->results[PRESSURE_RATIO].data[cycle * samples_per_cycle];
			}

			if (analysis->analysis_rqst[PRESSURE_RATIO_FRACTION] >= AnalysisRqst::CalculateAndStore)
			{
				pressure_ratio_fraction = &analysis->results[PRESSURE_RATIO_FRACTION].data[cycle * samples_per_cycle];
			}

			Return_Difference_Pressure(file,
				cycle,
				channel,
				motored_pressure,
				moving_pressure_average,
				&analysis->results[DIFFERENCE_PRESSURE_10].data[cycle],
				&analysis->results[NORMALISED_DIFF_PR_50].data[cycle],
				&analysis->results[PRESSURE_RATIO_MGMT_10].data[cycle],
				&analysis->results[PRESSURE_RATIO_MGMT_25].data[cycle],
				&analysis->results[DILPAR].data[cycle],
				&analysis->results[DIFFERENCE_PRESSURE_INTEGRAL].data[cycle],
				difference_pressure,
				pressure_ratio,
				pressure_ratio_fraction);
		}

		/* Calculate Air Charge Estimation */
		if (analysis->analysis_rqst[DELTA_P] > AnalysisRqst::NotCalculated)
		{
			if (analysis->analysis_rqst[MOVING_PRESSURE_AVERAGE] >= AnalysisRqst::CalculateAndStore)
			{
				moving_pressure_average = &analysis->results[MOVING_PRESSURE_AVERAGE].data[cycle * samples_per_cycle];
			}

			Return_Mass_Estimator_Data(file,
				channel,
				analysis->config.poly_comp_start_angle,
				analysis->config.poly_comp_finish_angle,
				analysis->config.t_ivc,
				analysis->results[POLY_COMP].data[cycle],
				analysis->config.R,
				volume,
				moving_pressure_average,
				&analysis->results[DELTA_P].data[cycle],
				&analysis->results[TRAPPED_MASS].data[cycle],
				&analysis->results[LOAD].data[cycle]);
		}
#endif

		/* Calculate IMEP */
		if (analysis->analysis_rqst[GROSS_IMEP] > AnalysisRqst::NotCalculated)
		{
			if (analysis->analysis_rqst[ANGULAR_TORQUE] >= AnalysisRqst::CalculateAndStore)
			{
				angular_torque = &analysis->results[ANGULAR_TORQUE].data[cycle * samples_per_cycle];
			}

			if (analysis->analysis_rqst[CRANK_ANGLE_IMEP] >= AnalysisRqst::CalculateAndStore)
			{
				crank_angle_imep = &analysis->results[CRANK_ANGLE_IMEP].data[cycle * samples_per_cycle];
			}

			Return_IMEP(file,
				cycle,
				channel,
				volume,
				dVolume,
				pressure_to_torque,
				analysis->config.crankcase_pressure,
				analysis->config.imep_method,
				analysis->config.deactivation_delta_p,
				&analysis->results[GROSS_IMEP].data[cycle],
				&analysis->results[NET_IMEP].data[cycle],
				&analysis->results[UPPER_PUMPING_IMEP].data[cycle],
				&analysis->results[LOWER_PUMPING_IMEP].data[cycle],
				&analysis->results[PUMPING_IMEP].data[cycle],
				&analysis->results[INDICATED_TORQUE_NET].data[cycle],
				&analysis->results[INDICATED_TORQUE].data[cycle],
				angular_torque,
				&total_angular_indicated_torque,
				crank_angle_imep,
				&cylinder_deactivated);

			if (cylinder_deactivated == true)
			{
				analysis->cycle_classification[cycle] |= CYCLE_DEACTIVATED & 0xFF;
			}
			else if (analysis->results[GROSS_IMEP].data[cycle] < analysis->config.misfire_imep)
			{
				analysis->cycle_classification[cycle] |= CYCLE_MISFIRE & 0xFF;
			}
			else if (analysis->results[GROSS_IMEP].data[cycle] < analysis->config.slowburn_imep)
			{
				analysis->cycle_classification[cycle] |= CYCLE_SLOW_BURN & 0xFF;
			}
			else
			{
				/* Do Nothing */
			}
		}

		if (analysis->analysis_rqst[INDICATED_POWER] > AnalysisRqst::NotCalculated)
		{
			Return_Indicated_Data(file,
				analysis->results[INDICATED_TORQUE].data[cycle],
				analysis->results[ENGINE_SPEED].data[cycle],
				&analysis->results[INDICATED_POWER].data[cycle],
				&analysis->results[INDICATED_POWER_HP].data[cycle],
				&analysis->results[INDICATED_TORQUE_LBFT].data[cycle]);
		}

		if (analysis->analysis_rqst[MISFIRE_F] > AnalysisRqst::NotCalculated)
		{
			Return_Misfire_Data(file,
				0,
				cycle,
				cycle,
				analysis->config.misfire_cycles_f,
				analysis->config.misfire_cycles_m,
				analysis->config.misfire_cycles_s,
				analysis->cycle_classification,
				&analysis->results[MISFIRE_F].data[cycle],
				&analysis->results[MISFIRE_M].data[cycle],
				&analysis->results[MISFIRE_S].data[cycle]);
		}

		/* Calculate maximum pressure and maximum pressure rise rate */
		if (analysis->analysis_rqst[MIN_PRESSURE] >= AnalysisRqst::CalculateAndStore)
		{
			if (analysis->analysis_rqst[PRESSURE_RISE_RATE] >= AnalysisRqst::CalculateAndStore)
			{
				pressure_rise_rate = &analysis->results[PRESSURE_RISE_RATE].data[cycle * samples_per_cycle];
			}

			Return_Pressure_Analysis(file,
				cycle,
				channel,
				analysis->config.pressure_rise_start_angle,
				analysis->config.pressure_rise_finish_angle,
				analysis->config.pressure_rise_range,
				&analysis->results[MIN_PRESSURE].data[cycle],
				&analysis->results[MIN_PRESSURE_CA].data[cycle],
				&analysis->results[MAX_PRESSURE].data[cycle],
				&analysis->results[MAX_PRESSURE_CA].data[cycle],
				&analysis->results[MAX_PRESSURE_RISE_RATE].data[cycle],
				&analysis->results[MAX_PRESSURE_RISE_RATE_CA].data[cycle],
				pressure_rise_rate);
		}
#ifdef __CATOOLRT__
		/* Calculate pressure second derivative */
		if (analysis->analysis_rqst[MAX_D2P] > AnalysisRqst::NotCalculated)
		{
			if (analysis->analysis_rqst[D2P] >= AnalysisRqst::CalculateAndStore)
			{
				d2p = &analysis->results[D2P].data[cycle * samples_per_cycle];
			}

			Return_Pressure_Second_Derivative(file,
				cycle,
				channel,
				analysis->config.d2p_window,
				&analysis->results[MAX_D2P].data[cycle],
				&analysis->results[MAX_D2P_CA].data[cycle],
				d2p);
		}
#endif
		/* Calculate maximum mean gas temperature */
		if (analysis->analysis_rqst[MAX_MEAN_GAS_TEMP] > AnalysisRqst::NotCalculated)
		{
			if (analysis->analysis_rqst[MOVING_PRESSURE_AVERAGE] >= AnalysisRqst::CalculateAndStore)
			{
				moving_pressure_average = &analysis->results[MOVING_PRESSURE_AVERAGE].data[cycle * samples_per_cycle];
			}

			if (analysis->analysis_rqst[MEAN_GAS_TEMP] >= AnalysisRqst::CalculateAndStore)
			{
				mean_gas_temp = &analysis->results[MEAN_GAS_TEMP].data[cycle * samples_per_cycle];
			}

			Return_Temperature_Data(file,
				channel,
				cycle,
				volume,
				analysis->config.temp_ref_ca,
				analysis->config.t_ivc,
				0.0f,/*analysis->results[POLY_COMP].data[cycle],*/
				analysis->config.gas_temp_model,
				moving_pressure_average,
				&analysis->results[MAX_MEAN_GAS_TEMP].data[cycle],
				&analysis->results[MAX_MEAN_GAS_TEMP_CA].data[cycle],
				mean_gas_temp);
		}

		/* Calculate peak knocking pressure */

		if (analysis->analysis_rqst[PEAK_KNOCKING_PRESSURE] > AnalysisRqst::NotCalculated)
		{
			if (analysis->analysis_rqst[MOVING_PRESSURE_AVERAGE] >= AnalysisRqst::CalculateAndStore)
			{
				moving_pressure_average = &analysis->results[MOVING_PRESSURE_AVERAGE].data[cycle * samples_per_cycle];
			}

			if (analysis->analysis_rqst[KNOCKING_PRESSURE] >= AnalysisRqst::CalculateAndStore)
			{
				knocking_pressure = &analysis->results[KNOCKING_PRESSURE].data[cycle * samples_per_cycle];
			}

			Return_PKP_Data(file,
				cycle,
				channel,
				dCA,
				moving_pressure_average,
				analysis->config.pkp_start_angle,
				analysis->config.pkp_finish_angle,
				analysis->config.knock_integral_type,
				analysis->config.knock_onset_threshold,
				&analysis->results[PEAK_KNOCKING_PRESSURE].data[cycle],
				&analysis->results[PEAK_KNOCKING_PRESSURE_CA].data[cycle],
				&analysis->results[KNOCK_INTEGRAL].data[cycle],
				&analysis->results[KNOCK_ONSET_CA].data[cycle],
				knocking_pressure);

			if (analysis->results[PEAK_KNOCKING_PRESSURE].data[cycle] > analysis->config.mega_knock_pkp)
			{
				analysis->cycle_classification[cycle] |= CYCLE_MEGA_KNOCK & 0xFF;
			}
			else if (analysis->results[PEAK_KNOCKING_PRESSURE].data[cycle] > analysis->config.knock_pkp)
			{
				analysis->cycle_classification[cycle] |= CYCLE_KNOCK & 0xFF;
			}
			else
			{
				/* Do Nothing */
			}
		}

		/* Calculate knock factor */
		if (analysis->analysis_rqst[KNOCK_FACTOR] > AnalysisRqst::NotCalculated)
		{
			if (analysis->analysis_rqst[KNOCKING_PRESSURE] >= AnalysisRqst::CalculateAndStore)
			{
				knocking_pressure = &analysis->results[KNOCKING_PRESSURE].data[cycle * samples_per_cycle];
			}

			Return_Knock_Factor(file,
				channel,
				analysis->results[ENGINE_SPEED].data[cycle],
				analysis->results[MAX_PRESSURE_CA].data[cycle],
				knocking_pressure,
				30.0f,
				30.0f,
				1000.0f,
				6000.0f,
				0.0f,
				10.0f,
				0.1f,
				&analysis->results[KNOCK_FACTOR].data[cycle]);
		}

		/* Calculate knock signal energy */
		if (analysis->analysis_rqst[KNOCK_SIGNAL_ENERGY] > AnalysisRqst::NotCalculated)
		{
			if (analysis->analysis_rqst[KNOCKING_PRESSURE] >= AnalysisRqst::CalculateAndStore)
			{
				knocking_pressure = &analysis->results[KNOCKING_PRESSURE].data[cycle * samples_per_cycle];
			}

			Return_Knock_Signal_Energy(file,
				channel,
				dCA,
				analysis->results[ENGINE_SPEED].data[cycle],
				analysis->results[KNOCK_ONSET_CA].data[cycle],
				knocking_pressure,
				&analysis->results[KNOCK_SIGNAL_ENERGY].data[cycle]);
		}

#ifdef __CATOOLRT__
		/* Calculate Fourier Analysis */
		if (analysis->analysis_rqst[FFT] > AnalysisRqst::NotCalculated)
		{
			if (analysis->analysis_rqst[FFT_FREQUENCY] >= AnalysisRqst::CalculateAndStore)
			{
				fft_frequency = &analysis->results[FFT_FREQUENCY].data[cycle * analysis->results[FFT_FREQUENCY].samples_per_cycle];
			}

			if (analysis->analysis_rqst[FFT] >= AnalysisRqst::CalculateAndStore)
			{
				fft = &analysis->results[FFT].data[cycle * analysis->results[FFT].samples_per_cycle];
			}

			Return_FFT(&fft_config,
				&file->channel_data[channel].data[cycle * file->channel_data[channel].samples_per_cycle],
				analysis->results[ENGINE_SPEED].data[cycle],
				false,
				fft_frequency,
				fft);
		}

		/* Calculate Fourier Analysis of knock window */
		if (analysis->analysis_rqst[FFT_KNOCK_PRESSURE] > AnalysisRqst::NotCalculated)
		{
			if (analysis->analysis_rqst[KNOCKING_PRESSURE] >= AnalysisRqst::CalculateAndStore)
			{
				knocking_pressure = &analysis->results[KNOCKING_PRESSURE].data[cycle * samples_per_cycle];
			}

			if (analysis->analysis_rqst[FFT_KNOCK_PRESSURE_FREQUENCY] >= AnalysisRqst::CalculateAndStore)
			{
				fft_knock_pressure_frequency = &analysis->results[FFT_KNOCK_PRESSURE_FREQUENCY].data[cycle * analysis->results[FFT_KNOCK_PRESSURE_FREQUENCY].samples_per_cycle];
			}

			if (analysis->analysis_rqst[FFT_KNOCK_PRESSURE] >= AnalysisRqst::CalculateAndStore)
			{
				fft_knock_pressure = &analysis->results[FFT_KNOCK_PRESSURE].data[cycle * analysis->results[FFT_KNOCK_PRESSURE].samples_per_cycle];
			}

			Return_FFT(&fft_knk_config,
				knocking_pressure,
				analysis->results[ENGINE_SPEED].data[cycle],
				false,
				fft_knock_pressure_frequency,
				fft_knock_pressure);
		}
#endif
		/* Calculate Checkel & Dale Knock Metrics */
		if (analysis->analysis_rqst[KNOCK_C_AND_D] > AnalysisRqst::NotCalculated)
		{
			Return_CandD_Data(file,
				cycle,
				channel,
				dCA,
				analysis->config.cd_start_angle,
				analysis->config.cd_finish_angle,
				cd_first,
				cd_second,
				cd_filt_sec,
				&analysis->results[KNOCK_C_AND_D].data[cycle],
				&analysis->results[KNOCK_C_AND_D_CA].data[cycle],
				NULL);
		}

		/* Calculate polytropic index */
		if (analysis->analysis_rqst[POLYTROPIC_INDICES] > AnalysisRqst::NotCalculated)
		{
			if (analysis->analysis_rqst[MOVING_PRESSURE_AVERAGE] >= AnalysisRqst::CalculateAndStore)
			{
				moving_pressure_average = &analysis->results[MOVING_PRESSURE_AVERAGE].data[cycle * samples_per_cycle];
			}

			if (analysis->analysis_rqst[POLYTROPIC_INDICES] >= AnalysisRqst::CalculateAndStore)
			{
				polytropic_indices = &analysis->results[POLYTROPIC_INDICES].data[cycle * samples_per_cycle];
			}

			Return_Polytropic_Indices_Data(file,
				channel,
				volume,
				moving_pressure_average,
				polytropic_indices);
		}

		/* Calculate temperate dependent gamma */
		if (analysis->analysis_rqst[GAMMA] > AnalysisRqst::NotCalculated)
		{
			if (analysis->analysis_rqst[POLYTROPIC_INDICES] >= AnalysisRqst::CalculateAndStore)
			{
				polytropic_indices = &analysis->results[POLYTROPIC_INDICES].data[cycle * samples_per_cycle];
			}

			if (analysis->analysis_rqst[MEAN_GAS_TEMP] >= AnalysisRqst::CalculateAndStore)
			{
				mean_gas_temp = &analysis->results[MEAN_GAS_TEMP].data[cycle * samples_per_cycle];
			}

			if (analysis->analysis_rqst[GAMMA] >= AnalysisRqst::CalculateAndStore)
			{
				gamma = &analysis->results[GAMMA].data[cycle * samples_per_cycle];
			}

			Return_Gamma_Data(file,
				channel,
				analysis->config.gamma_method,
				analysis->results[POLY_COMP].data[cycle],
				analysis->results[POLY_EXP].data[cycle],
				analysis->config.mfb_n,
				analysis->config.R,
				polytropic_indices,
				mean_gas_temp,
				gamma);
		}

#ifdef __CATOOLRT__
		if (analysis->analysis_rqst[EIVC] > AnalysisRqst::NotCalculated)
		{
			if (analysis->analysis_rqst[MOVING_PRESSURE_AVERAGE] >= AnalysisRqst::CalculateAndStore)
			{
				moving_pressure_average = &analysis->results[MOVING_PRESSURE_AVERAGE].data[cycle * samples_per_cycle];
			}

			if (analysis->analysis_rqst[POLYTROPIC_INDICES] >= AnalysisRqst::CalculateAndStore)
			{
				polytropic_indices = &analysis->results[POLYTROPIC_INDICES].data[cycle * samples_per_cycle];
			}

			if (analysis->analysis_rqst[EXTRAPOLATED_CYL_PRESSURE] >= AnalysisRqst::CalculateAndStore)
			{
				extrapolated_cyl_pressure = &analysis->results[EXTRAPOLATED_CYL_PRESSURE].data[cycle * samples_per_cycle];
			}

			Return_Valve_Timing_Estimation(file,
				cycle,
				channel,
				volume,
				analysis->config.poly_comp_start_angle,
				analysis->config.poly_comp_finish_angle,
				analysis->config.poly_exp_start_angle,
				analysis->config.poly_exp_finish_angle,
				analysis->results[POLY_COMP].data[cycle],
				analysis->results[POLY_EXP].data[cycle],
				file->engine.ivc_angle,
				file->engine.evo_angle,
				moving_pressure_average,
				polytropic_indices,
				&analysis->results[EIVC].data[cycle],
				&analysis->results[EEVO].data[cycle],
				extrapolated_cyl_pressure);
		}
#endif

		/* Determine SOC */
		if (analysis->analysis_rqst[START_OF_COMBUSTION] > AnalysisRqst::NotCalculated)
		{
			if (file->channel_data[channel].soc_config.type == SOC_POSITIVE_HR)
			{
				if (analysis->analysis_rqst[MOVING_PRESSURE_AVERAGE] >= AnalysisRqst::CalculateAndStore)
				{
					moving_pressure_average = &analysis->results[MOVING_PRESSURE_AVERAGE].data[cycle * samples_per_cycle];
				}

				Return_Postitive_HR_SOC(file,
					channel,
					file->engine.ivc_angle,
					file->engine.evo_angle,
					analysis->results[POLY_COMP].data[cycle],
					analysis->results[POLY_EXP].data[cycle],
					analysis->results[MAX_PRESSURE_CA].data[cycle],
					volume,
					moving_pressure_average,
					analysis->config.soc_threshold,
					&analysis->results[START_OF_COMBUSTION].data[cycle]);
			}
			else
			{
				Return_SOC(file,
					file_analysis,
					cycle,
					channel,
					NULL,
					soc_config_channel,
					file->channel_data[channel].soc_config.type,
					file->channel_data[channel].soc_config.start_window,
					file->channel_data[channel].soc_config.finish_window,
					file->channel_data[channel].soc_config.fixed_value,
					START_OF_COMBUSTION,
					file->channel_data[channel].soc_config.aligned,
					file->channel_data[channel].soc_config.invert,
					false,
					file->number_of_cycles,
					dCA,
					&analysis->results[START_OF_COMBUSTION].data[cycle]);
			}
		}

		/* Calculate MFB data */
		if (analysis->analysis_rqst[MAX_BURN_RATE] > AnalysisRqst::NotCalculated)
		{
			if (analysis->analysis_rqst[MOVING_PRESSURE_AVERAGE] >= AnalysisRqst::CalculateAndStore)
			{
				moving_pressure_average = &analysis->results[MOVING_PRESSURE_AVERAGE].data[cycle * samples_per_cycle];
			}

			if (analysis->analysis_rqst[MFB] >= AnalysisRqst::CalculateAndStore)
			{
				mfb = &analysis->results[MFB].data[cycle * samples_per_cycle];
			}

			Return_MFB_Data(file,
				cycle,
				channel,
				volume,
				dVolume,
				dCA,
				analysis->config.mfb_n,
				analysis->config.mfb_model,
				analysis->results[START_OF_COMBUSTION].data[cycle],
				analysis->results[EEOC].data[cycle],
				moving_pressure_average,
				&analysis->results[MAX_BURN_RATE].data[cycle],
				&analysis->results[MAX_BURN_RATE_CA].data[cycle],
				mfb);
		}

		if (analysis->analysis_rqst[BURN_ANGLE_1] > AnalysisRqst::NotCalculated)
		{
			if (analysis->analysis_rqst[MFB] >= AnalysisRqst::CalculateAndStore)
			{
				mfb = &analysis->results[MFB].data[cycle * samples_per_cycle];
			}

			Return_Burn_Angles(file,
				channel,
				mfb,
				analysis->results[START_OF_COMBUSTION].data[cycle],
				analysis->results[EEOC].data[cycle],
				analysis->results[KNOCK_ONSET_CA].data[cycle],
				analysis->config.interpolate_mfb,
				&analysis->results[BURN_ANGLE_1].data[cycle],
				&analysis->results[BURN_ANGLE_2].data[cycle],
				&analysis->results[BURN_ANGLE_5].data[cycle],
				&analysis->results[BURN_ANGLE_10].data[cycle],
				&analysis->results[BURN_ANGLE_20].data[cycle],
				&analysis->results[BURN_ANGLE_25].data[cycle],
				&analysis->results[BURN_ANGLE_50].data[cycle],
				&analysis->results[BURN_ANGLE_75].data[cycle],
				&analysis->results[BURN_ANGLE_80].data[cycle],
				&analysis->results[BURN_ANGLE_90].data[cycle],
				&analysis->results[BURN_ANGLE_95].data[cycle],
				&analysis->results[BURN_ANGLE_98].data[cycle],
				&analysis->results[BURN_ANGLE_99].data[cycle],
				&analysis->results[BURN_DURATION_0_2].data[cycle],
				&analysis->results[BURN_DURATION_0_5].data[cycle],
				&analysis->results[BURN_DURATION_0_10].data[cycle],
				&analysis->results[BURN_DURATION_0_90].data[cycle],
				&analysis->results[BURN_DURATION_2_90].data[cycle],
				&analysis->results[BURN_DURATION_5_90].data[cycle],
				&analysis->results[BURN_DURATION_10_90].data[cycle],
				&analysis->results[KNOCK_ONSET_UMFB].data[cycle]);
		}

		if (analysis->analysis_rqst[CENTRE_OF_GRAVITY] > AnalysisRqst::NotCalculated)
		{
			if (analysis->analysis_rqst[MFB] >= AnalysisRqst::CalculateAndStore)
			{
				mfb = &analysis->results[MFB].data[cycle * samples_per_cycle];
			}

			Return_COG(file,
				channel,
				analysis->results[BURN_ANGLE_10].data[cycle],
				analysis->results[BURN_ANGLE_90].data[cycle],
				mfb,
				&analysis->results[CENTRE_OF_GRAVITY].data[cycle]);
		}

		/* Calculate heat release data */

		if (analysis->analysis_rqst[MAX_HEAT_RELEASE_RATE] > AnalysisRqst::NotCalculated)
		{
			if (analysis->analysis_rqst[MEAN_GAS_TEMP] >= AnalysisRqst::CalculateAndStore)
			{
				mean_gas_temp = &analysis->results[MEAN_GAS_TEMP].data[cycle * samples_per_cycle];
			}

			if (analysis->analysis_rqst[GAMMA] >= AnalysisRqst::CalculateAndStore)
			{
				gamma = &analysis->results[GAMMA].data[cycle * samples_per_cycle];
			}

			if (analysis->analysis_rqst[MOVING_PRESSURE_AVERAGE] >= AnalysisRqst::CalculateAndStore)
			{
				moving_pressure_average = &analysis->results[MOVING_PRESSURE_AVERAGE].data[cycle * samples_per_cycle];
			}

			if (analysis->analysis_rqst[NET_HEAT_RELEASE_RATE] >= AnalysisRqst::CalculateAndStore)
			{
				net_heat_release_rate = &analysis->results[NET_HEAT_RELEASE_RATE].data[cycle * samples_per_cycle];
			}

			if (analysis->analysis_rqst[GROSS_HEAT_RELEASE_RATE] >= AnalysisRqst::CalculateAndStore)
			{
				gross_heat_release_rate = &analysis->results[GROSS_HEAT_RELEASE_RATE].data[cycle * samples_per_cycle];
			}

			if (analysis->analysis_rqst[H_COEFF] >= AnalysisRqst::CalculateAndStore)
			{
				h_coeff = &analysis->results[H_COEFF].data[cycle * samples_per_cycle];
			}

			Return_Heat_Release_Data(file,
				channel,
				cycle,
				dCA,
				volume,
				dVolume,
				heat_transfer_area,
				mean_gas_temp,
				gamma,
				moving_pressure_average,
				analysis->config.t_ivc,
				analysis->config.t_wall,
				analysis->config.R,
				analysis->results[START_OF_COMBUSTION].data[cycle],
				analysis->results[EEOC].data[cycle],
				analysis->results[ENGINE_SPEED].data[cycle],
				analysis->results[POLY_COMP].data[cycle],
				analysis->results[POLY_EXP].data[cycle],
				analysis->config.heat_release_model,
				analysis->config.heat_transfer_model,
				analysis->config.heat_release_window,
				analysis->config.annand_a,
				&analysis->results[MAX_HEAT_RELEASE_RATE].data[cycle],
				&analysis->results[MAX_HEAT_RELEASE_RATE_CA].data[cycle],
				&analysis->results[TOTAL_HEAT_RELEASE].data[cycle],
				net_heat_release_rate,
				gross_heat_release_rate,
				h_coeff);
		}

		/* Calculate TLA */

		if (analysis->analysis_rqst[TLA] > AnalysisRqst::NotCalculated)
		{
			if (analysis->analysis_rqst[MOVING_PRESSURE_AVERAGE] >= AnalysisRqst::CalculateAndStore)
			{
				moving_pressure_average = &analysis->results[MOVING_PRESSURE_AVERAGE].data[cycle * samples_per_cycle];
			}

			if (analysis->analysis_rqst[POLYFIT] >= AnalysisRqst::CalculateAndStore)
			{
				polyfit = &analysis->results[POLYFIT].data[cycle * samples_per_cycle];
			}

			Return_TLA(file,
				cycle,
				channel,
				analysis->config.tla_method,
				analysis->config.tla_range,
				tla_crank_angle_data,
				tla_pressure_data,
				analysis->results[MAX_PRESSURE_CA].data[cycle],
				moving_pressure_average,
				&analysis->results[TLA].data[cycle],
				NULL,
				polyfit);
		}

		/* Calculate Wiebe functions */

		if (analysis->analysis_rqst[WIEBE_A] > AnalysisRqst::NotCalculated)
		{
			if (analysis->analysis_rqst[WIEBE_MFB] >= AnalysisRqst::CalculateAndStore)
			{
				wiebe_mfb = &analysis->results[WIEBE_MFB].data[cycle * samples_per_cycle];
			}

			Return_Wiebe(file,
				channel,
				analysis->results[START_OF_COMBUSTION].data[cycle],
				analysis->results[EEOC].data[cycle],
				analysis->results[BURN_ANGLE_5].data[cycle],
				analysis->results[BURN_ANGLE_10].data[cycle],
				analysis->results[BURN_ANGLE_50].data[cycle],
				analysis->results[BURN_ANGLE_90].data[cycle],
				analysis->results[BURN_ANGLE_95].data[cycle],
				analysis->config.wiebe_a_start,
				analysis->config.wiebe_a_finish,
				analysis->config.wiebe_a_step,
				analysis->config.wiebe_m_start,
				analysis->config.wiebe_m_finish,
				analysis->config.wiebe_m_step,
				&analysis->results[WIEBE_A].data[cycle],
				&analysis->results[WIEBE_M].data[cycle],
				wiebe_mfb);
		}

		for (analysis_type = NUMBER_OF_ANALYSIS_CHANNELS; analysis_type < get_number_of_analysis_channels(); analysis_type++)
		{
			ReturnPluginAnalysis = (ReturnPluginAnalysisFunction)get_analysis_function(analysis_type);

			if ((ReturnPluginAnalysis != NULL) && (analysis->analysis_rqst[analysis_type] >= AnalysisRqst::CalculateAndStore))
			{
				ReturnPluginAnalysis(file, analysis, cycle, channel, analysis_type, geometry, &plugin_config[analysis_type - NUMBER_OF_ANALYSIS_CHANNELS], cycle + 1, 0, analysis->results[ENGINE_SPEED].data[cycle], file->number_of_cycles, 0);
			}
		}

		update_status(status,file->channel_data[channel].name, (float)cycle / (float)file->number_of_cycles);
	}
	
	if (analysis->analysis_rqst[TLA] > AnalysisRqst::NotCalculated)
	{
		Complete_TLA(analysis->config.tla_method, tla_crank_angle_data, tla_pressure_data);
	}
	
	if (analysis->analysis_rqst[EEOC] > AnalysisRqst::NotCalculated)
	{
		Complete_EEOC(&eeoc_config);
	}
	if (analysis->analysis_rqst[KNOCK_C_AND_D] > AnalysisRqst::NotCalculated)
	{
		Complete_CandD(cd_first, cd_second, cd_filt_sec);
	}

	if (analysis->analysis_rqst[POLY_COMP] > AnalysisRqst::NotCalculated)
	{
		Complete_Polytropic_Index(poly_comp_logPressure);
	}
	
	if (analysis->analysis_rqst[POLY_EXP] > AnalysisRqst::NotCalculated)
	{
		Complete_Polytropic_Index(poly_exp_logPressure);
	}

	if (analysis->analysis_rqst[FFT] > AnalysisRqst::NotCalculated)
	{
		Complete_FFT(&fft_config);
	}

	if (analysis->analysis_rqst[MOVING_PRESSURE_AVERAGE] > AnalysisRqst::NotCalculated)
	{
		Complete_SmoothedPressure_Data(&smoothing_coefficients);
	}

	for (analysis_type = NUMBER_OF_ANALYSIS_CHANNELS; analysis_type < get_number_of_analysis_channels(); analysis_type++)
	{
		CompletePluginChannels = (CompletePluginChannelsFunction)get_complete_function(analysis_type);

		if ((CompletePluginChannels != NULL) && (analysis->analysis_rqst[analysis_type] >= AnalysisRqst::CalculateAndStore))
		{
			CompletePluginChannels(file, analysis, channel, analysis_type, geometry, &plugin_config[analysis_type - NUMBER_OF_ANALYSIS_CHANNELS]);
		}
	}

	//FINISH_RESULTS(ENGINE_SPEED, engine_speed);

	FINISH_RESULTS(NET_HEAT_RELEASE_RATE, net_heat_release_rate);
	FINISH_RESULTS(GROSS_HEAT_RELEASE_RATE, gross_heat_release_rate);
	FINISH_RESULTS(H_COEFF, h_coeff);
	FINISH_RESULTS(ANGULAR_TORQUE, angular_torque);
	FINISH_RESULTS(KNOCKING_PRESSURE, knocking_pressure);
	FINISH_RESULTS(MEAN_GAS_TEMP, mean_gas_temp);
	FINISH_RESULTS(GAMMA, gamma);
	FINISH_RESULTS(MFB, mfb);
	FINISH_RESULTS(MOTORED_PRESSURE, motored_pressure);
	FINISH_RESULTS(MOVING_PRESSURE_AVERAGE, moving_pressure_average);
	FINISH_RESULTS(POLYFIT, polyfit);
	FINISH_RESULTS(POLYTROPIC_INDICES, polytropic_indices);
	FINISH_RESULTS(PRESSURE_RISE_RATE, pressure_rise_rate);
	FINISH_RESULTS(WIEBE_MFB, wiebe_mfb);
	FINISH_RESULTS(FFT_KNOCK_PRESSURE, fft_knock_pressure);
	FINISH_RESULTS(FFT_KNOCK_PRESSURE_FREQUENCY, fft_knock_pressure_frequency);
	FINISH_RESULTS(FFT, fft);
	FINISH_RESULTS(FFT_FREQUENCY, fft_frequency);
	FINISH_RESULTS(EXTRAPOLATED_CYL_PRESSURE, extrapolated_cyl_pressure);
	FINISH_RESULTS(DIFFERENCE_PRESSURE, difference_pressure);
	FINISH_RESULTS(PRESSURE_RATIO, pressure_ratio);
	FINISH_RESULTS(PRESSURE_RATIO_FRACTION, pressure_ratio_fraction);
	FINISH_RESULTS(D2P, d2p);
	FINISH_RESULTS(CRANK_ANGLE_IMEP, crank_angle_imep);

	free(volume);
	volume = NULL;
	
	free(dVolume);
	dVolume = NULL;
	
	free(dCA);
	dCA = NULL;
	
	free(heat_transfer_area);
	heat_transfer_area = NULL;
	
	free(logVolume);
	logVolume = NULL;

	free(pressure_to_torque);
	pressure_to_torque = NULL;
}

void cycle_classification(AnalysisChannel* analysis, const unsigned int number_of_cycles)
{
}

void sync_analysis(FileData* file)
{
	unsigned int first_tdc;
	unsigned int second_tdc;
	unsigned int channel;
	unsigned int number_missynced_channels = 0;
	unsigned int number_cylpr_channels = 0;
	unsigned int cycle;
	unsigned int crank_angle;
	unsigned int new_cycle;
	unsigned int new_crank_angle;
	unsigned int mid_cycle;
	float raw_data;
	
	if (file == NULL)
	{
		return;
	}

	for (channel=0;channel<file->number_of_channels;channel++)
	{
		if ((file->channel_data[channel].loaded == true) && (file->channel_data[channel].type == CHAN_CYL_PR))
		{
			first_tdc = DegreesToCrankAngle(file,-360.0f,channel);
			second_tdc = DegreesToCrankAngle(file,0.0f,channel);

			if (ReturnCAData(file,0,first_tdc,channel) > ReturnCAData(file,0,second_tdc,channel))
			{
				logmessage(NOTICE,"Pressure data is mis-synchronised on channel %u\n",channel);
			
				number_missynced_channels += 1;
			}
			
			number_cylpr_channels += 1;
		}
	}
	
	if ((number_missynced_channels > 0) && (number_cylpr_channels == number_missynced_channels))
	{
		logmessage(NOTICE,"Pressure data is mis-synchronised on all in-cylinder pressure channels\n");
		
		file->number_of_cycles -= 1;
		
		for (channel=0;channel<file->number_of_channels;channel++)
		{
			if ((file->channel_data[channel].loaded == true) && (file->number_of_cycles > 1))
			{
				logmessage(NOTICE,"Re-syncing channel %u\n",channel);
					
				mid_cycle = (unsigned int)(file->channel_data[channel].samples_per_cycle / 2);

				for (cycle=0;cycle<file->number_of_cycles;cycle++)
				{
					new_cycle = 0;
					new_crank_angle = mid_cycle;
					for (crank_angle=0;crank_angle<file->channel_data[channel].samples_per_cycle;crank_angle++)
					{
						raw_data = ReturnCAData(file,cycle,crank_angle,channel);
						
						SetCAData(file,new_cycle,new_crank_angle,channel,raw_data);
						
						new_crank_angle += 1;
						if (new_crank_angle >= file->channel_data[channel].samples_per_cycle)
						{
							new_crank_angle = 0;
							new_cycle += 1;
						}
					}
				}				
			}
		}
	}
	
	file->sync_checked = true;
}

void engine_speed_analysis(const FileData* file,float* data,const unsigned int channel)
{
	unsigned int cycle;
    float engine_speed;
	float duration;

	if ((file == NULL) || (data == NULL))
	{
		return;
	}

	if (channel >= file->number_of_channels)
	{
		return;
	}

	if (file->channel_data[channel].abscissa.type != ABSCISSA_CRANKANGLE)
	{
		return;
	}

	if (file->channel_data[channel].duration == NULL)
	{
		return;
	}

	for (cycle=0;cycle<file->number_of_cycles;cycle++)
	{
		duration = file->channel_data[channel].duration[cycle];

		if (duration < FLT_EPSILON)
		{
			engine_speed = 0.0f;
		}
		else
		{
			engine_speed = (float)file->engine.number_of_strokes*30000.0f/duration;
		}

		data[cycle] = engine_speed;
	}
}

bool speed_from_channel(FileData* file, const unsigned int channel, const unsigned int channel_number)
{
	if (file->channel_data[channel_number].loaded == false)
	{
		return(false);
	}

	if (file->channel_data[channel_number].abscissa.type != ABSCISSA_CYCLE)
	{
		return(false);
	}

	bool speed_calculated = true;
	float duration = 0.0f;
	unsigned int cycle;

	for (cycle = 0; cycle < file->number_of_cycles; cycle++)
	{
		if (cycle < file->channel_data[channel_number].samples_per_cycle)
		{
			if (file->channel_data[channel_number].data[cycle] > 0.0f)
			{
				duration = (float)file->engine.number_of_strokes * 30000.0f / file->channel_data[channel_number].data[cycle];
			}
		}

		file->channel_data[channel].duration[cycle] = duration;

		if (duration < FLT_EPSILON)
		{
			speed_calculated = false;
		}
	}

	return(speed_calculated);
}

bool cycle_duration_analysis(FileData* file,const unsigned int channel,const unsigned int engine_speed_type,const char* engine_speed_channel,const float default_engine_speed)
{
	unsigned int cycle;
    short* ptr_RZT = NULL;
    long* ptr_AZT = NULL;
    float engine_speed;
	float duration;
	bool speed_calculated = false;
	unsigned int channel_number;
	unsigned int aztlen;

	if (file == NULL)
	{
		return(false);
	}

	if ((file->channel_data == NULL) || (channel >= file->number_of_channels) || (file->number_of_channels == 0))
	{
		return(false);
	}

	if (file->channel_data[channel].abscissa.type != ABSCISSA_CRANKANGLE)
	{
		return(false);
	}

	if (file->channel_data[channel].duration == NULL)
	{
		file->channel_data[channel].duration = (float*)malloc(file->number_of_cycles * sizeof(float));

		if (file->channel_data[channel].duration == NULL)
		{
			logmessage(FATAL, "Failed to allocate memory\n");
		}
	}

	switch (engine_speed_type)
	{
		/* Equal to the average of another channel */
		case N_AVGCHNL:
		{
			if ((engine_speed_channel != NULL) && (channel_name_to_number(file, engine_speed_channel, ABSCISSA_UNKNOWN, &channel_number) == true))
			{
				speed_calculated = true;

				for (cycle=0;cycle<file->number_of_cycles;cycle++)
				{
					engine_speed = channel_average(file, channel_number,cycle,channel,(bool)false);

					if (engine_speed > FLT_EPSILON)
					{
						duration = (float)file->engine.number_of_strokes * 30000.0f / engine_speed;
					}
					
					file->channel_data[channel].duration[cycle] = duration;

					if (duration < FLT_EPSILON)
					{
						speed_calculated = false;
					}
				}
			}
			
			break;
		}
		/* Using AVL I-File RZT (Cycle duration) */
		case N_AVLRZT:
		{
			if ((file->ptr_DGB != NULL) && (file->ptr_DGB[file->ca_dgb].rztadr != NULL))
			{
				speed_calculated = true;
				ptr_RZT = (short*)file->ptr_DGB[file->ca_dgb].rztadr;
				duration = 0.0f;

				for (cycle=0;cycle<file->number_of_cycles;cycle++)
				{
					if (ptr_RZT[cycle] > 0.0)
					{
						duration = (float)((double)ptr_RZT[cycle] * file->ptr_DGB[file->ca_dgb].dltzei);
					}
					
					file->channel_data[channel].duration[cycle] = duration;

					if (duration < FLT_EPSILON)
					{
						speed_calculated = false;
					}
				}
			}
			
			if ((speed_calculated == false) && (channel_name_to_number(file, "CYCDUR", ABSCISSA_UNKNOWN, &channel_number) == true))
			{
				if (file->channel_data[channel_number].loaded == true)
				{
					speed_calculated = true;
					duration = 0.0f;

					for (cycle = 0; cycle<file->number_of_cycles; cycle++)
					{
						if (cycle < file->channel_data[channel_number].samples_per_cycle)
						{
							if (file->channel_data[channel_number].data[cycle] > 0.0f)
							{
								duration = file->channel_data[channel_number].data[cycle];
							}
						}

						file->channel_data[channel].duration[cycle] = duration;

						if (duration < FLT_EPSILON)
						{
							speed_calculated = false;
						}
					}
				}
				else
				{
					logmessage(NOTICE, "Found CYCDUR channel but it was not loaded so cannot use as a speed channel\n");
				}
			}

			if ((speed_calculated == false) && (engine_speed_channel != NULL) && (channel_name_to_number(file, engine_speed_channel, ABSCISSA_UNKNOWN, &channel_number) == true))
			{
				speed_calculated = speed_from_channel(file,channel,channel_number);
			}

			if ((speed_calculated == false) && (channel_name_to_number(file, "SPEED", ABSCISSA_UNKNOWN, &channel_number) == true))
			{
				/* AVL IFile */

				speed_calculated = speed_from_channel(file, channel, channel_number);
			}

			if ((speed_calculated == false) && (channel_name_to_number(file, "DREZAHL", ABSCISSA_UNKNOWN, &channel_number) == true))
			{
				/* AVL IFile */

				speed_calculated = speed_from_channel(file, channel, channel_number);
			}

			if ((speed_calculated == false) && (channel_name_to_number(file, "RPM", ABSCISSA_UNKNOWN, &channel_number) == true))
			{
				/* Alma TDMS */

				speed_calculated = speed_from_channel(file, channel, channel_number);
			}

			if ((speed_calculated == false) && (channel_name_to_number(file, "SPEED_AVG_0", ABSCISSA_UNKNOWN, &channel_number) == true))
			{
				/* Kistler .open */

				speed_calculated = speed_from_channel(file, channel, channel_number);
			}

			if ((speed_calculated == false) && (channel_name_to_number(file, "CYCTIME", ABSCISSA_UNKNOWN, &channel_number) == true))
			{
				if (file->channel_data[channel_number].loaded == true)
				{
					speed_calculated = true;
					duration = 0.0f;
					
					for (cycle = 0; cycle<file->number_of_cycles - 1; cycle++)
					{
						if (cycle < file->channel_data[channel_number].samples_per_cycle)
						{
							if (file->channel_data[channel_number].data[cycle] - file->channel_data[channel_number].data[cycle + 1] > 0.0f)
							{
								duration = file->channel_data[channel_number].data[cycle] - file->channel_data[channel_number].data[cycle + 1];
							}
						}

						file->channel_data[channel].duration[cycle] = duration;

						if (duration < FLT_EPSILON)
						{
							speed_calculated = false;
						}
					}

					file->channel_data[channel].duration[cycle] = duration;
				}
				else
				{
					logmessage(NOTICE, "Found CYCTIME channel but it was not loaded so cannot use as a speed channel\n");
				}
			}

			if ((speed_calculated == false) && (file->ptr_DGB != NULL) && (file->ptr_DGB[file->ca_dgb].aztadr != NULL))
			{
				ptr_AZT = (long*)file->ptr_DGB[file->ca_dgb].aztadr;

				aztlen = file->ptr_DGB[file->ca_dgb].aztlen;
				
				if (file->ptr_DGB[file->ca_dgb].azttyp == 0)
				{
					duration = 0.0f;
					speed_calculated = true;
					for (cycle=0;cycle<file->number_of_cycles-1;cycle++)
					{
						if (((cycle + 1) < aztlen) && (ptr_AZT[cycle + 1] - ptr_AZT[cycle] > 0.0f))
						{
							duration = (float)((double)(ptr_AZT[cycle + 1] - ptr_AZT[cycle])*file->ptr_DGB[file->ca_dgb].dltzei);
						}

						file->channel_data[channel].duration[cycle] = duration;
						
						if (duration < FLT_EPSILON)
						{
							speed_calculated = false;
						}
					}
					file->channel_data[channel].duration[cycle] = duration;
				}
				else if (file->ptr_DGB[file->ca_dgb].azttyp == 1)
				{
					duration = 0.0f;
					speed_calculated = true;
					for (cycle=0;cycle<file->number_of_cycles-1;cycle++)
					{
						if ((cycle / 16 + 1) * 4 < aztlen)
						{
							if (ptr_AZT[cycle / 16 + 1] - ptr_AZT[cycle / 16] > 0.0f)
							{
								duration = (float)((double)(ptr_AZT[cycle / 16 + 1] - ptr_AZT[cycle / 16])*file->ptr_DGB[file->ca_dgb].dltzei);
							}
						}

						file->channel_data[channel].duration[cycle] = duration;

						if (duration < FLT_EPSILON)
						{
							speed_calculated = false;
						}
					}
					file->channel_data[channel].duration[cycle] = duration;
				}
				else
				{
					logmessage(WARNING,"Found AZT but did not understand the azttyp %d\n",file->ptr_DGB[file->ca_dgb].azttyp);
				}
			}

			if (speed_calculated == false)
			{
				logmessage(WARNING,"Could not find speed data for all cycles in channel %s\n",file->channel_data[channel].name);
			}
				 
			break;
		}
		/* Assume default engine speed */
		case N_NOTHING:
		case N_SPECIFY:
		default:
		{
			break;
		}
	}
	
	if (speed_calculated == false)
	{
		engine_speed = default_engine_speed;
		
		if (engine_speed < 10.0f)
		{
			engine_speed = 10.0f;
		}
		
		duration = (float)file->engine.number_of_strokes*30000.0f/engine_speed;
	
		for (cycle=0;cycle<file->number_of_cycles;cycle++)
		{
			file->channel_data[channel].duration[cycle] = duration;
		}
	}

	return(true);
}

void* cycle_analysis(void* arguments)
{
	cycle_analysis_arguments* args = (cycle_analysis_arguments*)arguments;
	FileData* file = NULL;
	AnalysisChannel* analysis = NULL;
	Analysis* file_analysis = NULL;
	ProgressStatus* status = NULL;
	bool error_check;
	unsigned int channel_to_analyse;
	unsigned int cycle;
	unsigned int number_of_misfires;
	unsigned int number_of_slow_burns;
	unsigned int number_of_knocking_cycles;
	unsigned int number_of_mega_knocking_cycles;

	if (args == NULL)
	{
		return(NULL);
	}

	args->thread_status = THREAD_STATUS_INITIALISING;
	file = args->file;
	if (file == NULL)
	{
		args->thread_status = THREAD_STATUS_DONE;
		return(NULL);
	}

	if (args->channel >= file->number_of_channels)
	{
		args->thread_status = THREAD_STATUS_DONE;
		return(NULL);
	}

	file_analysis = args->analysis;
	if (file_analysis == NULL)
	{
		args->thread_status = THREAD_STATUS_DONE;
		return(NULL);
	}

	if (file_analysis->channel == NULL)
	{
		args->thread_status = THREAD_STATUS_DONE;
		return(NULL);
	}

	analysis = &file_analysis->channel[args->channel];

	status = &args->status;

	channel_to_analyse = analysis->channel;
	if (channel_to_analyse >= file->number_of_channels)
	{
		args->thread_status = THREAD_STATUS_DONE;
		return(NULL);
	}

	error_check = args->error_check;
	args->thread_status = THREAD_STATUS_RUNNING;

	if ((file->channel_data[channel_to_analyse].file_flag == true) && (file->channel_data[channel_to_analyse].loaded == true))
    {
		logmessage(NOTICE,"Analysing channel %s\n",file->channel_data[channel_to_analyse].name);

		if (file->channel_data[channel_to_analyse].abscissa.type == ABSCISSA_CRANKANGLE)
		{
			if ((file->channel_data[channel_to_analyse].filter_config.type != FILTER_NONE) && (file->channel_data[channel_to_analyse].filter_config.filtered == false))
			{
				logmessage(NOTICE, "Filtering channel...");

				filter_channel(file, channel_to_analyse, analysis->config.engine_speed);

				logmessage(NOTICE, "done\n");
			}

			logmessage(NOTICE,"Performing offset correction...");
					
		    offset_correction_channel(file,analysis,channel_to_analyse);
		        	
			logmessage(NOTICE,"done\n");

			switch (file->channel_data[channel_to_analyse].type)
			{
			case CHAN_CYL_PR:     		/* Cylinder Pressure */
			{				
				logmessage(NOTICE,"Performing pressure analysis...");
	       		
				pressure_cycle_analysis(file,analysis,file_analysis,status);
	       		
				logmessage(NOTICE,"done\n");

	            break;         	
	        }
			case CHAN_INJ_PR:			/* Injection Pressure */
			case CHAN_NEEDLE_LIFT:		/* Needle Lift */
			case CHAN_LOW_T:			/* Low Tension Coil Voltage */
			case CHAN_HIGH_T:			/* High Tension Pick Up */
			case CHAN_VALVE_LIFT:		/* Valve Lift */
			case CHAN_SPK_PRI_CURR:		/* Spark Primary Current */
			case CHAN_SPK_SEC_CURR:		/* Spark Secondary Current */
			case CHAN_IGN_ANG:			/* Ignition Angle Meter */
			case CHAN_CPS:				/* Crankshaft position sensor */
			case CHAN_CID:				/* Camshaft position sensor */
			case CHAN_INJECTOR:			/* Injector */
			case CHAN_BLOCK_ACC:		/* Engine Block Accelerometer */
			case CHAN_INLET_PR:			/* Inlet Manifold Pressure */
			case CHAN_EXH_PR:			/* Exhaust Manifold Pressure */
			case CHAN_TOOTH_PERIODS:	/* Tooth Periods */
			{
				fft_analysis(file, analysis, channel_to_analyse, status);

				start_of_combustion_analysis(file, analysis, file_analysis, channel_to_analyse, status);

				break;
			}
			case CHAN_UNKNOWN:			/* Unknown */
			case CHAN_GENERIC:			/* Generic crankangle base channel */
			case CHAN_ENG_SPEED:		/* Engine Speed */
			{
				/* Calculate min/max/mean */

				start_of_combustion_analysis(file, analysis, file_analysis, channel_to_analyse, status);

				break;
			}
			default:
			{
				logmessage(NOTICE, "Nothing to do with channel %s\n", file->channel_data[channel_to_analyse].name);
				
				break;
			}
			}
					
			logmessage(NOTICE,"done\n");

			if ((file->channel_data[channel_to_analyse].type == CHAN_CYL_PR) ||
				(file->channel_data[channel_to_analyse].type == CHAN_BLOCK_ACC))
			{
				/* Cycle classification */

				cycle_classification(analysis, analysis->number_of_cycles);

				number_of_misfires = 0;
				number_of_slow_burns = 0;
				number_of_knocking_cycles = 0;
				number_of_mega_knocking_cycles = 0;

				for (cycle = 0; cycle < file->number_of_cycles; cycle++)
				{
					if ((analysis->cycle_classification[cycle] & CYCLE_MISFIRE) > 0)
					{
						number_of_misfires += 1;
					}
					if ((analysis->cycle_classification[cycle] & CYCLE_SLOW_BURN) > 0)
					{
						number_of_slow_burns += 1;
					}
					if ((analysis->cycle_classification[cycle] & CYCLE_KNOCK) > 0)
					{
						number_of_knocking_cycles += 1;
					}
					if ((analysis->cycle_classification[cycle] & CYCLE_MEGA_KNOCK) > 0)
					{
						number_of_mega_knocking_cycles += 1;
					}
				}

				if (number_of_misfires > 0)
				{
					logmessage(NOTICE, "%s: Misfire detected in %g%% (%u) of cycles\n", file->channel_data[channel_to_analyse].name, (float)number_of_misfires / (float)file->number_of_cycles*100.0f, number_of_misfires);
				}
				if (number_of_slow_burns > 0)
				{
					logmessage(NOTICE, "%s: Slow burns detected in %g%% (%u) of cycles\n", file->channel_data[channel_to_analyse].name, (float)number_of_slow_burns / (float)file->number_of_cycles*100.0f, number_of_slow_burns);
				}
				if (number_of_knocking_cycles > 0)
				{
					logmessage(NOTICE, "%s: Knock detected in %g%% (%u) of cycles\n", file->channel_data[channel_to_analyse].name, (float)number_of_knocking_cycles / (float)file->number_of_cycles*100.0f, number_of_knocking_cycles);
				}
				if (number_of_mega_knocking_cycles > 0)
				{
					logmessage(NOTICE, "%s: Mega Knock detected in %g%% (%u) of cycles\n", file->channel_data[channel_to_analyse].name, (float)number_of_mega_knocking_cycles / (float)file->number_of_cycles*100.0f, number_of_mega_knocking_cycles);
				}
			}
		}
		else if (file->channel_data[channel_to_analyse].abscissa.type == ABSCISSA_TIME)
		{
			switch (file->channel_data[channel_to_analyse].type)
			{
			case CHAN_TIME:
			{
				break;
			}
			case CHAN_CID:			/* Camshaft identification sensor */
			{
				break;
			}
			case CHAN_CPS:			/* Crankshaft position sensor */
			{
				break;
			}
			case CHAN_LOW_T:		/* Low tension coil voltage */
			{
				break;
			}
			case CHAN_INJECTOR:		/* Injector */
			{
				break;
			}
			case CHAN_UNKNOWN:
			default:
			{
				logmessage(NOTICE,"Nothing to do with channel %s\n", file->channel_data[channel_to_analyse].name);
                
				break;
			}
			}			
		}
		else if (file->channel_data[channel_to_analyse].abscissa.type == ABSCISSA_CYCLE)
		{
			logmessage(NOTICE,"Not performing analysis of cycle results channel %s\n", file->channel_data[channel_to_analyse].name);
		}
		else if (file->channel_data[channel_to_analyse].abscissa.type == ABSCISSA_FREQUENCY)
		{
			logmessage(NOTICE, "Not performing analysis of frequency channel %s\n", file->channel_data[channel_to_analyse].name);
		}
		else
		{
			logmessage(NOTICE,"Cannot analyse channel %s as the abscissa type (%u) is not supported\n", file->channel_data[channel_to_analyse].name,file->channel_data[channel_to_analyse].abscissa.type);
		}	
		
		if (file->number_of_cycles > 0)
		{
			logmessage(NOTICE,"Performing statistical analysis...");

			raw_statistical_analysis(file,analysis,channel_to_analyse);

			statistical_analysis(analysis);

			logmessage(NOTICE,"done\n");
		}

		if (error_check == true)
		{
			logmessage(NOTICE, "Performing error checking...");

			error_check_analysis(file, analysis);

			logmessage(NOTICE, "done\n");
		}

		logmessage(NOTICE,"channel %u done\n",channel_to_analyse);
	}
	else
	{
			logmessage(WARNING,"Cannot analyse channel %s as it is not loaded\n", file->channel_data[channel_to_analyse].name);
	}

	args->thread_status = THREAD_STATUS_DONE;

#if defined(_USE_PTHREADS)
	if (file->threads > 1)
	{
		pthread_exit(NULL);
	}
#endif

	return(NULL);
}

void statistical_analysis(AnalysisChannel* analysis)
{
	unsigned int analysis_channel;
	unsigned int samples_per_cycle;
	unsigned int frequency_unit;

	if (analysis == NULL)
	{
		return;
	}

	if (analysis->results == NULL)
	{
		return;
	}

	for (analysis_channel = 0; analysis_channel<get_number_of_analysis_channels(); analysis_channel++)
	{
		samples_per_cycle = analysis->results[analysis_channel].samples_per_cycle;

		if (analysis->analysis_rqst[analysis_channel] > AnalysisRqst::NotCalculated)
		{
			if (analysis->results[analysis_channel].abscissa.type == ABSCISSA_CYCLE)
			{
				/* Using samples_per_cycle rather than number_of_cycles as there are some IFiles where the number of data points < number_of_cycles */

				calculate_cycle_stats(&analysis->statistics[analysis_channel], analysis->results[analysis_channel].data, samples_per_cycle, analysis->config.stats_type);
			}
			else if (analysis->analysis_rqst[analysis_channel] >= AnalysisRqst::CalculateAndStore)
			{
				if (analysis->results[analysis_channel].abscissa.type == ABSCISSA_CRANKANGLE)
				{
					calculate_ca_stats(&analysis->ca_statistics[analysis_channel], analysis->results[analysis_channel].data, samples_per_cycle, analysis->number_of_cycles);
				}
				else if (analysis->results[analysis_channel].abscissa.type == ABSCISSA_FREQUENCY)
				{
					unsigned int frequency_channel;

					switch (analysis_channel)
					{
					default:					frequency_channel = UINT_MAX;						break;
					case FFT:					frequency_channel = FFT_FREQUENCY;					frequency_unit = UNITS_KHZ;	break;
					case FFT_REFERENCE:			frequency_channel = FFT_REFERENCE_FREQUENCY;		frequency_unit = UNITS_KHZ;	break;
					case FFT_KNOCK_PRESSURE:	frequency_channel = FFT_KNOCK_PRESSURE_FREQUENCY;	frequency_unit = UNITS_KHZ;	break;
					case FFT_TORQUE_SUM:		frequency_channel = FFT_TORQUE_SUM_FREQUENCY;		frequency_unit = UNITS_KHZ;	break;
					case FFT_TOOTH_PERIOD:		frequency_channel = FFT_TOOTH_PERIOD_FREQUENCY;		frequency_unit = UNITS_HZ;	break;
					}

					if (frequency_channel < get_number_of_analysis_channels())
					{
						if ((samples_per_cycle > 0) &&
							(samples_per_cycle == analysis->results[analysis_channel].samples_per_cycle) &&
							(samples_per_cycle == analysis->results[frequency_channel].samples_per_cycle) &&
							(samples_per_cycle == analysis->results[analysis_channel].abscissa.number_of_samples[0]) &&
							(samples_per_cycle == analysis->results[frequency_channel].abscissa.number_of_samples[0]))
						{
							calculate_freq_stats(&analysis->ca_statistics[analysis_channel],
								analysis->results[analysis_channel].data,
								analysis->results[frequency_channel].data,
								frequency_unit,
								samples_per_cycle,
								analysis->number_of_cycles,
								analysis->config.fft_stats_maximum,
								analysis->config.fft_stats_resolution);
						}
						else
						{
							logmessage(MSG_ERROR, "samples_per_cycle do not match\n");
						}
					}
				}
				else
				{
					/* Do Nothing */
				}
			}
		}
	}
}

void raw_statistical_analysis(const FileData* file, AnalysisChannel* analysis, const unsigned int channel)
{
	if ((file == NULL) || (analysis == NULL))
	{
		return;
	}

	if (channel >= file->number_of_channels)
	{
		return;
	}

	if (file->channel_data[channel].abscissa.type == ABSCISSA_CYCLE)
	{
		/* Using samples_per_cycle rather than number_of_cycles as there are some IFiles where the number of data points < number_of_cycles */
#if 0
		Statistics stats;

		Copy_Statistics(&analysis->raw_statistics, &stats);
#endif
		calculate_cycle_stats(&analysis->raw_statistics, file->channel_data[channel].data, file->channel_data[channel].samples_per_cycle, analysis->config.stats_type);
#if 0
		if (Compare_Statistics(&stats, &analysis->raw_statistics) == false)
		{
			logmessage(WARNING, "%s: Stats have changed\n", file->channel_data[channel].name);
		}
		else
		{
			logmessage(WARNING, "%s: Stats are unchanged\n", file->channel_data[channel].name);
		}
#endif
	}
	else if ((file->channel_data[channel].abscissa.type == ABSCISSA_CRANKANGLE) || (file->channel_data[channel].abscissa.type == ABSCISSA_FREQUENCY))
	{
		calculate_ca_stats(&analysis->raw_ca_statistics, file->channel_data[channel].data, file->channel_data[channel].samples_per_cycle, file->number_of_cycles);
	}
	else
	{
		/* Do Nothing */
	}
}

void error_check_analysis(const FileData* file, const AnalysisChannel* analysis)
{
	unsigned int analysis_channel;
	unsigned int channel;

	if ((file == NULL) || (analysis == NULL))
	{
		return;
	}

	if ((file->channel_data == NULL) || (analysis->channel >= file->number_of_channels))
	{
		return;
	}

	channel = analysis->channel;

	for (analysis_channel = 0; analysis_channel < get_number_of_analysis_channels(); analysis_channel++)
	{
		if ((analysis->analysis_rqst[analysis_channel] > AnalysisRqst::NotCalculated) && (analysis->results[analysis_channel].abscissa.type == ABSCISSA_CYCLE))
		{
			if (analysis->statistics[analysis_channel].max > get_analysis_error_max(analysis_channel))
			{
				logmessage(WARNING, "ERROR: %s %s calculated max (%f) exceeds maximum (%f)\n", file->channel_data[channel].name, get_analysis_name(analysis_channel), analysis->statistics[analysis_channel].max, get_analysis_error_max(analysis_channel));
			}
			else if (analysis->statistics[analysis_channel].max > get_analysis_warning_max(analysis_channel))
			{
				logmessage(WARNING, "WARNING: %s %s calculated max (%f) exceeds maximum (%f)\n", file->channel_data[channel].name, get_analysis_name(analysis_channel), analysis->statistics[analysis_channel].max, get_analysis_warning_max(analysis_channel));
			}
			else
			{
				/* Do Nothing */
			}

			if (analysis->statistics[analysis_channel].min < get_analysis_error_min(analysis_channel))
			{
				logmessage(WARNING, "ERROR: %s %s calculated min (%f) less than minimum (%f)\n", file->channel_data[channel].name, get_analysis_name(analysis_channel), analysis->statistics[analysis_channel].min, get_analysis_error_min(analysis_channel));
			}
			else if (analysis->statistics[analysis_channel].min < get_analysis_warning_min(analysis_channel))
			{
				logmessage(WARNING, "WARNING: %s %s calculated min (%f) less than minimum (%f)\n", file->channel_data[channel].name, get_analysis_name(analysis_channel), analysis->statistics[analysis_channel].min, get_analysis_warning_min(analysis_channel));
			}
			else
			{
				/* Do Nothing */
			}

			if (analysis->statistics[analysis_channel].stddev > get_analysis_error_stddev(analysis_channel))
			{
				logmessage(WARNING, "ERROR: %s %s calculated std dev (%f) standard deviation excessive (%f)\n", file->channel_data[channel].name, get_analysis_error_max(analysis_channel), analysis->statistics[analysis_channel].stddev, get_analysis_error_stddev(analysis_channel));
			}
			else if (analysis->statistics[analysis_channel].stddev > get_analysis_warning_stddev(analysis_channel))
			{
				logmessage(WARNING, "WARNING: %s %s calculated std dev (%f) standard deviation excessive (%f)\n", file->channel_data[channel].name, get_analysis_error_max(analysis_channel), analysis->statistics[analysis_channel].stddev, get_analysis_warning_stddev(analysis_channel));
			}
			else
			{
				/* Do Nothing */
			}
		}
	}
}

void Return_Geometry_Data(const FileData* file,const unsigned int channel,float* dCA_data,float* volume_data,float* dVolume_data,float* HTA_data,float* logVolume_data,float* pressure_to_torque_data)
{
	unsigned int theta;
	float fixed_hta;
	unsigned int crank_angle;
	bool delete_volume;
	float crank_radius;
	float angle;
	float piston_area;
	float dCA = 0.0f;
	float dVolume = 0.0f;

	if (file == NULL)
	{
		return;
	}

	if ((channel >= file->number_of_channels) || (file->channel_data == NULL))
	{
		return;
	}

	if (file->channel_data[channel].samples_per_cycle == 0)
	{
		return;
	}

	if (file->channel_data[channel].abscissa.type != ABSCISSA_CRANKANGLE)
	{
		return;
	}
	
	if (dCA_data != NULL)
	{
		for (crank_angle=0;crank_angle<file->channel_data[channel].samples_per_cycle-1;crank_angle++)
		{
			dCA = CrankAngleToDegrees(file, crank_angle + 1, channel) - CrankAngleToDegrees(file, crank_angle, channel);

			dCA_data[crank_angle] = dCA;
		}

		dCA_data[file->channel_data[channel].samples_per_cycle - 1] = dCA;
	}

	if (volume_data != NULL)
	{
		for (crank_angle=0;crank_angle<file->channel_data[channel].samples_per_cycle;crank_angle++)
		{
			theta = ReturnTheta(file,crank_angle,channel);

			volume_data[crank_angle] = Volume(file,channel,theta);
		}
    }

	if (dVolume_data != NULL)
    {
        if (volume_data == NULL)
        {
            volume_data = (float*)malloc(file->channel_data[channel].samples_per_cycle * sizeof(float));
    
       	    if (volume_data == NULL)
       	    {
       		    logmessage(FATAL,"Memory could not be allocated\n");
            }

	        Return_Geometry_Data(file,channel,NULL,volume_data,NULL,NULL,NULL,NULL);
	        delete_volume = true;               
        }
        else
        {
			delete_volume = false;
		}
        
		for (crank_angle=0;crank_angle<file->channel_data[channel].samples_per_cycle-1;crank_angle++)
		{
			dVolume = volume_data[crank_angle + 1] - volume_data[crank_angle];

            dVolume_data[crank_angle] = dVolume;
		}

		dVolume_data[file->channel_data[channel].samples_per_cycle - 1] = dVolume;
		
	    if (delete_volume != false)
	    {
	        free(volume_data);
	        volume_data = NULL;
		}		
    }

	if ((HTA_data != NULL) && (file->engine.compression_ratio > 0.0f) && (file->engine.bore > 0.0f))
	{
        if (volume_data == NULL)
        {
            volume_data = (float*)malloc(file->channel_data[channel].samples_per_cycle * sizeof(float));
       	    if (volume_data == NULL)
       	    {
       		    logmessage(FATAL,"Memory could not be allocated\n");
				exit(EXIT_FAILURE);
            }

	        Return_Geometry_Data(file,channel,NULL,volume_data,NULL,NULL,NULL,NULL);
	        delete_volume = true;               
        }
        else
        {
			delete_volume = false;
		}

		fixed_hta = FLT_PI*file->engine.bore*file->engine.stroke / file->engine.compression_ratio + FLT_PI*file->engine.bore*file->engine.bore / 2.0f;

		for (crank_angle=0;crank_angle<file->channel_data[channel].samples_per_cycle;crank_angle++)
		{
			HTA_data[crank_angle] = fixed_hta + 4.0f/file->engine.bore*volume_data[crank_angle];
		}
		
	    if (delete_volume != false)
	    {
	        free(volume_data);
	        volume_data = NULL;
		}
	}
	
    if (logVolume_data != NULL)
	{
        if (volume_data == NULL)
        {
            volume_data = (float*)malloc(file->channel_data[channel].samples_per_cycle * sizeof(float));
       	    if (volume_data == NULL)
       	    {
       		    logmessage(FATAL,"Memory could not be allocated\n");
				exit(EXIT_FAILURE);
            }

	        Return_Geometry_Data(file,channel,NULL,volume_data,NULL,NULL,NULL,NULL);
	        delete_volume = true;               
        }
        else
        {
			delete_volume = false;
		}
        
		for (crank_angle=0;crank_angle<file->channel_data[channel].samples_per_cycle;crank_angle++)
		{
			logVolume_data[crank_angle] = logf(volume_data[crank_angle]);
		}
		
	    if (delete_volume != false)
	    {
	        free(volume_data);
		}
    }

	if (pressure_to_torque_data != NULL)
	{
		piston_area = FLT_PI * file->engine.bore * file->engine.bore / 4.0f * SQ_MM_TO_M * file->engine.piston_factor;

		for (crank_angle = 0; crank_angle < file->channel_data[channel].samples_per_cycle; crank_angle++)
		{
			angle = CrankAngleToDegrees(file, crank_angle, channel) / 360.0f * 2.0f * FLT_PI;

			crank_radius = sinf(angle) * file->engine.stroke * LEN_MM_TO_M / 2.0f;

			pressure_to_torque_data[crank_angle] = piston_area * crank_radius  * BAR_TO_PA;
		}
	}
}

float RawToEngineering(const float raw, const Calibration* calibration)
{
	float result;
	unsigned int factor;
	unsigned int number_of_terms;

	if (calibration == NULL)
	{
		return(0.0f);
	}

	number_of_terms = calibration->terms;

	if (number_of_terms > NUMBER_OF_CALIBRATION_TERMS)
	{
		number_of_terms = NUMBER_OF_CALIBRATION_TERMS;
	}

	result = 0.0f;

	for (factor = 0; factor < number_of_terms; factor++)
	{
		result += calibration->factor[factor] * powf(raw, (float)factor);
	}

	return(result);
}

bool wrap_angle(float* angle, const float minimum, const float maximum)
{
	bool result = false;

	if (angle == NULL)
	{
		return(false);
	}

	while (*angle > maximum)
	{
		*angle -= maximum *2.0f;
		result = true;
	}

	while (*angle < minimum)
	{
		*angle += minimum *2.0f;
		result = true;
	}

	return(result);
}

bool wrap_valve_angles(Engine* engine)
{
	float minimum;
	float maximum;
	bool result = false;

	if (engine == NULL)
	{
		return(false);
	}

	minimum = -90.0f * (float)engine->number_of_strokes;
	maximum = 90.0f * (float)engine->number_of_strokes;

	result |= wrap_angle(&engine->ivo_angle, minimum, maximum);
	result |= wrap_angle(&engine->ivc_angle, minimum, maximum);
	result |= wrap_angle(&engine->evo_angle, minimum, maximum);
	result |= wrap_angle(&engine->evc_angle, minimum, maximum);
	
	return(result);
}