/* Combustion Analysis Tool (CAT)
   www.catool.org
  
   Filename: atimat.c
  
   Purpose:  Provide common programming interface for processing acquired engine data
  
   Author:   Ben Brown
   Version:  1.2
   Date:     20.06.13

   Revision: GPL Release

   Copyright (C) Xarin Limited, 2000-2024

      This program is free software: you can redistribute it and/or modify
      it under the terms of version 2 of the GNU General Public License as
      published by the Free Software Foundation.

      This program is distributed in the hope that it will be useful,
      but WITHOUT ANY WARRANTY; without even the implied warranty of
      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
      GNU General Public License for more details.

      You should have received a copy of the GNU General Public License
      along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "cat.h"

void matlab_file_write_structure_atimat_header_device(FILE* file_handle, char* array_name)
{
	int structure_size_file_position;
	unsigned int flags;
	unsigned int array_name_length;
	unsigned int field_name_length;
	int end_of_structure_file_position;
	unsigned int number_of_fields;
	unsigned int total_field_name_length;
	int structure_size;
	int array_flags_subelement;
	char array_type;

	fwrite_int(miMATRIX,file_handle);

	structure_size_file_position = ftell(file_handle);
	
	fwrite_int(0,file_handle);			/* We'll go back later and fill in the number of bytes	*/

	/* Array Flags */

	array_type = mxSTRUCT_CLASS;
	flags = mfNONE;

	fwrite_int(miUINT32,file_handle);
	fwrite_int(8,file_handle);
	
	array_flags_subelement = (int)array_type | flags;
	
	fwrite_int(array_flags_subelement,file_handle);
    fwrite_int(0,file_handle);
	
	/* Dimensions Array */

	fwrite_int(miINT32,file_handle);
	fwrite_int(8,file_handle);
	fwrite_int(1,file_handle);
	fwrite_int(1,file_handle);

	/* Array Name */

	validate_variable_name(array_name);

	array_name_length = (unsigned int)strlen(array_name);

	if (array_name_length > SIZEOF_CHANNEL_NAME)
	{
		array_name_length = SIZEOF_CHANNEL_NAME;
	}

	matlab_file_write(file_handle,miINT8,array_name,array_name_length);

	/* Field Name Length */

	field_name_length = 12;
	
	matlab_file_write(file_handle,miINT32,&field_name_length,4);

	/* Field Names */

	number_of_fields = 6;

	total_field_name_length = number_of_fields * field_name_length;

	fwrite_int(miINT8,file_handle);
	fwrite_int(total_field_name_length,file_handle);

	fwrite_padded_char("name",field_name_length,file_handle);
	fwrite_padded_char("description",field_name_length,file_handle);
	fwrite_padded_char("comment",field_name_length,file_handle);
	fwrite_padded_char("strategy",field_name_length,file_handle);
	fwrite_padded_char("calibration",field_name_length,file_handle);
	fwrite_padded_char("serialnum",field_name_length,file_handle);

	/* name */

	matlab_file_write_string_numeric_array(file_handle,"catoolRT",50);

	/* description */

	matlab_file_write_string_numeric_array(file_handle,"AVL IFile converted to ATIMAT",50);

	/* comment */

	matlab_file_write_string_numeric_array(file_handle,"catoolRT",50);

	/* strategy */

	matlab_file_write_string_numeric_array(file_handle,"Unknown",50);

	/* calibration */

	matlab_file_write_string_numeric_array(file_handle,"Unknown",50);

	/* serialnum */

	matlab_file_write_string_numeric_array(file_handle,"None",50);
	
	end_of_structure_file_position = ftell(file_handle);
	
	if ((structure_size_file_position >= 0) && (end_of_structure_file_position > structure_size_file_position))
	{
		structure_size = end_of_structure_file_position - structure_size_file_position - 4;

		if (structure_size >= 0)
		{
			if (fseek(file_handle, structure_size_file_position, SEEK_SET) == 0)
			{
				fwrite_int(structure_size, file_handle);
				
				fseek(file_handle, end_of_structure_file_position, SEEK_SET);
			}
		}
	}
	else
	{
		logmessage(MSG_ERROR,"Unable to determine end of MATLAB file\n");
	}
}

#ifdef _CATOOL_UNICODE_
void matlab_file_write_structure_atimat_header(FILE* file_handle, char* array_name, wchar_t* filename, const unsigned int number_of_channels)
#else
void matlab_file_write_structure_atimat_header(FILE* file_handle, char* array_name, char* filename, const unsigned int number_of_channels)
#endif
{
	int structure_size_file_position;
	unsigned int flags;
	unsigned int array_name_length;
	unsigned int field_name_length;
	int end_of_structure_file_position;
	unsigned int number_of_fields;
	unsigned int total_field_name_length;
	int structure_size;
	int array_flags_subelement;
	char array_type;
	char device[] = "device";
	
	fwrite_int(miMATRIX,file_handle);

	structure_size_file_position = ftell(file_handle);
	
	fwrite_int(0,file_handle);			/* We'll go back later and fill in the number of bytes	*/

	/* Array Flags */

	array_type = mxSTRUCT_CLASS;
	flags = mfNONE;

	fwrite_int(miUINT32,file_handle);
	fwrite_int(8,file_handle);
	
	array_flags_subelement = (int)array_type | flags;
	
	fwrite_int(array_flags_subelement,file_handle);
    fwrite_int(0,file_handle);
	
	/* Dimensions Array */

	fwrite_int(miINT32,file_handle);
	fwrite_int(8,file_handle);
	fwrite_int(1,file_handle);
	fwrite_int(1,file_handle);

	/* Array Name */

	validate_variable_name(array_name);

	array_name_length = (unsigned int)strlen(array_name);

	if (array_name_length > SIZEOF_CHANNEL_NAME)
	{
		array_name_length = SIZEOF_CHANNEL_NAME;
	}

	matlab_file_write(file_handle,miINT8,array_name,array_name_length);

	/* Field Name Length */

	field_name_length = 12;
	
	matlab_file_write(file_handle,miINT32,&field_name_length,4);

	/* Field Names */

	number_of_fields = 16;

	total_field_name_length = number_of_fields * field_name_length;

	fwrite_int(miINT8,file_handle);
	fwrite_int(total_field_name_length,file_handle);

	fwrite_padded_char("version",field_name_length,file_handle);
	fwrite_padded_char("source",field_name_length,file_handle);
	fwrite_padded_char("project",field_name_length,file_handle);
	fwrite_padded_char("screen",field_name_length,file_handle);
	fwrite_padded_char("recorder",field_name_length,file_handle);
	fwrite_padded_char("startdate",field_name_length,file_handle);
	fwrite_padded_char("starttime",field_name_length,file_handle);
	fwrite_padded_char("startevent",field_name_length,file_handle);
	fwrite_padded_char("stopdate",field_name_length,file_handle);
	fwrite_padded_char("stoptime",field_name_length,file_handle);
	fwrite_padded_char("stopevent",field_name_length,file_handle);
	//fwrite_padded_char("user",field_name_length,file_handle);
	fwrite_padded_char("comment",field_name_length,file_handle);
	fwrite_padded_char("vehicle",field_name_length,file_handle);
	fwrite_padded_char("numchannels",field_name_length,file_handle);
	fwrite_padded_char("numdevices",field_name_length,file_handle);
	fwrite_padded_char("device",field_name_length,file_handle);

	/* version */

	matlab_file_write_string_numeric_array(file_handle,"Vision Matlab Structure 1.0",50);

	/* source */

#ifdef _CATOOL_UNICODE_
	matlab_file_write_wchar_numeric_array(file_handle, filename, 50);
#else
	matlab_file_write_string_numeric_array(file_handle,filename,50);
#endif
	/* project */

	matlab_file_write_string_numeric_array(file_handle,"",50);

	/* screen */

	matlab_file_write_string_numeric_array(file_handle,"",50);

	/* recorder */

	matlab_file_write_string_numeric_array(file_handle,"",50);

	/* startdate */

	matlab_file_write_string_numeric_array(file_handle,"",50);

	/* starttime */

	matlab_file_write_string_numeric_array(file_handle,"",50);

	/* startevent */

	matlab_file_write_string_numeric_array(file_handle,"",50);

	/* stopdate */

	matlab_file_write_string_numeric_array(file_handle,"",50);

	/* stoptime */

	matlab_file_write_string_numeric_array(file_handle,"",50);

	/* stopevent */

	matlab_file_write_string_numeric_array(file_handle,"",50);

	/* user */

	//matlab_file_write_string_numeric_array(file_handle,"Empty",50);

	/* comment */

	matlab_file_write_string_numeric_array(file_handle,"Converted from AVL IFile with catoolRT",50);

	/* vehicle */

	matlab_file_write_string_numeric_array(file_handle,"",50);

	/* numchannels */

	double value = (double)number_of_channels;
	matlab_file_write_double_numeric_array(file_handle,value);

	//matlab_file_write_uint8_numeric_array(file_handle,number_of_channels);

	/* numdevices */

	matlab_file_write_double_numeric_array(file_handle,1.0);

	/* device */

	matlab_file_write_structure_atimat_header_device(file_handle, device);

	end_of_structure_file_position = ftell(file_handle);
	
	if ((structure_size_file_position >= 0) && (end_of_structure_file_position > structure_size_file_position))
	{
		structure_size = end_of_structure_file_position - structure_size_file_position - 4;

		if (structure_size >= 0)
		{
			if (fseek(file_handle, structure_size_file_position, SEEK_SET) == 0)
			{
				fwrite_int(structure_size, file_handle);

				fseek(file_handle, end_of_structure_file_position, SEEK_SET);
			}
		}
	}
	else
	{
		logmessage(MSG_ERROR,"Unable to determine end of MATLAB file\n");
	}
}

void matlab_file_write_structure_atimat_signals(FILE* file_handle,char* array_name,int* dimensions,double* data)
{
	int structure_size_file_position;
	unsigned int flags;
	unsigned int array_name_length;
	unsigned int field_name_length;
	int end_of_structure_file_position;
	unsigned int number_of_fields;
	unsigned int total_field_name_length;
	int structure_size;
	int array_flags_subelement;
	char array_type;
	
	fwrite_int(miMATRIX,file_handle);

	structure_size_file_position = ftell(file_handle);
	
	fwrite_int(0,file_handle);			/* We'll go back later and fill in the number of bytes	*/

	/* Array Flags */

	array_type = mxSTRUCT_CLASS;
	flags = mfNONE;

	fwrite_int(miUINT32,file_handle);
	fwrite_int(8,file_handle);
	
	array_flags_subelement = (int)array_type | flags;
	
	fwrite_int(array_flags_subelement,file_handle);
    fwrite_int(0,file_handle);
	
	/* Dimensions Array */

	fwrite_int(miINT32,file_handle);
	fwrite_int(8,file_handle);
	fwrite_int(1,file_handle);
	fwrite_int(1,file_handle);

	/* Array Name */

	validate_variable_name(array_name);

	array_name_length = (unsigned int)strlen(array_name);

	if (array_name_length > SIZEOF_CHANNEL_NAME)
	{
		array_name_length = SIZEOF_CHANNEL_NAME;
	}

	matlab_file_write(file_handle,miINT8,array_name,array_name_length);

	/* Field Name Length */

	field_name_length = 12;
	
	matlab_file_write(file_handle,miINT32,&field_name_length,4);

	/* Field Names */

	number_of_fields = 2;

	total_field_name_length = number_of_fields * field_name_length;

	fwrite_int(miINT8,file_handle);
	fwrite_int(total_field_name_length,file_handle);

	fwrite_padded_char("values",field_name_length,file_handle);
	fwrite_padded_char("dimensions",field_name_length,file_handle);

	/* values */

	matlab_file_write_numeric_array(file_handle,miDOUBLE,false,false,NULL,2,dimensions,data,NULL);

	/* dimension */

	matlab_file_write_uint8_numeric_array(file_handle,1);

	end_of_structure_file_position = ftell(file_handle);
	
	if ((structure_size_file_position >= 0) && (end_of_structure_file_position > structure_size_file_position))
	{
		structure_size = end_of_structure_file_position - structure_size_file_position - 4;

		if (structure_size >= 0)
		{
			if (fseek(file_handle, structure_size_file_position, SEEK_SET) == 0)
			{
				fwrite_int(structure_size, file_handle);

				fseek(file_handle, end_of_structure_file_position, SEEK_SET);
			}
		}
	}
	else
	{
		logmessage(MSG_ERROR,"Unable to determine end of MATLAB file\n");
	}
}

void matlab_file_write_structure_atimat(FILE* file_handle,char* array_name,char* units,const double period,unsigned int data_dimension_0,unsigned int data_dimension_1,float* data,unsigned int axis_dimension_0,unsigned int axis_dimension_1,float* axis)
{
	int structure_size_file_position;
	unsigned int flags;
	unsigned int array_name_length;
	unsigned int field_name_length;
	int end_of_structure_file_position;
	unsigned int number_of_fields;
	unsigned int total_field_name_length;
	int structure_size;
	int dimensions[2];
	int array_flags_subelement;
	char array_type;
	int data_pointer;
	double* double_data = NULL;
	char signals[] = "signals";
	
	fwrite_int(miMATRIX,file_handle);

	structure_size_file_position = ftell(file_handle);
	
	fwrite_int(0,file_handle);			/* We'll go back later and fill in the number of bytes	*/

	/* Array Flags */

	array_type = mxSTRUCT_CLASS;
	flags = mfNONE;

	fwrite_int(miUINT32,file_handle);
	fwrite_int(8,file_handle);
	
	array_flags_subelement = (int)array_type | flags;
	
	fwrite_int(array_flags_subelement,file_handle);
    fwrite_int(0,file_handle);
	
	/* Dimensions Array */

	fwrite_int(miINT32,file_handle);
	fwrite_int(8,file_handle);
	fwrite_int(1,file_handle);
	fwrite_int(1,file_handle);

	/* Array Name */

	validate_variable_name(array_name);

	array_name_length = (unsigned int)strlen(array_name);

	if (array_name_length > SIZEOF_CHANNEL_NAME)
	{
		array_name_length = SIZEOF_CHANNEL_NAME;
	}

	matlab_file_write(file_handle,miINT8,array_name,array_name_length);

	/* Field Name Length */

	field_name_length = 12;
	
	matlab_file_write(file_handle,miINT32,&field_name_length,4);

	/* Field Names */

	number_of_fields = 10;

	total_field_name_length = number_of_fields * field_name_length;

	fwrite_int(miINT8,file_handle);
	fwrite_int(total_field_name_length,file_handle);

	fwrite_padded_char("Channelname",field_name_length,file_handle);
	fwrite_padded_char("Channeltype",field_name_length,file_handle);
	fwrite_padded_char("unit",field_name_length,file_handle);
	fwrite_padded_char("timeunit",field_name_length,file_handle);
	fwrite_padded_char("period",field_name_length,file_handle);
	fwrite_padded_char("device",field_name_length,file_handle);
	fwrite_padded_char("time",field_name_length,file_handle);
	fwrite_padded_char("signals",field_name_length,file_handle);
	fwrite_padded_char("min",field_name_length,file_handle);
	fwrite_padded_char("max",field_name_length,file_handle);

	/* Channelname */

	matlab_file_write_string_numeric_array(file_handle,array_name,SIZEOF_CHANNEL_NAME);

	/* Channeltype */

	matlab_file_write_string_numeric_array(file_handle,"DOUBLE",50);

	/* unit */

	matlab_file_write_string_numeric_array(file_handle,units,SIZEOF_UNITS);

	/* timeunit */

	matlab_file_write_string_numeric_array(file_handle,"Seconds",50);

	/* period */

	matlab_file_write_double_numeric_array(file_handle,period);

	/* device */

	matlab_file_write_string_numeric_array(file_handle,"catoolRT",50);

	/* time */

	dimensions[0] = axis_dimension_0;
	dimensions[1] = axis_dimension_1;

	double_data = (double*)malloc(dimensions[0] * sizeof(double));

	if (double_data == NULL)
	{
		fclose(file_handle);
		logmessage(FATAL,"Memory could not be allocated\n");
		exit(EXIT_FAILURE);
	}

	for (data_pointer=0;data_pointer<dimensions[0];data_pointer++)
	{
		double_data[data_pointer] = (double)axis[data_pointer];
	}

	matlab_file_write_numeric_array(file_handle,miDOUBLE,false,false,NULL,2,dimensions,double_data,NULL);

	free(double_data);

	/* signals */

	dimensions[0] = data_dimension_0;
	dimensions[1] = data_dimension_1;

	double_data = (double*)malloc(dimensions[0] * sizeof(double));

	if (double_data == NULL)
	{
		fclose(file_handle);
		logmessage(FATAL,"Memory could not be allocated\n");
		exit(EXIT_FAILURE);
	}

	for (data_pointer=0;data_pointer<dimensions[0];data_pointer++)
	{
		double_data[data_pointer] = (double)data[data_pointer];
	}

	matlab_file_write_structure_atimat_signals(file_handle, signals, dimensions, double_data);

	free(double_data);

	/* min */

	matlab_file_write_double_numeric_array(file_handle,0);

	/* max */

	matlab_file_write_double_numeric_array(file_handle,1);

	end_of_structure_file_position = ftell(file_handle);
	
	if ((structure_size_file_position >= 0) && (end_of_structure_file_position > structure_size_file_position))
	{
		structure_size = end_of_structure_file_position - structure_size_file_position - 4;

		if (structure_size >= 0)
		{
			if (fseek(file_handle, structure_size_file_position, SEEK_SET) == 0)
			{
				fwrite_int(structure_size, file_handle);

				fseek(file_handle, end_of_structure_file_position, SEEK_SET);
			}
		}
	}
	else
	{
		logmessage(MSG_ERROR,"Unable to determine end of MATLAB file\n");
	}
}

#ifdef _CATOOL_UNICODE_
bool save_atimat_file(FileData* file,wchar_t* filename,Analysis* analysis,const unsigned int ifile_output)
#else
bool save_atimat_file(FileData* file, char* filename, Analysis* analysis, const unsigned int ifile_output)
#endif
{
	FILE* matfile = NULL;
	unsigned int channel;
	unsigned int cycle;
	unsigned int analysis_number;
	unsigned int analysis_channel;
	unsigned int axis_size;
	char array_name[SIZEOF_CHANNEL_NAME];
	char units[SIZEOF_UNITS];
	unsigned int data_dimension_0;
	unsigned int data_dimension_1;
	unsigned int axis_dimension_0;
	unsigned int axis_dimension_1;
	unsigned int data_pointer;
	float* data = NULL;
	float* axis = NULL;
	unsigned int number_of_channels;
	double period;
	unsigned int resolution;
	double* cycle_percent = NULL;
	unsigned int crank_angle;
	double extra_time;
	unsigned int number_of_points;
	float average;
	double time;
	unsigned int t;
	char header[] = "header";

	bool export_angular_torque = false;
	
	if ((file->number_of_channels > 0) && (file->channel_data == NULL))
	{
		logmessage(MSG_ERROR,"Channels are configured but no data is present\n");

		return(false);
	}

	logmessage(DEBUG,"Generating MATLAB compatible channels names\n");
	generate_matlab_channelnames(file,analysis);

#ifdef _CATOOL_UNICODE_
	matfile = _wfopen(filename, L"wb");
#else
	matfile = fopen(filename,"wb");
#endif

	if ( matfile == NULL )
	{
		logmessage(MSG_ERROR,"MATLAB output file could not be opened\n");
		
		return(false);
	}
	
	number_of_channels = 0;
	for (channel=0;channel<file->number_of_channels;channel++)
	{
		if ((file->channel_data[channel].loaded == true) && 
			((file->channel_data[channel].abscissa.type == ABSCISSA_TIME) || (file->channel_data[channel].abscissa.type == ABSCISSA_CYCLE)))
		{
			number_of_channels += 1;
		}
	}

	for (analysis_number=0;analysis_number<analysis->number_of_channels;analysis_number++)
	{
		for (analysis_channel=0;analysis_channel<get_number_of_analysis_channels();analysis_channel++)
		{
			if ((analysis->channel[analysis_number].analysis_rqst[analysis_channel] >= AnalysisRqst::CalculateAndStore) &&
				(analysis->channel[analysis_number].results[analysis_channel].abscissa.type == ABSCISSA_CYCLE))
		    {
				number_of_channels += 1;
			}
		}
	}

	logmessage(DEBUG,"Exporting %u channels\n",number_of_channels);
	
	matlab_file_open(matfile);

	matlab_file_write_structure_atimat_header(matfile, header, filename, number_of_channels);

	for (channel=0;channel<file->number_of_channels;channel++)
	{
		if (file->channel_data[channel].loaded == true)
		{
			if ((file->channel_data[channel].abscissa.type == ABSCISSA_TIME) || (file->channel_data[channel].abscissa.type == ABSCISSA_CYCLE))
			{
				if (file->channel_data[channel].abscissa.type == ABSCISSA_TIME)
				{
					axis_size = file->channel_data[channel].samples_per_cycle;
				}
				else /* (file->channel_data[channel].abscissa.type == ABSCISSA_CYCLE) */
				{
					axis_size = file->number_of_cycles;
				}
			
				strncpy(array_name,file->channel_data[channel].matlab_name,SIZEOF_CHANNEL_NAME);
				array_name[SIZEOF_CHANNEL_NAME-1] = 0x00;
				strncpy(units,file->channel_data[channel].units,SIZEOF_UNITS);
				units[SIZEOF_UNITS-1] = 0x00;
			
				data_dimension_0 = axis_size;
				data_dimension_1 = 1;
				
				data = file->channel_data[channel].data;
			
				axis_dimension_0 = axis_size;
				axis_dimension_1 = 1;
				
				axis = (float*)malloc(axis_size * sizeof(float));
				
				if (axis == NULL)
				{
					fclose(matfile);
					logmessage(FATAL,"Memory could not be allocated\n");
					return(false);
				}

				if (file->channel_data[channel].abscissa.type == ABSCISSA_TIME)
				{
					period = (double)file->channel_data[channel].abscissa.resolution[0] * file->time_resolution;		/* ms */

					for(data_pointer=0;data_pointer<axis_size;data_pointer++)
					{
						axis[data_pointer] = (float)data_pointer*(float)period;
					}
				}
				else /* (file->channel_data[channel].abscissa.type == ABSCISSA_CYCLE) */
				{
					axis[0] = 0.0f;
					period = 0.0;

					for(cycle=1;cycle<axis_size;cycle++)
					{
						axis[cycle] = axis[cycle-1] + (file->channel_data[channel].duration[cycle-1] / 1000.0f);
						period += axis[cycle];
					}

					period /= axis_size;
				}

				logmessage(DEBUG,"Exporting raw channel '%s' with duration %f s\n",array_name,axis[axis_size-1]);

				matlab_file_write_structure_atimat(matfile,array_name,units,period,data_dimension_0,data_dimension_1,data,axis_dimension_0,axis_dimension_1,axis);
				
				free(axis);
				axis = NULL;
			}
		}
	}

	for (analysis_number=0;analysis_number<analysis->number_of_channels;analysis_number++)
	{
		for (analysis_channel=0;analysis_channel<get_number_of_analysis_channels();analysis_channel++)
		{
			if (analysis->channel[analysis_number].analysis_rqst[analysis_channel] >= AnalysisRqst::CalculateAndStore)
			{
				if (analysis->channel[analysis_number].results[analysis_channel].abscissa.type == ABSCISSA_CYCLE)
				{
					strncpy(array_name,analysis->channel[analysis_number].results[analysis_channel].matlab_name,SIZEOF_CHANNEL_NAME);
					array_name[SIZEOF_CHANNEL_NAME-1] = 0x00;
					strncpy(units,analysis->channel[analysis_number].results[analysis_channel].units,SIZEOF_UNITS);
					units[SIZEOF_UNITS-1] = 0x00;
				
					axis_size = file->number_of_cycles;
					
					data_dimension_0 = axis_size;
					data_dimension_1 = 1;
					
					data = analysis->channel[analysis_number].results[analysis_channel].data;
				
					axis_dimension_0 = axis_size;
					axis_dimension_1 = 1;
					
					axis = (float*)malloc(axis_size * sizeof(float));
					
					if (axis == NULL)
					{
						fclose(matfile);
						logmessage(FATAL,"Memory could not be allocated\n");
						return(false);
					}

					axis[0] = (file->channel_data[analysis->channel[analysis_number].channel].duration[0] / 1000.0f) * (file->channel_data[analysis->channel[analysis_number].channel].tdc_offset / (180.0f * (float)file->engine.number_of_strokes));
					period = 0.0;

					for(cycle=1;cycle<axis_size;cycle++)
					{
						axis[cycle] = axis[cycle - 1] + (file->channel_data[analysis->channel[analysis_number].channel].duration[cycle - 1] / 1000.0f);

						period += axis[cycle];
					}
					period /= axis_size;

					logmessage(DEBUG,"Exporting analysis channel '%s' with duration %f s\n",array_name,axis[axis_size-1]);

					matlab_file_write_structure_atimat(matfile,array_name,units,period,data_dimension_0,data_dimension_1,data,axis_dimension_0,axis_dimension_1,axis);
					
					free(axis);
					axis = NULL;
				}
				else if ((analysis->channel[analysis_number].results[analysis_channel].abscissa.type == ABSCISSA_CRANKANGLE) && (analysis_channel == ANGULAR_TORQUE) && (export_angular_torque == true))
				{
					channel = analysis->channel[analysis_number].channel;

					strncpy(array_name,analysis->channel[analysis_number].results[analysis_channel].matlab_name,SIZEOF_CHANNEL_NAME);
					array_name[SIZEOF_CHANNEL_NAME-1] = 0x00;
					strncpy(units,analysis->channel[analysis_number].results[analysis_channel].units,SIZEOF_UNITS);
					units[SIZEOF_UNITS-1] = 0x00;

					resolution = 1;		// ms
				
					period = 0.0;
					for (cycle=0;cycle<file->number_of_cycles;cycle++)
					{
						period += file->channel_data[channel].duration[cycle - 1];
					}

					axis_size = (unsigned int)(period / (double)resolution);
				
					data_dimension_0 = axis_size;
					data_dimension_1 = 1;
					
					data = (float*)malloc(axis_size * sizeof(float));
				
					if (data == NULL)
					{
						fclose(matfile);
						logmessage(FATAL,"Memory could not be allocated\n");
						return(false);
					}
					
					axis_dimension_0 = axis_size;
					axis_dimension_1 = 1;
					
					axis = (float*)malloc(axis_size * sizeof(float));
					
					if (axis == NULL)
					{
						fclose(matfile);
						logmessage(FATAL,"Memory could not be allocated\n");
						free(data);
						return(false);
					}

					cycle_percent = (double*)malloc(analysis->channel[analysis_number].results[analysis_channel].samples_per_cycle * sizeof(double));
					
					if (cycle_percent == NULL)
					{
						fclose(matfile);
						logmessage(FATAL,"Memory could not be allocated\n");
						free(data);
						free(axis);
						return(false);
					}
					
					for (crank_angle=0;crank_angle<analysis->channel[analysis_number].results[analysis_channel].samples_per_cycle-1;crank_angle++)
					{
						cycle_percent[crank_angle] = (double)(file->channel_data[analysis_number].abscissa.ca_to_deg[crank_angle + 1] -
							file->channel_data[analysis_number].abscissa.ca_to_deg[crank_angle]) /
							((double)file->engine.number_of_strokes*180.0);
					}
					
					cycle_percent[crank_angle] = (double)(((double)file->engine.number_of_strokes*180.0) -
						file->channel_data[analysis_number].abscissa.ca_to_deg[crank_angle]) /
						((double)file->engine.number_of_strokes*180.0);
					
					t = 0;
					cycle = 0;
					crank_angle = 0;
					extra_time = 0.0;
					while (t < axis_size)
					{
						axis[t] = (float)t / 1000.0f;
						
						number_of_points = 0;
						average = 0.0f;
						time = extra_time;
						while (time < (double)resolution)
						{
							time += (double)file->channel_data[channel].duration[cycle] * cycle_percent[crank_angle];
							
							if (cycle < file->number_of_cycles)
							{
								average += analysis->channel[analysis_number].results[analysis_channel].data[cycle*analysis->channel[analysis_number].results[analysis_channel].samples_per_cycle + crank_angle];
								number_of_points++;
							}
							
							crank_angle += 1;
								
							if (crank_angle >= analysis->channel[analysis_number].results[analysis_channel].samples_per_cycle)
							{
								crank_angle = 0;
								cycle += 1;
							}
						}
						
						extra_time = time - (double)resolution;
						
						average /= number_of_points;
						
						data[t] = average;
						
						t += resolution;
					}

					logmessage(DEBUG,"Exporting analysis channel '%s' with duration %f s\n",array_name,axis[axis_size-1]);

					matlab_file_write_structure_atimat(matfile,array_name,units,period,data_dimension_0,data_dimension_1,data,axis_dimension_0,axis_dimension_1,axis);
					
					free(axis);
					axis = NULL;
					
					free(data);
					data = NULL;
					
					free(cycle_percent);
					cycle_percent = NULL;
				}
				else
				{
					/* Do Nothing */
				}
			}
		}
	}
	
	matlab_file_close(matfile);
	
	return(true);
}
