/* Combustion Analysis Tool (CAT)
   www.catool.org

   Filename: analysis.h

   Purpose:  Provide common programming interface for processing acquired engine data

   Author:   Ben Brown
   Version:  1.2
   Date:     19.10.2015

   Revision: GPL Release

   Copyright (C) Xarin Limited, 2000-2024

      This program is free software: you can redistribute it and/or modify
      it under the terms of version 2 of the GNU General Public License as
      published by the Free Software Foundation.

      This program is distributed in the hope that it will be useful,
      but WITHOUT ANY WARRANTY; without even the implied warranty of
      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
      GNU General Public License for more details.

      You should have received a copy of the GNU General Public License
      along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef __ANALYSIS_H__
#define __ANALYSIS_H__

#include "cat.h"

void Init_FFTConfig(FFTConfig* config);
bool Prepare_FFT(const FileData* file,
				 const unsigned int channel,
				 const unsigned int units,
				 const float fft_start_window,
				 const float fft_finish_window,
				 const unsigned int fft_window_type,
	             const unsigned int number_of_cycles,
				 FFTConfig* config);
void Complete_FFT(FFTConfig* config);
void Return_FFT(FFTConfig* config,
				float* input,
				const float engine_speed,
				const bool engine_order,
				float* frequency,
				float* data);
void Return_MinMaxMean_Analysis(const FileData* file,
								const unsigned int cycle,
								const unsigned int channel,
								float* min,
								float* max,
								float *mean);
void Return_Angular_Torque_Results(const FileData* file, 
	                               const Analysis* analysis,
	                               const unsigned int cycle,
	                               const unsigned int channel,
	                               const unsigned int group, 
	                               const unsigned int number_of_cycles, 
	                               const float crankcase_pressure, 
	                               const float* pressure_to_torque, 
	                               float* data);
void Return_Burn_Angles(const FileData* file,
						const unsigned int channel,
						const float* mfb,
						const float start_of_combustion,
						const float eeoc,
						const float knock_onset_ca,
						const bool interpolate_mfb,
						float* burn_angle_1,
						float* burn_angle_2,
						float* burn_angle_5,
						float* burn_angle_10,
						float* burn_angle_20,
						float* burn_angle_25,
						float* burn_angle_50,
						float* burn_angle_75,
						float* burn_angle_80,
						float* burn_angle_90,
						float* burn_angle_95,
						float* burn_angle_98,
						float* burn_angle_99,
						float* burn_duration_0_2,
						float* burn_duration_0_5,
						float* burn_duration_0_10,
						float* burn_duration_0_90,
						float* burn_duration_2_90,
						float* burn_duration_5_90,
						float* burn_duration_10_90,
						float* knock_onset_umfb);
void Complete_CandD(float* first_differential,
                    float* second_differential,
                    float* second_filtered_differential);
void Prepare_CandD(const FileData* file,
                   const unsigned int channel,
                   float** first_differential,
                   float** second_differential,
                   float** second_filtered_differential);
void Return_CandD_Data(const FileData* file,
                       const unsigned int cycle,
                       const unsigned int channel,
                       const float* dCA,
					   const float cd_start_angle,
					   const float cd_finish_angle,
                       float* first_differential,
                       float* second_differential,
                       float* second_filtered_differential,
                       float* peak_cd_ki,
                       float* peak_cd_ki_ca,
                       float* data);
void Prepare_EEOC(const FileData* file,
				  const unsigned int channel,
	              const float eeoc_start_window,
	              const float eeoc_finish_window,
	              const float eeoc_index,
	              const float eeoc_default,
	              const unsigned int eeoc_method,
				  EEOCConfig* config);
void Complete_EEOC(EEOCConfig* config);
void Return_EEOC(const FileData* file,
				 const unsigned int cycle,
				 const unsigned int channel,
				 EEOCConfig* config,
				 float* eeoc);
void Return_Gamma_Data(const FileData* file,
                       const unsigned int channel,
					   const unsigned int method,
					   const float poly_comp,
					   const float poly_exp,
					   const float polytropic_index,
					   const float R,
					   const float* gamma,
					   const float* mean_gas_temp,
					   float* data);
void Return_Heat_Release_Data(const FileData* file,
							  const unsigned int channel,
	                          const unsigned int cycle,
							  float* dCA,
							  float* volume,
							  float* dVolume,
							  float* heat_transfer_area,
							  const float* mean_gas_temp,
							  const float* gamma,
							  const float* pressure,
							  const float t_ivc,
							  const float t_wall,
							  const float R,
							  const float start_of_combustion,
							  const float eeoc,
							  const float engine_speed,
							  const float poly_comp,
							  const float poly_exp,
							  const unsigned int heat_release_model,
							  const unsigned int heat_transfer_model,
	                          const unsigned int heat_release_window,
							  const float annand_a,
							  float* max_heat_release_rate,
							  float* max_heat_release_rate_ca,
							  float* total_heat_release,
							  float* net_heat_release_rate,
							  float* gross_heat_release_rate,
							  float* h);
void Return_IMEP(const FileData* file,
	             const unsigned int cycle,
				 const unsigned int channel,
	             float* volume,
				 float* dVolume,
				 float* pressure_to_torque,
				 const float crankcase_pressure,
				 const unsigned int method,
	             const float deactivation_delta_p,
				 float* gross_imep,
				 float* net_imep,
				 float* upper_pumping_imep,
				 float* lower_pumping_imep,
				 float* pumping_imep,
				 float* net_indicated_torque,
	             float* gross_indicated_torque,
				 float* angular_indicated_torque,
				 float* total_angular_indicated_torque,
	             float* crank_angle_imep,
				 bool* cylinder_deactivated);
void Return_MFB_Data(const FileData* file,
					 const unsigned int cycle,
					 const unsigned int channel,
					 const float* volume,
					 const float* dVolume,
					 const float* dCA,
					 const float mfb_n,
	                 const unsigned int mfb_pressure_method,
					 const float start_of_combustion,
					 const float eeoc,
					 const float* smoothed_pressure,
					 float* max_burn_rate,
					 float* max_burn_rate_ca,
					 float* data);
void Return_Motored_Pressure_Data(const FileData* file,
								  const unsigned int channel,
								  const unsigned int method,
								  const float poly_comp_finish_angle,
								  const float poly_comp_index,
								  const float* pressure,
								  const float* volume,
								  float* data);
void Return_PKP_Data(const FileData* file,
					 const unsigned int cycle,
					 const unsigned int channel,
					 float* dCA,
					 float* moving_pressure_average,
					 const float pkp_start_angle,
					 const float pkp_finish_angle,
					 const unsigned int knock_integral_type,
					 const float knock_onset_threshold,
					 float* peak_knocking_pressure,
					 float* peak_knocking_pressure_ca,
					 float* knock_integral,
					 float* knock_onset_ca,
					 float* data);
void Return_Piston_Acceleration_Data(const FileData* file,
									 const unsigned int channel,
									 const float engine_speed,
									 float* data);
void Return_Piston_Displacement_Data(const FileData* file,
									 const unsigned int channel,
									 float* data);
void Return_Piston_Velocity_Data(const FileData* file,
								 const unsigned int channel,
								 const float engine_speed,
								 float* data);
void Return_Polytropic_Indices_Data(const FileData* file,
									const unsigned int channel,
									float* volume,
									const float* pressure_data,
									float* data);
void Return_Pressure_Second_Derivative(const FileData* file,
	const unsigned int cycle,
	const unsigned int channel,
	const float window,
	float* d2p_max,
	float* d2p_max_ca,
	float* data);
void Return_Pressure_Analysis(const FileData* file,
	                          const unsigned int cycle,
							  const unsigned int channel,
                              float pressure_rise_window_start,
                              float pressure_rise_window_finish,
	                          float pressure_rise_window_range,
							  float* min_pressure,
							  float* min_pressure_ca,
							  float* max_pressure,
							  float* max_pressure_ca,
							  float* max_pressure_rise_rate,
							  float* max_pressure_rise_rate_ca,
							  float* data);
void Return_Pressure_Data(const FileData* file,
	                      const unsigned int cycle,
						  const unsigned int channel,
						  float* data);
void Prepare_SmoothedPressure_Data(const FileData* file,
	                               const unsigned int method,
	                               const float smoothing_range,
								   const float smoothing_resolution,
								   unsigned int* theta_range,
								   unsigned int* theta_resolution,
								   unsigned int* points_to_average,
								   float** coefficients);
void Complete_SmoothedPressure_Data(float** coefficients);
void Return_SmoothedPressure_Data(const FileData* file,
	                              const unsigned int cycle,
								  const unsigned int channel,
								  const unsigned int range,
								  const unsigned int resolution,
								  const unsigned int points_to_average,
								  const float* coefficients,
								  float* data);
void Return_SOC(const FileData* file,
	            const Analysis* analysis,
				const unsigned int cycle,
				const unsigned int channel,
	            float* digital_channel,
				const unsigned int start_of_combustion_channel,
				const unsigned int start_of_combustion_type,
				const float start_of_combustion_start_window,
				const float start_of_combustion_finish_window,
				const float start_of_combustion_angle,
				const unsigned int analysis_type,
				const bool aligned,
				const bool invert,
				const bool wraparound,
				const unsigned int number_of_cycles,
				float* dCA,
				float* start_of_combustion);
void Return_Temperature_Data(const FileData* file,
							 const unsigned int channel,
	                         const unsigned int cycle,
							 float* volume,
							 const float temp_ref_ca,
							 const float t_ivc,
	                         const float gamma,
	                         const unsigned int gas_temp_model,
							 float* pressure,
							 float* max_mean_gas_temp,
							 float* max_mean_gas_temp_ca,
							 float* data);
void Prepare_Polytropic_Index(const FileData* file,
	                          const unsigned int channel,
							  const float start_angle,
	                          const float finish_angle,
							  unsigned int* number_of_points,
							  unsigned int* start_crank_angle,
							  float** logPressure);
void Return_Polytropic_Index(const FileData* file,
                             const unsigned int cycle,
							 const unsigned int channel,
							 const unsigned int start_crank_angle,
							 const unsigned int number_of_points,
							 float* logVolume,
							 float* logPressure,
							 float* pressure,
							 float* gamma,
							 float* k);
void Return_Injection_Timing(const FileData* file,
							 const unsigned int channel,
							 const float engine_speed,
							 const float* digital_signal,
							 const float injection_window_start,
							 const float injection_window_finish,
							 const bool aligned,
							 const unsigned int max_number_of_injections,
							 float* soi1,
							 float* soi2,
							 float* soi3,
							 float* soi4,
							 float* soi5,
							 float* soi6,
							 float* eoi1,
							 float* eoi2,
							 float* eoi3,
							 float* eoi4,
							 float* eoi5,
							 float* eoi6,
							 float* noi,
							 float* injdur1,
							 float* injdur2,
							 float* injdur3,
							 float* injdur4,
							 float* injdur5,
							 float* injdur6,
							 float* separation1,
							 float* separation2,
							 float* separation3,
							 float* separation4,
							 float* separation5);
void Return_Digital_Channel(const FileData* file,
                            const unsigned int cycle,
							const unsigned int channel,
							const unsigned int type,
							const float latch_high,
							const float latch_low,
							const float angle_filter,
							float* data);

void Return_Camshaft_Analysis(const FileData* file,
	const unsigned int channel,
	const unsigned int cam_type,
	const unsigned int edge_type,
	const float reference_angle,
	const float offset,
	const float cam_edge_1,
	const float cam_edge_2,
	const float cam_edge_3,
	const float cam_edge_4,
	const float cam_edge_5,
	const float cam_edge_6,
	const float cam_edge_7,
	const float cam_edge_8,
	float* cam_angle,
	float* cam_advance);
void Complete_Polytropic_Index(float* logPressure);
void Complete_TLA(const unsigned int tla_method,
	              float* crank_angle_data,
                  float* pressure_data);
void Prepare_TLA(const FileData* file,
				 float tla_range,
	             const unsigned int tla_method,
                 float** crank_angle_data,
                 float** pressure_data);
void Return_TLA(const FileData* file,
	            const unsigned int cycle,
				const unsigned int channel,
				const unsigned int tla_method,
				const float tla_range,
				float* crank_angle_data,
				float* pressure_data,
				float max_pressure_angle,
				const float* smoothed_pressure_data,
				float* tla,
                float* combustion_symmetry,
				float* data);
void Return_Wiebe(const FileData* file,
	              const unsigned int channel,
	              const float start_of_combustion,
				  const float eeoc,
				  const float burn_angle_5,
				  const float burn_angle_10,
				  const float burn_angle_50,
				  const float burn_angle_90,
				  const float burn_angle_95,
				  const float wiebe_a_start,
				  const float wiebe_a_finish,
				  const float wiebe_a_step,
				  const float wiebe_m_start,
				  const float wiebe_m_finish,
				  const float wiebe_m_step,
				  float* wiebe_a,
				  float* wiebe_m,
				  float* data);
void Return_COG(const FileData* file,
	            const unsigned int channel,
				const float burn_angle_10,
				const float burn_angle_90,
				const float* mfb,
				float* centre_of_gravity);
void Return_Knock_Factor(const FileData* file,
	                     const unsigned int channel,
						 const float engine_speed,
						 const float max_pressure_ca,
						 const float* knkp,
						 const float knock_window_width,
						 const float reference_window_width,
						 const float speed_shift_start,
						 const float speed_shift_finish,
						 const float speed_shift_start_correction,
						 const float speed_shift_finish_correction,
						 const float noise_threshold,
						 float* knock_factor);
void Return_Postitive_HR_SOC(const FileData* file,
					   const unsigned int channel,
					   const float ivo,
					   const float evo,
					   const float poly_comp,
					   const float poly_exp,
	                   const float max_pressure_ca,
					   const float* volume,
					   const float* pressure,
					   const float threshold,
					   float* start_of_combustion);
void Return_Accelerometer_Data(const FileData* file,
	const unsigned int cycle,
	const unsigned int channel,
	const float* dCA,
	const float reference_start_angle,
	const float reference_finish_angle,
	const float knock_window_start_angle,
	const float knock_window_finish_angle,
	float* knock_integral,
	float* knock_factor);
void Return_Crank_Sensor_Analysis(const FileData* file,
	const unsigned int channel,
	const float* digital_signal,
	float* missing_tooth_angle_1,
	float* missing_tooth_angle_2,
	float* missing_tooth_ratio_min,
	float* missing_tooth_ratio_max,
	float* gap_ratio_min,
	float* gap_ratio_max);
void Return_Coil_Analysis(const FileData* file,
	const unsigned int cycle,
	const unsigned int channel,
	const bool calculate_max_current,
	const bool calculate_restrike,
	const float* digital_channel,
	const float engine_speed,
	float* dwell_time,
	float* max_coil_current,
	float* restrike_max_current,
	float* restrike_timing,
	float* restrike_delay_time,
	float* restrike_dwell_time);
void Return_Intake_Pressure_Analysis(const FileData* file,
	const unsigned int cycle,
	const unsigned int previous_cycle,
	const unsigned int channel,
	const float* smoothed_pressure_data,
	float ivo,
	float* max_pressure_ivo,
	float* max_pressure_ivo_ca);
void Return_Exhaust_Pressure_Analysis(const FileData* file,
	const unsigned int cycle,
	const unsigned int previous_cycle,
	const unsigned int channel,
	const float* smoothed_pressure_data,
	float evc,
	float* min_pressure_evc,
	float* min_pressure_evc_ca);
void Return_Indicated_Data(const FileData* file,
	const float indicated_torque,
	const float engine_speed,
	float* indicated_power,
	float* indicated_power_hp,
	float* indicated_tq_lbft);
void Return_Cam_Edges(const FileData* file,
	const unsigned int channel,
	const float* digital_signal,
	float* edge_1,
	float* edge_2,
	float* edge_3,
	float* edge_4,
	float* edge_5,
	float* edge_6,
	float* edge_7,
	float* edge_8);
void Return_Knock_Signal_Energy(const FileData* file,
	const unsigned int channel,
	const float* dCA,
	const float engine_speed,
	const float knock_onset_ca,
	const float* knocking_pressure,
	float* knock_signal_energy);
void Return_Misfire_Data(const FileData* file,
	const unsigned int start_cycle,
	const unsigned int finish_cycle,
	const unsigned int number_of_cycles,
	const unsigned int number_of_cycles_f,
	const unsigned int number_of_cycles_m,
	const unsigned int number_of_cycles_s,
	const unsigned char* cycle_classification,
	float* misfire_f,
	float* misfire_m,
	float* misfire_s);

#endif
