/* Combustion Analysis Tool (CAT)
   www.catool.org

   Filename: analysis.c

   Purpose:  Provide common programming interface for processing acquired engine data

   Author:   Ben Brown
   Version:  1.2
   Date:     19.10.2015

   Revision: GPL Release

   Copyright (C) Xarin Limited, 2000-2024

      This program is free software: you can redistribute it and/or modify
      it under the terms of version 2 of the GNU General Public License as
      published by the Free Software Foundation.

      This program is distributed in the hope that it will be useful,
      but WITHOUT ANY WARRANTY; without even the implied warranty of
      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
      GNU General Public License for more details.

      You should have received a copy of the GNU General Public License
      along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <limits.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>

#include "cat.h"

extern unsigned int debug_level;

void Init_FFTConfig(FFTConfig* config)
{
	config->buffer = NULL;
	config->ifac = NULL;
	config->samples_per_cycle = 1;
	config->start_crank_angle = 0;
	config->speed_to_freq = 1.0f;
	config->window = NULL;
	config->wsave = NULL;
}

bool Prepare_FFT(const FileData* file,
	             const unsigned int channel,
				 const unsigned int units,
				 const float fft_start_window,
				 const float fft_finish_window,
				 const unsigned int fft_window_type,
	             const unsigned int number_of_cycles,
				 FFTConfig* config)
{
	unsigned int finish_crank_angle;

	if ((file == NULL) || (config == NULL))
	{
		return(false);
	}

	if ((channel >= file->number_of_channels) || (file->number_of_channels == 0))
	{
		return(false);
	}

	if (number_of_cycles > 1)
	{
		config->start_crank_angle = 0;

		if (file->number_of_cycles >= number_of_cycles)
		{
			config->number_of_samples = file->channel_data[channel].samples_per_cycle * number_of_cycles;
		}
		else
		{
			config->number_of_samples = file->channel_data[channel].samples_per_cycle * file->number_of_cycles;
		}
	}
	else
	{
		config->start_crank_angle = DegreesToCrankAngle(file, fft_start_window, channel);
		finish_crank_angle = DegreesToCrankAngle(file, fft_finish_window, channel);

		config->number_of_samples = (finish_crank_angle - config->start_crank_angle + 1);

		if ((config->number_of_samples == 0) || (config->number_of_samples > file->channel_data[channel].samples_per_cycle))
		{
			return(false);
		}
	}

	if (config->number_of_samples % 2 == 0)
	{
		config->samples_per_cycle = config->number_of_samples / 2 - 1;
	}
	else
	{
		config->samples_per_cycle = (config->number_of_samples - 1) / 2;
	}

	if (number_of_cycles > 1)
	{
		config->speed_to_freq = 1.0f / (60.0f * file->engine.number_of_strokes) / number_of_cycles * 2.0f;
	}
	else
	{
		config->speed_to_freq = (360.0f / (fft_finish_window - fft_start_window)) * ((float)(finish_crank_angle - config->start_crank_angle + 1)) / 2.0f / 60.0f / (float)config->samples_per_cycle;
	}

	if (units == UNITS_EO)
	{
		config->speed_to_freq /= 60.0f * 1000.0f;
	}
	else if (units == UNITS_KHZ)
	{
		config->speed_to_freq /= 1000.0f;
	}
	else
	{
		/* Do Nothing */
	}

	if (config->buffer != NULL)
	{
		free(config->buffer);
	}

	config->buffer = (float*)malloc(config->number_of_samples * sizeof(float));

	if (config->buffer == NULL)
	{
		logmessage(FATAL,"Memory could not be allocated\n");
	}

	if (config->wsave != NULL)
	{
		free(config->wsave);
	}
	
	config->wsave = (float*)malloc((2 * config->number_of_samples + 15) * sizeof(float));
	
	if (config->wsave == NULL)
	{
		logmessage(FATAL,"Memory could not be allocated\n");
	}
	
	if (config->ifac != NULL)
	{
		free(config->ifac);
	}

	config->ifac = (int*)malloc((config->number_of_samples + 1) * sizeof(int));
	
	if (config->ifac == NULL)
	{
		logmessage(FATAL,"Memory could not be allocated\n");
	}

	if (config->window != NULL)
	{
		free(config->window);
	}

	config->window = (float*)malloc(config->number_of_samples * sizeof(float));

	if (config->window == NULL)
	{
		logmessage(FATAL,"Memory could not be allocated\n");
	}

	calculate_window(config->window, config->number_of_samples, fft_window_type);

	rffti(config->number_of_samples, config->wsave, config->ifac);

	return(true);
}

void Complete_FFT(FFTConfig* config)
{
	if (config == NULL)
	{
		return;
	}

	if (config->buffer != NULL)
	{
		free(config->buffer);
		config->buffer = NULL;
	}
	
	if (config->wsave != NULL)
	{
		free(config->wsave);
		config->wsave = NULL;
	}
	
	if (config->ifac != NULL)
	{
		free(config->ifac);
		config->ifac = NULL;
	}

	if (config->window != NULL)
	{
		free(config->window);
		config->window = NULL;
	}
}

void Return_FFT(FFTConfig* config,
				float* input,
			    const float engine_speed,
				const bool engine_order,
		  	    float* frequency,
				float* data)
{
	unsigned int crank_angle;

#ifdef _SAFE_MEMORY
    if ((input != NULL) && (data != NULL) && (frequency != NULL) && (config != NULL))
#endif
    {
		for (crank_angle=0;crank_angle<config->number_of_samples;crank_angle++)
    	{
    		config->buffer[crank_angle] = config->window[crank_angle] * input[config->start_crank_angle + crank_angle];
		}
		
		rfftf(config->number_of_samples, config->buffer, config->wsave, config->ifac);

		if (engine_order == true)
		{
			for (crank_angle = 1; crank_angle < config->samples_per_cycle; crank_angle++)
			{
				data[crank_angle - 1] = sqrtf(config->buffer[2 * crank_angle - 1] * config->buffer[2 * crank_angle - 1] + config->buffer[2 * crank_angle] * config->buffer[2 * crank_angle]) / (float)config->samples_per_cycle;	/* should be config->number_of_samples? */

				frequency[crank_angle - 1] = config->speed_to_freq * (float)crank_angle;
			}
		}
		else
		{
			for (crank_angle = 1; crank_angle < config->samples_per_cycle; crank_angle++)
			{
				data[crank_angle - 1] = sqrtf(config->buffer[2 * crank_angle - 1] * config->buffer[2 * crank_angle - 1] + config->buffer[2 * crank_angle] * config->buffer[2 * crank_angle]) / (float)config->samples_per_cycle;	/* should be config->number_of_samples? */

				frequency[crank_angle - 1] = config->speed_to_freq * engine_speed * (float)crank_angle;
			}
		}
    }
}

void Return_MinMaxMean_Analysis(const FileData* file,
	const unsigned int cycle,
	const unsigned int channel,
	float* min,
	float* max,
	float *mean)
{
	unsigned int crank_angle;
	float calculated_mean = 0.0f;
	unsigned int samples_per_cycle = file->channel_data[channel].samples_per_cycle;

	minmaxf(&file->channel_data[channel].data[cycle*samples_per_cycle], samples_per_cycle, min, max);

	for (crank_angle = 0; crank_angle < file->channel_data[channel].samples_per_cycle; crank_angle++)
	{
		calculated_mean += ReturnCAData(file, cycle, crank_angle, channel);
	}

	calculated_mean /= (float)file->channel_data[channel].samples_per_cycle;

#ifdef _SAFE_MEMORY
	if (mean != NULL)
#endif
	{
		*mean = calculated_mean;
	}
}

void Return_Angular_Torque_Results(const FileData* file, 
	                               const Analysis* analysis,
	                               const unsigned int cycle,
	                               const unsigned int channel,
	                               const unsigned int group, 
	                               const unsigned int number_of_cycles, 
	                               const float crankcase_pressure, 
	                               const float* pressure_to_torque, 
	                               float* data)
{
	int theta;
	float angle;
	unsigned int maximum_theta;
	unsigned int crank_angle;
	unsigned int group_channel;
	unsigned int other_channel;
	int adjusted_cycle;
	float pressure;

	for (crank_angle = 0; crank_angle < file->channel_data[channel].samples_per_cycle; crank_angle++)
	{
		data[crank_angle] = (ReturnCAData(file, cycle, crank_angle, channel) - crankcase_pressure) * pressure_to_torque[crank_angle];

		for (group_channel = 0; group_channel < analysis->group[group].number_of_channels; group_channel++)
		{
			other_channel = analysis->group[group].channels[group_channel];

			if (other_channel != channel)
			{
				maximum_theta = MaximumTheta(file, other_channel);

				angle = file->channel_data[channel].tdc_offset - file->channel_data[other_channel].tdc_offset;

				theta = ReturnTheta(file, crank_angle, channel) - DegreesToThetaAbs(file, angle);

				adjusted_cycle = cycle;

				while (theta < 0)
				{
					theta += maximum_theta;
					adjusted_cycle -= 1;
				}

				if (adjusted_cycle < 0)
				{
					adjusted_cycle += number_of_cycles;
				}

				pressure = ReturnThetaData(file, adjusted_cycle, theta, other_channel);

				data[crank_angle] += (pressure - crankcase_pressure) * ReturnThetaDataEx(file, theta, channel, pressure_to_torque);
			}
		}
	}
}

void Return_Intake_Pressure_Analysis(const FileData* file,
	const unsigned int current_cycle,
	const unsigned int previous_cycle,
	const unsigned int channel,
	const float* smoothed_pressure_data,
	float ivo,
	float* max_pressure_ivo,
	float* max_pressure_ivo_ca)
{
	unsigned int cycle;
	unsigned int crank_angle;
	unsigned int start_crank_angle;
	unsigned int finish_crank_angle;
	float start_angle;
	float finish_angle;
	float pressure;
	float max_pressure_tmp;
	unsigned int max_pressure_ca_tmp;

	start_angle = ivo - 45.0f;

	if (start_angle < (float)file->engine.number_of_strokes*-90.0f)
	{
		start_angle += (float)file->engine.number_of_strokes*180.0f;
	}

	finish_angle = ivo + 45.0f;

	if (finish_angle > (float)file->engine.number_of_strokes*90.0f)
	{
		finish_angle -= (float)file->engine.number_of_strokes*180.0f;
	}

	start_crank_angle = DegreesToCrankAngle(file, start_angle, channel);
	finish_crank_angle = DegreesToCrankAngle(file, finish_angle, channel);

	max_pressure_ca_tmp = start_crank_angle;

	if (start_crank_angle < finish_crank_angle)
	{
		if (start_angle < 0.0f)
		{
			cycle = previous_cycle;
		}
		else
		{
			cycle = current_cycle;
		}

		max_pressure_tmp = ReturnCAData(file, cycle, start_crank_angle, channel);

		for (crank_angle = start_crank_angle + 1; crank_angle <= finish_crank_angle; crank_angle++)
		{
			pressure = ReturnCAData(file, cycle, crank_angle, channel);

			if (pressure > max_pressure_tmp)
			{
				max_pressure_tmp = pressure;
				max_pressure_ca_tmp = crank_angle;
			}
		}
	}
	else
	{
		max_pressure_tmp = ReturnCAData(file, previous_cycle, start_crank_angle, channel);

		for (crank_angle = start_crank_angle + 1; crank_angle < file->channel_data[channel].samples_per_cycle; crank_angle++)
		{
			pressure = ReturnCAData(file, previous_cycle, crank_angle, channel);

			if (pressure > max_pressure_tmp)
			{
				max_pressure_tmp = pressure;
				max_pressure_ca_tmp = crank_angle;
			}
		}

		for (crank_angle = 0; crank_angle <= finish_crank_angle; crank_angle++)
		{
			pressure = ReturnCAData(file, current_cycle, crank_angle, channel);

			if (pressure > max_pressure_tmp)
			{
				max_pressure_tmp = pressure;
				max_pressure_ca_tmp = crank_angle;
			}
		}
	}

	*max_pressure_ivo = max_pressure_tmp;
	*max_pressure_ivo_ca = CrankAngleToDegrees(file,max_pressure_ca_tmp,channel);
}

void Return_Exhaust_Pressure_Analysis(const FileData* file,
	const unsigned int current_cycle,
	const unsigned int previous_cycle,
	const unsigned int channel,
	const float* smoothed_pressure_data,
	float evc,
	float* min_pressure_evc,
	float* min_pressure_evc_ca)
{
	unsigned int cycle;
	unsigned int crank_angle;
	unsigned int start_crank_angle;
	unsigned int finish_crank_angle;
	float start_angle;
	float finish_angle;
	float pressure;
	float min_pressure_tmp;
	unsigned int min_pressure_ca_tmp;

	start_angle = evc - 45.0f;

	if (start_angle < (float)file->engine.number_of_strokes*-90.0f)
	{
		start_angle += (float)file->engine.number_of_strokes*180.0f;
	}

	finish_angle = evc + 45.0f;

	if (finish_angle > (float)file->engine.number_of_strokes*90.0f)
	{
		finish_angle -= (float)file->engine.number_of_strokes*180.0f;
	}

	start_crank_angle = DegreesToCrankAngle(file, start_angle, channel);
	finish_crank_angle = DegreesToCrankAngle(file, finish_angle, channel);

	min_pressure_ca_tmp = start_crank_angle;

	if (start_crank_angle < finish_crank_angle)
	{
		if (start_angle < 0.0f)
		{
			cycle = previous_cycle;
		}
		else
		{
			cycle = current_cycle;
		}

		min_pressure_tmp = ReturnCAData(file, cycle, start_crank_angle, channel);

		for (crank_angle = start_crank_angle + 1; crank_angle <= finish_crank_angle; crank_angle++)
		{
			pressure = ReturnCAData(file, cycle, crank_angle, channel);

			if (pressure < min_pressure_tmp)
			{
				min_pressure_tmp = pressure;
				min_pressure_ca_tmp = crank_angle;
			}
		}
	}
	else
	{
		min_pressure_tmp = ReturnCAData(file, previous_cycle, start_crank_angle, channel);

		for (crank_angle = start_crank_angle + 1; crank_angle < file->channel_data[channel].samples_per_cycle; crank_angle++)
		{
			pressure = ReturnCAData(file, previous_cycle, crank_angle, channel);

			if (pressure < min_pressure_tmp)
			{
				min_pressure_tmp = pressure;
				min_pressure_ca_tmp = crank_angle;
			}
		}

		for (crank_angle = 0; crank_angle <= finish_crank_angle; crank_angle++)
		{
			pressure = ReturnCAData(file, current_cycle, crank_angle, channel);

			if (pressure < min_pressure_tmp)
			{
				min_pressure_tmp = pressure;
				min_pressure_ca_tmp = crank_angle;
			}
		}
	}

	*min_pressure_evc = min_pressure_tmp;
	*min_pressure_evc_ca = CrankAngleToDegrees(file,min_pressure_ca_tmp,channel);
}

/* Function: Return_Pressure_Analysis

   OK for partial cycle abscissas */

void Return_Pressure_Analysis(const FileData* file,
	                          const unsigned int cycle,
							  const unsigned int channel,
                              float pressure_rise_window_start,
                              float pressure_rise_window_finish,
	                          float pressure_rise_window_range,
							  float* min_pressure,
							  float* min_pressure_ca,
							  float* max_pressure,
							  float* max_pressure_ca,
							  float* max_pressure_rise_rate,
							  float* max_pressure_rise_rate_ca,
							  float* data)
{
	unsigned int crank_angle;
	float pressure;
	float dp;
	float max_pressure_tmp;
	unsigned int max_pressure_ca_tmp;
	float min_pressure_tmp;
	unsigned int min_pressure_ca_tmp;
	float max_pressure_rise_rate_tmp;
	unsigned int max_pressure_rise_rate_ca_tmp;
	unsigned int start_crank_angle;
	unsigned int finish_crank_angle;
	unsigned int theta_step_size;
	float range;
	float pressure_1;
	float pressure_2;
	unsigned int theta;

	min_pressure_tmp = ReturnCAData(file,cycle,0,channel);
	min_pressure_ca_tmp = 0;
	max_pressure_tmp = min_pressure_tmp;
	max_pressure_ca_tmp = 0;
	max_pressure_rise_rate_tmp = -10000.0f;
	max_pressure_rise_rate_ca_tmp = 0;

#ifdef _SAFE_MEMORY
	if ((max_pressure != NULL) || (max_pressure_ca != NULL) || (min_pressure != NULL) || (min_pressure_ca != NULL))
#endif
	{
		for (crank_angle=0;crank_angle<file->channel_data[channel].samples_per_cycle;crank_angle++)
		{
			pressure = ReturnCAData(file,cycle,crank_angle,channel);

			if (pressure > max_pressure_tmp)
			{
				max_pressure_tmp = pressure;
				max_pressure_ca_tmp = crank_angle;
			}
			else if (pressure < min_pressure_tmp)
			{
				min_pressure_tmp = pressure;
				min_pressure_ca_tmp = crank_angle;
			}
			else
			{
				/* Do Nothing */
			}
		}

#ifdef _SAFE_MEMORY		
		if (max_pressure != NULL)
#endif
		{
			*max_pressure = max_pressure_tmp;
		}
#ifdef _SAFE_MEMORY	
		if (max_pressure_ca != NULL)
#endif
		{
			*max_pressure_ca = CrankAngleToDegrees(file, max_pressure_ca_tmp, channel);
		}
#ifdef _SAFE_MEMORY		
		if (min_pressure != NULL)
#endif
		{
			*min_pressure = min_pressure_tmp;
		}
#ifdef _SAFE_MEMORY			
		if (min_pressure_ca != NULL)
#endif
		{	
			*min_pressure_ca = CrankAngleToDegrees(file, min_pressure_ca_tmp, channel);
		}
	}

#ifdef _SAFE_MEMORY
	if (data != NULL)
#endif
	{
		start_crank_angle = DegreesToCrankAngle(file, pressure_rise_window_start, channel);
		finish_crank_angle = DegreesToCrankAngle(file, pressure_rise_window_finish, channel);
		theta_step_size = DegreesToThetaAbs(file, pressure_rise_window_range);

		if (ReturnTheta(file, start_crank_angle, channel) < theta_step_size)
		{
			theta_step_size = 0;
		}

		if (ReturnTheta(file, finish_crank_angle, channel) + theta_step_size >= MaximumTheta(file, channel))
		{
			theta_step_size = 0;
		}

		range = 2.0f * pressure_rise_window_range;

		for (crank_angle = start_crank_angle; crank_angle < finish_crank_angle; crank_angle++)
		{
			theta = ReturnTheta(file, crank_angle, channel);

			pressure_1 = ReturnThetaData(file, cycle, theta - theta_step_size, channel);
			pressure_2 = ReturnThetaData(file, cycle, theta + theta_step_size, channel);

			dp = (pressure_2 - pressure_1) / range;

			data[crank_angle] = dp;

			if (dp > max_pressure_rise_rate_tmp)
			{
				max_pressure_rise_rate_tmp = dp;
				max_pressure_rise_rate_ca_tmp = crank_angle;
			}
		}

		for (crank_angle = 0; crank_angle<start_crank_angle; crank_angle++)
		{
			data[crank_angle] = data[start_crank_angle];
		}

		if (finish_crank_angle > 0)
		{
			for (crank_angle = finish_crank_angle; crank_angle < file->channel_data[channel].samples_per_cycle; crank_angle++)
			{
				data[crank_angle] = data[finish_crank_angle - 1];
			}
		}

#ifdef _SAFE_MEMORY
		if (max_pressure_rise_rate != NULL)
#endif
		{
			*max_pressure_rise_rate = max_pressure_rise_rate_tmp;
		}
#ifdef _SAFE_MEMORY
		if (max_pressure_rise_rate_ca != NULL)
#endif
		{
			*max_pressure_rise_rate_ca = CrankAngleToDegrees(file, max_pressure_rise_rate_ca_tmp, channel);
		}
	}
}

/* Function: Return_TLA

   PARTIAL CYCLE ABSCISSA requires check */

void Complete_TLA(const unsigned int tla_method,
	              float* crank_angle_data,
	              float* pressure_data)
{
	if (crank_angle_data != NULL)
	{
		free(crank_angle_data);
	}
	
	if (pressure_data != NULL)
	{
		free(pressure_data);
	}
}

void Prepare_TLA(const FileData* file,
	             float tla_range,
                 const unsigned int tla_method,
				 float** crank_angle_data,
				 float** pressure_data)
{
	unsigned int tla_range_theta;
	unsigned int theta_step_size;
	unsigned int number_of_data_points;

	if (*crank_angle_data != NULL)
	{
		free(*crank_angle_data);
	}

	if (*pressure_data != NULL)
	{
		free(*pressure_data);
	}

	if ((tla_method == TLA_POLYNOMIAL) || (tla_method == TLA_POLYNOMIAL_SMOOTHED))
	{
		tla_range_theta = DegreesToThetaAbs(file, tla_range);
		theta_step_size = DegreesToThetaAbs(file, 0.2f);
		number_of_data_points = (unsigned int)((2.0f * (float)tla_range_theta) / (float)theta_step_size + 1.0f);

		*crank_angle_data = (float*)malloc(number_of_data_points * sizeof(float));
		if (*crank_angle_data == NULL)
		{
			logmessage(FATAL, "Memory could not be allocated\n");
		}

		*pressure_data = (float*)malloc(number_of_data_points * sizeof(float));
		if (*pressure_data == NULL)
		{
			logmessage(FATAL, "Memory could not be allocated\n");
		}
	}
}

void Return_TLA(const FileData* file,
	            const unsigned int cycle,
				const unsigned int channel,
				const unsigned int tla_method,
				const float tla_range,
				float* crank_angle_data,
				float* pressure_data,
				float max_pressure_angle,
				const float* smoothed_pressure_data,
				float* tla,
                float* combustion_symmetry,
				float* data)
{
	unsigned int number_of_data_points;
	unsigned int data_pointer;
	unsigned int theta;
	unsigned int max_pressure_theta;
	unsigned int tla_range_theta;
	unsigned int start_theta;
	unsigned int finish_theta;
	unsigned int theta_step_size;
	unsigned int crank_angle;
	float a0;
	float a1;
	float a2;
	float point_of_inflexion;
	float angle;
	float pressure;
	float range = 90.0f*(float)file->engine.number_of_strokes - tla_range - 1.0f;
	unsigned int maximum_theta;

	if (max_pressure_angle < -range)
	{
		max_pressure_angle = -range;
	}

	if (max_pressure_angle > range)
	{
		max_pressure_angle = range;
	}

	max_pressure_theta = DegreesToTheta(file, max_pressure_angle);
	theta_step_size = DegreesToThetaAbs(file, 0.2f);

#if 0
	float sum = 0.0f;
	finish_theta = DegreesToThetaAbs(file, 60.0f);
	
	for (theta = 1; theta <= finish_theta; theta+=theta_step_size)
	{
		float error = ReturnThetaDataEx(file, max_pressure_theta + theta, channel, smoothed_pressure_data) - ReturnThetaDataEx(file, max_pressure_theta - theta, channel, smoothed_pressure_data);

		sum += error*error;
	}

#ifdef _SAFE_MEMORY
	if (combustion_symmetry != NULL)
#endif
	{
		*combustion_symmetry = sum;
	}
#endif

	if ((tla_method == TLA_POLYNOMIAL) || (tla_method == TLA_POLYNOMIAL_SMOOTHED))
	{
		tla_range_theta = DegreesToThetaAbs(file, tla_range);

		start_theta = max_pressure_theta - tla_range_theta;
		finish_theta = max_pressure_theta + tla_range_theta;
		maximum_theta = MaximumTheta(file, channel);

		if (start_theta >= maximum_theta)
		{
			start_theta = maximum_theta - 1;
		}

		if (finish_theta >= maximum_theta)
		{
			finish_theta = maximum_theta - 1;
		}

		number_of_data_points = (unsigned int)((2.0f * (float)tla_range_theta) / (float)theta_step_size + 1.0f);

		data_pointer = 0;
		
		if (tla_method == TLA_POLYNOMIAL)
		{
			for (theta = start_theta; theta <= finish_theta; theta += theta_step_size)
			{
				crank_angle_data[data_pointer] = ThetaToDegrees(file, theta) - max_pressure_angle;
				pressure_data[data_pointer] = ReturnThetaData(file, cycle, theta, channel);
				data_pointer += 1;
			}
		}
		else /* if (tla_method == TLA_POLYNOMIAL_SMOOTHED) */
		{
			for (theta = start_theta; theta <= finish_theta; theta += theta_step_size)
			{
				crank_angle_data[data_pointer] = ThetaToDegrees(file, theta) - max_pressure_angle;
				pressure_data[data_pointer] = ReturnThetaDataEx(file, theta, channel, smoothed_pressure_data);
				data_pointer += 1;
			}
		}

		if (calculate_second_order_least_squares(number_of_data_points, crank_angle_data, pressure_data, &a0, &a1, &a2) == 1)
		{
			if (fabs(a2) > 0.0f)
			{
				point_of_inflexion = -a1 / (2.0f * a2);
			}
			else
			{
				point_of_inflexion = 0.0f;
			}
		}
		else
		{
			point_of_inflexion = 0.0f;
		}
#ifdef _SAFE_MEMORY
		if (tla != NULL)
#endif
		{
			*tla = -(max_pressure_angle + point_of_inflexion);
		}
#ifdef _SAFE_MEMORY
		if (data != NULL)
#endif
		{
			for (crank_angle = 0; crank_angle < file->channel_data[channel].samples_per_cycle; crank_angle++)
			{
				angle = CrankAngleToDegrees(file, crank_angle, channel) - max_pressure_angle;
				pressure = a2 * angle * angle + a1 * angle + a0;

				if (pressure > 0.0f)
				{
					data[crank_angle] = pressure;
				}
				else
				{
					data[crank_angle] = 0.0f;
				}
			}
		}
	}
	else if (tla_method == TLA_SLICE)
	{
		float minus_40 = ReturnThetaDataEx(file, DegreesToTheta(file, -40.0f), channel, smoothed_pressure_data);
		float minus_35 = ReturnThetaDataEx(file, DegreesToTheta(file, -35.0f), channel, smoothed_pressure_data);
		float minus_30 = ReturnThetaDataEx(file, DegreesToTheta(file, -30.0f), channel, smoothed_pressure_data);
		float minus_25 = ReturnThetaDataEx(file, DegreesToTheta(file, -25.0f), channel, smoothed_pressure_data);
		float minus_20 = ReturnThetaDataEx(file, DegreesToTheta(file, -20.0f), channel, smoothed_pressure_data);
		float minus_15 = ReturnThetaDataEx(file, DegreesToTheta(file, -15.0f), channel, smoothed_pressure_data);
		float minus_10 = ReturnThetaDataEx(file, DegreesToTheta(file, -10.0f), channel, smoothed_pressure_data);
		float minus_5 = ReturnThetaDataEx(file, DegreesToTheta(file, -5.0f), channel, smoothed_pressure_data);

		start_theta = DegreesToTheta(file, 0.0f);
		finish_theta = DegreesToTheta(file, 50.0f);

		unsigned int plus_40 = finish_theta;
		unsigned int plus_35 = finish_theta;
		unsigned int plus_30 = finish_theta;
		unsigned int plus_25 = finish_theta;
		unsigned int plus_20 = finish_theta;
		unsigned int plus_15 = finish_theta;
		unsigned int plus_10 = finish_theta;
		unsigned int plus_5 = finish_theta;

		for (theta = start_theta; theta < finish_theta; theta++)
		{
			pressure = ReturnThetaDataEx(file, theta, channel, smoothed_pressure_data);

			if (pressure > minus_5)
			{
				plus_5 = theta;
			}
			else if (pressure > minus_10)
			{
				plus_10 = theta;
			}
			else if (pressure > minus_15)
			{
				plus_15 = theta;
			}
			else if (pressure > minus_20)
			{
				plus_20 = theta;
			}
			else if (pressure > minus_25)
			{
				plus_25 = theta;
			}
			else if (pressure > minus_30)
			{
				plus_30 = theta;
			}
			else if (pressure > minus_35)
			{
				plus_35 = theta;
			}
			else if (pressure > minus_40)
			{
				plus_40 = theta;
			}
			else
			{
				break;
			}
		}

		float error_40 = 40.0f - ThetaToDegrees(file, plus_40);
		float error_35 = 35.0f - ThetaToDegrees(file, plus_35);
		float error_30 = 30.0f - ThetaToDegrees(file, plus_30);
		float error_25 = 25.0f - ThetaToDegrees(file, plus_25);
		float error_20 = 20.0f - ThetaToDegrees(file, plus_20);
		float error_15 = 15.0f - ThetaToDegrees(file, plus_15);
		float error_10 = 10.0f - ThetaToDegrees(file, plus_10);
		float error_5 = 5.0f - ThetaToDegrees(file, plus_5);

		*tla = (error_40 + error_35 + error_30 + error_25 + error_20 + error_15 + error_10 + error_5) / 8.0f / 2.0f;
		
		for (crank_angle = 0; crank_angle < file->channel_data[channel].samples_per_cycle; crank_angle++)
		{
			data[crank_angle] = 0.0f;
		}
	}
	else
	{
		/* Do Nothing */
	}
}

/* Function: Return_Wiebe

   This is very slow, a reduced range and resolution of m and a would be helpful.
   OK for partial cycle abscissas */

void Return_Wiebe(const FileData* file,
	              const unsigned int channel,
				  const float start_of_combustion,
				  const float eeoc,
				  const float burn_angle_5,
				  const float burn_angle_10,
				  const float burn_angle_50,
				  const float burn_angle_90,
				  const float burn_angle_95,
				  const float wiebe_a_start,
				  const float wiebe_a_finish,
				  const float wiebe_a_step,
				  const float wiebe_m_start,
				  const float wiebe_m_finish,
				  const float wiebe_m_step,
				  float* wiebe_a,
				  float* wiebe_m,
				  float* data)
{
	unsigned int crank_angle;
	unsigned int eeoc_crank_angle;
	unsigned int start_of_combustion_crank_angle;
	float smallest_mfb_error = 1E30f;
	float a;
	float m;
	float combustion_percentage;
	float combustion_period = eeoc - start_of_combustion;
	float calc_mfb_5;
	float calc_mfb_10;
	float calc_mfb_50;
	float calc_mfb_90;
	float calc_mfb_95;
	float wiebe_error;
	float wiebe_m_tmp = 0.0f;
	float wiebe_a_tmp = 0.0f;
	
	/*
	
	SAE 2015-

	float td = burn_angle_10 - start_of_combustion;
	float tb = burn_angle_90 - burn_angle_10;

	wiebe_m_tmp = log(log(1.0f - 0.1f) - log(1.0f - 0.9f)) / (log(td) - log(td + tb)) - 1.0f;

	wiebe_a_tmp = -powf(combustion_period / td, wiebe_m_tmp + 1.0f) * log(1.0f - 0.1f);
	
	*/

	for (m=wiebe_m_start;m<=wiebe_m_finish;m+=wiebe_m_step)
	{
		for (a=wiebe_a_start;a<=wiebe_a_finish;a+=wiebe_a_step)
		{
			combustion_percentage = (burn_angle_5-start_of_combustion)/combustion_period;
			calc_mfb_5 = 1.0f - powf((float)FLT_e,-a*powf(combustion_percentage,m+1.0f));
			
			combustion_percentage = (burn_angle_10-start_of_combustion)/combustion_period;
			calc_mfb_10 = 1.0f - powf((float)FLT_e, -a*powf(combustion_percentage, m + 1.0f));
			
			combustion_percentage = (burn_angle_50-start_of_combustion)/combustion_period;
			calc_mfb_50 = 1.0f - powf((float)FLT_e, -a*powf(combustion_percentage, m + 1.0f));
			
			combustion_percentage = (burn_angle_90-start_of_combustion)/combustion_period;
			calc_mfb_90 = 1.0f - powf((float)FLT_e, -a*powf(combustion_percentage, m + 1.0f));
			
			combustion_percentage = (burn_angle_95-start_of_combustion)/combustion_period;
			calc_mfb_95 = 1.0f - powf((float)FLT_e, -a*powf(combustion_percentage, m + 1.0f));

			wiebe_error = ((0.50f - calc_mfb_50)*(0.50f - calc_mfb_50) +
				(0.10f - calc_mfb_10)*(0.10f - calc_mfb_10) +
				(0.90f - calc_mfb_90)*(0.90f - calc_mfb_90) +
				(0.05f - calc_mfb_5)*(0.05f - calc_mfb_5) +
				(0.95f - calc_mfb_95)*(0.95f - calc_mfb_95));

			if (wiebe_error < smallest_mfb_error)
			{
				wiebe_m_tmp = m;
				wiebe_a_tmp = a;
				smallest_mfb_error = wiebe_error;
			}
		}
	}

#ifdef _SAFE_MEMORY
	if (data != NULL)
#endif
	{
		start_of_combustion_crank_angle = DegreesToCrankAngle(file,start_of_combustion,channel);

		if (start_of_combustion_crank_angle < 1)
		{
			start_of_combustion_crank_angle = 1;
		}

		eeoc_crank_angle = DegreesToCrankAngle(file,eeoc,channel);

		for (crank_angle=0;crank_angle<start_of_combustion_crank_angle;crank_angle++)
		{
			data[crank_angle] = 0.0f;
		}
	
		for (crank_angle=start_of_combustion_crank_angle;crank_angle<=eeoc_crank_angle;crank_angle++)
		{
			if (combustion_period > 0.0f)
			{
				combustion_percentage = (float)(CrankAngleToDegrees(file, crank_angle, channel) - start_of_combustion) / combustion_period;
			}
			else
			{
				combustion_percentage = 1.0f;
			}

			data[crank_angle] = 100.0f*(1.0f - powf((float)FLT_e, -wiebe_a_tmp*powf(combustion_percentage, wiebe_m_tmp + 1)));
		}

		for (crank_angle=eeoc_crank_angle+1;crank_angle<file->channel_data[channel].samples_per_cycle;crank_angle++)
		{
			data[crank_angle] = 100.0f;
		}
	}

#ifdef _SAFE_MEMORY
	if (wiebe_a != NULL)
#endif
	{
		*wiebe_a = wiebe_a_tmp;
	}
#ifdef _SAFE_MEMORY
	if (wiebe_m != NULL)
#endif
	{
		*wiebe_m = wiebe_m_tmp;
	}
}

/* Function: Return_SOC
  
   Purpose: Calculates start of combustion from maximum positive rise rate of a channel.
            This may be injector needle lift, fuel injection pressure, low tension coil voltage or
            high tension pick up.
*/

void Return_SOC(const FileData* file,
	const Analysis* analysis,
	const unsigned int cycle,
	unsigned int channel,
	float* digital_channel,
	const unsigned int start_of_combustion_channel,
	const unsigned int start_of_combustion_type,
	const float start_of_combustion_start_window,
	const float start_of_combustion_finish_window,
	const float start_of_combustion_angle,
	const unsigned int analysis_type,
	const bool aligned,
	const bool invert,
	const bool wraparound,
	const unsigned int number_of_cycles,
	float* dCA,
	float* start_of_combustion)
{
	float old_value;
	float value;
	float start_of_combustion_tmp;
	float rise_rate;
	unsigned int start_window;
	unsigned int finish_window;
	unsigned int crank_angle;
	float max_value_rise_rate;
	unsigned int max_value_rise_rate_ca;

	switch (start_of_combustion_type)
    {
        case SOC_FIXED:
        {
			start_of_combustion_tmp = start_of_combustion_angle;
			break;
        }
		case SOC_DIGITAL_RISING_EDGE:
		{
			start_window = DegreesToCrankAngle(file, start_of_combustion_start_window, channel);
			finish_window = DegreesToCrankAngle(file, start_of_combustion_finish_window, channel);

			max_value_rise_rate_ca = UINT_MAX;
			crank_angle = start_window + 1;

			if (digital_channel != NULL)
			{
				while (crank_angle <= finish_window)
				{
					if ((digital_channel[crank_angle - 1] < 0.5f) && (digital_channel[crank_angle] > 0.5f))
					{
						max_value_rise_rate_ca = crank_angle;
						break;
					}

					crank_angle++;
				}
			}

			if (max_value_rise_rate_ca == UINT_MAX)
			{
				start_of_combustion_tmp = 0.0f;
			}
			else
			{
				start_of_combustion_tmp = CrankAngleToDegrees(file, max_value_rise_rate_ca, channel);
			}

			break;
		}
		case SOC_DIGITAL_FALLING_EDGE:
		{
			start_window = DegreesToCrankAngle(file, start_of_combustion_start_window, channel);
			finish_window = DegreesToCrankAngle(file, start_of_combustion_finish_window, channel);

			max_value_rise_rate_ca = UINT_MAX;
			crank_angle = start_window + 1;

			if (digital_channel != NULL)
			{
				while (crank_angle <= finish_window)
				{
					if ((digital_channel[crank_angle - 1] > 0.5f) && (digital_channel[crank_angle] < 0.5f))
					{
						max_value_rise_rate_ca = crank_angle;
						break;
					}

					crank_angle++;
				}
			}

			if (max_value_rise_rate_ca == UINT_MAX)
			{
				start_of_combustion_tmp = 0.0f;
			}
			else
			{
				start_of_combustion_tmp = CrankAngleToDegrees(file, max_value_rise_rate_ca, channel);
			}

			break;
		}
    	case SOC_CA_CHANNEL_RISE:
		{
			if (aligned == true)
			{
				start_window = DegreesToCrankAngle(file, start_of_combustion_start_window + file->channel_data[channel].tdc_offset - file->channel_data[start_of_combustion_channel].tdc_offset, start_of_combustion_channel);
				finish_window = DegreesToCrankAngle(file, start_of_combustion_finish_window + file->channel_data[channel].tdc_offset - file->channel_data[start_of_combustion_channel].tdc_offset, start_of_combustion_channel);
			}
			else
			{
				start_window = DegreesToCrankAngle(file, start_of_combustion_start_window, start_of_combustion_channel);
				finish_window = DegreesToCrankAngle(file, start_of_combustion_finish_window, start_of_combustion_channel);
			}

			if ((start_window+2 < file->channel_data[start_of_combustion_channel].samples_per_cycle) && (finish_window < file->channel_data[start_of_combustion_channel].samples_per_cycle))
			{
				clip_crank_angle(file,&start_window,start_of_combustion_channel);
				clip_crank_angle(file,&finish_window,start_of_combustion_channel);

				if (aligned == true)
				{
					old_value = ReturnReferencedCAData(file, cycle, start_window, start_of_combustion_channel, channel, number_of_cycles, wraparound);
					value = ReturnReferencedCAData(file, cycle, start_window + 1, start_of_combustion_channel, channel, number_of_cycles, wraparound);
				}
				else
				{
					old_value = ReturnCAData(file, cycle, start_window, start_of_combustion_channel);
					value = ReturnCAData(file, cycle, start_window + 1, start_of_combustion_channel);
				}

				max_value_rise_rate = ( value - old_value ) / dCA[start_window];
				max_value_rise_rate_ca = start_window + 1;
				
				if (aligned == true)
				{
					for (crank_angle = start_window + 2; crank_angle <= finish_window; crank_angle++)
					{
						value = ReturnReferencedCAData(file, cycle, crank_angle, start_of_combustion_channel, channel, number_of_cycles, wraparound);
						rise_rate = (value - old_value) / dCA[crank_angle-1];

						if (rise_rate > max_value_rise_rate)
						{
							max_value_rise_rate = rise_rate;
							max_value_rise_rate_ca = crank_angle;
						}

						old_value = value;
					}
				}
				else
				{
					for (crank_angle = start_window + 2; crank_angle <= finish_window; crank_angle++)
					{
						value = ReturnCAData(file, cycle, crank_angle, start_of_combustion_channel);
						rise_rate = (value - old_value) / dCA[crank_angle-1];

						if (rise_rate > max_value_rise_rate)
						{
							max_value_rise_rate = rise_rate;
							max_value_rise_rate_ca = crank_angle;
						}

						old_value = value;
					}
				}

    			start_of_combustion_tmp = CrankAngleToDegrees(file, max_value_rise_rate_ca, start_of_combustion_channel);
			}
			else
			{
				start_of_combustion_tmp = 0.0f;
			}

            break;
        }
		case SOC_CA_CHANNEL_FALL:
        {
			if (aligned == true)
			{
				start_window = DegreesToCrankAngle(file, start_of_combustion_start_window + file->channel_data[channel].tdc_offset - file->channel_data[start_of_combustion_channel].tdc_offset, start_of_combustion_channel);
				finish_window = DegreesToCrankAngle(file, start_of_combustion_finish_window + file->channel_data[channel].tdc_offset - file->channel_data[start_of_combustion_channel].tdc_offset, start_of_combustion_channel);
			}
			else
			{
				start_window = DegreesToCrankAngle(file, start_of_combustion_start_window, start_of_combustion_channel);
				finish_window = DegreesToCrankAngle(file, start_of_combustion_finish_window, start_of_combustion_channel);
			}

			if ((start_window+2 < file->channel_data[start_of_combustion_channel].samples_per_cycle) && (finish_window < file->channel_data[start_of_combustion_channel].samples_per_cycle))
			{
	            clip_crank_angle(file,&start_window,start_of_combustion_channel);
	            clip_crank_angle(file,&finish_window,start_of_combustion_channel);
	
				if (aligned == true)
				{
					old_value = ReturnReferencedCAData(file, cycle, start_window, start_of_combustion_channel, channel, number_of_cycles, wraparound);
					value = ReturnReferencedCAData(file, cycle, start_window + 1, start_of_combustion_channel, channel, number_of_cycles, wraparound);
				}
				else
				{
					old_value = ReturnCAData(file, cycle, start_window, start_of_combustion_channel);
					value = ReturnCAData(file, cycle, start_window + 1, start_of_combustion_channel);
				}

				max_value_rise_rate = ( old_value - value ) / dCA[start_window];
				max_value_rise_rate_ca = start_window+1;
	
				if (aligned == true)
				{
					for (crank_angle = start_window + 2; crank_angle <= finish_window; crank_angle++)
					{
						value = ReturnReferencedCAData(file, cycle, crank_angle, start_of_combustion_channel, channel, number_of_cycles, wraparound);
						rise_rate = (old_value - value) / dCA[crank_angle-1];

						if (rise_rate > max_value_rise_rate)
						{
							max_value_rise_rate = rise_rate;
							max_value_rise_rate_ca = crank_angle;
						}

						old_value = value;
					}
				}
				else
				{
					for (crank_angle = start_window + 2; crank_angle <= finish_window; crank_angle++)
					{
						value = ReturnCAData(file, cycle, crank_angle, start_of_combustion_channel);
						rise_rate = (old_value - value) / dCA[crank_angle-1];

						if (rise_rate > max_value_rise_rate)
						{
							max_value_rise_rate = rise_rate;
							max_value_rise_rate_ca = crank_angle;
						}

						old_value = value;
					}
				}
	
	    		start_of_combustion_tmp = CrankAngleToDegrees(file, max_value_rise_rate_ca, start_of_combustion_channel);
			}
			else
			{
				start_of_combustion_tmp = 0.0f;
			}
			
            break;
        }
		case SOC_CA_CHANNEL_AVG:
		{
			start_window = DegreesToCrankAngle(file, start_of_combustion_start_window, start_of_combustion_channel);
			finish_window = DegreesToCrankAngle(file, start_of_combustion_finish_window, start_of_combustion_channel);

			if ((start_window + 2 < file->channel_data[start_of_combustion_channel].samples_per_cycle) && (finish_window < file->channel_data[start_of_combustion_channel].samples_per_cycle))
			{
				clip_crank_angle(file, &start_window, start_of_combustion_channel);
				clip_crank_angle(file, &finish_window, start_of_combustion_channel);

				value = 0.0f;
				if (aligned == true)
				{
					for (crank_angle = start_window; crank_angle <= finish_window; crank_angle++)
					{
						value += ReturnReferencedCAData(file, cycle, crank_angle, start_of_combustion_channel, channel, number_of_cycles, wraparound);
					}
				}
				else
				{
					for (crank_angle = start_window; crank_angle <= finish_window; crank_angle++)
					{
						value += ReturnCAData(file, cycle, crank_angle, start_of_combustion_channel);
					}
				}

				start_of_combustion_tmp = value / (float)(finish_window - start_window + 1);
			}
			else
			{
				start_of_combustion_tmp = 0.0f;
			}

			break;
		}
		case SOC_CYC_CHANNEL:
		{
			switch (file->channel_data[start_of_combustion_channel].abscissa.type)
			{
			case ABSCISSA_CYCLE:
			{
				start_of_combustion_tmp = file->channel_data[start_of_combustion_channel].data[cycle];

				break;
			}
			case ABSCISSA_CRANKANGLE:
			{
				unsigned int soc_analysis_channel = 0;

				while((analysis->channel[soc_analysis_channel].channel != start_of_combustion_channel) && (soc_analysis_channel < analysis->number_of_channels))
				{
					soc_analysis_channel++;
				}

				if ((analysis->channel[soc_analysis_channel].channel == start_of_combustion_channel) && (analysis->channel[soc_analysis_channel].analysis_rqst[analysis_type] >= AnalysisRqst::TemporaryCalculation))
				{
					start_of_combustion_tmp = analysis->channel[soc_analysis_channel].results[analysis_type].data[cycle];
				}
				else
				{
					start_of_combustion_tmp = 0.0f;
				}

				break;
			}
			default:
			{
				start_of_combustion_tmp = 0.0f;
									
				break;
			}
			}

			break;
		}
        case SOC_INVALID:
		default:
        {
            start_of_combustion_tmp = 0.0f;
            break;
        }
	}

	if (invert == true)
	{
		start_of_combustion_tmp *= -1.0f;
	}

	if (start_of_combustion_tmp < -90.0f*file->engine.number_of_strokes)
	{
		start_of_combustion_tmp = 0.0f;
	}
	else if (start_of_combustion_tmp > 90.0f*file->engine.number_of_strokes)
	{
		start_of_combustion_tmp = 0.0f;
	}
	else
	{
		/* Do Nothing */
	}
#ifdef _SAFE_MEMORY
	if (start_of_combustion != NULL)
#endif
	{
		*start_of_combustion = start_of_combustion_tmp;
	}
}
/* Function: Return_EEOC

   PARTIAL CYCLE ABSCISSA requires check */
   
void Prepare_EEOC(const FileData* file,
	const unsigned int channel,
	const float eeoc_start_window,
	const float eeoc_finish_window,
	const float eeoc_index,
	const float eeoc_default,
	const unsigned int eeoc_method,
	EEOCConfig* config)
{
	unsigned int maximum_theta = MaximumTheta(file, channel);

	if (config == NULL)
	{
		return;
	}

	config->index = eeoc_index;
	config->method = eeoc_method;

	config->start_window_theta = DegreesToTheta(file, eeoc_start_window);
	config->finish_window_theta = DegreesToTheta(file, eeoc_finish_window);

	config->averaging_range = DegreesToThetaAbs(file, 5.0f);
	config->step_size = DegreesToThetaAbs(file, 0.2f);

	if (config->averaging_range % config->step_size > 0)
	{
		config->averaging_range = 25 * config->step_size;
	}

	if (config->finish_window_theta + config->averaging_range >= maximum_theta)
	{
		config->method = EEOC_METHOD_FIXED;
	}

	if (config->values != NULL)
	{
		free(config->values);
	}

	config->values = (float*)malloc(maximum_theta * sizeof(float));
	
	if (config->values == NULL)
    {
    	logmessage(FATAL,"Memory could not be allocated\n");
    }
}

void Complete_EEOC(EEOCConfig* config)
{
	if (config == NULL)
	{
		return;
	}

	if (config->values != NULL)
	{
		free(config->values);
		config->values = NULL;
	}
}

void Return_EEOC(const FileData* file,
				 const unsigned int cycle,
				 const unsigned int channel,
				 EEOCConfig* config,
				 float* eeoc)
{
	float eeoc_sum;
	float eeoc_pressure;
	float eeoc_volume;
	float max_eeoc_sum;
	
	unsigned int theta;
	unsigned int eeoc_theta;

	if (config->method == EEOC_METHOD_FIXED)
	{
#ifdef _SAFE_MEMORY
		if (eeoc != NULL)
#endif
		{
			*eeoc = config->default_value;

			return;
		}
	}

	for (theta = config->start_window_theta - config->averaging_range; theta <= config->finish_window_theta + config->averaging_range; theta += config->step_size)
	{
		eeoc_pressure = ReturnThetaData(file, cycle, theta, channel);
		eeoc_volume = Volume(file, channel, theta);
		config->values[theta] = eeoc_pressure * (float)powf(eeoc_volume, config->index);
	}

    eeoc_sum = 0.0f;
	for (theta = config->start_window_theta - config->averaging_range; theta <= config->start_window_theta + config->averaging_range; theta += config->step_size)
	{
		eeoc_sum += config->values[theta];
	}

	eeoc_theta = config->start_window_theta;
	max_eeoc_sum = eeoc_sum;

	for (theta= config->start_window_theta+ config->step_size;theta<= config->finish_window_theta;theta+= config->step_size)
	{
		eeoc_sum = eeoc_sum - config->values[theta- config->averaging_range- config->step_size] + config->values[theta+ config->averaging_range];

		if (eeoc_sum > max_eeoc_sum)
		{
			eeoc_theta = theta;
			max_eeoc_sum = eeoc_sum;
		}
	}
#ifdef _SAFE_MEMORY
	if (eeoc != NULL)
#endif
	{
		*eeoc = ThetaToDegrees(file,eeoc_theta) + 10.0f;
	}
}

void Return_Indicated_Data(const FileData* file,
						    const float indicated_torque,
	                        const float engine_speed,
	                        float* indicated_power,
	                        float* indicated_power_hp,
	                        float* indicated_tq_lbft)
{
#ifdef _SAFE_MEMORY
	if (indicated_power != NULL)
#endif
	{
		*indicated_power = indicated_torque * engine_speed / 9548.8f;
	}

#ifdef _SAFE_MEMORY
	if (indicated_power_hp != NULL)
#endif
	{
		*indicated_power_hp = indicated_torque * engine_speed / 9548.8f * KW_TO_HP;
	}

#ifdef _SAFE_MEMORY
	if (indicated_tq_lbft != NULL)
#endif
	{
		*indicated_tq_lbft = indicated_torque * NM_TO_LBFT;
	}
}

/* Function: Return_IMEP
  
   Calculating upper and lower pumping MEP adds significantly to the processing time as each
   pressure must be compared to 1.0 bar
  
   PARTIAL CYCLE ABSCISSA requires check */

void Return_IMEP(const FileData* file,
	const unsigned int cycle,
	const unsigned int channel,
	float* volume,
	float* dVolume,
	float* pressure_to_torque,
	const float crankcase_pressure,
	const unsigned int method,
	const float deactivation_delta_p,
	float* gross_imep,
	float* net_imep,
	float* upper_pumping_imep,
	float* lower_pumping_imep,
	float* pumping_imep,
	float* net_indicated_torque,
	float* gross_indicated_torque,
	float* angular_indicated_torque,
	float* total_angular_indicated_torque,
	float* crank_angle_imep,
	bool* cylinder_deactivated)
{
	unsigned int crank_angle;
	unsigned int crank_angle_180;
	unsigned int crank_angle_540;
	float p1;
	float p2;
	float upper_pumping_imep_tmp = 0.0f;
	float lower_pumping_imep_tmp = 0.0f;
	float gross_imep_tmp = 0.0f;
	float piston_pressure;
	float total_angular_indicated_torque_tmp;
	float pumping_imep_tmp;
	float net_imep_tmp;
	bool cyl_deact = false;
	float pdV;
	float sv2 = 2.0f * file->engine.swept_volume;

	/* Calc IMEP */

#ifdef _SAFE_MEMORY 
	if ((upper_pumping_imep == NULL) || 
		(lower_pumping_imep == NULL) || 
		(gross_imep == NULL) ||
		(total_angular_indicated_torque == NULL) ||
		(angular_indicated_torque == NULL) ||
		(pumping_imep == NULL) ||
		(net_imep == NULL) ||
		(net_indicated_torque == NULL) ||
		(gross_indicated_torque == NULL) ||
		(cylinder_deactivated == NULL))
	{
		return;
	}
#endif

	if (file->engine.swept_volume < FLT_EPSILON)
	{
		return;
	}

	if (file->engine.number_of_strokes == 2)
	{
		if (method == IMEP_MEAN_PRESSURE)
		{
			for (crank_angle = 0; crank_angle < file->channel_data[channel].samples_per_cycle-1; crank_angle++)
			{
				p1 = ReturnCAData(file, cycle, crank_angle, channel);
				p2 = ReturnCAData(file, cycle, crank_angle + 1, channel);

				pdV = (p1 + p2) * dVolume[crank_angle];

				gross_imep_tmp += pdV;

				crank_angle_imep[crank_angle] = gross_imep_tmp / sv2;
			}

			gross_imep_tmp /= sv2;
		}
		else if (method == IMEP_MEAN_PRESSURE_1DEG)
		{
			unsigned int theta;
			unsigned int theta_359 = DegreesToThetaAbs(file, 359.0f);
			unsigned int theta_1 = DegreesToThetaAbs(file, 1.0f);
			float current_volume = ReturnThetaDataEx(file, 0, channel, volume);
			float next_volume;

			for (theta = 0; theta < theta_359; theta += theta_1)
			{
				next_volume = ReturnThetaDataEx(file, theta + theta_1, channel, volume);

				p1 = ReturnThetaData(file, cycle, theta, channel);
				p2 = ReturnThetaData(file, cycle, theta + theta_1, channel);

				pdV = (p1 + p2) * (next_volume - current_volume);

				gross_imep_tmp += pdV;

				crank_angle = ReturnCrankAngle(file, theta, channel);

				crank_angle_imep[crank_angle] = gross_imep_tmp / sv2;

				current_volume = next_volume;
			}

			gross_imep_tmp /= sv2;
		}
		else
		{
			for (crank_angle = 0; crank_angle < file->channel_data[channel].samples_per_cycle; crank_angle++)
			{
				p1 = ReturnCAData(file, cycle, crank_angle, channel);

				pdV = p1 * dVolume[crank_angle];

				gross_imep_tmp += pdV;

				crank_angle_imep[crank_angle] = gross_imep_tmp / file->engine.swept_volume;
			}

			gross_imep_tmp /= file->engine.swept_volume;
		}
	}
	else /*(file->engine.number_of_strokes == 4 */
	{
		crank_angle_180 = DegreesToCrankAngle(file, -180.0f, channel);
		crank_angle_540 = DegreesToCrankAngle(file, 180.0f, channel);

		p1 = ReturnCAData(file, cycle, 0, channel);
		p2 = ReturnCAData(file, cycle, crank_angle_180, channel);

		if ((p1 - p2) > deactivation_delta_p)
		{
			/* Intake deactivated */

			cyl_deact = true;
		}
		else
		{
			p1 = ReturnCAData(file, cycle, crank_angle_540, channel);
			p2 = ReturnCAData(file, cycle, file->channel_data[channel].samples_per_cycle - 1, channel);

			if ((p2 - p1) > deactivation_delta_p)
			{
				/* Exhaust deactivated */

				cyl_deact = true;
			}
		}

		if (method == IMEP_MEAN_PRESSURE)
		{
			for (crank_angle = 0; crank_angle < crank_angle_180; crank_angle++)
			{
				p1 = ReturnCAData(file, cycle, crank_angle, channel);
				p2 = ReturnCAData(file, cycle, crank_angle + 1, channel);

				pdV = (p1 + p2) * dVolume[crank_angle];

				if (p1 > 1.0f)
				{
					upper_pumping_imep_tmp += pdV;
				}
				else
				{
					lower_pumping_imep_tmp += pdV;
				}

				crank_angle_imep[crank_angle] = (upper_pumping_imep_tmp + lower_pumping_imep_tmp) / sv2;
			}

			float low_sum = upper_pumping_imep_tmp + lower_pumping_imep_tmp;

			if (cyl_deact == true)
			{
				low_sum /= sv2;

				for (crank_angle = crank_angle_180; crank_angle < crank_angle_540; crank_angle++)
				{
					crank_angle_imep[crank_angle] = low_sum;
				}
			}
			else
			{
				for (crank_angle = crank_angle_180; crank_angle < crank_angle_540; crank_angle++)
				{
					p1 = ReturnCAData(file, cycle, crank_angle, channel);
					p2 = ReturnCAData(file, cycle, crank_angle + 1, channel);

					pdV = (p1 + p2) * dVolume[crank_angle];

					gross_imep_tmp += pdV;

					crank_angle_imep[crank_angle] = (low_sum + gross_imep_tmp) / sv2;
				}
			}

			for (crank_angle = crank_angle_540; crank_angle < file->channel_data[channel].samples_per_cycle - 1; crank_angle++)
			{
				p1 = ReturnCAData(file, cycle, crank_angle, channel);
				p2 = ReturnCAData(file, cycle, crank_angle + 1, channel);

				pdV = (p1 + p2) * dVolume[crank_angle];

				if (p1 > 1.0f)
				{
					upper_pumping_imep_tmp += pdV;
				}
				else
				{
					lower_pumping_imep_tmp += pdV;
				}

				crank_angle_imep[crank_angle] = (gross_imep_tmp + upper_pumping_imep_tmp + lower_pumping_imep_tmp) / sv2;
			}

			crank_angle_imep[crank_angle] = (gross_imep_tmp + upper_pumping_imep_tmp + lower_pumping_imep_tmp) / sv2;

			gross_imep_tmp /= sv2;
			upper_pumping_imep_tmp /= sv2;
			lower_pumping_imep_tmp /= sv2;
		}
		else if (method == IMEP_MEAN_PRESSURE_1DEG)
		{
			unsigned int theta;
			unsigned int theta_180 = DegreesToTheta(file, -180.0f);
			unsigned int theta_540 = DegreesToTheta(file, 180.0f);
			unsigned int theta_719 = DegreesToTheta(file, 359.0f);
			unsigned int theta_1 = DegreesToThetaAbs(file, 1.0f);
			float current_volume = ReturnThetaDataEx(file, 0, channel, volume);
			float next_volume;
			unsigned int last_crank_angle = 0;
			unsigned int ca;

			for (theta = 0; theta < theta_180; theta += theta_1)
			{
				next_volume = ReturnThetaDataEx(file, theta + theta_1, channel, volume);

				p1 = ReturnThetaData(file, cycle, theta, channel);
				p2 = ReturnThetaData(file, cycle, theta + theta_1, channel);

				pdV = (p1 + p2) * (next_volume - current_volume);

				if (p1 > 1.0f)
				{
					upper_pumping_imep_tmp += pdV;
				}
				else
				{
					lower_pumping_imep_tmp += pdV;
				}

				crank_angle = ReturnCrankAngle(file, theta, channel);

				for (ca = last_crank_angle + 1; ca < crank_angle; ca++)
				{
					crank_angle_imep[ca] = crank_angle_imep[last_crank_angle];
				}
				
				crank_angle_imep[crank_angle] = (upper_pumping_imep_tmp + lower_pumping_imep_tmp) / sv2;

				current_volume = next_volume;
				last_crank_angle = crank_angle;
			}

			float low_sum = upper_pumping_imep_tmp + lower_pumping_imep_tmp;

			if (cyl_deact == true)
			{
				low_sum /= sv2;

				for (theta = theta_180; theta < theta_540; theta += theta_1)
				{
					crank_angle = ReturnCrankAngle(file, theta, channel);

					for (ca = last_crank_angle + 1; ca < crank_angle; ca++)
					{
						crank_angle_imep[ca] = crank_angle_imep[last_crank_angle];
					}

					crank_angle_imep[crank_angle] = low_sum;

					last_crank_angle = crank_angle;
				}
			}
			else
			{
				current_volume = ReturnThetaDataEx(file, theta_180, channel, volume);

				for (theta = theta_180; theta < theta_540; theta += theta_1)
				{
					next_volume = ReturnThetaDataEx(file, theta + theta_1, channel, volume);

					p1 = ReturnThetaData(file, cycle, theta, channel);
					p2 = ReturnThetaData(file, cycle, theta + theta_1, channel);

					pdV = (p1 + p2) * (next_volume - current_volume);

					gross_imep_tmp += pdV;

					crank_angle = ReturnCrankAngle(file, theta, channel);

					for (ca = last_crank_angle + 1; ca < crank_angle; ca++)
					{
						crank_angle_imep[ca] = crank_angle_imep[last_crank_angle];
					}

					crank_angle_imep[crank_angle] = (low_sum + gross_imep_tmp) / sv2;

					current_volume = next_volume;
					last_crank_angle = crank_angle;
				}
			}

			current_volume = ReturnThetaDataEx(file, theta_540, channel, volume);

			for (theta = theta_540; theta < theta_719; theta += theta_1)
			{
				next_volume = ReturnThetaDataEx(file, theta + theta_1, channel, volume);

				p1 = ReturnThetaData(file, cycle, theta, channel);
				p2 = ReturnThetaData(file, cycle, theta + theta_1, channel);

				pdV = (p1 + p2) * (next_volume - current_volume);

				if (p1 > 1.0f)
				{
					upper_pumping_imep_tmp += pdV;
				}
				else
				{
					lower_pumping_imep_tmp += pdV;
				}

				crank_angle = ReturnCrankAngle(file, theta, channel);

				for (ca = last_crank_angle + 1; ca < crank_angle; ca++)
				{
					crank_angle_imep[ca] = crank_angle_imep[last_crank_angle];
				}

				crank_angle_imep[crank_angle] = (gross_imep_tmp + upper_pumping_imep_tmp + lower_pumping_imep_tmp) / sv2;

				current_volume = next_volume;
				last_crank_angle = crank_angle;
			}

			crank_angle = ReturnCrankAngle(file, theta, channel);

			crank_angle_imep[crank_angle] = (gross_imep_tmp + upper_pumping_imep_tmp + lower_pumping_imep_tmp) / sv2;

			gross_imep_tmp /= sv2;
			upper_pumping_imep_tmp /= sv2;
			lower_pumping_imep_tmp /= sv2;
		}
		else
		{
			for (crank_angle = 0; crank_angle < crank_angle_180; crank_angle++)
			{
				p1 = ReturnCAData(file, cycle, crank_angle, channel);

				pdV = p1 * dVolume[crank_angle];

				if (p1 > 1.0f)
				{
					upper_pumping_imep_tmp += pdV;
				}
				else
				{
					lower_pumping_imep_tmp += pdV;
				}

				crank_angle_imep[crank_angle] = (upper_pumping_imep_tmp + lower_pumping_imep_tmp) / file->engine.swept_volume;
			}

			float low_sum = upper_pumping_imep_tmp + lower_pumping_imep_tmp;

			if (cyl_deact == true)
			{
				low_sum /= file->engine.swept_volume;

				for (crank_angle = crank_angle_180; crank_angle < crank_angle_540; crank_angle++)
				{
					crank_angle_imep[crank_angle] = low_sum;
				}
			}
			else
			{
				for (crank_angle = crank_angle_180; crank_angle < crank_angle_540; crank_angle++)
				{
					p1 = ReturnCAData(file, cycle, crank_angle, channel);

					pdV = p1 * dVolume[crank_angle];

					gross_imep_tmp += pdV;

					crank_angle_imep[crank_angle] = (low_sum + gross_imep_tmp) / file->engine.swept_volume;
				}
			}

			for (crank_angle = crank_angle_540; crank_angle < file->channel_data[channel].samples_per_cycle - 1; crank_angle++)
			{
				p1 = ReturnCAData(file, cycle, crank_angle, channel);

				pdV = p1 * dVolume[crank_angle];

				if (p1 > 1.0f)
				{
					upper_pumping_imep_tmp += pdV;
				}
				else
				{
					lower_pumping_imep_tmp += pdV;
				}

				crank_angle_imep[crank_angle] = (gross_imep_tmp + upper_pumping_imep_tmp + lower_pumping_imep_tmp) / file->engine.swept_volume;
			}

			gross_imep_tmp /= file->engine.swept_volume;
			upper_pumping_imep_tmp /= file->engine.swept_volume;
			lower_pumping_imep_tmp /= file->engine.swept_volume;
		}

		total_angular_indicated_torque_tmp = 0.0f;

		for (crank_angle = 0; crank_angle < file->channel_data[channel].samples_per_cycle; crank_angle++)
		{
			piston_pressure = (ReturnCAData(file, cycle, crank_angle, channel) - crankcase_pressure);

			angular_indicated_torque[crank_angle] = piston_pressure * pressure_to_torque[crank_angle];

			total_angular_indicated_torque_tmp += angular_indicated_torque[crank_angle];
		}

		*total_angular_indicated_torque = total_angular_indicated_torque_tmp;
	}

	pumping_imep_tmp = lower_pumping_imep_tmp + upper_pumping_imep_tmp;
	net_imep_tmp = gross_imep_tmp + pumping_imep_tmp;

	*gross_imep = gross_imep_tmp;
	*upper_pumping_imep = upper_pumping_imep_tmp;
	*lower_pumping_imep = lower_pumping_imep_tmp;
	*pumping_imep = pumping_imep_tmp;
	*net_imep = net_imep_tmp;
	*net_indicated_torque = net_imep_tmp * BAR_TO_PA * file->engine.swept_volume * CUBIC_MM_TO_M / RADIANS_PER_REV / (file->engine.number_of_strokes / 2.0f);
	*gross_indicated_torque = gross_imep_tmp * BAR_TO_PA * file->engine.swept_volume * CUBIC_MM_TO_M / RADIANS_PER_REV / (file->engine.number_of_strokes / 2.0f);
	*cylinder_deactivated = cyl_deact;
}

/* Function: Return_SmoothedPressure_Data
  
   This is quite slow due to the use of Theta data rather than CA.  However, this enables the specified
   crank angles to be used using interpolation of the data.
  
   PARTIAL CYCLE ABSCISSA requires check */

void Prepare_SmoothedPressure_Data(const FileData* file,
	                               const unsigned int method,
	                               const float smoothing_range,
								   const float smoothing_resolution,
								   unsigned int* theta_range,
								   unsigned int* theta_resolution,
								   unsigned int* points_to_average,
								   float** coefficients)
{
	unsigned int resolution;
	unsigned int range;
	unsigned int points;
	unsigned int i;

	if (coefficients == NULL)
	{
		return;
	}

	resolution = DegreesToThetaAbs(file, smoothing_resolution);

	range = DegreesToThetaAbs(file, smoothing_range);

	if (range == 0)
	{
		*theta_range = 0;
		*theta_resolution = 0;
		*points_to_average = 0;

		return;
	}

	if (resolution > range)
	{
		resolution = range;
	}

	if (resolution == 0)
	{
		resolution = 1;
	}

	if (range % 2 != 0)
	{
		range += 1;
	}

	points = ((2 * range) / resolution) + 1;

	*coefficients = (float*)malloc(points * sizeof(float));

	if (*coefficients == NULL)
	{
		logmessage(FATAL, "Memory could not be allocated\n");
	}

	if (method == SMOOTH_SAVITZKY_GOLAY)
	{
		calc_sg_coefficients(points, *coefficients);

		points = 1;
	}
	else
	{
		for (i = 0; i < points; i++)
		{
			(*coefficients)[i] = 1.0f;
		}
	}

	*theta_range = range;
	*theta_resolution = resolution;
	*points_to_average = points;
}

void Complete_SmoothedPressure_Data(float** coefficients)
{
	if (coefficients != NULL)
	{
		if (*coefficients != NULL)
		{
			free(*coefficients);
			*coefficients = NULL;
		}
	}
}

void Return_SmoothedPressure_Data(const FileData* file,
	                              const unsigned int cycle,
								  const unsigned int channel,
								  const unsigned int range,
								  const unsigned int resolution,
								  const unsigned int points_to_average,
								  const float* coefficients,
								  float* data)
{
    int start_theta;
    int finish_theta;
    int theta;
	unsigned int crank_angle;
	unsigned int maximum_theta = MaximumTheta(file,channel);
	unsigned int i;
	float* theta_data = &file->channel_data[channel].data[cycle * file->channel_data[channel].samples_per_cycle];

#ifdef _SAFE_MEMORY
	if (data != NULL)
#endif
	{
		if (range == 0)
		{
			memcpy(data, &file->channel_data[channel].data[cycle*file->channel_data[channel].samples_per_cycle], file->channel_data[channel].samples_per_cycle * sizeof(float));
		}
		else
		{
			for (crank_angle = 0; crank_angle < file->channel_data[channel].samples_per_cycle; crank_angle++)
			{
				theta = ReturnTheta(file, crank_angle, channel);

				start_theta = theta - range;
				finish_theta = theta + range;

				data[crank_angle] = 0.0f;
				i = 0;
				for (theta = start_theta; theta <= finish_theta; theta += resolution)
				{
					if (theta < 0)
					{
						data[crank_angle] += coefficients[i] * ReturnThetaDataEx(file, 0, channel, theta_data);
					}
					else if (theta >= (int)maximum_theta)
					{
						data[crank_angle] += coefficients[i] * ReturnThetaDataEx(file, maximum_theta - 1, channel, theta_data);
					}
					else
					{
						data[crank_angle] += coefficients[i] * ReturnThetaDataEx(file, theta, channel, theta_data);
					}

					i++;
				}

				data[crank_angle] /= (float)points_to_average;
			}
		}
	}
}

/* Function: Return_PKP_Data
  
   PARTIAL CYCLE ABSCISSA requires check
   Check for valid engine speed */

void Return_PKP_Data(const FileData* file,
					 const unsigned int cycle,
					 const unsigned int channel,
					 float* dCA,
					 float* moving_pressure_average,
					 const float pkp_start_angle,
					 const float pkp_finish_angle,
					 const unsigned int knock_integral_type,
					 const float knock_onset_threshold,
					 float* peak_knocking_pressure,
					 float* peak_knocking_pressure_ca,
					 float* knock_integral,
					 float* knock_onset_ca,
					 float* data)
{
	unsigned int crank_angle;
	unsigned int knock_start_ca;
	unsigned int knock_finish_ca;
	unsigned int peak_knocking_pressure_ca_tmp = 0;
	unsigned int knock_onset_ca_tmp = UINT_MAX;
	float pressure;
	float knocking_pressure;
	float peak_knocking_pressure_tmp = 0.0f;
	float knock_integral_tmp = 0.0f;
	float knock_squared_integral_tmp = 0.0f;

	/* Peak Knocking Pressure (SAE 980896) */

	knock_start_ca = DegreesToCrankAngle(file,pkp_start_angle,channel);
	knock_finish_ca = DegreesToCrankAngle(file,pkp_finish_angle,channel);

#ifdef _SAFE_MEMORY
	if ((peak_knocking_pressure != NULL) || (peak_knocking_pressure_ca != NULL))
#endif
	{
		for (crank_angle=knock_start_ca;crank_angle<=knock_finish_ca;crank_angle++)
		{
			pressure = ReturnCAData(file,cycle,crank_angle,channel);

			knocking_pressure = fabsf((pressure - moving_pressure_average[crank_angle]));

			if (knocking_pressure > peak_knocking_pressure_tmp)
			{
				peak_knocking_pressure_tmp = knocking_pressure;
				peak_knocking_pressure_ca_tmp = crank_angle;
			}
			
			knock_integral_tmp += knocking_pressure * dCA[crank_angle];
			knock_squared_integral_tmp += knocking_pressure * knocking_pressure * dCA[crank_angle];

			if ((knock_onset_ca_tmp == UINT_MAX) && (knocking_pressure > knock_onset_threshold))
			{
				knock_onset_ca_tmp = crank_angle;
			}
		}
#ifdef _SAFE_MEMORY
		if (peak_knocking_pressure != NULL)
#endif
		{
			*peak_knocking_pressure = peak_knocking_pressure_tmp;
		}
#ifdef _SAFE_MEMORY
		if (peak_knocking_pressure_ca != NULL)
#endif
		{
			*peak_knocking_pressure_ca = CrankAngleToDegrees(file,peak_knocking_pressure_ca_tmp,channel);
		}
#ifdef _SAFE_MEMORY
		if (knock_integral != NULL)
#endif
		{
			if (knock_integral_type == KNOCK_RECTIFIED_INTEGRAL)
			{
				*knock_integral = knock_integral_tmp;
			}
			else if (knock_integral_type == KNOCK_SQUARED_INTEGRAL)
			{
				*knock_integral = knock_squared_integral_tmp;
			}
			else
			{
				*knock_integral = 0.0f;
			}
		}
	}
#ifdef _SAFE_MEMORY
	if (data != NULL)
#endif
	{
		for (crank_angle=0;crank_angle<knock_start_ca;crank_angle++)
		{
			data[crank_angle] = 0.0f;
		}

		for (crank_angle=knock_start_ca;crank_angle<=knock_finish_ca;crank_angle++)
		{
			pressure = ReturnCAData(file,cycle,crank_angle,channel);
			knocking_pressure = pressure - moving_pressure_average[crank_angle];

			data[crank_angle] = knocking_pressure;
		}

		for (crank_angle=knock_finish_ca+1;crank_angle<file->channel_data[channel].samples_per_cycle;crank_angle++)
		{
			data[crank_angle] = 0.0f;
		}
	}
#ifdef _SAFE_MEMORY
	if (knock_onset_ca != NULL)
#endif
	{
		if (knock_onset_ca_tmp == UINT_MAX)
		{
			*knock_onset_ca = 0.0f;
		}
		else
		{
			*knock_onset_ca = CrankAngleToDegrees(file, knock_onset_ca_tmp, channel);
		}
	}
}

void Prepare_CandD(const FileData* file,
	               const unsigned int channel,
				   float** first_differential,
				   float** second_differential,
				   float** second_filtered_differential)
{
	if (*first_differential != NULL)
	{
		free(*first_differential);
	}
	
	*first_differential = (float*)malloc(file->channel_data[channel].samples_per_cycle * sizeof(float));
	if (*first_differential == NULL)
	{
		if (debug_level >= FATAL)
		{
			logmessage(FATAL,"Memory could not be allocated\n");
		}
	}

	if (*second_differential != NULL)
	{
		free(*second_differential);
	}
	
	*second_differential = (float*)malloc(file->channel_data[channel].samples_per_cycle * sizeof(float));
	if (*second_differential == NULL)
	{
		if (debug_level >= FATAL)
		{
			logmessage(FATAL,"Memory could not be allocated\n");
		}
	}

	if (*second_filtered_differential != NULL)
	{
		free(*second_filtered_differential);
	}
	
	*second_filtered_differential = (float*)malloc(file->channel_data[channel].samples_per_cycle * sizeof(float));
	if (*second_filtered_differential == NULL)
	{
		if (debug_level >= FATAL)
		{
			logmessage(FATAL,"Memory could not be allocated\n");
		}
	}
}

void Complete_CandD(float* first_differential,
	                float* second_differential,
					float* second_filtered_differential)
{
	if (first_differential != NULL)
	{
		free(first_differential);
	}
	
	if (second_differential != NULL)
	{
		free(second_differential);
	}
	
	if (second_filtered_differential != NULL)
	{
		free(second_filtered_differential);
	}
}

void Return_CandD_Data(const FileData* file,
	const unsigned int cycle,
	const unsigned int channel,
	const float* dCA,
	const float cd_start_angle,
	const float cd_finish_angle,
	float* first_differential,
	float* second_differential,
	float* second_filtered_differential,
	float* peak_cd_ki,
	float* peak_cd_ki_ca,
	float* data)
{
	unsigned int crank_angle;
	float peak_cd_ki_tmp = 0.0f;
	float cd_ki;
	unsigned int cd_start_ca;
	unsigned int cd_finish_ca;
	unsigned int peak_cd_ki_ca_tmp = 0;

	/* Checkel & Dale Third Pressure Differential (SAE 860028) */

	if (file->channel_data[channel].samples_per_cycle > 16)
	{
		cd_start_ca = DegreesToCrankAngle(file, cd_start_angle, channel);
		cd_finish_ca = DegreesToCrankAngle(file, cd_finish_angle, channel);

		if (cd_start_ca < 16)
		{
			cd_start_ca = 16;
		}

		if (cd_finish_ca > file->channel_data[channel].samples_per_cycle - 17)
		{
			cd_finish_ca = file->channel_data[channel].samples_per_cycle - 17;
		}

		for (crank_angle = cd_start_ca - 12; crank_angle <= cd_finish_ca + 12; crank_angle++)
		{
			first_differential[crank_angle] = (86.0f * (ReturnCAData(file, cycle, crank_angle - 4, channel) - ReturnCAData(file, cycle, crank_angle + 4, channel))
				+ 142.0f * (ReturnCAData(file, cycle, crank_angle - 3, channel) - ReturnCAData(file, cycle, crank_angle + 3, channel))
				+ 193.0f * (ReturnCAData(file, cycle, crank_angle - 2, channel) - ReturnCAData(file, cycle, crank_angle + 2, channel))
				+ 126.0f * (ReturnCAData(file, cycle, crank_angle - 1, channel) - ReturnCAData(file, cycle, crank_angle + 1, channel)))
				/ (1188.0f * dCA[crank_angle]);
		}

		for (crank_angle = cd_start_ca - 8; crank_angle <= cd_finish_ca + 8; crank_angle++)
		{
			second_differential[crank_angle] = (86.0f * (first_differential[crank_angle - 4] - first_differential[crank_angle + 4])
				+ 142.0f * (first_differential[crank_angle - 3] - first_differential[crank_angle + 3])
				+ 193.0f * (first_differential[crank_angle - 2] - first_differential[crank_angle + 2])
				+ 126.0f * (first_differential[crank_angle - 1] - first_differential[crank_angle + 1]))
				/ (1188.0f * dCA[crank_angle]);
		}

		for (crank_angle = cd_start_ca - 4; crank_angle <= cd_finish_ca + 4; crank_angle++)
		{
			second_filtered_differential[crank_angle] = (2.0f * (second_differential[crank_angle - 4] + second_differential[crank_angle + 4])
				+ 3.0f * (second_differential[crank_angle - 3] + second_differential[crank_angle + 3])
				+ 4.0f * (second_differential[crank_angle - 2] + second_differential[crank_angle + 2])
				+ 5.0f * (second_differential[crank_angle - 1] + second_differential[crank_angle] + second_differential[crank_angle + 1]))
				/ 33.0f;
		}

		/*for (crank_angle = 0; crank_angle < cd_start_ca; crank_angle++)
		{
			data[crank_angle] = 0.0f;
		}*/

		for (crank_angle = cd_start_ca; crank_angle <= cd_finish_ca; crank_angle++)
		{
			/*data[crank_angle] = */
			
			cd_ki = (86.0f * (second_filtered_differential[crank_angle - 4] - second_filtered_differential[crank_angle + 4])
				+ 142.0f * (second_filtered_differential[crank_angle - 3] - second_filtered_differential[crank_angle + 3])
				+ 193.0f * (second_filtered_differential[crank_angle - 2] - second_filtered_differential[crank_angle + 2])
				+ 126.0f * (second_filtered_differential[crank_angle - 1] - second_filtered_differential[crank_angle + 1]))
				/ (1188.0f * dCA[crank_angle]);

			if (/*data[crank_angle]*/ cd_ki > peak_cd_ki_tmp)
			{
				peak_cd_ki_tmp = cd_ki; /* data[crank_angle]; */
				peak_cd_ki_ca_tmp = crank_angle;
			}
		}

		/*for (crank_angle = cd_finish_ca + 1; crank_angle < file->channel_data[channel].samples_per_cycle; crank_angle++)
		{
			data[crank_angle] = 0.0f;
		}*/
	}

#ifdef _SAFE_MEMORY
	if (peak_cd_ki != NULL)
#endif
	{
		*peak_cd_ki = peak_cd_ki_tmp;
	}

#ifdef _SAFE_MEMORY
	if (peak_cd_ki_ca != NULL)
#endif
	{
		*peak_cd_ki_ca = CrankAngleToDegrees(file,peak_cd_ki_ca_tmp,channel);
	}
}

/* Function: Return_Pressure_Data
  
   OK for partial cycle abscissas */

void Return_Pressure_Data(const FileData* file,
	                      const unsigned int cycle,
						  const unsigned int channel,
						  float* data)
{
#ifdef _SAFE_MEMORY
	if (data != NULL)
#endif
	{
		memcpy(data, (void*)&file->channel_data[channel].data[cycle*file->channel_data[channel].samples_per_cycle], file->channel_data[channel].samples_per_cycle*sizeof(float));
	}
}

/* Function: Return_MFB_Data

   PARTIAL CYCLE ABSCISSA requires check */

void Return_MFB_Data(const FileData* file,
	                 const unsigned int cycle,
					 const unsigned int channel,
					 const float* volume,
					 const float* dVolume,
					 const float* dCA,
					 const float mfb_n,
	                 const unsigned int mfb_pressure_method,
					 const float start_of_combustion,
					 const float eeoc,
					 const float* smoothed_pressure,
					 float* max_burn_rate,
					 float* max_burn_rate_ca,
					 float* data)
{
	float delta_pc;
	float delta_pc_star;
	float pressure;
	float old_pressure;
	float old_volume;
	unsigned int start_of_combustion_crank_angle;
	unsigned int eeoc_crank_angle;
	unsigned int crank_angle;
	float max_burn_rate_tmp = 0.0f;
	unsigned int max_burn_rate_ca_tmp = 0;
	float normaliser = 1.0f;
	float burn_rate;

	start_of_combustion_crank_angle = DegreesToCrankAngle(file,start_of_combustion,channel);

	if (start_of_combustion_crank_angle < 1)
	{
		start_of_combustion_crank_angle = 1;
	}

	eeoc_crank_angle = DegreesToCrankAngle(file,eeoc,channel);

    for (crank_angle=0;crank_angle<start_of_combustion_crank_angle;crank_angle++)
    {
    	data[crank_angle] = 0.0f;
	}

	old_pressure = smoothed_pressure[start_of_combustion_crank_angle - 1];
    old_volume = volume[start_of_combustion_crank_angle-1];

	if (mfb_pressure_method == MFB_CATOOL)
	{
		for (crank_angle = start_of_combustion_crank_angle; crank_angle <= eeoc_crank_angle; crank_angle++)
		{
			pressure = smoothed_pressure[crank_angle];

			delta_pc = pressure - old_pressure * powf(old_volume / volume[crank_angle], mfb_n);

			delta_pc_star = delta_pc * old_volume / file->engine.clearance_volume;

			data[crank_angle] = data[crank_angle - 1] + delta_pc_star;

			burn_rate = delta_pc_star * dVolume[crank_angle] / dCA[crank_angle];

			if (burn_rate > max_burn_rate_tmp)
			{
				max_burn_rate_tmp = burn_rate;
				max_burn_rate_ca_tmp = crank_angle;
			}

			old_pressure = pressure;
			old_volume = volume[crank_angle];
		}
	}
	else if (mfb_pressure_method == MFB_KISTLER_RASSWEILER)
	{
		for (crank_angle = start_of_combustion_crank_angle; crank_angle <= eeoc_crank_angle; crank_angle++)
		{
			float res_rad = dCA[crank_angle] * DEG_TO_RAD;
			float ml = 1.0f / (mfb_n - 1.0f) / res_rad;

			pressure = smoothed_pressure[crank_angle];

			delta_pc = pressure - old_pressure * powf(old_volume / volume[crank_angle], mfb_n);

			delta_pc_star = ml * volume[crank_angle] * delta_pc;

			data[crank_angle] = data[crank_angle - 1] + delta_pc_star;

			burn_rate = delta_pc_star * dVolume[crank_angle] / dCA[crank_angle];

			if (burn_rate > max_burn_rate_tmp)
			{
				max_burn_rate_tmp = burn_rate;
				max_burn_rate_ca_tmp = crank_angle;
			}

			old_pressure = pressure;
			old_volume = volume[crank_angle];
		}
	}
	else /* if (mfb_pressure_method == MFB_KISTLER_HOHENBERG) */
	{
		for (crank_angle = start_of_combustion_crank_angle; crank_angle <= eeoc_crank_angle; crank_angle++)
		{
			float res_rad = dCA[crank_angle] * DEG_TO_RAD;
			float ml = 1.0f / (mfb_n - 1.0f) / res_rad;

			float m2 = mfb_n / (mfb_n - 1.0f);
			float d_in = smoothed_pressure[crank_angle + 1] - smoothed_pressure[crank_angle - 1];

			delta_pc_star = ml * volume[crank_angle] * d_in + m2 * smoothed_pressure[crank_angle] * dVolume[crank_angle];

			data[crank_angle] = data[crank_angle - 1] + delta_pc_star;

			burn_rate = delta_pc_star * dVolume[crank_angle] / dCA[crank_angle];

			if (burn_rate > max_burn_rate_tmp)
			{
				max_burn_rate_tmp = burn_rate;
				max_burn_rate_ca_tmp = crank_angle;
			}
		}
	}

	if (data[eeoc_crank_angle] > 0.0f)
	{
		normaliser = 100.0f / data[eeoc_crank_angle];
	}

	for (crank_angle = start_of_combustion_crank_angle; crank_angle <= eeoc_crank_angle; crank_angle++)
	{
		data[crank_angle] = data[crank_angle] * normaliser;
	}

    for (crank_angle=eeoc_crank_angle+1;crank_angle<file->channel_data[channel].samples_per_cycle;crank_angle++)
    {
    	data[crank_angle] = 100.0f;
	}

#ifdef _SAFE_MEMORY
    if (max_burn_rate != NULL)
#endif
    {
    	*max_burn_rate = max_burn_rate_tmp;
	}
#ifdef _SAFE_MEMORY
    if (max_burn_rate_ca != NULL)
#endif
    {
    	*max_burn_rate_ca = CrankAngleToDegrees(file, max_burn_rate_ca_tmp, channel);
	}
}

/* Function: Return_Burn_Angles

   PARTIAL CYCLE ABSCISSA requires check */

void Return_Burn_Angles(const FileData* file,
	                    const unsigned int channel,
						const float* mfb,
						const float start_of_combustion,
						const float eeoc,
						const float knock_onset_ca,
						const bool interpolate_mfb,
						float* burn_angle_1,
						float* burn_angle_2,
						float* burn_angle_5,
						float* burn_angle_10,
						float* burn_angle_20,
						float* burn_angle_25,
						float* burn_angle_50,
						float* burn_angle_75,
						float* burn_angle_80,
						float* burn_angle_90,
						float* burn_angle_95,
						float* burn_angle_98,
						float* burn_angle_99,
						float* burn_duration_0_2,
						float* burn_duration_0_5,
						float* burn_duration_0_10,
						float* burn_duration_0_90,
						float* burn_duration_2_90,
						float* burn_duration_5_90,
						float* burn_duration_10_90,
						float* knock_onset_umfb)
{
	unsigned int start_of_combustion_crank_angle;
	unsigned int eeoc_crank_angle;
	unsigned int crank_angle;
	unsigned int knock_onset_crank_angle;

	float burn_angle_99_tmp = eeoc;
	float burn_angle_98_tmp = eeoc;
	float burn_angle_95_tmp = eeoc;
	float burn_angle_90_tmp = eeoc;
	float burn_angle_80_tmp = eeoc;
	float burn_angle_75_tmp = eeoc;
	float burn_angle_50_tmp = ((start_of_combustion+eeoc)/2.0f);
	float burn_angle_25_tmp = start_of_combustion;
	float burn_angle_20_tmp = start_of_combustion;
	float burn_angle_10_tmp = start_of_combustion;
	float burn_angle_5_tmp = start_of_combustion;
	float burn_angle_2_tmp = start_of_combustion;
	float burn_angle_1_tmp = start_of_combustion;
	float burn_angle_prev;

	start_of_combustion_crank_angle = DegreesToCrankAngle(file,start_of_combustion,channel);
	knock_onset_crank_angle = DegreesToCrankAngle(file, knock_onset_ca, channel);

	if (start_of_combustion_crank_angle == 0)
	{
		/* Because of the crank_angle-1 below */
		start_of_combustion_crank_angle = 1;
	}
	
	eeoc_crank_angle = DegreesToCrankAngle(file,eeoc,channel);

	for (crank_angle=start_of_combustion_crank_angle;crank_angle<=eeoc_crank_angle;crank_angle++)
	{
		if (mfb[crank_angle-1] < 99.0f)
		{
			if (mfb[crank_angle] >= 99.0f)
			{
				burn_angle_99_tmp = CrankAngleToDegrees(file, crank_angle, channel);
				
				if (interpolate_mfb == true)
				{
					burn_angle_prev = CrankAngleToDegrees(file, crank_angle - 1, channel);

					burn_angle_99_tmp = burn_angle_prev + (burn_angle_99_tmp - burn_angle_prev) * ((99.0f - mfb[crank_angle - 1]) / (mfb[crank_angle] - mfb[crank_angle - 1]));
				}
			}
			else if (mfb[crank_angle-1] < 98.0f)
			{
				if (mfb[crank_angle] >= 98.0f)
				{
					burn_angle_98_tmp = CrankAngleToDegrees(file,crank_angle,channel);

					if (interpolate_mfb == true)
					{
						burn_angle_prev = CrankAngleToDegrees(file, crank_angle - 1, channel);

						burn_angle_98_tmp = burn_angle_prev + (burn_angle_98_tmp - burn_angle_prev) * ((98.0f - mfb[crank_angle - 1]) / (mfb[crank_angle] - mfb[crank_angle - 1]));
					}
				}
				else if (mfb[crank_angle-1] < 95.0f)
				{
					if (mfb[crank_angle] >= 95.0f)
					{
						burn_angle_95_tmp = CrankAngleToDegrees(file,crank_angle,channel);

						if (interpolate_mfb == true)
						{
							burn_angle_prev = CrankAngleToDegrees(file, crank_angle - 1, channel);

							burn_angle_95_tmp = burn_angle_prev + (burn_angle_95_tmp - burn_angle_prev) * ((95.0f - mfb[crank_angle - 1]) / (mfb[crank_angle] - mfb[crank_angle - 1]));
						}
					}
					else if (mfb[crank_angle-1] < 90.0f)
					{
						if (mfb[crank_angle] >= 90.0f)
						{
							burn_angle_90_tmp = CrankAngleToDegrees(file,crank_angle,channel);

							if (interpolate_mfb == true)
							{
								burn_angle_prev = CrankAngleToDegrees(file, crank_angle - 1, channel);

								burn_angle_90_tmp = burn_angle_prev + (burn_angle_90_tmp - burn_angle_prev) * ((90.0f - mfb[crank_angle - 1]) / (mfb[crank_angle] - mfb[crank_angle - 1]));
							}
						}
						else if (mfb[crank_angle-1] < 80.0f)
						{
							if (mfb[crank_angle] >= 80.0f)
							{
								burn_angle_80_tmp = CrankAngleToDegrees(file,crank_angle,channel);

								if (interpolate_mfb == true)
								{
									burn_angle_prev = CrankAngleToDegrees(file, crank_angle - 1, channel);

									burn_angle_80_tmp = burn_angle_prev + (burn_angle_80_tmp - burn_angle_prev) * ((80.0f - mfb[crank_angle - 1]) / (mfb[crank_angle] - mfb[crank_angle - 1]));
								}
							}
							else if (mfb[crank_angle-1] < 75.0f)
							{
								if (mfb[crank_angle] >= 75.0f)
								{
									burn_angle_75_tmp = CrankAngleToDegrees(file,crank_angle,channel);

									if (interpolate_mfb == true)
									{
										burn_angle_prev = CrankAngleToDegrees(file, crank_angle - 1, channel);

										burn_angle_75_tmp = burn_angle_prev + (burn_angle_75_tmp - burn_angle_prev) * ((75.0f - mfb[crank_angle - 1]) / (mfb[crank_angle] - mfb[crank_angle - 1]));
									}
								}
								else if (mfb[crank_angle-1] < 50.0f)
								{
									if (mfb[crank_angle] >= 50.0f)
									{
										burn_angle_50_tmp = CrankAngleToDegrees(file,crank_angle,channel);

										if (interpolate_mfb == true)
										{
											burn_angle_prev = CrankAngleToDegrees(file, crank_angle - 1, channel);

											burn_angle_50_tmp = burn_angle_prev + (burn_angle_50_tmp - burn_angle_prev) * ((50.0f - mfb[crank_angle - 1]) / (mfb[crank_angle] - mfb[crank_angle - 1]));
										}
									}
									else if (mfb[crank_angle-1] < 25.0f)
									{
										if (mfb[crank_angle] >= 25.0f)
										{
											burn_angle_25_tmp = CrankAngleToDegrees(file,crank_angle,channel);

											if (interpolate_mfb == true)
											{
												burn_angle_prev = CrankAngleToDegrees(file, crank_angle - 1, channel);

												burn_angle_25_tmp = burn_angle_prev + (burn_angle_25_tmp - burn_angle_prev) * ((25.0f - mfb[crank_angle - 1]) / (mfb[crank_angle] - mfb[crank_angle - 1]));
											}
										}
										else if (mfb[crank_angle-1] < 20.0f)
										{
											if (mfb[crank_angle] >= 20.0f)
											{
												burn_angle_20_tmp = CrankAngleToDegrees(file,crank_angle,channel);

												if (interpolate_mfb == true)
												{
													burn_angle_prev = CrankAngleToDegrees(file, crank_angle - 1, channel);

													burn_angle_20_tmp = burn_angle_prev + (burn_angle_20_tmp - burn_angle_prev) * ((20.0f - mfb[crank_angle - 1]) / (mfb[crank_angle] - mfb[crank_angle - 1]));
												}
											}
											else if (mfb[crank_angle-1] < 10.0f)
											{
												if (mfb[crank_angle] >= 10.0f)
												{
													burn_angle_10_tmp = CrankAngleToDegrees(file,crank_angle,channel);

													if (interpolate_mfb == true)
													{
														burn_angle_prev = CrankAngleToDegrees(file, crank_angle - 1, channel);

														burn_angle_10_tmp = burn_angle_prev + (burn_angle_10_tmp - burn_angle_prev) * ((10.0f - mfb[crank_angle - 1]) / (mfb[crank_angle] - mfb[crank_angle - 1]));
													}
												}
												else if (mfb[crank_angle-1] < 5.0f)
												{
													if (mfb[crank_angle] >= 5.0f)
													{
														burn_angle_5_tmp = CrankAngleToDegrees(file,crank_angle,channel);

														if (interpolate_mfb == true)
														{
															burn_angle_prev = CrankAngleToDegrees(file, crank_angle - 1, channel);

															burn_angle_5_tmp = burn_angle_prev + (burn_angle_5_tmp - burn_angle_prev) * ((5.0f - mfb[crank_angle - 1]) / (mfb[crank_angle] - mfb[crank_angle - 1]));
														}
													}
													else if (mfb[crank_angle-1] < 2.0f)
													{
														if (mfb[crank_angle] >= 2.0f)
														{
															burn_angle_2_tmp = CrankAngleToDegrees(file,crank_angle,channel);

															if (interpolate_mfb == true)
															{
																burn_angle_prev = CrankAngleToDegrees(file, crank_angle - 1, channel);

																burn_angle_2_tmp = burn_angle_prev + (burn_angle_2_tmp - burn_angle_prev) * ((2.0f - mfb[crank_angle - 1]) / (mfb[crank_angle] - mfb[crank_angle - 1]));
															}
														}
														else if (mfb[crank_angle-1] < 1.0f)
														{
															if (mfb[crank_angle] >= 1.0f)
															{
																burn_angle_1_tmp = CrankAngleToDegrees(file,crank_angle,channel);

																if (interpolate_mfb == true)
																{
																	burn_angle_prev = CrankAngleToDegrees(file, crank_angle - 1, channel);

																	burn_angle_1_tmp = burn_angle_prev + (burn_angle_1_tmp - burn_angle_prev) * ((1.0f - mfb[crank_angle - 1]) / (mfb[crank_angle] - mfb[crank_angle - 1]));
																}
															}
														}
														else
														{
															/* Do Nothing */
														}
													}
													else
													{
														/* Do Nothing */
													}
												}
												else
												{
													/* Do Nothing */
												}
											}
											else
											{
												/* Do Nothing */
											}
										}
										else
										{
											/* Do Nothing */
										}
									}
									else
									{
										/* Do Nothing */
									}
								}
								else
								{
									/* Do Nothing */
								}
							}
							else
							{
								/* Do Nothing */
							}
						}
						else
						{
							/* Do Nothing */
						}
					}
					else
					{
					/* Do Nothing */
					}
				}
				else
				{
					/* Do Nothing */
				}
			}
			else
			{
				/* Do Nothing */
			}
		}
	}

#ifdef _SAFE_MEMORY
	if (knock_onset_umfb != NULL)
#endif
	{
		*knock_onset_umfb = 100.0f - mfb[knock_onset_crank_angle];
	}

#ifdef _SAFE_MEMORY
	if (burn_angle_1 != NULL)
#endif
	{
		*burn_angle_1 = burn_angle_1_tmp;
	}
#ifdef _SAFE_MEMORY
	if (burn_angle_2 != NULL)
#endif
	{
		*burn_angle_2 = burn_angle_2_tmp;
	}
#ifdef _SAFE_MEMORY
	if (burn_angle_5 != NULL)
#endif
	{
		*burn_angle_5 = burn_angle_5_tmp;
	}
#ifdef _SAFE_MEMORY
	if (burn_angle_10 != NULL)
#endif
	{
		*burn_angle_10 = burn_angle_10_tmp;
	}
#ifdef _SAFE_MEMORY
	if (burn_angle_20 != NULL)
#endif
	{
		*burn_angle_20 = burn_angle_20_tmp;
	}
#ifdef _SAFE_MEMORY
	if (burn_angle_25 != NULL)
#endif
	{
		*burn_angle_25 = burn_angle_25_tmp;
	}
#ifdef _SAFE_MEMORY
	if (burn_angle_50 != NULL)
#endif
	{
		*burn_angle_50 = burn_angle_50_tmp;
	}
#ifdef _SAFE_MEMORY
	if (burn_angle_75 != NULL)
#endif
	{
		*burn_angle_75 = burn_angle_75_tmp;
	}
#ifdef _SAFE_MEMORY
	if (burn_angle_80 != NULL)
#endif
	{
		*burn_angle_80 = burn_angle_80_tmp;
	}
#ifdef _SAFE_MEMORY
	if (burn_angle_90 != NULL)
#endif
	{
		*burn_angle_90 = burn_angle_90_tmp;
	}
#ifdef _SAFE_MEMORY
	if (burn_angle_95 != NULL)
#endif
	{
		*burn_angle_95 = burn_angle_95_tmp;
	}
#ifdef _SAFE_MEMORY
	if (burn_angle_98 != NULL)
#endif
	{
		*burn_angle_98 = burn_angle_98_tmp;
	}
#ifdef _SAFE_MEMORY
	if (burn_angle_99 != NULL)
#endif
	{
		*burn_angle_99 = burn_angle_99_tmp;
	}
#ifdef _SAFE_MEMORY
	if (burn_duration_0_2 != NULL)
#endif
	{
		*burn_duration_0_2 = burn_angle_2_tmp - start_of_combustion;
	}
#ifdef _SAFE_MEMORY
	if (burn_duration_0_5 != NULL)
#endif
	{
		*burn_duration_0_5 = burn_angle_5_tmp - start_of_combustion;
	}
#ifdef _SAFE_MEMORY
	if (burn_duration_0_10 != NULL)
#endif
	{
		*burn_duration_0_10 = burn_angle_10_tmp - start_of_combustion;
	}
#ifdef _SAFE_MEMORY
	if (burn_duration_0_90 != NULL)
#endif
	{
		*burn_duration_0_90 = burn_angle_90_tmp - start_of_combustion;
	}
#ifdef _SAFE_MEMORY
	if (burn_duration_2_90 != NULL)
#endif
	{
		*burn_duration_2_90 = burn_angle_90_tmp - burn_angle_2_tmp;
	}
#ifdef _SAFE_MEMORY
	if (burn_duration_5_90 != NULL)
#endif
	{
		*burn_duration_5_90 = burn_angle_90_tmp - burn_angle_5_tmp;
	}
#ifdef _SAFE_MEMORY
	if (burn_duration_10_90 != NULL)
#endif
	{
		*burn_duration_10_90 = burn_angle_90_tmp - burn_angle_10_tmp;
	}
}

/* Function: Return_Heat_Release_Data
  
   PARTIAL CYCLE ABSCISSA requires check */

void Return_Heat_Release_Data(const FileData* file,
							  const unsigned int channel,
							  const unsigned int cycle,
							  float* dCA,
							  float* volume,
							  float* dVolume,
							  float* heat_transfer_area,
							  const float* mean_gas_temp,
							  const float* gamma,
							  const float* pressure,
							  const float t_ivc,
							  const float t_wall,
							  const float R,
							  const float start_of_combustion,
							  const float eeoc,
							  const float engine_speed,
							  const float poly_comp,
							  const float poly_exp,
							  const unsigned int heat_release_model,
							  const unsigned int heat_transfer_model,
							  const unsigned int heat_release_window,
							  const float annand_a,
							  float* max_heat_release_rate,
							  float* max_heat_release_rate_ca,
							  float* total_heat_release,
							  float* net_heat_release_rate,
							  float* gross_heat_release_rate,
							  float* h)
{
	float mean_piston_speed;
	float motored_pressure;
    float pressure_ivc;
	float volume_ivc;
	float c1;
	float c2;
	float u;
	float gas_density;
	float dynamic_viscosity;
	float cp;
	float thermal_conductivity;
    float reynolds_number;
	float annand_c;
	float TbyPVref;
	float first_law_gamma;
	float n;
	float dAbydTheta;
    float calc_h = 0.0f;
	float calc_net = 0.0f;
	float calc_gross = 0.0f;
    float max_heat_release_rate_tmp = 0.0f;
    unsigned int max_heat_release_rate_ca_tmp = 0;
    float total_heat_release_tmp = 0.0f;
	unsigned int crank_angle_360;
	unsigned int crank_angle_start;
	unsigned int crank_angle_finish;
	unsigned int crank_angle;
	unsigned int crank_angle_soc;
	unsigned int crank_angle_eeoc;
	unsigned int crank_angle_ivc;
	unsigned int crank_angle_evo;

	crank_angle_soc = DegreesToCrankAngle(file, start_of_combustion, channel);

	if (crank_angle_soc < 1)
	{
		crank_angle_soc = 1;
	}

	crank_angle_eeoc = DegreesToCrankAngle(file, eeoc, channel);

	if (crank_angle_eeoc < 1)
	{
		crank_angle_eeoc = 1;
	}

	crank_angle_ivc = DegreesToCrankAngle(file, file->engine.ivc_angle, channel);
	crank_angle_evo = DegreesToCrankAngle(file, file->engine.evo_angle, channel);

	if (crank_angle_ivc < 1)
	{
		crank_angle_ivc = 1;
	}

	crank_angle_360 = DegreesToCrankAngle(file, 0.0f, channel);

	pressure_ivc = pressure[crank_angle_ivc];
	volume_ivc = volume[crank_angle_ivc];

	if (heat_release_window == HR_WINDOW_IVC_EVO)
	{
		crank_angle_start = crank_angle_ivc;
		crank_angle_finish = crank_angle_evo;
	}
	else /* if (heat_release_window == HR_WINDOW_SOC_EEOC) */
	{
		crank_angle_start = crank_angle_soc;
		crank_angle_finish = crank_angle_eeoc;
	}

#ifdef _SAFE_MEMORY
    if (h != NULL)
#endif
    {
        for (crank_angle=0;crank_angle<crank_angle_ivc;crank_angle++)			/* was SOC */
        {
		    h[crank_angle] = 0.0f;
		}
	}
#ifdef _SAFE_MEMORY
    if (net_heat_release_rate != NULL)
#endif
    {
        for (crank_angle=0;crank_angle<crank_angle_ivc;crank_angle++)			/* was SOC */
        {
		    net_heat_release_rate[crank_angle] = 0.0f;
		}
	}
#ifdef _SAFE_MEMORY
    if (gross_heat_release_rate != NULL)
#endif
    {
        for (crank_angle=0;crank_angle<crank_angle_ivc;crank_angle++)			/* was SOC */
        {
		    gross_heat_release_rate[crank_angle] = 0.0f;
		}
	}

	/* Heat Release */
	switch (heat_release_model)
	{
	default:
	case HR_FIRSTLAW:		/* Single-Zone First Law Heat Release Model */
	{
		if ((pressure_ivc > 0.0f) && (volume_ivc > 0.0f))
		{
			TbyPVref = t_ivc / (pressure_ivc * BAR_TO_PA) / (volume_ivc * CUBIC_MM_TO_M);
		}
		else
		{
			TbyPVref = 0.0f;
		}

		for (crank_angle = crank_angle_ivc; crank_angle < crank_angle_evo; crank_angle++)			/* was SOC/<= EEOC */
		{
			mean_piston_speed = file->engine.stroke * LEN_MM_TO_M * engine_speed / 30.0f;

			/* Calculate Heat Transfer Coefficients (h = W / m^2 . K ) */

			switch (heat_transfer_model)
			{
			case HT_ANNAND:			/* Annand, I.Mech.E. Procedings Vol. 177 No. 36, 1963 */
			{
				if ((mean_gas_temp[crank_angle] > 0.0f) && (gamma[crank_angle] > 0.0f))
				{
					gas_density = pressure[crank_angle] * BAR_TO_PA / R / mean_gas_temp[crank_angle];
					dynamic_viscosity = 4.702E-7f * powf(mean_gas_temp[crank_angle], 0.645f);			/* Annand, I.Mech.E., Vol. 177 No. 36, 1963 */
					cp = R / (1.0f - 1.0f / gamma[crank_angle]);

					thermal_conductivity = cp * dynamic_viscosity / 0.7f;								/* Annand, I.Mech.E., Vol. 177 No. 36, 1963 */
					reynolds_number = mean_piston_speed * file->engine.bore * LEN_MM_TO_M * gas_density / dynamic_viscosity;

					if ((file->engine.type == ENGINE_SI) || (file->engine.type == ENGINE_DISI))
					{
						annand_c = 0.075f * 5.6E-8f;
					}
					else
					{
						annand_c = 0.576f * 5.6E-8f;
					}

					calc_h = annand_a * thermal_conductivity / (file->engine.bore * LEN_MM_TO_M) * powf((float)reynolds_number, 0.7f) + annand_c * (powf((float)mean_gas_temp[crank_angle], 4.0f) - powf((float)t_wall, 4.0f)) / (mean_gas_temp[crank_angle] - t_wall);
				}
				else
				{
					calc_h = 0.0f;
				}
				
				break;
			}
			case HT_WOSCHNI:		/* Woschni, SAE 670931 */
			{
				if (mean_gas_temp[crank_angle] > 0.0f)
				{
					motored_pressure = pressure_ivc * powf((float)(volume_ivc / volume[crank_angle]), 1.3f);

					c1 = 2.28f;

					if (file->engine.type == ENGINE_CI_IDI)
					{
						c2 = 6.22E-3f;
					}
					else
					{
						c2 = 3.24E-3f;
					}

					u = c1 * mean_piston_speed + c2 * TbyPVref * file->engine.swept_volume * CUBIC_MM_TO_M * (pressure[crank_angle] - motored_pressure) * BAR_TO_KPA;

					calc_h = 3.26f * powf((float)file->engine.bore * LEN_MM_TO_M, -0.2f) * powf(pressure[crank_angle] * BAR_TO_KPA, 0.8f) * powf((float)mean_gas_temp[crank_angle], -0.55f) * powf((float)u, 0.8f);
				}
				else
				{
					calc_h = 0.0f;
				}

				break;
			}
			case HT_WOSCHNI_GT:		/* Heywood, "Internal Combustion Engine Fundamentals", section 12.4.3 */
			{
				if (mean_gas_temp[crank_angle] > 0.0f)
				{
					motored_pressure = pressure_ivc * powf((float)(volume_ivc / volume[crank_angle]), 1.3f);

					if (crank_angle < crank_angle_ivc)
					{
						c1 = 6.18f;
						c2 = 0.0f;
					}
					else if (crank_angle < crank_angle_soc)
					{
						c1 = 2.28f;
						c2 = 0.0f;
					}
					else
					{
						c1 = 2.28f;
						c2 = 3.24E-3f;
					}

					u = c1 * mean_piston_speed + c2 * TbyPVref * file->engine.swept_volume * CUBIC_MM_TO_M * (pressure[crank_angle] - motored_pressure) * BAR_TO_KPA;

					calc_h = 3.26f * powf((float)file->engine.bore * LEN_MM_TO_M, -0.2f) * powf(pressure[crank_angle] * BAR_TO_KPA, 0.8f) * powf((float)mean_gas_temp[crank_angle], -0.55f) * powf((float)u, 0.8f);
				}
				else
				{
					calc_h = 0.0f;
				}

				break;
			}
			case HT_HOHENBERG:		/* Hohenberg, SAE 790825 */
			{
				if (mean_gas_temp[crank_angle] > 0.0f)
				{
					calc_h = 129.8f * powf((float)volume[crank_angle] * CUBIC_MM_TO_M, -0.06f) * powf(pressure[crank_angle], 0.8f) * powf((float)mean_gas_temp[crank_angle], -0.4f) * powf((float)mean_piston_speed + 1.4f, 0.8f);
				}
				else
				{
					calc_h = 0.0f;
				}

				break;
			}
			case HT_EICHELBERG:		/* Eichelberg */
			{
				calc_h = 2.43f * powf((float)mean_piston_speed, (1.0f / 3.0f)) * powf((float)(pressure[crank_angle] * mean_gas_temp[crank_angle]), 0.5f);
				break;
			}
			case HT_NUSSELT:        /* Nusselt */
			{
				calc_h = 0.99f * (1.0f + 1.24f * mean_piston_speed) * powf((float)(pressure[crank_angle] * pressure[crank_angle] * mean_gas_temp[crank_angle]), (1.0f / 3.0f));
				break;
			}
			case HT_BRILING:       /* Briling */
			{
				calc_h = 0.99f * (3.5f + 0.185f * mean_piston_speed) * powf((float)(pressure[crank_angle] * pressure[crank_angle] * mean_gas_temp[crank_angle]), (1.0f / 3.0f));
				break;
			}
			case HT_NONE:			/* None */
			default:
			{
				calc_h = 0.0f;
				break;
			}
			}

			dAbydTheta = (heat_transfer_area[crank_angle + 1] - heat_transfer_area[crank_angle - 1]) / (dCA[crank_angle - 1] + dCA[crank_angle]);

			calc_net = gamma[crank_angle] /
				(gamma[crank_angle] - 1.0f) *
				pressure[crank_angle] * BAR_TO_PA *
				dVolume[crank_angle] / dCA[crank_angle] * CUBIC_MM_TO_M +
				1.0f / (gamma[crank_angle] - 1.0f) *
				volume[crank_angle] * CUBIC_MM_TO_M *
				(pressure[crank_angle + 1] - pressure[crank_angle]) * BAR_TO_PA / dCA[crank_angle];		/* J / deg */

			calc_gross = calc_net +
				calc_h *
				(mean_gas_temp[crank_angle] - t_wall) *
				dAbydTheta * SQ_MM_TO_M;																/* J / deg */

#ifdef _SAFE_MEMORY
			if (h != NULL)
#endif
			{
				h[crank_angle] = calc_h;
			}
#ifdef _SAFE_MEMORY
			if (net_heat_release_rate != NULL)
#endif
			{
				net_heat_release_rate[crank_angle] = calc_net;
			}
#ifdef _SAFE_MEMORY       
			if (gross_heat_release_rate != NULL)
#endif
			{
				gross_heat_release_rate[crank_angle] = calc_gross;
			}

			if ((crank_angle >= crank_angle_start) && (crank_angle < crank_angle_finish))
			{
				if (calc_net > max_heat_release_rate_tmp)
				{
					max_heat_release_rate_tmp = calc_net;
					max_heat_release_rate_ca_tmp = crank_angle;
				}

				total_heat_release_tmp += calc_net * dCA[crank_angle];
			}
		}

		break;
	}
	case HR_AVL_THERMO1:	/* Ref: AVL 670 Indimaster / AVL 660 Indistation - Operating Instructions, 1997 */
	{
		calc_h = 0.0f;

		float step = 1.0f;

		unsigned int theta_step = DegreesToThetaAbs(file, step);
		unsigned int theta;
		float gamma_i;
		float p_i;
		float p_i_plus_1;
		float p_i_minus_1;
		float v_i;
		float v_i_plus_1;
		float v_i_minus_1;

		for (crank_angle = crank_angle_ivc; crank_angle < crank_angle_evo; crank_angle++)
		{
			theta = ReturnTheta(file, crank_angle, channel);

			gamma_i = ReturnThetaDataEx(file, theta, channel, gamma);
			p_i = ReturnThetaData(file, cycle, theta, channel);
			p_i_plus_1 = ReturnThetaData(file, cycle, theta + theta_step, channel);
			p_i_minus_1 = ReturnThetaData(file, cycle, theta - theta_step, channel);
			v_i = ReturnThetaDataEx(file, theta, channel, volume) / file->engine.swept_volume;
			v_i_plus_1 = ReturnThetaDataEx(file, theta + theta_step, channel, volume) / file->engine.swept_volume;
			v_i_minus_1 = ReturnThetaDataEx(file, theta - theta_step, channel, volume) / file->engine.swept_volume;

			/* C = 100 = JOULES_TO_KJ * BAR_TO_PA */

			calc_net = JOULES_TO_KJ / (gamma_i - 1.0f) *
				(gamma_i * p_i * BAR_TO_PA * (v_i_plus_1 - v_i_minus_1) +
					v_i * (p_i_plus_1 - p_i_minus_1) * BAR_TO_PA) / (2.0f * step);				/* kJ / m^3 / deg */

			calc_gross = calc_net * file->engine.swept_volume * CUBIC_MM_TO_M / JOULES_TO_KJ;		/* J / deg */

			h[crank_angle] = calc_h;
			net_heat_release_rate[crank_angle] = calc_net;
			gross_heat_release_rate[crank_angle] = calc_gross;

			if ((crank_angle >= crank_angle_start) && (crank_angle < crank_angle_finish))
			{
				if (calc_net > max_heat_release_rate_tmp)
				{
					max_heat_release_rate_tmp = calc_net;
					max_heat_release_rate_ca_tmp = crank_angle;
				}

				total_heat_release_tmp += calc_net * dCA[crank_angle];
			}
		}

		break;
	}
	case HR_POLYFIRST:		/* Polytropic Index First Law (SAE 981052 , SAE 1999-01-0187)
					            No correction of polytropic indices over +/- 50 deg TDC */
	{
		calc_h = 0.0f;

		for (crank_angle = crank_angle_ivc; crank_angle < crank_angle_evo; crank_angle++)			/* was SOC/<= EEOC */
		{
			if ((file->engine.type == ENGINE_CI_DI) || (file->engine.type == ENGINE_CI_IDI))
			{
				first_law_gamma = 1.35f - 6.0E-5f * mean_gas_temp[crank_angle] + 1.0E-8f * mean_gas_temp[crank_angle] * mean_gas_temp[crank_angle];
			}
			else
			{
				first_law_gamma = 1.338f - 6.0E-5f * mean_gas_temp[crank_angle] + 1.0E-8f * mean_gas_temp[crank_angle] * mean_gas_temp[crank_angle];
			}

			if (crank_angle < crank_angle_360)
			{
				n = poly_comp;
			}
			else
			{
				n = poly_exp;
			}

			calc_gross = (volume[crank_angle] + volume[crank_angle - 1]) / dCA[crank_angle - 1] / 2.0f * CUBIC_MM_TO_M *
				(pressure[crank_angle] * BAR_TO_PA -
					pressure[crank_angle - 1] * BAR_TO_PA * powf(volume[crank_angle - 1] / volume[crank_angle], n)) / (first_law_gamma - 1.0f);

			calc_net = (volume[crank_angle] + volume[crank_angle - 1]) / dCA[crank_angle - 1] / 2.0f * CUBIC_MM_TO_M *
				(pressure[crank_angle] * BAR_TO_PA -
					pressure[crank_angle - 1] * BAR_TO_PA * powf(volume[crank_angle - 1] / volume[crank_angle], first_law_gamma)) / (first_law_gamma - 1.0f);

#ifdef _SAFE_MEMORY
			if (h != NULL)
#endif
			{
				h[crank_angle] = calc_h;
			}
#ifdef _SAFE_MEMORY
			if (net_heat_release_rate != NULL)
#endif
			{
				net_heat_release_rate[crank_angle] = calc_net;
			}
#ifdef _SAFE_MEMORY        
			if (gross_heat_release_rate != NULL)
#endif
			{
				gross_heat_release_rate[crank_angle] = calc_gross;
			}

			if ((crank_angle >= crank_angle_start) && (crank_angle < crank_angle_finish))
			{
				if (calc_net > max_heat_release_rate_tmp)
				{
					max_heat_release_rate_tmp = calc_net;
					max_heat_release_rate_ca_tmp = crank_angle;
				}

				total_heat_release_tmp += calc_net * dCA[crank_angle];
			}
		}

        break;
	}
	}

#ifdef _SAFE_MEMORY
    if (h != NULL)
#endif
    {
        for (crank_angle=crank_angle_evo;crank_angle<file->channel_data[channel].samples_per_cycle;crank_angle++)		/* was EEOC + 1 */
        {
		    h[crank_angle] = 0.0f;
		}
	}
#ifdef _SAFE_MEMORY
    if (net_heat_release_rate != NULL)
#endif
    {
        for (crank_angle=crank_angle_evo;crank_angle<file->channel_data[channel].samples_per_cycle;crank_angle++)		/* was EEOC + 1 */
        {
		    net_heat_release_rate[crank_angle] = 0.0f;
		}
	}
#ifdef _SAFE_MEMORY
    if (gross_heat_release_rate != NULL)
#endif
    {
        for (crank_angle=crank_angle_evo;crank_angle<file->channel_data[channel].samples_per_cycle;crank_angle++)		/* was EEOC + 1 */
        {
		    gross_heat_release_rate[crank_angle] = 0.0f;
		}
	}
#ifdef _SAFE_MEMORY
	if (max_heat_release_rate != NULL)
#endif
	{
		*max_heat_release_rate = max_heat_release_rate_tmp;
	}
#ifdef _SAFE_MEMORY
	if (max_heat_release_rate_ca != NULL)
#endif
	{
		*max_heat_release_rate_ca = CrankAngleToDegrees(file, max_heat_release_rate_ca_tmp, channel);
	}
#ifdef _SAFE_MEMORY
	if (total_heat_release != NULL)
#endif
	{
		*total_heat_release = total_heat_release_tmp;
	}
}

/* Function: Return_Motored_Pressure_Data
   Small speed increase possible by calculating (Vivc/V)^1.3 table for use on all cycles
   PARTIAL CYCLE ABSCISSA requires check */

void Return_Motored_Pressure_Data(const FileData* file,
								  const unsigned int channel,
								  const unsigned int method,
								  const float poly_comp_finish_angle,
								  const float poly_comp_index,
								  const float* pressure,
								  const float* volume,
								  float* data)
{
	unsigned int crank_angle;
	unsigned int ivc_crank_angle;
	unsigned int evo_crank_angle;
	unsigned int start_crank_angle;
	float calc_pressure;
	float ivc_pressure;
	float actual_pressure;
	float n;
	float ivc_ca;
	float evo_ca;
	float start_ca;

	if ((method & MOTORED_POLY_COMP) > 0)
	{
		n = poly_comp_index;
	}
	else /*if (method & MOTORED_FIXED_N > 0)*/
	{
		n = 1.3f;
	}

	if ((method & MOTORED_POLY_FINISH) > 0)
	{
		ivc_ca = poly_comp_finish_angle;
		evo_ca = file->engine.evo_angle;
	}
	else /* if ((method & MOTORED_IVC) > 0) */
	{
		ivc_ca = file->engine.ivc_angle;
		evo_ca = file->engine.evo_angle;
	}

	if ((method & MOTORED_BDC) > 0)
	{
		start_ca = -180.0f;
	}
	else
	{
		start_ca = ivc_ca;
	}

	
	if (data != NULL)
	{
        ivc_crank_angle = DegreesToCrankAngle(file,ivc_ca,channel);
        evo_crank_angle = DegreesToCrankAngle(file,evo_ca,channel);
		start_crank_angle = DegreesToCrankAngle(file,start_ca, channel);

		ivc_pressure = pressure[ivc_crank_angle];

    	for (crank_angle=0;crank_angle<start_crank_angle;crank_angle++)
    	{
    		data[crank_angle] = pressure[crank_angle];
		}

    	for (crank_angle=start_crank_angle;crank_angle<=evo_crank_angle;crank_angle++)
    	{	
			data[crank_angle] = ivc_pressure*powf((float)(volume[ivc_crank_angle] / volume[crank_angle]), n);
		}

		calc_pressure = data[evo_crank_angle];

    	for (crank_angle=evo_crank_angle+1;crank_angle<file->channel_data[channel].samples_per_cycle;crank_angle++)
    	{
    		actual_pressure = pressure[crank_angle];

    		if (calc_pressure < actual_pressure)
    		{
    			data[crank_angle] = calc_pressure;
			}
			else
			{
				data[crank_angle] = actual_pressure;
			}
    	}
    }
}

void Return_Digital_Channel(const FileData* file,
                            const unsigned int cycle,
							const unsigned int channel,
							const unsigned int type,
							const float latch_high,
							const float latch_low,
							const float angle_filter,
							float* data)
{
	unsigned int crank_angle;
	unsigned int samples_per_cycle = file->channel_data[channel].samples_per_cycle;
	float threshold;
	float value;
	float new_value;
	Statistics stats;
	unsigned int histogram[10];
	unsigned int bin;
	unsigned int sample_threshold;
	unsigned int number_of_samples;
	bool found_sample;
	unsigned int low_bin;
	unsigned int high_bin;
	float lower_threshold;
	float upper_threshold;
	bool looking_for_downedge;
	unsigned int last_edge = 0;
	unsigned int crank_angle_2;
	float factor;

	if (type == DIGITIZE_AUTO)
	{
		calculate_single_cycle_stats(&stats, &file->channel_data[channel].data[cycle * samples_per_cycle], samples_per_cycle, STATS_RATIO);

		if (stats.range < 0.3f)
		{
			if (stats.mean < 1.0f)
			{
				value = 0.0f;
			}
			else
			{
				value = 1.0f;
			}

			for (crank_angle = 0; crank_angle < samples_per_cycle; crank_angle++)
			{
				data[crank_angle] = value;
			}
		}
		else
		{
			for (bin = 0; bin < 10; bin++)
			{
				histogram[bin] = 0;
			}

			factor = 10.0f / stats.range;

			for (crank_angle = 0; crank_angle < samples_per_cycle; crank_angle++)
			{
				value = ReturnCAData(file, cycle, crank_angle, channel);

				bin = (unsigned int)((value - stats.min) * factor);

				if (bin > 9)
				{
					bin = 9;
				}

				histogram[bin]++;
			}

			sample_threshold = (unsigned int)(0.30f * (float)samples_per_cycle);
			number_of_samples = 0;
			found_sample = false;

			for (bin = 0; bin < 10; bin++)
			{
				if (histogram[bin] > sample_threshold)
				{
					number_of_samples += 1;
				}
			}

			if (number_of_samples == 0)
			{
				for (crank_angle = 0; crank_angle < samples_per_cycle; crank_angle++)
				{
					data[crank_angle] = 0.0f;
				}
			}
			else if (number_of_samples == 1)
			{
				low_bin = 0;
				for (bin = 0; bin < 10; bin++)
				{
					if (histogram[bin] > sample_threshold)
					{
						low_bin = bin;
					}
				}

				threshold = stats.min + (float)(low_bin + 1.0f) / 10.0f * stats.range;

				if (low_bin < 5)
				{
					lower_threshold = threshold + stats.stddev*0.5f;

					upper_threshold = threshold + stats.stddev;

					new_value = 0.0f;
				}
				else
				{
					lower_threshold = threshold - stats.stddev;

					upper_threshold = threshold - stats.stddev*0.5f;

					new_value = 1.0f;
				}

				//value = ReturnCAData(file, cycle, 0, channel);

				for (crank_angle = 0; crank_angle < samples_per_cycle; crank_angle++)
				{
					value = ReturnCAData(file, cycle, crank_angle, channel);

					if (value < lower_threshold)
					{
						new_value = 0.0f;
					}
					else if (value > upper_threshold)
					{
						new_value = 1.0f;
					}
					else
					{
						/* Do Nothing */
					}

					data[crank_angle] = new_value;
				}
			}
			else if (number_of_samples == 2)
			{
				found_sample = false;
				low_bin = 0;
				high_bin = 9;
				for (bin = 0; bin < 10; bin++)
				{
					if (histogram[bin] > sample_threshold)
					{
						if (found_sample == false)
						{
							low_bin = bin;
							found_sample = true;
						}
						else
						{
							high_bin = bin;
						}
					}
				}

				lower_threshold = (float)low_bin / 10.0f * stats.range + stats.min + stats.range / 10.0f / 2.0f + stats.stddev;

				upper_threshold = (float)high_bin / 10.0f * stats.range + stats.min + stats.range / 10.0f / 2.0f - stats.stddev;

				/* Calculate starting value */

				value = ReturnCAData(file, cycle, 0, channel);

				bin = (unsigned int)((value - stats.min) / stats.range * 10.0f);

				if (bin > 9)
				{
					bin = 9;
				}

				if (abs((int)bin - (int)low_bin) < abs((int)bin - (int)high_bin))
				{
					new_value = 0.0f;
				}
				else
				{
					new_value = 1.0f;
				}

				for (crank_angle = 0; crank_angle < samples_per_cycle; crank_angle++)
				{
					value = ReturnCAData(file, cycle, crank_angle, channel);

					if (value < lower_threshold)
					{
						new_value = 0.0f;
					}
					else if (value > upper_threshold)
					{
						new_value = 1.0f;
					}
					else
					{
						/* Do Nothing */
					}

					data[crank_angle] = new_value;
				}
			}
			else if (number_of_samples == 3)
			{
				for (crank_angle = 0; crank_angle < samples_per_cycle; crank_angle++)
				{
					value = ReturnCAData(file, cycle, crank_angle, channel);

					if (value - stats.min < stats.max - value)
					{
						new_value = 0.0f;
					}
					else
					{
						new_value = 1.0f;
					}

					data[crank_angle] = new_value;
				}
			}
			else
			{
				/* With a sample threshold of 30% this is not possible! */

				for (crank_angle = 0; crank_angle < samples_per_cycle; crank_angle++)
				{
					data[crank_angle] = 0.0f;
				}
			}

			looking_for_downedge = (data[0] > 0.5f);

			for (crank_angle = 0; crank_angle < samples_per_cycle; crank_angle++)
			{
				if (data[crank_angle] > 0.5f)
				{
					if (looking_for_downedge == false)
					{
						/* Found a upedge */

						looking_for_downedge = true;

						if (CrankAngleToDegrees(file, crank_angle, channel) - CrankAngleToDegrees(file, last_edge, channel) < angle_filter)
						{
							for (crank_angle_2 = last_edge; crank_angle_2 < crank_angle; crank_angle_2++)
							{
								data[crank_angle_2] = 1.0f;
							}
						}

						if (CrankAngleToDegrees(file, crank_angle, channel) - CrankAngleToDegrees(file, 0, channel) > angle_filter)
						{
							last_edge = crank_angle;
						}
					}
				}
				else
				{
					if (looking_for_downedge == true)
					{
						/* Found an downedge */

						looking_for_downedge = false;

						if (CrankAngleToDegrees(file, crank_angle, channel) - CrankAngleToDegrees(file, last_edge, channel) < angle_filter)
						{
							for (crank_angle_2 = last_edge; crank_angle_2 < crank_angle; crank_angle_2++)
							{
								data[crank_angle_2] = 0.0f;
							}
						}

						if (CrankAngleToDegrees(file, crank_angle, channel) - CrankAngleToDegrees(file, 0, channel) > angle_filter)
						{
							last_edge = crank_angle;
						}
					}
				}

				data[crank_angle] = (float)looking_for_downedge;
			}

			/*if (channel == 0)
				logmessage(DEBUG, "%u: %d %d %d %d %d %d %d %d %d %d %u %u %u %f %f %f",
				cycle,
				histogram[0],
				histogram[1],
				histogram[2],
				histogram[3],
				histogram[4],
				histogram[5],
				histogram[6],
				histogram[7],
				histogram[8],
				histogram[9],
				number_of_samples,
				samples_per_cycle,
				sample_threshold,
				threshold,
				upper_threshold,
				lower_threshold);*/
		}
	}
	else if (type == DIGITIZE_THRESHOLD)
	{
		lower_threshold = latch_low;
		upper_threshold = latch_high;
		
		new_value = 0.0f;

		for (crank_angle = 0; crank_angle < samples_per_cycle; crank_angle++)
		{
			value = ReturnCAData(file, cycle, crank_angle, channel);

			if (value < lower_threshold)
			{
				new_value = 0.0f;
			}
			else if (value > upper_threshold)
			{
				new_value = 1.0f;
			}
			else
			{
				/* Do Nothing */
			}

			data[crank_angle] = new_value;
		}
	}
	else /* if (type == DIGITIZE_NONE) */
	{
		for (crank_angle = 0; crank_angle < samples_per_cycle; crank_angle++)
		{
			data[crank_angle] = 0.0f;
		}
	}
}

void Return_Injection_Timing(const FileData* file,
							 const unsigned int channel,
							 const float engine_speed,
							 const float* digital_signal,
							 const float injection_window_start,
							 const float injection_window_finish,
							 const bool aligned,
							 const unsigned int max_number_of_injections,
							 float* soi1,
							 float* soi2,
							 float* soi3,
							 float* soi4,
							 float* soi5,
							 float* soi6,
							 float* eoi1,
							 float* eoi2,
							 float* eoi3,
							 float* eoi4,
							 float* eoi5,
							 float* eoi6,
							 float* noi,
							 float* injdur1,
							 float* injdur2, 
							 float* injdur3, 
							 float* injdur4, 
							 float* injdur5, 
							 float* injdur6, 
							 float* separation1,
							 float* separation2, 
							 float* separation3, 
							 float* separation4, 
							 float* separation5)
{
	unsigned int crank_angle;
	unsigned int number_of_edges = 0;
	float value;
	float previous_value;
	float edges[12];
	float injector_duration[6];
	float max_angle;
	float separation[6];
	unsigned int edge;
	unsigned int injection_window_start_ca = DegreesToCrankAngle(file,injection_window_start,channel);
	unsigned int injection_window_finish_ca = DegreesToCrankAngle(file, injection_window_finish, channel);
	unsigned int injection;

	memset(&edges, 0, max_number_of_injections * 2 * sizeof(float));

	previous_value = digital_signal[injection_window_start_ca];
	for (crank_angle=injection_window_start_ca+1;crank_angle<injection_window_finish_ca;crank_angle++)
	{
		value = digital_signal[crank_angle];

		if (fabs(value - previous_value) > 0.5)
		{
			if ((number_of_edges < 12) && (number_of_edges < max_number_of_injections * 2))
			{
				edges[number_of_edges] = CrankAngleToDegrees(file, crank_angle, channel);

				number_of_edges++;
			}
		}

		previous_value = value;
	}

	for (injection = 0; injection < 6; injection++)
	{
		if (number_of_edges > injection * 2)
		{
			injector_duration[injection] = (edges[injection*2+1] - edges[injection*2]) / engine_speed / 6.0f*1000.0f;	/* ms */
		}
		else
		{
			injector_duration[injection] = 0.0f;
		}

		if (number_of_edges > injection * 2 + 3)
		{
			separation[injection] = (edges[injection * 2 + 2] - edges[injection * 2 + 1]) / engine_speed / 6.0f*1000.0f;	/* ms */
		}
		else
		{
			separation[injection] = 0.0f;
		}
	}

	if (aligned == true)
	{
		max_angle = 90.0f*file->engine.number_of_strokes;

		for (edge = 0; edge < number_of_edges; edge++)
		{
			edges[edge] += file->channel_data[channel].tdc_offset;

			while (edges[edge] > max_angle)
			{
				edges[edge] -= 2.0f * max_angle;
			}
		}
	}

#ifdef _SAFE_MEMORY
	if (soi1 != NULL)
#endif
	{
		*soi1 = edges[0];
	}
#ifdef _SAFE_MEMORY
	if (eoi1 != NULL)
#endif
	{
		*eoi1 = edges[1];
	}
#ifdef _SAFE_MEMORY
	if (soi2 != NULL)
#endif
	{
		*soi2 = edges[2];
	}
#ifdef _SAFE_MEMORY
	if (eoi2 != NULL)
#endif
	{
		*eoi2 = edges[3];
	}
#ifdef _SAFE_MEMORY
	if (soi3 != NULL)
#endif
	{
		*soi3 = edges[4];
	}
#ifdef _SAFE_MEMORY
	if (eoi3 != NULL)
#endif
	{
		*eoi3 = edges[5];
	}
#ifdef _SAFE_MEMORY
	if (soi4 != NULL)
#endif
	{
		*soi4 = edges[6];
	}
#ifdef _SAFE_MEMORY
	if (eoi4 != NULL)
#endif
	{
		*eoi4 = edges[7];
	}
#ifdef _SAFE_MEMORY
	if (soi5 != NULL)
#endif
	{
		*soi5 = edges[8];
	}
#ifdef _SAFE_MEMORY
	if (eoi5 != NULL)
#endif
	{
		*eoi5 = edges[9];
	}
#ifdef _SAFE_MEMORY
	if (soi6 != NULL)
#endif
	{
		*soi6 = edges[10];
	}
#ifdef _SAFE_MEMORY
	if (eoi6 != NULL)
#endif
	{
		*eoi6 = edges[11];
	}
#ifdef _SAFE_MEMORY
	if (noi != NULL)
#endif
	{
		*noi = (float)(number_of_edges/2);
	}
#ifdef _SAFE_MEMORY
	if (injdur1 != NULL)
#endif
	{
		*injdur1 = injector_duration[0];
	}
#ifdef _SAFE_MEMORY
	if (injdur2 != NULL)
#endif
	{
		*injdur2 = injector_duration[1];
	}
#ifdef _SAFE_MEMORY
	if (injdur3 != NULL)
#endif
	{
		*injdur3 = injector_duration[2];
	}
#ifdef _SAFE_MEMORY
	if (injdur4 != NULL)
#endif
	{
		*injdur4 = injector_duration[3];
	}
#ifdef _SAFE_MEMORY
	if (injdur5 != NULL)
#endif
	{
		*injdur5 = injector_duration[4];
	}
#ifdef _SAFE_MEMORY
	if (injdur6 != NULL)
#endif
	{
		*injdur6 = injector_duration[5];
	}
#ifdef _SAFE_MEMORY
	if (separation1 != NULL)
#endif
	{
		*separation1 = separation[0];
	}
#ifdef _SAFE_MEMORY
	if (separation2 != NULL)
#endif
	{
		*separation2 = separation[1];
	}
#ifdef _SAFE_MEMORY
	if (separation3 != NULL)
#endif
	{
		*separation3 = separation[2];
	}
#ifdef _SAFE_MEMORY
	if (separation4 != NULL)
#endif
	{
		*separation4 = separation[3];
	}
#ifdef _SAFE_MEMORY
	if (separation5 != NULL)
#endif
	{
		*separation5 = separation[4];
	}
}

void Return_Cam_Edges(const FileData* file,
	const unsigned int channel,
	const float* digital_signal,
	float* edge_1,
	float* edge_2,
	float* edge_3,
	float* edge_4,
	float* edge_5,
	float* edge_6,
	float* edge_7,
	float* edge_8)
{
	unsigned int crank_angle;
	float value;
	unsigned int number_of_down_edges = 0;
	unsigned int number_of_up_edges = 0;
	float edges[8];
	unsigned int edge;
	float default_value = CrankAngleToDegrees(file, file->channel_data[channel].samples_per_cycle - 1, channel);

	for (edge = 0; edge < 8; edge++)
	{
		edges[edge] = default_value;
	}

	bool searching_for_upedge = digital_signal[0] < 0.5f;

	for (crank_angle = 1; crank_angle < file->channel_data[channel].samples_per_cycle; crank_angle++)
	{
		value = digital_signal[crank_angle];

		if ((searching_for_upedge == true) && (value > 0.5f))
		{
			/* Found an upedge */

			if (number_of_up_edges < 4)
			{
				edges[number_of_up_edges] = CrankAngleToDegrees(file, crank_angle, channel);
			}

			number_of_up_edges++;

			searching_for_upedge = false;
		}
		else if ((searching_for_upedge == false) && (value < 0.5f))
		{
			/* Found a downedge */

			if (number_of_down_edges < 4)
			{
				edges[4 + number_of_down_edges] = CrankAngleToDegrees(file, crank_angle, channel);
			}

			number_of_down_edges++;

			searching_for_upedge = true;
		}
		else
		{
			/* Do Nothing */
		}
	}

#ifdef _SAFE_MEMORY
	if (edge_1 != NULL)
#endif
	{
		*edge_1 = edges[0];
	}
#ifdef _SAFE_MEMORY
	if (edge_2 != NULL)
#endif
	{
		*edge_2 = edges[1];
	}
#ifdef _SAFE_MEMORY
	if (edge_3 != NULL)
#endif
	{
		*edge_3 = edges[2];
	}
#ifdef _SAFE_MEMORY
	if (edge_4 != NULL)
#endif
	{
		*edge_4 = edges[3];
	}
#ifdef _SAFE_MEMORY
	if (edge_5 != NULL)
#endif
	{
		*edge_5 = edges[4];
	}
#ifdef _SAFE_MEMORY
	if (edge_6 != NULL)
#endif
	{
		*edge_6 = edges[5];
	}
#ifdef _SAFE_MEMORY
	if (edge_7 != NULL)
#endif
	{
		*edge_7 = edges[6];
	}
#ifdef _SAFE_MEMORY
	if (edge_8 != NULL)
#endif
	{
		*edge_8 = edges[7];
	}
}

void Return_Camshaft_Analysis(const FileData* file,
	const unsigned int channel,
	const unsigned int cam_type,
	const unsigned int edge_type,
	const float reference_angle,
	const float offset,
	const float cam_edge_1,
	const float cam_edge_2,
	const float cam_edge_3,
	const float cam_edge_4,
	const float cam_edge_5,
	const float cam_edge_6,
	const float cam_edge_7,
	const float cam_edge_8,
	float* cam_angle,
	float* cam_advance)
{
	float distance[8];
	float abs_min_distance;
	unsigned int edge;
	unsigned int closest_edge = UINT_MAX;

	abs_min_distance = FLT_MAX;

	if ((edge_type == EDGE_RISING) || (edge_type == EDGE_BOTH))
	{
		distance[0] = reference_angle - cam_edge_1;
		distance[1] = reference_angle - cam_edge_2;
		distance[2] = reference_angle - cam_edge_3;
		distance[3] = reference_angle - cam_edge_4;

		for (edge = 0; edge < 4; edge++)
		{
			if (fabsf(distance[edge]) < abs_min_distance)
			{
				abs_min_distance = fabsf(distance[edge]);
				closest_edge = edge;
			}
		}
	}

	if ((edge_type == EDGE_FALLING) || (edge_type == EDGE_BOTH))
	{
		distance[4] = reference_angle - cam_edge_5;
		distance[5] = reference_angle - cam_edge_6;
		distance[6] = reference_angle - cam_edge_7;
		distance[7] = reference_angle - cam_edge_8;

		for (edge = 4; edge < 8; edge++)
		{
			if (fabsf(distance[edge]) < abs_min_distance)
			{
				abs_min_distance = fabsf(distance[edge]);
				closest_edge = edge;
			}
		}
	}

	if (closest_edge < 8)
	{
		*cam_advance = distance[closest_edge] + offset;

		switch (closest_edge)
		{
		default:
		case 0:	*cam_angle = cam_edge_1; break;
		case 1:	*cam_angle = cam_edge_2; break;
		case 2:	*cam_angle = cam_edge_3; break;
		case 3:	*cam_angle = cam_edge_4; break;
		case 4:	*cam_angle = cam_edge_5; break;
		case 5:	*cam_angle = cam_edge_6; break;
		case 6:	*cam_angle = cam_edge_7; break;
		case 7:	*cam_angle = cam_edge_8; break;
		}
	}
	else
	{
		*cam_advance = 0.0f;
		*cam_angle = 0.0f;
	}
}

/* Function: Return_Temperature_Data

   PARTIAL CYCLE ABSCISSA requires check */

void Return_Temperature_Data(const FileData* file,
							 const unsigned int channel,
	                         const unsigned int cycle,
							 float* volume,
							 const float temp_ref_ca,
							 const float t_ivc,
	                         const float gamma,
	                         const unsigned int gas_temp_model,
							 float* pressure,
							 float* max_mean_gas_temp,
							 float* max_mean_gas_temp_ca,
							 float* data)
{
	unsigned int crank_angle;
	unsigned int ivc_crank_angle;
	unsigned int evo_crank_angle;
	unsigned int temp_ref_crank_angle;
	float TbyPVref = 0.0f;
	float mean_gas_temp;
	float Pref;
	float Vref;
	float Tref;
	float k = gamma;
	float max_mean_gas_temp_tmp = 0.0f;
	unsigned int max_mean_gas_temp_ca_tmp = 0;

	ivc_crank_angle = DegreesToCrankAngle(file, file->engine.ivc_angle, channel);
	evo_crank_angle = DegreesToCrankAngle(file, file->engine.evo_angle, channel);

	switch (gas_temp_model)
	{
	case GAS_TEMP_AVL_THERMO1:
	{
		float vol_eff = 0.9f;

		float p_manifold = (ReturnThetaData(file, cycle, DegreesToTheta(file, -180.0f), channel) +
			ReturnThetaData(file, cycle, DegreesToTheta(file, -179.0f), channel) +
			ReturnThetaData(file, cycle, DegreesToTheta(file, -178.0f), channel)) / 3.0f;

		float t_manifold = 20.0f + C_TO_K;

		if (p_manifold > 0.0f)
		{
			TbyPVref = t_manifold / (vol_eff * file->engine.swept_volume * p_manifold);
		}

		break;
	}
	case GAS_TEMP_REF_CA:
	{
		temp_ref_crank_angle = DegreesToCrankAngle(file, temp_ref_ca, channel);

		Pref = pressure[temp_ref_crank_angle];
		Vref = volume[temp_ref_crank_angle];

		if (k < 1.0f)
		{
			if ((file->engine.type == ENGINE_SI) || (file->engine.type == ENGINE_DISI))
			{
				k = 1.32f;
			}
			else
			{
				k = 1.35f;
			}
		}

		Tref = t_ivc * powf((float)(volume[ivc_crank_angle] / Vref), (k - 1.0f));

		if ((Pref > 0.0f) && (Vref > 0.0f))
		{
			TbyPVref = Tref / Pref / Vref;
		}

		break;
	}
	case GAS_TEMP_IVC:
	{
		Pref = pressure[ivc_crank_angle];
		Vref = volume[ivc_crank_angle];
		Tref = t_ivc;

		if ((Pref > 0.0f) && (Vref > 0.0f))
		{
			TbyPVref = Tref / Pref / Vref;
		}

		break;
	}
	}

	for (crank_angle = ivc_crank_angle; crank_angle <= evo_crank_angle; crank_angle++)
	{
		mean_gas_temp = pressure[crank_angle] * volume[crank_angle] * TbyPVref;

		data[crank_angle] = mean_gas_temp;

		if (mean_gas_temp > max_mean_gas_temp_tmp)
		{
			max_mean_gas_temp_ca_tmp = crank_angle;
			max_mean_gas_temp_tmp = mean_gas_temp;
		}
	}

	for (crank_angle = 0; crank_angle < ivc_crank_angle; crank_angle++)
	{
		data[crank_angle] = data[ivc_crank_angle];
	}

	for (crank_angle = evo_crank_angle + 1; crank_angle < file->channel_data[channel].samples_per_cycle; crank_angle++)
	{
		data[crank_angle] = data[evo_crank_angle];
	}

#ifdef _SAFE_MEMORY
	if (max_mean_gas_temp != NULL)
#endif
	{
		*max_mean_gas_temp = max_mean_gas_temp_tmp;
	}
#ifdef _SAFE_MEMORY
	if (max_mean_gas_temp_ca != NULL)
#endif
	{
		*max_mean_gas_temp_ca = CrankAngleToDegrees(file, max_mean_gas_temp_ca_tmp, channel);
	}
}

/* Function: Return_Gamma_Data

   PARTIAL CYCLE ABSCISSA requires check */

void Return_Gamma_Data(const FileData* file,
                       const unsigned int channel,
					   const unsigned int method,
					   const float poly_comp,
					   const float poly_exp,
					   const float polytropic_index,
					   const float R,
					   const float* gamma,
					   const float* mean_gas_temp,
					   float* data)
{
	unsigned int crank_angle;
	unsigned int ivc_crank_angle;
	unsigned int evo_crank_angle;
	unsigned int three_sixty_degrees;
	float x;

	ivc_crank_angle = DegreesToCrankAngle(file, file->engine.ivc_angle, channel);
	evo_crank_angle = DegreesToCrankAngle(file, file->engine.evo_angle, channel);
	three_sixty_degrees = DegreesToCrankAngle(file, 0.0f, channel);

	for (crank_angle=0;crank_angle<ivc_crank_angle;crank_angle++)
	{
		data[crank_angle] = 0.0f;
	}

	switch (method)
	{
	case GAMMA_BRUNT:
	default:
	{
		switch (file->engine.type)
		{
		case ENGINE_CI_DI:	/* Brunt et al, SAE 1999-01-0187 */
		case ENGINE_CI_IDI:
		{
			for (crank_angle = ivc_crank_angle; crank_angle <= evo_crank_angle; crank_angle++)
			{
				data[crank_angle] = 1.35f - 6.0E-5f*mean_gas_temp[crank_angle] + 1.0E-8f*mean_gas_temp[crank_angle] * mean_gas_temp[crank_angle];
			}

			break;
		}
		case ENGINE_SI:		/* Brunt et al, SAE 981052 */
		case ENGINE_DISI:
		default:
		{
			for (crank_angle = ivc_crank_angle; crank_angle <= evo_crank_angle; crank_angle++)
			{
				data[crank_angle] = 1.338f - 6.0E-5f*mean_gas_temp[crank_angle] + 1.0E-8f*mean_gas_temp[crank_angle] * mean_gas_temp[crank_angle];
			}

			break;
		}
		}
		break;
	}
	case GAMMA_SPECIFIED:
	{
		for (crank_angle = ivc_crank_angle; crank_angle <= evo_crank_angle; crank_angle++)
		{
			data[crank_angle] = polytropic_index;
		}

		break;
	}
	case GAMMA_AIR:
	{
		for (crank_angle = ivc_crank_angle; crank_angle <= evo_crank_angle; crank_angle++)
		{
			data[crank_angle] = 1.4f - 8.0E-5f * mean_gas_temp[crank_angle];
		}

		break;
	}
	case GAMMA_KULZER:		/* Kulzer et al, SAE 2009-01-0501 */
	{
		for (crank_angle = ivc_crank_angle; crank_angle <= evo_crank_angle; crank_angle++)
		{
			data[crank_angle] = 1.0f + R / (653.6f + 0.22785f * mean_gas_temp[crank_angle]);
		}

		break;
	}
	case GAMMA_AVL_THERM1_CALC:
	case GAMMA_MEASURED_MEAN:
	{
		for (crank_angle = ivc_crank_angle; crank_angle < three_sixty_degrees; crank_angle++)
		{
			data[crank_angle] = poly_comp;
		}

		for (crank_angle = three_sixty_degrees; crank_angle <= evo_crank_angle; crank_angle++)
		{
			data[crank_angle] = poly_exp;
		}

		break;
	}
	case GAMMA_MEASURED:
	{
		for (crank_angle = ivc_crank_angle; crank_angle <= evo_crank_angle; crank_angle++)
		{
			data[crank_angle] = gamma[crank_angle];
		}

		break;
	}
	case GAMMA_INDOLENE:		/* SAE 841359 */
	{
		for (crank_angle = ivc_crank_angle; crank_angle <= evo_crank_angle; crank_angle++)
		{
			data[crank_angle] = 1.392f - 8.13E-5f*mean_gas_temp[crank_angle];
		}

		break;
	}
	case GAMMA_PROPANE:			/* SAE 841359 */
	{
		for (crank_angle = ivc_crank_angle; crank_angle <= evo_crank_angle; crank_angle++)
		{
			data[crank_angle] = 1.375f - 6.99E-5f*mean_gas_temp[crank_angle];
		}

		break;
	}
	case GAMMA_HAYES:			/* SAE 860029 */
	{
		for (crank_angle = ivc_crank_angle; crank_angle <= evo_crank_angle; crank_angle++)
		{
			if (mean_gas_temp[crank_angle] < 1000.0f)
			{
				x = (
						3.6359f - 
						1.33735E-3f * mean_gas_temp[crank_angle] + 
						3.29421E-6f * mean_gas_temp[crank_angle] * mean_gas_temp[crank_angle] - 
						1.9112E-9f * mean_gas_temp[crank_angle] * mean_gas_temp[crank_angle] * mean_gas_temp[crank_angle] + 
						0.275462E-12f * mean_gas_temp[crank_angle] * mean_gas_temp[crank_angle] * mean_gas_temp[crank_angle] * mean_gas_temp[crank_angle]
					);
			}
			else
			{
				x = (
						3.04473f + 
						1.33805E-3f * mean_gas_temp[crank_angle] - 
						0.488256E-6f * mean_gas_temp[crank_angle] * mean_gas_temp[crank_angle] + 
						0.0855475E-9f * mean_gas_temp[crank_angle] * mean_gas_temp[crank_angle] * mean_gas_temp[crank_angle] - 
						0.005701327E-12f * mean_gas_temp[crank_angle] * mean_gas_temp[crank_angle] * mean_gas_temp[crank_angle] * mean_gas_temp[crank_angle]
					);
			}

			data[crank_angle] = x / (x - 1.0f);
		}

		break;
	}
	case GAMMA_CHANG:			/* Chang et al, SAE 2004-01-2996 */
	{
		for (crank_angle = ivc_crank_angle; crank_angle <= evo_crank_angle; crank_angle++)
		{
			data[crank_angle] = 1.396f 
				- 1.436E-4f * mean_gas_temp[crank_angle]
				+ 6.207E-8f * mean_gas_temp[crank_angle] * mean_gas_temp[crank_angle]
				- 9.967E-12f * mean_gas_temp[crank_angle] * mean_gas_temp[crank_angle] * mean_gas_temp[crank_angle];
		}

		break;
	}
	case GAMMA_AVL_THERM1_FIXED:
	{
		switch (file->engine.type)
		{
		case ENGINE_SI:
		default:
		{
			for (crank_angle = ivc_crank_angle; crank_angle <= evo_crank_angle; crank_angle++)
			{
				data[crank_angle] = 1.32f;
			}

			break;
		}
		case ENGINE_CI_DI:
		case ENGINE_CI_IDI:
		{
			for (crank_angle = ivc_crank_angle; crank_angle <<= evo_crank_angle; crank_angle++)
			{
				data[crank_angle] = 1.37f;
			}

			break;
		}
		case ENGINE_DISI:
		{
			for (crank_angle = ivc_crank_angle; crank_angle <= evo_crank_angle; crank_angle++)
			{
				data[crank_angle] = 1.35f;
			}

			break;
		}
		}

		break;
	}
	case GAMMA_AVL_THERM1_A:
	{
		switch (file->engine.type)
		{
		case ENGINE_SI:
		default:
		{
			for (crank_angle = ivc_crank_angle; crank_angle < three_sixty_degrees; crank_angle++)
			{
				data[crank_angle] = 1.32f;
			}

			for (crank_angle = three_sixty_degrees; crank_angle <= evo_crank_angle; crank_angle++)
			{
				data[crank_angle] = 1.27f;
			}

			break;
		}
		case ENGINE_CI_DI:
		case ENGINE_CI_IDI:
		{
			for (crank_angle = ivc_crank_angle; crank_angle < three_sixty_degrees; crank_angle++)
			{
				data[crank_angle] = 1.37f;
			}

			for (crank_angle = three_sixty_degrees; crank_angle <= evo_crank_angle; crank_angle++)
			{
				data[crank_angle] = 1.30f;
			}

			break;
		}
		case ENGINE_DISI:
		{
			for (crank_angle = ivc_crank_angle; crank_angle < three_sixty_degrees; crank_angle++)
			{
				data[crank_angle] = 1.35f;
			}

			for (crank_angle = three_sixty_degrees; crank_angle <= evo_crank_angle; crank_angle++)
			{
				data[crank_angle] = 1.30f;
			}

			break;
		}
		}

		break;
	}
	}

	for (crank_angle=evo_crank_angle+1;crank_angle<file->channel_data[channel].samples_per_cycle;crank_angle++)
	{
		data[crank_angle] = 0.0f;
	}
}

void Prepare_Polytropic_Index(const FileData* file,
	                          const unsigned int channel,
	                          const float start_angle,
	                          const float finish_angle,
							  unsigned int* number_of_points,
							  unsigned int* start_crank_angle,
							  float** logPressure)
{
	unsigned int start_ca;
	unsigned int finish_ca;
	unsigned int data_size;
	
	start_ca = DegreesToCrankAngle(file,start_angle,channel);
	finish_ca = DegreesToCrankAngle(file,finish_angle,channel);
	data_size = (finish_ca-start_ca+1);
	
	*logPressure = (float*)malloc(data_size * sizeof(float));
	if (*logPressure == NULL)
	{
		logmessage(FATAL,"Memory could not be allocated\n");
	}

	if (number_of_points != NULL)
	{
		*number_of_points = data_size;
	}

	if (start_crank_angle != NULL)
	{
		*start_crank_angle = start_ca;
	}
}

void Return_Polytropic_Index(const FileData* file,
	                         const unsigned int cycle,
							 const unsigned int channel,
							 const unsigned int start_crank_angle,
							 const unsigned int number_of_points,
							 float* logVolume,
							 float* logPressure,
							 float* pressure,
							 float* gamma,
							 float* k)
{
	float a0 = 0.0f;
	float a1 = 0.0f;
	float value;
	unsigned int crank_angle;

	for (crank_angle=0;crank_angle<number_of_points;crank_angle++)
	{
		value = pressure[start_crank_angle + crank_angle];

		if (value > 0.0f)
		{
			logPressure[crank_angle] = logf(value);
		}
		else
		{
			logPressure[crank_angle] = 0.0f;
		}
	}

	calculate_first_order_least_squares(number_of_points, &logVolume[start_crank_angle], logPressure, &a0, &a1);

#ifdef _SAFE_MEMORY
	if (gamma != NULL)
#endif   
	{
		*gamma = -a1;
	}

#ifdef _SAFE_MEMORY
	if (k != NULL)
#endif
	{
		*k = powf(10.0f, a0);
	}
}

void Complete_Polytropic_Index(float* logPressure)
{
	if (logPressure != NULL)
	{
		free(logPressure);
	}
}

/* Function: Return_Polytropic_Indices_Data

   OK for partial cycle abscissas */

void Return_Polytropic_Indices_Data(const FileData* file,
	const unsigned int channel,
	float* volume,
	const float* pressure_data,
	float* data)
{
	float previous_pressure;
	float next_pressure;
	float previous_volume;
	float next_volume;
	unsigned int crank_angle;
	unsigned int crank_angle_ivc;
	unsigned int crank_angle_evo;

#ifdef _SAFE_MEMORY
	if (data == NULL)
	{
		return;
	}
#endif 

	if (file->channel_data[channel].samples_per_cycle >= 5)
	{
		crank_angle_ivc = DegreesToCrankAngle(file, -180.0f/*file->engine.ivc_ca*/, channel);

		if (crank_angle_ivc < 4)
		{
			crank_angle_ivc = 4;
		}

		crank_angle_evo = DegreesToCrankAngle(file, 180.0f/*file->engine.evo_ca*/, channel);

		if (crank_angle_evo > file->channel_data[channel].samples_per_cycle - 5)
		{
			crank_angle_evo = file->channel_data[channel].samples_per_cycle - 5;
		}

		for (crank_angle = 0; crank_angle < crank_angle_ivc; crank_angle++)
		{
			data[crank_angle] = 0.0f;
		}

		for (crank_angle = crank_angle_ivc; crank_angle <= crank_angle_evo; crank_angle++)
		{
			next_pressure = pressure_data[crank_angle + 4];
			previous_pressure = pressure_data[crank_angle - 4];

			next_volume = volume[crank_angle + 4];
			previous_volume = volume[crank_angle - 4];

			if ((next_pressure > 0.0f) && 
				(previous_pressure > 0.0f) && 
				((next_volume > previous_volume) || (next_volume < previous_volume)))
			{
				data[crank_angle] = logf(previous_pressure / next_pressure) / logf(next_volume / previous_volume);
			}
			else
			{
				data[crank_angle] = 500.0f;
			}
		}

		for (crank_angle = crank_angle_evo + 1; crank_angle < file->channel_data[channel].samples_per_cycle; crank_angle++)
		{
			data[crank_angle] = 0.0f;
		}
	}
	else
	{
		for (crank_angle = 0; crank_angle < file->channel_data[channel].samples_per_cycle; crank_angle++)
		{
			data[crank_angle] = 0.0f;
		}
	}
}

/* Function: Return_Piston_Displacement_Data

   Division by zero when l = 0
   Protected by logic in fileprocessor.cpp
   OK for partial cycle abscissas */

void Return_Piston_Displacement_Data(const FileData* file,
									 const unsigned int channel,
									 float* data)
{
	float l = file->engine.conrod_length;
	float r = file->engine.stroke/2.0f;
	float theta_rad;
	unsigned int crank_angle;

	if (data != NULL)
    {
        for (crank_angle=0;crank_angle<file->channel_data[channel].samples_per_cycle;crank_angle++)
    	{
    		theta_rad = CrankAngleToDegrees(file,crank_angle,channel)*DEG_TO_RAD;
    		data[crank_angle] = l+r-r*cosf(theta_rad)-l*sqrtf(1.0f-r*r/l/l*sinf(theta_rad)*sinf(theta_rad));
    	}
    }
}

/* Function: Return_Piston_Velocity_Data

   Division by zero when r = 0 OR l/r <= 1
   Protected by logic in fileprocessor.cpp 
   OK for partial cycle abscissas */

void Return_Piston_Velocity_Data(const FileData* file,
								 const unsigned int channel,
								 const float engine_speed,
								 float* data)
{
	float l = file->engine.conrod_length;
	float r = file->engine.stroke/2.0f;
	float theta_rad;
	unsigned int crank_angle;

    if (data != NULL)
    {
    	for (crank_angle=0;crank_angle<file->channel_data[channel].samples_per_cycle;crank_angle++)
    	{
    		theta_rad = CrankAngleToDegrees(file,crank_angle,channel)*DEG_TO_RAD;
    		data[crank_angle] = r*engine_speed*sinf(theta_rad)*(1.0f+cosf(theta_rad))/sqrtf(l*l/r/r-sinf(theta_rad)*sinf(theta_rad));
    	}
	}
}

/* Function: Return_Piston_Acceleration_Data

   Division by zero when r = 0 OR l/r <= 1
   Protected by logic in fileprocessor.cpp
   OK for partial cycle abscissas */

void Return_Piston_Acceleration_Data(const FileData* file,
									 const unsigned int channel,
									 const float engine_speed,
									 float* data)
{
	float l = file->engine.conrod_length;
	float r = file->engine.stroke/2.0f;
	float theta_rad;
	unsigned int crank_angle;

	if (data != NULL)
	{
        for (crank_angle=0;crank_angle<file->channel_data[channel].samples_per_cycle;crank_angle++)
    	{
    		theta_rad = CrankAngleToDegrees(file,crank_angle,channel)*DEG_TO_RAD;
    		data[crank_angle] = r*engine_speed*engine_speed*(cosf(theta_rad)+((l*l/r/r*cosf(2.0f*engine_speed)+powf(sinf(theta_rad),4.0f))/powf(l*l/r/r-sinf(theta_rad)*sinf(theta_rad),1.5f)));
    	}
	}
}


void Return_COG(const FileData* file, const unsigned int channel, const float burn_angle_10, const float burn_angle_90, const float* mfb, float* centre_of_gravity)
{
	unsigned int burn_angle_10_ca;
	unsigned int burn_angle_90_ca;
	float sum;
	unsigned int crank_angle;
	unsigned int maximum_theta = MaximumTheta(file, channel);
	unsigned int theta;

	burn_angle_10_ca = DegreesToCrankAngle(file, burn_angle_10, channel);
	burn_angle_90_ca = DegreesToCrankAngle(file, burn_angle_90, channel);

	sum = 0.0f;

	for (crank_angle = burn_angle_10_ca; crank_angle < burn_angle_90_ca; crank_angle++)
	{
		sum += (mfb[crank_angle + 1] - mfb[crank_angle]) * (ReturnTheta(file, crank_angle, channel) + ReturnTheta(file, crank_angle + 1, channel)) / 2.0f;
	}
#ifdef _SAFE_MEMORY
	if (centre_of_gravity != NULL)
#endif
	{
		theta = (unsigned int)(sum / 80.0f);

		if (theta < maximum_theta)
		{
			*centre_of_gravity = ThetaToDegrees(file, theta);
		}
		else
		{
			*centre_of_gravity = 0.0f;
		}
	}
}

void Return_Knock_Factor(const FileData* file,
	                     const unsigned int channel,
						 const float engine_speed,
						 const float max_pressure_ca,
						 const float* knkp,
						 const float knock_window_width,
						 const float reference_window_width,
						 const float speed_shift_start,
						 const float speed_shift_finish,
						 const float speed_shift_start_correction,
						 const float speed_shift_finish_correction,
						 const float noise_threshold,
						 float* knock_factor)
{
	float correction;
	unsigned int reference_start_ca;
	unsigned int reference_finish_ca;
	unsigned int knock_start_ca;
	unsigned int knock_finish_ca;
	float reference_integral;
	float knock_integral;
	unsigned int crank_angle;

	/* Mannesmann VDO -> Siemens -> Continenal method uses a +/- 4 degree moving average filter to find peak pressure */
	/* It then uses a 5 kHz moving average filter to calculate the knocking pressure */

	/* Calculate windows */

	if (engine_speed < speed_shift_start)
	{
		correction = speed_shift_start_correction;
	}
	else if (engine_speed > speed_shift_finish)
	{
		correction = speed_shift_finish_correction;
	}
	else
	{
		correction = speed_shift_start_correction + (engine_speed - speed_shift_start) / (speed_shift_finish - speed_shift_start)*(speed_shift_finish_correction - speed_shift_start_correction);
	}

	reference_start_ca = DegreesToCrankAngle(file, max_pressure_ca - reference_window_width - correction,channel);
	reference_finish_ca = DegreesToCrankAngle(file, max_pressure_ca - correction,channel);
	knock_start_ca = reference_finish_ca;
	knock_finish_ca = DegreesToCrankAngle(file, max_pressure_ca + knock_window_width - correction,channel);

	/* Integrate reference window */

	reference_integral = 0.0f;
	for (crank_angle = reference_start_ca; crank_angle<reference_finish_ca; crank_angle++)
	{
		if (knkp[crank_angle] < noise_threshold)
		{
			reference_integral += noise_threshold;
		}
		else
		{
			reference_integral += knkp[crank_angle];
		}
	}

	/* Integrate knock window */

	knock_integral = 0.0f;
	for (crank_angle = knock_start_ca; crank_angle<knock_finish_ca; crank_angle++)
	{
		if (knkp[crank_angle] < noise_threshold)
		{
			knock_integral += noise_threshold;
		}
		else
		{
			knock_integral += knkp[crank_angle];
		}
	}

	/* Calculate knock factor */

	if (reference_integral < FLT_EPSILON)
	{
		reference_integral = FLT_EPSILON;
	}

#ifdef _SAFE_MEMORY
	if (knock_factor != NULL)
#endif
	{
		*knock_factor = knock_integral / reference_integral;
	}
}

void Return_Postitive_HR_SOC(const FileData* file,
							 const unsigned int channel,
							 const float ivc,
							 const float evo,
							 const float poly_comp,
							 const float poly_exp,
	                         const float max_pressure_ca,
							 const float* volume,
							 const float* pressure,
							 const float threshold,
							 float* start_of_combustion)
{
	unsigned int crank_angle;
	unsigned int ivc_ca = DegreesToCrankAngle(file, ivc, channel);
	unsigned int three_sixty_degrees = DegreesToCrankAngle(file, 0.0f, channel);
	float calc;
	float n;

	crank_angle = DegreesToCrankAngle(file, max_pressure_ca, channel);

	if (crank_angle == 0)
	{
		crank_angle = 1;
	}

	do
	{
		if (crank_angle < three_sixty_degrees)
		{
			n = poly_comp;
		}
		else
		{
			n = poly_exp;
		}

		calc = pressure[crank_angle] - pressure[crank_angle-1] * powf(volume[crank_angle - 1] / volume[crank_angle], n);

		crank_angle -= 1;
	}
	while ((calc > threshold) && (crank_angle > ivc_ca));

#ifdef _SAFE_MEMORY
	if (start_of_combustion != NULL)
#endif
	{
		*start_of_combustion = CrankAngleToDegrees(file, crank_angle, channel);
	}
}

/* Function: Return_Accelerometer_Data */

void Return_Accelerometer_Data(const FileData* file,
	const unsigned int cycle,
	const unsigned int channel,
	const float* dCA,
	const float reference_start_angle,
	const float reference_finish_angle,
	const float knock_window_start_angle,
	const float knock_window_finish_angle,
	float* knock_integral,
	float* knock_factor)
{
	unsigned int crank_angle;
	unsigned int knock_ref_start_ca;
	unsigned int knock_ref_finish_ca;
	unsigned int knock_window_start_ca;
	unsigned int knock_window_finish_ca;
	float reference_integral = 0.0f;
	float knock_integral_tmp = 0.0f;
	float data;

	knock_ref_start_ca = DegreesToCrankAngle(file, reference_start_angle, channel);
	knock_ref_finish_ca = DegreesToCrankAngle(file, reference_finish_angle, channel);
	knock_window_start_ca = DegreesToCrankAngle(file, knock_window_start_angle, channel);
	knock_window_finish_ca = DegreesToCrankAngle(file, knock_window_finish_angle, channel);

	for (crank_angle = knock_ref_start_ca; crank_angle < knock_ref_finish_ca; crank_angle++)
	{
		data = fabsf(ReturnCAData(file, cycle, crank_angle, channel));

		reference_integral += data * dCA[crank_angle];
	}

	if (reference_finish_angle > reference_start_angle)
	{
		/* Scale the reference window in case it is a difference size to the knock window */

		reference_integral *= (knock_window_finish_angle - knock_window_start_angle) / (reference_finish_angle - reference_start_angle);
	}
	else
	{
		/* If there is no reference window then force the knock factor to 1.0 */

		reference_integral = 0.0f;
	}

	for (crank_angle = knock_window_start_ca; crank_angle < knock_window_finish_ca; crank_angle++)
	{
		data = fabsf(ReturnCAData(file, cycle, crank_angle, channel));

		knock_integral_tmp += data * dCA[crank_angle];
	}

#ifdef _SAFE_MEMORY
	if (knock_integral != NULL)
#endif
	{
		*knock_integral = knock_integral_tmp;
	}

#ifdef _SAFE_MEMORY
	if (knock_factor != NULL)
#endif
	{
		if (reference_integral > 0.0f)
		{
			*knock_factor = knock_integral_tmp / reference_integral;
		}
		else
		{
			*knock_factor = 1.0f;
		}
	}
}

void Return_Crank_Sensor_Analysis(const FileData* file,
	const unsigned int channel,
	const float* digital_signal,
	float* missing_tooth_angle_1,
	float* missing_tooth_angle_2,
	float* missing_tooth_ratio_min,
	float* missing_tooth_ratio_max,
	float* gap_ratio_min,
	float* gap_ratio_max)
{
	unsigned int crank_angle;
	bool looking_for_upedge;
	//unsigned int number_of_upedges = 0;
	unsigned int number_of_downedges = 0;
	unsigned int last_downedge = 0;
	unsigned int downedge_period[3];
	unsigned int period;
	float ratio_1;
	float ratio_2;
	unsigned int downedge_missing_tooth_crank_angle_1;
	unsigned int downedge_missing_tooth_crank_angle_2;
	unsigned int number_of_downedge_missing_teeth = 0;
	float min_ratio = 100.0f;
	float max_ratio = 0.0f;
	float min_mt_ratio = 100.0f;
	float max_mt_ratio = 0.0f;

	for (period = 0; period < 3; period++)
	{
		downedge_period[period] = 0;
	}

	looking_for_upedge = (digital_signal[0] < 0.5f);

	for (crank_angle = 1; crank_angle < file->channel_data[channel].samples_per_cycle; crank_angle++)
	{
		if (looking_for_upedge == true)
		{
			if (digital_signal[crank_angle] > 0.5f)
			{
				looking_for_upedge = false;
				//number_of_upedges++;
			}
		}
		else
		{
			if (digital_signal[crank_angle] < 0.5f)
			{
				if (number_of_downedges > 0)
				{
					period = ReturnTheta(file, crank_angle, channel) - ReturnTheta(file, last_downedge, channel);

					downedge_period[2] = downedge_period[1];
					downedge_period[1] = downedge_period[0];
					downedge_period[0] = period;

					if (number_of_downedges > 2)
					{
						ratio_1 = (float)downedge_period[1] / (float)downedge_period[0];
						ratio_2 = (float)downedge_period[1] / (float)downedge_period[2];

						if ((ratio_1 > 1.5f) && (ratio_2 > 1.5f))
						{
							if (number_of_downedge_missing_teeth == 0)
							{
								downedge_missing_tooth_crank_angle_1 = crank_angle;
							}
							else if (number_of_downedge_missing_teeth == 1)
							{
								downedge_missing_tooth_crank_angle_2 = crank_angle;
							}
							else
							{
								/* Do Nothing */
							}

							number_of_downedge_missing_teeth++;

							if (ratio_1 > max_mt_ratio)
							{
								max_mt_ratio = ratio_1;
							}
							else if (ratio_1 < min_mt_ratio)
							{
								min_mt_ratio = ratio_1;
							}
							else
							{
								/* Do Nothing */
							}

							if (ratio_2 > max_mt_ratio)
							{
								max_mt_ratio = ratio_2;
							}
							else if (ratio_2 < min_mt_ratio)
							{
								min_mt_ratio = ratio_2;
							}
							else
							{
								/* Do Nothing */
							}
						}
						else if ((ratio_1 > 0.75f) && (ratio_2 > 0.75f))
						{
							if (ratio_1 > max_ratio)
							{
								max_ratio = ratio_1;
							}
							else if (ratio_1 < min_ratio)
							{
								min_ratio = ratio_1;
							}
							else
							{
								/* Do Nothing */
							}

							if (ratio_2 > max_ratio)
							{
								max_ratio = ratio_2;
							}
							else if (ratio_2 < min_ratio)
							{
								min_ratio = ratio_2;
							}
							else
							{
								/* Do Nothing */
							}
						}
						else
						{
							/* Do Nothing - Probably about to find a missing tooth */
						}
					}
				}

				looking_for_upedge = true;
				number_of_downedges++;
				last_downedge = crank_angle;
			}
		}
	}

	if (number_of_downedge_missing_teeth > 0)
	{
		*missing_tooth_angle_1 = CrankAngleToDegrees(file, downedge_missing_tooth_crank_angle_1, channel);

		if (number_of_downedge_missing_teeth > 1)
		{
			*missing_tooth_angle_2 = CrankAngleToDegrees(file, downedge_missing_tooth_crank_angle_2, channel);
		}
		else
		{
			*missing_tooth_angle_2 = *missing_tooth_angle_1;
		}

		*missing_tooth_ratio_min = min_mt_ratio;
		*missing_tooth_ratio_max = max_mt_ratio;
	}
	else
	{
		*missing_tooth_angle_1 = -360.0f;
		*missing_tooth_angle_2 = -360.0f;
		*missing_tooth_ratio_min = 0.0f;
		*missing_tooth_ratio_max = 0.0f;
	}

	if (number_of_downedges > 2)
	{
		*gap_ratio_min = min_ratio;
		*gap_ratio_max = max_ratio;
	}
	else
	{
		*gap_ratio_min = 0.0f;
		*gap_ratio_max = 0.0f;
	}
}

void Return_Coil_Analysis(const FileData* file,
	const unsigned int cycle,
	const unsigned int channel,
	const bool calculate_max_current,
	const bool calculate_restrike,
	const float* digital_channel,
	const float engine_speed,
	float* dwell_time,
	float* max_coil_current,
	float* restrike_max_current,
	float* restrike_timing,
	float* restrike_delay_time,
	float* restrike_dwell_time)
{
	unsigned int crank_angle;
	float value;
	float low_current_mean = 0.0f;
	unsigned int data_points = 0;
	float max_current = 0.0f;
	unsigned int first_edge = 0;
	unsigned int second_edge = 0;
	bool looking_for_upedge = (digital_channel[0] < 0.5f);
	bool looking_for_downedge = false;
	float dwell_angle;

	for (crank_angle = 0; crank_angle < file->channel_data[channel].samples_per_cycle; crank_angle++)
	{
		value = ReturnCAData(file, cycle, crank_angle, channel);

		if (digital_channel[crank_angle] < 0.5f)
		{
			low_current_mean += value;
			data_points++;

			if (looking_for_downedge == true)
			{
				looking_for_downedge = false;
				
				second_edge = crank_angle;
			}
		}
		else
		{
			if (value > max_current)
			{
				max_current = value;
			}

			if (looking_for_upedge == true)
			{
				looking_for_upedge = false;
				looking_for_downedge = true;

				first_edge = crank_angle;
			}
		}

	}

	if ((calculate_max_current == true) && (data_points > 0))
	{
		low_current_mean /= (float)data_points;

		max_current -= low_current_mean;

		*max_coil_current = max_current;
	}
	else
	{
		*max_coil_current = 0.0f;
	}

	if ((looking_for_downedge == false) && (looking_for_upedge == false) && (first_edge < second_edge) && (engine_speed > 0.0f))
	{
		dwell_angle = CrankAngleToDegrees(file, second_edge, channel) - CrankAngleToDegrees(file, first_edge, channel);

		*dwell_time = dwell_angle / engine_speed / 6.0f * 1000.0f;		/* ms */
	}
	else
	{
		*dwell_time = 0.0f;
	}
}

void Return_Pressure_Second_Derivative(const FileData* file,
	const unsigned int cycle,
	const unsigned int channel,
	const float window,
	float* d2p_max,
	float* d2p_max_ca,
	float* data)
{
	unsigned int crank_angle;
	unsigned int theta_step = DegreesToThetaAbs(file, window);
	unsigned int theta;
	float d2p;
	float max_d2p = -FLT_MAX;
	unsigned int max_d2p_ca = 0;
	float factor = window * window;
	unsigned int max_theta = MaximumTheta(file, channel) - theta_step;

	for (crank_angle = 0; crank_angle < file->channel_data[channel].samples_per_cycle; crank_angle++)
	{
		theta = ReturnTheta(file, crank_angle, channel);

		if ((theta >= theta_step) && (theta < max_theta))
		{
			d2p = (ReturnThetaData(file, cycle, theta - theta_step, channel) +
				ReturnThetaData(file, cycle, theta + theta_step, channel) -
				2.0f * ReturnCAData(file,cycle,crank_angle,channel)) / factor;
		}
		else
		{
			d2p = 0.0f;
		}

		if (d2p > max_d2p)
		{
			max_d2p = d2p;
			max_d2p_ca = crank_angle;
		}

		data[crank_angle] = d2p;
	}

	*d2p_max = max_d2p;
	*d2p_max_ca = CrankAngleToDegrees(file, max_d2p_ca, channel);
}

void Return_Knock_Signal_Energy(const FileData* file,
	const unsigned int channel,
	const float* dCA,
	const float engine_speed,
	const float knock_onset_ca,
	const float* knocking_pressure,
	float* knock_signal_energy)
{
	unsigned int crank_angle;
	float resolution = 7.0f;
	unsigned int start_crank_angle = DegreesToCrankAngle(file, knock_onset_ca, channel);
	unsigned int finish_crank_angle = DegreesToCrankAngle(file, knock_onset_ca + resolution, channel);
	float ki = 0.0f;

	for (crank_angle = start_crank_angle; crank_angle < finish_crank_angle; crank_angle++)
	{
		ki += knocking_pressure[crank_angle] * knocking_pressure[crank_angle] * dCA[crank_angle];
	}

	if (engine_speed > 0.0f)
	{
		ki *= 1.0f / 6.0f / resolution / engine_speed;
	}
	else
	{
		ki = 0.0f;
	}

	*knock_signal_energy = ki;
}

void Return_Misfire_Data(const FileData* file,
	const unsigned int cycle_start,
	const unsigned int cycle_finish,
	const unsigned int number_of_cycles,
	const unsigned int number_of_cycles_f,
	const unsigned int number_of_cycles_m,
	const unsigned int number_of_cycles_s,
	const unsigned char* cycle_classification,
	float* misfire_f,
	float* misfire_m,
	float* misfire_s)
{
	unsigned int cycle;
	unsigned int cycle_to_analyse;
	unsigned int number_of_misfires_f = 0;
	unsigned int number_of_misfires_m = 0;
	unsigned int number_of_misfires_s = 0;
	unsigned int number_of_valid_cycles_f = 0;
	unsigned int number_of_valid_cycles_m = 0;
	unsigned int number_of_valid_cycles_s = 0;

	unsigned int cycle_finish_mod = cycle_finish;

	if (cycle_finish_mod < cycle_start)
	{
		cycle_finish_mod = number_of_cycles + cycle_finish_mod;
	}

	int min_cycle_s = cycle_finish_mod - number_of_cycles_s + 1;

	if (min_cycle_s < 0)
	{
		min_cycle_s = 0;
	}

	int min_cycle_m = cycle_finish_mod - number_of_cycles_m + 1;

	if (min_cycle_m < 0)
	{
		min_cycle_m = 0;
	}

	int min_cycle_f = cycle_finish_mod - number_of_cycles_f + 1;

	if (min_cycle_f < 0)
	{
		min_cycle_f = 0;
	}

	for (cycle_to_analyse = cycle_start; cycle_to_analyse <= cycle_finish_mod; cycle_to_analyse++)
	{
		if (cycle_to_analyse >= number_of_cycles)
		{
			cycle = cycle_to_analyse - number_of_cycles;
		}
		else
		{
			cycle = cycle_to_analyse;
		}

		bool misfire = ((cycle_classification[cycle] & CYCLE_MISFIRE) != 0) && ((cycle_classification[cycle] & CYCLE_DEACTIVATED) == 0);

		if ((int)cycle_to_analyse >= min_cycle_s)
		{
			if (misfire == true)
			{
				number_of_misfires_s++;
			}

			number_of_valid_cycles_s++;

			if ((int)cycle_to_analyse >= min_cycle_m)
			{
				if (misfire == true)
				{
					number_of_misfires_m++;
				}

				number_of_valid_cycles_m++;

				if ((int)cycle_to_analyse >= min_cycle_f)
				{
					if (misfire == true)
					{
						number_of_misfires_f++;
					}

					number_of_valid_cycles_f++;
				}
			}
		}
	}

	if (number_of_valid_cycles_f > 0)
	{
		*misfire_f = (float)number_of_misfires_f / (float)number_of_valid_cycles_f * 100.0f;
	}
	else
	{
		*misfire_f = 0.0f;
	}

	if (number_of_valid_cycles_m > 0)
	{
		*misfire_m = (float)number_of_misfires_m / (float)number_of_valid_cycles_m * 100.0f;
	}
	else
	{
		*misfire_m = 0.0f;
	}

	if (number_of_valid_cycles_s > 0)
	{
		*misfire_s = (float)number_of_misfires_s / (float)number_of_valid_cycles_s * 100.0f;
	}
	else
	{
		*misfire_s = 0.0f;
	}
}