/* Combustion Analysis Tool (CAT)
   www.catool.org

   Filename: abscissa.c

   Purpose:  Provide common programming interface for processing acquired engine data

   Author:   Ben Brown
   Version:  1.2
   Date:     19.10.2015

   Revision: GPL Release

   Copyright (C) Xarin Limited, 2000-2024

      This program is free software: you can redistribute it and/or modify
      it under the terms of version 2 of the GNU General Public License as
      published by the Free Software Foundation.

      This program is distributed in the hope that it will be useful,
      but WITHOUT ANY WARRANTY; without even the implied warranty of
      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
      GNU General Public License for more details.

      You should have received a copy of the GNU General Public License
      along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "stdlib.h"
#include "string.h"

#include "cat.h"

extern unsigned int debug_level;

unsigned int Copy_Abscissa(AbscissaData* source, AbscissaData* dest)
{
	unsigned int samples_per_cycle;
	unsigned int measurement_table;

	if (dest == NULL)
	{
		return(0);
	}

	Empty_Abscissa(dest);

	if (source == NULL)
	{
		dest->type = ABSCISSA_CRANKANGLE;
		dest->number_of_measurement_tables = 0;
		dest->number_of_samples = NULL;
		dest->resolution = NULL;
		dest->start = NULL;
		dest->units[0] = 0x00;
		dest->slope = 1.0;
		dest->offset = 0.0;

		samples_per_cycle = 0;
	}
	else
	{
		dest->type = source->type;
		strncpy(dest->units, source->units, SIZEOF_UNITS);
		dest->units[SIZEOF_UNITS - 1] = 0x00;
		dest->number_of_measurement_tables = source->number_of_measurement_tables;

		if (dest->number_of_measurement_tables > 0)
		{
			dest->number_of_samples = (unsigned int*)malloc(sizeof(unsigned int) * dest->number_of_measurement_tables);
			if (dest->number_of_samples == NULL)
			{
				logmessage(FATAL, "Memory could not be allocated\n");
				exit(EXIT_FAILURE);
			}

			dest->start = (unsigned int*)malloc(sizeof(unsigned int) * dest->number_of_measurement_tables);
			if (dest->start == NULL)
			{
				logmessage(FATAL, "Memory could not be allocated\n");
				exit(EXIT_FAILURE);
			}

			dest->resolution = (unsigned int*)malloc(sizeof(unsigned int) * dest->number_of_measurement_tables);
			if (dest->resolution == NULL)
			{
				logmessage(FATAL, "Memory could not be allocated\n");
				exit(EXIT_FAILURE);
			}

			samples_per_cycle = 0;
			for (measurement_table = 0; measurement_table<dest->number_of_measurement_tables; measurement_table++)
			{
				dest->number_of_samples[measurement_table] = source->number_of_samples[measurement_table];
				dest->start[measurement_table] = source->start[measurement_table];
				dest->resolution[measurement_table] = source->resolution[measurement_table];

				samples_per_cycle += dest->number_of_samples[measurement_table];
			}

			if (source->axis != NULL)
			{
				size_t data_size;

				switch (source->axis_type)
				{
				case VARIABLE_INT64: data_size = samples_per_cycle * sizeof(int64_t);	break;
				case VARIABLE_INT32: data_size = samples_per_cycle * sizeof(int32_t);	break;
				case VARIABLE_FLOAT: data_size = samples_per_cycle * sizeof(float);		break;
				case VARIABLE_DOUBLE: data_size = samples_per_cycle * sizeof(double);	break;
				default: data_size = 0; break;
				}

				dest->axis = malloc(data_size);

				memcpy(dest->axis, source->axis, data_size);
			}
		}
		else
		{
			dest->number_of_samples = NULL;
			dest->resolution = NULL;
			dest->start = NULL;
			samples_per_cycle = 0;
		}

		dest->axis_type = source->axis_type;
		dest->slope = source->slope;
		dest->offset = source->offset;
	}

	return(samples_per_cycle);
}

void Zero_Abscissa(AbscissaData* abscissa)
{
	if (abscissa == NULL)
	{
		return;
	}

	abscissa->axis = NULL;
	abscissa->slope = 1.0;
	abscissa->offset = 0.0;
	abscissa->ca_to_deg = NULL;
	abscissa->ca_to_theta = NULL;
	abscissa->number_of_measurement_tables = 0;
	abscissa->number_of_samples = NULL;
	abscissa->resolution = NULL;
	abscissa->start = NULL;
	abscissa->theta_to_ca = NULL;
	abscissa->theta_weighting = NULL;
	abscissa->theta_weighting_inv = NULL;
	abscissa->type = ABSCISSA_UNKNOWN;
	abscissa->units[0] = 0x00;
	abscissa->volume = NULL;
	abscissa->slope = 1.0;
	abscissa->offset = 0.0;
	abscissa->axis_type = VARIABLE_NONE;
}

void Empty_Abscissa(AbscissaData* abscissa)
{
	if (abscissa == NULL)
	{
		return;
	}

	abscissa->units[0] = 0x00;
	abscissa->number_of_measurement_tables = 0;
	abscissa->type = ABSCISSA_UNKNOWN;

	if (abscissa->number_of_samples != NULL)
	{
		free(abscissa->number_of_samples);
		abscissa->number_of_samples = NULL;
	}

	if (abscissa->start != NULL)
	{
		free(abscissa->start);
		abscissa->start = NULL;
	}

	if (abscissa->resolution != NULL)
	{
		free(abscissa->resolution);
		abscissa->resolution = NULL;
	}

	if (abscissa->theta_to_ca != NULL)
	{
		free(abscissa->theta_to_ca);
		abscissa->theta_to_ca = NULL;
	}

	if (abscissa->ca_to_theta != NULL)
	{
		free(abscissa->ca_to_theta);
		abscissa->ca_to_theta = NULL;
	}

	if (abscissa->theta_weighting != NULL)
	{
		free(abscissa->theta_weighting);
		abscissa->theta_weighting = NULL;
	}

	if (abscissa->theta_weighting_inv != NULL)
	{
		free(abscissa->theta_weighting_inv);
		abscissa->theta_weighting_inv = NULL;
	}

	if (abscissa->volume != NULL)
	{
		free(abscissa->volume);
		abscissa->volume = NULL;
	}

	if (abscissa->ca_to_deg != NULL)
	{
		free(abscissa->ca_to_deg);
		abscissa->ca_to_deg = NULL;
	}

	if (abscissa->axis != NULL)
	{
		free(abscissa->axis);
		abscissa->axis = NULL;
	}
}

bool Calculate_Abscissa_Lookups(AbscissaData* abscissa, const unsigned int pulses_per_cycle, const unsigned int number_of_strokes, const float minimum_resolution)
{
	unsigned int samples_per_cycle = 0;
	unsigned int crank_angle;
	unsigned int mt;
	unsigned int theta;
	unsigned int start_theta;
	unsigned int finish_theta;
	unsigned int theta_range;
	unsigned int maximum_theta;
	bool result = true;

	if (abscissa == NULL)
	{
		return(false);
	}

	if ((abscissa->type != ABSCISSA_CRANKANGLE) ||
		(abscissa->number_of_measurement_tables == 0) ||
		(abscissa->number_of_samples == NULL) ||
		(abscissa->resolution == NULL) ||
		(abscissa->start == NULL) ||
		(pulses_per_cycle == 0) ||
		((number_of_strokes != 2) && (number_of_strokes != 4)) ||
		(minimum_resolution < FLT_EPSILON))
	{
		return(false);
	}

	/* Calculate our own number of samples */

	for (mt = 0; mt < abscissa->number_of_measurement_tables; mt++)
	{
		samples_per_cycle += abscissa->number_of_samples[mt];
	}

	if (samples_per_cycle == 0)
	{
		return(false);
	}

	/* Populate crank angle to theta lookup table */

	if (abscissa->ca_to_theta != NULL)
	{
		free(abscissa->ca_to_theta);
	}

	abscissa->ca_to_theta = (unsigned int*)malloc(samples_per_cycle * sizeof(unsigned int));
	if (abscissa->ca_to_theta == NULL)
	{
		logmessage(FATAL, "Memory could not be allocated\n");
	}

	for (crank_angle = 0; crank_angle<samples_per_cycle; crank_angle++)
	{
		abscissa->ca_to_theta[crank_angle] = ReturnCalculatedTheta(abscissa, crank_angle);
	}

	mt = abscissa->number_of_measurement_tables - 1;
	maximum_theta = abscissa->start[mt] + abscissa->number_of_samples[mt] * abscissa->resolution[mt];

	if (pulses_per_cycle * 2 > maximum_theta)
	{
		maximum_theta = pulses_per_cycle * 2;
	}

	if (abscissa->theta_to_ca != NULL)
	{
		free(abscissa->theta_to_ca);
	}

	abscissa->theta_to_ca = (unsigned int*)malloc(maximum_theta * sizeof(unsigned int));
	if (abscissa->theta_to_ca == NULL)
	{
		logmessage(FATAL, "Memory could not be allocated\n");
		exit(EXIT_FAILURE);
	}

	if (abscissa->theta_weighting != NULL)
	{
		free(abscissa->theta_weighting);
	}

	abscissa->theta_weighting = (float*)malloc(maximum_theta * sizeof(float));
	if (abscissa->theta_weighting == NULL)
	{
		logmessage(FATAL, "Memory could not be allocated\n");
		exit(EXIT_FAILURE);
	}

	if (abscissa->theta_weighting_inv != NULL)
	{
		free(abscissa->theta_weighting_inv);
	}

	abscissa->theta_weighting_inv = (float*)malloc(maximum_theta * sizeof(float));
	if (abscissa->theta_weighting_inv == NULL)
	{
		logmessage(FATAL, "Memory could not be allocated\n");
		exit(EXIT_FAILURE);
	}

	if (abscissa->ca_to_deg != NULL)
	{
		free(abscissa->ca_to_deg);
	}

	abscissa->ca_to_deg = (float*)malloc(samples_per_cycle* sizeof(float));
	if (abscissa->ca_to_deg == NULL)
	{
		logmessage(FATAL, "Memory could not be allocated\n");
		exit(EXIT_FAILURE);
	}

	for (theta = 0; theta<maximum_theta; theta++)
	{
		abscissa->theta_to_ca[theta] = 0;
		abscissa->theta_weighting[theta] = 0.5f;
		abscissa->theta_weighting_inv[theta] = 0.5f;
	}

	float offset = (float)(MINIMUM_THETA_ANGLE_MULT * number_of_strokes);

	for (crank_angle = 0; crank_angle<samples_per_cycle; crank_angle++)
	{
		start_theta = abscissa->ca_to_theta[crank_angle];

		if (crank_angle >= samples_per_cycle - 1)
		{
			finish_theta = maximum_theta;
		}
		else
		{
			finish_theta = abscissa->ca_to_theta[crank_angle + 1];
		}

		theta_range = finish_theta - start_theta;

		for (theta = start_theta; theta<finish_theta; theta++)
		{
			if (theta < maximum_theta)
			{
				abscissa->theta_to_ca[theta] = crank_angle;
				abscissa->theta_weighting[theta] = ((float)finish_theta - (float)theta) / (float)theta_range;
				abscissa->theta_weighting_inv[theta] = 1.0f - abscissa->theta_weighting[theta];
			}
		}

		if (finish_theta > maximum_theta)
		{
			result = false;
		}

		abscissa->ca_to_deg[crank_angle] = ((float)abscissa->ca_to_theta[crank_angle] / minimum_resolution) - offset;
	}

	return(result);
}

bool Build_Abscissa(const FileData* file,const AbscissaBlocks* channel_blocks, AbscissaData* abscissa)
{
	unsigned int block;
	unsigned int start_theta;
	unsigned int finish_theta;
	unsigned int resolution_theta;

	if ((file == NULL) || (channel_blocks == NULL) || (abscissa == NULL))
	{
		return(false);
	}

	if ((channel_blocks->number_of_blocks == 0) || (channel_blocks->number_of_blocks > 5))
	{
		return(false);
	}

	Empty_Abscissa(abscissa);

	abscissa->number_of_measurement_tables = channel_blocks->number_of_blocks;
	abscissa->type = ABSCISSA_CRANKANGLE;

	abscissa->start = (unsigned int*)malloc(channel_blocks->number_of_blocks * sizeof(unsigned int));
	if (abscissa->start == NULL)
	{
		logmessage(MSG_ERROR, "Unable to allocate memory\n");

		return(false);
	}

	abscissa->number_of_samples = (unsigned int*)malloc(channel_blocks->number_of_blocks * sizeof(unsigned int));
	if (abscissa->number_of_samples == NULL)
	{
		logmessage(MSG_ERROR, "Unable to allocate memory\n");

		free(abscissa->start);
		abscissa->start = NULL;

		return(false);
	}

	abscissa->resolution = (unsigned int*)malloc(channel_blocks->number_of_blocks * sizeof(unsigned int));
	if (abscissa->resolution == NULL)
	{
		logmessage(MSG_ERROR, "Unable to allocate memory\n");

		free(abscissa->start);
		abscissa->start = NULL;

		free(abscissa->number_of_samples);
		abscissa->number_of_samples = NULL;

		return(false);
	}

	for (block = 0; block < channel_blocks->number_of_blocks; block++)
	{
		start_theta = DegreesToTheta(file, channel_blocks->start[block]);
		finish_theta = DegreesToTheta(file, channel_blocks->finish[block]);
		resolution_theta = DegreesToThetaAbs(file, channel_blocks->resolution[block]);

		abscissa->start[block] = start_theta;
		abscissa->number_of_samples[block] = (finish_theta - start_theta) / resolution_theta;
		abscissa->resolution[block] = resolution_theta;
	}

	return(true);
}

void Add_Abscissa(AbscissaData* abscissa, const unsigned int number_of_strokes, const unsigned int min_res_ppr, const unsigned int mt, const float start_angle, const float finish_angle, const float resolution)
{
	FileData file;
	file.engine.number_of_strokes = number_of_strokes;
	file.minimum_resolution = (float)min_res_ppr / 360.0f;

	if ((number_of_strokes != 2) && (number_of_strokes != 4))
	{
		return;
	}

	if (min_res_ppr == 0)
	{
		return;
	}

	if (abscissa == NULL)
	{
		return;
	}

	if (finish_angle <= start_angle)
	{
		return;
	}

	if (resolution < FLT_EPSILON)
	{
		return;
	}

	if (mt >= abscissa->number_of_measurement_tables)
	{
		return;
	}

	abscissa->start[mt] = DegreesToTheta(&file, start_angle);
	abscissa->resolution[mt] = DegreesToThetaAbs(&file, resolution);
	abscissa->number_of_samples[mt] = DegreesToThetaAbs(&file, (finish_angle - start_angle)) / abscissa->resolution[mt];
}

unsigned int Abscissa_Samples(const AbscissaData* abscissa)
{
	unsigned int mt;
	unsigned int samples_per_cycle = 0;

	if (abscissa == NULL)
	{
		return(0);
	}

	if (abscissa->number_of_samples == NULL)
	{
		return(0);
	}

	if (abscissa->number_of_measurement_tables > 5)
	{
		return(0);
	}

	for (mt = 0; mt < abscissa->number_of_measurement_tables; mt++)
	{
		samples_per_cycle += abscissa->number_of_samples[mt];
	}

	return(samples_per_cycle);
}

void GetAxis(const AbscissaData* abscissa, const unsigned int variable_type, void* axis, const size_t axis_size)
{
	if ((abscissa == NULL) || 
		(axis == NULL) || 
		(axis_size == 0) || 
		(variable_type == VARIABLE_NONE))
	{
		return;
	}

	if ((abscissa->axis_type == VARIABLE_NONE) ||
		(abscissa->axis == NULL))
	{
		return;
	}

	if (variable_type == abscissa->axis_type)
	{
		/* Same variable type, just copy memory */

		size_t data_size;

		switch (variable_type)
		{
		case VARIABLE_INT64: data_size = axis_size * sizeof(int64_t);	break;
		case VARIABLE_INT32: data_size = axis_size * sizeof(int32_t);	break;
		case VARIABLE_FLOAT: data_size = axis_size * sizeof(float);		break;
		case VARIABLE_DOUBLE: data_size = axis_size * sizeof(double);	break;
		default: data_size = 0; break;
		}

		if (data_size > 0)
		{
			memcpy(axis, abscissa->axis, data_size);
		}

		return;
	}

	size_t data_pointer;

	switch (variable_type)
	{
	case VARIABLE_INT32:
	{
		switch (abscissa->axis_type)
		{
		case VARIABLE_INT64:
		{
			for (data_pointer = 0; data_pointer < axis_size; data_pointer++)
			{
				((int32_t*)axis)[data_pointer] = (int32_t)((int64_t*)abscissa->axis)[data_pointer];
			}

			break;
		}
		case VARIABLE_FLOAT:
		{
			for (data_pointer = 0; data_pointer < axis_size; data_pointer++)
			{
				((int32_t*)axis)[data_pointer] = (int32_t)((float*)abscissa->axis)[data_pointer];
			}

			break;
		}
		case VARIABLE_DOUBLE:
		{
			for (data_pointer = 0; data_pointer < axis_size; data_pointer++)
			{
				((int32_t*)axis)[data_pointer] = (int32_t)((double*)abscissa->axis)[data_pointer];
			}

			break;
		}
		}

		break;
	}
	case VARIABLE_INT64:
	{
		switch (abscissa->axis_type)
		{
		case VARIABLE_INT32:
		{
			for (data_pointer = 0; data_pointer < axis_size; data_pointer++)
			{
				((int64_t*)axis)[data_pointer] = (int64_t)((int32_t*)abscissa->axis)[data_pointer];
			}

			break;
		}
		case VARIABLE_FLOAT:
		{
			for (data_pointer = 0; data_pointer < axis_size; data_pointer++)
			{
				((int64_t*)axis)[data_pointer] = (int64_t)((float*)abscissa->axis)[data_pointer];
			}

			break;
		}
		case VARIABLE_DOUBLE:
		{
			for (data_pointer = 0; data_pointer < axis_size; data_pointer++)
			{
				((int64_t*)axis)[data_pointer] = (int64_t)((double*)abscissa->axis)[data_pointer];
			}

			break;
		}
		}

		break;
	}
	case VARIABLE_FLOAT:
	{
		switch (abscissa->axis_type)
		{
		case VARIABLE_INT32:
		{
			for (data_pointer = 0; data_pointer < axis_size; data_pointer++)
			{
				((float*)axis)[data_pointer] = (float)((int32_t*)abscissa->axis)[data_pointer];
			}

			break;
		}
		case VARIABLE_INT64:
		{
			for (data_pointer = 0; data_pointer < axis_size; data_pointer++)
			{
				((float*)axis)[data_pointer] = (float)((int64_t*)abscissa->axis)[data_pointer];
			}

			break;
		}
		case VARIABLE_DOUBLE:
		{
			for (data_pointer = 0; data_pointer < axis_size; data_pointer++)
			{
				((float*)axis)[data_pointer] = (float)((double*)abscissa->axis)[data_pointer];
			}

			break;
		}
		}

		break;
	}
	case VARIABLE_DOUBLE:
	{
		switch (abscissa->axis_type)
		{
		case VARIABLE_INT32:
		{
			for (data_pointer = 0; data_pointer < axis_size; data_pointer++)
			{
				((double*)axis)[data_pointer] = (double)((int32_t*)abscissa->axis)[data_pointer];
			}

			break;
		}
		case VARIABLE_INT64:
		{
			for (data_pointer = 0; data_pointer < axis_size; data_pointer++)
			{
				((double*)axis)[data_pointer] = (double)((int64_t*)abscissa->axis)[data_pointer];
			}

			break;
		}
		case VARIABLE_FLOAT:
		{
			for (data_pointer = 0; data_pointer < axis_size; data_pointer++)
			{
				((double*)axis)[data_pointer] = (double)((float*)abscissa->axis)[data_pointer];
			}

			break;
		}
		}

		break;
	}
	}
}